/*------------------------------------------------------------------------
 *  Copyright 2008 (c) Jeff Brown <spadix@users.sourceforge.net>
 *
 *  This file is part of the Zebra Barcode Library.
 *
 *  The Zebra Barcode Library is free software; you can redistribute it
 *  and/or modify it under the terms of the GNU Lesser Public License as
 *  published by the Free Software Foundation; either version 2.1 of
 *  the License, or (at your option) any later version.
 *
 *  The Zebra Barcode Library is distributed in the hope that it will be
 *  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 *  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser Public License
 *  along with the Zebra Barcode Library; if not, write to the Free
 *  Software Foundation, Inc., 51 Franklin St, Fifth Floor,
 *  Boston, MA  02110-1301  USA
 *
 *  http://sourceforge.net/projects/zebra
 *------------------------------------------------------------------------*/

#include <zebra.h>
#include "image.h"
#include "error.h"

#define LOGO_WIDTH 432
#define LOGO_HEIGHT 304
#define LOGO_RLE_SIZE 0x62c

static const uint8_t logo_rle[LOGO_RLE_SIZE] = {
    0x87,0x66, 0x04, 0x01, 0x60, 0x03, /*col 0*/
    0x8f,0x0e, 0x02, 0x80,0x02, 0x02, /*col 1*/
    0x8f,0x0a, 0x04, 0x80,0x06, 0x02, /*col 2*/
    0x8f,0x07, 0x80,0x0a, /*col 3*/
    0xaf,0xbf, 0x01, 0x80,0x15, /*col 8*/
    0x87,0xf8, 0x04, 0x80,0x17, /*col 9*/
    0x87,0xf7, 0x80,0x19, 0x01, /*col 10*/
    0x87,0xf4, 0x80,0x1c, /*col 11*/
    0xaf,0xa8, 0x06, 0x01, 0x80,0x27, /*col 16*/
    0x87,0xe3, 0x06, 0x03, 0x80,0x2b, /*col 17*/
    0x87,0xd3, 0x06, 0x04, 0x03, 0x42, 0x01, 0x80,0x2f, 0x01, /*col 18*/
    0x87,0xcc, 0x03, 0x01, 0x80,0x41, 0x04, /*col 19*/
    0x87,0xc9, 0x03, 0x80,0x45, /*col 20*/
    0x87,0xc7, 0x06, 0x80,0x48, /*col 21*/
    0x87,0xc6, 0x01, 0x80,0x48, 0x05, /*col 22*/
    0x87,0xc4, 0x05, 0x80,0x4a, /*col 23*/
    0xaf,0x64, 0x01, 0x80,0x67, /*col 28*/
    0x87,0xa7, 0x80,0x68, /*col 29*/
    0x87,0xa7, 0x80,0x68, /*col 30*/
    0x87,0xa7, 0x80,0x68, /*col 31*/
    0x87,0xa7, 0x80,0x68, /*col 32*/
    0x87,0xa6, 0x06, 0x80,0x68, /*col 33*/
    0x87,0xa6, 0x02, 0x80,0x68, /*col 34*/
    0x87,0xa6, 0x80,0x68, 0x01, /*col 35*/
    0x87,0xa6, 0x80,0x68, 0x01, /*col 36*/
    0x87,0xa5, 0x05, 0x80,0x68, 0x02, /*col 37*/
    0x87,0xa5, 0x03, 0x80,0x68, 0x01, /*col 38*/
    0x87,0xa5, 0x02, 0x80,0x69, /*col 39*/
    0xc7,0x02,0x20, 0x03, 0x80,0x67, 0x03, /*col 48*/
    0x87,0xa7, 0x80,0x68, /*col 49*/
    0x87,0xa8, 0x80,0x67, /*col 50*/
    0x87,0xa9, 0x80,0x66, 0x06, /*col 51*/
    0xc7,0x06,0xe8, 0x03, 0x80,0x5c, /*col 64*/
    0x87,0xb3, 0x03, 0x80,0x5b, /*col 65*/
    0x87,0xb4, 0x03, 0x80,0x5a, /*col 66*/
    0x87,0xb5, 0x03, 0x80,0x59, 0x06, /*col 67*/
    0xc7,0x0b,0xbb, 0x80,0x5b, 0x01, /*col 84*/
    0x87,0xb4, 0x03, 0x80,0x5c, /*col 85*/
    0x87,0xb4, 0x80,0x5d, /*col 86*/
    0x87,0xb3, 0x80,0x5e, /*col 87*/
    0xaf,0x73, 0x80,0x62, /*col 92*/
    0x87,0xae, 0x04, 0x80,0x62, /*col 93*/
    0x87,0xae, 0x80,0x62, 0x01, /*col 94*/
    0x87,0xad, 0x80,0x63, 0x04, /*col 95*/
    0xaf,0x6d, 0x01, 0x80,0x66, /*col 100*/
    0x87,0xa9, 0x06, 0x80,0x67, /*col 101*/
    0x87,0xa9, 0x80,0x67, 0x01, /*col 102*/
    0x87,0xa8, 0x80,0x68, 0x01, /*col 103*/
    0x87,0xa7, 0x04, 0x80,0x68, 0x02, /*col 104*/
    0x87,0xa7, 0x80,0x69, 0x02, /*col 105*/
    0x87,0xa6, 0x80,0x6a, 0x01, /*col 106*/
    0x87,0xa5, 0x02, 0x80,0x6a, 0x01, /*col 107*/
    0xc7,0x06,0xe8, 0x80,0x7f, 0x03, /*col 120*/
    0x87,0x90, 0x80,0x81, 0x06, /*col 121*/
    0x87,0x8e, 0x80,0x82, 0x01, /*col 122*/
    0x87,0x8d, 0x03, 0x80,0x83, /*col 123*/
    0xc7,0x02,0x10, 0x80,0x89, 0x04, /*col 132*/
    0x87,0x86, 0x06, 0x80,0x8b, 0x03, 0x87,0x67, 0x01, 0x08, 0x05, /*col 133*/
    0x87,0x09, 0x80,0x8e, 0x03, 0x87,0x62, 0x01, 0x20, 0x02, /*col 134*/
    0x87,0x08, 0x05, 0x80,0x90, 0x03, 0x87,0x5d, 0x06, 0x38, 0x06, /*col 135*/
    0x87,0x08, 0x80,0x93, 0x02, 0x87,0x59, 0x04, 0x48, /*col 136*/
    0x87,0x08, 0x05, 0x80,0x95, 0x02, 0x87,0x55, 0x02, 0x50, /*col 137*/
    0x87,0x09, 0x80,0x98, 0x05, 0x87,0x51, 0x02, 0x58, /*col 138*/
    0x87,0x09, 0x04, 0x80,0x99, 0x01, 0x87,0x4e, 0x01, 0x60, 0x03, /*col 139*/
    0x87,0x09, 0x80,0x9c, 0x03, 0x87,0x4a, 0x78, /*col 140*/
    0x87,0x09, 0x03, 0x80,0x9e, 0x04, 0x87,0x46, 0x01, 0x78, /*col 141*/
    0x87,0x0a, 0x80,0xa1, 0x01, 0x05, 0x87,0x41, 0x02, 0x80,0x01, /*col 142*/
    0x87,0x0a, 0x04, 0x80,0xa5, 0x02, 0x05, 0x87,0x3b, 0x01, 0x80,0x02, /*col 143*/
    0xa7,0xcd, 0x03, 0x80,0xbe, 0x02, 0x05, 0x87,0x17, 0x03, 0x80,0x09, 0x01, /*col 148*/
    0x87,0x0e, 0x01, 0x80,0xc3, 0x01, 0x03, 0x04, 0x86,0x0c, 0x04, 0x01, 0x80,0x0c, 0x05, /*col 149*/
    0x87,0x0f, 0x03, 0x80,0xff, /*col 150*/
    0x87,0x11, 0x05, 0x80,0xfe, /*col 151*/
    0xa7,0xd8, 0x80,0xf1, 0x04, /*col 156*/
    0x87,0x1f, 0x01, 0x80,0xea, 0x06, /*col 157*/
    0x87,0x25, 0x05, 0x80,0xe7, /*col 158*/
    0x87,0x2a, 0x80,0xe5, /*col 159*/
    0x87,0x2c, 0x02, 0x80,0xe2, /*col 160*/
    0x87,0x2f, 0x01, 0x80,0xdf, /*col 161*/
    0x87,0x31, 0x05, 0x80,0xdd, 0x05, /*col 162*/
    0x87,0x33, 0x80,0xac, 0x01, 0x02, 0x03, 0x0c, 0x05, 0x06, 0x3f, 0x06, 0x0d, 0x04, 0x03, 0x02, 0x01, 0x80,0x0a, /*col 163*/
    0x87,0x35, 0x80,0x8d, 0x09, 0x02, 0x03, 0x04, 0x0e, 0x87,0x28, 0x06, 0x04, 0x01, 0x48, 0x01, /*col 164*/
    0x87,0x36, 0x02, 0x80,0x77, 0x02, 0x03, 0x0d, 0x06, 0x87,0x45, 0x04, 0x01, 0x18, 0x06, /*col 165*/
    0x87,0x38, 0x80,0x74, 0x05, /*col 166*/
    0x87,0x9b, 0x80,0x76, 0x06, /*col 167*/
    0x87,0x99, 0x80,0x78, 0x06, /*col 168*/
    0x87,0x97, 0x80,0x7a, 0x05, /*col 169*/
    0x87,0x95, 0x01, 0x80,0x7b, 0x02, /*col 170*/
    0x87,0x93, 0x03, 0x80,0x7e, 0x06, /*col 171*/
    0xaf,0x51, 0x80,0x8c, 0x02, 0x06, /*col 176*/
    0x87,0x82, 0x80,0x90, 0x02, 0x05, /*col 177*/
    0x87,0x7e, 0x80,0x94, 0x01, 0x04, /*col 178*/
    0x87,0x7a, 0x01, 0x80,0x98, 0x01, 0x05, /*col 179*/
    0xaf,0x36, 0x80,0xa8, 0x01, /*col 184*/
    0x87,0x67, 0x80,0xaa, 0x03, /*col 185*/
    0x87,0x65, 0x80,0xac, 0x06, 0x87,0x23, 0x02, 0x10, 0x01, /*col 186*/
    0x87,0x2b, 0x80,0xae, 0x06, 0x87,0x1f, 0x01, 0x30, /*col 187*/
    0xa7,0xeb, 0x04, 0x80,0xc2, 0x04, 0x7f, 0x06, 0x78, 0x01, /*col 192*/
    0x87,0x2b, 0x80,0xc9, 0x09, 0x02, 0x03, 0x04, 0x0d, 0x04, 0x80,0x03, 0x01, /*col 193*/
    0x87,0x2b, 0x80,0xe4, 0x01, /*col 194*/
    0x87,0x2b, 0x06, 0x80,0xe3, 0x01, /*col 195*/
    0xa7,0xed, 0x80,0x6e, 0x87,0x22, 0x06, 0x04, 0x02, 0x80,0x2f, 0x02, /*col 200*/
    0x87,0x2d, 0x05, 0x80,0x6e, 0x87,0x29, 0x05, 0x03, 0x01, 0x80,0x27, 0x04, /*col 201*/
    0x87,0x2e, 0x80,0x6e, 0x05, 0x87,0x2f, 0x06, 0x04, 0x02, 0x80,0x20, /*col 202*/
    0x87,0x2f, 0x80,0x6f, 0x87,0x36, 0x06, 0x04, 0x02, 0x80,0x18, 0x01, /*col 203*/
    0x87,0x2f, 0x05, 0x80,0x6e, 0x05, 0x87,0x3d, 0x05, 0x03, 0x01, 0x80,0x0f, 0x02, /*col 204*/
    0x87,0x31, 0x80,0x6f, 0x87,0x44, 0x06, 0x03, 0x78, 0x01, 0x06, /*col 205*/
    0x87,0x39, 0x80,0x6f, 0x87,0x48, 0x06, 0x01, 0x48, 0x04, /*col 206*/
    0x87,0x3c, 0x01, 0x80,0x6e, 0x01, 0x87,0x4b, 0x04, 0x18, 0x04, /*col 207*/
    0xaf,0x00, 0x01, 0x80,0x6f, 0x06, /*col 212*/
    0x87,0x9f, 0x03, 0x80,0x6f, 0x03, /*col 213*/
    0x87,0x9f, 0x03, 0x80,0x70, /*col 214*/
    0x87,0x9f, 0x03, 0x80,0x70, /*col 215*/
    0x87,0x9f, 0x03, 0x80,0x70, /*col 216*/
    0x87,0x9f, 0x03, 0x80,0x70, 0x04, /*col 217*/
    0x87,0x9e, 0x03, 0x80,0x71, /*col 218*/
    0x87,0x9e, 0x03, 0x80,0x71, /*col 219*/
    0xc7,0x02,0x1e, 0x03, 0x80,0x74, /*col 228*/
    0x87,0x9b, 0x02, 0x80,0x74, /*col 229*/
    0x87,0x9b, 0x01, 0x80,0x74, 0x06, /*col 230*/
    0x87,0x9a, 0x80,0x75, 0x03, /*col 231*/
    0x87,0x9a, 0x80,0x75, 0x01, /*col 232*/
    0x87,0x9a, 0x80,0x76, /*col 233*/
    0x87,0x9a, 0x80,0x76, /*col 234*/
    0x87,0x9a, 0x80,0x76, /*col 235*/
    0xc7,0x02,0x19, 0x80,0x77, /*col 244*/
    0x87,0x98, 0x06, 0x80,0x77, /*col 245*/
    0x87,0x98, 0x04, 0x80,0x77, /*col 246*/
    0x87,0x98, 0x02, 0x80,0x77, 0x06, /*col 247*/
    0x87,0x97, 0x80,0x78, 0x06, /*col 248*/
    0x87,0x97, 0x80,0x78, 0x05, /*col 249*/
    0x87,0x96, 0x04, 0x80,0x78, 0x02, /*col 250*/
    0x87,0x96, 0x80,0x7a, /*col 251*/
    0xaf,0x55, 0x80,0x7b, 0x06, /*col 256*/
    0x87,0x93, 0x02, 0x80,0x7b, 0x06, /*col 257*/
    0x87,0x93, 0x80,0x7c, /*col 258*/
    0x87,0x94, 0x80,0x7c, /*col 259*/
    0x87,0x93, 0x04, 0x80,0x7c, /*col 260*/
    0x87,0x93, 0x80,0x7d, /*col 261*/
    0x87,0x93, 0x80,0x7c, 0x02, /*col 262*/
    0x87,0x92, 0x05, 0x80,0x7c, 0x05, /*col 263*/
    0x87,0x92, 0x01, 0x80,0x7c, /*col 264*/
    0x87,0x93, 0x80,0x7d, /*col 265*/
    0x87,0x93, 0x80,0x7c, 0x04, /*col 266*/
    0x87,0x92, 0x02, 0x80,0x7c, /*col 267*/
    0xc7,0x02,0x11, 0x80,0x7c, /*col 276*/
    0x87,0x94, 0x80,0x7b, 0x04, /*col 277*/
    0x87,0x93, 0x06, 0x80,0x7b, /*col 278*/
    0x87,0x94, 0x05, 0x80,0x7a, 0x02, /*col 279*/
    0xaf,0x54, 0x80,0x79, /*col 284*/
    0x87,0x97, 0x80,0x78, 0x01, /*col 285*/
    0x87,0x97, 0x80,0x78, /*col 286*/
    0x87,0x98, 0x80,0x77, 0x03, /*col 287*/
    0x87,0x97, 0x04, 0x80,0x77, /*col 288*/
    0x87,0x98, 0x80,0x77, 0x03, /*col 289*/
    0x87,0x98, 0x80,0x77, /*col 290*/
    0x87,0x99, 0x80,0x76, 0x04, /*col 291*/
    0xc7,0x02,0x16, 0x05, 0x80,0x72, /*col 300*/
    0x87,0x9d, 0x01, 0x80,0x71, /*col 301*/
    0x87,0x9e, 0x80,0x71, /*col 302*/
    0x87,0x9f, 0x80,0x6f, 0x01, /*col 303*/
    0x87,0x9f, 0x06, 0x80,0x6e, 0x03, /*col 304*/
    0x87,0xa0, 0x02, 0x80,0x6d, 0x03, /*col 305*/
    0x87,0xa1, 0x80,0x6d, 0x03, /*col 306*/
    0x87,0xa2, 0x80,0x6c, 0x02, /*col 307*/
    0xaf,0x62, 0x05, 0x80,0x6f, 0x01, /*col 312*/
    0x87,0x9f, 0x05, 0x80,0x71, 0x02, 0x87,0x54, 0x05, 0x0a, 0x04, /*col 313*/
    0x87,0x35, 0x05, 0x80,0x73, 0x03, 0x87,0x4e, 0x06, 0x38, 0x06, /*col 314*/
    0x87,0x33, 0x05, 0x80,0x75, 0x04, 0x87,0x4a, 0x04, 0x50, 0x02, /*col 315*/
    0x87,0x32, 0x05, 0x80,0x77, 0x05, 0x87,0x46, 0x02, 0x68, 0x01, /*col 316*/
    0x87,0x31, 0x05, 0x80,0x79, 0x06, 0x87,0x42, 0x06, 0x80,0x01, /*col 317*/
    0x87,0x31, 0x05, 0x80,0x7b, 0x87,0x40, 0x80,0x03, 0x03, /*col 318*/
    0x87,0x30, 0x05, 0x80,0x7c, 0x01, 0x87,0x3b, 0x04, 0x80,0x06, /*col 319*/
    0xa7,0xf0, 0x05, 0x80,0x85, 0x87,0x24, 0x05, 0x01, 0x80,0x13, /*col 324*/
    0x87,0x31, 0x05, 0x80,0x86, 0x01, 0x87,0x1e, 0x06, 0x02, 0x80,0x16, 0x01, /*col 325*/
    0x87,0x31, 0x05, 0x80,0x88, 0x02, 0x87,0x17, 0x05, 0x03, 0x01, 0x80,0x1a, /*col 326*/
    0x87,0x32, 0x05, 0x80,0x8b, 0x01, 0x82,0x0e, 0x01, 0x80,0x21, 0x06, /*col 327*/
    0x87,0x32, 0x05, 0x80,0xdb, 0x05, /*col 328*/
    0x87,0x33, 0x05, 0x80,0xda, 0x02, /*col 329*/
    0x87,0x34, 0x05, 0x80,0xca, 0x01, 0x03, 0x14, 0x48, 0x03, /*col 330*/
    0x87,0x35, 0x05, 0x80,0xc4, 0x01, 0x04, 0x47, 0x40, 0x01, /*col 331*/
    0xc7,0x01,0xb7, 0x01, 0x80,0xd7, 0x05, /*col 340*/
    0x87,0x38, 0x80,0xd6, 0x06, /*col 341*/
    0x87,0x39, 0x01, 0x80,0xd4, /*col 342*/
    0x87,0x3c, 0x80,0xd2, 0x06, /*col 343*/
    0xc7,0x01,0xc2, 0x04, 0x80,0x65, 0x01, 0x7f, 0x02, 0x80,0x1d, 0x03, 0x06, /*col 352*/
    0x87,0x6a, 0x80,0x64, 0x05, 0x87,0x02, 0x06, 0x80,0x16, 0x02, 0x04, /*col 353*/
    0x87,0x71, 0x80,0x62, 0x06, 0x87,0x05, 0x01, 0x80,0x0f, 0x03, 0x06, /*col 354*/
    0x87,0x76, 0x02, 0x80,0x5f, 0x04, 0x87,0x08, 0x04, 0x80,0x08, 0x01, 0x04, /*col 355*/
    0xc7,0x06,0xc7, 0x01, 0x80,0x33, 0x01, /*col 368*/
    0x87,0xdc, 0x02, 0x80,0x30, 0x03, /*col 369*/
    0x87,0xdf, 0x04, 0x80,0x2d, 0x04, /*col 370*/
    0x87,0xe2, 0x06, 0x80,0x2a, 0x03, /*col 371*/
    0xa7,0xa7, 0x04, 0x01, 0x78, 0x09, 0x02, 0x03, 0x04, 0x06, /*col 375*/
    0x87,0xde, 0x04, 0x01, 0x80,0x34, 0x01, 0x03, 0x06, /*col 376*/
    0x87,0xdc, 0x05, 0x03, 0x01, 0x80,0x33, 0x01, 0x06, /*col 377*/
    0x87,0xe0, 0x06, 0x05, 0x04, 0x02, 0x01, 0x80,0x01, 0x02, 0x01, 0x80,0x19, 0x03, /*col 378*/
    0x87,0xf3, 0x0a, 0x01, 0x80,0x1b, 0x02, /*col 379*/
    0x87,0xf8, 0x04, 0x80,0x18, 0x05, /*col 380*/
    0x87,0xfa, 0x06, 0x02, 0x80,0x15, /*col 381*/
    0x87,0xfd, 0x05, 0x01, 0x80,0x12, 0x06, /*col 382*/
    0x87,0xfe, 0x06, 0x01, 0x80,0x10, 0x05, /*col 383*/
    0x8f,0x00, 0x06, 0x01, 0x80,0x0e, 0x02, /*col 384*/
    0x8f,0x02, 0x06, 0x01, 0x80,0x0d, /*col 385*/
    0x8f,0x04, 0x04, 0x80,0x0c, /*col 386*/
    0x8f,0x05, 0x06, 0x80,0x0b, /*col 387*/
    0x8f,0x07, 0x01, 0x80,0x09, /*col 388*/
    0x8f,0x08, 0x03, 0x80,0x07, 0x01, /*col 389*/
    0x8f,0x09, 0x03, 0x80,0x06, 0x03, /*col 390*/
    0x8f,0x0a, 0x80,0x06, 0x06, /*col 391*/
    0x8f,0x0a, 0x04, 0x80,0x05, /*col 392*/
    0x8f,0x0c, 0x80,0x05, /*col 393*/
    0x8f,0x0c, 0x80,0x05, /*col 394*/
    0x8f,0x0c, 0x03, 0x80,0x03, 0x04, /*col 395*/
    0x8f,0x0c, 0x06, 0x80,0x03, /*col 396*/
    0x8f,0x0e, 0x80,0x03, /*col 397*/
    0x8f,0x0e, 0x01, 0x80,0x01, 0x05, /*col 398*/
    0x8f,0x0e, 0x06, 0x80,0x01, /*col 399*/
    0x8f,0x10, 0x01, 0x70, 0x01, /*col 400*/
    0x8f,0x11, 0x70, 0x06, /*col 401*/
    0x8f,0x11, 0x03, 0x68, /*col 402*/
    0x8f,0x13, 0x60, 0x04, /*col 403*/
    0x8f,0x13, 0x03, 0x58, /*col 404*/
    0x8f,0x14, 0x06, 0x50, 0x02, /*col 405*/
    0x8f,0x15, 0x50, /*col 406*/
    0x8f,0x16, 0x48, 0x01, /*col 407*/
    0x8f,0x15, 0x06, 0x48, /*col 408*/
    0x8f,0x16, 0x01, 0x48, /*col 409*/
    0x8f,0x16, 0x48, 0x05, /*col 410*/
    0x8f,0x15, 0x02, 0x48, /*col 411*/
    0x8f,0x16, 0x50, /*col 412*/
    0x8f,0x15, 0x05, 0x48, 0x03, /*col 413*/
    0x8f,0x15, 0x50, /*col 414*/
    0x8f,0x15, 0x06, 0x50, /*col 415*/
    0x8f,0x15, 0x01, 0x48, 0x03, /*col 416*/
    0x8f,0x15, 0x50, /*col 417*/
    0x8f,0x15, 0x02, 0x48, 0x01, /*col 418*/
    0x8f,0x15, 0x50, /*col 419*/
    0x8f,0x15, 0x05, 0x48, 0x03, /*col 420*/
    0x8f,0x15, 0x48, /*col 421*/
    0x8f,0x16, 0x06, 0x40, /*col 422*/
    0x8f,0x17, 0x01, 0x38, /*col 423*/
    0x8f,0x18, 0x38, /*col 424*/
    0x8f,0x18, 0x03, 0x30, 0x05, /*col 425*/
    0x8f,0x18, 0x10, 0x07, 0x18, /*col 426*/
    0x8f,0x18, 0x05, 0x08, 0x0f, 0x05, 0x08, 0x05, /*col 427*/
    0x8f,0x18, 0x08, 0x1f, 0x03, 0x00, /*col 428*/
    0x8f,0x19, 0x00, /*col 429*/
    0x8f,0x1f, 0x01, /*col 430*/
    0x8f,0x99,
};

int _zebra_draw_logo (zebra_image_t *img)
{
    unsigned w = zebra_image_get_width(img);
    unsigned h = zebra_image_get_height(img);
    if(w < 128 || h < 90) {
        w = 128;
        h = 90;
        zebra_image_set_size(img, w, h);
    }

    zebra_image_set_format(img, fourcc('Y','8','0','0'));
    zprintf(5, "fitting (%d x %d) into (%d x %d)\n",
            LOGO_WIDTH, LOGO_HEIGHT, w, h);

    unsigned sw = 1, sh = 1;
    int sf;
    for(sf = 3; sf >= -2; sf--) {
        sw = (LOGO_WIDTH << 3) >> (3 - sf);
        sh = (LOGO_HEIGHT << 3) >> (3 - sf);
        /* FIXME add minimum border */
        zprintf(6, "    trying << %d => %d x %d\n", sf, sw, sh);
        if(sw <= w && sh <= h)
            break;
    }

    int datalen = w * h;
    uint8_t *data = malloc(datalen);
    assert(data);
    zebra_image_set_data(img, data, datalen, zebra_image_free_data);
    memset(data, 0xff, datalen);

    unsigned xpad = (w - sw + 1) / 2;
    unsigned ypad = (h - sh + 1) / 2;

    const uint8_t *rle = logo_rle;
    uint8_t *p = NULL; /* (((pix & 0x7) * 255 + 6) / 7) */
    unsigned n = 0, pix = 0x7;
    unsigned x, y, i, j, mask = (sf >= 0) ? (1 << sf) - 1 : (1 << -sf) - 1;

    for(x = 0; x < LOGO_WIDTH; x++) {
        p = data + (ypad * w) + xpad + ((x << 3) >> (3 - sf));
        zprintf(64, "  [%03d]", x);
        for(y = 0; y < LOGO_HEIGHT; y++) {
            if(!n) {
                pix = *(rle++);
                if((pix & 0xc0) == 0xc0) {
                    n = *(rle++) << 8;
                    n += *(rle++) + 0x810;
                }
                else if(pix & 0x80)
                    n = (((pix & 0x38) << 5) | *(rle++)) + 0x10;
                else
                    n = ((pix >> 3) & 0x0f) + 1;
                pix &= 0x7;
                if(sf >= 0)
                    pix = ((pix * 255) + 6) / 7;
                if(_zebra_verbosity >= 64)
                    fprintf(stderr, " (%03d) pix=%02x n=%04x", y, pix, n);
                
            }

            if(!sf) {
                *p = pix;
                p += w;
            }
            else if(sf > 0)
                for(i = 0; i <= mask; i++) {
                    for(j = 0; j <= mask; j++)
                        *(p++) = pix;
                    p += w - mask - 1;
                }
            else {
                if(!((x | y) & mask))
                    *p = pix;
                else
                    *p += pix;
                if((y & mask) == mask)
                    p += w;
            }
            n--;
        }
        if(_zebra_verbosity >= 64)
            fprintf(stderr, "\n");
    }

    if(sf < 0) {
        /* make 2nd pass to finish merging subsamples */
        p = data + (ypad * w) + xpad;
        unsigned div = (mask + 1) * (mask + 1) * 7;
        for(y = 0; y < sh; y++) {
            for(x = 0; x < sw; x++, p++)
                *p = (((unsigned)*p) * 255 + div - 1) / div;
            p += w - sw;
        }
    }

    zprintf(6, "    rle=%p + size=%x = %p\n", logo_rle, LOGO_RLE_SIZE, rle);
    if(rle != logo_rle + LOGO_RLE_SIZE)
        fprintf(stderr, "WARNING: zebra internal: logo rle not at end?\n");
    zprintf(6, "    data=%p + len=%x = %p\n", data, datalen, p);

    return(0);
}
