/* This is for emacs: -*-Mode: C++;-*- */
/*  
  Copyright 2002, Andreas Rottmann

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/
#if !defined(_INC_YEHIA_NODE_H)
#define _INC_YEHIA_NODE_H

#include <glib.h>

namespace Yehia
{

template <class T> class G_Node_Iterator;
template <class T> class G_Node_Recursive_Iterator;

template <class T>
class G_Node
{
    friend class G_Node_Iterator<T>;
  public:
    /** Iterator over children. */
    typedef G_Node_Iterator<T> iterator;
    /** Const iterator over children. */
    typedef G_Node_Iterator<T> const_iterator;
    /** Depth-first search iterator. */
    typedef G_Node_Recursive_Iterator<T> recursive_iterator;
    typedef G_Node_Recursive_Iterator<T> const_recursive_iterator;
    
    G_Node(G_Node_Iterator<T> it) : node_(it.node) { }
    G_Node(const T& t) : node_(g_node_new(new T(t))) { }
    G_Node(const G_Node<T>& node) : node_(node.node_) { }
    ~G_Node() {
      if (node_ && !node_->parent)
      {
        erase(begin(), end());
        if (node_ && node_->data) delete (T *)node_->data;
        g_node_destroy(node_);
      }
    }

    /** Get iterator at first child */
    iterator begin() { 
      return g_node_first_child(node_) ?  g_node_first_child(node_) : 0; 
    }
    /** Get iterator at first child */
    const_iterator begin() const { 
      return g_node_first_child(node_) ?  g_node_first_child(node_) : 0; 
    }
    /** Get iterator after last child */
    iterator end() { return 0; }
    /** Get iterator after last child */
    const_iterator end() const { return 0; }
    
    iterator insert(int pos, const T& t) { 
      return g_node_insert(node_, pos, G_Node(t).node_);
    }
    iterator insert_before(iterator it, const T& t) {
      return g_node_insert_before(node_, it.node, G_Node<T>(t).node_);
    }
    iterator insert_after(iterator it, const T& t) {
      return g_node_insert_before(node_, it.node, G_Node<T>(t).node_);
    }
    iterator push_front(const T& t) {
      return g_node_prepend(node_, G_Node(t).node_);
    }
    iterator push_back(const T& t) {
      return g_node_append(node_, G_Node(t).node_);
    }
    void reverse() { g_node_reverse_children(node_); }
    
    iterator root() { 
      return g_node_get_root(node_); 
    }
    iterator parent() { 
      return node_->parent; 
    }
    /** Remove node.
     * \param it Iterator. */
    void erase(iterator it) { G_Node<T>(g_node_unlink(it.node)); }
    void erase(iterator start, iterator end) { 
      for (iterator it = start; it != end; ++it)
        G_Node<T>(g_node_unlink(it.node)); 
    }
    /** Test relationship.
     * \return boolean; true if this node is an ancestor of \p descendant */
    bool is_ancestor_of(const iterator& descendant) const {
      return g_node_is_ancestor(node_, descendant.node);
    }
  private:
    GNode *node_;

    static gboolean delete_node_data(GNode *node, gpointer data) {
      delete (T *)node->data;
      return FALSE;
    }
};

template <class T> 
class G_Node_Iterator
{
  public:
    typedef T value_type;
    typedef T* pointer;
    typedef T& reference;

    GNode *node;
    
    typedef G_Node_Iterator<T> Self;
    
    G_Node_Iterator(GNode *n = 0) : node(n) { }
    G_Node_Iterator(const G_Node<T>& n) : node(n.node_) { }
    G_Node_Iterator(const G_Node_Iterator<T>& it) : node(it.node) { }
    
    bool operator ==(const Self& x) const { return node == x.node; }
    bool operator !=(const Self& x) const { return node != x.node; }

    Self&  operator++() {
      if (node)
        node = g_node_next_sibling(node);
      return *this;
    }

    
    Self operator++(int) {
      Self tmp = *this;
      ++*this;
      return tmp;
    }
    
    Self&  operator--() {
      if (node)
        node = g_node_prev_sibling(node);
      return *this;
    }

    Self operator--(int) {
      Self tmp = *this;
      --*this;
      return tmp;
    }

    Self parent() const { return node ? node->parent : 0; }
    
    reference operator*() const {
      return *(pointer)(node ? node->data : 0);
    }
    pointer operator->() const {
      return (pointer)(node ? node->data : 0);
    }
};

template <class T> class G_Node_Recursive_Iterator : public G_Node_Iterator<T>
{
  private:
    GNode *end_;
  public:
    typedef G_Node_Recursive_Iterator<T> Self;
    typedef G_Node_Iterator<T> Parent;
    
    G_Node_Recursive_Iterator(GNode *n) 
        : Parent(n), end_(n ? n->parent : n) { }
    G_Node_Recursive_Iterator(const G_Node<T>& n) 
        : Parent(n), end_(n ? n.node_->parent : n) { }
    G_Node_Recursive_Iterator(const G_Node_Iterator<T>& it) 
        : Parent(it), end_(it.node ? it.node->parent : it.node) { }
    
    Self& operator++() {
      if (node)
      {
        if (node->children) 
          node = g_node_first_child(node);
        else if (node == end_)
          node = 0;
        else if (node->next) 
          node = g_node_next_sibling(node);
        else if (node->parent)
        {
          do
          {
            node = node->parent;
          } while (node && !node->next);
          if (node)
            node = g_node_next_sibling(node);
        }
        else node = 0;
      }
      return *this;
    }
    
    Self operator++(int) {
      Self tmp = *this;
      ++*this;
      return tmp;
    }
    
    Self&  operator--() {
      if (node)
        node = g_node_prev_sibling(node);
      return *this;
    }

    Self operator--(int) {
      Self tmp = *this;
      --*this;
      return tmp;
    }
};


}

#endif
