/*  
  Copyright 2002, Andreas Rottmann

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/
#include <yehia/plugin.h>

#include <libguile.h>

#include "guile-script.h"
#include "guile-loader.h"

using namespace std;
using namespace SigC;
using namespace SigCX;
using namespace SigCX::Threads;

namespace Yehia
{

namespace Script
{

namespace
{

void thrower(std::exception& e)
{
  throw e;
}

}

class GuilePlugin : public Plugin
{
    static void main_prog(void *, int, char*[]);
  public:
    GuilePlugin(PluginManager& mgr);
    virtual ~GuilePlugin();

    virtual std::string name() const { return "guile"; }
    virtual std::string description() const { return "Guile support"; }
    Language *language() { return lang_; }
    guileObjectFactory& factory() { return obj_factory_; }

    bool as_extension() const { return as_extension_; }
    Tunnel *tunnel() { return tunnel_; }
    PluginLoader *loader() { return loader_; }
    void init();
    
    void run();
  private:
    Dispatcher *disp_;
    Language *lang_;
    Script::guileObjectFactory obj_factory_;
    Script::guileObject root_ns_;
    ThreadTunnel *tunnel_;
    Plugin *yehia_;
    guilePluginLoader *loader_;
    Threads::Mutex mutex_;
    bool as_extension_;
};


class GuileDispatcher : public StandardDispatcher
{
    static void main_prog(void *closure, int argc, char *argv[]);
  public:
    GuileDispatcher(GuilePlugin& plugin) { }
    virtual bool run(bool inf) {
      char *argv[1];
      argv[0] = NULL;
      try
      {
        scm_boot_guile(0, argv, main_prog, this);
      }
      catch (std::exception& e)
      {
        tunnel<void, std::exception&>(
                slot(&thrower), e, LanguageManager::instance().main_tunnel());
      }
      return true;
    }
};

GuilePlugin::GuilePlugin(PluginManager& mgr)
    : Plugin(mgr), root_ns_(obj_factory_)
{
  if ((yehia_ = mgr.load_plugin("yehia")) == 0)
    throw std::runtime_error("loading yehia plugin failed");
  
  yehia_->reference();

  as_extension_ = (bool)getenv("YEHIA_GUILE_AS_EXTENSION");
  
  lang_ = 0; // this must be created after guile is booted
  loader_ = 0;
  tunnel_ = 0;
  disp_ = 0;
}


GuilePlugin::~GuilePlugin()
{
  mutex_.lock();
  yehia_->unreference();
  manager().unregister_plugin_loader(*loader_);
  //manager()->unregister_language("guile");
  if (tunnel_) 
  {
    SigCX::tunnel(slot(*disp_, &Dispatcher::exit), tunnel_, true);
    delete tunnel_;
  }
  mutex_.unlock();
}

void GuilePlugin::init()
{
  scm_load_goops();

  // We must register the following types: bool, long, unsigned long, 
  // double, std::string, Any, std::list<Any>
  mutex_.lock();

  obj_factory_.register_class(typeid(bool), 
                              obj_factory_.create_object(scm_class_boolean));
  obj_factory_.register_class(typeid(long), 
                              obj_factory_.create_object(scm_class_integer));
  obj_factory_.register_class(typeid(unsigned long), 
                              obj_factory_.create_object(scm_class_integer));
  obj_factory_.register_class(typeid(double), 
                              obj_factory_.create_object(scm_class_real));
  obj_factory_.register_class(typeid(std::string), 
                              obj_factory_.create_object(scm_class_string));
  obj_factory_.register_class(typeid(Any), 
                              obj_factory_.create_object(scm_class_top));
  obj_factory_.register_class(typeid(std::list<Any>), 
                              obj_factory_.create_object(scm_class_list));

  loader_ = manage(new guilePluginLoader(manager()));
  lang_ = manage(new Language(*root_ns_.namespace_interface(), 
                              obj_factory_, tunnel_));

  mutex_.unlock();
}

void GuilePlugin::run()
{
  mutex_.lock();
  disp_ = manage(new GuileDispatcher(*this));
  tunnel_ = new ThreadTunnel(*disp_);
  mutex_.unlock();
}

void GuileDispatcher::main_prog(void *closure, int argc, char *argv[])
{
  GuileDispatcher *disp = (GuileDispatcher *)closure;
  disp->StandardDispatcher::run();
}

} // Script

}

using namespace Yehia;

extern "C" Plugin *yehia_guile_plugin_init(PluginManager *mgr)
{
  using namespace Yehia::Script;
  
  try
  {
    GuilePlugin *plugin = SigC::manage(new GuilePlugin(*mgr));
    
    plugin->reference();
    
    // If we are loaded as a Guile extension module, not to embed
    // Guile into a program, we run single-threaded and don't have to
    // boot Guile.
    if (plugin->as_extension())
      plugin->init();
    else
    {
      plugin->run();
      tunnel(slot(*plugin, &GuilePlugin::init), plugin->tunnel(), true);
    }
    LanguageManager::instance().register_language("guile", 
                                                   *plugin->language());
    mgr->register_plugin_loader(*plugin->loader());

    // Now the yehia plugin has registered itself with Guile, we do
    // some fixups
    plugin->factory().yehia_init_fixup();
    
    return plugin;
  }
  catch (...)
  {
    mgr->set_error("GuilePlugin initialisation failed");
    return 0;
  }
}
