// file      : xsde/cxx/elements.cxx
// author    : Boris Kolpackov <boris@codesynthesis.com>
// copyright : Copyright (c) 2005-2007 Code Synthesis Tools CC
// license   : GNU GPL v2 + exceptions; see accompanying LICENSE file

#include <cxx/elements.hxx>

#include <backend-elements/regex.hxx>

#include <sstream>
#include <iostream>
#include <algorithm>

using std::wcerr;
using std::endl;

namespace CXX
{
  //
  //
  wchar_t
  upcase (wchar_t c)
  {
    return std::toupper (c);
  }

  std::locale Context::locale ("C");

  // Potentially-qualified C++ id.
  //
  static String cxx_id_expr (L"/^(::)?([a-zA-Z_]\\w*)(::[a-zA-Z_]\\w*)*$//");

  // Context
  //

  Context::
  Context (std::wostream& o,
           SemanticGraph::Schema& root,
           NarrowString const& char_type__,
           Boolean include_with_brackets__,
           NarrowString const& include_prefix__,
           NarrowString const& esymbol__,
           Containers::Vector<NarrowString> const& nsm,
           Containers::Vector<NarrowString> const& nsr,
           Boolean trace_namespace_regex_,
           Containers::Vector<NarrowString> const& ir,
           Boolean trace_include_regex_,
           Boolean inline_)
      : os (o),
        schema_root (root),
        char_type (char_type_),
        L (L_),
        string_type (string_type_),
        include_with_brackets (include_with_brackets_),
        include_prefix (include_prefix_),
        esymbol (esymbol_),
        inl (inl_),
        ns_mapping_cache (ns_mapping_cache_),
        char_type_ (char_type__),
        L_ (char_type == L"wchar_t" ? L"L" : L""),
        string_type_ (L"::std::basic_string< " + char_type + L" >"),
        include_with_brackets_ (include_with_brackets__),
        include_prefix_ (include_prefix__),
        esymbol_ (esymbol__),
        inl_ (inline_ ? L"inline\n" : L""),
        trace_namespace_regex (trace_namespace_regex_),
        nsr_mapping (nsr_mapping_),
        nsm_mapping (nsm_mapping_),
        include_mapping (include_mapping_),
        trace_include_regex (trace_include_regex_)
  {
    using BackendElements::Regex::perl_s;

    // Default mapping.
    //
    nsr_mapping_.push_back (
      L"#^.* (.*?/)??"L"(([a-zA-Z_]\\w*)(/[a-zA-Z_]\\w*)*)/?$#$2#");
    nsr_mapping_.push_back (
      L"#^.* http://www\\.w3\\.org/2001/XMLSchema$#xml_schema#");

    // Custom regex mapping.
    //
    for (Containers::Vector<NarrowString>::ConstIterator
           i (nsr.begin ()), e (nsr.end ()); i != e; ++i)
    {
      nsr_mapping_.push_back (*i);
    }

    // Custom direct mapping.
    //

    for (Containers::Vector<NarrowString>::ConstIterator
           i (nsm.begin ()), e (nsm.end ()); i != e; ++i)
    {
      String s (*i);

      // Split the string in two parts at the last '='.
      //
      Size pos (s.rfind ('='));

      if (pos == String::npos)
        throw InvalidNamespaceMapping (s, "no delimiter ('=') found");

      // Empty xml_ns designates the no-namespace case.
      //
      String xml_ns (s, 0, pos);
      String cxx_ns (s, pos + 1);

      if (!cxx_ns.empty () && !perl_s (cxx_ns, cxx_id_expr).empty ())
        throw InvalidNamespaceMapping (s, "invalid C++ identifier");

      nsm_mapping_[xml_ns] = cxx_ns;
    }

    // Include path regex
    //
    for (Containers::Vector<NarrowString>::ConstIterator
           i (ir.begin ()), e (ir.end ()); i != e; ++i)
    {
      include_mapping_.push_back (*i);
    }
  }

  String Context::
  ns_name (SemanticGraph::Namespace& ns)
  {
    using BackendElements::Regex::perl_s;

    using SemanticGraph::Schema;
    using SemanticGraph::Includes;
    using SemanticGraph::Imports;
    using SemanticGraph::Implies;
    using SemanticGraph::Sources;



    String tmp;
    MapMapping::ConstIterator i (nsm_mapping.find (ns.name ()));


    if (i != nsm_mapping.end ())
    {
      tmp = i->second;
    }
    else
    {
      SemanticGraph::Path path;
      Schema& schema (dynamic_cast<Schema&> (ns.scope ()));

      if (schema.used ())
      {
        SemanticGraph::Uses& u (*schema.used_begin ());
        path = u.path ();
      }

      String pair (path.empty () ? "" : path.string ());

      pair += L' ' + ns.name ();

      // Check cache first
      //
      MappingCache::ConstIterator i (ns_mapping_cache.find (pair));

      if (i != ns_mapping_cache.end ())
      {
        tmp = i->second;
      }
      else
      {
        if (trace_namespace_regex)
          wcerr << "pat: '" << pair << "'" << endl;

        Boolean found (false);

        for (RegexMapping::ConstReverseIterator e (nsr_mapping.rbegin ());
             e != nsr_mapping.rend (); ++e)
        {
          tmp = perl_s (pair, *e);
          tmp = perl_s (tmp, L"#/#::#"); // replace `/' with `::'

          // Check the result.
          //
          if (perl_s (tmp, cxx_id_expr).empty ())
            found = true;

          if (trace_namespace_regex)
            wcerr << "try: '" << *e << "' : '" << tmp << "' : "
                  << (found ? '+' : '-') << endl;

          if (found)
            break;
        }

        if (!found)
        {
          // Check if the name is valid by itself.
          //
          if (ns.name ().empty ())
          {
            // Empty name denotes a no-namespace case.
            //
            tmp = ns.name ();
          }
          else
          {
            tmp = perl_s (ns.name (), L"#/#::#"); // replace `/' with `::'

            if (!perl_s (tmp, cxx_id_expr).empty ())
            {
              throw NoNamespaceMapping (
                ns.file (), ns.line (), ns.column (), ns.name ());
            }
          }
        }

        // Add the mapping to the cache.
        //
        ns_mapping_cache[pair] = tmp;
      }
    }


    // Parse resulting namespace string and id() each name.
    //
    String r;
    String::size_type b (0), e;

    do
    {
      e = tmp.find (L"::", b);

      String name (tmp, b, e == tmp.npos ? e : e - b);

      if (!name.empty ())
        r += L"::" + escape (name);

      b = e;

      if (b == tmp.npos)
        break;

      b += 2;

    } while (true);

    return r;
  }

  String Context::
  xs_ns_name ()
  {
    SemanticGraph::Scope::NamesIteratorPair r (
      schema_root.find ("http://www.w3.org/2001/XMLSchema"));

    assert (r.first != r.second);

    return ns_name (
      dynamic_cast<SemanticGraph::Namespace&> (r.first->named ()));
  }

  SemanticGraph::Namespace& Context::
  namespace_ (SemanticGraph::Nameable& n)
  {
    // The basic idea goes like this: go up Names edges until you
    // reach Namespace. There are, however, anonymous types which
    // need special handling. In the case of an anonymous type we
    // will go up the first Belongs edge (because the first edge
    // is where the type was defined.
    //

    if (n.named ())
    {
      SemanticGraph::Scope& s (n.scope ());

      SemanticGraph::Namespace* ns (
        dynamic_cast<SemanticGraph::Namespace*> (&n));

      return ns ? *ns : namespace_ (s);
    }
    else
    {
      SemanticGraph::Type& t (dynamic_cast<SemanticGraph::Type&> (n));

      SemanticGraph::Belongs& b (*t.classifies_begin ());

      return namespace_ (b.instance ());
    }
  }

  String Context::
  xml_ns_name (SemanticGraph::Nameable& n)
  {
    return namespace_ (n).name ();
  }

  String Context::
  fq_name (SemanticGraph::Nameable& n, Char const* name_key)
  {
    using namespace SemanticGraph;

    String r;

    if (dynamic_cast<Schema*> (&n))
    {
      return L""; // Map to global namespace.
    }
    else if (SemanticGraph::Namespace* ns =
             dynamic_cast<SemanticGraph::Namespace*> (&n))
    {
      r = ns_name (*ns);
    }
    else
    {
      r = fq_name (n.scope ());
      r += L"::";
      r += n.context ().get<String> (name_key);
    }

    return r;
  }

  namespace
  {
    WideChar* keywords[] = {
      L"and",
      L"asm",
      L"auto",
      L"bitand",
      L"bitor",
      L"bool",
      L"break",
      L"case",
      L"catch",
      L"char",
      L"class",
      L"compl",
      L"const",
      L"const_cast",
      L"continue",
      L"default",
      L"delete",
      L"do",
      L"double",
      L"dynamic_cast",
      L"else",
      L"end_eq",
      L"enum",
      L"explicit",
      L"export",
      L"extern",
      L"false",
      L"float",
      L"for",
      L"friend",
      L"goto",
      L"if",
      L"inline",
      L"int",
      L"long",
      L"mutable",
      L"namespace",
      L"new",
      L"not",
      L"not_eq",
      L"operator",
      L"or",
      L"or_eq",
      L"private",
      L"protected",
      L"public",
      L"register",
      L"reinterpret_cast",
      L"return",
      L"short",
      L"signed",
      L"sizeof",
      L"static",
      L"static_cast",
      L"struct",
      L"switch",
      L"template",
      L"this",
      L"throw",
      L"true",
      L"try",
      L"typedef",
      L"typeid",
      L"typename",
      L"union",
      L"unsigned",
      L"using",
      L"virtual",
      L"void",
      L"volatile",
      L"wchar_t",
      L"while",
      L"xor",
      L"xor_eq"
    };
  }

  String Context::
  escape (String const& name)
  {
    Size const size (sizeof (keywords) / sizeof (WideChar*));

    if (std::binary_search (keywords, keywords + size, name))
      return name + L'_';

    String r (name);

    if (!std::isalpha (r[0], locale) && r[0] != L'_')
      r = L"cxx_" + r;

    for (String::Iterator i (r.begin () + 1), e (r.end ()); i != e; ++i)
      if (!std::isalnum (*i, locale) && *i != L'_')
        *i = L'_';

    return r;
  }

  String Context::
  escape_str (String const& str)
  {
    String r;
    r.reserve (str.size ());

    for (String::ConstIterator i (str.begin ()), e (str.end ()); i != e; ++i)
    {
      switch (*i)
      {
      case L'"':
        {
          r.append (L"\\\"");
          break;
        }
      case L'\\':
        {
          r.append (L"\\\\");
          break;
        }
      default:
        {
          r.push_back (*i);
          break;
        }
      }
    }

    return r;
  }

  String Context::
  process_include_path (String const& name) const
  {
    using BackendElements::Regex::perl_s;

    String path (include_prefix + name);

    if (trace_include_regex)
      wcerr << "pat: '" << path << "'" << endl;

    String r;
    Boolean found (false);

    for (RegexMapping::ConstReverseIterator e (include_mapping.rbegin ());
         e != include_mapping.rend (); ++e)
    {
      r = perl_s (path, *e);

      found = (r != path);

      if (trace_include_regex)
        wcerr << "try: '" << *e << "' : '" << r << "' : "
              << (found ? '+' : '-') << endl;

      if (found)
        break;
    }

    if (!found)
      r = path;

    if (!r.empty () && r[0] != L'"' && r[0] != L'<')
    {
      WideChar op (include_with_brackets ? L'<' : L'"');
      WideChar cl (include_with_brackets ? L'>' : L'"');
      r = op + r + cl;
    }

    return r;
  }

  // Namespace
  //

  Void Namespace::
  pre (Type& n)
  {
    String ns (ctx_.ns_name (n));

    String::size_type b (0), e;

    if (st_)
      st_->enter (L"");

    do
    {
      e = ns.find (L"::", b);

      String name (ns, b, e == ns.npos ? e : e - b);

      if (!name.empty ())
      {
        ctx_.os << "namespace " << name << "{";

        if (st_)
          st_->enter (name);
      }


      b = e;

      if (b == ns.npos)
        break;

      b += 2;

    } while (true);
  }

  Void Namespace::
  post (Type& n)
  {
    String ns (ctx_.ns_name (n));

    String::size_type b (0), e;

    do
    {
      e = ns.find (L"::", b);

      String name (ns, b, e == ns.npos ? e : e - b);

      if (!name.empty ())
      {
        ctx_.os << "}";

        if (st_)
          st_->leave ();
      }


      b = e;

      if (b == ns.npos)
        break;

      b += 2;

    }
    while (true);

    if (st_)
      st_->leave ();
  }

  // Include
  //
  Void Includes::
  traverse (SemanticGraph::Path const& path)
  {
    using BackendElements::Regex::perl_s;

    // Note the use of the portable representation of the path.
    //
    String name (perl_s (String (path.string ()), expr_));

    ctx_.os << "#include " << ctx_.process_include_path (name) << endl
            << endl;
  }
}
