/*-
# X-BASED RUBIK'S CUBE(tm)
#
#  Rubik.c
#
###
#
#  Copyright (c) 1994 - 2007	David A. Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/* Methods file for Rubik */

#include "file.h"
#include "rngs.h"
#include "sound.h"
#include "RubikP.h"
#include "Rubik2dP.h"
#include "Rubik3dP.h"
#ifdef HAVE_OPENGL
#include "RubikGLP.h"
#endif

#ifdef WINVER
#ifndef LOGPATH
#define LOGPATH "/usr/tmp"
#endif
#ifndef INIFILE
#define INIFILE "wrubik.ini"
#endif

#define SECTION "setup"

static const char *faceColorString[MAX_FACES] =
{
	"255 0 0",
	"255 255 0",
	"255 255 255",
	"0 255 0",
	"255 165 0",
	"0 0 255"
};

static const char faceColorChar[MAX_FACES] =
{'R', 'Y', 'W', 'G', 'O', 'B'};
#else

#ifndef LOGPATH
#ifdef VMS
#define LOGPATH "SYS$SCRATCH:"
#else
#define LOGPATH "/usr/tmp"
#endif
#endif

static Boolean SetValuesPuzzle(Widget current, Widget request, Widget renew);
static void DestroyPuzzle(Widget old);
static void InitializePuzzle(Widget request, Widget renew);

RubikClassRec rubikClassRec =
{
	{
		(WidgetClass) & widgetClassRec,		/* superclass */
		(char *) "Rubik",	/* class name */
		sizeof (RubikRec),	/* widget size */
		NULL,		/* class initialize */
		NULL,		/* class part initialize */
		FALSE,		/* class inited */
		(XtInitProc) InitializePuzzle,	/* initialize */
		NULL,		/* initialize hook */
		XtInheritRealize,	/* realize */
		NULL,		/* actions */
		0,		/* num actions */
		NULL,		/* resources */
		0,		/* num resources */
		NULLQUARK,	/* xrm class */
		TRUE,		/* compress motion */
		TRUE,		/* compress exposure */
		TRUE,		/* compress enterleave */
		TRUE,		/* visible interest */
		(XtWidgetProc) DestroyPuzzle,	/* destroy */
		NULL,		/* resize */
		NULL,		/* expose */
		(XtSetValuesFunc) SetValuesPuzzle,	/* set values */
		NULL,		/* set values hook */
		XtInheritSetValuesAlmost,	/* set values almost */
		NULL,		/* get values hook */
		NULL,		/* accept focus */
		XtVersion,	/* version */
		NULL,		/* callback private */
		NULL,		/* tm table */
		NULL,		/* query geometry */
		NULL,		/* display accelerator */
		NULL		/* extension */
	},
	{
		0		/* ignore */
	}
};

WidgetClass rubikWidgetClass = (WidgetClass) & rubikClassRec;

void
setPuzzle(RubikWidget w, int reason)
{
	rubikCallbackStruct cb;

	cb.reason = reason;
	XtCallCallbacks((Widget) w, (char *) XtNselectCallback, &cb);
}

static void
setPuzzleMove(RubikWidget w, int reason, int face, int position, int direction,
		Boolean control, int fast)
{
	rubikCallbackStruct cb;

	cb.reason = reason;
	cb.face = face;
	cb.position = position;
	cb.direction = direction;
	cb.control = control;
	cb.fast = fast;
	XtCallCallbacks((Widget) w, (char *) XtNselectCallback, &cb);
}
#endif

static void
loadFont(RubikWidget w)
{
#ifndef WINVER
	Display *display = XtDisplay(w);
	const char *altfontname = "-*-times-*-r-*-*-*-180-*";
	char buf[512];

	if (w->rubik.fontInfo) {
		XUnloadFont(XtDisplay(w), w->rubik.fontInfo->fid);
		XFreeFont(XtDisplay(w), w->rubik.fontInfo);
	}
	if (w->rubik.font && (w->rubik.fontInfo =
			XLoadQueryFont(display, w->rubik.font)) == NULL) {
		(void) sprintf(buf,
			"Can not open %s font.\nAttempting %s font as alternate\n",
			w->rubik.font, altfontname);
		DISPLAY_WARNING(buf);
		if ((w->rubik.fontInfo = XLoadQueryFont(display,
				altfontname)) == NULL) {
			(void) sprintf(buf,
				"Can not open %s alternate font.\nUse the -font option to specify a font to use.\n",
				altfontname);
			DISPLAY_WARNING(buf);
		}
	}
	if (w->rubik.fontInfo) {
		w->rubik.letterOffset.x = XTextWidth(w->rubik.fontInfo, "8", 1)
			/ 2;
		w->rubik.letterOffset.y = w->rubik.fontInfo->max_bounds.ascent
			/ 2;
	} else
#endif
	{
		w->rubik.letterOffset.x = 3;
		w->rubik.letterOffset.y = 4;
	}
}

#ifndef LOGFILE
#define LOGFILE "rubik.log"
#endif

static int facesToDirection[MAX_FACES][MAX_FACES] =
{
	{-1, 3, 2, 1, -3, 0},
	{0, -1, 1, -3, 2, 3},
	{0, 3, -1, 1, 2, -3},
	{0, -3, 3, -1, 2, 1},
	{-3, 3, 0, 1, -1, 2},
	{2, 3, -3, 1, 0, -1}
};
static RubikLoc slideNextRow[MAX_FACES][MAX_ORIENT] =
{
	{
		{5, TOP},
		{3, RIGHT},
		{2, TOP},
		{1, LEFT}
	},
	{
		{0, RIGHT},
		{2, TOP},
		{4, LEFT},
		{5, BOTTOM}
	},
	{
		{0, TOP},
		{3, TOP},
		{4, TOP},
		{1, TOP}
	},
	{
		{0, LEFT},
		{5, BOTTOM},
		{4, RIGHT},
		{2, TOP}
	},
	{
		{2, TOP},
		{3, LEFT},
		{5, TOP},
		{1, RIGHT}
	},
	{
		{4, TOP},
		{3, BOTTOM},
		{0, TOP},
		{1, BOTTOM}
	}
};
static int rowToRotate[MAX_FACES][MAX_ORIENT] =
{
	{3, 2, 1, 5},
	{2, 4, 5, 0},
	{3, 4, 1, 0},
	{5, 4, 2, 0},
	{3, 5, 1, 2},
	{3, 0, 1, 4}
};

static RubikStack undo = {NULL, NULL, NULL, 0};
static RubikStack redo = {NULL, NULL, NULL, 0};

static void
CheckPieces(RubikWidget w)
{
	char *buf1 = NULL, *buf2 = NULL;

	if (w->rubik.sizex < MIN_FACETS) {
		intCat(&buf1,
			"Number of rubik in X direction out of bounds, use at least ",
			MIN_FACETS);
		stringCat(&buf2, buf1, ", defaulting to ");
		free(buf1);
		intCat(&buf1, buf2, DEFAULTFACETS);
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
		w->rubik.sizex = DEFAULTFACETS;
	}
	if (w->rubik.sizey < MIN_FACETS) {
		intCat(&buf1,
			"Number of rubik in Y direction out of bounds, use at least ",
			MIN_FACETS);
		stringCat(&buf2, buf1, ", defaulting to ");
		free(buf1);
		intCat(&buf1, buf2, DEFAULTFACETS);
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
		w->rubik.sizey = DEFAULTFACETS;
	}
	if (w->rubik.sizez < MIN_FACETS) {
		intCat(&buf1,
			"Number of rubik in Z direction out of bounds, use at least ",
			MIN_FACETS);
		stringCat(&buf2, buf1, ", defaulting to ");
		free(buf1);
		intCat(&buf1, buf2, DEFAULTFACETS);
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
		w->rubik.sizez = DEFAULTFACETS;
	}
	if (w->rubik.delay < 0) {
		intCat(&buf1, "Delay can not be negative (",
			w->rubik.delay);
		stringCat(&buf2, buf1, "), taking absolute value");
		free(buf1);
		DISPLAY_WARNING(buf2);
		free(buf2);
		w->rubik.delay = -w->rubik.delay;
	}
}

void
faceSizes(const RubikWidget w, const int face,
		int *sizeOfRow, int *sizeOfColumn)
{
	switch (face) {
	case 0:	/* TOP */
	case 4:	/* BOTTOM */
		*sizeOfRow = w->rubik.sizex;
		*sizeOfColumn = w->rubik.sizez;
		break;
	case 1:	/* LEFT */
	case 3:	/* RIGHT */
		*sizeOfRow = w->rubik.sizez;
		*sizeOfColumn = w->rubik.sizey;
		break;
	case 2:	/* FRONT */
	case 5:	/* BACK */
		*sizeOfRow = w->rubik.sizex;
		*sizeOfColumn = w->rubik.sizey;
		break;
	default:
		{
			char *buf;

			intCat(&buf, "faceSizes: face ", face);
			DISPLAY_WARNING(buf);
			free(buf);
		}
	}
}

int
sizeFace(const RubikWidget w, const int face)
{
	int sizeOfRow, sizeOfColumn;

	faceSizes(w, face, &sizeOfRow, &sizeOfColumn);
	return (sizeOfRow * sizeOfColumn);
}

int
sizeRow(const RubikWidget w, const int face)
{
	int sizeOfRow, sizeOfColumn;

	faceSizes(w, face, &sizeOfRow, &sizeOfColumn);
	return sizeOfRow;
}

static Boolean
checkFaceSquare(RubikWidget w, int face)
{
	int sizeOfRow, sizeOfColumn;

	faceSizes(w, face, &sizeOfRow, &sizeOfColumn);
	return (sizeOfRow == sizeOfColumn);
	/* Cubes can be made square with a 4x2 face where 90 degree turns
	 * should be permitted but that is kind of complicated for me.
	 * This can be done in 2 ways, where the side of the facets are
	 * the same size and one where one side (the side with half the
	 * number of facets) is twice the size of the other.  The first is
	 * complicated because faces of cubies can go under other faces.
	 * The second way is similar to "banded rubik" where scotch tape
	 * restricts the moves of the puzzle.  Here you have to keep track
	 * of the restrictions and show banded cubies graphically as one
	 * cube.
	 */
}

Boolean
CheckSolved(const RubikWidget w, Boolean rotation)
{
	int face, position;
	RubikLoc test = {0, FALSE};

	for (face = 0; face < MAX_FACES; face++)
		for (position = 0; position < sizeFace(w, face); position++) {
			if (!position) {
				test.face = w->rubik.cubeLoc[face][position].face;
				test.rotation = w->rubik.cubeLoc[face][position].rotation;
			} else if (test.face !=		/* face */
				   w->rubik.cubeLoc[face][position].face ||
				   (rotation && test.rotation !=		/* STRT - MAX_ORIENT */
				  w->rubik.cubeLoc[face][position].rotation))
				return False;
		}
	return True;
}

#ifdef DEBUG
void
PrintCube(RubikWidget w)
{
	int face, position, sizeOfRow, sizeOfColumn;

	for (face = 0; face < MAX_FACES; face++) {
		faceSizes(w, face, &sizeOfRow, &sizeOfColumn);
		for (position = 0; position < sizeOfRow * sizeOfColumn; position++) {
			(void) printf("%d %d  ", w->rubik.cubeLoc[face][position].face,
				w->rubik.cubeLoc[face][position].rotation);
			if (!((position + 1) % sizeOfRow))
				(void) printf("\n");
		}
		(void) printf("\n");
	}
	(void) printf("\n");
}

void
PrintRow(RubikWidget w, int orient, int size)
{
	int i;

	(void) printf("Row %d:\n", orient);
	for (i = 0; i < size; i++)
		(void) printf("%d %d  ", w->rubik.rowLoc[orient][i].face,
			w->rubik.rowLoc[orient][i].rotation);
	(void) printf("\n");
}

#endif

static void
DrawSquare(const RubikWidget w, const int face, const int position, const int offset)
{
	if (w->rubik.dim == 2)
		DrawSquare2D((Rubik2DWidget) w, face, position, offset);
	else if (w->rubik.dim == 3)
		DrawSquare3D((Rubik3DWidget) w, face, position, offset);
}

void
DrawAllPieces(const RubikWidget w)
{
	int face, position;

#ifdef HAVE_OPENGL
	if (w->rubik.dim == 4) {
		DrawAllPiecesGL((RubikGLWidget) w);
		return;
	}
#endif
	for (face = 0; face < MAX_FACES; face++)
		for (position = 0; position < sizeFace(w, face); position++)
			DrawSquare(w, face, position, FALSE);
}

static void
DrawFrame(const RubikWidget w, const Boolean focus)
{
	if (w->rubik.dim == 2)
		DrawFrame2D((Rubik2DWidget) w, focus);
	else if (w->rubik.dim == 3)
		DrawFrame3D((Rubik3DWidget) w, focus);
#ifdef HAVE_OPENGL
	else if (w->rubik.dim == 4)
		DrawFrameGL((RubikGLWidget) w, focus);
#endif
}

static void
MoveNoPieces(RubikWidget w)
{
	setPuzzle(w, ACTION_ILLEGAL);
}

static void
RotateFace(RubikWidget w, int face, int direction)
{
	int position, i, j, sizeOfRow, sizeOfColumn, sizeOnPlane;
	RubikLoc *faceLoc = NULL;

	faceSizes(w, face, &sizeOfRow, &sizeOfColumn);
	sizeOnPlane = sizeOfRow * sizeOfColumn;
	if ((faceLoc = (RubikLoc *) malloc(sizeOnPlane * sizeof (RubikLoc))) ==
				NULL) {
			DISPLAY_ERROR("Not enough memory for rubik face position info, exiting.");
	}
	/* Read Face */
	for (position = 0; position < sizeOnPlane; position++)
		faceLoc[position] = w->rubik.cubeLoc[face][position];
	/* Write Face */
	for (position = 0; position < sizeOnPlane; position++) {
		i = position % sizeOfRow;
		j = position / sizeOfRow;
		if (direction == CW)
			w->rubik.cubeLoc[face][position] =
				faceLoc[(sizeOfRow - i - 1) * sizeOfRow + j];
		else if (direction == CCW)
			w->rubik.cubeLoc[face][position] =
				faceLoc[i * sizeOfRow + sizeOfColumn - j - 1];
		else		/* (direction == HALF) */
			w->rubik.cubeLoc[face][position] =
				faceLoc[sizeOfRow - i - 1 + (sizeOfColumn - j - 1) * sizeOfRow];

		w->rubik.cubeLoc[face][position].rotation =
			(w->rubik.cubeLoc[face][position].rotation +
			 direction - MAX_ORIENT) % MAX_ORIENT;
		DrawSquare(w, face, position, FALSE);
	}
	if (faceLoc != NULL)
		free(faceLoc);
}

/* Yeah this is big and ugly */
static void
SlideRC(RubikWidget w, int face, int direction, int h, int sizeOnOppAxis,
	int *newFace, int *newDirection, int *newH,
	int *rotate, Boolean *reverse)
{
	*newFace = slideNextRow[face][direction].face;
	*rotate = slideNextRow[face][direction].rotation;
	*newDirection = (*rotate + direction) % MAX_ORIENT;
	switch (*rotate) {
	case TOP:
		*newH = h;
		*reverse = False;
		break;
	case RIGHT:
		if (*newDirection == TOP || *newDirection == BOTTOM) {
			*newH = sizeOnOppAxis - 1 - h;
			*reverse = False;
		} else {	/* *newDirection == RIGHT ||
				   *newDirection == LEFT */
			*newH = h;
			*reverse = True;
		}
		break;
	case BOTTOM:
		*newH = sizeOnOppAxis - 1 - h;
		*reverse = True;
		break;
	case LEFT:
		if (*newDirection == TOP || *newDirection == BOTTOM) {
			*newH = h;
			*reverse = True;
		} else {	/* *newDirection == RIGHT ||
				   *newDirection == LEFT */
			*newH = sizeOnOppAxis - 1 - h;
			*reverse = False;
		}
		break;
	default:
		{
			char *buf;

			intCat(&buf, "SlideRC: rotate ", *rotate);
			DISPLAY_WARNING(buf);
			free(buf);
		}
	}
}

static void
ReadRC(RubikWidget w, int face, int dir, int h, int orient, int size)
{
	int g, sizeOfRow;

	sizeOfRow = sizeRow(w, face);
	if (dir == TOP || dir == BOTTOM)
		for (g = 0; g < size; g++)
			w->rubik.rowLoc[orient][g] =
				w->rubik.cubeLoc[face][g * sizeOfRow + h];
	else			/* dir == RIGHT || dir == LEFT */
		for (g = 0; g < size; g++)
			w->rubik.rowLoc[orient][g] =
				w->rubik.cubeLoc[face][h * sizeOfRow + g];
}

static void
RotateRC(RubikWidget w, int rotate, int orient, int size)
{
	int g;

	for (g = 0; g < size; g++)
		w->rubik.rowLoc[orient][g].rotation =
			(w->rubik.rowLoc[orient][g].rotation + rotate) % MAX_ORIENT;
}

static void
ReverseRC(RubikWidget w, int orient, int size)
{
	int g;
	RubikLoc temp;

	for (g = 0; g < size / 2; g++) {
		temp = w->rubik.rowLoc[orient][size - 1 - g];
		w->rubik.rowLoc[orient][size - 1 - g] = w->rubik.rowLoc[orient][g];
		w->rubik.rowLoc[orient][g] = temp;
	}
}

static void
WriteRC(RubikWidget w, int face, int dir, int h, int orient, int size)
{
	int g, position, sizeOfRow;

	sizeOfRow = sizeRow(w, face);
	if (dir == TOP || dir == BOTTOM) {
		for (g = 0; g < size; g++) {
			position = g * sizeOfRow + h;
			w->rubik.cubeLoc[face][position] = w->rubik.rowLoc[orient][g];
			DrawSquare(w, face, position, FALSE);
		}
	} else {		/* dir == RIGHT || dir == LEFT */
		for (g = 0; g < size; g++) {
			position = h * sizeOfRow + g;
			w->rubik.cubeLoc[face][position] = w->rubik.rowLoc[orient][g];
			DrawSquare(w, face, position, FALSE);
		}
	}
}

static void
MovePieces(RubikWidget w, int face, int position, int direction,
		Boolean control, int fast)
{
	int newFace, newDirection, rotate;
	int h, k, newH = 0;
	int i, j, sizeOfRow, sizeOfColumn, sizeOnAxis, sizeOnOppAxis;
	Boolean reverse = FALSE;

#ifdef HAVE_OPENGL
	if (!control && fast != INSTANT && w->rubik.dim == 4) {
		MovePiecesGL((RubikGLWidget) w, face, position, direction,
			control, fast);
	}
#endif
	faceSizes(w, face, &sizeOfRow, &sizeOfColumn);
	i = position % sizeOfRow;
	j = position / sizeOfRow;
	h = (direction == TOP || direction == BOTTOM) ? i : j;
	if (direction == TOP || direction == BOTTOM) {
		sizeOnAxis = sizeOfColumn;
		sizeOnOppAxis = sizeOfRow;
	} else {
		sizeOnAxis = sizeOfRow;
		sizeOnOppAxis = sizeOfColumn;
	}
	/* rotate sides CW or CCW or HALF */

	if (h == sizeOnOppAxis - 1) {
		newDirection = (direction == TOP || direction == BOTTOM) ?
			TOP : RIGHT;
		if (w->rubik.degreeTurn == 180)
			RotateFace(w, rowToRotate[face][newDirection], HALF);
		else if (direction == TOP || direction == RIGHT)
			RotateFace(w, rowToRotate[face][newDirection], CW);
		else		/* direction == BOTTOM || direction == LEFT */
			RotateFace(w, rowToRotate[face][newDirection], CCW);
	}
	if (h == 0) {
		newDirection = (direction == TOP || direction == BOTTOM) ?
			BOTTOM : LEFT;
		if (w->rubik.degreeTurn == 180)
			RotateFace(w, rowToRotate[face][newDirection], HALF);
		else if (direction == TOP || direction == RIGHT)
			RotateFace(w, rowToRotate[face][newDirection], CCW);
		else		/* direction == BOTTOM || direction == LEFT */
			RotateFace(w, rowToRotate[face][newDirection], CW);
	}
	/* Slide rows */
	ReadRC(w, face, direction, h, 0, sizeOnAxis);
	if (w->rubik.degreeTurn == 180) {
		int sizeOnDepthAxis;

		SlideRC(w, face, direction, h, sizeOnOppAxis,
			&newFace, &newDirection, &newH, &rotate, &reverse);
		sizeOnDepthAxis = sizeFace(w, newFace) / sizeOnOppAxis;
		ReadRC(w, newFace, newDirection, newH, 1, sizeOnDepthAxis);
		RotateRC(w, rotate, 0, sizeOnAxis);
		if (reverse == True)
			ReverseRC(w, 0, sizeOnAxis);
		face = newFace;
		direction = newDirection;
		h = newH;
		for (k = 2; k <= MAX_ORIENT + 1; k++) {
			SlideRC(w, face, direction, h, sizeOnOppAxis,
			  &newFace, &newDirection, &newH, &rotate, &reverse);
			if (k != MAX_ORIENT && k != MAX_ORIENT + 1)
				ReadRC(w, newFace, newDirection, newH, k,
				     (k % 2) ? sizeOnDepthAxis : sizeOnAxis);
			RotateRC(w, rotate, k - 2,
				 (k % 2) ? sizeOnDepthAxis : sizeOnAxis);
			if (k != MAX_ORIENT + 1)
				RotateRC(w, rotate, k - 1,
				     (k % 2) ? sizeOnAxis : sizeOnDepthAxis);
			if (reverse == True) {
				ReverseRC(w, k - 2,
				     (k % 2) ? sizeOnDepthAxis : sizeOnAxis);
				if (k != MAX_ORIENT + 1)
					ReverseRC(w, k - 1,
						(k % 2) ? sizeOnAxis : sizeOnDepthAxis);
			}
			WriteRC(w, newFace, newDirection, newH, k - 2,
				(k % 2) ? sizeOnDepthAxis : sizeOnAxis);
			face = newFace;
			direction = newDirection;
			h = newH;
		}
	} else {
		for (k = 1; k <= MAX_ORIENT; k++) {
			SlideRC(w, face, direction, h, sizeOnOppAxis,
			  &newFace, &newDirection, &newH, &rotate, &reverse);
			if (k != MAX_ORIENT)
				ReadRC(w, newFace, newDirection, newH, k, sizeOnAxis);
			RotateRC(w, rotate, k - 1, sizeOnAxis);
			if (reverse == TRUE)
				ReverseRC(w, k - 1, sizeOnAxis);
			WriteRC(w, newFace, newDirection, newH, k - 1, sizeOnAxis);
			face = newFace;
			direction = newDirection;
			h = newH;
		}
	}
#ifdef HAVE_OPENGL
	if (!control && fast == INSTANT && w->rubik.dim == 4) {
		DrawAllPiecesGL((RubikGLWidget) w);
	}
#endif
#ifdef DEBUG
	PrintCube(w);
#endif
}

static Boolean
MoveAllPieces(RubikWidget w, const int face, const int position,
		const int direction, const Boolean control, int fast)
{
	int sizeOfRow, sizeOfColumn, sizeOnOppAxis, factor;
	Boolean newControl;

	faceSizes(w, face, &sizeOfRow, &sizeOfColumn);
	if (direction == TOP || direction == BOTTOM) {
		sizeOnOppAxis = sizeOfRow;
		factor = 1;
	} else {
		sizeOnOppAxis = sizeOfColumn;
		factor = sizeOfRow;
	}
	w->rubik.degreeTurn = (checkFaceSquare(w,
				rowToRotate[face][direction])) ? 90 : 180;
	if (control || sizeOnOppAxis == 1) {
		int k, newPosition, newFast;

		newControl = True;
		newFast = fast;

#ifdef HAVE_OPENGL
		if (newControl && w->rubik.dim == 4) {
			MovePiecesGL((RubikGLWidget) w, face, position,
				direction, newControl, fast);
			newFast = INSTANT;
		}
#endif
		for (k = 0; k < sizeOnOppAxis; k++) {
			newPosition = k * factor;
			MovePieces(w, face, newPosition, direction,
				newControl, newFast);
		}
	} else {
		newControl = False;
		MovePieces(w, face, position, direction, control, fast);
	}
	return newControl;
}

void
MovePuzzle(RubikWidget w, const int face, const int position,
		const int direction, const Boolean control, int fast)
{
	setPuzzleMove(w, ACTION_MOVED, face, position, direction,
		MoveAllPieces(w, face, position, direction, control, fast),
		fast);
#ifdef USE_SOUND
	if (w->rubik.sound) {
		playSound((char *) MOVESOUND);
	}
#endif
	setMove(&undo, face, position, direction, control);
	flushMoves(w, &redo, FALSE);
}

void
MovePuzzleDelay(RubikWidget w, const int face, const int position,
		const int direction, const Boolean control)
{
	MovePuzzle(w, face, position, direction, control, NORMAL);
	Sleep((unsigned int) w->rubik.delay);
}

static Boolean
SelectPieces(RubikWidget w, const int x, const int y, int *face, int *position)
{
	if (w->rubik.dim == 2)
		return SelectPieces2D((Rubik2DWidget) w, x, y,
					face, position);
	else if (w->rubik.dim == 3)
		return SelectPieces3D((Rubik3DWidget) w, x, y,
					face, position);
	return False;
}

static int
CheckMoveDir(RubikWidget w, int face, int position1, int position2,
		int *direction)
{
	int count = 0;
	int column1, column2, row1, row2, sizeOfRow;

	sizeOfRow = sizeRow(w, face);
	column1 = position1 % sizeOfRow;
	column2 = position2 % sizeOfRow;
	row1 = position1 / sizeOfRow;
	row2 = position2 / sizeOfRow;
	if (column1 == column2 && row1 != row2) {
		*direction = (row2 > row1) ? BOTTOM : TOP;
		count = 1;
	} else if (row1 == row2 && column1 != column2) {
		*direction = (column2 > column1) ? RIGHT : LEFT;
		count = 1;
	} else if (row1 == row2 && column1 == column2)
		count = 2;
	return count;
}


static Boolean
NarrowSelection(RubikWidget w, int *face, int *position, int *direction)
{
	if (w->rubik.dim == 2)
		return NarrowSelection2D((Rubik2DWidget) w, face, position, direction);
	else if (w->rubik.dim == 3)
		return NarrowSelection3D((Rubik3DWidget) w, face, position, direction);
	return False;
}

static Boolean
PositionPieces(RubikWidget w, int x, int y, int *face, int *position, int *direction)
{
	if (!SelectPieces(w, x, y, face, position))
		return False;
	return NarrowSelection(w, face, position, direction);
}

void
MovePuzzleInput(RubikWidget w, int x, int y, int direction, int control)
{
	int face, position;

	if (!w->rubik.practice && !control && CheckSolved(w, w->rubik.orient)) {
		MoveNoPieces(w);
		return;
	}
	if (!PositionPieces(w, x, y, &face, &position, &direction))
		return;
	control = (control) ? 1 : 0;
	MovePuzzle(w, face, position, direction, control, NORMAL);
	if (!control && CheckSolved(w, w->rubik.orient)) {
		setPuzzle(w, ACTION_SOLVED);
	}
}

void
ResetPieces(RubikWidget w)
{
	int face, position, orient, sizeOfFace;

	for (face = 0; face < MAX_FACES; face++) {
		sizeOfFace = sizeFace(w, face);
		if (w->rubik.cubeLoc[face])
			free(w->rubik.cubeLoc[face]);
		if (!(w->rubik.cubeLoc[face] = (RubikLoc *)
				malloc(sizeof (RubikLoc) * sizeOfFace))) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
		if (startLoc[face])
			free(startLoc[face]);
		if (!(startLoc[face] = (RubikLoc *)
				malloc(sizeof (RubikLoc) * sizeOfFace))) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
		for (position = 0; position < sizeOfFace; position++) {
			w->rubik.cubeLoc[face][position].face = face;
			w->rubik.cubeLoc[face][position].rotation = STRT - MAX_ORIENT;
		}
	}
	for (orient = 0; orient < MAX_ORIENT; orient++) {
		if (w->rubik.rowLoc[orient])
			free(w->rubik.rowLoc[orient]);
		if (!(w->rubik.rowLoc[orient] = (RubikLoc *)
				malloc(sizeof (RubikLoc) * MAX_MAX_SIZE))) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
	}
	flushMoves(w, &undo, TRUE);
	flushMoves(w, &redo, FALSE);
	w->rubik.started = False;
	w->rubik.currentFace = IGNORE_DIR;
	w->rubik.currentDirection = IGNORE_DIR;
}

static void
GetPieces(RubikWidget w)
{
	FILE *fp;
	int c, i, sizex, sizey, sizez, orient, practice, moves;
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, LOGFILE);
	lname = buf1;
	stringCat(&buf1, LOGPATH, FINALDELIM);
	stringCat(&buf2, buf1, LOGFILE);
	free(buf1);
	fname = buf2;
	/* Try current directory first. */
	name = lname;
	if ((fp = fopen(name, "r")) == NULL) {
		name = fname;
		if ((fp = fopen(name, "r")) == NULL) {
			stringCat(&buf1, "Can not read (get) ", lname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not read (get) ", lname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
		}
#endif
	}
	flushMoves(w, &undo, TRUE);
	flushMoves(w, &redo, FALSE);
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &sizex);
	if (sizex >= MIN_FACETS) {
		for (i = w->rubik.sizex; i < sizex; i++) {
			setPuzzle(w, ACTION_INCX);
		}
		for (i = w->rubik.sizex; i > sizex; i--) {
			setPuzzle(w, ACTION_DECX);
		}
	} else {
		stringCat(&buf1, name, " corrupted: sizex ");
		intCat(&buf2, buf1, sizex);
		free(buf1);
		stringCat(&buf1, buf2, " should be between ");
		free(buf2);
		intCat(&buf2, buf1, MIN_FACETS);
		free(buf1);
		stringCat(&buf1, buf2, " and MAXINT");
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
	}
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &sizey);
	if (sizey >= MIN_FACETS) {
		for (i = w->rubik.sizey; i < sizey; i++) {
			setPuzzle(w, ACTION_INCY);
		}
		for (i = w->rubik.sizey; i > sizey; i--) {
			setPuzzle(w, ACTION_DECY);
		}
	} else {
		stringCat(&buf1, name, " corrupted: sizey ");
		intCat(&buf2, buf1, sizey);
		free(buf1);
		stringCat(&buf1, buf2, " should be between ");
		free(buf2);
		intCat(&buf2, buf1, MIN_FACETS);
		free(buf1);
		stringCat(&buf1, buf2, " and MAXINT");
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
	}
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &sizez);
	if (sizez >= MIN_FACETS) {
		for (i = w->rubik.sizez; i < sizez; i++) {
			setPuzzle(w, ACTION_INCZ);
		}
		for (i = w->rubik.sizez; i > sizez; i--) {
			setPuzzle(w, ACTION_DECZ);
		}
	} else {
		stringCat(&buf1, name, " corrupted: sizez ");
		intCat(&buf2, buf1, sizez);
		free(buf1);
		stringCat(&buf1, buf2, " should be between ");
		free(buf2);
		intCat(&buf2, buf1, MIN_FACETS);
		free(buf1);
		stringCat(&buf1, buf2, " and MAXINT");
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
	}
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &orient);
	if (w->rubik.orient != (Boolean) orient) {
		setPuzzle(w, ACTION_ORIENTIZE);
	}
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &practice);
	if (w->rubik.practice != (Boolean) practice) {
		setPuzzle(w, ACTION_PRACTICE);
	}
#ifdef WINVER
	ResetPieces(w);
#endif
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &moves);
	scanStartPosition(fp, w);
	setPuzzle(w, ACTION_RESTORE);
#ifdef WINVER
	setStartPosition(w);
#endif
	scanMoves(fp, w, moves);
	(void) fclose(fp);
	(void) printf("%s: sizex %d, sizey %d, sizez %d, orient %s, practice %s, moves %d.\n",
		name, sizex, sizey, sizez,
		BOOL_STRING(orient), BOOL_STRING(practice), moves);
	free(lname);
	free(fname);
	w->rubik.cheat = True; /* Assume the worst. */
}

static void
WritePieces(RubikWidget w)
{
	FILE *fp;
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, LOGFILE);
	lname = buf1;
	stringCat(&buf1, LOGPATH, FINALDELIM);
	stringCat(&buf2, buf1, LOGFILE);
	free(buf1);
	fname = buf2;
	/* Try current directory first. */
	name = lname;
	if ((fp = fopen(name, "w")) == NULL) {
		name = fname;
		if ((fp = fopen(name, "w")) == NULL) {
			stringCat(&buf1, "Can not write to ", lname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not write to ", lname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
		}
#endif
	}
	(void) fprintf(fp, "sizex%c %d\n", SYMBOL, w->rubik.sizex);
	(void) fprintf(fp, "sizey%c %d\n", SYMBOL, w->rubik.sizey);
	(void) fprintf(fp, "sizez%c %d\n", SYMBOL, w->rubik.sizez);
	(void) fprintf(fp, "orient%c %d\n", SYMBOL,
		(w->rubik.orient) ? 1 : 0);
	(void) fprintf(fp, "practice%c %d\n", SYMBOL,
		(w->rubik.practice) ? 1 : 0);
	(void) fprintf(fp, "moves%c %d\n", SYMBOL,
		numMoves(&undo));
	printStartPosition(fp, w);
	printMoves(fp, &undo);
	(void) fclose(fp);
	(void) printf("Saved to %s.\n", name);
	free(lname);
	free(fname);
}

static void
UndoPieces(RubikWidget w)
{
	if (madeMoves(&undo)) {
		int face, position, direction, control;

		getMove(&undo,
			&face, &position, &direction, &control);
		setMove(&redo,
			face, position, direction, control);
		direction = (direction < MAX_ORIENT) ?
			(direction + MAX_ORIENT / 2) % MAX_ORIENT :
			3 * MAX_ORIENT - direction;
		setPuzzleMove(w, ACTION_UNDO, face, position, direction,
			MoveAllPieces(w, face, position, direction, control,
			DOUBLE), DOUBLE);
		if (!control && CheckSolved(w, w->rubik.orient)) {
			setPuzzle(w, ACTION_SOLVED);
		}
	}
}

static void
RedoPieces(RubikWidget w)
{
	if (madeMoves(&redo)) {
		int face, position, direction, control;

		getMove(&redo,
			&face, &position, &direction, &control);
		setMove(&undo,
			face, position, direction, control);
		setPuzzleMove(w, ACTION_REDO, face, position, direction,
			MoveAllPieces(w, face, position, direction, control,
			DOUBLE), DOUBLE);
		if (!control && CheckSolved(w, w->rubik.orient)) {
			setPuzzle(w, ACTION_SOLVED);
		}
	}
}

static void
ClearPieces(RubikWidget w)
{
	setPuzzle(w, ACTION_CLEAR);
}

static void
PracticePieces(RubikWidget w)
{
	setPuzzle(w, ACTION_PRACTICE);
}

static void
RandomizePieces(RubikWidget w)
{
	int face, position, direction;
	int big = w->rubik.sizex * w->rubik.sizey * w->rubik.sizez *
			3 + NRAND(2);

	w->rubik.cheat = False;
	if (big > 100)
		big = 100;
	if (w->rubik.practice)
		PracticePieces(w);
	setPuzzle(w, ACTION_RESET);
#ifdef DEBUG
	big = 3;
#endif
	while (big--) {
		face = NRAND(MAX_FACES);
		position = NRAND(sizeFace(w, face));
		direction = NRAND(MAX_ORIENT);
		MovePuzzle(w, face, position, direction, FALSE, INSTANT);
	}
	flushMoves(w, &undo, TRUE);
	flushMoves(w, &redo, FALSE);
	setPuzzle(w, ACTION_RANDOMIZE);
	if (CheckSolved(w, w->rubik.orient)) {
		setPuzzle(w, ACTION_SOLVED);
	}
}

static void
SolvePieces(RubikWidget w)
{
	if (CheckSolved(w, w->rubik.orient))
		return;
	if ((w->rubik.sizex == 2 && w->rubik.sizey == 2 && w->rubik.sizez == 2) ||
	    (w->rubik.sizex == 3 && w->rubik.sizey == 3 && w->rubik.sizez == 3))
		SolveSomePieces(w);
	else {
		setPuzzle(w, ACTION_SOLVE_MESSAGE);
	}
}

static void
OrientizePieces(RubikWidget w)
{
	setPuzzle(w, ACTION_ORIENTIZE);
}

static void
IncrementPieces(RubikWidget w)
{
	setPuzzle(w, ACTION_INCX);
	setPuzzle(w, ACTION_INCY);
	setPuzzle(w, ACTION_INCZ);
}

static void
DecrementPieces(RubikWidget w)
{
	if (w->rubik.sizex > MIN_FACETS) {
		setPuzzle(w, ACTION_DECX);
	}
	if (w->rubik.sizey > MIN_FACETS) {
		setPuzzle(w, ACTION_DECY);
	}
	if (w->rubik.sizez > MIN_FACETS) {
		setPuzzle(w, ACTION_DECZ);
	}
}

static void
ViewPieces(RubikWidget w)
{
	setPuzzle(w, ACTION_VIEW);
}

static void
SpeedPieces(RubikWidget w)
{
	w->rubik.delay -= 5;
	if (w->rubik.delay < 0)
		w->rubik.delay = 0;
#ifdef HAVE_MOTIF
	setPuzzle(w, ACTION_SPEED);
#endif
}

static void
SlowPieces(RubikWidget w)
{
	w->rubik.delay += 5;
#ifdef HAVE_MOTIF
	setPuzzle(w, ACTION_SPEED);
#endif
}

static void
SoundPieces(RubikWidget w)
{
	w->rubik.sound = !w->rubik.sound;
}

#ifdef WINVER
static void
SetValuesPuzzle(RubikWidget w)
{
	struct tagColor {
		int red, green, blue;
	} color;
	char szBuf[80], buf[20], charbuf[2];
	int face;

	w->rubik.sizex = GetPrivateProfileInt(SECTION, "sizeX",
		DEFAULTFACETS, INIFILE);
	w->rubik.sizey = GetPrivateProfileInt(SECTION, "sizeY",
		DEFAULTFACETS, INIFILE);
	w->rubik.sizez = GetPrivateProfileInt(SECTION, "sizeZ",
		DEFAULTFACETS, INIFILE);
	w->rubik.orient = (BOOL) GetPrivateProfileInt(SECTION, "orient",
		DEFAULT_ORIENT, INIFILE);
	w->rubik.practice = (BOOL) GetPrivateProfileInt(SECTION, "practice",
		DEFAULT_PRACTICE, INIFILE);
	w->rubik.dim = GetPrivateProfileInt(SECTION, "dim", 3, INIFILE);
	w->rubik.mono = (BOOL) GetPrivateProfileInt(SECTION, "mono",
		DEFAULT_MONO, INIFILE);
	w->rubik.reverse = (BOOL) GetPrivateProfileInt(SECTION, "reverseVideo",
		DEFAULT_REVERSE, INIFILE);
	/* cyan */
	(void) GetPrivateProfileString(SECTION, "frameColor", "0 255 255",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->rubik.frameGC = RGB(color.red, color.green, color.blue);
	/* gray25 */
	(void) GetPrivateProfileString(SECTION, "pieceBorder", "64 64 64",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->rubik.borderGC = RGB(color.red, color.green, color.blue);
	/* #AEB2C3 */
	(void) GetPrivateProfileString(SECTION, "background", "174 178 195",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->rubik.inverseGC = RGB(color.red, color.green, color.blue);
	for (face = 0; face < MAX_FACES; face++) {
		(void) sprintf(buf, "faceColor%d", face);
		(void) GetPrivateProfileString(SECTION, buf,
			faceColorString[face],
			szBuf, sizeof (szBuf), INIFILE);
		(void) sscanf(szBuf, "%d %d %d",
			&(color.red), &(color.green), &(color.blue));
		w->rubik.faceGC[face] =
			RGB(color.red, color.green, color.blue);
		(void) sprintf(buf, "faceChar%d", face);
		charbuf[0] = faceColorChar[face];
		charbuf[1] = '\0';
		(void) GetPrivateProfileString(SECTION, buf, charbuf,
			szBuf, sizeof (szBuf), INIFILE);
		w->rubik.faceChar[face] = szBuf[0];
	}
	(void) GetPrivateProfileString(SECTION, "userName", "Guest",
		szBuf, sizeof (szBuf), INIFILE);
	(void) strcpy(w->rubik.userName, szBuf);
		w->rubik.userName[80] = 0;
	(void) GetPrivateProfileString(SECTION, "scoreFile", "",
		szBuf, sizeof (szBuf), INIFILE);
	(void) strcpy(w->rubik.scoreFile, szBuf);
		w->rubik.scoreFile[80] = 0;
}

void
DestroyPuzzle(HBRUSH brush)
{
	deleteMoves(&undo);
	deleteMoves(&redo);
	(void) DeleteObject(brush);
	PostQuitMessage(0);
}

void
ResizePuzzle(RubikWidget w)
{
	if (w->rubik.dim == 2)
		ResizePuzzle2D((Rubik2DWidget) w);
	else if (w->rubik.dim == 3)
		ResizePuzzle3D((Rubik3DWidget) w);
#ifdef HAVE_OPENGL
	else if (w->rubik.dim == 4)
		ResizePuzzleGL((RubikGLWidget) w);
#endif
}

void
SizePuzzle(RubikWidget w)
{
	ResetPieces(w);
	ResizePuzzle(w);
}

void
ExposePuzzle(RubikWidget w)
{
	if (w->rubik.dim == 2)
		ExposePuzzle2D((Rubik2DWidget) w);
	else if (w->rubik.dim == 3)
		ExposePuzzle3D((Rubik3DWidget) w);
#ifdef HAVE_OPENGL
	else if (w->rubik.dim == 4)
		ExposePuzzleGL((RubikGLWidget) w);
#endif
}

#else
static void
GetColor(RubikWidget w, int face)
{
	XGCValues values;
	XtGCMask valueMask;
	XColor colorCell, rgb;

	valueMask = GCForeground | GCBackground;
	if (w->rubik.reverse) {
		values.background = w->rubik.foreground;
	} else {
		values.background = w->rubik.background;
	}
	if (!w->rubik.mono || w->rubik.dim == 4) {
		if (!w->rubik.faceName[face]) {
			char *buf1;

			intCat(&buf1, "Color name null for face ", face);
			DISPLAY_WARNING(buf1);
			free(buf1);
		} else if (XAllocNamedColor(XtDisplay(w),
				DefaultColormapOfScreen(XtScreen(w)),
				w->rubik.faceName[face], &colorCell, &rgb)) {
			values.foreground = w->rubik.faceColor[face] =
				colorCell.pixel;
			if (w->rubik.faceGC[face])
				XtReleaseGC((Widget) w, w->rubik.faceGC[face]);
			w->rubik.faceGC[face] = XtGetGC((Widget) w, valueMask,
				&values);
			return;
		} else {
			char *buf1, *buf2;

			stringCat(&buf1, "Color name \"",
				w->rubik.faceName[face]);
			stringCat(&buf2, buf1, "\" is not defined for face ");
			free(buf1);
			intCat(&buf1, buf2, face);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
		}
	}
	if (w->rubik.reverse) {
		values.background = w->rubik.foreground;
		values.foreground = w->rubik.background;
	} else {
		values.background = w->rubik.background;
		values.foreground = w->rubik.foreground;
	}
	if (w->rubik.faceGC[face])
		XtReleaseGC((Widget) w, w->rubik.faceGC[face]);
	w->rubik.faceGC[face] = XtGetGC((Widget) w, valueMask, &values);
}

void
SetAllColors(RubikWidget w)
{
	XGCValues values;
	XtGCMask valueMask;
	int face;

	valueMask = GCForeground | GCBackground;

	if (w->rubik.reverse) {
		values.background = w->rubik.background;
		values.foreground = w->rubik.foreground;
	} else {
		values.foreground = w->rubik.background;
		values.background = w->rubik.foreground;
	}
	if (w->rubik.inverseGC)
		XtReleaseGC((Widget) w, w->rubik.inverseGC);
	w->rubik.inverseGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->rubik.mono) {
		if (w->rubik.reverse) {
			values.background = w->rubik.foreground;
			values.foreground = w->rubik.background;
		} else {
			values.foreground = w->rubik.foreground;
			values.background = w->rubik.background;
		}
	} else {
		values.foreground = w->rubik.frameColor;
		values.background = w->rubik.background;
	}
	if (w->rubik.frameGC)
		XtReleaseGC((Widget) w, w->rubik.frameGC);
	w->rubik.frameGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->rubik.mono) {
		if (w->rubik.reverse) {
			values.background = w->rubik.foreground;
			values.foreground = w->rubik.background;
		} else {
			values.foreground = w->rubik.foreground;
			values.background = w->rubik.background;
		}
	} else {
		values.foreground = w->rubik.borderColor;
		values.background = w->rubik.background;
	}
	if (w->rubik.borderGC)
		XtReleaseGC((Widget) w, w->rubik.borderGC);
	w->rubik.borderGC = XtGetGC((Widget) w, valueMask, &values);
	for (face = 0; face < MAX_FACES; face++)
		GetColor(w, face);
	if (w->rubik.fontInfo)
		XSetFont(XtDisplay(w), w->rubik.borderGC,
			w->rubik.fontInfo->fid);
}

static Boolean
SetValuesPuzzle(Widget current, Widget request, Widget renew)
{
	RubikWidget c = (RubikWidget) current, w = (RubikWidget) renew;
	Boolean redraw = False, setColors = False;
	int face;

	CheckPieces(w);
	for (face = 0; face < MAX_FACES; face++) {
		if (strcmp(w->rubik.faceName[face], c->rubik.faceName[face])) {
			setColors = True;
			break;
		}
	}
	if (w->rubik.font != c->rubik.font ||
			w->rubik.borderColor != c->rubik.borderColor ||
			w->rubik.reverse != c->rubik.reverse ||
			w->rubik.mono != c->rubik.mono) {
		loadFont(w);
		SetAllColors(w);
		redraw = True;
	} else if (w->rubik.background != c->rubik.background ||
			w->rubik.foreground != c->rubik.foreground ||
			setColors) {
		SetAllColors(w);
		redraw = True;
	}
	if (w->rubik.orient != c->rubik.orient) {
		ResetPieces(w);
		redraw = True;
	} else if (w->rubik.practice != c->rubik.practice) {
		ResetPieces(w);
		redraw = True;
	}
	if (w->rubik.menu != ACTION_IGNORE) {
		int menu = w->rubik.menu;

		w->rubik.menu = ACTION_IGNORE;
		switch (menu) {
		case ACTION_GET:
			GetPieces(w);
			break;
		case ACTION_WRITE:
			WritePieces(w);
			break;
		case ACTION_UNDO:
			UndoPieces(w);
			break;
		case ACTION_REDO:
			RedoPieces(w);
			break;
		case ACTION_CLEAR:
			ClearPieces(w);
			break;
		case ACTION_PRACTICE:
			PracticePieces(w);
			break;
		case ACTION_RANDOMIZE:
			RandomizePieces(w);
			break;
		case ACTION_SOLVE:
			SolvePieces(w);
			break;
		case ACTION_INCREMENT:
			IncrementPieces(w);
			break;
		case ACTION_DECREMENT:
			(void) DecrementPieces(w);
			break;
		case ACTION_ORIENTIZE:
			OrientizePieces(w);
			break;
		case ACTION_SPEED:
			SpeedPieces(w);
			break;
		case ACTION_SLOW:
			SlowPieces(w);
			break;
		case ACTION_SOUND:
			SoundPieces(w);
			break;
		case ACTION_VIEW:
			ViewPieces(w);
			break;
		default:
			break;
		}
	}
	if (w->rubik.currentDirection == RESTORE_DIR) {
		setStartPosition(w);
		w->rubik.currentDirection = IGNORE_DIR;
	} else if (w->rubik.currentDirection == CLEAR_DIR) {
		ResetPieces(w);
		redraw = True;
		w->rubik.currentDirection = IGNORE_DIR;
	} else if (w->rubik.currentDirection > IGNORE_DIR) {
		(void) MoveAllPieces(w, w->rubik.currentFace,
			w->rubik.currentPosition, w->rubik.currentDirection,
			w->rubik.currentControl, w->rubik.currentFast);
		w->rubik.currentDirection = IGNORE_DIR;
	}
	return redraw;
}

static void
DestroyPuzzle(Widget old)
{
	RubikWidget w = (RubikWidget) old;
	Display *display = XtDisplay(w);
	int face;

	for (face = 0; face < MAX_FACES; face++)
		XtReleaseGC(old, w->rubik.faceGC[face]);
	XtReleaseGC(old, w->rubik.borderGC);
	XtReleaseGC(old, w->rubik.frameGC);
	XtReleaseGC(old, w->rubik.inverseGC);
	XtRemoveCallbacks(old, XtNselectCallback, w->rubik.select);
	if (w->rubik.fontInfo) {
		XUnloadFont(display, w->rubik.fontInfo->fid);
		XFreeFont(display, w->rubik.fontInfo);
	}
	deleteMoves(&undo);
	deleteMoves(&redo);
}

void
QuitPuzzle(RubikWidget w, XEvent *event, char **args, int nArgs)
{
	XtCloseDisplay(XtDisplay(w));
	exit(0);
}
#endif

#ifndef WINVER
static
#endif
void
InitializePuzzle(
#ifdef WINVER
RubikWidget w, HBRUSH brush
#else
Widget request, Widget renew
#endif
)
{
	int face, orient;
#ifdef WINVER
	SetValuesPuzzle(w);
#else
	RubikWidget w = (RubikWidget) renew;

	w->rubik.mono = (DefaultDepthOfScreen(XtScreen(w)) < 2 ||
		w->rubik.mono);
	w->rubik.fontInfo = NULL;
	for (face = 0; face < MAX_FACES; face++)
		w->rubik.faceGC[face] = NULL;
	w->rubik.borderGC = NULL;
	w->rubik.frameGC = NULL;
	w->rubik.inverseGC = NULL;
#endif
	w->rubik.focus = False;
	loadFont(w);
	for (face = 0; face < MAX_FACES; face++)
		w->rubik.cubeLoc[face] = NULL;
	for (orient = 0; orient < MAX_ORIENT; orient++)
		w->rubik.rowLoc[orient] = NULL;
	CheckPieces(w);
	newMoves(&undo);
	newMoves(&redo);
	w->rubik.cheat = False;
	ResetPieces(w);
#ifdef WINVER
	brush = CreateSolidBrush(w->rubik.inverseGC);
	SETBACK(w->core.hWnd, brush);
	(void) SRAND(time(NULL));
#else
	(void) SRAND(getpid());
#endif
}

void
HidePuzzle(RubikWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	setPuzzle(w, ACTION_HIDE);
}

void
SelectPuzzle(RubikWidget w
#ifdef WINVER
, const int x, const int y, const int control
#else
, XEvent *event, char **args, int nArgs
#endif
)
{
#ifndef WINVER
	int x = event->xbutton.x, y = event->xbutton.y;
	int control = (int) (event->xkey.state & ControlMask);
#endif

	if (SelectPieces(w, x, y,
		     &(w->rubik.currentFace), &(w->rubik.currentPosition))) {
		if (control || w->rubik.practice || !CheckSolved(w, w->rubik.orient))
			DrawSquare(w, w->rubik.currentFace, w->rubik.currentPosition,
				TRUE);
	} else {
		w->rubik.currentFace = IGNORE_DIR;
		w->rubik.currentDirection = IGNORE_DIR;
	}
}

void
ReleasePuzzle(RubikWidget w
#ifdef WINVER
, const int x, const int y, const int control
#else
, XEvent *event, char **args, int nArgs
#endif
)
{
	int face, position, count = -1, direction = -1;
#ifndef WINVER
	int x = event->xbutton.x, y = event->xbutton.y;
	int control = (int) (event->xkey.state & ControlMask);
#endif

	if (w->rubik.currentFace <= IGNORE_DIR)
		return;
	DrawSquare(w, w->rubik.currentFace, w->rubik.currentPosition, FALSE);
	if (!control && !w->rubik.practice && CheckSolved(w, w->rubik.orient))
		MoveNoPieces(w);
	else if (SelectPieces(w, x, y, &face, &position)) {
		if (face == w->rubik.currentFace) {
			if (position != w->rubik.currentPosition)
				count = CheckMoveDir(w, face,
					w->rubik.currentPosition, position,
					&direction);
		} else {
			direction = facesToDirection[w->rubik.currentFace][face];
			count = (direction == AMBIGUOUS_DIR) ? 2 : 1;
			/* can look closer to minimize this, but then
			must also be able to handle double moves */
		}
		if (count == 1) {
			MovePuzzle(w, face, w->rubik.currentPosition,
				direction, (control) ? 1 : 0, NORMAL);
			if (!control && CheckSolved(w, w->rubik.orient)) {
				setPuzzle(w, ACTION_SOLVED);
			}
		} else if (count == 2) {
			setPuzzle(w, ACTION_AMBIGUOUS);
		} else if (count == 0)
			MoveNoPieces(w);
	}
	w->rubik.currentFace = IGNORE_DIR;
	w->rubik.currentDirection = IGNORE_DIR;
}

#ifndef WINVER
void
PracticePuzzleMaybe(RubikWidget w
, XEvent *event, char **args, int nArgs
)
{
	if (!w->rubik.started)
		PracticePieces(w);
#ifdef HAVE_MOTIF
	else {
		setPuzzle(w, ACTION_PRACTICE_QUERY);
	}
#endif
}

void
PracticePuzzle2(RubikWidget w
, XEvent *event, char **args, int nArgs
)
{
#ifdef HAVE_MOTIF
	if (!w->rubik.started)
#endif
		PracticePieces(w);
}

void
RandomizePuzzleMaybe(RubikWidget w
, XEvent *event, char **args, int nArgs
)
{
	if (!w->rubik.started)
		RandomizePieces(w);
#ifdef HAVE_MOTIF
	else {
		setPuzzle(w, ACTION_RANDOMIZE_QUERY);
	}
#endif
}

void
RandomizePuzzle2(RubikWidget w
, XEvent *event, char **args, int nArgs
)
{
#ifdef HAVE_MOTIF
	if (!w->rubik.started)
#endif
		RandomizePieces(w);
}
#endif

void
GetPuzzle(RubikWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	GetPieces(w);
}

void
WritePuzzle(RubikWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	WritePieces(w);
}

void
UndoPuzzle(RubikWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	UndoPieces(w);
}

void
RedoPuzzle(RubikWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	RedoPieces(w);
}

void
ClearPuzzle(RubikWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	ClearPieces(w);
}

void
RandomizePuzzle(RubikWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	RandomizePieces(w);
}

void
SolvePuzzle(RubikWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	SolvePieces(w);
}

void
PracticePuzzle(RubikWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	PracticePieces(w);
}

void
OrientizePuzzle(RubikWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	OrientizePieces(w);
}

void
DecrementPuzzle(RubikWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	DecrementPieces(w);
}

void
IncrementPuzzle(RubikWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	IncrementPieces(w);
}

void
ViewPuzzle(RubikWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	ViewPieces(w);
}

void
SpeedPuzzle(RubikWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	SpeedPieces(w);
}

void
SlowPuzzle(RubikWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	SlowPieces(w);
}

void
SoundPuzzle(RubikWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	SoundPieces(w);
}

void
EnterPuzzle(RubikWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	w->rubik.focus = True;
	DrawFrame(w, w->rubik.focus);
}

void
LeavePuzzle(RubikWidget w
#ifndef WINVER
, XEvent *event, char **args, int nArgs
#endif
)
{
	w->rubik.focus = False;
	DrawFrame(w, w->rubik.focus);
}

#ifdef WINVER
void
IncrementAxisPuzzle(RubikWidget w, char axis)
{
	switch (axis) {
	case 'X':
		if (w->rubik.sizex > MIN_FACETS)
			setPuzzle(w, ACTION_DECX);
		return;
	case 'x':
		setPuzzle(w, ACTION_INCX);
		return;
	case 'Y':
		if (w->rubik.sizey > MIN_FACETS)
			setPuzzle(w, ACTION_DECY);
		return;
	case 'y':
		setPuzzle(w, ACTION_INCY);
		return;
	case 'Z':
		if (w->rubik.sizez > MIN_FACETS)
			setPuzzle(w, ACTION_DECZ);
		return;
	case 'z':
		setPuzzle(w, ACTION_INCZ);
		return;
	}
	return;
}

void
DimPuzzle(RubikWidget w)
{
	setPuzzle(w, ACTION_DIM);
}

#else
void
IncrementXPuzzle(RubikWidget w, XEvent *event, char **args, int nArgs)
{
	if (event->xbutton.state & (ShiftMask | LockMask)) {
		if (w->rubik.sizex <= MIN_FACETS)
			return;
		setPuzzle(w, ACTION_DECX);
	} else
		setPuzzle(w, ACTION_INCX);
}

void
IncrementYPuzzle(RubikWidget w , XEvent *event, char **args, int nArgs)
{
	if (event->xbutton.state & (ShiftMask | LockMask)) {
		if (w->rubik.sizey <= MIN_FACETS)
			return;
		setPuzzle(w, ACTION_DECY);
	} else
		setPuzzle(w, ACTION_INCY);
}

void
IncrementZPuzzle(RubikWidget w , XEvent *event, char **args, int nArgs)
{
	if (event->xbutton.state & (ShiftMask | LockMask)) {
		if (w->rubik.sizez <= MIN_FACETS)
			return;
		setPuzzle(w, ACTION_DECZ);
	} else
		setPuzzle(w, ACTION_INCZ);
}

void
MovePuzzleCcw(RubikWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput(w, event->xbutton.x, event->xbutton.y, CCW,
		(int) (event->xbutton.state & ControlMask));
}

void
MovePuzzleCw(RubikWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput(w, event->xbutton.x, event->xbutton.y, CW,
		(int) (event->xkey.state & ControlMask));
}
#endif
