/*-
# X-BASED DINOSAUR CUBE
#
#  Dino3d.c
#
###
#
#  Copyright (c) 1995 - 2007	David A. Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/* Methods file for Dino3d */

#include "DinoP.h"
#include "Dino3dP.h"

#ifndef WINVER
static Boolean SetValuesPuzzle3D(Widget current, Widget request, Widget renew);
static void ResizePuzzle3D(Dino3DWidget w);
static void InitializePuzzle3D(Widget request, Widget renew);
static void ExposePuzzle3D(Widget renew, XEvent *event, Region region);
static void MovePuzzle3DCcw(Dino3DWidget w,
	XEvent *event, char **args, int nArgs);
static void MovePuzzle3DTl(Dino3DWidget w,
	XEvent *event, char **args, int nArgs);
static void MovePuzzle3DTop(Dino3DWidget w,
	XEvent *event, char **args, int nArgs);
static void MovePuzzle3DTr(Dino3DWidget w,
	XEvent *event, char **args, int nArgs);
static void MovePuzzle3DLeft(Dino3DWidget w,
	XEvent *event, char **args, int nArgs);
static void MovePuzzle3DCw(Dino3DWidget w,
	XEvent *event, char **args, int nArgs);
static void MovePuzzle3DRight(Dino3DWidget w,
	XEvent *event, char **args, int nArgs);
static void MovePuzzle3DBl(Dino3DWidget w,
	XEvent *event, char **args, int nArgs);
static void MovePuzzle3DBottom(Dino3DWidget w,
	XEvent *event, char **args, int nArgs);
static void MovePuzzle3DBr(Dino3DWidget w,
	XEvent *event, char **args, int nArgs);

static char defaultTranslationsPuzzle3D[] =
"<KeyPress>q: Quit()\n\
 Ctrl<KeyPress>C: Quit()\n\
 <KeyPress>osfCancel: Hide()\n\
 <KeyPress>Escape: Hide()\n\
 <KeyPress>osfEscape: Hide()\n\
 Ctrl<KeyPress>[: Hide()\n\
 <KeyPress>0x1B: Hide()\n\
 <KeyPress>0x2E: Speed()\n\
 <KeyPress>0x3E: Speed()\n\
 <KeyPress>0x3C: Slow()\n\
 <KeyPress>0x2C: Slow()\n\
 Shift<KeyPress>2: Sound()\n\
 <KeyPress>KP_Divide: MoveCcw()\n\
 <KeyPress>R5: MoveCcw()\n\
 <KeyPress>Home: MoveTl()\n\
 <KeyPress>KP_7: MoveTl()\n\
 <KeyPress>R7: MoveTl()\n\
 <KeyPress>Up: MoveTop()\n\
 <KeyPress>osfUp: MoveTop()\n\
 <KeyPress>KP_Up: MoveTop()\n\
 <KeyPress>KP_8: MoveTop()\n\
 <KeyPress>R8: MoveTop()\n\
 <KeyPress>Prior: MoveTr()\n\
 <KeyPress>KP_9: MoveTr()\n\
 <KeyPress>R9: MoveTr()\n\
 <KeyPress>Left: MoveLeft()\n\
 <KeyPress>osfLeft: MoveLeft()\n\
 <KeyPress>KP_Left: MoveLeft()\n\
 <KeyPress>KP_4: MoveLeft()\n\
 <KeyPress>R10: MoveLeft()\n\
 <KeyPress>Begin: MoveCw()\n\
 <KeyPress>KP_5: MoveCw()\n\
 <KeyPress>R11: MoveCw()\n\
 <KeyPress>Right: MoveRight()\n\
 <KeyPress>osfRight: MoveRight()\n\
 <KeyPress>KP_Right: MoveRight()\n\
 <KeyPress>KP_6: MoveRight()\n\
 <KeyPress>R12: MoveRight()\n\
 <KeyPress>End: MoveBl()\n\
 <KeyPress>KP_1: MoveBl()\n\
 <KeyPress>R13: MoveBl()\n\
 <KeyPress>Down: MoveBottom()\n\
 <KeyPress>osfDown: MoveBottom()\n\
 <KeyPress>KP_Down: MoveBottom()\n\
 <KeyPress>KP_2: MoveBottom()\n\
 <KeyPress>R14: MoveBottom()\n\
 <KeyPress>Next: MoveBr()\n\
 <KeyPress>KP_3: MoveBr()\n\
 <KeyPress>R15: MoveBr()\n\
 <Btn1Down>: Select()\n\
 <Btn1Up>: Release()\n\
 <Btn2Down>: PracticeMaybe()\n\
 <Btn2Down>(2+): Practice2()\n\
 <Btn3Down>: RandomizeMaybe()\n\
 <Btn3Down>(2+): Randomize2()\n\
 <Btn4Down>: MoveTop()\n\
 <Btn5Down>: MoveBottom()\n\
 <KeyPress>g: Get()\n\
 <KeyPress>w: Write()\n\
 <KeyPress>u: Undo()\n\
 <KeyPress>r: Redo()\n\
 <KeyPress>c: Clear()\n\
 <KeyPress>z: Randomize()\n\
 <KeyPress>s: Solve()\n\
 <KeyPress>p: Practice()\n\
 <KeyPress>o: Orientize()\n\
 <KeyPress>2: Period2()\n\
 <KeyPress>3: Period3()\n\
 <KeyPress>b: Both()\n\
 <KeyPress>v: View()\n\
 <EnterWindow>: Enter()\n\
 <LeaveWindow>: Leave()";

static XtActionsRec actionsListPuzzle3D[] =
{
	{(char *) "Quit", (XtActionProc) QuitPuzzle},
	{(char *) "Hide", (XtActionProc) HidePuzzle},
	{(char *) "MoveCcw", (XtActionProc) MovePuzzle3DCcw},
	{(char *) "MoveTl", (XtActionProc) MovePuzzle3DTl},
	{(char *) "MoveTop", (XtActionProc) MovePuzzle3DTop},
	{(char *) "MoveTr", (XtActionProc) MovePuzzle3DTr},
	{(char *) "MoveLeft", (XtActionProc) MovePuzzle3DLeft},
	{(char *) "MoveCw", (XtActionProc) MovePuzzle3DCw},
	{(char *) "MoveRight", (XtActionProc) MovePuzzle3DRight},
	{(char *) "MoveBl", (XtActionProc) MovePuzzle3DBl},
	{(char *) "MoveBottom", (XtActionProc) MovePuzzle3DBottom},
	{(char *) "MoveBr", (XtActionProc) MovePuzzle3DBr},
	{(char *) "Select", (XtActionProc) SelectPuzzle},
	{(char *) "Release", (XtActionProc) ReleasePuzzle},
	{(char *) "PracticeMaybe", (XtActionProc) PracticePuzzleMaybe},
	{(char *) "Practice2", (XtActionProc) PracticePuzzle2},
	{(char *) "RandomizeMaybe", (XtActionProc) RandomizePuzzleMaybe},
	{(char *) "Randomize2", (XtActionProc) RandomizePuzzle2},
	{(char *) "Get", (XtActionProc) GetPuzzle},
	{(char *) "Write", (XtActionProc) WritePuzzle},
	{(char *) "Undo", (XtActionProc) UndoPuzzle},
	{(char *) "Redo", (XtActionProc) RedoPuzzle},
	{(char *) "Clear", (XtActionProc) ClearPuzzle},
	{(char *) "Randomize", (XtActionProc) RandomizePuzzle},
	{(char *) "Solve", (XtActionProc) SolvePuzzle},
	{(char *) "Practice", (XtActionProc) PracticePuzzle},
	{(char *) "Orientize", (XtActionProc) OrientizePuzzle},
	{(char *) "Period2", (XtActionProc) Period2ModePuzzle},
	{(char *) "Period3", (XtActionProc) Period3ModePuzzle},
	{(char *) "Both", (XtActionProc) BothModePuzzle},
	{(char *) "View", (XtActionProc) ViewPuzzle},
	{(char *) "Speed", (XtActionProc) SpeedPuzzle},
	{(char *) "Slow", (XtActionProc) SlowPuzzle},
	{(char *) "Sound", (XtActionProc) SoundPuzzle},
	{(char *) "Enter", (XtActionProc) EnterPuzzle},
	{(char *) "Leave", (XtActionProc) LeavePuzzle}
};

static XtResource resourcesPuzzle3D[] =
{
	{XtNwidth, XtCWidth, XtRDimension, sizeof (Dimension),
	 XtOffset(DinoWidget, core.width),
	 XtRString, (caddr_t) "250"},
	{XtNheight, XtCHeight, XtRDimension, sizeof (Dimension),
	 XtOffset(DinoWidget, core.height),
	 XtRString, (caddr_t) "400"},
	{XtNmono, XtCMono, XtRBoolean, sizeof (Boolean),
	 XtOffset(DinoWidget, dino.mono),
	 XtRString, (caddr_t) "FALSE"},
	{XtNreverseVideo, XtCReverseVideo, XtRBoolean, sizeof (Boolean),
	 XtOffset(DinoWidget, dino.reverse),
	 XtRString, (caddr_t) "FALSE"},
	{XtNforeground, XtCForeground, XtRPixel, sizeof (Pixel),
	 XtOffset(DinoWidget, dino.foreground),
	 XtRString, (caddr_t) XtDefaultForeground},
	{XtNbackground, XtCBackground, XtRPixel, sizeof (Pixel),
	 XtOffset(DinoWidget, dino.background),
	 XtRString, (caddr_t) XtDefaultBackground},
	{XtNframeColor, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(DinoWidget, dino.frameColor),
	 XtRString, (caddr_t) "cyan" /*XtDefaultForeground*/},
	{XtNfaceColor0, XtCLabel, XtRString, sizeof (String),
	 XtOffset(DinoWidget, dino.faceName[0]),
	 XtRString, (caddr_t) "Red"},
	{XtNfaceColor1, XtCLabel, XtRString, sizeof (String),
	 XtOffset(DinoWidget, dino.faceName[1]),
	 XtRString, (caddr_t) "Blue"},
	{XtNfaceColor2, XtCLabel, XtRString, sizeof (String),
	 XtOffset(DinoWidget, dino.faceName[2]),
	 XtRString, (caddr_t) "Yellow"},
	{XtNfaceColor3, XtCLabel, XtRString, sizeof (String),
	 XtOffset(DinoWidget, dino.faceName[3]),
	 XtRString, (caddr_t) "Green"},
	{XtNfaceColor4, XtCLabel, XtRString, sizeof (String),
	 XtOffset(DinoWidget, dino.faceName[4]),
	 XtRString, (caddr_t) "White"},
	{XtNfaceColor5, XtCLabel, XtRString, sizeof (String),
	 XtOffset(DinoWidget, dino.faceName[5]),
	 XtRString, (caddr_t) "Orange"},
	{XtNpieceBorder, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(DinoWidget, dino.borderColor),
	 XtRString, (caddr_t) "gray25" /*XtDefaultForeground*/},
	{XtNdelay, XtCDelay, XtRInt, sizeof (int),
	 XtOffset(DinoWidget, dino.delay),
	 XtRString, (caddr_t) "10"},
	{XtNsound, XtCSound, XtRBoolean, sizeof (Boolean),
	 XtOffset(DinoWidget, dino.sound),
	 XtRString, (caddr_t) "FALSE"},
	{XtNmoveSound, XtCMoveSound, XtRString, sizeof (String),
	 XtOffset(DinoWidget, dino.moveSound),
	 XtRString, (caddr_t) MOVESOUND},
	{XtNfont, XtCFont, XtRString, sizeof (String),
	 XtOffset(DinoWidget, dino.font),
	 XtRString, (caddr_t) "9x15bold"},
	{XtNmode, XtCMode, XtRInt, sizeof (int),
	 XtOffset(DinoWidget, dino.mode),
	 XtRString, (caddr_t) "3"}, /*DEFAULT_MODE */
	{XtNorient, XtCOrient, XtRBoolean, sizeof (Boolean),
	 XtOffset(DinoWidget, dino.orient),
	 XtRString, (caddr_t) "FALSE"},	/* DEFAULT_ORIENT */
	{XtNpractice, XtCPractice, XtRBoolean, sizeof (Boolean),
	 XtOffset(DinoWidget, dino.practice),
	 XtRString, (caddr_t) "TRUE"}, /* DEFAULT_PRACTICE */
	{XtNuserName, XtCUserName, XtRString, sizeof (String),
	 XtOffset(DinoWidget, dino.userName),
	 XtRString, (caddr_t) ""},
	{XtNscoreFile, XtCScoreFile, XtRString, sizeof (String),
	 XtOffset(DinoWidget, dino.scoreFile),
	 XtRString, (caddr_t) ""},
	{XtNscoreOnly, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(DinoWidget, dino.scoreOnly),
	 XtRString, (caddr_t) "FALSE"},
	{XtNversionOnly, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(DinoWidget, dino.versionOnly),
	 XtRString, (caddr_t) "FALSE"},
	{XtNmenu, XtCMenu, XtRInt, sizeof (int),
	 XtOffset(DinoWidget, dino.menu),
	 XtRString, (caddr_t) "999"}, /* ACTION_IGNORE */
	{XtNstart, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(DinoWidget, dino.started),
	 XtRString, (caddr_t) "FALSE"},
	{XtNcheat, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(DinoWidget, dino.cheat),
	 XtRString, (caddr_t) "FALSE"},
	{XtNface, XtCFace, XtRInt, sizeof (int),
	 XtOffset(DinoWidget, dino.currentFace),
	 XtRString, (caddr_t) "-1"},
	{XtNpos, XtCPos, XtRInt, sizeof (int),
	 XtOffset(DinoWidget, dino.currentPosition),
	 XtRString, (caddr_t) "-1"},
	{XtNdirection, XtCDirection, XtRInt, sizeof (int),
	 XtOffset(DinoWidget, dino.currentDirection),
	 XtRString, (caddr_t) "-1"},
	{XtNstyle, XtCStyle, XtRInt, sizeof (int),
	 XtOffset(DinoWidget, dino.style),
	 XtRString, (caddr_t) "-1"},
	{XtNpixmapSize, XtCPixmapSize, XtRInt, sizeof (int),
	 XtOffset(DinoWidget, dino.pixmapSize),
	 XtRString, (caddr_t) "64"},
	{XtNselectCallback, XtCCallback, XtRCallback, sizeof (caddr_t),
	 XtOffset(DinoWidget, dino.select),
	 XtRCallback, (caddr_t) NULL}
};

Dino3DClassRec dino3dClassRec =
{
	{
		(WidgetClass) & dinoClassRec,	/* superclass */
		(char *) "Dino3D",	/* class name */
		sizeof (Dino3DRec),	/* widget size */
		NULL,		/* class initialize */
		NULL,		/* class part initialize */
		FALSE,		/* class inited */
		(XtInitProc) InitializePuzzle3D,	/* initialize */
		NULL,		/* initialize hook */
		XtInheritRealize,	/* realize */
		actionsListPuzzle3D,	/* actions */
		XtNumber(actionsListPuzzle3D),	/* num actions */
		resourcesPuzzle3D,	/* resources */
		XtNumber(resourcesPuzzle3D),	/* num resources */
		NULLQUARK,	/* xrm class */
		TRUE,		/* compress motion */
		TRUE,		/* compress exposure */
		TRUE,		/* compress enterleave */
		TRUE,		/* visible interest */
		NULL,		/* destroy */
		(XtWidgetProc) ResizePuzzle3D,	/* resize */
		(XtExposeProc) ExposePuzzle3D,	/* expose */
		(XtSetValuesFunc) SetValuesPuzzle3D,	/* set values */
		NULL,		/* set values hook */
		XtInheritSetValuesAlmost,	/* set values almost */
		NULL,		/* get values hook */
		XtInheritAcceptFocus,	/* accept focus */
		XtVersion,	/* version */
		NULL,		/* callback private */
		defaultTranslationsPuzzle3D,	/* tm table */
		NULL,		/* query geometry */
		NULL,		/* display accelerator */
		NULL		/* extension */
	},
	{
		0		/* ignore */
	},
	{
		0		/* ignore */
	}
};

WidgetClass dino3dWidgetClass = (WidgetClass) & dino3dClassRec;
#endif

static Point faceLoc3D[MAX_FACES][MAX_ORIENT];
static Point cubeLoc3D[MAX_FACES][MAX_ORIENT];
static Point diamondLoc3D[MAX_FACES][MAX_ORIENT];
static Point tinyDiamondLoc3D[MAX_FACES][MAX_ORIENT];
static Point triangleLoc3D[MAX_FACES][MAX_ORIENT][4];
static Point letter3DList[MAX_FACES][MAX_ORIENT];
static Point orientTriangle[MAX_FACES][MAX_ORIENT][2];

static Pixmap dr = 0; /* dummy for future double buffering */

static void
DiamondOffset3D(Dino3DWidget w, int face, int *dx, int *dy)
{
	if (w->dino.vertical) {
		switch (face) {
		case 0:
			*dx = (2 * faceLoc3D[0][0].x +
				faceLoc3D[0][1].x +
				faceLoc3D[0][2].x) / 2;
			*dy = (2 * faceLoc3D[0][0].y +
				faceLoc3D[0][1].y) / 2;
			break;
		case 1:
			*dx = faceLoc3D[1][0].x + faceLoc3D[1][1].x;
			*dy = faceLoc3D[1][0].y;
			break;
		case 2:
			*dx = (2 * faceLoc3D[2][0].x +
				faceLoc3D[2][1].x +
				faceLoc3D[2][2].x) / 2;
			*dy = (2 * faceLoc3D[2][0].y +
				faceLoc3D[2][1].y +
				faceLoc3D[2][2].y) / 2;
			break;
		case 3:
			*dx = faceLoc3D[3][0].x + faceLoc3D[3][1].x;
			*dy = faceLoc3D[3][0].y;
			break;
		case 4:
			*dx = (2 * faceLoc3D[4][0].x +
				faceLoc3D[4][1].x +
				faceLoc3D[4][2].x) / 2;
			*dy = (2 * faceLoc3D[4][0].y +
				faceLoc3D[4][1].y +
			faceLoc3D[4][2].y) / 2;
			break;
		case 5:
			*dx = (2 * faceLoc3D[5][0].x +
				faceLoc3D[5][1].x +
				faceLoc3D[5][2].x) / 2;
			*dy = (2 * faceLoc3D[5][0].y +
				faceLoc3D[5][1].y) / 2;
			break;
		default:
			{
				char *buf;

				intCat(&buf, "DiamondOffset3D: face ",
					face);
				DISPLAY_WARNING(buf);
				free(buf);
			}
		}
	} else {
		switch (face) {
		case 0:
			*dx = faceLoc3D[0][0].x;
			*dy = faceLoc3D[0][0].y + faceLoc3D[0][1].y;
			break;
		case 1:
			*dx = (2 * faceLoc3D[1][0].x +
				faceLoc3D[1][1].x +
				faceLoc3D[1][2].x) / 2;
			*dy = (2 * faceLoc3D[1][0].y +
				faceLoc3D[1][1].y +
				faceLoc3D[1][2].y) / 2;
			break;
		case 2:
			*dx = (2 * faceLoc3D[2][0].x +
				faceLoc3D[2][1].x) / 2;
			*dy = (2 * faceLoc3D[2][0].y +
				faceLoc3D[2][1].y +
				faceLoc3D[2][2].y) / 2;
			break;
		case 3:
			*dx = (2 * faceLoc3D[3][0].x +
				faceLoc3D[3][1].x) / 2;
			*dy = (2 * faceLoc3D[3][0].y +
				faceLoc3D[3][1].y +
				faceLoc3D[3][2].y) / 2;
			break;
		case 4:
			*dx = faceLoc3D[4][0].x;
			*dy = faceLoc3D[4][0].y + faceLoc3D[4][1].y;
			break;
		case 5:
			*dx = (2 * faceLoc3D[5][0].x +
				faceLoc3D[5][1].x +
				faceLoc3D[5][2].x) / 2;
			*dy = (2 * faceLoc3D[5][0].y +
				faceLoc3D[5][1].y +
				faceLoc3D[5][2].y) / 2;
			break;
		default:
			{
				char *buf;

				intCat(&buf, "DiamondOffset3D: face ", face);
				DISPLAY_WARNING(buf);
				free(buf);
			}
		}
	}
}

static void
CubeOffset3D(Dino3DWidget w, int face, int side, int *dx, int *dy)
{
	int corner;

	DiamondOffset3D(w, face, dx, dy);
	for (corner = 1; corner <= side; corner++) {
		*dx += tinyDiamondLoc3D[face][corner].x;
		*dy += tinyDiamondLoc3D[face][corner].y;
	}
}

static void
MapTo3D(Dino3DWidget w, int face, int corner, int *side)
{
	switch (face) {
	case 0:
		*side = (corner + 2) % MAX_ORIENT;
		break;
	case 1:
	case 5:
		*side = corner;
		break;
	case 2:
		*side = (corner + 1) % MAX_ORIENT;
		break;
	case 3:
	case 4:
		*side = (corner + 3) % MAX_ORIENT;
		break;
	default:
		{
			char *buf;

			intCat(&buf, "MapTo3D: face ", face);
			DISPLAY_WARNING(buf);
			free(buf);
		}
	}
}

static void
MapFrom3D(Dino3DWidget w, int face, int side, int *corner)
{
	switch (face) {
	case 0:
		*corner = (side + 2) % MAX_ORIENT;
		break;
	case 1:
	case 5:
		*corner = side;
		break;
	case 2:
		*corner = (side + 3) % MAX_ORIENT;
		break;
	case 3:
	case 4:
		*corner = (side + 1) % MAX_ORIENT;
		break;
	default:
		{
			char *buf;

			intCat(&buf, "MapFrom3D: face ", face);
			DISPLAY_WARNING(buf);
			free(buf);
		}
	}
}

void
DrawTriangle3D(Dino3DWidget w, int face, int position, int offset)
{
	GC faceGC, borderGC;
	int corner = 0, dx, dy, rotation, letterX, letterY;

	MapTo3D(w, face, position, &corner);
	CubeOffset3D(w, face, corner, &dx, &dy);
	rotation = w->dino.cubeLoc[face][position].rotation;
	letterX = dx + letter3DList[face][corner].x;
	letterY = dy + letter3DList[face][corner].y;
	triangleLoc3D[face][corner][0].x = dx;
	triangleLoc3D[face][corner][0].y = dy;
	if (offset) {
		borderGC = w->dino.faceGC[(int) w->dino.cubeLoc[face][position].face];
		if (w->dino.mono) {
			faceGC = w->dino.inverseGC;
		} else {
			faceGC = w->dino.borderGC;
		}
	} else {
		faceGC = w->dino.faceGC[(int) w->dino.cubeLoc[face][position].face];
		borderGC = w->dino.borderGC;
	}
	POLYGON((DinoWidget) w, dr, faceGC, borderGC,
		triangleLoc3D[face][corner], 3, True, False);
	if (w->dino.mono) {
		char buf[2];

		buf[0] =
#ifdef WINVER
			w->dino.faceChar[w->dino.cubeLoc
				[face][position].face];
#else
			w->dino.faceName[w->dino.cubeLoc
				[face][position].face][0];
#endif
		buf[1] = '\0';
		if (offset) {
			borderGC = w->dino.borderGC;
		} else {
			borderGC = w->dino.inverseGC;
		}
		DRAWTEXT(w, dr, borderGC,
			letterX + w->dino.letterOffset.x, letterY + w->dino.letterOffset.y,
			buf, 1);
	}
	if (w->dino.orient) {
		DRAWLINE(w, dr, borderGC,
			letterX +
			orientTriangle[face][rotation][0].x,
			letterY +
			orientTriangle[face][rotation][0].y,
			letterX +
			orientTriangle[face][rotation][1].x,
			letterY +
			orientTriangle[face][rotation][1].y);
	}
}

void
DrawFrame3D(Dino3DWidget w, Boolean focus)
{
	int face, dx, dy;
	GC gc = (focus) ? w->dino.frameGC : w->dino.borderGC;

	dx = w->dino3d.viewSize.x + w->dino.puzzleOffset.x;
	dy = w->dino3d.viewSize.y + w->dino.puzzleOffset.y;
	if (w->dino.vertical) {
		DRAWLINE(w, dr, w->dino.frameGC,
			0, dy, dx + w->dino.puzzleOffset.x + 1, dy);
		DRAWTEXT(w, dr, w->dino.borderGC,
			(int) (2 * w->dino.delta),
			(int) (3 * w->dino.delta + w->dino.letterOffset.y),
			"Front", 5);
		DRAWTEXT(w, dr, w->dino.borderGC, (int)
			(-4 * w->dino.delta + 2 * 6 * w->dino.letterOffset.x + w->core.width),
			(int) (-w->dino.delta - 2 * w->dino.letterOffset.y + w->core.height),
			"Back", 4);
	} else {
		DRAWLINE(w, dr, w->dino.frameGC,
			dx, 0, dx, dy + w->dino.puzzleOffset.y + 1);
		DRAWTEXT(w, dr, w->dino.borderGC,
			(int) (2 * w->dino.delta),
			(int) (3 * w->dino.delta + w->dino.letterOffset.y),
			"Front", 5);
		DRAWTEXT(w, dr, w->dino.borderGC, (int)
			(-4 * w->dino.delta + 2 * 6 * w->dino.letterOffset.x + w->core.width),
			(int) (-w->dino.delta - 2 * w->dino.letterOffset.y + w->core.height),
			"Back", 4);
	}
	for (face = 0; face < MAX_FACES; face++) {
		POLYLINE((DinoWidget) w, dr, gc,
			faceLoc3D[face], MAX_ORIENT, False);
	}
}

static void
ResizePieces(Dino3DWidget w)
{
	int face, orient, side, corner;
	Point subcubeLoc3D[MAX_FACES][MAX_ORIENT];
	Point orientCubeLoc3D[2][MAX_FACES][MAX_ORIENT];
	Point subdiamondLoc3D[MAX_FACES][MAX_ORIENT];
	Point orientDiamondLoc3D[2][MAX_FACES][MAX_ORIENT];

	w->dino.letterOffset.x = -2;
	w->dino.letterOffset.y = 3;
	w->dino3d.viewMiddle.x = w->dino3d.faceSize.x +
		w->dino.puzzleOffset.x;
	w->dino3d.viewMiddle.y = w->dino3d.faceSize.y +
		w->dino.puzzleOffset.y;
	for (face = 0; face < MAX_FACES; face++) {
		faceLoc3D[face][0].x = w->dino3d.viewMiddle.x;
		faceLoc3D[face][0].y = w->dino3d.viewMiddle.y;
		for (orient = 1; orient < MAX_ORIENT; orient++) {
			faceLoc3D[face][orient].x = w->dino3d.faceSize.x;
			faceLoc3D[face][orient].y = w->dino3d.faceSize.y;
		}
	}
	if (w->dino.vertical) {
		faceLoc3D[0][1].x /= -2;
		faceLoc3D[0][1].y /= -1;
		faceLoc3D[0][2].y = 0;
		faceLoc3D[0][3].x /= 2;

		faceLoc3D[1][1].x /= -2;
		faceLoc3D[1][2].x /= -2;
		faceLoc3D[1][2].y /= -1;
		faceLoc3D[1][3].x /= 2;
		faceLoc3D[1][3].y /= -1;

		faceLoc3D[2][1].y = 0;
		faceLoc3D[2][2].x /= -2;
		faceLoc3D[2][3].x /= -1;
		faceLoc3D[2][3].y = 0;

		for (face = MAX_FACES / 2; face < MAX_FACES; face++)
			faceLoc3D[face][0].y += w->dino3d.viewSize.y + 3;

		faceLoc3D[3][1].x /= 2;
		faceLoc3D[3][1].y /= -1;
		faceLoc3D[3][2].x /= 2;
		faceLoc3D[3][3].x /= -2;

		faceLoc3D[4][1].x /= -1;
		faceLoc3D[4][1].y = 0;
		faceLoc3D[4][2].x /= 2;
		faceLoc3D[4][2].y /= -1;
		faceLoc3D[4][3].y = 0;

		faceLoc3D[5][1].x /= 2;
		faceLoc3D[5][2].x /= -1;
		faceLoc3D[5][2].y = 0;
		faceLoc3D[5][3].x /= -2;
		faceLoc3D[5][3].y /= -1;
	} else {
		faceLoc3D[0][1].x /= -1;
		faceLoc3D[0][1].y /= -2;
		faceLoc3D[0][2].y /= -2;
		faceLoc3D[0][3].y /= 2;

		faceLoc3D[1][1].x = 0;
		faceLoc3D[1][2].x /= -1;
		faceLoc3D[1][2].y /= -2;
		faceLoc3D[1][3].x = 0;
		faceLoc3D[1][3].y /= -1;

		faceLoc3D[2][1].y /= -2;
		faceLoc3D[2][2].x = 0;
		faceLoc3D[2][3].y /= 2;
		faceLoc3D[2][3].x /= -1;

		for (face = MAX_FACES / 2; face < MAX_FACES; face++)
			faceLoc3D[face][0].x += w->dino3d.viewSize.x + 3;

		faceLoc3D[3][1].x /= -1;
		faceLoc3D[3][1].y /= 2;
		faceLoc3D[3][2].x = 0;
		faceLoc3D[3][2].y /= -1;
		faceLoc3D[3][3].y /= -2;

		faceLoc3D[4][1].y /= 2;
		faceLoc3D[4][2].x /= -1;
		faceLoc3D[4][2].y /= 2;
		faceLoc3D[4][3].x /= -1;
		faceLoc3D[4][3].y /= -2;

		faceLoc3D[5][1].x = 0;
		faceLoc3D[5][1].y /= -1;
		faceLoc3D[5][2].y /= 2;
		faceLoc3D[5][3].x = 0;
	}

	for (face = 0; face < MAX_FACES; face++) {
		cubeLoc3D[face][0].x = faceLoc3D[face][0].x;
		cubeLoc3D[face][0].y = faceLoc3D[face][0].y;
		subcubeLoc3D[face][0].x = faceLoc3D[face][0].x;
		subcubeLoc3D[face][0].y = faceLoc3D[face][0].y;
		orientCubeLoc3D[0][face][0].x = orientCubeLoc3D[1][face][0].x = 0;
		orientCubeLoc3D[0][face][0].y = orientCubeLoc3D[1][face][0].y = 0;
		for (orient = 1; orient < MAX_ORIENT; orient++) {
			cubeLoc3D[face][orient].x = faceLoc3D[face][orient].x - 3 *
				w->dino.delta * faceLoc3D[face][orient].x /
				w->dino3d.faceSize.x;
			cubeLoc3D[face][orient].y = faceLoc3D[face][orient].y - 3 *
				w->dino.delta * faceLoc3D[face][orient].y /
				w->dino3d.faceSize.y;
			subcubeLoc3D[face][orient].x = faceLoc3D[face][orient].x -
				4 * faceLoc3D[face][orient].x * w->dino.delta /
				w->dino3d.faceSize.x;
			subcubeLoc3D[face][orient].y = faceLoc3D[face][orient].y -
				4 * faceLoc3D[face][orient].y * w->dino.delta /
				w->dino3d.faceSize.y;
			orientCubeLoc3D[0][face][orient].x = (faceLoc3D[face][orient].x -
				5 * faceLoc3D[face][orient].x * w->dino.delta /
				w->dino3d.faceSize.x) / 3;
			orientCubeLoc3D[0][face][orient].y = (faceLoc3D[face][orient].y -
				5 * faceLoc3D[face][orient].y * w->dino.delta /
				w->dino3d.faceSize.y) / 3;
			orientCubeLoc3D[1][face][orient].x = (faceLoc3D[face][orient].x -
				7 * faceLoc3D[face][orient].x * w->dino.delta /
				w->dino3d.faceSize.x) / 6;
			orientCubeLoc3D[1][face][orient].y = (faceLoc3D[face][orient].y -
				7 * faceLoc3D[face][orient].y * w->dino.delta /
				w->dino3d.faceSize.y) / 6;
		}

		MapFrom3D(w, face, 1, &corner);
		orientTriangle[face][corner][0].x = -orientCubeLoc3D[0][face][2].x / 2;
		orientTriangle[face][corner][0].y = -orientCubeLoc3D[0][face][2].y / 2;
		orientTriangle[face][corner][1].x = -orientCubeLoc3D[1][face][2].x / 2;
		orientTriangle[face][corner][1].y = -orientCubeLoc3D[1][face][2].y / 2;
		for (orient = 0; orient < MAX_ORIENT - 1; orient++) {
			diamondLoc3D[face][orient].x = (cubeLoc3D[face][orient].x +
				cubeLoc3D[face][orient + 1].x) / 2;
			diamondLoc3D[face][orient].y = (cubeLoc3D[face][orient].y +
				cubeLoc3D[face][orient + 1].y) / 2;
			subdiamondLoc3D[face][orient].x = (subcubeLoc3D[face][orient].x +
				subcubeLoc3D[face][orient + 1].x) / 2;
			subdiamondLoc3D[face][orient].y = (subcubeLoc3D[face][orient].y +
				subcubeLoc3D[face][orient + 1].y) / 2;
			tinyDiamondLoc3D[face][orient + 1].x = diamondLoc3D[face][orient].x -
				subdiamondLoc3D[face][orient].x;
			tinyDiamondLoc3D[face][orient + 1].y = diamondLoc3D[face][orient].y -
				subdiamondLoc3D[face][orient].y;
			orientDiamondLoc3D[0][face][orient].x =
				(orientCubeLoc3D[0][face][orient].x +
				orientCubeLoc3D[0][face][orient + 1].x) / 2;
			orientDiamondLoc3D[0][face][orient].y =
				(orientCubeLoc3D[0][face][orient].y +
				orientCubeLoc3D[0][face][orient + 1].y) / 2;
			orientDiamondLoc3D[1][face][orient].x =
				(orientCubeLoc3D[1][face][orient].x +
				orientCubeLoc3D[1][face][orient + 1].x) / 2;
			orientDiamondLoc3D[1][face][orient].y =
				(orientCubeLoc3D[1][face][orient].y +
				orientCubeLoc3D[1][face][orient + 1].y) / 2;
		}
		/* Its a parallelagram so take advantage of that */
		diamondLoc3D[face][orient].x = (cubeLoc3D[face][MAX_ORIENT - 1].x -
			cubeLoc3D[face][MAX_ORIENT / 2].x) / 2;
		diamondLoc3D[face][orient].y = (cubeLoc3D[face][MAX_ORIENT - 1].y -
			cubeLoc3D[face][MAX_ORIENT / 2].y) / 2;
		subdiamondLoc3D[face][orient].x = (subcubeLoc3D[face][MAX_ORIENT - 1].x -
			subcubeLoc3D[face][MAX_ORIENT / 2].x) / 2;
		subdiamondLoc3D[face][orient].y = (subcubeLoc3D[face][MAX_ORIENT - 1].y -
			subcubeLoc3D[face][MAX_ORIENT / 2].y) / 2;
		tinyDiamondLoc3D[face][0].x = diamondLoc3D[face][orient].x -
			subdiamondLoc3D[face][orient].x;
		tinyDiamondLoc3D[face][0].y = diamondLoc3D[face][orient].y -
			subdiamondLoc3D[face][orient].y;
		orientDiamondLoc3D[0][face][orient].x =
			(orientCubeLoc3D[0][face][MAX_ORIENT - 1].x -
			orientCubeLoc3D[0][face][MAX_ORIENT / 2].x) / 2;
		orientDiamondLoc3D[0][face][orient].y =
			(orientCubeLoc3D[0][face][MAX_ORIENT - 1].y -
			orientCubeLoc3D[0][face][MAX_ORIENT / 2].y) / 2;
		orientDiamondLoc3D[1][face][orient].x =
			(orientCubeLoc3D[1][face][MAX_ORIENT - 1].x -
			orientCubeLoc3D[1][face][MAX_ORIENT / 2].x) / 2;
		orientDiamondLoc3D[1][face][orient].y =
			(orientCubeLoc3D[1][face][MAX_ORIENT - 1].y -
			orientCubeLoc3D[1][face][MAX_ORIENT / 2].y) / 2;

		MapFrom3D(w, face, 1, &corner);
		orientTriangle[face][corner][0].x = -orientCubeLoc3D[0][face][2].x / 2;
		orientTriangle[face][corner][0].y = -orientCubeLoc3D[0][face][2].y / 2;
		orientTriangle[face][corner][1].x = -orientCubeLoc3D[1][face][2].x / 2;
		orientTriangle[face][corner][1].y = -orientCubeLoc3D[1][face][2].y / 2;

		triangleLoc3D[face][0][1].x = subdiamondLoc3D[face][2].x;
		triangleLoc3D[face][0][2].x = -subdiamondLoc3D[face][1].x -
			subdiamondLoc3D[face][2].x;
		triangleLoc3D[face][0][1].y = subdiamondLoc3D[face][2].y;
		triangleLoc3D[face][0][2].y = -subdiamondLoc3D[face][1].y -
			subdiamondLoc3D[face][2].y;
		triangleLoc3D[face][1][1].x = -subdiamondLoc3D[face][1].x;
		triangleLoc3D[face][1][2].x = subdiamondLoc3D[face][1].x -
			subdiamondLoc3D[face][2].x;
		triangleLoc3D[face][1][1].y = -subdiamondLoc3D[face][1].y;
		triangleLoc3D[face][1][2].y = subdiamondLoc3D[face][1].y -
			subdiamondLoc3D[face][2].y;
		triangleLoc3D[face][2][1].x = -subdiamondLoc3D[face][2].x;
		triangleLoc3D[face][2][2].x = subdiamondLoc3D[face][1].x +
			subdiamondLoc3D[face][2].x;
		triangleLoc3D[face][2][1].y = -subdiamondLoc3D[face][2].y;
		triangleLoc3D[face][2][2].y = subdiamondLoc3D[face][1].y +
			subdiamondLoc3D[face][2].y;
		triangleLoc3D[face][3][1].x = subdiamondLoc3D[face][1].x;
		triangleLoc3D[face][3][2].x = subdiamondLoc3D[face][2].x -
			subdiamondLoc3D[face][1].x;
		triangleLoc3D[face][3][1].y = subdiamondLoc3D[face][1].y;
		triangleLoc3D[face][3][2].y = subdiamondLoc3D[face][2].y -
			subdiamondLoc3D[face][1].y;
		for (orient = 0; orient < MAX_ORIENT; orient++) {
			letter3DList[face][orient].x =
				(2 * triangleLoc3D[face][orient][1].x +
				triangleLoc3D[face][orient][2].x) / 3;
			letter3DList[face][orient].y =
				(2 * triangleLoc3D[face][orient][1].y +
				triangleLoc3D[face][orient][2].y) / 3;
			triangleLoc3D[face][orient][3].x =
				-triangleLoc3D[face][orient][1].x - triangleLoc3D[face][orient][2].x;
			triangleLoc3D[face][orient][3].y =
				-triangleLoc3D[face][orient][1].y - triangleLoc3D[face][orient][2].y;
		}
		for (orient = 1; orient < MAX_ORIENT; orient++) {
			side = corner;
			MapFrom3D(w, face, (orient + 1) % MAX_ORIENT, &corner);
			orientTriangle[face][corner][0].x =
				orientTriangle[face][side][0].x +
				orientDiamondLoc3D[0][face][orient].x;
			orientTriangle[face][corner][0].y =
				orientTriangle[face][side][0].y +
				orientDiamondLoc3D[0][face][orient].y;
			orientTriangle[face][corner][1].x =
				orientTriangle[face][side][1].x +
				orientDiamondLoc3D[1][face][orient].x;
			orientTriangle[face][corner][1].y =
				orientTriangle[face][side][1].y +
				orientDiamondLoc3D[1][face][orient].y;
		}
	}
	w->dino3d.cubeSize.x = w->dino3d.faceSize.x - 2 * w->dino.delta;
	w->dino3d.cubeSize.y = w->dino3d.faceSize.y - 2 * w->dino.delta;
	w->dino3d.cubeDiagonal = w->dino3d.faceDiagonal - 2 * w->dino.delta;
	w->dino3d.cubeDiag = w->dino3d.faceDiagonal + 2 * w->dino.delta;
}

Boolean
SelectPieces3D(Dino3DWidget w, int x, int y, int *face, int *position)
{
	int u, v, front, tl, ur, ul, found, corner, dx, dy;
	int x_1, y_1, x_2, y_2;

	x_1 = x;
	y_1 = y;
	if (w->dino.vertical) {
		x -= w->dino3d.viewMiddle.x;
		front = (y < w->dino3d.viewSize.y + w->dino.puzzleOffset.y);
		if (!front)
			y -= (w->dino3d.viewSize.y);
		tl = (y < w->dino3d.viewMiddle.y);
		y -= w->dino3d.viewMiddle.y;
		u = -w->dino3d.faceSize.y * x + w->dino3d.faceDiagonal * y;
		v = w->dino3d.faceSize.y * x + w->dino3d.faceDiagonal * y;
		ur = (u < 0);
		ul = (v < 0);
		if (front) {
			if (tl)
				*face = (ur) ? 0 : 1;
			else
				*face = (ul) ? 1 : 2;
		} else {
			if (tl)
				*face = (ul) ? 4 : 3;
			else
				*face = (ur) ? 3 : 5;
		}
	} else {
		y -= w->dino3d.viewMiddle.y;
		front = (x < w->dino3d.viewSize.x + w->dino.puzzleOffset.x);
		if (!front)
			x -= (w->dino3d.viewSize.x);
		tl = (x < w->dino3d.viewMiddle.x);
		x -= w->dino3d.viewMiddle.x;
		u = -w->dino3d.faceSize.x * y + w->dino3d.faceDiagonal * x;
		v = w->dino3d.faceSize.x * y + w->dino3d.faceDiagonal * x;
		ur = (u < 0);
		ul = (v < 0);
		if (front) {
			if (tl)
				*face = (ur) ? 1 : 0;
			else
				*face = (ul) ? 0 : 2;
		} else {
			if (tl)
				*face = (ul) ? 3 : 4;
			else
				*face = (ur) ? 4 : 5;
		}
	}
	x = x_1;
	y = y_1;
	found = 0;
	DiamondOffset3D(w, *face, &dx, &dy);
	for (corner = 0; corner < MAX_ORIENT; corner++) {
		int ok = 0;

		x_1 = dx + triangleLoc3D[*face][corner][1].x;
		y_1 = dy + triangleLoc3D[*face][corner][1].y;
		x_2 = x_1 + triangleLoc3D[*face][corner][2].x;
		y_2 = y_1 + triangleLoc3D[*face][corner][2].y;
		if ((x_1 - dx) * (y_2 - dy) <= (y_1 - dy) * (x_2 - dx)) {
			if ((x_1 - dx) * (y - dy) <= (y_1 - dy) * (x - dx))
				ok++;
		} else if ((x_1 - dx) * (y - dy) >= (y_1 - dy) * (x - dx))
			ok++;
		if ((x_2 - dx) * (y_1 - dy) <= (y_2 - dy) * (x_1 - dx)) {
			if ((x_2 - dx) * (y - dy) <= (y_2 - dy) * (x - dx))
				ok++;
		} else if ((x_2 - dx) * (y - dy) >= (y_2 - dy) * (x - dx))
			ok++;
		if (ok == 2) {
			MapFrom3D(w, *face, corner, position);
			found++;
		}
	}
	if (found != 1)
		return False;
	return True;
}

Boolean
NarrowSelection3D(Dino3DWidget w, int *face, int *direction)
{
	switch (*direction) {
	case TR:
	case BR:
	case BL:
	case TL:
		if (w->dino.vertical) {
			if (*face == 1)
				*direction = *direction + 2 * MAX_ORIENT;
			else if (*face == 3)
				*direction = (*direction + 3) % MAX_ORIENT +
					2 * MAX_ORIENT;
		} else {
			if (*face == 0)
				*direction = (*direction + 3) % MAX_ORIENT +
					2 * MAX_ORIENT;
			else if (*face == 4)
				*direction = (*direction + 2) % MAX_ORIENT +
					2 * MAX_ORIENT;
			if (*face == 5)
				*direction = (*direction + 2) % MAX_ORIENT;
		}
		break;
	case CCW:
	case CW:
		if (*face == 3 || *face == 4)
			*direction = (TR + *direction - CW) % MAX_ORIENT;
		else if (*face == 0)
			*direction = (BR + *direction - CW) % MAX_ORIENT;
		else if (*face == 2)
			*direction = (BL + *direction - CW) % MAX_ORIENT;
		else if (*face == 1 || *face == 5)
			*direction = (TL + *direction - CW) % MAX_ORIENT;
		break;
	case TOP:
	case RIGHT:
	case BOTTOM:
	case LEFT:
		if (w->dino.vertical) {
			if (*face == 1)
				*direction = (TL + *direction) % MAX_ORIENT;
			else if (*face == 3)
				*direction = (BL + *direction) % MAX_ORIENT;
		} else {
			if (*face == 0)
				*direction = (TR + *direction) % MAX_ORIENT;
			else if (*face == 4)
				*direction = (BR + *direction) % MAX_ORIENT;
			else if (*face == 5)
				*direction = (BL + *direction) % MAX_ORIENT +
					2 * MAX_ORIENT;
		}
		break;
	default:
		return False;
	}
	return True;
}

#ifndef WINVER
static Boolean
SetValuesPuzzle3D(Widget current, Widget request, Widget renew)
{
	Dino3DWidget c = (Dino3DWidget) current, w = (Dino3DWidget) renew;
	Boolean redraw = False;

	if (w->dino3d.cubeSize.x != c->dino3d.cubeSize.x) {
		ResizePuzzle3D(w);
		redraw = True;
	}
	return (redraw);
}
#endif

#ifndef WINVER
static
#endif
void
ResizePuzzle3D(Dino3DWidget w)
{
	Point tempSize;
#ifdef WINVER
	RECT rect;

	/* Determine size of client area */
	(void) GetClientRect(w->core.hWnd, &rect);
	w->core.width = rect.right;
	w->core.height = rect.bottom;
#endif
	w->dino.delta = 4;
	w->dino.vertical = (w->core.height >= w->core.width);
	if (w->dino.vertical) {
		tempSize.y = w->core.height / MAX_VIEWS;
		tempSize.x = w->core.width;
		if (tempSize.x >= DIVIDE(tempSize.y)) {
			w->dino3d.cubeSize.y = MAX((tempSize.y -
				3 * w->dino.delta) / 2 - w->dino.delta - 2, 0);
			w->dino3d.cubeSize.x = DIVIDE(w->dino3d.cubeSize.y);
		} else {
			w->dino3d.cubeSize.x = MAX((tempSize.x -
				2 * w->dino.delta - 7) / 2 - w->dino.delta, 0);
			w->dino3d.cubeSize.y = MULTIPLY(w->dino3d.cubeSize.x);
		}
		w->dino3d.cubeDiagonal = w->dino3d.cubeSize.x / 2;
		w->dino3d.faceSize.x = w->dino3d.cubeSize.x + 2 * w->dino.delta + 1;
		w->dino3d.faceSize.y = w->dino3d.cubeSize.y + 2 * w->dino.delta + 1;
		w->dino3d.faceDiagonal = w->dino3d.faceSize.x / 2;
		w->dino3d.viewSize.x = 2 * w->dino3d.faceSize.x + 3;
		w->dino3d.viewSize.y = 2 * w->dino3d.faceSize.y + 3;
		w->dino.puzzleSize.x = w->dino3d.viewSize.x + 1;
		w->dino.puzzleSize.y = MAX_VIEWS * w->dino3d.viewSize.y + 1;
	} else {
		tempSize.x = w->core.width / MAX_VIEWS;
		tempSize.y = w->core.height;
		if (tempSize.y >= DIVIDE(tempSize.x)) {
			w->dino3d.cubeSize.x = MAX((tempSize.x -
				3 * w->dino.delta) / 2 - w->dino.delta - 2, 0);
			w->dino3d.cubeSize.y = DIVIDE(w->dino3d.cubeSize.x);
		} else {
			w->dino3d.cubeSize.y = MAX((tempSize.y -
				2 * w->dino.delta - 7) / 2 - w->dino.delta, 0);
			w->dino3d.cubeSize.x = MULTIPLY(w->dino3d.cubeSize.y);
		}
		w->dino3d.cubeDiagonal = w->dino3d.cubeSize.y / 2;
		w->dino3d.faceSize.y = w->dino3d.cubeSize.y + 2 * w->dino.delta + 1;
		w->dino3d.faceSize.x = w->dino3d.cubeSize.x + 2 * w->dino.delta + 1;
		w->dino3d.faceDiagonal = w->dino3d.faceSize.y / 2;
		w->dino3d.viewSize.y = 2 * w->dino3d.faceSize.y + 3;
		w->dino3d.viewSize.x = 2 * w->dino3d.faceSize.x + 3;
		w->dino.puzzleSize.y = w->dino3d.viewSize.y + 1;
		w->dino.puzzleSize.x = MAX_VIEWS * w->dino3d.viewSize.x + 1;
	}
	w->dino.puzzleOffset.x = ((int) w->core.width - w->dino.puzzleSize.x) / 2;
	w->dino.puzzleOffset.y = ((int) w->core.height - w->dino.puzzleSize.y) /
		2;
	ResizePieces(w);
}

#ifndef WINVER
static
#endif
void
InitializePuzzle3D(
#ifdef WINVER
Dino3DWidget w
#else
Widget request, Widget renew
#endif
)
{
#ifndef WINVER
	Dino3DWidget w = (Dino3DWidget) renew;

	SetAllColors((DinoWidget) w);
#endif
	w->dino.dim = 3;
	ResizePuzzle3D(w);
}

#ifndef WINVER
static
#endif
void
ExposePuzzle3D(
#ifdef WINVER
Dino3DWidget w
#else
Widget renew, XEvent *event, Region region
#endif
)
{
#ifndef WINVER
	Dino3DWidget w = (Dino3DWidget) renew;

	if (!w->core.visible)
		return;
#endif
	FILLRECTANGLE(w, dr, w->dino.inverseGC,
		0, 0, w->core.width, w->core.height);
	DrawFrame3D(w, w->dino.focus);
	DrawAllPieces((DinoWidget) w);
}

#ifndef WINVER
static void
MovePuzzle3DTl(Dino3DWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((DinoWidget) w, event->xbutton.x, event->xbutton.y, TL,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & Mod1Mask));
}

static void
MovePuzzle3DTop(Dino3DWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((DinoWidget) w, event->xbutton.x, event->xbutton.y, TOP,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & Mod1Mask));
}

static void
MovePuzzle3DTr(Dino3DWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((DinoWidget) w, event->xbutton.x, event->xbutton.y, TR,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & Mod1Mask));
}

static void
MovePuzzle3DLeft(Dino3DWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((DinoWidget) w, event->xbutton.x, event->xbutton.y, LEFT,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & Mod1Mask));
}

static void
MovePuzzle3DCw(Dino3DWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((DinoWidget) w, event->xbutton.x, event->xbutton.y, CW,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & Mod1Mask));
}

static void
MovePuzzle3DRight(Dino3DWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((DinoWidget) w, event->xbutton.x, event->xbutton.y, RIGHT,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & Mod1Mask));
}

static void
MovePuzzle3DBl(Dino3DWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((DinoWidget) w, event->xbutton.x, event->xbutton.y, BL,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & Mod1Mask));
}

static void
MovePuzzle3DBottom(Dino3DWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((DinoWidget) w, event->xbutton.x, event->xbutton.y, BOTTOM,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & Mod1Mask));
}

static void
MovePuzzle3DBr(Dino3DWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((DinoWidget) w, event->xbutton.x, event->xbutton.y, BR,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & Mod1Mask));
}

static void
MovePuzzle3DCcw(Dino3DWidget w, XEvent *event, char **args, int nArgs)
{
	MovePuzzleInput((DinoWidget) w, event->xbutton.x, event->xbutton.y, CCW,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask),
		(int) (event->xkey.state & Mod1Mask));
}
#endif
