//
//  Copyright (c) 1999 by Mike Romberg (romberg@fsl.noaa.gov)
//
//  This file may be distributed under terms of the GPL
//
// $Id$
//

#include "diskmeter.h"
#include "xosview.h"
#include <fstream>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#define MAX_PROCSTAT_LENGTH 2048

DiskMeter::DiskMeter( XOSView *parent, float max ) : FieldMeterGraph(
  parent, 3, "DISK", "READ/WRITE/IDLE"), _vmstat(false),
  _statFileName("/proc/stat")
    {
    read_prev_ = 0;
    write_prev_ = 0;
    maxspeed_ = max;

    struct stat buf;
    if (stat("/proc/vmstat", &buf) == 0
      && buf.st_mode & S_IFREG)
        {
        _vmstat = true;
        _statFileName = "/proc/vmstat";
        getvmdiskinfo();
        }
    else
        getdiskinfo();
    }

DiskMeter::~DiskMeter( void )
    {
    }

void DiskMeter::checkResources( void )
    {
    FieldMeterGraph::checkResources();

    setfieldcolor( 0, parent_->getResource("diskReadColor") );
    setfieldcolor( 1, parent_->getResource("diskWriteColor") );
    setfieldcolor( 2, parent_->getResource("diskIdleColor") );
    priority_ = atoi (parent_->getResource( "diskPriority" ) );
    dodecay_ = parent_->isResourceTrue("diskDecay" );
    useGraph_ = parent_->isResourceTrue( "diskGraph" );
    SetUsedFormat(parent_->getResource("diskUsedFormat"));
    }

void DiskMeter::checkevent( void )
    {
    if (_vmstat)
        getvmdiskinfo();
    else
        getdiskinfo();
    drawfields();
    }

void DiskMeter::updateinfo(unsigned long one, unsigned long two,
  int fudgeFactor)
    {
    // assume each "unit" is 1k.
    // This is true for ext2, but seems to be 512 bytes
    // for vfat and 2k for cdroms
    // work in 512-byte blocks

    // tw: strange, on my system, a ext2fs (read and write)
    // unit seems to be 2048. kernel 2.2.12 and the file system
    // is on a SW-RAID5 device (/dev/md0).

    // So this is a FIXME - but how ???

    float itim = IntervalTimeInMicrosecs();
    unsigned long read_curr = one * fudgeFactor;  // FIXME!
    unsigned long write_curr = two * fudgeFactor; // FIXME!

    // avoid strange values at first call
    if(read_prev_ == 0) read_prev_ = read_curr;
    if(write_prev_ == 0) write_prev_ = write_curr;

    // calculate rate in bytes per second
    fields_[0] = ((read_curr - read_prev_) * 1e6 * 512) / itim;
    fields_[1] = ((write_curr - write_prev_) * 1e6 * 512) / itim;

    // fix overflow (conversion bug?)
    if (fields_[0] < 0.0)
        fields_[0] = 0.0;
    if (fields_[1] < 0.0)
        fields_[1] = 0.0;

    if (fields_[0] + fields_[1] > total_)
       	total_ = fields_[0] + fields_[1];

    fields_[2] = total_ - (fields_[0] + fields_[1]);

    read_prev_ = read_curr;
    write_prev_ = write_curr;

    setUsed((fields_[0]+fields_[1]), total_);
    IntervalTimerStart();
    }


void DiskMeter::getvmdiskinfo(void)
    {
    IntervalTimerStop();
    total_ = maxspeed_;
    char buf[MAX_PROCSTAT_LENGTH];
    std::ifstream stats(_statFileName);
    unsigned long one, two;

    if ( !stats )
        {
        std::cerr <<"Can not open file : " << _statFileName << std::endl;
        exit( 1 );
        }


    stats >> buf;
    // kernel >= 2.5
    while (!stats.eof() && strncmp(buf, "pgpgin", 7))
        {
        stats.ignore(1024, '\n');
        stats >> buf;
        }

    // read first value
    stats >> one;

    while (!stats.eof() && strncmp(buf, "pgpgout", 7))
        {
        stats.ignore(1024, '\n');
        stats >> buf;
        }

    // read second value
    stats >> two;

    updateinfo(one, two, 4);
    }

void DiskMeter::getdiskinfo( void )
    {
    IntervalTimerStop();
    total_ = maxspeed_;
    char buf[MAX_PROCSTAT_LENGTH];
    std::ifstream stats(_statFileName);

    if ( !stats )
        {
        std::cerr <<"Can not open file : " << _statFileName << std::endl;
        exit( 1 );
        }

    // Find the line with 'page'
    stats >> buf;
    while (strncmp(buf, "page", 9))
        {
        stats.ignore(MAX_PROCSTAT_LENGTH, '\n');
        stats >> buf;
        if (stats.eof())
            break;
        }

	// read values
    unsigned long one, two;
    stats >> one >> two;

    updateinfo(one, two, 2);
    }
