/*
    Copyright (C) 2000-2002  Tamas Pal <tpal@egon.gyaloglo.hu>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
/*
  goodnight.c - an XMMS plugin to stop playing at a given time and perform some 
  action
*/

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <time.h>
#include <pthread.h>
#include <glib.h>
#include <gtk/gtk.h>
#include <string.h>
#include "xmmsctrl.h"
#include "configfile.h"
#include "plugin.h"

#include "config.h"
#include "save_xmms.h"

#define CONFIGFILE "/.xmms/goodnight"

#define STOPSTR "Stop"
#define QUITSTR "Quit"
#define STANDBYSTR "Standby"
#define SHUTDOWNSTR "Shutdown"
#define ATTIME "At_time"
#define FROMNOW "From_now"
#define LASTSONG "Last_song"

static void *goodnight_thread(void *args);
void goodnight_init();
void goodnight_about();
void goodnight_config();
static void goodnight_config_save(GtkWidget * wid, gpointer data);
static void do_it_now(int vol_save);
static gint fade_down();

static void show_status_window();
static void update_status_window();
static void destroy_status_window(GtkWidget * wid, gpointer data);

static int active_toggle_click(GtkWidget * wid, gpointer data);
static void get_config_from_dialog(void);

static int debugmsg(const char *msg, ...);

enum {
    AT_TIME, FROM_NOW, LAST_SONG
};

enum {
    STOP, QUIT, STANDBY, SHUTDOWN
};

/* Windows */
static GtkWidget *conf_dialog, *about_win;
static GtkWidget *status_win;

/* static widgets needed for setup */
static GtkWidget *quit_radio, *shutdown_radio, *standby_radio, *stop_radio,
    *shutdown_entry, *standby_entry;
static GtkWidget *at_radio, *fn_radio, *lastsong_radio;
#ifdef DEBUG
static GtkWidget *debug_toggle;
#endif

static GtkWidget *status_label;
static GtkWidget *status_toggle;

static GtkWidget *active_next_toggle;
static GtkWidget *active_toggle;


/* thread */
static pthread_t tid;

GtkObject *athour_w, *atminute_w, *fadespeed_o;
GtkObject *fnhour_w, *fnminute_w;

/* configuration options */
gint athour = -1, atminute = -1, fnhour = -1, fnminute = -1, fadespeed =
    -1, timing_mode = AT_TIME, action_mode = STOP;
gchar *shutdown_command = NULL, *standby_command = NULL;
gboolean show_status = FALSE, debug = FALSE;
gboolean active = TRUE;
gboolean active_next_time = TRUE;

gchar *dummy = "";
time_t fn_time;

#ifdef DEBUG
#define debugm(msg,...) debugmsg(msg, ##__VA_ARGS__)
#else
#define debugm(msg,...)
#endif

/* Simple function to print debugging messages to stdout */
int debugmsg(const char *msg, ...)
{
    va_list args;

    time_t tim;
    struct tm ltime;
    static char stamp[30];
    if (debug == FALSE)
	return 1;

    memset(stamp, 0, 30);
    memset(&ltime, 0, sizeof(struct tm));
    tim = time(NULL);
    localtime_r(&tim, &ltime);
    strftime(stamp, 29, "%H:%M:%S ", &ltime);
    printf("%s", stamp);
    va_start(args, msg);
    vprintf(msg, args);
    fflush(stdout);
    return 1;
}

void goodnight_cleanup()
{
    destroy_status_window(NULL, NULL);
    pthread_cancel(tid);
}

GeneralPlugin goodnight = {
    NULL,
    NULL,
    -1,
#ifdef DEBUG
    "XMMS-Goodnight plugin development version " VERSION,
#else
    "XMMS-Goodnight plugin " VERSION,
#endif
    goodnight_init,
    goodnight_about,
    goodnight_config,
    goodnight_cleanup,
};

GeneralPlugin *get_gplugin_info()
{
    return (&goodnight);
}

static void goodnight_config_read(GtkWidget * wid, gpointer data)
{
    gchar *configfile, *action = NULL, *timing_mod = NULL;
    ConfigFile *config;

    configfile = g_strconcat(g_get_home_dir(), CONFIGFILE, NULL);
    if ((config = xmms_cfg_open_file(configfile)) != NULL) {
	xmms_cfg_read_int(config, "goodnight", "athour", &athour);
	xmms_cfg_read_int(config, "goodnight", "atminute", &atminute);
	xmms_cfg_read_int(config, "goodnight", "fnhour", &fnhour);
	xmms_cfg_read_int(config, "goodnight", "fnminute", &fnminute);
	xmms_cfg_read_int(config, "goodnight", "fadespeed", &fadespeed);
	xmms_cfg_read_string(config, "goodnight", "shutdown_command",
			     &shutdown_command);
	xmms_cfg_read_string(config, "goodnight", "standby_command",
			     &standby_command);

	xmms_cfg_read_string(config, "goodnight", "exit_action", &action);
	xmms_cfg_read_string(config, "goodnight", "timing_mode",
			     &timing_mod);
	xmms_cfg_read_boolean(config, "goodnight", "debug", &debug);
	xmms_cfg_read_boolean(config, "goodnight", "show_status",
			      &show_status);

	xmms_cfg_read_boolean(config, "goodnight", "active_next_time",
			      &active_next_time);
	debugm("goodnight_config_read(): active_next_time: %d\n",
	       active_next_time);

	xmms_cfg_free(config);
    }

    /* some "airbag"... */
    if ((athour < 0) || (athour > 23))
	athour = 0;
    if ((atminute < 0) || (atminute > 59))
	atminute = 0;
    if ((fnhour < 0) || (fnhour > 1000))
	fnhour = 0;
    if ((fnminute < 0) || (fnminute > 1000))
	fnminute = 0;
    if ((fadespeed < 1) || (fadespeed > 20))
	fadespeed = 2;

    if (!shutdown_command)
	shutdown_command = "";
    if (!standby_command)
	standby_command = "";
    if (action) {
	if (!strcasecmp(action, QUITSTR))
	    action_mode = QUIT;
	if (!strcasecmp(action, STANDBYSTR))
	    action_mode = STANDBY;
	if (!strcasecmp(action, SHUTDOWNSTR))
	    action_mode = SHUTDOWN;
	if (!strcasecmp(action, STOPSTR))
	    action_mode = STOP;
    }
    if (timing_mod) {
	if (!strcasecmp(timing_mod, ATTIME))
	    timing_mode = AT_TIME;
	if (!strcasecmp(timing_mod, FROMNOW))
	    timing_mode = FROM_NOW;
	if (!strcasecmp(timing_mod, LASTSONG))
	    timing_mode = LAST_SONG;
    }

    fn_time = time(NULL) + ((fnhour * 3600) + (fnminute * 60));

    active = active_next_time;
    debugm("goodnight_config_read(): active_next_time: %d, active: %d\n",
	   active_next_time, active);

    g_free(configfile);
}

void goodnight_init()
{
    pthread_attr_t attr;

    goodnight_config_read(NULL, NULL);

    pthread_attr_init(&attr);
    pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
    pthread_attr_setinheritsched(&attr, PTHREAD_EXPLICIT_SCHED);
    pthread_attr_setschedpolicy(&attr, SCHED_OTHER);
    pthread_attr_setscope(&attr, PTHREAD_SCOPE_SYSTEM);

    pthread_create(&tid, &attr, goodnight_thread, NULL);
    return;

}

static void *goodnight_thread(void *args)
{
    gint vol_save, playlist_len, playlist_act, lastsong_full_time,
	lastsong_act_time;
    struct tm t_now;
    time_t timenow;
    struct timespec mosec;
    gboolean end_last_song = FALSE;
    mosec.tv_nsec = 0;

    debugm("Yaw running....\n");
/* initial sleeping. I need it because the status window screwed up at startup 
   when some XMMS windows materialized on it. Don't ask me why ... */
    sleep(1);

    while (1) {
	timenow = time(NULL);

	if ((show_status == TRUE) && (status_win == NULL))
	    show_status_window();
	if ((show_status == FALSE) && (status_win != NULL))
	    destroy_status_window(NULL, NULL);

	if ((show_status == TRUE) && (status_win != NULL))
	    update_status_window();

	if (active == TRUE) {
	    switch (timing_mode) {

	    case AT_TIME:
		localtime_r(&timenow, &t_now);
		if (t_now.tm_hour == athour)
		    if (t_now.tm_min == atminute) {
			vol_save = fade_down();
			do_it_now(vol_save);
		    }
		mosec.tv_sec = 60;
		break;
	    case FROM_NOW:
		debugm
		    ("goodnight_thread(): fn_time:%ld, timenow:%ld difftime(fn_time, timenow)):,%g\n",
		     fn_time, timenow, difftime(fn_time, timenow));
		mosec.tv_sec = 1;
		if (timenow == fn_time) {
		    vol_save = fade_down();
		    do_it_now(vol_save);
		}
		if (timenow > fn_time)
		    active = FALSE;
		mosec.tv_sec = 1;
		break;

	    case LAST_SONG:
		/* quite lame but I hasn't got a better idea now ... */
		playlist_len =
		    xmms_remote_get_playlist_length(goodnight.
						    xmms_session);
		playlist_act =
		    xmms_remote_get_playlist_pos(goodnight.xmms_session);
		debugm
		    ("repeat:%d,playing:%d,playlist_len:%d,playlist_act:%d,end_last_song:%d\n",
		     xmms_remote_is_repeat(goodnight.xmms_session),
		     xmms_remote_is_playing(goodnight.xmms_session),
		     playlist_len, playlist_act, end_last_song);

		if ((xmms_remote_is_repeat(goodnight.xmms_session)) ==
		    TRUE) {
		    xmms_remote_toggle_repeat(goodnight.xmms_session);
		}
		if (((playlist_len - 1) == playlist_act)
		    ) {
		    lastsong_full_time =
			xmms_remote_get_playlist_time(goodnight.
						      xmms_session,
						      playlist_act)
			/ 1000;
		    lastsong_act_time =
			xmms_remote_get_output_time(goodnight.xmms_session)
			/ 1000;
		    debugm("full-act: %d-%d\n",
			   lastsong_full_time, lastsong_act_time);
		    if (lastsong_full_time - lastsong_act_time <= 1) {
			xmms_remote_set_playlist_pos(goodnight.
						     xmms_session, 0);
			vol_save =
			    xmms_remote_get_main_volume(goodnight.
							xmms_session);
			do_it_now(vol_save);
		    }
		}
		/* we just fall through to the default action...Just lazyness ...:) */
		/* !! FALL THROUGH !! */
	    default:
		mosec.tv_sec = 1;
		break;
	    }
	} else
	    mosec.tv_sec = 1;
	nanosleep(&mosec, NULL);
    }
}

static void show_status_window()
{
    if (status_win)
	destroy_status_window(NULL, NULL);
    debugm("show_status_window(): Showing status window...\n");

    status_win = gtk_window_new(GTK_WINDOW_DIALOG);
    gtk_window_set_title(GTK_WINDOW(status_win), ("Goonight status"));
    gtk_window_set_policy(GTK_WINDOW(status_win), FALSE, FALSE, FALSE);
    gtk_window_set_position(GTK_WINDOW(status_win), GTK_WIN_POS_NONE);

    gtk_container_set_border_width(GTK_CONTAINER(status_win), 5);

    status_label = gtk_label_new(NULL);
    gtk_container_add(GTK_CONTAINER(status_win), status_label);

    gtk_signal_connect(GTK_OBJECT(status_win), "destroy",
		       GTK_SIGNAL_FUNC(destroy_status_window),
		       &status_win);
    update_status_window();
    gtk_widget_show_all(status_win);
}

static void update_status_window()
{
    gchar *mode = NULL;
    gchar *act = NULL;
    gchar *act_m = NULL;
    gchar *label;
    time_t tnow;

    switch (timing_mode) {
    case AT_TIME:
	act = g_strdup_printf("at %02d:%02d", athour, atminute);
	break;
    case FROM_NOW:
	tnow = time(NULL);
	if (active == TRUE)
	    act =
		g_strdup_printf("in %02ld:%02ld", ((fn_time - tnow) / 60),
				(fn_time - tnow) % 60);
	else
	    act = g_strdup_printf("in %02d:00", (fnhour * 60 + fnminute));
	break;
    case LAST_SONG:
	act = g_strdup_printf("at last song");
	break;
    default:
	break;
    }

    switch (action_mode) {
    case STOP:
	mode = g_strdup_printf("Stop playing ");
	break;
    case QUIT:
	mode = g_strdup_printf("Quits ");
	break;
    case STANDBY:
	mode = g_strdup_printf("%s ", STANDBYSTR);
	break;
    case SHUTDOWN:
	mode = g_strdup_printf("%s ", SHUTDOWNSTR);
	break;
    default:
	break;
    }

    if (active == TRUE)
	act_m = g_strdup_printf("(active)");
    else
	act_m = g_strdup_printf("(inactive)");

    label = g_strdup_printf("%s%s%s", mode, act, act_m);
    gtk_label_set_text(GTK_LABEL(status_label), label);
    debugmsg("update_status_win().label:%s\n", label);
    g_free(act);
    g_free(mode);
    g_free(act_m);
    g_free(label);

}

void destroy_status_window(GtkWidget * wid, gpointer data)
{
    if (status_win) {
	debugm("destroy_status_window(): Closing status window...\n");
	gtk_widget_destroy(status_win);
	status_win = NULL;
	show_status = FALSE;
    }
}

static int active_toggle_click(GtkWidget * wid, gpointer data)
{
    active =
	gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(active_toggle));
    debugm("active_toggle_click(): active: %d\n", active);
    if (active == TRUE) {
	get_config_from_dialog();
	fn_time = time(NULL) + (fnhour * 3600) + (fnminute * 60);
    }
    return 0;
}

static void get_config_from_dialog(void)
{
    fadespeed = (gint) GTK_ADJUSTMENT(fadespeed_o)->value;
    if ((gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(standby_radio))) ==
	TRUE)
	action_mode = STANDBY;
    if ((gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(shutdown_radio)))
	== TRUE)
	action_mode = SHUTDOWN;
    if ((gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(quit_radio))) ==
	TRUE)
	action_mode = QUIT;
    if ((gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(stop_radio))) ==
	TRUE)
	action_mode = STOP;
    athour = (gint) GTK_ADJUSTMENT(athour_w)->value;
    atminute = (gint) GTK_ADJUSTMENT(atminute_w)->value;
    fnhour = (gint) GTK_ADJUSTMENT(fnhour_w)->value;
    fnminute = (gint) GTK_ADJUSTMENT(fnminute_w)->value;
    shutdown_command =
	g_strdup(gtk_entry_get_text(GTK_ENTRY(shutdown_entry)));
    standby_command =
	g_strdup(gtk_entry_get_text(GTK_ENTRY(standby_entry)));


//  printf("relative: %ld\nnow: %ld\n",(long)relative,(long)now);
    if ((gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(at_radio))) ==
	TRUE) {
	timing_mode = AT_TIME;
    }
    if ((gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(fn_radio))) ==
	TRUE) {
	timing_mode = FROM_NOW;
    }
    if ((gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lastsong_radio)))
	== TRUE) {
	timing_mode = LAST_SONG;
    }
#ifdef DEBUG
    debug = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(debug_toggle));
#endif

    show_status =
	gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(status_toggle));

    active_next_time =
	gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON
				     (active_next_toggle));
}

void goodnight_about()
{
    GtkWidget *button, *label, *bigbox, *buttonbox;

    if (about_win)
	return;
    about_win = gtk_window_new(GTK_WINDOW_DIALOG);
    gtk_window_set_title(GTK_WINDOW(about_win), ("About"));
    gtk_window_set_policy(GTK_WINDOW(about_win), FALSE, FALSE, FALSE);
    gtk_window_set_position(GTK_WINDOW(about_win), GTK_WIN_POS_MOUSE);

    bigbox = gtk_vbox_new(FALSE, 5);
    gtk_container_add(GTK_CONTAINER(about_win), bigbox);

    label =
	gtk_label_new
	("XMMS-Goodnight plugin \n\n Stops XMMS at the given time.\n"
	 "Created by Tamas Pal <folti@fiktiv.szgtikol.kando.hu>");
    gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_CENTER);
    gtk_container_add(GTK_CONTAINER(bigbox), label);

    buttonbox = gtk_hbutton_box_new();
    gtk_button_box_set_layout(GTK_BUTTON_BOX(buttonbox),
			      GTK_BUTTONBOX_DEFAULT_STYLE);
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(buttonbox), 5);
    gtk_box_pack_start(GTK_BOX(bigbox), buttonbox, FALSE, FALSE, 0);

    gtk_signal_connect(GTK_OBJECT(about_win), "destroy",
		       GTK_SIGNAL_FUNC(gtk_widget_destroyed), &about_win);

    button = gtk_button_new_with_label("Ok");
    GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
    gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
			      GTK_SIGNAL_FUNC(gtk_widget_destroy),
			      (gpointer) about_win);
    gtk_box_pack_start(GTK_BOX((buttonbox)), button, FALSE, TRUE, 5);

    gtk_widget_show_all(about_win);

}

void goodnight_config_save(GtkWidget * wid, gpointer data)
{
    gchar *configfile, *action = NULL, *timing_mod = NULL;
    ConfigFile *config;
    configfile = g_strconcat(g_get_home_dir(), CONFIGFILE, NULL);

    if ((config = xmms_cfg_open_file(configfile)) == NULL)
	config = xmms_cfg_new();

    get_config_from_dialog();
    switch (timing_mode) {
    case AT_TIME:
	timing_mod = ATTIME;
	break;
    case FROM_NOW:
	timing_mod = FROMNOW;
	break;
    case LAST_SONG:
	timing_mod = LASTSONG;
	break;
    }

    switch (action_mode) {
    case QUIT:
	action = QUITSTR;
	break;
    case SHUTDOWN:
	action = SHUTDOWNSTR;
	break;
    case STANDBY:
	action = STANDBYSTR;
	break;
    case STOP:
	action = STOPSTR;
	break;
    }
    xmms_cfg_write_int(config, "goodnight", "athour", athour);
    xmms_cfg_write_int(config, "goodnight", "atminute", atminute);
    xmms_cfg_write_int(config, "goodnight", "fnhour", fnhour);
    xmms_cfg_write_int(config, "goodnight", "fnminute", fnminute);
    xmms_cfg_write_int(config, "goodnight", "fadespeed", fadespeed);
    if (shutdown_command)
	xmms_cfg_write_string(config, "goodnight", "shutdown_command",
			      shutdown_command);
    if (standby_command)
	xmms_cfg_write_string(config, "goodnight", "standby_command",
			      standby_command);
    xmms_cfg_write_string(config, "goodnight", "timing_mode", timing_mod);
    xmms_cfg_write_string(config, "goodnight", "exit_action", action);
    xmms_cfg_write_boolean(config, "goodnight", "debug", debug);
    xmms_cfg_write_boolean(config, "goodnight", "show_status",
			   show_status);

    xmms_cfg_write_boolean(config, "goodnight", "active_next_time",
			   active_next_time);

    xmms_cfg_write_file(config, configfile);
    xmms_cfg_free(config);
    g_free(configfile);

    debugm("goodnight_config_save(): active_next_time: %d, active: %d\n",
	   active_next_time, active);
    return;
}

void goodnight_config_ok(GtkWidget * wid, gpointer data)
{
    goodnight_config_save(NULL, NULL);
    if (show_status == FALSE)
	destroy_status_window(NULL, NULL);
    gtk_widget_destroy(conf_dialog);
    conf_dialog = NULL;
    return;
}


void goodnight_config()
{
    GtkWidget *ok_button, *apply_button, *cancel_button;
    GtkWidget *attimebox, *athourfield, *atminutefield /*, *fadefield */ ;
    GtkWidget *fntimebox, *fnhourfield, *fnminutefield;
    GtkWidget *timtype_frame, *timtype_table, *fade_hbox;
    GtkWidget *exit_actions;
    GtkWidget *actionframe, *textframe, *fadeframe;
    GtkWidget *bigbox, *buttonbox, *commands_table, *shutd_label,
	*standby_label;
    GtkWidget *fadescale;
#ifdef DEBUG
#endif
    if ((athour == -1) || (atminute == -1) || (fnhour == -1)
	|| (fnminute == -1)) {
	goodnight_config_read(NULL, NULL);
    }

    if (conf_dialog)
	return;

    conf_dialog = gtk_window_new(GTK_WINDOW_DIALOG);

    gtk_window_set_title(GTK_WINDOW(conf_dialog),
			 ("XMMS-Goodnight setup"));
    gtk_window_set_policy(GTK_WINDOW(conf_dialog), FALSE, FALSE, FALSE);
    gtk_window_set_position(GTK_WINDOW(conf_dialog), GTK_WIN_POS_MOUSE);

    gtk_container_set_border_width(GTK_CONTAINER(conf_dialog), 5);

    gtk_signal_connect(GTK_OBJECT(conf_dialog), "destroy",
		       GTK_SIGNAL_FUNC(gtk_widget_destroyed),
		       &conf_dialog);

    bigbox = gtk_vbox_new(FALSE, 5);
    gtk_container_add(GTK_CONTAINER(GTK_WINDOW(conf_dialog)), bigbox);

    timtype_frame = gtk_frame_new("Timing actions");
    gtk_container_add(GTK_CONTAINER(bigbox), timtype_frame);

    timtype_table = gtk_table_new(3, 2, FALSE);
    gtk_container_add(GTK_CONTAINER(timtype_frame), timtype_table);


    at_radio = gtk_radio_button_new_with_label(NULL, "At:");
    gtk_table_attach_defaults(GTK_TABLE(timtype_table), at_radio, 0, 1, 0,
			      1);

    fn_radio =
	gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON
						    (at_radio),
						    "From now:");
    gtk_table_attach_defaults(GTK_TABLE(timtype_table), fn_radio, 0, 1, 1,
			      2);

    lastsong_radio =
	gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON
						    (at_radio),
						    "Last song");
    gtk_table_attach_defaults(GTK_TABLE(timtype_table), lastsong_radio, 0,
			      1, 2, 3);

    if (timing_mode == AT_TIME)
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(at_radio), TRUE);

    if (timing_mode == FROM_NOW)
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(fn_radio), TRUE);

    if (timing_mode == LAST_SONG)
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(lastsong_radio),
				     TRUE);

/* in this box we could set the "At" time (an exact time to action) */
    attimebox = gtk_hbox_new(FALSE, 5);
    gtk_container_set_border_width(GTK_CONTAINER(attimebox), 5);
    gtk_table_attach_defaults(GTK_TABLE(timtype_table), attimebox, 1, 2, 0,
			      1);

    athour_w = gtk_adjustment_new(athour, 0, 23, 1, 1, 1);
    athourfield = gtk_spin_button_new(GTK_ADJUSTMENT(athour_w), 1.0, 0);

    atminute_w = gtk_adjustment_new(atminute, 0, 59, 1, 1, 1);
    atminutefield =
	gtk_spin_button_new(GTK_ADJUSTMENT(atminute_w), 1.0, 0);

    gtk_box_pack_start(GTK_BOX(attimebox), gtk_label_new("Hour: "), FALSE,
		       FALSE, 0);
    gtk_box_pack_start(GTK_BOX(attimebox), athourfield, FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(attimebox), gtk_label_new("Minute: "),
		       FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(attimebox), atminutefield, FALSE, FALSE, 0);

/* "from now" timebox this time is relative to the setting */
    fntimebox = gtk_hbox_new(FALSE, 5);
    gtk_container_set_border_width(GTK_CONTAINER(fntimebox), 5);
    gtk_table_attach_defaults(GTK_TABLE(timtype_table), fntimebox, 1, 2, 1,
			      2);

    fnhour_w = gtk_adjustment_new(fnhour, 0, 999, 1, 1, 1);
    fnhourfield = gtk_spin_button_new(GTK_ADJUSTMENT(fnhour_w), 1.0, 0);

    fnminute_w = gtk_adjustment_new(fnminute, 0, 999, 1, 1, 1);
    fnminutefield =
	gtk_spin_button_new(GTK_ADJUSTMENT(fnminute_w), 1.0, 0);

    gtk_box_pack_start(GTK_BOX(fntimebox), gtk_label_new("Hour: "), FALSE,
		       FALSE, 0);
    gtk_box_pack_start(GTK_BOX(fntimebox), fnhourfield, FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(fntimebox), gtk_label_new("Minute: "),
		       FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(fntimebox), fnminutefield, FALSE, FALSE, 0);

/* fade setting frame */
    fadeframe = gtk_frame_new(NULL);
    gtk_container_add(GTK_CONTAINER(bigbox), fadeframe);

    fade_hbox = gtk_hbox_new(FALSE, 5);
    gtk_container_add(GTK_CONTAINER(fadeframe), fade_hbox);

    fadespeed_o = gtk_adjustment_new(fadespeed, 1, 20, 1.0, 5.0, 1.0);
    fadescale = gtk_spin_button_new(GTK_ADJUSTMENT(fadespeed_o), 1.0, 0);

    gtk_box_pack_start(GTK_BOX(fade_hbox), gtk_label_new("Fade speed:"),
		       FALSE, FALSE, 10);
    gtk_box_pack_start(GTK_BOX(fade_hbox), fadescale, FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(fade_hbox), gtk_label_new("%/sec"), FALSE,
		       FALSE, 0);

/* frame for the action types */
    actionframe = gtk_frame_new("Exit actions:");
    gtk_container_add(GTK_CONTAINER(bigbox), actionframe);

    exit_actions = gtk_vbox_new(FALSE, 0);
    gtk_container_set_border_width(GTK_CONTAINER(exit_actions), 5);

    gtk_container_add(GTK_CONTAINER(actionframe), exit_actions);

    shutdown_radio =
	gtk_radio_button_new_with_label(NULL, "Shutdown computer");
    gtk_box_pack_start(GTK_BOX(exit_actions), shutdown_radio, TRUE, TRUE,
		       0);

    standby_radio =
	gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON
						    (shutdown_radio),
						    "Standby");
    gtk_box_pack_start(GTK_BOX(exit_actions), standby_radio, TRUE, TRUE,
		       0);

    quit_radio =
	gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON
						    (shutdown_radio),
						    "Exit XMMS");
    gtk_box_pack_start(GTK_BOX(exit_actions), quit_radio, TRUE, TRUE, 0);

    stop_radio =
	gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON
						    (shutdown_radio),
						    "Stop playing");
    gtk_box_pack_start(GTK_BOX(exit_actions), stop_radio, TRUE, TRUE, 0);

    if ((action_mode == SHUTDOWN))
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(shutdown_radio),
				     TRUE);

    if ((action_mode == STANDBY))
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(standby_radio),
				     TRUE);

    if ((action_mode == QUIT))
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(quit_radio), TRUE);

    if ((action_mode == STOP))
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(stop_radio), TRUE);

/* frame to set the external commands for shutdown and standby */
    textframe = gtk_frame_new("Commands(executed by a system() call!):");
    gtk_container_add(GTK_CONTAINER(bigbox), textframe);
    commands_table = gtk_table_new(2, 2, FALSE);
    gtk_container_add(GTK_CONTAINER(textframe), commands_table);

    shutd_label = gtk_label_new("Shutdown command:");
    standby_label = gtk_label_new("Standby command:");

    gtk_table_attach_defaults(GTK_TABLE(commands_table), shutd_label, 0, 1,
			      0, 1);
    gtk_table_attach_defaults(GTK_TABLE(commands_table), standby_label, 0,
			      1, 1, 2);

    shutdown_entry = gtk_entry_new();
    gtk_entry_set_text(GTK_ENTRY(shutdown_entry), shutdown_command);
    standby_entry = gtk_entry_new();
    gtk_entry_set_text(GTK_ENTRY(standby_entry), standby_command);

    gtk_table_attach_defaults(GTK_TABLE(commands_table), shutdown_entry, 1,
			      2, 0, 1);
    gtk_table_attach_defaults(GTK_TABLE(commands_table), standby_entry, 1,
			      2, 1, 2);
#ifdef DEBUG
    debug_toggle =
	gtk_check_button_new_with_label("Print debug messages to stdout");
    gtk_container_add(GTK_CONTAINER(bigbox), debug_toggle);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(debug_toggle), debug);
#endif
/* Plugin status window's check button */
    status_toggle =
	gtk_check_button_new_with_label("Show plugin status in window");
    gtk_container_add(GTK_CONTAINER(bigbox), status_toggle);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(status_toggle),
				 show_status);
    active_next_toggle =
	gtk_check_button_new_with_label("Activate plugin next time");
    gtk_container_add(GTK_CONTAINER(bigbox), active_next_toggle);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(active_next_toggle),
				 active_next_time);


    active_toggle = gtk_toggle_button_new_with_label("Active");
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(active_toggle), active);
    gtk_signal_connect_object(GTK_OBJECT(active_toggle), "clicked",
			      GTK_SIGNAL_FUNC(active_toggle_click), NULL);
    gtk_container_add(GTK_CONTAINER(bigbox), active_toggle);

/* three default buttons at the bottom of the window :) */
    buttonbox = gtk_hbutton_box_new();
    gtk_button_box_set_layout(GTK_BUTTON_BOX(buttonbox),
			      GTK_BUTTONBOX_END);
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(buttonbox), 5);
    gtk_box_pack_start(GTK_BOX(bigbox), buttonbox, FALSE, FALSE, 0);

    ok_button = gtk_button_new_with_label("Ok");
    apply_button = gtk_button_new_with_label("Apply");
    cancel_button = gtk_button_new_with_label("Cancel");

    gtk_signal_connect_object(GTK_OBJECT(cancel_button), "clicked",
			      GTK_SIGNAL_FUNC(gtk_widget_destroy),
			      (gpointer) conf_dialog);

    gtk_signal_connect_object(GTK_OBJECT(apply_button), "clicked",
			      GTK_SIGNAL_FUNC(goodnight_config_save),
			      NULL);
    gtk_signal_connect_object(GTK_OBJECT(ok_button), "clicked",
			      GTK_SIGNAL_FUNC(goodnight_config_ok), NULL);

    GTK_WIDGET_SET_FLAGS(ok_button, GTK_CAN_DEFAULT);
    GTK_WIDGET_SET_FLAGS(cancel_button, GTK_CAN_DEFAULT);
    GTK_WIDGET_SET_FLAGS(apply_button, GTK_CAN_DEFAULT);


    gtk_box_pack_start(GTK_BOX(buttonbox), ok_button, TRUE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(buttonbox), cancel_button, TRUE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(buttonbox), apply_button, TRUE, TRUE, 0);

    gtk_widget_show_all(conf_dialog);

}

void do_it_now(int vol_save)
{
    struct timespec mosec;
    mosec.tv_nsec = 0;

    debugm("do_it_now()...\n");

    xmms_remote_stop(goodnight.xmms_session);
    save_cfgfile(goodnight.xmms_session);
    save_playlist(goodnight.xmms_session);
    mosec.tv_sec = 1;
    nanosleep(&mosec, NULL);	/* wait one sec before resetting volume */
    xmms_remote_set_main_volume(goodnight.xmms_session, vol_save);
    if (action_mode == QUIT) {
	debugm("Trying to quit...\n");
#ifdef HAVE_XMMS_REMOTE_QUIT
	xmms_remote_quit(goodnight.xmms_session);
#else
	/* a bit brute but works... */
	raise(SIGTERM);
#endif
    }
    if ((action_mode == SHUTDOWN) && (shutdown_command))
	system(shutdown_command);
    if ((action_mode == STANDBY) && (standby_command))
	system(standby_command);

    active = FALSE;
}

gint fade_down()
{
    struct timespec tsec;
    gint volume;
    gint vol_start = volume =
	xmms_remote_get_main_volume(goodnight.xmms_session);

    debugm("fade_down(): Fading down ...\n");

    tsec.tv_nsec = 0;
    while (volume >= 10) {
	volume -= fadespeed;
	xmms_remote_set_main_volume(goodnight.xmms_session, volume);
	tsec.tv_sec = 1;
	nanosleep(&tsec, NULL);
    }
    return vol_start;
}
