// *************************************************************************
// * Xgsm - mobile phone manager
// *
// * File:    xgsm_pref.h
// *
// * Purpose: Preferences editor
// *
// * Author:  Peter Hofmann (software@pxh.de)
// *
// * Created: 28.5.2000
// *************************************************************************

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "xgsm_pref.h"
#include "xgsm_util.h"
#include "xgsm_main.h"

#include <gtk--/menu.h>
#include <gtk--/label.h>
#include <gtk--/entry.h>
#include <gtk--/optionmenu.h>
#include <gtk--/menuitem.h>
#include <gnome--/file-entry.h>
#include <gnome--/dialog.h>
#include <sigc++/signal_system.h>
#include <gnome.h>
#include <gsmlib/gsm_error.h>
#include <termios.h>
#include <typeinfo>
#include <iostream>

extern "C" {
#include "interface.h"
#include "support.h"
}

using namespace std;
using namespace Xgsm;
using namespace gsmlib;

// Configuration data tables

static char *dataSourceToStr[] = {"from_file", "from_device"};
static char *sortOrderToStr[] =
{"by_text", "by_telephone", "by_index", "by_date"};

// Configuration members

DataSource Configuration::checkDataSource(string title, string dataSource)
{
  dataSource = lowercase(dataSource);
  if (dataSource == "from_file")
    return FromFile;
  else if (dataSource == "from_device")
    return FromDevice;
  else
  {
    mainWindow->message(title + ": " +
                        stringPrintf(_("Invalid data source '%s'"),
                        dataSource.c_str()), true);
    _loadError = true;
    return FromFile;
  }
}

SortOrder Configuration::checkSortOrder(string title, string sortOrder,
                                        bool isSMS)
{
  sortOrder = lowercase(sortOrder);
  if (sortOrder == "by_text")
    return ByText;
  if (sortOrder == "by_telephone")
    return ByTelephone;
  if (sortOrder == "by_index")
    return ByIndex;
  if (sortOrder == "by_date" && isSMS)
    return ByDate;
  else
  {
    mainWindow->message(title + ": " +
                        stringPrintf(_("Invalid sort order '%s'"),
                        sortOrder.c_str()), true);
    _loadError = true;
    return ByIndex;
  }
}

Configuration::Configuration()
{
  // build baud rate option menu
  
}
  
void Configuration::load(bool loadDialogSizes) throw(GsmException)
{
  _loadError = false;

  // device page
  gnome_config_push_prefix("/" XGSM_CONFIG_PATH "/device/");
  gchar *tmpc = gnome_config_get_string("phone_device=/dev/mobilephone");
  _phoneDevice = tmpc;
  g_free(tmpc);

  tmpc = gnome_config_get_string("baud_rate=38400");
  _baudRate = tmpc;
  g_free(tmpc);

  if (_baudRate != "300" &&
      _baudRate != "600" &&
      _baudRate != "1200" &&
      _baudRate != "2400" &&
      _baudRate != "4800" &&
      _baudRate != "9600" &&
      _baudRate != "19200" &&
      _baudRate != "38400" &&
#ifdef B57600
      _baudRate != "57600" &&
#endif
#ifdef B115200
      _baudRate != "115200" &&
#endif
#ifdef B230400
      _baudRate != "230400" &&
#endif
#ifdef B460800
      _baudRate != "460800"
#endif
      )
  {
    mainWindow->message(
      stringPrintf(_("Invalid baudrate '%s'. Setting to 38400."),
                   _baudRate.c_str()), true);
    _baudRate = "38400";
    _loadError = true;
  }

  _softwareHandshake = gnome_config_get_bool("software_handshake=false");

  tmpc = gnome_config_get_string("init_string=E0");
  _initString = tmpc;
  g_free(tmpc);
  
  tmpc = gnome_config_get_string("character_set=GSM");
  _characterSet = tmpc;
  g_free(tmpc);

  _characterEncoding = gnome_config_get_int("character_encoding=0");
  if (_characterEncoding < DCS_DEFAULT_ALPHABET ||
      _characterEncoding > DCS_SIXTEEN_BIT_ALPHABET)
  {
    mainWindow->message(
      stringPrintf(_("Invalid character encoding '%d'. Setting to 7-bit."),
                   _characterEncoding), true);
    _characterEncoding = DCS_DEFAULT_ALPHABET;
    _loadError = true;
  }

  gnome_config_pop_prefix();
  
  // phonebooks page
  gnome_config_push_prefix("/" XGSM_CONFIG_PATH "/phonebooks_left/");
  tmpc = gnome_config_get_string("data_source=from_device");
  string tmps = tmpc;
  g_free(tmpc);
  _phoneEditorDefault[0]._dataSource =
    checkDataSource(_("Left Phonebook"), tmps);
  
  tmpc = gnome_config_get_string("store_name=SM");
  _phoneEditorDefault[0]._storeName = tmpc;
  g_free(tmpc);

  tmpc = gnome_config_get_string("sort_order=by_text");
  tmps = tmpc;
  g_free(tmpc);
  _phoneEditorDefault[0]._sortOrder =
    checkSortOrder(_("Left Phonebook"), tmps, false);
  gnome_config_pop_prefix();

  gnome_config_push_prefix("/" XGSM_CONFIG_PATH "/phonebooks_right/");
  tmpc = gnome_config_get_string("data_source=from_file");
  tmps = tmpc;
  g_free(tmpc);
  _phoneEditorDefault[1]._dataSource =
    checkDataSource(_("Right Phonebook"), tmps);
  
  tmpc = gnome_config_get_string("store_name=SM");
  _phoneEditorDefault[1]._storeName = tmpc;
  g_free(tmpc);

  tmpc = gnome_config_get_string("sort_order=by_text");
  tmps = tmpc;
  g_free(tmpc);
  _phoneEditorDefault[1]._sortOrder =
    checkSortOrder(_("Right Phonebook"), tmps, false);
  gnome_config_pop_prefix();

  // SMS stores page
  gnome_config_push_prefix("/" XGSM_CONFIG_PATH "/sms_stores_left/");
  tmpc = gnome_config_get_string("data_source=from_device");
  tmps = tmpc;
  g_free(tmpc);
  _smsEditorDefault[0]._dataSource =
    checkDataSource(_("Left SMS Stores"), tmps);
  
  tmpc = gnome_config_get_string("store_name=SM");
  _smsEditorDefault[0]._storeName = tmpc;
  g_free(tmpc);

  tmpc = gnome_config_get_string("sort_order=by_date");
  tmps = tmpc;
  g_free(tmpc);
  _smsEditorDefault[0]._sortOrder =
    checkSortOrder(_("Left SMS Stores"), tmps, true);
  gnome_config_pop_prefix();

  gnome_config_push_prefix("/" XGSM_CONFIG_PATH "/sms_stores_right/");
  tmpc = gnome_config_get_string("data_source=from_file");
  tmps = tmpc;
  g_free(tmpc);
  _smsEditorDefault[1]._dataSource =
    checkDataSource(_("Right SMS Stores"), tmps);
  
  tmpc = gnome_config_get_string("store_name=SM");
  _smsEditorDefault[1]._storeName = tmpc;
  g_free(tmpc);

  tmpc = gnome_config_get_string("sort_order=by_date");
  tmps = tmpc;
  g_free(tmpc);
  _smsEditorDefault[1]._sortOrder =
    checkSortOrder(_("Right SMS Stores"), tmps, true);
  gnome_config_pop_prefix();

  // SMS page
  gnome_config_push_prefix("/" XGSM_CONFIG_PATH "/sms/");
  _showLogWindow = gnome_config_get_bool("show_log_window=false");

  _longSMS = gnome_config_get_bool("long_sms=false");

  tmpc = gnome_config_get_string("sent_sms_color=blue");
  _sentSMSColor = tmpc;
  g_free(tmpc);

  tmpc = gnome_config_get_string("received_sms_color=black");
  _receivedSMSColor = tmpc;
  g_free(tmpc);
  
  tmpc = gnome_config_get_string("long_sms_color=blue");
  _longSMSColor = tmpc;
  g_free(tmpc);

  _incomingSMSNotification =
    gnome_config_get_bool("incoming_sms_notification=false");

  gnome_config_pop_prefix();

  // gui page
  gnome_config_push_prefix("/" XGSM_CONFIG_PATH "/gui/");
  tmpc = gnome_config_get_string("warning_color=red");
  _warningColor = tmpc;
  g_free(tmpc);

  _multipleListSelectionStyle =
    gnome_config_get_bool("list_multiple_selection=false");
  
  // load dialog sizes
  if (loadDialogSizes)
  {
    _mainWidth = gnome_config_get_int("main_width=-1");
    _mainHeight = gnome_config_get_int("main_height=-1");

    _pbWidth = gnome_config_get_int("pb_width=-1");
    _pbHeight = gnome_config_get_int("pb_height=-1");

    _smsWidth = gnome_config_get_int("sms_width=-1");
    _smsHeight = gnome_config_get_int("sms_height=-1");

    _newSmsWidth = gnome_config_get_int("new_sms_width=-1");
    _newSmsHeight = gnome_config_get_int("new_sms_height=-1");

    _chatSmsWidth = gnome_config_get_int("chat_sms_width=-1");
    _chatSmsHeight = gnome_config_get_int("chat_sms_height=-1");

    _incomingSmsWidth = gnome_config_get_int("incoming_sms_width=-1");
    _incomingSmsHeight = gnome_config_get_int("incoming_sms_height=-1");

    _smsPanedPosition = gnome_config_get_int("sms_paned_position=-1");
  }
  gnome_config_pop_prefix();

  if (_loadError)
  {
    // set all variables that could have caused the _loadError
    setPhonebookSortOrder(Left, _phoneEditorDefault[0]._sortOrder);
    setPhonebookSortOrder(Right, _phoneEditorDefault[1]._sortOrder);
    setPhonebookDataSource(Left, _phoneEditorDefault[0]._dataSource);
    setPhonebookDataSource(Right, _phoneEditorDefault[1]._dataSource);
    setSMSSortOrder(Left, _smsEditorDefault[0]._sortOrder);
    setSMSSortOrder(Right, _smsEditorDefault[1]._sortOrder);
    setSMSDataSource(Left, _smsEditorDefault[0]._dataSource);
    setSMSDataSource(Right, _smsEditorDefault[1]._dataSource);

    throw GsmException(_("Errors when parsing\nconfiguration file"),
                       ParameterError);
  }
}

void Configuration::sync()
{
  gnome_config_sync();
}

void Configuration::drop()
{
  gnome_config_drop_all();
  load(false);                  // restore previous values
}

void Configuration::setPhoneDevice(string phoneDevice)
{
  _phoneDevice = phoneDevice;
  gnome_config_set_string("/" XGSM_CONFIG_PATH "/device/phone_device",
                          phoneDevice.c_str());
}

void Configuration::setBaudRate(string baudRate)
{
  _baudRate = baudRate;
  gnome_config_set_string("/" XGSM_CONFIG_PATH "/device/baud_rate",
                          baudRate.c_str());
}

void Configuration::setSoftwareHandshake(bool softwareHandshake)
{
  _softwareHandshake = softwareHandshake;
  gnome_config_set_bool("/" XGSM_CONFIG_PATH "/device/software_handshake",
                        softwareHandshake);
}

void Configuration::setInitString(string initString)
{
  _initString = initString;
  gnome_config_set_string("/" XGSM_CONFIG_PATH "/device/init_string",
                          initString.c_str());
}

void Configuration::setCharacterSet(string characterSet)
{
  _characterSet = characterSet;
  gnome_config_set_string("/" XGSM_CONFIG_PATH "/device/character_set",
                          characterSet.c_str());
}

void Configuration::setCharacterEncoding(int characterEncoding)
{
  _characterEncoding = characterEncoding;
  gnome_config_set_int("/" XGSM_CONFIG_PATH "/device/character_encoding",
                       characterEncoding);
}

void Configuration::setPhonebookDataSource(LeftOrRight lr,
                                           DataSource dataSource)
{
  _phoneEditorDefault[(int)lr]._dataSource = dataSource;
  if (lr == Left)
    gnome_config_set_string("/" XGSM_CONFIG_PATH
                            "/phonebooks_left/data_source",
                            dataSourceToStr[dataSource]);
  else
    gnome_config_set_string("/" XGSM_CONFIG_PATH
                            "/phonebooks_right/data_source",
                            dataSourceToStr[dataSource]);
}

void Configuration::setPhonebookStoreName(LeftOrRight lr, string storeName)
{
  _phoneEditorDefault[(int)lr]._storeName = storeName;
  if (lr == Left)
    gnome_config_set_string("/" XGSM_CONFIG_PATH
                            "/phonebooks_left/store_name",
                            storeName.c_str());
  else
    gnome_config_set_string("/" XGSM_CONFIG_PATH
                            "/phonebooks_right/store_name",
                            storeName.c_str());
}
    
void Configuration::setPhonebookSortOrder(LeftOrRight lr, SortOrder sortOrder)
{
  _phoneEditorDefault[(int)lr]._sortOrder = sortOrder;
  if (lr == Left)
    gnome_config_set_string("/" XGSM_CONFIG_PATH
                            "/phonebooks_left/sort_order",
                            sortOrderToStr[sortOrder]);
  else
    gnome_config_set_string("/" XGSM_CONFIG_PATH
                            "/phonebooks_right/sort_order",
                            sortOrderToStr[sortOrder]);
}

void Configuration::setSMSDataSource(LeftOrRight lr, DataSource dataSource)
{
  _smsEditorDefault[(int)lr]._dataSource = dataSource;
  if (lr == Left)
    gnome_config_set_string("/" XGSM_CONFIG_PATH
                            "/sms_stores_left/data_source",
                            dataSourceToStr[dataSource]);
  else
    gnome_config_set_string("/" XGSM_CONFIG_PATH
                            "/sms_stores_right/data_source",
                            dataSourceToStr[dataSource]);
}

void Configuration::setSMSStoreName(LeftOrRight lr, string storeName)
{
  _smsEditorDefault[(int)lr]._storeName = storeName;
  if (lr == Left)
    gnome_config_set_string("/" XGSM_CONFIG_PATH
                            "/sms_stores_left/store_name",
                            storeName.c_str());
  else
    gnome_config_set_string("/" XGSM_CONFIG_PATH
                            "/sms_stores_right/store_name",
                            storeName.c_str());
}
    
void Configuration::setSMSSortOrder(LeftOrRight lr, SortOrder sortOrder)
{
  _smsEditorDefault[(int)lr]._sortOrder = sortOrder;
  if (lr == Left)
    gnome_config_set_string("/" XGSM_CONFIG_PATH
                            "/sms_stores_left/sort_order",
                            sortOrderToStr[sortOrder]);
  else
    gnome_config_set_string("/" XGSM_CONFIG_PATH
                            "/sms_stores_right/sort_order",
                            sortOrderToStr[sortOrder]);
}

void Configuration::setVoice(bool voice)
{
  // FIXME
}

void Configuration::setFax(bool voice)
{
  // FIXME
}

void Configuration::setData(bool voice)
{
  // FIXME
}

void Configuration::setShowLogWindow(bool showLogWindow)
{
  _showLogWindow = showLogWindow;
  gnome_config_set_bool("/" XGSM_CONFIG_PATH "/sms/show_log_window",
                        showLogWindow);
}

void Configuration::setLongSMS(bool longSMS)
{
  _longSMS = longSMS;
  gnome_config_set_bool("/" XGSM_CONFIG_PATH "/sms/long_sms",
                        longSMS);
}

void Configuration::setSentSMSColor(string sentSMSColor)
{
  _sentSMSColor = sentSMSColor;
  gnome_config_set_string("/" XGSM_CONFIG_PATH "/sms/sent_sms_color",
                          sentSMSColor.c_str());
}

void Configuration::setReceivedSMSColor(string receivedSMSColor)
{
  _receivedSMSColor = receivedSMSColor;
  gnome_config_set_string("/" XGSM_CONFIG_PATH "/sms/received_sms_color",
                          receivedSMSColor.c_str());
}

void Configuration::setLongSMSColor(string longSMSColor)
{
  _longSMSColor = longSMSColor;
  gnome_config_set_string("/" XGSM_CONFIG_PATH "/sms/long_sms_color",
                          longSMSColor.c_str());
}

void Configuration::setIncomingSMSNotification(bool incomingSMSNotification)
{
  _incomingSMSNotification = incomingSMSNotification;
  gnome_config_set_bool("/" XGSM_CONFIG_PATH "/sms/incoming_sms_notification",
                        incomingSMSNotification);
}

void Configuration::setWarningColor(string warningColor)
{
  _warningColor = warningColor;
  gnome_config_set_string("/" XGSM_CONFIG_PATH "/gui/warning_color",
                          _warningColor.c_str());
}

void Configuration::
setMultipleListSelectionStyle(bool multipleListSelectionStyle)
{
  _multipleListSelectionStyle = multipleListSelectionStyle;
  gnome_config_set_bool("/" XGSM_CONFIG_PATH "/gui/list_multiple_selection",
                        multipleListSelectionStyle);
}

bool Configuration::getMainSizes(int &width, int &height)
{
  width = _mainWidth;
  height = _mainHeight;
  return _mainWidth != -1 && _mainHeight != -1;
}

void Configuration::setMainSizes(int width, int height)
{
  _mainWidth = width;
  _mainHeight = height;
}
    
bool Configuration::getPbSizes(int &width, int &height)
{
  width = _pbWidth;
  height = _pbHeight;
  return _pbWidth != -1 && _pbHeight != -1;
}

void Configuration::setPbSizes(int width, int height)
{
  _pbWidth = width;
  _pbHeight = height;
}
    
bool Configuration::getSmsSizes(int &width, int &height)
{
  width = _smsWidth;
  height = _smsHeight;
  return _smsWidth != -1 && _smsHeight != -1;
}

void Configuration::setSmsSizes(int width, int height)
{
  _smsWidth = width;
  _smsHeight = height;
}
    
bool Configuration::getNewSmsSizes(int &width, int &height)
{
  width = _newSmsWidth;
  height = _newSmsHeight;
  return _newSmsWidth != -1 && _newSmsHeight != -1;
}

void Configuration::setNewSmsSizes(int width, int height)
{
  _newSmsWidth = width;
  _newSmsHeight = height;
}
    
bool Configuration::getChatSmsSizes(int &width, int &height)
{
  width = _chatSmsWidth;
  height = _chatSmsHeight;
  return _chatSmsWidth != -1 && _chatSmsHeight != -1;
}

void Configuration::setChatSmsSizes(int width, int height)
{
  _chatSmsWidth = width;
  _chatSmsHeight = height;
}
    
bool Configuration::getIncomingSmsSizes(int &width, int &height)
{
  width = _incomingSmsWidth;
  height = _incomingSmsHeight;
  return _incomingSmsWidth != -1 && _incomingSmsHeight != -1;
}

void Configuration::setIncomingSmsSizes(int width, int height)
{
  _incomingSmsWidth = width;
  _incomingSmsHeight = height;
}

bool Configuration::getSmsPanedPosition(int &position)
{
  position = _smsPanedPosition;
  return _smsPanedPosition != -1;
}

void Configuration::setSmsPanedPosition(int position)
{
  _smsPanedPosition = position;
}
    
Configuration::~Configuration()
{
  // drop pending changes (user has terminated xgsm while preferences window
  // was open)
  gnome_config_drop_all();

  gnome_config_set_int("/" XGSM_CONFIG_PATH "/gui/main_width", _mainWidth);
  gnome_config_set_int("/" XGSM_CONFIG_PATH "/gui/main_height", _mainHeight);

  gnome_config_set_int("/" XGSM_CONFIG_PATH "/gui/pb_width", _pbWidth);
  gnome_config_set_int("/" XGSM_CONFIG_PATH "/gui/pb_height", _pbHeight);

  gnome_config_set_int("/" XGSM_CONFIG_PATH "/gui/sms_width", _smsWidth);
  gnome_config_set_int("/" XGSM_CONFIG_PATH "/gui/sms_height", _smsHeight);

  gnome_config_set_int("/" XGSM_CONFIG_PATH "/gui/new_sms_width",
                       _newSmsWidth);
  gnome_config_set_int("/" XGSM_CONFIG_PATH "/gui/new_sms_height",
                       _newSmsHeight);

  gnome_config_set_int("/" XGSM_CONFIG_PATH "/gui/chat_sms_width",
                       _chatSmsWidth);
  gnome_config_set_int("/" XGSM_CONFIG_PATH "/gui/chat_sms_height",
                       _chatSmsHeight);

  gnome_config_set_int("/" XGSM_CONFIG_PATH "/gui/incoming_sms_width",
                       _incomingSmsWidth);
  gnome_config_set_int("/" XGSM_CONFIG_PATH "/gui/incoming_sms_height",
                       _incomingSmsHeight);

  gnome_config_set_int("/" XGSM_CONFIG_PATH "/gui/sms_paned_position",
                       _smsPanedPosition);

  sync();
}

// Configuration global var

Configuration Xgsm::config;

// Preferences members

bool Preferences::checkColor(string color)
{
  if (Gdk_Color().parse(color))
    return true;
  else
  {
    Gnome::Dialogs::error(
      gsmlib::stringPrintf(_("Error: '%s' is not a valid color specification"),
                           color.c_str()))->run_and_close();
    return false;
  }
}

void Preferences::onInfoResponse(ResponseRef response)
{
  if (typeid(response()) == typeid(ErrorResponse))
  {
    // handle error response
    if (config.getPhoneDevice() != "")
      mainWindow->message(
        (gsmlib::stringPrintf(_("Error reading charsets from device %s\n("),
                              config.getPhoneDevice().c_str()) +
         ((ErrorResponse&)response()).exception().what() + ")"), true);
    onClose();
  }
  else
  {
    assert(typeid(response()) == typeid(DeviceInfoResponse));
    vector<string> charSets =
      ((DeviceInfoResponse&)response()).getSupportedCharSets();
    _characterSetMenu.clear();
    for (vector<string>::iterator i = charSets.begin();
         i != charSets.end(); ++i)
    {
      cout << "Adding charset " << *i << endl;
      _characterSetMenu.append(*i, config.getCharacterSet(),
                               slot(this, &Preferences::onCharacterSetSet));
    }
  }
}

void Preferences::onOpenDone(ResponseRef response)
{
  if (typeid(response()) == typeid(ErrorResponse))
  {
    // handle error response
    if (config.getPhoneDevice() != "")
      mainWindow->message(
        (gsmlib::stringPrintf(_("Error opening %s\n("),
                              config.getPhoneDevice().c_str()) +
         ((ErrorResponse&)response()).exception().what() + ")"), true);
    onClose();
  }
  else
  {
    DeviceHelper::onOpenDone(response);
    cout << "Sending info request" << endl;
    RequestRef infoRequest =
      new DeviceInfoRequest(DeviceInfoRequest::CharSetInfo,
                            slot(this, &Preferences::onInfoResponse));
    _dev->request(infoRequest);
  }
}

void Preferences::onPhoneDevice()
{
  DEBUG_START(1);
  DEBUG_END;
  if (config.getPhoneDevice() != _phoneDeviceEntry->get_text())
  {
    config.setPhoneDevice(_phoneDeviceEntry->get_text());
    changed();
  }
}

void Preferences::onBaudRateSet()
{
  string baudRate = _baudRateMenu.getActiveChoice();
  assert(baudRate != "");
  if (config.getBaudRate() != baudRate)
  {
    config.setBaudRate(baudRate);
    changed();
  }
}

void Preferences::onSoftwareHandshake()
{
  bool newState = _softwareHandshakeButton->get_active();
  if (newState != config.getSoftwareHandshake())
  {
    config.setSoftwareHandshake(newState);
    changed();
  }
}

void Preferences::onInitString()
{
  string newInitString =
    ((Gtk::Entry*)_initStringEntry->gtk_entry())->get_text();
  if (config.getInitString() != newInitString)
  {
    config.setInitString(newInitString);
    changed();
  }
}

void Preferences::onCharacterSetSet()
{
  string characterSet = _characterSetMenu.getActiveChoice();
  assert(characterSet != "");
  if (config.getCharacterSet() != characterSet)
  {
    config.setCharacterSet(characterSet);
    changed();
  }
}

void Preferences::onCharacterEncodingSet()
{
  int characterEncoding = getActivatedMenuItem(_characterEncodingMenu);
  if (config.getCharacterEncoding() != characterEncoding)
  {
    config.setCharacterEncoding(characterEncoding);
    changed();
  }
}

void Preferences::onPbLeftFileOrDevice()
{
  DataSource newSource =
    (DataSource)getActivatedMenuItem(_pbLeftFileOrDevice);
  if (config.getPhonebookDataSource(Left) != newSource)
  {
    config.setPhonebookDataSource(Left, newSource);
    changed();
  }
}

void Preferences::onPbLeftName()
{
  string newName =
    ((Gtk::Entry*)_pbLeftPhonebookName->gtk_entry())->get_text();
  if (config.getPhonebookStoreName(Left) != newName)
  {
    config.setPhonebookStoreName(Left, newName);
    changed();
  }
}

void Preferences::onPbLeftSortOrder()
{
  SortOrder newOrder = (SortOrder)getActivatedMenuItem(_pbLeftSortOrder);
  if (config.getPhonebookSortOrder(Left) != newOrder)
  {
    config.setPhonebookSortOrder(Left, newOrder);
    changed();
  }
}

void Preferences::onPbRightFileOrDevice()
{
  DataSource newSource =
    (DataSource)getActivatedMenuItem(_pbRightFileOrDevice);
  if (config.getPhonebookDataSource(Right) != newSource)
  {
    config.setPhonebookDataSource(Right, newSource);
    changed();
  }
}

void Preferences::onPbRightName()
{
  string newName =
    ((Gtk::Entry*)_pbRightPhonebookName->gtk_entry())->get_text();
  if (config.getPhonebookStoreName(Right) != newName)
  {
    config.setPhonebookStoreName(Right, newName);
    changed();
  }
}

void Preferences::onPbRightSortOrder()
{
  DataSource newSource =
    (DataSource)getActivatedMenuItem(_pbRightFileOrDevice);
  if (config.getPhonebookDataSource(Right) != newSource)
  {
    config.setPhonebookDataSource(Right, newSource);
    changed();
  }
}

void Preferences::onSmsLeftFileOrDevice()
{
  DataSource newSource =
    (DataSource)getActivatedMenuItem(_smsLeftFileOrDevice);
  if (config.getSMSDataSource(Left) != newSource)
  {
    config.setSMSDataSource(Left, newSource);
    changed();
  }
}

void Preferences::onSmsLeftName()
{
  string newName =
    ((Gtk::Entry*)_smsLeftSMSStoreName->gtk_entry())->get_text();
  if (config.getSMSStoreName(Left) != newName)
  {
    config.setSMSStoreName(Left, newName);
    changed();
  }
}

void Preferences::onSmsLeftSortOrder()
{
  SortOrder newOrder = (SortOrder)getActivatedMenuItem(_smsLeftSortOrder);
  if (config.getSMSSortOrder(Left) != newOrder)
  {
    config.setSMSSortOrder(Left, newOrder);
    changed();
  }
}

void Preferences::onSmsRightFileOrDevice()
{
  DataSource newSource =
    (DataSource)getActivatedMenuItem(_smsRightFileOrDevice);
  if (config.getSMSDataSource(Right) != newSource)
  {
    config.setSMSDataSource(Right, newSource);
    changed();
  }
}

void Preferences::onSmsRightName()
{
  string newName =
    ((Gtk::Entry*)_smsRightSMSStoreName->gtk_entry())->get_text();
  if (config.getSMSStoreName(Right) != newName)
  {
    config.setSMSStoreName(Right, newName);
    changed();
  }
}

void Preferences::onSmsRightSortOrder()
{
  DataSource newSource =
    (DataSource)getActivatedMenuItem(_smsRightFileOrDevice);
  if (config.getSMSDataSource(Right) != newSource)
  {
    config.setSMSDataSource(Right, newSource);
    changed();
  }
}

void Preferences::onShowMessageLog()
{
  bool newState = _showMessageLogButton->get_active();
  if (newState != config.getShowLogWindow())
  {
    config.setShowLogWindow(newState);
    changed();
  }
}

void Preferences::onLongSMS()
{
  bool newState = _longSMSButton->get_active();
  if (newState != config.getLongSMS())
  {
    config.setLongSMS(newState);
    changed();
  }
}

void Preferences::onSentSMSColorChanged()
{
  string newText = ((Gtk::Entry*)_sentSMSColorEntry->gtk_entry())->get_text();
  if (newText != config.getSentSMSColor())
  {
    config.setSentSMSColor(newText);
    changed();
  }
}

void Preferences::onSentSMSColorClicked()
{
  string colorString =
    ((Gtk::Entry*)_sentSMSColorEntry->gtk_entry())->get_text();

  if (colorString.length() == 0 || checkColor(colorString))
  {
    string newColor = runColorSelectionDialog(_("Color of Sent SMS"),
                                              config.getSentSMSColor());

    if (newColor != config.getSentSMSColor())
    {
      ((Gtk::Entry*)_sentSMSColorEntry->gtk_entry())->set_text(newColor);
      config.setSentSMSColor(newColor);
      changed();
    }
  }
}

void Preferences::onReveivedSMSColorChanged()
{
  string newText =
    ((Gtk::Entry*)_receivedSMSColorEntry->gtk_entry())->get_text();
  if (newText != config.getReceivedSMSColor())
  {
    config.setReceivedSMSColor(newText);
    changed();
  }
}

void Preferences::onReceivedSMSColorClicked()
{
  string colorString = ((Gtk::Entry*)_receivedSMSColorEntry->gtk_entry())->get_text();

  if (colorString.length() == 0 || checkColor(colorString))
  {
    string newColor = runColorSelectionDialog(_("Color of Received SMS"),
                                              config.getReceivedSMSColor());

    if (newColor != config.getReceivedSMSColor())
    {
      ((Gtk::Entry*)_receivedSMSColorEntry->gtk_entry())->set_text(newColor);
      config.setReceivedSMSColor(newColor);
      changed();
    }
  }
}

void Preferences::onLongSMSColorChanged()
{
  string newText = ((Gtk::Entry*)_longSMSColorEntry->gtk_entry())->get_text();
  if (newText != config.getLongSMSColor())
  {
    config.setLongSMSColor(newText);
    changed();
  }
}

void Preferences::onLongSMSColorClicked()
{
  string colorString =
    ((Gtk::Entry*)_longSMSColorEntry->gtk_entry())->get_text();

  if (colorString.length() == 0 || checkColor(colorString))
  {
    string newColor = runColorSelectionDialog(_("2nd Color for Long SMS"),
                                              config.getLongSMSColor());

    if (newColor != config.getLongSMSColor())
    {
      ((Gtk::Entry*)_longSMSColorEntry->gtk_entry())->set_text(newColor);
      config.setLongSMSColor(newColor);
      changed();
    }
  }
}

void Preferences::onNewSMSNotify()
{
  bool newState = _newSMSNotifyButton->get_active();
  if (newState != config.getIncomingSMSNotification())
  {
    config.setIncomingSMSNotification(newState);
    changed();
  }
}

void Preferences::onListSelectionStyle()
{
  bool newState = getActivatedMenuItem(_listSelectionStyleMenu) == 1;
  if (newState != config.getMultipleListSelectionStyle())
  {
    config.setMultipleListSelectionStyle(newState);
    changed();
  }
}

void Preferences::onWarningColorChanged()
{
  string newText = ((Gtk::Entry*)_warningColorEntry->gtk_entry())->get_text();
  if (newText != config.getWarningColor())
  {
    config.setWarningColor(newText);
    changed();
  }
}

void Preferences::onWarningColorClicked()
{
  string colorString =
    ((Gtk::Entry*)_warningColorEntry->gtk_entry())->get_text();

  if (colorString.length() == 0 || checkColor(colorString))
  {
    string newColor = runColorSelectionDialog(_("Warning Color"),
                                              config.getWarningColor());

    if (newColor != config.getWarningColor())
    {
      ((Gtk::Entry*)_warningColorEntry->gtk_entry())->set_text(newColor);
      config.setWarningColor(newColor);
      changed();
    }
  }
}

gboolean Preferences::onWindowClose()
{
  DEBUG_START(1);
  cout << "onClose called" << endl;
  DEBUG_END;

  if (! _okToClose)
    return true;
  else
  {
    // this may be called after sync(), but then it doesn't matter anyway
    config.drop();
    delete this;
    return FALSE;
  }
}

void Preferences::onApply(gint pageNum)
{
  DEBUG_START(1);
  cout << "onApply called " << pageNum << endl;
  DEBUG_END;
  if (pageNum == -1)
  {
    // check colors
    bool colorError = false;
    if (! checkColor(config.getSentSMSColor()))
    {
      ((Gtk::Entry*)_sentSMSColorEntry->gtk_entry())->
        set_text(_oldSentSMSColor);
      colorError = true;
    }
    if (! checkColor(config.getReceivedSMSColor()))
    {
      ((Gtk::Entry*)_receivedSMSColorEntry->gtk_entry())->
        set_text(_oldReceivedSMSColor);
      colorError = true;
    }
    if (! checkColor(config.getLongSMSColor()))
    {
      ((Gtk::Entry*)_longSMSColorEntry->gtk_entry())->
        set_text(_oldLongSMSColor);
      colorError = true;
    }
    if (! checkColor(config.getWarningColor()))
    {
      ((Gtk::Entry*)_warningColorEntry->gtk_entry())->
        set_text(_oldWarningColor);
      colorError = true;
    }

    if (colorError)
    {
      _okToClose = false;
      changed();
    }
    else
    {
      _okToClose = true;

      // device page history
      if (config.getPhoneDevice() != _oldPhoneDevice)
      {
        ((Gnome::Entry*)_phoneDeviceFileEntry->gnome_entry())->
          append_history(TRUE, config.getPhoneDevice());
        ((Gnome::Entry*)_phoneDeviceFileEntry->gnome_entry())->save_history();
        _oldPhoneDevice = config.getPhoneDevice();
      }
      if (_oldInitString != config.getInitString())
      {
        _initStringEntry->append_history(TRUE, config.getInitString());
        _initStringEntry->save_history();
        _oldInitString = config.getInitString();
      }

      // phonebooks page history
      if (_oldPhonebookStoreNameLeft != config.getPhonebookStoreName(Left))
      {
        _pbLeftPhonebookName->
          append_history(TRUE, config.getPhonebookStoreName(Left));
        _pbLeftPhonebookName->save_history();
        _oldPhonebookStoreNameLeft = config.getPhonebookStoreName(Left);
      }
      if (_oldPhonebookStoreNameRight != config.getPhonebookStoreName(Right))
      {
        _pbRightPhonebookName->
          append_history(TRUE, config.getPhonebookStoreName(Right));
        _pbRightPhonebookName->save_history();
        _oldPhonebookStoreNameRight = config.getPhonebookStoreName(Right);
      }

      // SMS store page history
      if (_oldSSMSStoreNameLeft != config.getSMSStoreName(Left))
      {
        _smsLeftSMSStoreName->
          append_history(TRUE, config.getSMSStoreName(Left));
        _smsLeftSMSStoreName->save_history();
        _oldSSMSStoreNameLeft = config.getSMSStoreName(Left);
      }
      if (_oldSSMSStoreNameRight != config.getSMSStoreName(Right))
      {
        _smsRightSMSStoreName->
          append_history(TRUE, config.getSMSStoreName(Right));
        _smsRightSMSStoreName->save_history();
        _oldSSMSStoreNameRight = config.getSMSStoreName(Right);
      }

      // SMS page
      if (_oldSentSMSColor != config.getSentSMSColor())
      {
        _sentSMSColorEntry->append_history(TRUE, config.getSentSMSColor());
        _sentSMSColorEntry->save_history();
        _oldSentSMSColor = config.getSentSMSColor();
      }
      if (_oldReceivedSMSColor != config.getReceivedSMSColor())
      {
        _receivedSMSColorEntry->
          append_history(TRUE, config.getReceivedSMSColor());
        _receivedSMSColorEntry->save_history();
        _oldReceivedSMSColor = config.getReceivedSMSColor();
      }
      if (_oldLongSMSColor != config.getLongSMSColor())
      {
        _longSMSColorEntry->append_history(TRUE, config.getLongSMSColor());
        _longSMSColorEntry->save_history();
        _oldLongSMSColor = config.getLongSMSColor();
      }

      // GUI page
      if (_oldWarningColor != config.getWarningColor())
      {
        _warningColorEntry->append_history(TRUE, config.getWarningColor());
        _warningColorEntry->save_history();
        _oldWarningColor = config.getWarningColor();
      }

      // put all synchronisation, history saving etc. here
      config.sync();
    }
  }
}

void Preferences::onHelp(gint pageNum)
{
  // FIXME
}

void Preferences::onClicked(gint p0)
{
  cout << "clicked " << p0 << endl;
  if (p0 == 2)
  {
    _okToClose = true;
    onWindowClose();
  }
}

Preferences::Preferences(bool loadError) :
  Gnome::PropertyBox((GnomePropertyBox*)create_propertybox()),
  DeviceHelper("default device"), ToplevelHelper(this), _okToClose(true)
{
  // try to open default device
  openDefaultDevice();

  // connect handlers
  close.connect(slot(this, &Preferences::onWindowClose));
  apply.connect(slot(this, &Preferences::onApply));
  help.connect(slot(this, &Preferences::onHelp));
  clicked.connect(slot(this, &Preferences::onClicked));

  // device page
  LOOKUP_WIDGET(_phoneDeviceFileEntry, "pref_phone_device", GnomeFileEntry,
                GNOME_IS_FILE_ENTRY, this);
  _phoneDeviceEntry = (Gtk::Entry*)_phoneDeviceFileEntry->gtk_entry();
  _phoneDeviceEntry->
    changed.connect(slot(this, &Preferences::onPhoneDevice));
  _phoneDeviceEntry->set_text(config.getPhoneDevice());
  _oldPhoneDevice = config.getPhoneDevice();
  ((Gnome::Entry*)_phoneDeviceFileEntry->gnome_entry())->load_history();
  
  Gtk::OptionMenu *om;
  LOOKUP_WIDGET(om, "pref_baud_rate", GtkOptionMenu,
                GTK_IS_OPTION_MENU, this);
  _baudRateMenu = DynamicOptionMenu(om);
  _baudRateMenu.append("300", config.getBaudRate(),
                       slot(this, &Preferences::onBaudRateSet));
  _baudRateMenu.append("600", config.getBaudRate(),
                       slot(this, &Preferences::onBaudRateSet));
  _baudRateMenu.append("1200", config.getBaudRate(),
                       slot(this, &Preferences::onBaudRateSet));
  _baudRateMenu.append("2400", config.getBaudRate(),
                       slot(this, &Preferences::onBaudRateSet));
  _baudRateMenu.append("4800", config.getBaudRate(),
                       slot(this, &Preferences::onBaudRateSet));
  _baudRateMenu.append("9600", config.getBaudRate(),
                       slot(this, &Preferences::onBaudRateSet));
  _baudRateMenu.append("19200", config.getBaudRate(),
                       slot(this, &Preferences::onBaudRateSet));
  _baudRateMenu.append("38400", config.getBaudRate(),
                       slot(this, &Preferences::onBaudRateSet));
#ifdef B57600
  _baudRateMenu.append("57600", config.getBaudRate(),
                       slot(this, &Preferences::onBaudRateSet));
#endif
#ifdef B115200
  _baudRateMenu.append("115200", config.getBaudRate(),
                       slot(this, &Preferences::onBaudRateSet));
#endif
#ifdef B230400
  _baudRateMenu.append("230400", config.getBaudRate(),
                       slot(this, &Preferences::onBaudRateSet));
#endif
#ifdef B460800
  _baudRateMenu.append("460800", config.getBaudRate(),
                       slot(this, &Preferences::onBaudRateSet));
#endif

  LOOKUP_WIDGET(_initStringEntry, "pref_init_string_entry",
                GnomeEntry, GNOME_IS_ENTRY, this);
  ((Gtk::Entry*)_initStringEntry->gtk_entry())->changed.
    connect(slot(this, &Preferences::onInitString));
  ((Gtk::Entry*)_initStringEntry->gtk_entry())->
    set_text(config.getInitString());
  _initStringEntry->load_history();
  _oldInitString = config.getInitString();

  LOOKUP_WIDGET(_softwareHandshakeButton, "pref_software_handshake",
                GtkCheckButton, GTK_IS_CHECK_BUTTON, this);
  _softwareHandshakeButton->clicked.
    connect(slot(this, &Preferences::onSoftwareHandshake));
  _softwareHandshakeButton->set_active(config.getSoftwareHandshake());

  LOOKUP_WIDGET(om, "pref_character_set", GtkOptionMenu,
                GTK_IS_OPTION_MENU, this);
  _characterSetMenu = DynamicOptionMenu(om);
  _characterSetMenu.append(config.getCharacterSet(), config.getCharacterSet(),
                           slot(this, &Preferences::onCharacterSetSet));

  LOOKUP_WIDGET(om, "pref_character_encoding", GtkOptionMenu,
                GTK_IS_OPTION_MENU, this);
  _characterEncodingMenu = om->get_menu();
  _characterEncodingMenu->
    selection_done.connect(slot(this, &Preferences::onCharacterEncodingSet));

  // phonebooks page
  LOOKUP_WIDGET(om, "pref_pb_left_file_or_device", GtkOptionMenu,
                GTK_IS_OPTION_MENU, this);
  _pbLeftFileOrDevice = om->get_menu();
  _pbLeftFileOrDevice->selection_done.
    connect(slot(this, &Preferences::onPbLeftFileOrDevice));
  om->set_history(config.getPhonebookDataSource(Left));
  
  LOOKUP_WIDGET(_pbLeftPhonebookName, "pref_pb_left_phonebook_name",
                GnomeEntry, GNOME_IS_ENTRY, this);
  ((Gtk::Entry*)_pbLeftPhonebookName->gtk_entry())->changed.
    connect(slot(this, &Preferences::onPbLeftName));
  ((Gtk::Entry*)_pbLeftPhonebookName->gtk_entry())->
    set_text(config.getPhonebookStoreName(Left));
  _pbLeftPhonebookName->load_history();
  _oldPhonebookStoreNameLeft = config.getPhonebookStoreName(Left);

  LOOKUP_WIDGET(om, "pref_pb_left_sort_order", GtkOptionMenu,
                GTK_IS_OPTION_MENU, this);
  _pbLeftSortOrder = om->get_menu();
  _pbLeftSortOrder->selection_done.
  connect(slot(this, &Preferences::onPbLeftSortOrder));
  om->set_history(config.getPhonebookSortOrder(Left));
  
  LOOKUP_WIDGET(om, "pref_pb_right_file_or_device", GtkOptionMenu,
                GTK_IS_OPTION_MENU, this);
  _pbRightFileOrDevice = om->get_menu();
  _pbRightFileOrDevice->selection_done.
    connect(slot(this, &Preferences::onPbRightFileOrDevice));
  om->set_history(config.getPhonebookDataSource(Right));

  LOOKUP_WIDGET(_pbRightPhonebookName, "pref_pb_right_phonebook_name",
                GnomeEntry, GNOME_IS_ENTRY, this);
  ((Gtk::Entry*)_pbRightPhonebookName->gtk_entry())->changed.
    connect(slot(this, &Preferences::onPbRightName));
  ((Gtk::Entry*)_pbRightPhonebookName->gtk_entry())->
    set_text(config.getPhonebookStoreName(Right));
  _pbRightPhonebookName->load_history();
  _oldPhonebookStoreNameRight = config.getPhonebookStoreName(Right);

  LOOKUP_WIDGET(om, "pref_pb_right_sort_order", GtkOptionMenu,
                GTK_IS_OPTION_MENU, this);
  _pbRightSortOrder = om->get_menu();
  _pbRightSortOrder->selection_done.
    connect(slot(this, &Preferences::onPbRightSortOrder));
  om->set_history(config.getPhonebookSortOrder(Right));

  // SMS stores page
  LOOKUP_WIDGET(om, "pref_sms_left_file_or_device", GtkOptionMenu,
                GTK_IS_OPTION_MENU, this);
  _smsLeftFileOrDevice = om->get_menu();
  _smsLeftFileOrDevice->selection_done.
    connect(slot(this, &Preferences::onSmsLeftFileOrDevice));
  om->set_history(config.getSMSDataSource(Left));

  LOOKUP_WIDGET(_smsLeftSMSStoreName, "pref_sms_left_store_name",
                GnomeEntry, GNOME_IS_ENTRY, this);
  ((Gtk::Entry*)_smsLeftSMSStoreName->gtk_entry())->changed.
    connect(slot(this, &Preferences::onSmsLeftName));
  ((Gtk::Entry*)_smsLeftSMSStoreName->gtk_entry())->
    set_text(config.getSMSStoreName(Left));
  _smsLeftSMSStoreName->load_history();
  _oldSSMSStoreNameLeft = config.getSMSStoreName(Left);

  LOOKUP_WIDGET(om, "pref_sms_left_sort_order", GtkOptionMenu,
                GTK_IS_OPTION_MENU, this);
  _smsLeftSortOrder = om->get_menu();
  _smsLeftSortOrder->selection_done.
    connect(slot(this, &Preferences::onSmsLeftSortOrder));
  om->set_history(config.getSMSSortOrder(Left));
  
  LOOKUP_WIDGET(om, "pref_sms_right_file_or_device", GtkOptionMenu,
                GTK_IS_OPTION_MENU, this);
  _smsRightFileOrDevice = om->get_menu();
  _smsRightFileOrDevice->selection_done.
    connect(slot(this, &Preferences::onSmsRightFileOrDevice));
  om->set_history(config.getSMSDataSource(Right));

  LOOKUP_WIDGET(_smsRightSMSStoreName, "pref_sms_right_store_name",
                GnomeEntry, GNOME_IS_ENTRY, this);
  ((Gtk::Entry*)_smsRightSMSStoreName->gtk_entry())->changed.
    connect(slot(this, &Preferences::onSmsRightName));
  ((Gtk::Entry*)_smsRightSMSStoreName->gtk_entry())->
    set_text(config.getSMSStoreName(Right));
  _smsRightSMSStoreName->load_history();
  _oldSSMSStoreNameRight = config.getSMSStoreName(Right);

  LOOKUP_WIDGET(om, "pref_sms_right_sort_order", GtkOptionMenu,
                GTK_IS_OPTION_MENU, this);
  _smsRightSortOrder = om->get_menu();
  _smsRightSortOrder->selection_done.
    connect(slot(this, &Preferences::onSmsRightSortOrder));
  om->set_history(config.getSMSSortOrder(Right));

  // SMS page
  LOOKUP_WIDGET(_showMessageLogButton, "pref_show_message_log", GtkCheckButton,
                GTK_IS_CHECK_BUTTON, this);
  _showMessageLogButton->clicked.
    connect(slot(this, &Preferences::onShowMessageLog));
  _showMessageLogButton->set_active(config.getShowLogWindow());

  LOOKUP_WIDGET(_longSMSButton, "pref_long_sms", GtkCheckButton,
                GTK_IS_CHECK_BUTTON, this);
  _longSMSButton->clicked.connect(slot(this, &Preferences::onLongSMS));
  _longSMSButton->set_active(config.getLongSMS());

  LOOKUP_WIDGET(_sentSMSColorEntry, "pref_sent_sms_col_entry", GnomeEntry,
                GNOME_IS_ENTRY, this);
  ((Gtk::Entry*)_sentSMSColorEntry->gtk_entry())->changed.
    connect(slot(this, &Preferences::onSentSMSColorChanged));
  ((Gtk::Entry*)_sentSMSColorEntry->gtk_entry())->
    set_text(config.getSentSMSColor());
  _sentSMSColorEntry->load_history();
  _oldSentSMSColor = config.getSentSMSColor();

  Gtk::Button *selectButton;
  LOOKUP_WIDGET(selectButton, "pref_sent_sms_col_select", GtkButton,
                GTK_IS_BUTTON, this);
  selectButton->clicked.
    connect(slot(this, &Preferences::onSentSMSColorClicked));

  LOOKUP_WIDGET(_receivedSMSColorEntry, "pref_rec_sms_col_entry", GnomeEntry,
                GNOME_IS_ENTRY, this);
  ((Gtk::Entry*)_receivedSMSColorEntry->gtk_entry())->changed.
    connect(slot(this, &Preferences::onReveivedSMSColorChanged));
  ((Gtk::Entry*)_receivedSMSColorEntry->gtk_entry())->
    set_text(config.getReceivedSMSColor());
  _receivedSMSColorEntry->load_history();
  _oldReceivedSMSColor = config.getReceivedSMSColor();

  LOOKUP_WIDGET(selectButton, "pref_rec_sms_col_select", GtkButton,
                GTK_IS_BUTTON, this);
  selectButton->clicked.
    connect(slot(this, &Preferences::onReceivedSMSColorClicked));

  LOOKUP_WIDGET(_longSMSColorEntry, "pref_long_sms_col_entry", GnomeEntry,
                GNOME_IS_ENTRY, this);
  ((Gtk::Entry*)_longSMSColorEntry->gtk_entry())->changed.
    connect(slot(this, &Preferences::onLongSMSColorChanged));
  ((Gtk::Entry*)_longSMSColorEntry->gtk_entry())->
    set_text(config.getLongSMSColor());
  _longSMSColorEntry->load_history();
  _oldLongSMSColor = config.getLongSMSColor();

  LOOKUP_WIDGET(selectButton, "pref_long_sms_col_select", GtkButton,
                GTK_IS_BUTTON, this);
  selectButton->clicked.
    connect(slot(this, &Preferences::onLongSMSColorClicked));

  LOOKUP_WIDGET(_newSMSNotifyButton, "pref_new_sms_notify", GtkCheckButton,
                GTK_IS_CHECK_BUTTON, this);
  _newSMSNotifyButton->clicked.
    connect(slot(this, &Preferences::onNewSMSNotify));
  _newSMSNotifyButton->set_active(config.getIncomingSMSNotification());

  // GUI page
  LOOKUP_WIDGET(om, "pref_list_sel_style", GtkOptionMenu,
                GTK_IS_OPTION_MENU, this);
  _listSelectionStyleMenu = om->get_menu();
  _listSelectionStyleMenu->
    selection_done.connect(slot(this, &Preferences::onListSelectionStyle));
  om->set_history(config.getMultipleListSelectionStyle() ? 1 : 0);

  LOOKUP_WIDGET(_warningColorEntry, "pref_warn_col_entry", GnomeEntry,
                GNOME_IS_ENTRY, this);
  ((Gtk::Entry*)_warningColorEntry->gtk_entry())->changed.
    connect(slot(this, &Preferences::onWarningColorChanged));
  ((Gtk::Entry*)_warningColorEntry->gtk_entry())->
    set_text(config.getWarningColor());
  _warningColorEntry->load_history();
  _oldWarningColor = config.getWarningColor();

  LOOKUP_WIDGET(selectButton, "pref_warn_col_select", GtkButton,
                GTK_IS_BUTTON, this);
  selectButton->clicked.
    connect(slot(this, &Preferences::onWarningColorClicked));
  

  // if there was a load error, the configuration was changed by
  // the load() routine
  set_modified(loadError);

  show_all();
}

Preferences::~Preferences()
{
}
