/*
 *  xfmedia - simple gtk2 media player based on xine
 *
 *  Copyright (c) 2004-2005 Brian Tarricone, <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_GETOPT_H
#define _GNU_SOURCE
#include <getopt.h>
#endif

#ifdef HAVE_LOCALE_H
#include <locale.h>
#endif

#include <glib.h>

/* for _() only; no need to link in the actual library */
#include <libxfce4util/libxfce4util.h>

#include <xfmedia/xfmedia-remote-client.h>

enum {
    OPT_SESSION = 1000,
    OPT_PING,
    OPT_PAUSE,
    OPT_STOP,
    OPT_PREV,
    OPT_NEXT
};    

static const gchar *short_opts = "Vhp::ta:r:cl:s:Qj";
static const struct option long_opts[] =
{
    { "version", no_argument, NULL, 'V' },
    { "help", no_argument, NULL, 'h' },
    
    { "session", required_argument, NULL, OPT_SESSION },
    { "ping", no_argument, NULL, OPT_PING },
    
    { "play", optional_argument, NULL, 'p' },
    { "toggle", no_argument, NULL, 't' },
    { "pause", no_argument, NULL, OPT_PAUSE },
    { "stop", no_argument, NULL, OPT_STOP },
    { "previous", no_argument, NULL, OPT_PREV },
    { "next", no_argument, NULL, OPT_NEXT },
    
    { "playlist-add", required_argument, NULL, 'a' },
    { "playlist-remove", required_argument, NULL, 'r' },
    { "playlist-clear", no_argument, NULL, 'c' },
    { "playlist-load", required_argument, NULL, 'l' },
    { "playlist-save", required_argument, NULL, 's' },
    
    { "quit", no_argument, NULL, 'Q' },
    
    { "jump-to-file", no_argument, NULL, 'j' },
    
    { NULL, 0, NULL, 0 }
};

static void
xfmedia_remote_print_usage(const gchar *argv0)
{
    gchar *cmd;
	
	if((cmd=g_strrstr(argv0, "/")))
		cmd++;
	else
		cmd = (gchar *)argv0;
    
    g_print(_("Xfmedia %s.  Released under the terms of the GNU GPL.  See the COPYING\nfile for details.\n\n"), VERSION);
	g_print(_("Usage: %s [options]\n\n"), cmd);
	g_print(_("    -h, --help           Print this help message and exit.\n"));
	g_print(_("    -V, --version        Print version information and exit.\n"));
    g_print(_("    --session=N          Specify the number of the already-running instance of\n"
              "                         Xfmedia.  (Defaults to 0, or the first instance found.)\n"));
    g_print(_("    --ping               Checks to see if an instance of Xfmedia is running.\n"));
    g_print(_("    -p, --play[=N]       Causes Xfmedia to play the file at playlist idx N.\n"
              "                         If N is not specified, Xfmedia will play the currently-\n"
              "                         selected entry, or, if no item is selected, the first\n"
              "                         entry.  If used in conjunction with --playlist-add,\n"
              "                         will cause Xfmedia to play the newly-added entry.\n"));
    g_print(_("    --toggle             Causes Xfmedia to pause if playing, or play if paused\n"
              "                         or stopped.\n"));
    g_print(_("    --pause              Causes Xfmedia to pause/unpause.\n"));
    g_print(_("    --stop               Causes Xfmedia to stop playback.\n"));
    g_print(_("    --previous           Causes Xfmedia to play the entry played before the\n"
              "                         currently-selected (or -playing) entry.\n"));
    g_print(_("    --next               Causes Xfmedia to play the next entry in the playlist.\n"
              "                         If Xfmedia is in shuffle mode, a new entry will be\n"
              "                         selected randomly.\n"));
    g_print(_("    -a, --playlist-add=FILE\n"
              "                         Adds FILE to the end of the playlist.\n"));
    g_print(_("    -r, --playlist-remove=N\n"
              "                         Removes the specified playlist entry from the playlist.\n"));
    g_print(_("    -c, --playlist-clear\n"
              "                         Removes all entries from the playlist.\n"));
    g_print(_("    -l, --playlist-load=FILE\n"
              "                         Loads the specified playlist.  Note that this does NOT\n"
              "                         first clear the playlist.\n"));
    g_print(_("    -s, --playlist-save=FILE\n"
              "                         Saves the current playlist to the specified FILE.\n"));
    g_print(_("    -Q, --quit           Instructs Xfmedia to quit.\n"));
    g_print(_("    -j, --jump-to-file   Causes Xfmedia to show the main window (if not visible)\n"
              "                         and show the jump-to-file entry box."));
}

int
main(int argc, char **argv)
{
    gint c = 0, option_index = 0, session_id = -1, tmpi;
    const gchar *command = NULL;
    gpointer arg1 = NULL, arg2 = NULL;
    gboolean possibly_play_too = FALSE;
    XfmediaRemoteStatus ret;
    
    xfce_textdomain(GETTEXT_PACKAGE, LOCALEDIR, "UTF-8");
    if(!setlocale(LC_ALL, ""))
        g_warning("Locale not supported; using 'C'.");
    
    if(argc == 1) {
        xfmedia_remote_print_usage(argv[0]);
        return 1;
    }
    
    opterr = 0;
    for(;;) {
        c = getopt_long(argc, argv, short_opts, long_opts, &option_index);
        if(c == -1)
            break;
        
        switch(c) {
            case 'V':
                g_print(_("Xfmedia version %s, Copyright (c) Brian Tarricone, <bjt23@cornell.edu>\n"), VERSION);
                g_print(_("Released under the terms of the GNU General Public License.\n"));
                exit(0);
                break;
            
            case 'h':
                xfmedia_remote_print_usage(argv[0]);
                exit(0);
                break;
            
            case OPT_SESSION:
                session_id = atoi(optarg);
                if(session_id < 0) {
                    g_print(_("Argument to --session must be greater than or equal to 0.\n"));
                    exit(1);
                }
                break;
            
            case OPT_PING:
                command = XFMEDIA_REMOTE_IS_RUNNING;
                break;
            
            case 'p':
                if(!command) {
                    command = XFMEDIA_REMOTE_PLAY;
                    if(optarg && *optarg && *optarg != ':') {
                        tmpi = atoi(optarg);
                        if(tmpi < 0) {
                            g_print(_("Argument to --play must be greater than or equal to 0.\n"));
                            exit(1);
                        } else
                            arg1 = GINT_TO_POINTER(tmpi);
                    } else
                        arg1 = GINT_TO_POINTER(-1);
                }
                possibly_play_too = TRUE;
                break;
            
            case 't':
                command = XFMEDIA_REMOTE_TOGGLE_PLAY;
                break;
            
            case OPT_PAUSE:
                command = XFMEDIA_REMOTE_PAUSE;
                break;
            
            case OPT_STOP:
                command = XFMEDIA_REMOTE_STOP;
                break;
            
            case OPT_PREV:
                command = XFMEDIA_REMOTE_PREV;
                break;
            
            case OPT_NEXT:
                command = XFMEDIA_REMOTE_NEXT;
                break;
            
            case 'a':
                command = XFMEDIA_REMOTE_ADD_FILE;
                if(*optarg != '/') {
                    arg1 = g_strdup_printf("%s%s%s", g_get_current_dir(),
                            G_DIR_SEPARATOR_S, optarg);
                } else
                    arg1 = optarg;
                arg2 = GINT_TO_POINTER(-1);
                break;
            
            case 'r':
                command = XFMEDIA_REMOTE_REMOVE_FILE;
                tmpi = atoi(optarg);
                if(tmpi < 0) {
                    g_print(_("Argument to --playlist-remove must be greater than or equal to 0.\n"));
                    exit(1);
                } else
                    arg1 = GINT_TO_POINTER(tmpi);
                break;
            
            case 'c':
                command = XFMEDIA_REMOTE_CLEAR_PLAYLIST;
                break;
            
            case 'l':
                command = XFMEDIA_REMOTE_LOAD_PLAYLIST;
                if(*optarg != '/') {
                    arg1 = g_strdup_printf("%s%s%s", g_get_current_dir(),
                            G_DIR_SEPARATOR_S, optarg);
                } else
                    arg1 = optarg;
                break;
            
            case 's':
                command = XFMEDIA_REMOTE_SAVE_PLAYLIST;
                if(*optarg != '/') {
                    arg1 = g_strdup_printf("%s%s%s", g_get_current_dir(),
                            G_DIR_SEPARATOR_S, optarg);
                } else
                    arg1 = optarg;
                break;
            
            case 'Q':
                command = XFMEDIA_REMOTE_QUIT;
                break;
            
            case 'j':
                command = XFMEDIA_REMOTE_TRIGGER_JTF;
                break;
            
            default:
                g_printerr(_("Unrecognized option: %s\n"), argv[optind]);
                xfmedia_remote_print_usage(argv[0]);
                exit(1);
        }
    }
    
    DBG("command is %s %p %p\n", command?command:"(null)", arg1, arg2);
    
    if(!command)
        return 1;
    
    if(session_id < 0) {
        session_id = xfmedia_remote_send_command(-1, XFMEDIA_REMOTE_IS_RUNNING);
        if(session_id < 0) {
            /* can't use xfce_exec() here because i don't want to have to
             * pull in libxfcegui4 */
            if(!g_spawn_command_line_async(BINDIR "/xfmedia", NULL)
                   && !g_spawn_command_line_async("xfmedia", NULL))
            {
                g_warning(_("Unable to find a running Xfmedia instance."));
                return 1;
            }
            while(session_id < 0)
                session_id = xfmedia_remote_send_command(-1, XFMEDIA_REMOTE_IS_RUNNING);
        }
    }
    
    ret = xfmedia_remote_send_command(session_id, command, arg1, arg2);
    
    if(possibly_play_too && ret >= 0 && !strcmp(command, XFMEDIA_REMOTE_ADD_FILE))
        ret = xfmedia_remote_send_command(session_id, XFMEDIA_REMOTE_PLAY, ret);
    
    return (ret >= XFMEDIA_REMOTE_COMMAND_SUCCEEDED ? 0 : 1);
}
