/*
 *  xfmedia - simple gtk2 media player based on xine
 *
 *  Copyright (c) 2004-2005 Brian Tarricone, <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <xfmedia/xfmedia-interface.h>

#include <libxfce4util/libxfce4util.h>

#include "xfmedia-common.h"
#include "mainwin.h"
#include <xfmedia/xfmedia-playlist.h>
#include "gtkxine.h"

gint
xfmedia_interface_get_session_number(XfMediaPlugin *plugin)
{
    XfMediaMainwin *mwin;
    
    g_return_val_if_fail(plugin, -1);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    return mwin->session_id;
}

gboolean
xfmedia_engine_open(XfMediaPlugin *plugin, const gchar *location)
{
    XfMediaMainwin *mwin;
    
    g_return_val_if_fail(plugin && location, FALSE);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    return gtk_xine_open(mwin->gtx, location);
}

gboolean
xfmedia_engine_play(XfMediaPlugin *plugin)
{
    XfMediaMainwin *mwin;
    
    g_return_val_if_fail(plugin, FALSE);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    if(gtk_xine_get_status(mwin->gtx) == XINE_STATUS_PLAY
            && gtk_xine_get_param(mwin->gtx, XINE_PARAM_SPEED) != XINE_SPEED_NORMAL)
    {
        if(gtk_xine_get_param(mwin->gtx, XINE_PARAM_SPEED) == XINE_SPEED_PAUSE)
            gtk_xine_set_param(mwin->gtx, XINE_PARAM_AUDIO_CLOSE_DEVICE, 0);
        gtk_xine_set_param(mwin->gtx, XINE_PARAM_SPEED, XINE_SPEED_NORMAL);
        return TRUE;
    } else
        return gtk_xine_play(mwin->gtx, 0, 0);
}

void
xfmedia_engine_pause(XfMediaPlugin *plugin)
{
    XfMediaMainwin *mwin;
    
    g_return_if_fail(plugin);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    if(gtk_xine_get_status(mwin->gtx) == XINE_STATUS_PLAY) {
        gtk_xine_set_param(mwin->gtx, XINE_PARAM_AUDIO_CLOSE_DEVICE, 1);
        gtk_xine_set_param(mwin->gtx, XINE_PARAM_SPEED, XINE_SPEED_PAUSE);
    }
}

void
xfmedia_engine_stop(XfMediaPlugin *plugin)
{
    XfMediaMainwin *mwin;
    
    g_return_if_fail(plugin);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    gtk_xine_stop(mwin->gtx);
}

gboolean
xfmedia_engine_seek(XfMediaPlugin *plugin, gint time)
{
    XfMediaMainwin *mwin;
    
    g_return_val_if_fail(plugin && time >= 0, FALSE);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    gtk_xine_set_param(mwin->gtx, XINE_PARAM_AUDIO_CLOSE_DEVICE, 0);
    return gtk_xine_play(mwin->gtx, 0, time);
}

gboolean
xfmedia_engine_seek_stream_pos(XfMediaPlugin *plugin, gint position)
{
    XfMediaMainwin *mwin;
    
    g_return_val_if_fail(plugin && position >= 0, FALSE);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    gtk_xine_set_param(mwin->gtx, XINE_PARAM_AUDIO_CLOSE_DEVICE, 0);
    return gtk_xine_play(mwin->gtx, position, 0);
}

gint
xfmedia_engine_get_current_time(XfMediaPlugin *plugin)
{
    XfMediaMainwin *mwin;
    gint pos_stream, pos_time, length_time;
    
    g_return_val_if_fail(plugin, 0);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    if(gtk_xine_get_pos_length(mwin->gtx, &pos_stream, &pos_time, &length_time))
        return pos_time;
    else
        return 0;
}

gint
xfmedia_engine_get_current_stream_pos(XfMediaPlugin *plugin)
{
    XfMediaMainwin *mwin;
    gint pos_stream, pos_time, length_time;
    
    g_return_val_if_fail(plugin, 0);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    if(gtk_xine_get_pos_length(mwin->gtx, &pos_stream, &pos_time, &length_time))
        return pos_stream;
    else
        return 0;
}

gint
xfmedia_engine_get_total_time(XfMediaPlugin *plugin)
{
    XfMediaMainwin *mwin;
    gint pos_stream, pos_time, length_time;
    
    g_return_val_if_fail(plugin, 0);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    if(gtk_xine_get_pos_length(mwin->gtx, &pos_stream, &pos_time, &length_time))
        return length_time;
    else
        return 0;
}

XfMediaStatus
xfmedia_engine_get_status(XfMediaPlugin *plugin)
{
    XfMediaMainwin *mwin;
    
    g_return_val_if_fail(plugin, XFMEDIA_STATUS_UNAVAILABLE);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    return (XfMediaStatus)gtk_xine_get_status(mwin->gtx);
}

XfMediaSpeed
xfmedia_engine_get_speed(XfMediaPlugin *plugin)
{
    XfMediaMainwin *mwin;
    
    g_return_val_if_fail(plugin, XFMEDIA_SPEED_NORMAL);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    return (XfMediaSpeed)gtk_xine_get_param(mwin->gtx, XINE_PARAM_SPEED);
}

void
xfmedia_engine_set_speed(XfMediaPlugin *plugin, XfMediaSpeed speed)
{
    XfMediaMainwin *mwin;
    
    g_return_if_fail(plugin && speed >= XFMEDIA_SPEED_PAUSED
            && speed <= XFMEDIA_SPEED_FASTER);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    if(gtk_xine_get_param(mwin->gtx, XINE_PARAM_SPEED) == XINE_SPEED_PAUSE
             && speed != XFMEDIA_SPEED_PAUSED)
    {
        gtk_xine_set_param(mwin->gtx, XINE_PARAM_AUDIO_CLOSE_DEVICE, 0);
    } else if(speed == XFMEDIA_SPEED_PAUSED)
        gtk_xine_set_param(mwin->gtx, XINE_PARAM_AUDIO_CLOSE_DEVICE, 1);
    
    gtk_xine_set_param(mwin->gtx, XINE_PARAM_SPEED, (gint)speed);
}

gboolean
xfmedia_get_fullscreen(XfMediaPlugin *plugin)
{
    XfMediaMainwin *mwin;
    
    g_return_val_if_fail(plugin, FALSE);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    return gtk_xine_is_fullscreen(mwin->gtx);
}

void
xfmedia_set_fullscreen(XfMediaPlugin *plugin, gboolean setting)
{
    XfMediaMainwin *mwin;
    
    g_return_if_fail(plugin);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    gtk_xine_set_fullscreen(mwin->gtx, setting);
}

gboolean
xfmedia_get_video_window_visible(XfMediaPlugin *plugin)
{
    XfMediaMainwin *mwin;
    
    g_return_val_if_fail(plugin, FALSE);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    return (gtk_xine_is_fullscreen(mwin->gtx)
            || GTK_WIDGET_VISIBLE(mwin->video_window));
}

void
xfmedia_set_video_window_visible(XfMediaPlugin *plugin, gboolean setting)
{
    XfMediaMainwin *mwin;
    
    g_return_if_fail(plugin);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    if(!setting && gtk_xine_is_fullscreen(mwin->gtx))
        gtk_xine_set_fullscreen(mwin->gtx, FALSE);
    else if(!gtk_xine_is_fullscreen(mwin->gtx)) {
        if(setting)
            gtk_widget_show(mwin->video_window);
        else
            gtk_widget_hide(mwin->video_window);
    }
}

G_CONST_RETURN gchar *
xfmedia_info_get_location(XfMediaPlugin *plugin)
{
    XfMediaMainwin *mwin;
    
    g_return_val_if_fail(plugin, NULL);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    return gtk_xine_get_current_mrl(mwin->gtx);
}


gchar *
xfmedia_info_get_name(XfMediaPlugin *plugin)
{
    XfMediaMainwin *mwin;
    gint index;
    gchar *name = NULL;
    
    g_return_val_if_fail(plugin, NULL);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    if(!mwin->cur_playing)
        return NULL;
    
    index = xfmedia_playlist_entry_ref_get_index(mwin->cur_playing);
    if(index != -1)
        xfmedia_playlist_get(mwin->plist, index, &name, NULL, NULL);
    
    return name;
}

gboolean
xfmedia_info_get_has_video(XfMediaPlugin *plugin)
{
    XfMediaMainwin *mwin;
    
    g_return_val_if_fail(plugin, FALSE);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    return gtk_xine_get_stream_info(mwin->gtx, XINE_STREAM_INFO_HAS_VIDEO);
}

gboolean
xfmedia_info_get_has_audio(XfMediaPlugin *plugin)
{
    XfMediaMainwin *mwin;
    
    g_return_val_if_fail(plugin, FALSE);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    return gtk_xine_get_stream_info(mwin->gtx, XINE_STREAM_INFO_HAS_AUDIO);
}

gboolean
xfmedia_info_get_audio_info(XfMediaPlugin *plugin, gchar **codec, gint *bitrate,
        gint *sample_rate, gint *channels, gint *bits_per_sample)
{
    XfMediaMainwin *mwin;
    gint status;
    
    g_return_val_if_fail(plugin, FALSE);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    status = gtk_xine_get_status(mwin->gtx);
    if(status == XINE_STATUS_IDLE || status == XINE_STATUS_QUIT
            || !gtk_xine_get_stream_info(mwin->gtx, XINE_STREAM_INFO_HAS_AUDIO))
    {
        return FALSE;
    }
    
    if(codec)
        *codec = g_strdup(gtk_xine_get_meta_info(mwin->gtx, XINE_META_INFO_AUDIOCODEC));
    if(bitrate)
        *bitrate = gtk_xine_get_stream_info(mwin->gtx, XINE_STREAM_INFO_AUDIO_BITRATE);
    if(sample_rate)
        *sample_rate = gtk_xine_get_stream_info(mwin->gtx, XINE_STREAM_INFO_AUDIO_SAMPLERATE);
    if(channels)
        *channels = gtk_xine_get_stream_info(mwin->gtx, XINE_STREAM_INFO_AUDIO_CHANNELS);
    if(bits_per_sample)
        *bits_per_sample = gtk_xine_get_stream_info(mwin->gtx, XINE_STREAM_INFO_AUDIO_BITS);
    
    return TRUE;
}

gboolean
xfmedia_info_get_video_info(XfMediaPlugin *plugin, gchar **codec, gint *bitrate,
        gdouble *fps, gint *width, gint *height, gdouble *aspect)
{
    XfMediaMainwin *mwin;
    gint status;
    
    g_return_val_if_fail(plugin, FALSE);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    status = gtk_xine_get_status(mwin->gtx);
    if(status == XINE_STATUS_IDLE || status == XINE_STATUS_QUIT
            || !gtk_xine_get_stream_info(mwin->gtx, XINE_STREAM_INFO_HAS_VIDEO))
    {
        return FALSE;
    }
    
    if(codec)
        *codec = g_strdup(gtk_xine_get_meta_info(mwin->gtx, XINE_META_INFO_VIDEOCODEC));
    if(bitrate)
        *bitrate = gtk_xine_get_stream_info(mwin->gtx, XINE_STREAM_INFO_BITRATE);
    if(fps) {
        gint frame_duration = gtk_xine_get_stream_info(mwin->gtx, XINE_STREAM_INFO_FRAME_DURATION);
        if(frame_duration == 0)
            *fps = 0;
        else
            *fps = 90000.0 / frame_duration;
    }
    if(width)
        *width = gtk_xine_get_stream_info(mwin->gtx, XINE_STREAM_INFO_VIDEO_WIDTH);
    if(height)
        *height = gtk_xine_get_stream_info(mwin->gtx, XINE_STREAM_INFO_VIDEO_HEIGHT);
    if(aspect)
        *aspect = gtk_xine_get_stream_info(mwin->gtx, XINE_STREAM_INFO_VIDEO_RATIO) / 10000.0;
    
    return TRUE;
}
