/*
 *  xfmedia - simple gtk2 media player based on xine
 *
 *  Copyright (c) 2004-2005 Brian Tarricone, <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#include <gtk/gtk.h>
#include <gdk-pixbuf/gdk-pixdata.h>

#include <libxfce4util/libxfce4util.h>
#include <libxfcegui4/libxfcegui4.h>

#define EXO_API_SUBJECT_TO_CHANGE
#include <exo/exo.h>

#include "gtkxine.h"
#include "mainwin.h"
#include <xfmedia/xfmedia-playlist.h>
#include <xfmedia/xfmedia-settings.h>
#include "trayicon.h"
#include "keybindings.h"
#include "main.h"
#include "jumptofilewin.h"
#include "xfmedia-internal.h"
#include "mediamarks.h"
/*#include "equaliser.h"*/

#define MAX_SHUFFLE_QUEUE_DEPTH 100
#define AUTOHIDE_TIMEOUT        500

enum {
	DROP_TGT_REORDER = 0,
	DROP_TGT_URILIST,
	DROP_TGT_STRING
};

gboolean
xfmedia_mainwin_do_autohide(XfMediaMainwin *mwin)
{
    gboolean do_it = FALSE;
    GdkScreen *gscreen;
    gint scr_w, scr_h, x, y, w, h, a_x, a_y, a_w, a_h;
    GdkRectangle extents;
    
    gscreen = gtk_widget_get_screen(mwin->window);
    scr_w = gdk_screen_get_width(gscreen);
    scr_h = gdk_screen_get_height(gscreen);
    
    gdk_window_get_frame_extents(mwin->window->window, &extents);
    x = extents.x;
    y = extents.y;
    w = extents.width;
    h = extents.height;
    
    if(y == 0) {
        a_x = x;
        a_y = 0;
        a_w = w;
        a_h = 3;
        do_it = TRUE;
    } else if(y == scr_h - h) {
        a_x = x;
        a_y = scr_h - 3;
        a_w = w;
        a_h = 3;
        do_it = TRUE;
    } else if(x == 0) {
        a_x = 0;
        a_y = y;
        a_w = 3;
        a_h = h;
        gtk_window_move(GTK_WINDOW(mwin->autohide_window), 0, y);
        do_it = TRUE;
    } else if(x == scr_w - w) {
        a_x = scr_w - 3;
        a_y = y;
        a_w = 3;
        a_h = h;
        do_it = TRUE;
    }
    
    if(do_it) {
        gtk_widget_set_size_request(mwin->autohide_window, a_w, a_h);
        gtk_widget_realize(mwin->autohide_window);
        gtk_window_resize(GTK_WINDOW(mwin->autohide_window), a_w, a_h);
        gtk_window_move(GTK_WINDOW(mwin->autohide_window), a_x, a_y);
        if(xfmedia_settings_get_bool("/xfmedia/general/window_sticky"))
            gtk_window_stick(GTK_WINDOW(mwin->autohide_window));
        else
            gtk_window_unstick(GTK_WINDOW(mwin->autohide_window));
        gtk_window_set_keep_above(GTK_WINDOW(mwin->autohide_window), TRUE);
        gtk_window_set_skip_taskbar_hint(GTK_WINDOW(mwin->autohide_window), TRUE);
        gtk_window_set_accept_focus(GTK_WINDOW(mwin->autohide_window), FALSE);
        gtk_widget_show(mwin->autohide_window);
        gtk_widget_hide(mwin->window);
    }
    
    mwin->autohide_timer = 0;
    
    return FALSE;
}

static void
instantiate_extra_xine(XfMediaMainwin *mwin)
{
    mwin->extra_vp = xine_open_video_driver(gtk_xine_get_raw_xine_engine(mwin->gtx),
            "none", XINE_VISUAL_TYPE_NONE, NULL);
    mwin->extra_ap = xine_open_audio_driver(gtk_xine_get_raw_xine_engine(mwin->gtx),
            "none", NULL);
    mwin->extra_stream = xine_stream_new(gtk_xine_get_raw_xine_engine(mwin->gtx),
            mwin->extra_ap, mwin->extra_vp);
}

static void
create_autohide_window(XfMediaMainwin *mwin)
{
    GtkWidget *win;
    guint opacity;
    gint w, h;
    
    win = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(win), "xfmedia-autohide-window");
    gtk_window_set_decorated(GTK_WINDOW(win), FALSE);
    gtk_window_set_keep_above(GTK_WINDOW(win), TRUE);
    gtk_window_set_skip_taskbar_hint(GTK_WINDOW(win), TRUE);
    gtk_window_set_skip_pager_hint(GTK_WINDOW(win), TRUE);
    g_signal_connect(G_OBJECT(win), "enter-notify-event",
            G_CALLBACK(xfmedia_autohidewin_enter_notify), mwin);
    
    gtk_window_get_size(GTK_WINDOW(mwin->window), &w, &h);
    gtk_widget_set_size_request(win, w, h);
    
    /* set 40% transparency (if supported) */
    opacity = (guint)(0.6*0xffffffff);
    gdk_property_change(mwin->fs_controls->window,
            gdk_atom_intern("_NET_WM_WINDOW_OPACITY", FALSE),
            gdk_atom_intern("CARDINAL", FALSE), 32,
            GDK_PROP_MODE_REPLACE, (const guchar *)&opacity, sizeof(guint));
    
    mwin->autohide_window = win;
}

static void
mainwin_create_progress_box(XfMediaMainwin *mwin)
{
    PangoFontDescription *fdesc;
    
    mwin->progress_box = gtk_hbox_new(FALSE, BORDER/2);
    
    mwin->progress_label = gtk_label_new("");
    gtk_widget_show(mwin->progress_label);
    gtk_box_pack_start(GTK_BOX(mwin->progress_box), mwin->progress_label,
            FALSE, FALSE, 0);
    
    fdesc = pango_font_description_from_string("italic");
    gtk_widget_modify_font(mwin->progress_label, fdesc);
    pango_font_description_free(fdesc);
    
    mwin->progress_bar = gtk_progress_bar_new();
    gtk_progress_bar_set_orientation(GTK_PROGRESS_BAR(mwin->progress_bar),
            GTK_PROGRESS_LEFT_TO_RIGHT);
    gtk_widget_show(mwin->progress_bar);
    gtk_box_pack_start(GTK_BOX(mwin->progress_box), mwin->progress_bar,
            TRUE, TRUE, 0);
}

static void
mainwin_create_ui_msg_label(XfMediaMainwin *mwin)
{
    PangoFontDescription *fdesc;
    
    mwin->ui_message_label = exo_ellipsized_label_new("");
    exo_ellipsized_label_set_ellipsize(EXO_ELLIPSIZED_LABEL(mwin->ui_message_label),
            EXO_PANGO_ELLIPSIZE_END);
    gtk_misc_set_alignment(GTK_MISC(mwin->ui_message_label), 0.0, 0.5);
    
    fdesc = pango_font_description_from_string("italic");
    gtk_widget_modify_font(mwin->ui_message_label, fdesc);
    pango_font_description_free(fdesc);
}

static void
create_fs_control_window(XfMediaMainwin *mwin)
{
    GtkWidget *tophbox, *img, *btn, *lbl, *slider, *evtbox;
    gint iw, ih;
    guint opacity;
    
    mwin->fs_controls = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(mwin->fs_controls),
            "xfmedia-fullscreen-control-window");
    gtk_window_set_decorated(GTK_WINDOW(mwin->fs_controls), FALSE);
    gtk_window_set_keep_above(GTK_WINDOW(mwin->fs_controls), TRUE);
    gtk_container_set_border_width(GTK_CONTAINER(mwin->fs_controls), BORDER/2);
    
    tophbox = gtk_hbox_new(FALSE, BORDER);
    gtk_widget_show(tophbox);
    gtk_container_add(GTK_CONTAINER(mwin->fs_controls), tophbox);
    
    gtk_icon_size_lookup(GTK_ICON_SIZE_BUTTON, &iw, &ih);
    
    img = xfmedia_squished_stock_icon_load(GTK_STOCK_MEDIA_PREVIOUS,
            GTK_ICON_SIZE_BUTTON);
    gtk_widget_show(img);
    btn = gtk_button_new();
    gtk_container_add(GTK_CONTAINER(btn), img);
    gtk_widget_show(btn);
    gtk_box_pack_start(GTK_BOX(tophbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "clicked",
            G_CALLBACK(xfmedia_prev_cb), mwin);
    
    mwin->fs_play_btn_img = img = xfmedia_squished_stock_icon_load(GTK_STOCK_MEDIA_PLAY,
            GTK_ICON_SIZE_BUTTON);
    g_object_ref(G_OBJECT(img));
    gtk_widget_show(img);
    mwin->fs_playpause_btn = btn = gtk_button_new();
    gtk_container_add(GTK_CONTAINER(btn), mwin->fs_play_btn_img);
    gtk_widget_show(btn);
    gtk_box_pack_start(GTK_BOX(tophbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "clicked",
            G_CALLBACK(xfmedia_playpause_cb), mwin);
    
    mwin->fs_pause_btn_img = img = xfmedia_squished_stock_icon_load(GTK_STOCK_MEDIA_PAUSE,
            GTK_ICON_SIZE_BUTTON);
    g_object_ref(G_OBJECT(img));
    gtk_widget_show(img);
    
    img = xfmedia_squished_stock_icon_load(GTK_STOCK_MEDIA_STOP,
            GTK_ICON_SIZE_BUTTON);
    gtk_widget_show(img);
    btn = gtk_button_new();
    gtk_container_add(GTK_CONTAINER(btn), img);
    gtk_widget_show(btn);
    gtk_box_pack_start(GTK_BOX(tophbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "clicked",
            G_CALLBACK(xfmedia_stop_cb), mwin);
    
    img = xfmedia_squished_stock_icon_load(GTK_STOCK_MEDIA_NEXT,
            GTK_ICON_SIZE_BUTTON);
    gtk_widget_show(img);
    btn = gtk_button_new();
    gtk_container_add(GTK_CONTAINER(btn), img);
    gtk_widget_show(btn);
    gtk_box_pack_start(GTK_BOX(tophbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "clicked",
            G_CALLBACK(xfmedia_next_cb), mwin);
    
    evtbox = gtk_event_box_new();
    gtk_event_box_set_visible_window(GTK_EVENT_BOX(evtbox), FALSE);
    gtk_widget_show(evtbox);
    gtk_box_pack_start(GTK_BOX(tophbox), evtbox, FALSE, FALSE, 0);
    gtk_widget_add_events(evtbox, GDK_BUTTON_PRESS);
    g_signal_connect(G_OBJECT(evtbox), "button-press-event",
            G_CALLBACK(xfmedia_pos_lbl_click_cb), mwin);
    
    mwin->fs_time_label = lbl = gtk_label_new(NULL);
    gtk_widget_show(lbl);
    gtk_container_add(GTK_CONTAINER(evtbox), lbl);
    
    mwin->fs_position_slider = slider = gtk_hscale_new(NULL);
    gtk_scale_set_draw_value(GTK_SCALE(slider), FALSE);
    gtk_range_set_increments(GTK_RANGE(slider), 1, 10);
    gtk_range_set_update_policy(GTK_RANGE(slider), GTK_UPDATE_DISCONTINUOUS);
    gtk_widget_show(slider);
    gtk_box_pack_start(GTK_BOX(tophbox), slider, TRUE, TRUE, 0);
    gtk_widget_add_events(mwin->fs_position_slider, GDK_MOTION_NOTIFY);
    g_signal_connect(G_OBJECT(slider), "value-changed",
            G_CALLBACK(xfmedia_position_slider_cb), mwin);
    g_signal_connect(G_OBJECT(slider), "button-press-event",
            G_CALLBACK(xfmedia_pos_slider_btn_press_cb), mwin);
    g_signal_connect(G_OBJECT(slider), "button-release-event",
            G_CALLBACK(xfmedia_pos_slider_btn_rls_cb), mwin);
    
    gtk_widget_realize(mwin->fs_controls);
    
    /* set 40% transparency (if supported) */
    opacity = (guint)(0.6*0xffffffff);
    gdk_property_change(mwin->fs_controls->window,
            gdk_atom_intern("_NET_WM_WINDOW_OPACITY", FALSE),
            gdk_atom_intern("CARDINAL", FALSE), 32,
            GDK_PROP_MODE_REPLACE, (const guchar *)&opacity, sizeof(guint));
}

static GtkWidget *
create_autoplay_menu_item(XfMediaMainwin *mwin)
{
    GtkWidget *menu, *ap_mi = NULL, *mi, *img;
    GList *autoplay_plugins, *l;
    
    /* xine's autoplay menu */
    autoplay_plugins = gtk_xine_get_autoplay_input_plugin_ids(mwin->gtx);
    if(autoplay_plugins) {
        img = gtk_image_new_from_stock(GTK_STOCK_NEW, GTK_ICON_SIZE_MENU);
        gtk_widget_show(img);
        ap_mi = mi = gtk_image_menu_item_new_with_mnemonic(_("_Other"));
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(mi), img);
        gtk_widget_show(mi);
        
        menu = gtk_menu_new();
        gtk_widget_show(menu);
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(mi), menu);
        
        for(l = autoplay_plugins; l; l = l->next) {
            mi = gtk_image_menu_item_new_with_label((const gchar *)l->data);
            gtk_widget_show(mi);
            if(strstr((const gchar *)l->data, "DVD") == l->data
                    || strstr((const gchar *)l->data, "CD") == l->data
                    || strstr((const gchar *)l->data, "VCD") == l->data)
            {
                img = gtk_image_new_from_stock(GTK_STOCK_CDROM,
                        GTK_ICON_SIZE_MENU);
                gtk_widget_show(img);
                gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(mi), img);
            }
            gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
            g_signal_connect(G_OBJECT(mi), "activate",
                    G_CALLBACK(xfmedia_mainwin_playlist_add_autoplay_cb), mwin);
        }
        g_list_free(autoplay_plugins);
    }
    
    return ap_mi;
}

XfMediaMainwin *
xfmedia_mainwin_new(GtkXine *gtx, GtkWindow *video_window)
{
    XfMediaMainwin *mwin;
    GtkWidget *topwin, *topvbox, *playlist_pane, *menu2;
    GtkWidget *hbox, *slider, *vbox, *btn, *lbl, *img, *menu, *mi, *evtbox;
    gint iw, ih, w, h_s, h_l;
    GtkTooltips *ttips;
    gchar *user_kb_file;
    
    g_return_val_if_fail(gtx != NULL, NULL);
    
    mwin = g_new0(XfMediaMainwin, 1);
    mwin->session_id = -1;
    
    DBG("setting up gtx signals");
    mwin->gtx = gtx;
    g_signal_connect(G_OBJECT(gtx), "button-press-event",
            G_CALLBACK(xfmedia_vwin_button_press_cb), mwin);
    g_signal_connect(G_OBJECT(gtx), "motion-notify-event",
            G_CALLBACK(xfmedia_vwin_motion_notify_cb), mwin);
    g_signal_connect(G_OBJECT(gtx), "stream-ended",
            G_CALLBACK(xfmedia_gtkxine_stream_ended_cb), mwin);
    g_signal_connect(G_OBJECT(gtx), "ui-message",
            G_CALLBACK(xfmedia_gtkxine_ui_message_cb), mwin);
    g_signal_connect(G_OBJECT(gtx), "progress-message",
            G_CALLBACK(xfmedia_gtkxine_progress_message_cb), mwin);
    g_signal_connect(G_OBJECT(gtx), "format-changed",
            G_CALLBACK(xfmedia_gtkxine_format_changed_cb), mwin);
    g_signal_connect(G_OBJECT(gtx), "mrl-reference",
            G_CALLBACK(xfmedia_gtkxine_mrl_reference_cb), mwin);
    DBG("  done");
    
    instantiate_extra_xine(mwin);
    
    mwin->video_window = GTK_WIDGET(video_window);
    gtk_widget_add_events(GTK_WIDGET(video_window), GDK_BUTTON_PRESS_MASK|GDK_KEY_PRESS_MASK);
    g_signal_connect(G_OBJECT(video_window), "button-press-event",
            G_CALLBACK(xfmedia_vwin_button_press_cb), mwin);
    g_signal_connect(G_OBJECT(video_window), "motion-notify-event",
            G_CALLBACK(xfmedia_vwin_motion_notify_cb), mwin);
    g_signal_connect(G_OBJECT(video_window), "delete-event",
            G_CALLBACK(xfmedia_vwin_delete_cb), mwin);
    g_signal_connect(G_OBJECT(video_window), "key-press-event",
            G_CALLBACK(xfmedia_mainwin_keypress_cb), mwin);
    g_signal_connect(G_OBJECT(video_window), "map-event",
            G_CALLBACK(xfmedia_vwin_mapunmap_cb), mwin);
    g_signal_connect(G_OBJECT(video_window), "unmap-event",
            G_CALLBACK(xfmedia_vwin_mapunmap_cb), mwin);
    
    mwin->tooltips = ttips = gtk_tooltips_new();
    
    mwin->window = topwin = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(topwin), "Xfmedia");
    gtk_container_set_border_width(GTK_CONTAINER(topwin), BORDER/2);
    if(xfmedia_settings_get_bool("/xfmedia/playlist/playlist_expanded")) {
        w = xfmedia_settings_get_int("/xfmedia/general/width");
        h_l = xfmedia_settings_get_int("/xfmedia/general/height_large");
        if(h_l != -1 &&  w != -1)
            gtk_window_resize(GTK_WINDOW(topwin), w, h_l);
        else
            gtk_window_resize(GTK_WINDOW(topwin), 475, 550);
    } else {
        w = xfmedia_settings_get_int("/xfmedia/general/width");
        h_s = xfmedia_settings_get_int("/xfmedia/general/height_small");
        if(h_s != -1 && w != -1)
            gtk_window_resize(GTK_WINDOW(topwin), w, h_s);
        else {
            gtk_window_get_size(GTK_WINDOW(topwin), &w, &h_s);
            gtk_window_resize(GTK_WINDOW(topwin), 500, h_s);
        }
    }
    gtk_widget_add_events(topwin, GDK_BUTTON_PRESS|GDK_CONFIGURE|GDK_KEY_PRESS);
    g_signal_connect(G_OBJECT(topwin), "button-press-event",
            G_CALLBACK(xfmedia_mainwin_menu_cb), mwin);
    g_signal_connect(G_OBJECT(topwin), "configure-event",
            G_CALLBACK(xfmedia_mainwin_configure_cb), mwin);
    g_signal_connect(G_OBJECT(topwin), "window-state-event",
            G_CALLBACK(xfmedia_mainwin_winstate_cb), mwin);
    g_signal_connect(G_OBJECT(topwin), "leave-notify-event",
            G_CALLBACK(xfmedia_mainwin_enterleave_notify_cb), mwin);
    g_signal_connect(G_OBJECT(topwin), "enter-notify-event",
            G_CALLBACK(xfmedia_mainwin_enterleave_notify_cb), mwin);
    g_signal_connect(G_OBJECT(topwin), "map-event",
            G_CALLBACK(xfmedia_mainwin_map_cb), mwin);
    g_signal_connect(G_OBJECT(topwin), "unmap-event",
            G_CALLBACK(xfmedia_mainwin_unmap_cb), mwin);
    g_signal_connect(G_OBJECT(topwin), "key-press-event",
            G_CALLBACK(xfmedia_mainwin_keypress_cb), mwin);
    
    xfmedia_keybindings_init();
    user_kb_file = xfce_resource_save_location(XFCE_RESOURCE_CONFIG,
            "xfmedia/keybindings.rc", FALSE);
    if(user_kb_file && g_file_test(user_kb_file, G_FILE_TEST_EXISTS)) {
        xfmedia_keybindings_load(SYSCONFDIR "/xdg/xfmedia/keybindings.rc");
        xfmedia_keybindings_load(user_kb_file);
    } else
        xfmedia_keybindings_load(NULL);
    if(user_kb_file)
        g_free(user_kb_file);
    
    topvbox = gtk_vbox_new(FALSE, 2);
    gtk_widget_show(topvbox);
    gtk_container_add(GTK_CONTAINER(topwin), topvbox);
    gtk_widget_add_events(topvbox, GDK_BUTTON_PRESS);
    g_signal_connect(G_OBJECT(topvbox), "button-press-event",
            G_CALLBACK(xfmedia_mainwin_menu_cb), mwin);
    
    evtbox = gtk_event_box_new();
    gtk_event_box_set_visible_window(GTK_EVENT_BOX(evtbox), FALSE);
    gtk_widget_show(evtbox);
    gtk_box_pack_start(GTK_BOX(topvbox), evtbox, FALSE, FALSE, 0);
    gtk_widget_add_events(evtbox, GDK_BUTTON_PRESS);
    g_signal_connect(G_OBJECT(evtbox), "button-press-event",
            G_CALLBACK(xfmedia_mainwin_menu_cb), mwin);
            
    hbox = gtk_hbox_new(FALSE, BORDER);
    gtk_widget_show(hbox);
    gtk_container_add(GTK_CONTAINER(evtbox), hbox);
    
    evtbox = gtk_event_box_new();
    gtk_event_box_set_visible_window(GTK_EVENT_BOX(evtbox), FALSE);
    gtk_widget_show(evtbox);
    gtk_box_pack_start(GTK_BOX(hbox), evtbox, FALSE, FALSE, 0);
    gtk_widget_add_events(evtbox, GDK_BUTTON_PRESS);
    g_signal_connect(G_OBJECT(evtbox), "button-press-event",
            G_CALLBACK(xfmedia_pos_lbl_click_cb), mwin);
    
    mwin->time_label = lbl = gtk_label_new(NULL);
    gtk_widget_show(lbl);
    gtk_container_add(GTK_CONTAINER(evtbox), lbl);
    
    mwin->song_label = lbl = exo_ellipsized_label_new("");
    exo_ellipsized_label_set_ellipsize(EXO_ELLIPSIZED_LABEL(lbl),
            EXO_PANGO_ELLIPSIZE_END);
    gtk_label_set_markup(GTK_LABEL(lbl), "<span size=\"large\"> </span>");
    gtk_label_set_use_markup(GTK_LABEL(lbl), TRUE);
    gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
    gtk_widget_show(lbl);
    gtk_box_pack_start(GTK_BOX(hbox), lbl, TRUE, TRUE, 0);
    
    mainwin_create_progress_box(mwin);
    gtk_box_pack_start(GTK_BOX(hbox), mwin->progress_box, TRUE, TRUE, 0);
    
    mainwin_create_ui_msg_label(mwin);
    gtk_box_pack_start(GTK_BOX(hbox), mwin->ui_message_label, TRUE, TRUE, 0);
    
    evtbox = gtk_event_box_new();
    gtk_event_box_set_visible_window(GTK_EVENT_BOX(evtbox), FALSE);
    gtk_widget_show(evtbox);
    gtk_box_pack_start(GTK_BOX(topvbox), evtbox, FALSE, FALSE, 0);
    gtk_widget_add_events(evtbox, GDK_BUTTON_PRESS);
    g_signal_connect(G_OBJECT(evtbox), "button-press-event",
            G_CALLBACK(xfmedia_mainwin_menu_cb), mwin);
    
    hbox = gtk_hbox_new(FALSE, BORDER-2);
    gtk_widget_show(hbox);
    gtk_container_add(GTK_CONTAINER(evtbox), hbox);
    
    mwin->position_slider = slider = gtk_hscale_new(NULL);
    gtk_scale_set_draw_value(GTK_SCALE(slider), FALSE);
    gtk_range_set_increments(GTK_RANGE(slider), 1, 10);
    gtk_range_set_update_policy(GTK_RANGE(slider), GTK_UPDATE_DISCONTINUOUS);
    gtk_widget_show(slider);
    gtk_box_pack_start(GTK_BOX(hbox), slider, TRUE, TRUE, 0);
    gtk_widget_add_events(mwin->position_slider, GDK_MOTION_NOTIFY);
    g_signal_connect(G_OBJECT(slider), "value-changed",
            G_CALLBACK(xfmedia_position_slider_cb), mwin);
    g_signal_connect(G_OBJECT(slider), "button-press-event",
            G_CALLBACK(xfmedia_pos_slider_btn_press_cb), mwin);
    g_signal_connect(G_OBJECT(slider), "button-release-event",
            G_CALLBACK(xfmedia_pos_slider_btn_rls_cb), mwin);
    
    gtk_icon_size_lookup(GTK_ICON_SIZE_BUTTON, &iw, &ih);
    
    img = xfmedia_squished_stock_icon_load(GTK_STOCK_MEDIA_PREVIOUS,
            GTK_ICON_SIZE_BUTTON);
    gtk_widget_show(img);
    btn = gtk_button_new();
    gtk_container_add(GTK_CONTAINER(btn), img);
    gtk_widget_show(btn);
    gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "clicked",
            G_CALLBACK(xfmedia_prev_cb), mwin);
    gtk_tooltips_set_tip(ttips, btn, _("Previous Track"), NULL);
    
    mwin->play_btn_img = img = xfmedia_squished_stock_icon_load(GTK_STOCK_MEDIA_PLAY,
            GTK_ICON_SIZE_BUTTON);
    g_object_ref(G_OBJECT(img));
    gtk_widget_show(img);
    mwin->playpause_btn = btn = gtk_button_new();
    gtk_container_add(GTK_CONTAINER(btn), img);
    gtk_widget_show(btn);
    gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "clicked",
            G_CALLBACK(xfmedia_playpause_cb), mwin);
    gtk_tooltips_set_tip(ttips, btn, _("Play/Pause Track"), NULL);
    
    mwin->pause_btn_img = img = xfmedia_squished_stock_icon_load(GTK_STOCK_MEDIA_PAUSE,
            GTK_ICON_SIZE_BUTTON);
    g_object_ref(G_OBJECT(img));
    gtk_widget_show(img);
    
    img = xfmedia_squished_stock_icon_load(GTK_STOCK_MEDIA_STOP,
            GTK_ICON_SIZE_BUTTON);
    gtk_widget_show(img);
    btn = gtk_button_new();
    gtk_container_add(GTK_CONTAINER(btn), img);
    gtk_widget_show(btn);
    gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "clicked",
            G_CALLBACK(xfmedia_stop_cb), mwin);
    gtk_tooltips_set_tip(ttips, btn, _("Stop Playback"), NULL);
    
    img = xfmedia_squished_stock_icon_load(GTK_STOCK_MEDIA_NEXT,
            GTK_ICON_SIZE_BUTTON);
    gtk_widget_show(img);
    btn = gtk_button_new();
    gtk_container_add(GTK_CONTAINER(btn), img);
    gtk_widget_show(btn);
    gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "clicked",
            G_CALLBACK(xfmedia_next_cb), mwin);
    gtk_tooltips_set_tip(ttips, btn, _("Next Track"), NULL);
    
    vbox = gtk_vbox_new(FALSE, 0);
    gtk_widget_show(vbox);
    hbox = gtk_hbox_new(FALSE, BORDER-4);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
    
    mwin->playlist_pane = playlist_pane = gtk_expander_new_with_mnemonic(_("_Playlist"));
    if(xfmedia_settings_get_bool("/xfmedia/playlist/playlist_expanded"))
        gtk_expander_set_expanded(GTK_EXPANDER(playlist_pane), TRUE);
    else
        gtk_expander_set_expanded(GTK_EXPANDER(playlist_pane), FALSE);
    gtk_widget_show(playlist_pane);
    gtk_box_pack_start(GTK_BOX(topvbox), playlist_pane, TRUE, TRUE, 0);
    g_signal_connect_after(G_OBJECT(playlist_pane), "activate",
            G_CALLBACK(xfmedia_mainwin_expander_activate_cb), mwin);
    
    mwin->plist = XFMEDIA_PLAYLIST(xfmedia_playlist_new());
    xfmedia_playlist_set_shuffle_state(mwin->plist,
            xfmedia_settings_get_bool("/xfmedia/playlist/shuffle"));
    xfmedia_playlist_set_repeat_state(mwin->plist,
           xfmedia_settings_get_bool("/xfmedia/playlist/repeat"));
    gtk_widget_show(GTK_WIDGET(mwin->plist));
    gtk_container_add(GTK_CONTAINER(playlist_pane), GTK_WIDGET(mwin->plist));
    g_signal_connect(G_OBJECT(mwin->plist), "playlist-scrolled",
            G_CALLBACK(xfmedia_mainwin_playlist_scrolled_cb), mwin);
    g_signal_connect(G_OBJECT(mwin->plist), "entry-activated",
            G_CALLBACK(xfmedia_mainwin_playlist_entry_activated_cb), mwin);
    g_signal_connect(G_OBJECT(mwin->plist), "shuffle-toggled",
            G_CALLBACK(xfmedia_mainwin_playlist_shuffle_toggled_cb), mwin);
    g_signal_connect(G_OBJECT(mwin->plist), "repeat-toggled",
            G_CALLBACK(xfmedia_mainwin_playlist_repeat_toggled_cb), mwin);
    g_signal_connect(G_OBJECT(mwin->plist), "playlist-cleared",
            G_CALLBACK(xfmedia_mainwin_playlist_repeat_toggled_cb), mwin);
    
    mi = create_autoplay_menu_item(mwin);
    if(mi)
        xfmedia_playlist_augment_add_menu(mwin->plist, GTK_MENU_ITEM(mi));
    
    img = gtk_image_new_from_stock(GTK_STOCK_ADD, GTK_ICON_SIZE_MENU);
    gtk_widget_show(mi);
    mi = gtk_image_menu_item_new_with_mnemonic(_("Add _Mediamark..."));
    gtk_widget_show(mi);
    gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(mi), img);
    xfmedia_playlist_context_menu_prepend_item(mwin->plist, GTK_MENU_ITEM(mi));
    g_signal_connect(G_OBJECT(mi), "activate",
            G_CALLBACK(xfmedia_mainwin_playlist_add_mm_cb), mwin);
    
    mi = gtk_separator_menu_item_new();
    gtk_widget_show(mi);
    xfmedia_playlist_context_menu_prepend_item(mwin->plist, GTK_MENU_ITEM(mi));
    
    img = gtk_image_new_from_stock(GTK_STOCK_DIALOG_INFO, GTK_ICON_SIZE_MENU);
    gtk_widget_show(img);
    mi = gtk_image_menu_item_new_with_mnemonic(_("_File Info"));
    gtk_widget_show(mi);
    gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(mi), img);
    xfmedia_playlist_context_menu_prepend_item(mwin->plist, GTK_MENU_ITEM(mi));
    g_signal_connect(G_OBJECT(mi), "activate",
            G_CALLBACK(xfmedia_mainwin_playlist_file_info_cb), mwin);
    
    /* right-click popup menu */
    
    mwin->popup_menu = menu = gtk_menu_new();
    gtk_widget_show(menu);
    
    mi = gtk_image_menu_item_new_with_mnemonic(_("About _Xfmedia"));
    img = gtk_image_new_from_stock(GTK_STOCK_ABOUT, GTK_ICON_SIZE_MENU);
    gtk_widget_show(img);
    gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(mi), img);
    gtk_widget_show(mi);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
    g_signal_connect(G_OBJECT(mi), "activate",
            G_CALLBACK(xfmedia_about_cb), mwin);
    
    mi = gtk_separator_menu_item_new();
    gtk_widget_show(mi);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
    
    img = gtk_image_new_from_stock(GTK_STOCK_PREFERENCES, GTK_ICON_SIZE_MENU);
    gtk_widget_show(img);
    mi = gtk_image_menu_item_new_with_mnemonic(_("_Preferences"));
    gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(mi), img);
    gtk_widget_show(mi);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
    g_signal_connect_swapped(G_OBJECT(mi), "activate",
            G_CALLBACK(xfmedia_settings_show_dialog), mwin);
    
#if 0
    mi = gtk_image_menu_item_new_with_mnemonic(_("_Equalizer..."));
    gtk_widget_show(mi);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
    g_signal_connect_swapped(G_OBJECT(mi), "activate",
            G_CALLBACK(xfmedia_equaliser_show), mwin);
#endif

    mi = gtk_separator_menu_item_new();
    gtk_widget_show(mi);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);

    mi = gtk_image_menu_item_new_with_mnemonic(_("_Jump to File..."));
    img = gtk_image_new_from_stock(GTK_STOCK_JUMP_TO, GTK_ICON_SIZE_MENU);
    gtk_widget_show(img);
    gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(mi), img);
    gtk_widget_show(mi);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
    g_signal_connect_swapped(G_OBJECT(mi), "activate",
            G_CALLBACK(xfmedia_playlist_trigger_jump_to_file), mwin->plist);
    
    mwin->mmarks_mi = mi = gtk_image_menu_item_new_with_mnemonic(_("_Mediamarks"));
    gtk_widget_show(mi);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
    menu2 = xfmedia_mediamarks_create_menu(mwin);
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(mi), menu2);
    xfmedia_mediamarks_hook_changed(xfmedia_mainwin_mediamarks_changed_cb, mwin);
    
    mi = gtk_image_menu_item_new_with_mnemonic(_("Re_order"));
    gtk_widget_show(mi);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
    
    menu2 = gtk_menu_new();
    gtk_widget_show(menu2);
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(mi), menu2);
    
    mi = gtk_image_menu_item_new_with_mnemonic(_("Sort by _Title"));
    gtk_widget_show(mi);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu2), mi);
    g_signal_connect(G_OBJECT(mi), "activate",
            G_CALLBACK(xfmedia_mainwin_sort_title_cb), mwin);
    
    mi = gtk_image_menu_item_new_with_mnemonic(_("Sort by _Filename"));
    gtk_widget_show(mi);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu2), mi);
    g_signal_connect(G_OBJECT(mi), "activate",
            G_CALLBACK(xfmedia_mainwin_sort_filename_cb), mwin);
            
    mi = gtk_image_menu_item_new_with_mnemonic(_("_Randomize List"));
    gtk_widget_show(mi);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu2), mi);
    g_signal_connect(G_OBJECT(mi), "activate",
            G_CALLBACK(xfmedia_mainwin_randomise_playlist_cb), mwin);
    
    mi = gtk_separator_menu_item_new();
    gtk_widget_show(mi);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
    
    gtk_icon_size_lookup(GTK_ICON_SIZE_MENU, &iw, &ih);
    
    mwin->shuffle_mi = mi = gtk_check_menu_item_new_with_mnemonic(_("_Shuffle"));
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mi),
            xfmedia_settings_get_bool("/xfmedia/playlist/shuffle"));
    gtk_widget_show(mi);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
    g_signal_connect(G_OBJECT(mi), "toggled",
            G_CALLBACK(xfmedia_shuffle_menu_toggle_cb), mwin);
    
    mwin->repeat_mi = mi = gtk_check_menu_item_new_with_mnemonic(_("_Repeat"));
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mi),
            xfmedia_settings_get_bool("/xfmedia/playlist/repeat"));
    gtk_widget_show(mi);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
    g_signal_connect(G_OBJECT(mi), "toggled",
            G_CALLBACK(xfmedia_repeat_menu_toggle_cb), mwin);
    
    mi = gtk_check_menu_item_new_with_mnemonic(_("Repeat Single _Track"));
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mi),
            xfmedia_settings_get_bool("/xfmedia/playlist/repeat_one"));
    gtk_widget_show(mi);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
    g_signal_connect(G_OBJECT(mi), "toggled",
            G_CALLBACK(xfmedia_repeat_one_menu_toggle_cb), mwin);
    
    mwin->svis_mi = mi = gtk_check_menu_item_new_with_mnemonic(_("Show Audio _Visualization"));
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mi),
            xfmedia_settings_get_bool("/xfmedia/general/show_audio_vis"));
    gtk_widget_show(mi);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
    g_signal_connect(G_OBJECT(mi), "toggled",
            G_CALLBACK(xfmedia_show_vis_menu_toggle_cb), mwin);
    
    mi = gtk_separator_menu_item_new();
    gtk_widget_show(mi);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
    
    mi = gtk_image_menu_item_new_from_stock(GTK_STOCK_QUIT, NULL);
    gtk_widget_show(mi);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
    g_signal_connect(G_OBJECT(mi), "activate",
            G_CALLBACK(xfmedia_mainwin_quit_mi_cb), mwin);
    
    gtk_widget_realize(topwin);
    
    create_fs_control_window(mwin);
    create_autohide_window(mwin);
    
    xfmedia_mainwin_set_time_label(mwin, 0, FALSE, NULL);
    
    mwin->shuffle_history = g_queue_new();
    
    h_s = xfmedia_settings_get_int("/xfmedia/general/height_small");
    if(xfmedia_settings_get_bool("/xfmedia/playlist/playlist_expanded")) {
        h_l = xfmedia_settings_get_int("/xfmedia/general/height_large");
        if(h_l == -1) {
            gtk_window_get_size(GTK_WINDOW(topwin), &w, &h_l);
            xfmedia_settings_set_int("/xfmedia/general/height_large", h_l);
            xfmedia_settings_set_int("/xfmedia/general/width", w);
        }
    } else if(h_s == -1) {
        gtk_window_get_size(GTK_WINDOW(topwin), &w, &h_s);
        xfmedia_settings_set_int("/xfmedia/general/height_small", h_s);
        xfmedia_settings_set_int("/xfmedia/general/width", w);
    }
    
    if(xfmedia_settings_get_bool("/xfmedia/general/window_sticky"))
        gtk_window_stick(GTK_WINDOW(topwin));
    gtk_window_set_keep_above(GTK_WINDOW(topwin),
            xfmedia_settings_get_bool("/xfmedia/general/window_above"));
    
    mwin->pos_id = g_timeout_add(500, xfmedia_position_cb, mwin);
    
    return mwin;
}

void
xfmedia_mainwin_set_infobar_type(XfMediaMainwin *mwin, XfMediaInfobarType type)
{
    g_return_if_fail(mwin && type < XFMEDIA_INFOBAR_N_TYPES);
    
    switch(type) {
        case XFMEDIA_INFOBAR_SONG_LABEL:
            if(GTK_WIDGET_VISIBLE(mwin->progress_box))
                gtk_widget_hide(mwin->progress_box);
            if(GTK_WIDGET_VISIBLE(mwin->ui_message_label))
                gtk_widget_hide(mwin->ui_message_label);
            
            if(!GTK_WIDGET_VISIBLE(mwin->song_label))
                gtk_widget_show(mwin->song_label);
            break;
        
        case XFMEDIA_INFOBAR_PROGRESS_MESSAGE:
            if(GTK_WIDGET_VISIBLE(mwin->song_label))
                gtk_widget_hide(mwin->song_label);
            if(GTK_WIDGET_VISIBLE(mwin->ui_message_label))
                gtk_widget_hide(mwin->ui_message_label);
            
            if(!GTK_WIDGET_VISIBLE(mwin->progress_box))
                gtk_widget_show(mwin->progress_box);
            break;
        
        case XFMEDIA_INFOBAR_UI_MESSAGE:
            if(GTK_WIDGET_VISIBLE(mwin->song_label))
                gtk_widget_hide(mwin->song_label);
            if(GTK_WIDGET_VISIBLE(mwin->progress_box))
                gtk_widget_hide(mwin->progress_box);
            
            if(!GTK_WIDGET_VISIBLE(mwin->ui_message_label))
                gtk_widget_show(mwin->ui_message_label);
            break;
        
        default:
            break;
    }
}

gboolean
xfmedia_mainwin_get_file_info(XfMediaMainwin *mwin, const gchar *filename,
        gchar **title, gint *length)
{
    xine_stream_t *stream;
    
    g_return_val_if_fail(title && length, FALSE);
    
    if(filename) {
        stream = mwin->extra_stream;
        if(!xine_open(stream, filename))
            return FALSE;
    } else
        stream = gtk_xine_get_raw_stream(mwin->gtx);
    
    if(stream) {
        gint pos_stream, pos_time;
        
        *title = xfmedia_sinfo_to_string(stream,
                xfmedia_settings_get_string("/xfmedia/playlist/playlist_format"),
                filename ? filename : gtk_xine_get_current_mrl(mwin->gtx));
        
        /* if this fails, return -2 as the length */
        if(filename) {
            if(xine_get_pos_length(stream, &pos_stream, &pos_time, length))
                *length /= 1000;
            else
                *length = -2;
        } else {
            if(gtk_xine_get_pos_length(mwin->gtx, &pos_stream, &pos_time, length))
                *length /= 1000;
            else
                *length = -2;
        }
        
        if(filename)
            xine_close(stream);
    } else
        return FALSE;
    
    if(*title) {
        gchar *title_utf8 = xfmedia_brutal_utf8_validate(*title);
        if(title_utf8) {
            g_free(*title);
            *title = title_utf8;
        }
    }
    
    return TRUE;
}

void
xfmedia_mainwin_set_time_label(XfMediaMainwin *mwin, gint time,
        gboolean is_remaining, const gchar *extra_pango_markup)
{
    gchar time_lbl[128];
    
    g_snprintf(time_lbl, 128,
            "<span font_family=\"monospace\" size=\"large\" weight=\"bold\" %s>%c%02d:%02d</span>",
            extra_pango_markup ? extra_pango_markup : "",
            is_remaining ? '-' : ' ',
            time / 60, time % 60);
    gtk_label_set_markup(GTK_LABEL(mwin->time_label), time_lbl);
    
    g_snprintf(time_lbl, 128,
            "<span font_family=\"monospace\" %s>%c%02d:%02d</span>",
            extra_pango_markup ? extra_pango_markup : "",
            is_remaining ? '-' : ' ',
            time / 60, time % 60);
    gtk_label_set_markup(GTK_LABEL(mwin->fs_time_label), time_lbl);
}

static gint
get_base_text_size()
{
    GtkWidget *dummy;
    gint base_size = -1;
    
    dummy = gtk_invisible_new();
    gtk_widget_realize(dummy);
    
    if(dummy->style)
        base_size = pango_font_description_get_size(dummy->style->font_desc);
    
    gtk_widget_destroy(dummy);
    
    return base_size;
}

void
xfmedia_mainwin_set_song_label(XfMediaMainwin *mwin, const gchar *str)
{
    gint base_size;
    
    xfmedia_mainwin_set_infobar_type(mwin, XFMEDIA_INFOBAR_SONG_LABEL);
    
    g_return_if_fail(mwin && str);
    
    base_size = get_base_text_size();
    if(base_size != -1) {
        PangoFontDescription *font_desc = pango_font_description_new();
        pango_font_description_set_size(font_desc, base_size*PANGO_SCALE_LARGE);
        gtk_widget_modify_font(mwin->song_label, font_desc);
        pango_font_description_free(font_desc);
        gtk_label_set_use_markup(GTK_LABEL(mwin->song_label), FALSE);
        gtk_label_set_text(GTK_LABEL(mwin->song_label), str);
    } else {
        gchar *str_esc, *str_markup;
        str_esc = g_markup_escape_text(str, strlen(str));
        str_markup = g_strdup_printf("<span size=\"large\">%s</span>", str_esc);
        g_free(str_esc);
        gtk_label_set_use_markup(GTK_LABEL(mwin->song_label), TRUE);
        gtk_label_set_markup(GTK_LABEL(mwin->song_label), str_markup);
        g_free(str_markup);
    }
    
    if(xfmedia_settings_get_bool("/xfmedia/general/show_tray_icon")) {
        gchar *str_tt = g_strdup_printf("%s: %s", _("Playing"), str);
        xfmedia_tray_icon_set_tooltip(mwin->tray_icon, str_tt);
        g_free(str_tt);
    }
}

static GtkWidget *
tray_icon_menu_create(XfMediaMainwin *mwin)
{
    GtkWidget *menu, *mi;
    gint iw, ih;
    
    gtk_icon_size_lookup(GTK_ICON_SIZE_MENU, &iw, &ih);
    
    menu = gtk_menu_new();
    gtk_widget_show(menu);
    
    mi = gtk_image_menu_item_new_with_mnemonic(_("P_revious Track"));
    gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(mi),
            gtk_image_new_from_stock(GTK_STOCK_MEDIA_PREVIOUS, GTK_ICON_SIZE_MENU));
    gtk_widget_show(mi);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
    g_signal_connect(G_OBJECT(mi), "activate",
            G_CALLBACK(xfmedia_prev_cb), mwin);
    
    mwin->tray_icon_playpause_btn = mi =
            gtk_image_menu_item_new_with_mnemonic(_("_Play"));
    mwin->tray_icon_play_img = gtk_image_new_from_stock(GTK_STOCK_MEDIA_PLAY,
            GTK_ICON_SIZE_MENU);
    g_object_ref(G_OBJECT(mwin->tray_icon_play_img));
    if(gtk_xine_get_param(mwin->gtx, XINE_PARAM_SPEED) != XINE_SPEED_NORMAL
            || gtk_xine_get_status(mwin->gtx) != XINE_STATUS_PLAY)
    {
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(mi),
                mwin->tray_icon_play_img);
    }
    mwin->tray_icon_pause_img = gtk_image_new_from_stock(GTK_STOCK_MEDIA_PAUSE,
            GTK_ICON_SIZE_MENU);
    g_object_ref(G_OBJECT(mwin->tray_icon_pause_img));
    if(gtk_xine_get_param(mwin->gtx, XINE_PARAM_SPEED) == XINE_SPEED_NORMAL
            && gtk_xine_get_status(mwin->gtx) == XINE_STATUS_PLAY)
    {
        gtk_label_set_text(GTK_LABEL(gtk_bin_get_child(GTK_BIN(mi))),
                _("_Pause"));
        gtk_label_set_use_underline(GTK_LABEL(gtk_bin_get_child(GTK_BIN(mi))),
                TRUE);
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(mi),
                mwin->tray_icon_pause_img);
    }
    gtk_widget_show(mi);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
    g_signal_connect(G_OBJECT(mi), "activate",
            G_CALLBACK(xfmedia_playpause_cb), mwin);
    
    mi = gtk_image_menu_item_new_with_mnemonic(_("_Stop"));
    gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(mi),
            gtk_image_new_from_stock(GTK_STOCK_MEDIA_STOP, GTK_ICON_SIZE_MENU));
    gtk_widget_show(mi);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
    g_signal_connect(G_OBJECT(mi), "activate",
            G_CALLBACK(xfmedia_stop_cb), mwin);
    
    mi = gtk_image_menu_item_new_with_mnemonic(_("_Next Track"));
    gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(mi),
            gtk_image_new_from_stock(GTK_STOCK_MEDIA_NEXT, GTK_ICON_SIZE_MENU));
    gtk_widget_show(mi);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
    g_signal_connect(G_OBJECT(mi), "activate",
            G_CALLBACK(xfmedia_next_cb), mwin);
    
    mi = gtk_separator_menu_item_new();
    gtk_widget_show(mi);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
    
    mi = gtk_image_menu_item_new_from_stock(GTK_STOCK_QUIT, NULL);
    gtk_widget_show(mi);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
    g_signal_connect_swapped(G_OBJECT(mi), "activate",
            G_CALLBACK(gtk_widget_destroy), mwin->window);
    
    return menu;
}

void
xfmedia_mainwin_activate_systray(XfMediaMainwin *mwin)
{
    GdkScreen *gscreen = gtk_widget_get_screen(mwin->window);
    
    if(!mwin->tray_icon_menu)
        mwin->tray_icon_menu = tray_icon_menu_create(mwin);
    
    mwin->tray_icon = xfmedia_tray_icon_create(gscreen, mwin,
            GTK_MENU(mwin->tray_icon_menu));
}

void
xfmedia_mainwin_deactivate_systray(XfMediaMainwin *mwin)
{
    g_return_if_fail(mwin->tray_icon);
    
    xfmedia_tray_icon_destroy(mwin->tray_icon);
    mwin->tray_icon = NULL;
}

void
xfmedia_mainwin_connect_slider_callback(XfMediaMainwin *mwin)
{
    g_signal_handlers_unblock_by_func(G_OBJECT(mwin->position_slider),
            G_CALLBACK(xfmedia_position_slider_cb), mwin);
    g_signal_handlers_unblock_by_func(G_OBJECT(mwin->fs_position_slider),
            G_CALLBACK(xfmedia_position_slider_cb), mwin);
}

void
xfmedia_mainwin_disconnect_slider_callback(XfMediaMainwin *mwin)
{
    g_signal_handlers_block_by_func(G_OBJECT(mwin->position_slider),
            G_CALLBACK(xfmedia_position_slider_cb), mwin);
    g_signal_handlers_block_by_func(G_OBJECT(mwin->fs_position_slider),
            G_CALLBACK(xfmedia_position_slider_cb), mwin);
}

void
xfmedia_mainwin_set_play_buttons(XfMediaMainwin *mwin)
{
    gtk_container_remove(GTK_CONTAINER(mwin->playpause_btn),
            gtk_bin_get_child(GTK_BIN(mwin->playpause_btn)));
    gtk_container_add(GTK_CONTAINER(mwin->playpause_btn),
            mwin->play_btn_img);
    gtk_container_remove(GTK_CONTAINER(mwin->fs_playpause_btn),
            gtk_bin_get_child(GTK_BIN(mwin->fs_playpause_btn)));
    gtk_container_add(GTK_CONTAINER(mwin->fs_playpause_btn),
            mwin->fs_play_btn_img);
    if(mwin->tray_icon_playpause_btn && mwin->tray_icon_play_img) {
        GtkWidget *lbl;
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(mwin->tray_icon_playpause_btn),
                mwin->tray_icon_play_img);
        lbl = gtk_bin_get_child(GTK_BIN(mwin->tray_icon_playpause_btn));
        gtk_label_set_text(GTK_LABEL(lbl), _("_Play"));
        gtk_label_set_use_underline(GTK_LABEL(lbl), TRUE);
    }
}

void
xfmedia_mainwin_set_pause_buttons(XfMediaMainwin *mwin)
{
    gtk_container_remove(GTK_CONTAINER(mwin->playpause_btn),
            gtk_bin_get_child(GTK_BIN(mwin->playpause_btn)));
    gtk_container_add(GTK_CONTAINER(mwin->playpause_btn),
            mwin->pause_btn_img);
    gtk_container_remove(GTK_CONTAINER(mwin->fs_playpause_btn),
            gtk_bin_get_child(GTK_BIN(mwin->fs_playpause_btn)));
    gtk_container_add(GTK_CONTAINER(mwin->fs_playpause_btn),
            mwin->fs_pause_btn_img);
    if(mwin->tray_icon_playpause_btn && mwin->tray_icon_pause_img) {
        GtkWidget *lbl;
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(mwin->tray_icon_playpause_btn),
                mwin->tray_icon_pause_img);
        lbl = gtk_bin_get_child(GTK_BIN(mwin->tray_icon_playpause_btn));
        gtk_label_set_text(GTK_LABEL(lbl), _("_Pause"));
        gtk_label_set_use_underline(GTK_LABEL(lbl), TRUE);
    }
}

void
xfmedia_mainwin_destroy(XfMediaMainwin *mwin)
{    
    xfmedia_keybindings_cleanup();
    
    if(mwin->gtx)
        gtk_xine_stop(mwin->gtx);
    
    if(mwin->pos_id) {
        g_source_remove(mwin->pos_id);
        mwin->pos_id = 0;
    }
    
    if(mwin->tooltips) {
        gtk_object_sink(GTK_OBJECT(mwin->tooltips));
        mwin->tooltips = NULL;
    }
    
    if(mwin->shuffle_history) {
        gint i, tot = g_queue_get_length(mwin->shuffle_history);
        for(i = 0; i < tot; i++)
            xfmedia_playlist_entry_ref_destroy((XfMediaPlaylistEntryRef *)g_queue_peek_nth(mwin->shuffle_history, i));
        g_queue_free(mwin->shuffle_history);
        mwin->shuffle_history = NULL;
    }
    
    DBG("closing extra stream and a/v drivers");
    if(mwin->extra_stream) {
        xine_close(mwin->extra_stream);
        xine_dispose(mwin->extra_stream);
        xine_close_audio_driver(gtk_xine_get_raw_xine_engine(mwin->gtx),
                mwin->extra_ap);
        xine_close_video_driver(gtk_xine_get_raw_xine_engine(mwin->gtx),
                mwin->extra_vp);
        mwin->extra_stream = NULL;
        mwin->extra_ap = NULL;
        mwin->extra_vp = NULL;
    }
    
    DBG("killing tray icon");
    if(mwin->tray_icon) {
        xfmedia_tray_icon_destroy(mwin->tray_icon);
        mwin->tray_icon = NULL;
    }
    
    if(mwin->fs_controls) {
        gtk_widget_destroy(mwin->fs_controls);
        mwin->fs_controls = NULL;
    }
    
    if(mwin->autohide_window) {
        gtk_widget_destroy(mwin->autohide_window);
        mwin->autohide_window = NULL;
    }
    
    g_free(mwin);
    mwin = NULL;
}

gboolean
xfmedia_mainwin_play_uri(XfMediaMainwin *mwin, const gchar *uri)
{
    gboolean ret = FALSE;
    
    gtk_xine_stop(mwin->gtx);   
    DBG("about to call gtk_xine_open()"); 
    if(gtk_xine_open(mwin->gtx, uri)) {
        gboolean was_visible = (GTK_WIDGET_VISIBLE(mwin->video_window)
                || gtk_xine_is_fullscreen(mwin->gtx));
        gboolean has_video = gtk_xine_get_stream_info(mwin->gtx,
                XINE_STREAM_INFO_HAS_VIDEO);
        gboolean needs_vwin_show = FALSE;
        
        DBG("  gtk_xine_open() returned");
        
        xfmedia_plugins_forward_signal_VOID("stream-loaded");
        
        has_video = gtk_xine_get_stream_info(mwin->gtx,
                XINE_STREAM_INFO_HAS_VIDEO);
        
        if(!xfmedia_settings_get_bool("/xfmedia/general/show_audio_vis")
                && was_visible && !has_video)
        {
            if(gtk_xine_is_fullscreen(mwin->gtx)) {
                if(GTK_WIDGET_VISIBLE(mwin->fs_controls))
                    gtk_widget_hide(mwin->fs_controls);
                gtk_xine_set_fullscreen(mwin->gtx, FALSE);
            }
            gtk_widget_hide(mwin->video_window);
        } else if(has_video) {
            if(!was_visible)
                needs_vwin_show = TRUE;
            gtk_xine_set_vis(mwin->gtx, NULL);
        } else if(xfmedia_settings_get_bool("/xfmedia/general/show_audio_vis")
                && !has_video)
        {
            needs_vwin_show = TRUE;
            gtk_xine_set_vis(mwin->gtx,
                    xfmedia_settings_get_string("/xfmedia/general/selected_vis"));
        }
        
        DBG("dealt with video window");
        
        gtk_xine_set_param(mwin->gtx, XINE_PARAM_AUDIO_CLOSE_DEVICE, 0);
        if(!gtk_xine_play(mwin->gtx, 0, 0)) {
            DBG("gtk_xine_play() failed");
            if(!was_visible)
                gtk_widget_hide(mwin->video_window);
            ret = FALSE;
            gtk_xine_set_param(mwin->gtx, XINE_PARAM_AUDIO_CLOSE_DEVICE, 1);
        } else {
            gchar *name = NULL;
            gint song_len = -1;
            
            if(!xfmedia_mainwin_get_file_info(mwin, NULL, &name, &song_len) || !name)
                name = xfmedia_filename_to_name(uri);
            xfmedia_mainwin_set_song_label(mwin, name);
            g_free(name);
            
            if(has_video) {
                gint width, height;
                
                width = gtk_xine_get_stream_info(mwin->gtx,
                        XINE_STREAM_INFO_VIDEO_WIDTH);
                height = gtk_xine_get_stream_info(mwin->gtx,
                        XINE_STREAM_INFO_VIDEO_HEIGHT);
                if(width < 1 || height < 1) {
                    width = 480;
                    height = 360;
                }
                
                if(needs_vwin_show || xfmedia_settings_get_bool("/xfmedia/general/show_audio_vis")) {
                    GdkGeometry geom;
                    
                    geom.base_width = width;
                    geom.base_height = height;
                    geom.min_width = geom.min_height = 1;
                    geom.min_aspect = geom.max_aspect = (gdouble)width / height;
                    DBG("window size: %dx%d, aspect: %f", width, height, geom.max_aspect);
                    gtk_window_set_geometry_hints(GTK_WINDOW(mwin->video_window),
                            GTK_WIDGET(mwin->gtx), &geom,
                            GDK_HINT_MIN_SIZE|GDK_HINT_ASPECT|GDK_HINT_BASE_SIZE);
                    
                    gtk_widget_show(mwin->video_window);
                } else {
                    gtk_window_resize(GTK_WINDOW(mwin->video_window),
                            width, height);
                }
            }
            
            xfmedia_mainwin_disconnect_slider_callback(mwin);
            gtk_range_set_value(GTK_RANGE(mwin->position_slider), 0.0);
            gtk_range_set_value(GTK_RANGE(mwin->fs_position_slider), 0.0);
            xfmedia_mainwin_connect_slider_callback(mwin);
            
            if(gtk_xine_get_stream_info(mwin->gtx, XINE_STREAM_INFO_SEEKABLE)) {
                gtk_widget_set_sensitive(mwin->position_slider, TRUE);
                gtk_widget_set_sensitive(mwin->fs_position_slider, TRUE);
            } else {
                gtk_widget_set_sensitive(mwin->position_slider, FALSE);
                gtk_widget_set_sensitive(mwin->fs_position_slider, FALSE);
            }
            
            mwin->cur_playing_length = song_len;
            if(song_len <= 0)
                song_len = 1;
            
            gtk_range_set_range(GTK_RANGE(mwin->position_slider), 0.0,
                    song_len>0 ? (gdouble)song_len : 1.0);
            gtk_range_set_range(GTK_RANGE(mwin->fs_position_slider), 0.0,
                    song_len>0 ? (gdouble)song_len : 1.0);
            
            xfmedia_mainwin_set_pause_buttons(mwin);
            
            xfmedia_plugins_forward_signal_VOID("stream-started");
            ret = TRUE;
        }
    }
    
    return ret;
}

static gint
mainwin_get_next_index(XfMediaMainwin *mwin, gint last_index)
{
    gint tot_items, next_id, attempts = 0, cutoff;
    
    if(xfmedia_settings_get_bool("/xfmedia/playlist/shuffle")) {
        tot_items = xfmedia_playlist_get_n_entries(mwin->plist);
        cutoff = tot_items * 0.75;
        
        do {
            if(attempts > 0)
                DBG("shuffle collision!  trying again (attempt %d)", attempts);
            
#if HAVE_SRANDOM
            next_id = random() % tot_items;
#else
            next_id = rand() % tot_items;
#endif
            if(++attempts > cutoff) {
                xfmedia_playlist_shuffle_unset_all(mwin->plist);
                attempts = 0;
            }
        } while(!xfmedia_playlist_shuffle_check(mwin->plist, next_id));
    } else if(last_index == -1 && mwin->cur_playing) {
        tot_items = xfmedia_playlist_get_n_entries(mwin->plist);
        if(tot_items == 0)
            return -1;
        last_index = xfmedia_playlist_entry_ref_get_index(mwin->cur_playing);
        if(last_index >= tot_items - 1) {
            /* this means we're at the end of the list (i hope) */
            if(xfmedia_settings_get_bool("/xfmedia/playlist/repeat"))
                next_id = 0;
            else {
                gtk_xine_stop(mwin->gtx);
                if(gtk_xine_is_fullscreen(mwin->gtx))
                    gtk_xine_set_fullscreen(mwin->gtx, FALSE);
                if(GTK_WIDGET_VISIBLE(mwin->video_window))
                    gtk_widget_hide(mwin->video_window);
                next_id = -1;
            }
        } else
            next_id = last_index + 1;
    } else
        next_id = 0;
    
    return next_id;
}

gboolean
xfmedia_mainwin_play_file_at_index(XfMediaMainwin *mwin, gint index)
{
    gchar *filename;
    gboolean ret = FALSE;
    gint index_start = -1, index_end = -1;
    
    if(index == -1) {
        index = mainwin_get_next_index(mwin, -1);
        if(index == -1)
            return FALSE;
    }
    
    if(mwin->cur_playing) {
        xfmedia_playlist_entry_ref_destroy(mwin->cur_playing);
        mwin->cur_playing = NULL;
        mwin->cur_playing_length = 0;
    }
    
    if(xfmedia_playlist_get(mwin->plist, index, NULL, NULL, &filename) && filename) {
        if(!xfmedia_playlist_get_metadata_loaded(mwin->plist, index)) {
            gchar *title = NULL;
            gint length = 0;
            
            if(xfmedia_mainwin_get_file_info(mwin, filename, &title, &length)) {
                if((title && !*title) || !title) {
                    g_free(title);
                    title = xfmedia_filename_to_name(filename);
                }
                xfmedia_playlist_modify_entry(mwin->plist, index, title, length,
                        filename);
                g_free(title);
                xfmedia_playlist_set_metadata_loaded(mwin->plist, index, TRUE);
            }
        }
        
        if(xfmedia_playlist_get_visible_range(mwin->plist, &index_start, &index_end)) {
            if(index_start > index || index_end < index)
                xfmedia_playlist_scroll_to_index(mwin->plist, index, 0.5);
        }
        mwin->cur_playing = xfmedia_playlist_entry_ref_new_for_index(mwin->plist, index);
        xfmedia_playlist_set_bold_entry(mwin->plist, mwin->cur_playing);
        xfmedia_settings_set_int("/xfmedia/playlist/playlist_position", index);
        xfmedia_playlist_shuffle_check_set(mwin->plist, index);
        
        if(xfmedia_mainwin_play_uri(mwin, filename))
            ret = TRUE;
        
        g_free(filename);
    }
    
    return ret;
}

gboolean
xfmedia_mainwin_play_file_at_index_persistent(XfMediaMainwin *mwin, gint index)
{
    gboolean done = FALSE, played = FALSE;
    gboolean auto_advance = xfmedia_settings_get_bool("/xfmedia/playlist/auto_advance");
    
    do {
        played = xfmedia_mainwin_play_file_at_index(mwin, index);
        if(played || !auto_advance)
            done = TRUE;
        
        if(!done) {
            index = mainwin_get_next_index(mwin, index);
            if(index == -1)
                done = TRUE;
        }
    } while(!done);
    
    return played;
}

static inline void
mainwin_set_tooltip_playing(XfMediaMainwin *mwin)
{
    if(mwin->cur_playing && xfmedia_settings_get_bool("/xfmedia/general/show_tray_icon")) {
        gint index = xfmedia_playlist_entry_ref_get_index(mwin->cur_playing);
        gchar *title = NULL;
        
        if(index != -1
                && xfmedia_playlist_get(mwin->plist, index, &title, NULL, NULL)
                && title)
        {
            gchar *str = g_strdup_printf("%s: %s", _("Playing"), title);
            xfmedia_tray_icon_set_tooltip(mwin->tray_icon, str);
            g_free(str);
        }
    }
}

void
xfmedia_mainwin_toggle_playpause(XfMediaMainwin *mwin)
{
    gint speed, status;
    
    speed = gtk_xine_get_param(mwin->gtx, XINE_PARAM_SPEED);
    status = gtk_xine_get_status(mwin->gtx);
    
    if(status == XINE_STATUS_PLAY && speed != XINE_SPEED_PAUSE) {
        xfmedia_mainwin_set_play_buttons(mwin);
        gtk_xine_set_param(mwin->gtx, XINE_PARAM_SPEED, XINE_SPEED_PAUSE);
        gtk_xine_set_param(mwin->gtx, XINE_PARAM_AUDIO_CLOSE_DEVICE, 1);
        xfmedia_plugins_forward_signal_VOID("speed-changed");
        if(xfmedia_settings_get_bool("/xfmedia/general/show_tray_icon"))
            xfmedia_tray_icon_set_tooltip(mwin->tray_icon, _("Paused"));
    } else if(status == XINE_STATUS_STOP || status == XINE_STATUS_IDLE) {
        if(!gtk_xine_play(mwin->gtx, 0, 0)) {
            gint index = -1;
            
            if(mwin->cur_playing)
            	index = xfmedia_playlist_entry_ref_get_index(mwin->cur_playing);
            if(index < 0)
                index = xfmedia_playlist_get_selected(mwin->plist);
            if(index < 0)
                index = 0;
            
            xfmedia_mainwin_play_file_at_index(mwin, index);
        } else {
            if((gtk_xine_get_stream_info(mwin->gtx, XINE_STREAM_INFO_HAS_VIDEO)
                    || xfmedia_settings_get_bool("/xfmedia/general/show_audio_vis"))
                    && !gtk_xine_is_fullscreen(mwin->gtx)
                    && !GTK_WIDGET_VISIBLE(mwin->video_window))
            {
                gint width, height;
                GdkGeometry geom;
                
                width = gtk_xine_get_stream_info(mwin->gtx,
                        XINE_STREAM_INFO_VIDEO_WIDTH);
                height = gtk_xine_get_stream_info(mwin->gtx,
                        XINE_STREAM_INFO_VIDEO_HEIGHT);
                if(width < 1 || height < 1) {
                    width = 480;
                    height = 360;
                }
                geom.base_width = width;
                geom.base_height = height;
                geom.min_width = geom.min_height = 1;
                geom.min_aspect = geom.max_aspect = (gdouble)width / height;
                DBG("window size: %dx%d, aspect: %f", width, height, geom.max_aspect);
                gtk_window_set_geometry_hints(GTK_WINDOW(mwin->video_window),
                        GTK_WIDGET(mwin->gtx), &geom,
                        GDK_HINT_MIN_SIZE|GDK_HINT_ASPECT|GDK_HINT_BASE_SIZE);
                gtk_widget_show(mwin->video_window);
                gtk_window_resize(GTK_WINDOW(mwin->video_window),
                        width, height);
                
                mainwin_set_tooltip_playing(mwin);
            }
        }
    } else if(status == XINE_STATUS_PLAY && speed == XINE_SPEED_PAUSE) {
        xfmedia_mainwin_set_pause_buttons(mwin);
        gtk_xine_set_param(mwin->gtx, XINE_PARAM_AUDIO_CLOSE_DEVICE, 0);
        gtk_xine_set_param(mwin->gtx, XINE_PARAM_SPEED, XINE_SPEED_NORMAL);
        xfmedia_plugins_forward_signal_VOID("speed-changed");
        mainwin_set_tooltip_playing(mwin);
    }
}

void
xfmedia_mainwin_stop(XfMediaMainwin *mwin)
{
    gint speed, status;
    
    speed = gtk_xine_get_param(mwin->gtx, XINE_PARAM_SPEED);
    status = gtk_xine_get_status(mwin->gtx);
    
    if(status == XINE_STATUS_PLAY && speed != XINE_SPEED_PAUSE)
        xfmedia_mainwin_set_play_buttons(mwin);
    
    if(status == XINE_STATUS_PLAY) {
        gtk_xine_stop(mwin->gtx);
        
        xfmedia_mainwin_disconnect_slider_callback(mwin);
        gtk_range_set_value(GTK_RANGE(mwin->position_slider), 0.0);
        gtk_range_set_value(GTK_RANGE(mwin->fs_position_slider), 0.0);
        xfmedia_mainwin_connect_slider_callback(mwin);
        
        gtk_label_set_markup(GTK_LABEL(mwin->time_label),
                "<span font_family=\"monospace\" size=\"large\" weight=\"bold\"> 00:00</span>");
        gtk_label_set_markup(GTK_LABEL(mwin->fs_time_label),
                "<span font_family=\"monospace\"> 00:00</span>");
        
        if(GTK_WIDGET_VISIBLE(mwin->video_window))
            gtk_widget_hide(mwin->video_window);
        if(gtk_xine_is_fullscreen(mwin->gtx)) {
            if(GTK_WIDGET_VISIBLE(mwin->fs_controls))
                gtk_widget_hide(mwin->fs_controls);
            gtk_xine_set_fullscreen(mwin->gtx, FALSE);
        }
        gtk_xine_set_param(mwin->gtx, XINE_PARAM_AUDIO_CLOSE_DEVICE, 1);
    }
    
    if(xfmedia_settings_get_bool("/xfmedia/general/show_tray_icon"))
        xfmedia_tray_icon_set_tooltip(mwin->tray_icon, _("Stopped"));
}

void
xfmedia_mainwin_next(XfMediaMainwin *mwin)
{
    gint next_index = -1;
    
    if(mwin->cur_playing &&
            xfmedia_settings_get_bool("/xfmedia/playlist/shuffle"))
    {
        DBG("putting item on shuffle history");
        g_queue_push_head(mwin->shuffle_history,
                xfmedia_playlist_entry_ref_copy(mwin->cur_playing));
    }
    
    xfmedia_mainwin_play_file_at_index_persistent(mwin, next_index);
}

void
xfmedia_mainwin_prev(XfMediaMainwin *mwin)
{
    gint tot_items, index = -1;
    
    if(xfmedia_settings_get_bool("/xfmedia/playlist/shuffle")) {
        XfMediaPlaylistEntryRef *ref = NULL;
        gboolean playing = FALSE;
        
        if(mwin->cur_playing) {
        	index = xfmedia_playlist_entry_ref_get_index(mwin->cur_playing);
			if(index >= 0)
                xfmedia_playlist_shuffle_unset(mwin->plist, index);
        }
        
        do {
            if(index != -1)
                xfmedia_playlist_shuffle_unset(mwin->plist, index);
            ref = g_queue_pop_head(mwin->shuffle_history);
            DBG("got ref from shuffle history %p", ref);
            if(ref) {
                index = xfmedia_playlist_entry_ref_get_index(ref);
                xfmedia_playlist_entry_ref_destroy(ref);
            } else {
                tot_items = xfmedia_playlist_get_n_entries(mwin->plist);
#if HAVE_SRANDOM
                index = random() % tot_items;
#else
                index = rand() % tot_items;
#endif
                xfmedia_playlist_shuffle_unset_all(mwin->plist);
            }
        } while(index != -1
                && !(playing=xfmedia_mainwin_play_file_at_index(mwin, index)));
        
        if(!playing)
            xfmedia_mainwin_play_file_at_index_persistent(mwin, -1);
    } else if(mwin->cur_playing
            && xfmedia_playlist_entry_ref_valid(mwin->cur_playing))
    {
        index = xfmedia_playlist_entry_ref_get_index(mwin->cur_playing);
        do {
            if(index == 0 && xfmedia_settings_get_bool("/xfmedia/playlist/repeat")) {
                tot_items = xfmedia_playlist_get_n_entries(mwin->plist);
                index = tot_items - 1;
            } else if(index == -1)
                index = 0;
            else
                index--;
        } while(!xfmedia_mainwin_play_file_at_index(mwin, index));
    }
}
