/*
 *  xfmedia - simple gtk2 media player based on xine
 *
 *  Copyright (c) 2004-2005 Brian Tarricone, <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#include <gdk/gdkkeysyms.h>

#include <gtk/gtk.h>
#include <gdk-pixbuf/gdk-pixdata.h>

#include <libxfce4util/libxfce4util.h>
#include <libxfcegui4/libxfcegui4.h>

#define EXO_API_SUBJECT_TO_CHANGE
#include <exo/exo.h>

#include "xfmedia-internal.h"

#include "gtkxine.h"
#include "mainwin.h"
#include <xfmedia/xfmedia-playlist.h>
#include <xfmedia/xfmedia-settings.h>
#include "trayicon.h"
#include "keybindings.h"
#include "xfmedia-keybindings.h"
#include "main.h"
#include "mediamarks.h"

#define MAX_SHUFFLE_QUEUE_DEPTH 100
#define AUTOHIDE_TIMEOUT        500

static gboolean
xfmedia_mainwin_playlist_load_info_idled(gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    guint idx_start, idx_end, i;
    gchar *title = NULL, *filename = NULL;
    gint length = -1;
    
    gdk_threads_enter();
    
    if(!xfmedia_playlist_get_visible_range(mwin->plist, &idx_start, &idx_end)) {
        gdk_threads_leave();
        return FALSE;
    }
    
    for(i = idx_start; i <= idx_end; i++) {
        if(xfmedia_playlist_get_metadata_loaded(mwin->plist, i))
            continue;
        
        if(!xfmedia_playlist_get(mwin->plist, i, NULL, NULL, &filename) || !filename)
            continue;
        
        if(xfmedia_mainwin_get_file_info(mwin, filename, &title, &length)) {
            if(!title)
                title = xfmedia_filename_to_name(filename);
            xfmedia_playlist_modify_entry(mwin->plist, i, title, length, NULL);
            xfmedia_playlist_set_metadata_loaded(mwin->plist, i, TRUE);
            g_free(title);
        }
        g_free(filename);
        
        while(gtk_events_pending())
            gtk_main_iteration();
    }
    
    gdk_threads_leave();
    
    return FALSE;
}

void
xfmedia_mainwin_playlist_scrolled_cb(XfMediaPlaylist *plist, gpointer user_data)
{
    g_idle_add(xfmedia_mainwin_playlist_load_info_idled, user_data);
}

void
xfmedia_mainwin_playlist_entry_activated_cb(XfMediaPlaylist *plist, gint index,
        gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    
    if(!xfmedia_mainwin_play_file_at_index(mwin, index)) {
        xfce_message_dialog(GTK_WINDOW(mwin->window), "Xfmedia",
                GTK_STOCK_DIALOG_INFO, _("Unable to play file."),
                _("Xfmedia was unable to play the selected file.  Be sure that the file exists and that you have permission to access it."),
                GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE, NULL);
    }
}

void
xfmedia_mainwin_playlist_shuffle_toggled_cb(XfMediaPlaylist *plist,
        gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    DBG("got shuffle-toggled");
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mwin->shuffle_mi),
            !gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(mwin->shuffle_mi)));
}

void
xfmedia_mainwin_playlist_repeat_toggled_cb(XfMediaPlaylist *plist,
        gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    DBG("got repeat-toggled");
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mwin->repeat_mi),
            !gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(mwin->repeat_mi)));
}

void
xfmedia_mainwin_playlist_cleared_cb(XfMediaPlaylist *plist, gpointer user_data)
{
    xfmedia_settings_set_int("/xfmedia/playlist/playlist_position", -1);
}

void
xfmedia_mainwin_playlist_file_info_cb(GtkWidget *w, gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    gint index;
#define TMPBUF 64
    gchar *filename = NULL, tmpbuf[TMPBUF], *tmpbuf1 = NULL;
    const gchar *value;
    GtkWidget *win, *lbl, *entry, *topvbox, *hbox, *frame, *vbox, *btn;
    GtkSizeGroup *sg;
    struct stat st;
    gint pos_stream, pos_time, length_time = 0, old_xine_status, old_xine_speed;
    guint32 fourcc;
    
    index = xfmedia_playlist_get_selected(mwin->plist);
    if(index < 0)
        return;
    
    if(!xfmedia_playlist_get(mwin->plist, index, NULL, NULL, &filename))
        return;
    
    old_xine_status = gtk_xine_get_status(mwin->gtx);
    old_xine_speed = gtk_xine_get_param(mwin->gtx, XINE_PARAM_SPEED);
    
    if(!xine_open(mwin->extra_stream, filename)) {
        xfce_message_dialog(GTK_WINDOW(mwin->window), "Xfmedia",
                GTK_STOCK_DIALOG_INFO, _("Unable to open file."),
                _("Xfmedia was unable to open the selected file.  Be sure that the file exists and that you have permission to access it."),
                GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE, NULL);
        g_free(filename);
        return;
    }
    
    win = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(win), _("File Info"));
    gtk_window_set_transient_for(GTK_WINDOW(win), GTK_WINDOW(mwin->window));
    gtk_container_set_border_width(GTK_CONTAINER(win), BORDER);
    
    topvbox = gtk_vbox_new(FALSE, BORDER/2);
    gtk_widget_show(topvbox);
    gtk_container_add(GTK_CONTAINER(win), topvbox);
    
    sg = gtk_size_group_new(GTK_SIZE_GROUP_HORIZONTAL);
    
    frame = xfce_framebox_new(_("Filename"), TRUE);
    gtk_widget_show(frame);
    gtk_box_pack_start(GTK_BOX(topvbox), frame, FALSE, FALSE, 0);
    
    lbl = exo_ellipsized_label_new(filename);
    exo_ellipsized_label_set_ellipsize(EXO_ELLIPSIZED_LABEL(lbl),
            EXO_PANGO_ELLIPSIZE_START);
    gtk_label_set_selectable(GTK_LABEL(lbl), TRUE);
    gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
    gtk_label_set_text(GTK_LABEL(lbl), filename);
    gtk_widget_show(lbl);
    xfce_framebox_add(XFCE_FRAMEBOX(frame), lbl);
    
    frame = xfce_framebox_new(_("Media Info"), TRUE);
    gtk_widget_show(frame);
    gtk_box_pack_start(GTK_BOX(topvbox), frame, FALSE, FALSE, 0);
    
    vbox = gtk_vbox_new(FALSE, BORDER);
    gtk_widget_show(vbox);
    xfce_framebox_add(XFCE_FRAMEBOX(frame), vbox);
    
    hbox = gtk_hbox_new(FALSE, BORDER);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    
    lbl = gtk_label_new(_("Title:"));
    gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
    gtk_widget_show(lbl);
    gtk_size_group_add_widget(sg, lbl);
    gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
    
    entry = gtk_entry_new();
    value = xine_get_meta_info(mwin->extra_stream, XINE_META_INFO_TITLE);
    if(value)
        gtk_entry_set_text(GTK_ENTRY(entry), value);
    gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
    gtk_widget_show(entry);
    gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
    
    hbox = gtk_hbox_new(FALSE, BORDER);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    
    lbl = gtk_label_new(_("Artist:"));
    gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
    gtk_widget_show(lbl);
    gtk_size_group_add_widget(sg, lbl);
    gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
    
    entry = gtk_entry_new();
    value = xine_get_meta_info(mwin->extra_stream, XINE_META_INFO_ARTIST);
    if(value)
        gtk_entry_set_text(GTK_ENTRY(entry), value);
    gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
    gtk_widget_show(entry);
    gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
    
    hbox = gtk_hbox_new(FALSE, BORDER);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    
    lbl = gtk_label_new(_("Album:"));
    gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
    gtk_widget_show(lbl);
    gtk_size_group_add_widget(sg, lbl);
    gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
    
    entry = gtk_entry_new();
    value = xine_get_meta_info(mwin->extra_stream, XINE_META_INFO_ALBUM);
    if(value)
        gtk_entry_set_text(GTK_ENTRY(entry), value);
    gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
    gtk_widget_show(entry);
    gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
    
    hbox = gtk_hbox_new(FALSE, BORDER);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    
    lbl = gtk_label_new(_("Genre:"));
    gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
    gtk_widget_show(lbl);
    gtk_size_group_add_widget(sg, lbl);
    gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
    
    entry = gtk_entry_new();
    value = xine_get_meta_info(mwin->extra_stream, XINE_META_INFO_GENRE);
    if(value)
        gtk_entry_set_text(GTK_ENTRY(entry), value);
    gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
    gtk_widget_show(entry);
    gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
    
    lbl = gtk_label_new(_("Track:"));
    gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
    gtk_widget_show(lbl);
    gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
    
    entry = gtk_entry_new();
    value = xine_get_meta_info(mwin->extra_stream, XINE_META_INFO_TRACK_NUMBER);
    if(value && *value) {
        gtk_entry_set_text(GTK_ENTRY(entry), value);
        gtk_entry_set_width_chars(GTK_ENTRY(entry), strlen(value));
    } else
        gtk_entry_set_width_chars(GTK_ENTRY(entry), 2);
    gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
    gtk_widget_show(entry);
    gtk_box_pack_start(GTK_BOX(hbox), entry, FALSE, FALSE, 0);
    
    lbl = gtk_label_new(_("Year:"));
    gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
    gtk_widget_show(lbl);
    gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
    
    entry = gtk_entry_new();
    value = xine_get_meta_info(mwin->extra_stream, XINE_META_INFO_YEAR);
    if(value)
        gtk_entry_set_text(GTK_ENTRY(entry), value);
    gtk_entry_set_width_chars(GTK_ENTRY(entry), 4);
    gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
    gtk_widget_show(entry);
    gtk_box_pack_start(GTK_BOX(hbox), entry, FALSE, FALSE, 0);
    
    hbox = gtk_hbox_new(FALSE, BORDER);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    
    lbl = gtk_label_new(_("Comment:"));
    gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
    gtk_widget_show(lbl);
    gtk_size_group_add_widget(sg, lbl);
    gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
    
    entry = gtk_entry_new();
    value = xine_get_meta_info(mwin->extra_stream, XINE_META_INFO_COMMENT);
    if(value)
        gtk_entry_set_text(GTK_ENTRY(entry), value);
    gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
    gtk_widget_show(entry);
    gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
    
    hbox = gtk_hbox_new(FALSE, BORDER);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    
    lbl = gtk_label_new(_("Length:"));
    gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
    gtk_widget_show(lbl);
    gtk_size_group_add_widget(sg, lbl);
    gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
    
    entry = gtk_entry_new();
    if(xine_get_pos_length(mwin->extra_stream, &pos_stream, &pos_time, &length_time)) {
        length_time /= 1000;
        g_snprintf(tmpbuf, TMPBUF, "%02d:%02d", length_time/60, length_time%60);
        gtk_entry_set_text(GTK_ENTRY(entry), tmpbuf);
        gtk_entry_set_width_chars(GTK_ENTRY(entry), strlen(tmpbuf));
    }
    gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
    gtk_widget_show(entry);
    gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
    
    lbl = gtk_label_new(_("File size (b):"));
    gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
    gtk_widget_show(lbl);
    gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
    
    entry = gtk_entry_new();
    if(!stat(filename, &st)) {
        tmpbuf1 = xfmedia_num_prettify((gint)st.st_size);
        gtk_entry_set_text(GTK_ENTRY(entry), tmpbuf1);
        gtk_entry_set_width_chars(GTK_ENTRY(entry), strlen(tmpbuf1));
        g_free(tmpbuf1);
    }
    gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
    gtk_widget_show(entry);
    gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
    
    if(xine_get_stream_info(mwin->extra_stream, XINE_STREAM_INFO_HAS_AUDIO)) {
        frame = xfce_framebox_new(_("Audio Info"), TRUE);
        gtk_widget_show(frame);
        gtk_box_pack_start(GTK_BOX(topvbox), frame, FALSE, FALSE, 0);
        
        vbox = gtk_vbox_new(FALSE, BORDER);
        gtk_widget_show(vbox);
        xfce_framebox_add(XFCE_FRAMEBOX(frame), vbox);
        
        hbox = gtk_hbox_new(FALSE, BORDER);
        gtk_widget_show(hbox);
        gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
        
        lbl = gtk_label_new(_("Codec:"));
        gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
        gtk_widget_show(lbl);
        gtk_size_group_add_widget(sg, lbl);
        gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
        
        entry = gtk_entry_new();
        value = xine_get_meta_info(mwin->extra_stream,
                XINE_META_INFO_AUDIOCODEC);
        if(value)
            gtk_entry_set_text(GTK_ENTRY(entry), value);
        gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
        gtk_widget_show(entry);
        gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
        
        lbl = gtk_label_new(_("Samples/sec:"));
        gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
        gtk_widget_show(lbl);
        gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
        
        entry = gtk_entry_new();
        tmpbuf1 = xfmedia_num_prettify(xine_get_stream_info(mwin->extra_stream,
                XINE_STREAM_INFO_AUDIO_SAMPLERATE));
        gtk_entry_set_text(GTK_ENTRY(entry), tmpbuf1);
        gtk_entry_set_width_chars(GTK_ENTRY(entry), strlen(tmpbuf1));
        g_free(tmpbuf1);
        gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
        gtk_widget_show(entry);
        gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
        
        hbox = gtk_hbox_new(FALSE, BORDER);
        gtk_widget_show(hbox);
        gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
        
        lbl = gtk_label_new(_("Bits/sample:"));
        gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
        gtk_widget_show(lbl);
        gtk_size_group_add_widget(sg, lbl);
        gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
        
        entry = gtk_entry_new();
        g_snprintf(tmpbuf, TMPBUF, "%d",
        		xine_get_stream_info(mwin->extra_stream,
                	XINE_STREAM_INFO_AUDIO_BITS));
        gtk_entry_set_text(GTK_ENTRY(entry), tmpbuf);
        gtk_entry_set_width_chars(GTK_ENTRY(entry), strlen(tmpbuf));
        gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
        gtk_widget_show(entry);
        gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
        
        lbl = gtk_label_new(_("Channels:"));
        gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
        gtk_widget_show(lbl);
        gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
        
        entry = gtk_entry_new();
        g_snprintf(tmpbuf, TMPBUF, "%d",
        		xine_get_stream_info(mwin->extra_stream,
                	XINE_STREAM_INFO_AUDIO_CHANNELS));
        gtk_entry_set_text(GTK_ENTRY(entry), tmpbuf);
        gtk_entry_set_width_chars(GTK_ENTRY(entry), strlen(tmpbuf));
        gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
        gtk_widget_show(entry);
        gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
        
        lbl = gtk_label_new(_("Bitrate (kbit):"));
        gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
        gtk_widget_show(lbl);
        gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
        
        entry = gtk_entry_new();
        tmpbuf1 = xfmedia_num_prettify(xine_get_stream_info(mwin->extra_stream,
                XINE_STREAM_INFO_AUDIO_BITRATE)/1000);
        gtk_entry_set_text(GTK_ENTRY(entry), tmpbuf1);
        gtk_entry_set_width_chars(GTK_ENTRY(entry), strlen(tmpbuf1));
        g_free(tmpbuf1);
        gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
        gtk_widget_show(entry);
        gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
    }
    
    if(xine_get_stream_info(mwin->extra_stream, XINE_STREAM_INFO_HAS_VIDEO)) {
        frame = xfce_framebox_new(_("Video Info"), TRUE);
        gtk_widget_show(frame);
        gtk_box_pack_start(GTK_BOX(topvbox), frame, FALSE, FALSE, 0);
        
        vbox = gtk_vbox_new(FALSE, BORDER);
        gtk_widget_show(vbox);
        xfce_framebox_add(XFCE_FRAMEBOX(frame), vbox);
        
        hbox = gtk_hbox_new(FALSE, BORDER);
        gtk_widget_show(hbox);
        gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
        
        lbl = gtk_label_new(_("Codec:"));
        gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
        gtk_widget_show(lbl);
        gtk_size_group_add_widget(sg, lbl);
        gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
        
        entry = gtk_entry_new();
        value = xine_get_meta_info(mwin->extra_stream,
                XINE_META_INFO_VIDEOCODEC);
        if(value)
            gtk_entry_set_text(GTK_ENTRY(entry), value);
        gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
        gtk_widget_show(entry);
        gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
        
        lbl = gtk_label_new(_("FourCC:"));
        gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
        gtk_widget_show(lbl);
        gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
        
        entry = gtk_entry_new();
        fourcc = xine_get_stream_info(mwin->extra_stream,
                XINE_STREAM_INFO_VIDEO_FOURCC);
        tmpbuf[0] = ((gchar *)&fourcc)[0];
        tmpbuf[1] = ((gchar *)&fourcc)[1];
        tmpbuf[2] = ((gchar *)&fourcc)[2];
        tmpbuf[3] = ((gchar *)&fourcc)[3];
        tmpbuf[4] = 0;
        if(g_utf8_validate(tmpbuf, -1, NULL))
            gtk_entry_set_text(GTK_ENTRY(entry), tmpbuf);
        gtk_entry_set_width_chars(GTK_ENTRY(entry), 4);
        gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
        gtk_widget_show(entry);
        gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
        
        hbox = gtk_hbox_new(FALSE, BORDER);
        gtk_widget_show(hbox);
        gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
        
        lbl = gtk_label_new(_("Frame size:"));
        gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
        gtk_widget_show(lbl);
        gtk_size_group_add_widget(sg, lbl);
        gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
        
        entry = gtk_entry_new();
        g_snprintf(tmpbuf, TMPBUF, "%dx%d",
                xine_get_stream_info(mwin->extra_stream,
                    XINE_STREAM_INFO_VIDEO_WIDTH),
                xine_get_stream_info(mwin->extra_stream,
                    XINE_STREAM_INFO_VIDEO_HEIGHT));
        gtk_entry_set_text(GTK_ENTRY(entry), tmpbuf);
        gtk_entry_set_width_chars(GTK_ENTRY(entry), strlen(tmpbuf));
        gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
        gtk_widget_show(entry);
        gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
        
        lbl = gtk_label_new(_("Channels:"));
        gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
        gtk_widget_show(lbl);
        gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
        
        entry = gtk_entry_new();
        g_snprintf(tmpbuf, TMPBUF, "%d", xine_get_stream_info(mwin->extra_stream,
                XINE_STREAM_INFO_VIDEO_CHANNELS));
        gtk_entry_set_text(GTK_ENTRY(entry), tmpbuf);
        gtk_entry_set_width_chars(GTK_ENTRY(entry), strlen(tmpbuf));
        gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
        gtk_widget_show(entry);
        gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
        
        lbl = gtk_label_new(_("Bitrate (kbit):"));
        gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
        gtk_widget_show(lbl);
        gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
        
        entry = gtk_entry_new();
        tmpbuf1 = xfmedia_num_prettify(xine_get_stream_info(mwin->extra_stream,
                XINE_STREAM_INFO_VIDEO_BITRATE)/1000);
        gtk_entry_set_text(GTK_ENTRY(entry), tmpbuf1);
        gtk_entry_set_width_chars(GTK_ENTRY(entry), strlen(tmpbuf1));
        g_free(tmpbuf1);
        gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
        gtk_widget_show(entry);
        gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
    }
    
    xine_close(mwin->extra_stream);
    g_free(filename);
    
    hbox = gtk_hbox_new(FALSE, BORDER);
    gtk_widget_show(hbox);
    gtk_box_pack_end(GTK_BOX(topvbox), hbox, FALSE, FALSE, 0);
    
    btn = gtk_button_new_from_stock(GTK_STOCK_CLOSE);
    gtk_widget_show(btn);
    gtk_box_pack_end(GTK_BOX(hbox), btn, FALSE, FALSE, 0);
    g_signal_connect_swapped(G_OBJECT(btn), "clicked",
            G_CALLBACK(gtk_widget_destroy), win);
    
    if(old_xine_status == XINE_STATUS_PLAY
            && old_xine_speed != gtk_xine_get_param(mwin->gtx, XINE_PARAM_SPEED))
    {
        DBG("xine changed speed while getting stream info, fixing");
        gtk_xine_set_param(mwin->gtx, XINE_PARAM_SPEED, old_xine_speed);
    }
    
    gtk_widget_show(win);
#undef TMPBUF
}


void
xfmedia_mainwin_playlist_add_mm_cb(GtkWidget *w, gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    gint index;
    gchar *title = NULL, *filename = NULL;
    
    index = xfmedia_playlist_get_selected(mwin->plist);
    if(index < 0)
        return;
    
    if(!xfmedia_playlist_get(mwin->plist, index, &title, NULL, &filename))
        return;
    
    xfmedia_mediamarks_add(mwin, GTK_WINDOW(mwin->window), title, filename);
    
    g_free(title);
    g_free(filename);
}

void
xfmedia_mainwin_playlist_add_autoplay_cb(GtkMenuItem *mi, gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    GtkWidget *lbl = gtk_bin_get_child(GTK_BIN(mi));
    const gchar *plugin_id = gtk_label_get_text(GTK_LABEL(lbl));
    GList *mrls, *l;
    guint nadded = 0;
    
    if(plugin_id) {
        mrls = gtk_xine_get_autoplay_mrls(mwin->gtx, plugin_id);
        for(l = mrls; l; l = l->next) {
            gchar *title_utf8 = xfmedia_filename_to_name(l->data);
            xfmedia_playlist_append_entry(mwin->plist, title_utf8, -1,
                    l->data, FALSE);
            g_free(title_utf8);
            nadded++;
        }
        if(mrls) {
            g_list_free(mrls);
            //xfmedia_playlist_shuffle_add_entries(mwin->plist, -1, nadded);
        }
    }
}

gboolean
xfmedia_position_cb(gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    gint pos_stream, pos_time, length_time;
    
    if(gtk_xine_get_status(mwin->gtx) != XINE_STATUS_PLAY
            || gtk_xine_get_param(mwin->gtx, XINE_PARAM_SPEED) == XINE_SPEED_PAUSE
            || mwin->slider_sticky)
    {
        return TRUE;
    }
    
    if(!gtk_xine_get_pos_length(mwin->gtx, &pos_stream, &pos_time, &length_time))
        return TRUE;
    pos_time /= 1000;
    length_time /= 1000;
    
    /* FIXME: i *think* it's ok not to lock gdk when calling into xine like
     * that, but i should check. */
    gdk_threads_enter();
    
    if(xfmedia_settings_get_bool("/xfmedia/general/show_remaining") && length_time > 0)
        xfmedia_mainwin_set_time_label(mwin, length_time-pos_time, TRUE, NULL);
    else
        xfmedia_mainwin_set_time_label(mwin, pos_time, FALSE, NULL);
    
    /* sometimes xine fails to get the song length on the first try, so we'll
     * check it here if we have a zero length */
    if(mwin->cur_playing && length_time != 0
            && xfmedia_settings_get_int("/xfmedia/playlist/metadata_load_impetus") <= 1)
    {
        gint cur_length = -1, index;
        index = xfmedia_playlist_entry_ref_get_index(mwin->cur_playing);
        if(index != -1 && xfmedia_playlist_get(mwin->plist, index, NULL, &cur_length, NULL)
                && (cur_length != length_time || mwin->cur_playing_length <= 0))
        {
            xfmedia_playlist_modify_entry(mwin->plist, index, NULL, length_time, NULL);
            gtk_range_set_range(GTK_RANGE(mwin->position_slider), 0, length_time);
            gtk_range_set_range(GTK_RANGE(mwin->fs_position_slider), 0, length_time);
            mwin->cur_playing_length = length_time;
            
        }
    } else if(index < 0 && length_time != 0
            && mwin->cur_playing_length != length_time)
    {
        gtk_range_set_range(GTK_RANGE(mwin->position_slider), 0, length_time);
        gtk_range_set_range(GTK_RANGE(mwin->fs_position_slider), 0, length_time);
        mwin->cur_playing_length = length_time;
    }
    
    if(length_time > 0) {
        xfmedia_mainwin_disconnect_slider_callback(mwin);
        gtk_range_set_value(GTK_RANGE(mwin->position_slider), (gdouble)pos_time);
        gtk_range_set_value(GTK_RANGE(mwin->fs_position_slider), (gdouble)pos_time);
        xfmedia_mainwin_connect_slider_callback(mwin);
    }
    
    gdk_threads_leave();
    
    return TRUE;
}

void
xfmedia_position_slider_cb(GtkWidget *w, gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    gint status, speed, new_pos;
    
    status = gtk_xine_get_status(mwin->gtx);
    speed = gtk_xine_get_param(mwin->gtx, XINE_PARAM_SPEED);
    
    new_pos = (gint)gtk_range_get_value(GTK_RANGE(w)) * 1000;
    DBG("seeking to %d\n", new_pos);
    if(gtk_xine_play(mwin->gtx, 0, new_pos)) {
        if(status != XINE_STATUS_PLAY || speed == XINE_SPEED_PAUSE)
            xfmedia_mainwin_set_pause_buttons(mwin);
    }
}

static gboolean
xfmedia_pos_slider_mmotion_cb(GtkWidget *w, GdkEventMotion *evt,
        gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    gint cur_value;
    
    if(w == mwin->position_slider)
        cur_value = gtk_range_get_value(GTK_RANGE(mwin->position_slider));
    else if(w == mwin->fs_position_slider)
        cur_value = gtk_range_get_value(GTK_RANGE(mwin->fs_position_slider));
    else {
        DBG("er, this shouldn't happen.");
        return FALSE;
    }
    
    if(xfmedia_settings_get_bool("/xfmedia/general/show_remaining")) {
        gint pos_stream, pos_time, length_time = 0;
        if(!gtk_xine_get_pos_length(mwin->gtx, &pos_stream, &pos_time, &length_time) || length_time == 0)
            return FALSE;
        length_time /= 1000;
        xfmedia_mainwin_set_time_label(mwin, length_time-cur_value, TRUE, "style='italic'");
    } else
        xfmedia_mainwin_set_time_label(mwin, cur_value, FALSE, "style='italic'");
    
    return FALSE;
}

gboolean
xfmedia_pos_slider_btn_press_cb(GtkWidget *w, GdkEventButton *evt,
        gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    
    if(evt->button == 1 || evt->button == 2) {
        mwin->slider_sticky = TRUE;
        
        g_signal_connect(G_OBJECT(mwin->position_slider), "motion-notify-event",
                G_CALLBACK(xfmedia_pos_slider_mmotion_cb), mwin);
        g_signal_connect(G_OBJECT(mwin->fs_position_slider), "motion-notify-event",
                G_CALLBACK(xfmedia_pos_slider_mmotion_cb), mwin);
    }
    
    return FALSE;
}

gboolean
xfmedia_pos_slider_btn_rls_cb(GtkWidget *w, GdkEventButton *evt,
        gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    
    if(evt->button == 1 || evt->button == 2) {
        mwin->slider_sticky = FALSE;
        
        g_signal_handlers_disconnect_by_func(G_OBJECT(mwin->position_slider),
                G_CALLBACK(xfmedia_pos_slider_mmotion_cb), mwin);
        g_signal_handlers_disconnect_by_func(G_OBJECT(mwin->fs_position_slider),
                G_CALLBACK(xfmedia_pos_slider_mmotion_cb), mwin);
    }
    
    return FALSE;
}

void
xfmedia_playpause_cb(GtkWidget *w, gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    
    xfmedia_mainwin_toggle_playpause(mwin);
}

void
xfmedia_stop_cb(GtkWidget *w, gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    
    xfmedia_mainwin_stop(mwin);
}

void
xfmedia_prev_cb(GtkWidget *w, gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    
    xfmedia_mainwin_prev(mwin);
}

void
xfmedia_next_cb(GtkWidget *w, gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    
    xfmedia_mainwin_next(mwin);
}

gboolean
xfmedia_pos_lbl_click_cb(GtkWidget *w, GdkEventButton *evt, gpointer user_data)
{
    if(evt->button == 1) {
        xfmedia_settings_set_bool("/xfmedia/general/show_remaining",
                !xfmedia_settings_get_bool("/xfmedia/general/show_remaining"));
        /* FIXME: hackish */
        gdk_threads_leave();
        xfmedia_position_cb(user_data);
        gdk_threads_enter();
    }
    
    return xfmedia_mainwin_menu_cb(w, evt, user_data);
}

void
xfmedia_shuffle_menu_toggle_cb(GtkCheckMenuItem *mi, gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    gboolean set;
    
    set = gtk_check_menu_item_get_active(mi);
    xfmedia_playlist_set_shuffle_state(mwin->plist, set);
    xfmedia_settings_set_bool("/xfmedia/playlist/shuffle", set);
    
    if(!set && !g_queue_is_empty(mwin->shuffle_history)) {
        gint i, tot = g_queue_get_length(mwin->shuffle_history);
        for(i = tot-1; i >= 0; i--)
            xfmedia_playlist_entry_ref_destroy((XfMediaPlaylistEntryRef *)g_queue_pop_nth(mwin->shuffle_history, i));
    }
    
    if(set)
        xfmedia_playlist_shuffle_unset_all(mwin->plist);
}

void
xfmedia_repeat_menu_toggle_cb(GtkCheckMenuItem *mi, gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    gboolean set;
    
    set = gtk_check_menu_item_get_active(mi);
    xfmedia_playlist_set_repeat_state(mwin->plist, set);
    xfmedia_settings_set_bool("/xfmedia/playlist/repeat", set);
}

void
xfmedia_repeat_one_menu_toggle_cb(GtkCheckMenuItem *mi, gpointer user_data)
{
    gboolean set;
    
    set = gtk_check_menu_item_get_active(mi);
    xfmedia_settings_set_bool("/xfmedia/playlist/repeat_one", set);
}

void
xfmedia_show_vis_menu_toggle_cb(GtkCheckMenuItem *mi, gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    gboolean set, has_video;

    set = gtk_check_menu_item_get_active(mi);
    xfmedia_settings_set_bool("/xfmedia/general/show_audio_vis", set);
    
    has_video = gtk_xine_get_stream_info(mwin->gtx, XINE_STREAM_INFO_HAS_VIDEO);
    if(!set) {
        if(!has_video)
            gtk_widget_hide(mwin->video_window);
        gtk_xine_set_vis(mwin->gtx, NULL);
    } else {
        if(!has_video && gtk_xine_get_status(mwin->gtx) == XINE_STATUS_PLAY) {
            gint width, height;
            GdkGeometry geom;
            
            gtk_xine_set_vis(mwin->gtx,
                    xfmedia_settings_get_string("/xfmedia/general/selected_vis"));
            
            width = gtk_xine_get_stream_info(mwin->gtx,
                    XINE_STREAM_INFO_VIDEO_WIDTH);
            height = gtk_xine_get_stream_info(mwin->gtx,
                    XINE_STREAM_INFO_VIDEO_HEIGHT);
            if(width < 1 || height < 1) {
                width = 480;
                height = 360;
            }
            geom.base_width = width;
            geom.base_height = height;
            geom.min_width = geom.min_height = 1;
            geom.min_aspect = geom.max_aspect = (gdouble)width / height;
            DBG("window size: %dx%d, aspect: %f", width, height, geom.max_aspect);
            gtk_window_set_geometry_hints(GTK_WINDOW(mwin->video_window),
                    GTK_WIDGET(mwin->gtx), &geom,
                    GDK_HINT_MIN_SIZE|GDK_HINT_ASPECT|GDK_HINT_BASE_SIZE);
            gtk_widget_show(mwin->video_window);
            gtk_window_resize(GTK_WINDOW(mwin->video_window),
                    width, height);
        }
    }
}

gboolean
xfmedia_vwin_delete_cb(GtkWidget *w, GdkEvent *evt, gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    
    DBG("entering");
    
    if(gtk_xine_get_stream_info(mwin->gtx, XINE_STREAM_INFO_HAS_VIDEO)) {
        gtk_xine_stop(mwin->gtx);
        DBG("checking plist (%p)", mwin->plist);
        xfmedia_quit(mwin, XFMEDIA_QUIT_MODE_EXT);
    } else {
        gtk_xine_set_vis(mwin->gtx, NULL);
        gtk_widget_hide(w);
        gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mwin->svis_mi), FALSE);
        xfmedia_settings_set_bool("/xfmedia/general/show_audio_vis", FALSE);
    }
    
    /* never let the window get destroyed here */
    return TRUE;
}

gboolean
xfmedia_autohidewin_enter_notify(GtkWidget *w, GdkEventCrossing *evt,
        gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    gint x, y;
    
    x = xfmedia_settings_get_int("/xfmedia/general/pos_x");
    y = xfmedia_settings_get_int("/xfmedia/general/pos_y");
    
    gtk_window_move(GTK_WINDOW(mwin->window), x, y);
    if(xfmedia_settings_get_bool("/xfmedia/general/window_sticky"))
        gtk_window_stick(GTK_WINDOW(mwin->window));
    else
        gtk_window_unstick(GTK_WINDOW(mwin->window));
    if(xfmedia_settings_get_bool("/xfmedia/general/window_above"))
        gtk_window_set_keep_above(GTK_WINDOW(mwin->window), TRUE);
    gtk_widget_show(mwin->window);
    gtk_widget_hide(mwin->autohide_window);
    
    return FALSE;
}

gboolean
xfmedia_mainwin_enterleave_notify_cb(GtkWidget *w, GdkEventCrossing *evt, gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    
    if(evt->type == GDK_ENTER_NOTIFY) {
        if(mwin->autohide_timer) {
            g_source_remove(mwin->autohide_timer);
            mwin->autohide_timer = 0;
        }
    } else if(evt->type == GDK_LEAVE_NOTIFY) {
        if(evt->detail == GDK_NOTIFY_NONLINEAR_VIRTUAL
                && !mwin->temp_disable_autohide
                && xfmedia_settings_get_bool("/xfmedia/general/dock_autohide")
                && !mwin->autohide_timer)
        {
            mwin->autohide_timer = g_timeout_add(AUTOHIDE_TIMEOUT,
                    (GSourceFunc)xfmedia_mainwin_do_autohide, mwin);
        }
    }
    
    return FALSE;
}

GdkFilterReturn
xfmedia_mainwin_decs_filter(GdkXEvent *gxevt, GdkEvent *evt, gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    XCrossingEvent *xevt = (XCrossingEvent *)gxevt;

    if(xevt->type == EnterNotify) {
        if(mwin->autohide_timer) {
            g_source_remove(mwin->autohide_timer);
            mwin->autohide_timer = 0;
        }
    } else if(xevt->type == LeaveNotify) {
        if(xevt->detail == NotifyNonlinearVirtual
                && !(xevt->state & (ButtonPressMask|ButtonMotionMask
                                    |Button1MotionMask|Button2MotionMask
                                    |Button3MotionMask|Button4MotionMask
                                    |Button5MotionMask))
                && !mwin->temp_disable_autohide
                && xfmedia_settings_get_bool("/xfmedia/general/dock_autohide")
                && !mwin->autohide_timer)
        {
            mwin->autohide_timer = g_timeout_add(AUTOHIDE_TIMEOUT,
                    (GSourceFunc)xfmedia_mainwin_do_autohide, mwin);
        }
    } else if(xevt->type == MotionNotify) {
        if(mwin->autohide_timer) {
            g_source_remove(mwin->autohide_timer);
            mwin->autohide_timer = 0;
        }
    }
    
    return GDK_FILTER_CONTINUE;
}

gboolean
xfmedia_mainwin_configure_cb(GtkWidget *w, GdkEventConfigure *evt,
        gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    gint x = -1, y = -1;
    
    if(GTK_WIDGET_VISIBLE(w))
        gtk_window_get_position(GTK_WINDOW(w), &x, &y);
    else
        return FALSE;
    
    xfmedia_settings_set_int("/xfmedia/general/pos_x", x);
    xfmedia_settings_set_int("/xfmedia/general/pos_y", y);

    xfmedia_settings_set_int("/xfmedia/general/width", evt->width);
    if(gtk_expander_get_expanded(GTK_EXPANDER(mwin->playlist_pane)))
        xfmedia_settings_set_int("/xfmedia/general/height_large", evt->height);
    else
        xfmedia_settings_set_int("/xfmedia/general/height_small", evt->height);
    
    return FALSE;
}

gboolean
xfmedia_mainwin_winstate_cb(GtkWidget *w, GdkEventWindowState *evt,
        gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    
    mwin->is_iconified = (evt->new_window_state & GDK_WINDOW_STATE_ICONIFIED);
    
    if(xfmedia_settings_get_bool("/xfmedia/general/show_tray_icon")
            && xfmedia_settings_get_bool("/xfmedia/general/minimize_to_tray")
            && mwin->is_iconified)
    {
        /*DBG("setting skip taskbar hint");*/
        gtk_window_set_skip_taskbar_hint(GTK_WINDOW(w), TRUE);
        gtk_widget_hide(mwin->window);
        gtk_window_deiconify(GTK_WINDOW(mwin->window));
    } else {
        /*DBG("unsetting skip taskbar hint");*/
        gtk_window_set_skip_taskbar_hint(GTK_WINDOW(w), FALSE);
    }
    
    return FALSE;
}

/* NOTE: this is a dirty hack and needs to be fixed */
static gboolean
expand_treescroll_idle(gpointer data)
{
    //XfMediaMainwin *mwin = data;
    
    gdk_threads_enter();
    //xfmedia_playlist_treeview_scroll_cb(gtk_tree_view_get_vadjustment(GTK_TREE_VIEW(mwin->plist->treeview)), mwin);
    gdk_threads_leave();
    
    return FALSE;
}

void
xfmedia_mainwin_expander_activate_cb(GtkExpander *widget, gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    gint h_l, h_s, w;
    
    if(gtk_expander_get_expanded(widget)) {
        gtk_window_get_size(GTK_WINDOW(mwin->window), &w, &h_s);
        h_l = xfmedia_settings_get_int("/xfmedia/general/height_large");
        if(h_l == -1) {
            h_l = h_s * 3;
            xfmedia_settings_set_int("/xfmedia/general/height_large", h_l);
        }
        xfmedia_settings_set_int("/xfmedia/general/height_small", h_s);
        xfmedia_settings_set_int("/xfmedia/general/width", w);
        
        gtk_widget_set_size_request(mwin->window, -1, -1);
        gtk_window_resize(GTK_WINDOW(mwin->window), w, h_l);
        gtk_widget_show(GTK_WIDGET(mwin->plist));
        xfmedia_settings_set_bool("/xfmedia/playlist/playlist_expanded", TRUE);
        g_timeout_add(350, expand_treescroll_idle, mwin);
    } else {
        gtk_window_get_size(GTK_WINDOW(mwin->window), &w, &h_l);
        gtk_widget_hide(GTK_WIDGET(mwin->plist));
        h_s = xfmedia_settings_get_int("/xfmedia/general/height_small");
        gtk_widget_set_size_request(mwin->window, -1, h_s);
        gtk_window_resize(GTK_WINDOW(mwin->window), w, h_s);
        xfmedia_settings_set_int("/xfmedia/general/height_large", h_l);
        xfmedia_settings_set_int("/xfmedia/general/width", w);
        xfmedia_settings_set_bool("/xfmedia/playlist/playlist_expanded", FALSE);
    }
}

static void
about_dlg_zero_pointer(GtkWidget **dlg)
{
    *dlg = NULL;
}

static struct
{
    gchar *name;
    gchar *email;
    gchar *langcode;
} translator_list[] = {
    { "Bernhard Walle", "bernhard.walle@gmx.de", "de" },
    { "pi", "pi@beobide.net", "eu" },
    { "Jari Rahkonen", "jari.rahkonen@pp1.inet.fi", "fi" },
    { "Stephane Roy", "sroy@j2n.net", "fr" },
    { "XaNaX", "xanax@no-log.org", "fr" },
    { "Rimas Kudelis", "rg@akl.lt", "lt" },
    { "Robert Kurowski", "koorek@o2.pl", "pl" },
    { "Tomas Schertel", "tschertel@gmail.com", "pt_BR" },
    { "Juraz Brosz", "juro@jurajbrosz.info", "sk" },
    { NULL, NULL, NULL }
};

void
xfmedia_about_cb(GtkWidget *w, gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    XfceAboutInfo *ainfo;
    static GtkWidget *dlg = NULL;
    GtkWidget *toplevel = NULL;
    gint i;
    gchar *TRANSLATOR = _("Translator"), translator_str[1024];
    
    if(dlg) {
        gtk_window_present(GTK_WINDOW(dlg));
        return;
    }
    
    if(w)
        toplevel = gtk_widget_get_toplevel(w);
    if(!toplevel)
        toplevel = mwin->window;
    
    ainfo = xfce_about_info_new("Xfmedia", VERSION,
            _("Xfmedia is a lightweight media player,\nbased on the xine engine"),
            XFCE_COPYRIGHT_TEXT("2004-2005", "Brian Tarricone"), XFCE_LICENSE_GPL);
    xfce_about_info_set_homepage(ainfo, "http://spuriousinterrupt.org/projects/xfmedia/");
    
    xfce_about_info_add_credit(ainfo, "Brian Tarricone", "bjt23@cornell.edu",
            _("Original Author/Maintainer"));
    
    for(i = 0; translator_list[i].name; i++) {
        g_snprintf(translator_str, 1024, "%s (%s)", TRANSLATOR,
                translator_list[i].langcode);
        xfce_about_info_add_credit(ainfo, translator_list[i].name,
                translator_list[i].email, translator_str);
    }
    
    dlg = xfce_about_dialog_new(GTK_WINDOW(toplevel), ainfo, NULL);
    g_signal_connect(G_OBJECT(dlg), "response",
            G_CALLBACK(gtk_widget_destroy), NULL);
    g_signal_connect_swapped(G_OBJECT(dlg), "destroy",
            G_CALLBACK(about_dlg_zero_pointer), &dlg);
    gtk_widget_show_all(dlg);
}

gboolean
xfmedia_vwin_button_press_cb(GtkWidget *w, GdkEventButton *evt,
        gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    gint x, y;
    
    if(evt->button == 1 && evt->type == GDK_2BUTTON_PRESS) {
        gboolean is_fs = gtk_xine_is_fullscreen(mwin->gtx);
        
        gtk_xine_set_fullscreen(mwin->gtx, !is_fs);
        if(!is_fs) {
            gtk_window_get_position(GTK_WINDOW(mwin->video_window), &x, &y);
            xfmedia_settings_set_int("/xfmedia/general/vwin_pos_x", x);
            xfmedia_settings_set_int("/xfmedia/general/vwin_pos_y", y);
            gtk_widget_hide(mwin->video_window);
        } else {
            x = xfmedia_settings_get_int("/xfmedia/general/vwin_pos_x");
            y = xfmedia_settings_get_int("/xfmedia/general/vwin_pos_y");
            gtk_window_move(GTK_WINDOW(mwin->video_window), x, y);
            gtk_widget_show(mwin->video_window);
        }
        
        return TRUE;
    }
    
    return FALSE;
}

gboolean
xfmedia_vwin_motion_notify_cb(GtkWidget *w, GdkEventMotion *evt,
        gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    GdkScreen *gscreen;
    gint sheight, swidth;
    
    if(!gtk_xine_is_fullscreen(mwin->gtx))
        return FALSE;
    
    gscreen = gtk_widget_get_screen(mwin->window);
    swidth = gdk_screen_get_width(gscreen);
    sheight = gdk_screen_get_height(gscreen);
    
    if(!GTK_WIDGET_VISIBLE(mwin->fs_controls) && evt->y == sheight - 1) {
        Window fs_window;
        
        fs_window = GDK_WINDOW_XWINDOW(gtk_xine_get_fullscreen_window(mwin->gtx));
        gdk_property_change(mwin->fs_controls->window,
                gdk_atom_intern("WM_TRANSIENT_FOR", FALSE),
                gdk_atom_intern("WINDOW", FALSE),
                32, GDK_PROP_MODE_REPLACE,
                (const guchar *)&fs_window, sizeof(Window));
        
        gtk_widget_set_size_request(mwin->fs_controls, swidth, -1);
        gtk_widget_realize(mwin->fs_controls);
        gtk_window_move(GTK_WINDOW(mwin->fs_controls), 0,
                sheight - mwin->fs_controls->allocation.height);
        gtk_widget_show(mwin->fs_controls);
    } else if(GTK_WIDGET_VISIBLE(mwin->fs_controls)
            && evt->y < sheight - mwin->fs_controls->allocation.height)
    {
        gtk_widget_hide(mwin->fs_controls);
    }
    
    return FALSE;
}

gboolean
xfmedia_vwin_mapunmap_cb(GtkWidget *w, GdkEvent *evt, gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    int visible;
    
    if(evt->type == GDK_MAP)
        visible = 1;
    else if(evt->type == GDK_UNMAP)
        visible = 0;
    else
        return FALSE;
    
    gtk_xine_port_send_gui_data(mwin->gtx, XINE_GUI_SEND_VIDEOWIN_VISIBLE,
            &visible);
    
    return FALSE;
}

void
xfmedia_gtkxine_stream_ended_cb(GtkXine *gtx, gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    gboolean ret = FALSE;
    
    if(mwin->cur_playing
            && !xfmedia_settings_get_bool("/xfmedia/playlist/repeat_one")
            && xfmedia_settings_get_bool("/xfmedia/playlist/shuffle"))
    {
        DBG("putting item on shuffle history");
        g_queue_push_head(mwin->shuffle_history,
                xfmedia_playlist_entry_ref_copy(mwin->cur_playing));
    }
    
    if(xfmedia_settings_get_bool("/xfmedia/general/show_tray_icon"))
        xfmedia_tray_icon_set_tooltip(mwin->tray_icon, _("Stopped"));
    xfmedia_mainwin_set_play_buttons(mwin);
    
    xfmedia_mainwin_disconnect_slider_callback(mwin);
    gtk_range_set_value(GTK_RANGE(mwin->position_slider), 0.0);
    gtk_range_set_value(GTK_RANGE(mwin->fs_position_slider), 0.0);
    xfmedia_mainwin_connect_slider_callback(mwin);
    gtk_label_set_markup(GTK_LABEL(mwin->time_label),
            "<span font_family=\"monospace\" size=\"large\" weight=\"bold\"> 00:00</span>");
    gtk_label_set_markup(GTK_LABEL(mwin->fs_time_label),
            "<span font_family=\"monospace\"> 00:00</span>");
    
    xfmedia_plugins_forward_signal_VOID("stream-ended");
    
    if(xfmedia_settings_get_bool("/xfmedia/playlist/auto_advance")) {
        if(xfmedia_settings_get_bool("/xfmedia/playlist/repeat_one")) {
            if(mwin->cur_playing) {
                gint index = xfmedia_playlist_entry_ref_get_index(mwin->cur_playing);
                ret = xfmedia_mainwin_play_file_at_index(mwin, index);
            } else
                ret = gtk_xine_play(mwin->gtx, 0, 0);
        }
        if(!ret)
            ret = xfmedia_mainwin_play_file_at_index_persistent(mwin, -1);
    }
    
    if(!ret && GTK_WIDGET_VISIBLE(mwin->video_window))
        gtk_widget_hide(mwin->video_window);
}

static gboolean
gtkxine_hide_ui_msg_label(gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    xfmedia_mainwin_set_infobar_type(mwin, XFMEDIA_INFOBAR_SONG_LABEL);
    return FALSE;
}

void
xfmedia_gtkxine_ui_message_cb(GtkXine *gtx, const gchar *message,
        gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    
    DBG("got UI message: '%s'", message);
    
    xfmedia_mainwin_set_infobar_type(mwin, XFMEDIA_INFOBAR_UI_MESSAGE);
    gtk_label_set_text(GTK_LABEL(mwin->ui_message_label), message);
    
    g_timeout_add(3000, (GSourceFunc)gtkxine_hide_ui_msg_label, mwin);
}

void
xfmedia_gtkxine_progress_message_cb(GtkXine *gtx, const gchar *message, gint percent,
        gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    
    DBG("got progress message: '%s' at %d%%", message, percent);
    
    if(percent > 99)
        xfmedia_mainwin_set_infobar_type(mwin, XFMEDIA_INFOBAR_SONG_LABEL);
    else {
        xfmedia_mainwin_set_infobar_type(mwin, XFMEDIA_INFOBAR_PROGRESS_MESSAGE);
        
        gtk_label_set_text(GTK_LABEL(mwin->progress_label), message);
        gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(mwin->progress_bar),
                percent / 100.0);
    }
}

void
xfmedia_gtkxine_format_changed_cb(GtkXine *gtx, gint width, gint height,
        gint aspect_code, gboolean is_pan_scan, gpointer user_data)
{
    DBG("got format changed signal: %dx%d, aspect_code: %d, is_pan_scan: %s",
            width, height, aspect_code, is_pan_scan?"yes":"no");
}

/* FIXME: this actually won't work properly if the list contains more than one,
 * unique entry, but it's a start. */
void
xfmedia_gtkxine_mrl_reference_cb(GtkXine *gtx, const gchar *mrl,
        gint alternative, gpointer user_data)
{
#if 0  /* this thing sucks anyway */
    XfMediaMainwin *mwin = user_data;
    GtkTreePath *path;
    gint index = -1;
    
    DBG("got mrl reference signal: (%d) %s", alternative, mrl);
    
    if(alternative != 0)
        return;
    
    if(mwin->plist->cur_playing) {
        path = gtk_tree_row_reference_get_path(mwin->plist->cur_playing);
        if(path) {
            index = xfmedia_tree_path_to_index(path);
            index++;
            gtk_tree_path_free(path);
        }
    }
    
    index = xfmedia_mainwin_add_file(mwin, mrl, index, TRUE);
    
    path = xfmedia_playlist_get_path_at_index(mwin->plist, index);
    if(path) {
        xfmedia_playlist_play_file_at_path(mwin, path);
        gtk_tree_path_free(path);
    }
#endif
}
        

void
xfmedia_mainwin_quit_mi_cb(GtkMenuItem *mi, gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    
    xfmedia_quit(mwin, XFMEDIA_QUIT_MODE_EXT);
}

gboolean
xfmedia_mainwin_map_cb(GtkWidget *w, GdkEvent *evt, gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    GdkWindow *decs;
    
    decs = xfmedia_get_window_decorations(w->window);
    
    gdk_error_trap_push();
    
    if(decs) {
        mwin->window_decs = decs;
        gdk_error_trap_push();
        gdk_window_set_events(decs, gdk_window_get_events(decs)
                | GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK
                | GDK_POINTER_MOTION_MASK);
        gdk_window_add_filter(decs, xfmedia_mainwin_decs_filter, mwin);
    }
    
    gdk_error_trap_pop();
    
    return FALSE;
}

gboolean
xfmedia_mainwin_unmap_cb(GtkWidget *w, GdkEvent *evt, gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    
    if(mwin->window_decs) {
        gdk_error_trap_push();
        gdk_window_remove_filter(mwin->window_decs,
                xfmedia_mainwin_decs_filter, mwin);
        g_object_unref(G_OBJECT(mwin->window_decs));
        mwin->window_decs = NULL;
        gdk_error_trap_pop();
    }
    
    return FALSE;
}

gboolean
xfmedia_mainwin_keypress_cb(GtkWidget *w, GdkEventKey *evt, gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    const gchar *keybind_id;
    
    if(xfmedia_playlist_has_focus(mwin->plist)) {
        //if(gtk_major_version == 2 && gtk_minor_version >= 6) {
            /* ignore all keypresses in the treeview for gtk 2.6 and above, as
             * you can do type-ahead find in the treeview */
        //    DBG("ignored");
        //    return FALSE;
        /*} else*/ if(evt->keyval == GDK_Up || evt->keyval == GDK_KP_Up
                || evt->keyval == GDK_Down || evt->keyval == GDK_KP_Down
                || evt->keyval == GDK_Return || evt->keyval == GDK_KP_Enter)
        {
            /* ignore these, as they should be for playlist navigation */
            DBG("ignored");
            return FALSE;
        }
    }
    
    keybind_id = xfmedia_keybindings_lookup(evt->keyval, evt->state);
    if(keybind_id)
        xfmedia_keybindings_activate(mwin, keybind_id);
    
    return FALSE;
}

static void
mainwin_popupmenu_deactivate_cb(GtkWidget *w, gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    guint sigid = GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(w),
            "xfmedia-deactivate-sigid"));
    
    g_signal_handler_disconnect(G_OBJECT(w), sigid);
    mwin->temp_disable_autohide--;
}

gboolean
xfmedia_mainwin_menu_cb(GtkWidget *w, GdkEventButton *evt, gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    
    if(evt->button == 3 && mwin->popup_menu) {
        guint sigid = g_signal_connect(G_OBJECT(mwin->popup_menu), "deactivate",
                G_CALLBACK(mainwin_popupmenu_deactivate_cb), mwin);
        g_object_set_data(G_OBJECT(mwin->popup_menu),
                "xfmedia-deactivate-sigid", GUINT_TO_POINTER(sigid));
        mwin->temp_disable_autohide++;
        gtk_menu_popup(GTK_MENU(mwin->popup_menu), NULL, NULL, NULL, NULL,
                evt->button, evt->time);
        
        return TRUE;
    }
    
    return FALSE;
}

void
xfmedia_mainwin_mediamarks_changed_cb(gpointer data)
{
    XfMediaMainwin *mwin = data;
    GtkWidget *old_menu, *menu;
    
    old_menu = gtk_menu_item_get_submenu(GTK_MENU_ITEM(mwin->mmarks_mi));
    gtk_widget_destroy(old_menu);
    
    menu = xfmedia_mediamarks_create_menu(mwin);
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(mwin->mmarks_mi), menu);
}

void
xfmedia_mainwin_sort_title_cb(GtkWidget *w, gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    
    xfmedia_playlist_sort(mwin->plist, XFMEDIA_PLAYLIST_SORT_TITLE);
}

void
xfmedia_mainwin_sort_filename_cb(GtkWidget *w, gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    
    xfmedia_playlist_sort(mwin->plist, XFMEDIA_PLAYLIST_SORT_FILENAME);
}

void
xfmedia_mainwin_randomise_playlist_cb(GtkWidget *w, gpointer user_data)
{
    XfMediaMainwin *mwin = user_data;
    
    xfmedia_playlist_sort(mwin->plist, XFMEDIA_PLAYLIST_SORT_RANDOM);
}
