/*
 * Copyright (C) 2001-2003 the xine project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: gtkxine.c,v 1.42 2005/02/23 22:35:52 brian Exp $
 *
 * the xine engine in a widget - implementation
 *
 * some code originating from totem's gtkxine widget
 *
 * Modified: Copyright (c) 2004-2005 Brian Tarricone <bjt23@cornell.edu>
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <math.h>
#include <pwd.h>
#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
       
#include <X11/Xlib.h>
#include <X11/keysym.h>
#include <X11/cursorfont.h>
#include <X11/Xatom.h>
#ifdef HAVE_XTESTEXTENSION
#include <X11/extensions/XTest.h>
#endif /* HAVE_XTESTEXTENSION */

#include <glib-object.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>
#include <gtk/gtkwidget.h>
#include <gtk/gtkmain.h>
#include <gdk/gdkx.h>
#include <libxfce4util/libxfce4util.h>
#include <xine.h>

#include "gtkxine.h"

enum {
    SIG_STREAM_ENDED = 0,
    SIG_SET_TITLE,
    SIG_MESSAGE,
    SIG_FORMAT_CHANGE,
    SIG_PROGRESS_MSG,
    SIG_MRL_REFERENCE,
    N_SIGNALS
};

/*
 * global variable
 */

static gint verbosity = 0;

#define DEFAULT_WIDTH  420
#define DEFAULT_HEIGHT 315
#define DEFAULT_VIS_WIDTH  480
#define DEFAULT_VIS_HEIGHT 360

/* missing stuff from X includes */
#ifndef XShmGetEventBase
extern int XShmGetEventBase(Display *);
#endif

static void gtk_xine_class_init(GtkXineClass *klass);
static void gtk_xine_init(GtkXine *gxine);

static void gtk_xine_finalize(GObject *object);
static void gtk_xine_realize(GtkWidget *widget);
static void gtk_xine_unrealize(GtkWidget *widget);

static gint gtk_xine_expose(GtkWidget *widget, GdkEventExpose *event);

static void gtk_xine_size_allocate(GtkWidget *widget, GtkAllocation *allocation);

static GtkWidgetClass *parent_class = NULL;
static guint gtkxine_sigs[N_SIGNALS] = { 0 };

struct _GtkXinePriv
{
    xine_t *xine;
    xine_stream_t *stream;
    gchar *mrl;
    xine_event_queue_t *ev_queue;

    gchar *configfile;

    gchar *video_driver_id;
    gchar *audio_driver_id;

    xine_video_port_t *none_video_port;
    xine_video_port_t *video_port;
    xine_audio_port_t *audio_port;
    x11_visual_t vis;
    gdouble display_ratio;
    Display *display;
    gint screen;
    Window video_window;
    GC gc;
    GThread *thread;
    GAsyncQueue *thread_queue;
    gint completion_event;
    gint child_pid;

    gint xpos, ypos;
    gfloat resize_factor;
    gint oldwidth, oldheight;

    /* fullscreen stuff */

    gboolean fullscreen_mode;
    GdkRectangle fullscreen_geom;
    Window fullscreen_window, toplevel;
    GdkWindow *fullscreen_gwindow;
    Cursor no_cursor;
    Cursor on_cursor;
    gboolean cursor_visible;

    GdkVisibilityState visibility;

    gboolean have_xtest;
    gint xtest_keycode;

    /* visualization */
    xine_post_t *vis_plugin;
    gchar *vis_plugin_id;
};

/* VOID:INT,INT,INT,BOOLEAN */
static void
gtkxine_marshal_VOID__INT_INT_INT_BOOLEAN(GClosure *closure,
        GValue *return_value, guint n_param_values, const GValue *param_values,
        gpointer invocation_hint, gpointer marshal_data)
{
    typedef void (*GMarshalFunc_VOID__INT_INT_INT_BOOLEAN)(gpointer data1,
            gint arg_1, gint arg_2, gint arg_3, gboolean arg_4, gpointer data2);
    
    register GMarshalFunc_VOID__INT_INT_INT_BOOLEAN callback;
    register GCClosure *cc = (GCClosure*)closure;
    register gpointer data1, data2;
    
    g_return_if_fail (n_param_values == 5);
    
    if(G_CCLOSURE_SWAP_DATA(closure)) {
        data1 = closure->data;
        data2 = g_value_peek_pointer(param_values + 0);
    } else {
        data1 = g_value_peek_pointer(param_values + 0);
        data2 = closure->data;
    }
    
    callback = (GMarshalFunc_VOID__INT_INT_INT_BOOLEAN)(marshal_data ? marshal_data : cc->callback);
    
    callback(data1, g_value_get_int(param_values + 1),
            g_value_get_int(param_values + 2),
            g_value_get_int(param_values + 3),
            g_value_get_boolean(param_values + 4), data2);
}

/* VOID:STRING,INT */
static void
gtkxine_marshal_VOID__STRING_INT(GClosure *closure,
        GValue *return_value, guint n_param_values, const GValue *param_values,
        gpointer invocation_hint, gpointer marshal_data)
{
    typedef void (*GMarshalFunc_VOID__STRING_INT)(gpointer data1, const gchar *arg_1,
            gint arg_2, gpointer data2);
    
    register GMarshalFunc_VOID__STRING_INT callback;
    register GCClosure *cc = (GCClosure*)closure;
    register gpointer data1, data2;
    
    g_return_if_fail (n_param_values == 3);
    
    if(G_CCLOSURE_SWAP_DATA(closure)) {
        data1 = closure->data;
        data2 = g_value_peek_pointer(param_values + 0);
    } else {
        data1 = g_value_peek_pointer(param_values + 0);
        data2 = closure->data;
    }
    
    callback = (GMarshalFunc_VOID__STRING_INT)(marshal_data ? marshal_data : cc->callback);
    
    callback(data1, g_value_get_string(param_values + 1),
            g_value_get_int(param_values + 2), data2);
}

GtkType
gtk_xine_get_type(void)
{
    static GtkType gtk_xine_type = 0;

    if(!gtk_xine_type) {
        static const GTypeInfo gtk_xine_info = {
            sizeof(GtkXineClass),
            (GBaseInitFunc) NULL,
            (GBaseFinalizeFunc) NULL,
            (GClassInitFunc) gtk_xine_class_init,
            (GClassFinalizeFunc) NULL,
            NULL /* class_data */ ,
            sizeof(GtkXine),
            0 /* n_preallocs */ ,
            (GInstanceInitFunc) gtk_xine_init
        };

        gtk_xine_type = g_type_register_static(GTK_TYPE_WIDGET,
                "GtkXine", &gtk_xine_info, (GTypeFlags)0);
    }

    return gtk_xine_type;
}

static void
gtk_xine_class_init(GtkXineClass *klass)
{
    GObjectClass *object_class;
    GtkWidgetClass *widget_class;

    object_class = (GObjectClass *)klass;
    widget_class = (GtkWidgetClass *)klass;

    parent_class = g_type_class_peek_parent(klass);

    /* GtkWidget */
    widget_class->realize = gtk_xine_realize;
    widget_class->unrealize = gtk_xine_unrealize;
    widget_class->size_allocate = gtk_xine_size_allocate;
    widget_class->expose_event = gtk_xine_expose;

    /* GObject */
    object_class->set_property = NULL;
    object_class->get_property = NULL;
    object_class->finalize = gtk_xine_finalize;
    
    /* signals */
    gtkxine_sigs[SIG_STREAM_ENDED] = g_signal_new("stream-ended",
            GTK_TYPE_XINE, G_SIGNAL_RUN_FIRST,
            G_STRUCT_OFFSET(GtkXineClass, stream_ended), NULL, NULL,
            g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
    
    gtkxine_sigs[SIG_SET_TITLE] = g_signal_new("set-title",
            GTK_TYPE_XINE, G_SIGNAL_RUN_FIRST,
            G_STRUCT_OFFSET(GtkXineClass, set_title), NULL, NULL,
            g_cclosure_marshal_VOID__STRING, G_TYPE_NONE, 1, G_TYPE_STRING);
    
    gtkxine_sigs[SIG_MESSAGE] = g_signal_new("ui-message",
            GTK_TYPE_XINE, G_SIGNAL_RUN_FIRST,
            G_STRUCT_OFFSET(GtkXineClass, ui_message), NULL, NULL,
            g_cclosure_marshal_VOID__STRING, G_TYPE_NONE, 1, G_TYPE_STRING);
    
    gtkxine_sigs[SIG_FORMAT_CHANGE] = g_signal_new("format-changed",
            GTK_TYPE_XINE, G_SIGNAL_RUN_FIRST,
            G_STRUCT_OFFSET(GtkXineClass, format_changed), NULL, NULL,
            gtkxine_marshal_VOID__INT_INT_INT_BOOLEAN, G_TYPE_NONE, 4,
            G_TYPE_INT, G_TYPE_INT, G_TYPE_INT, G_TYPE_BOOLEAN);
    
    gtkxine_sigs[SIG_PROGRESS_MSG] = g_signal_new("progress-message",
            GTK_TYPE_XINE, G_SIGNAL_RUN_FIRST,
            G_STRUCT_OFFSET(GtkXineClass, progress_message), NULL, NULL,
            gtkxine_marshal_VOID__STRING_INT, G_TYPE_NONE, 2,
            G_TYPE_STRING, G_TYPE_INT);
    
    gtkxine_sigs[SIG_MRL_REFERENCE] = g_signal_new("mrl-reference",
            GTK_TYPE_XINE, G_SIGNAL_RUN_FIRST,
            G_STRUCT_OFFSET(GtkXineClass, mrl_reference), NULL, NULL,
            gtkxine_marshal_VOID__STRING_INT, G_TYPE_NONE, 2,
            G_TYPE_STRING, G_TYPE_INT);
}

static void
gtk_xine_init(GtkXine *gtx)
{

    GTK_WIDGET(gtx)->requisition.width = DEFAULT_WIDTH;
    GTK_WIDGET(gtx)->requisition.height = DEFAULT_HEIGHT;

    /* create a new xine instance, load config values */
    
    gtx->priv = g_new0(GtkXinePriv, 1);

    gtx->priv->xine = xine_new();
    
    gtx->priv->configfile = xfce_get_homefile(".xine", "config", NULL);
    xine_config_load(gtx->priv->xine, gtx->priv->configfile);

    xine_engine_set_param(gtx->priv->xine, XINE_ENGINE_PARAM_VERBOSITY, verbosity);

    gtx->priv->resize_factor = 1.0;

    xine_init(gtx->priv->xine);

#ifdef HAVE_XTESTEXTENSION
    {
        int a, b, c, d;

        gtx->priv->have_xtest = XTestQueryExtension(gdk_display, &a, &b, &c, &d);
        if(gtx->priv->have_xtest == True)
            gtx->priv->xtest_keycode = XKeysymToKeycode(gdk_display, XK_Shift_L);
    }
#endif

    gtx->priv->vis_plugin = NULL;
    gtx->priv->vis_plugin_id = NULL;

#if 0
    printf("gtkxine: extensions: '%s'\n", xine_get_file_extensions(gtx->priv->xine));
#endif

}

static void
gtk_xine_finalize(GObject *object)
{

    GtkXine *gtx = GTK_XINE(object);
    
    if(gtx->priv->mrl) {
        g_free(gtx->priv->mrl);
        gtx->priv->mrl = NULL;
    }
    
    /* save configuration */
    xine_config_save(gtx->priv->xine, gtx->priv->configfile);
    g_free(gtx->priv->configfile);

    /* exit xine */
    xine_exit(gtx->priv->xine);
    
    g_free(gtx->priv);
    gtx->priv = NULL;

    G_OBJECT_CLASS(parent_class)->finalize(object);

    gtx = NULL;
}

static void
gtk_xine_not_so_idle_add(GSourceFunc callback, gpointer data)
{
    GSource *idle_source = g_idle_source_new();
    GMainContext *mainctx = g_main_context_default();
    
    g_source_set_priority(idle_source, G_PRIORITY_HIGH_IDLE);
    g_source_set_callback(idle_source, callback, data, NULL);
    g_source_attach(idle_source, mainctx);
    g_source_unref(idle_source);
}

typedef enum {
    GTK_XINE_SIGPARAM_INT = 0,
    GTK_XINE_SIGPARAM_BOOL,
    GTK_XINE_SIGPARAM_FLOAT,
    GTK_XINE_SIGPARAM_STRING
} GtkXineSignalParamType;

typedef struct
{
    guint signal;
    GList *params;
    GtkXine *gtx;
} GtkXineSignalData;

typedef struct
{
    GtkXineSignalParamType type;
    gpointer data;
} GtkXineSignalParam;

static gboolean
xine_event_emit_idled(gpointer data)
{
    GtkXineSignalData *sigdata = data;
    GList *l;
    
    switch(sigdata->signal) {
        case SIG_STREAM_ENDED:
            gdk_threads_enter();
            g_signal_emit(G_OBJECT(sigdata->gtx), gtkxine_sigs[sigdata->signal], 0);
            gdk_threads_leave();
            break;
        
        case SIG_SET_TITLE:
        case SIG_MESSAGE:
            gdk_threads_enter();
            g_signal_emit(G_OBJECT(sigdata->gtx), gtkxine_sigs[sigdata->signal], 0,
                    ((GtkXineSignalParam *)g_list_nth_data(sigdata->params, 0))->data);
            gdk_threads_leave();
            break;
        
        case SIG_FORMAT_CHANGE:
            gdk_threads_enter();
            g_signal_emit(G_OBJECT(sigdata->gtx), gtkxine_sigs[sigdata->signal], 0,
                    GPOINTER_TO_INT(((GtkXineSignalParam *)g_list_nth_data(sigdata->params, 0))->data),
                    GPOINTER_TO_INT(((GtkXineSignalParam *)g_list_nth_data(sigdata->params, 1))->data),
                    GPOINTER_TO_INT(((GtkXineSignalParam *)g_list_nth_data(sigdata->params, 2))->data),
                    GPOINTER_TO_INT(((GtkXineSignalParam *)g_list_nth_data(sigdata->params, 3))->data)?TRUE:FALSE);
            gdk_threads_leave();
            break;
        
        case SIG_PROGRESS_MSG:
        case SIG_MRL_REFERENCE:
            gdk_threads_enter();
            g_signal_emit(G_OBJECT(sigdata->gtx), gtkxine_sigs[sigdata->signal], 0,
                    ((GtkXineSignalParam *)g_list_nth_data(sigdata->params, 0))->data,
                    GPOINTER_TO_INT(((GtkXineSignalParam *)g_list_nth_data(sigdata->params, 1))->data));
            gdk_threads_leave();
            break;
        
        default:
            g_critical("GtkXine: Got bad signal type (%d)", sigdata->signal);
            break;
    }
    
    for(l = sigdata->params; l; l = l->next) {
        GtkXineSignalParam *param = l->data;
        if(param->type == GTK_XINE_SIGPARAM_STRING)
            g_free(param->data);
        g_free(param);
    }
    if(sigdata->params)
        g_list_free(sigdata->params);    
    g_free(sigdata);
    
    return FALSE;
}

static void
event_listener_cb(void *user_data, const xine_event_t *event)
{
    GtkXine *gtx = user_data;
    GtkXineSignalData *sigdata;
    GtkXineSignalParam *sigparam;
    
    switch(event->type) {
        case XINE_EVENT_UI_PLAYBACK_FINISHED:
            sigdata = g_new0(GtkXineSignalData, 1);
            sigdata->signal = SIG_STREAM_ENDED;
            sigdata->gtx = gtx;
            break;
        
        case XINE_EVENT_UI_SET_TITLE:
            {
                xine_ui_data_t *uidata = (xine_ui_data_t *)event->data;
                
                sigdata = g_new0(GtkXineSignalData, 1);
                sigdata->signal = SIG_SET_TITLE;
                sigdata->gtx = gtx;
                
                sigparam = g_new0(GtkXineSignalParam, 1);
                sigparam->type = GTK_XINE_SIGPARAM_STRING;
                sigparam->data = g_strdup(uidata->str);
                sigdata->params = g_list_append(sigdata->params, sigparam);
            }
            break;
        
        case XINE_EVENT_UI_MESSAGE:
            {
                xine_ui_message_data_t *mdata = (xine_ui_message_data_t *)event->data;
                
                sigdata = g_new0(GtkXineSignalData, 1);
                sigdata->signal = SIG_MESSAGE;
                sigdata->gtx = gtx;
                
                sigparam = g_new0(GtkXineSignalParam, 1);
                sigparam->type = GTK_XINE_SIGPARAM_STRING;
                sigparam->data = g_strdup(mdata->messages);
                sigdata->params = g_list_append(sigdata->params, sigparam);
            }
            break;
        
        case XINE_EVENT_FRAME_FORMAT_CHANGE:
            {
                xine_format_change_data_t *fdata = (xine_format_change_data_t *)event->data;
                
                sigdata = g_new0(GtkXineSignalData, 1);
                sigdata->signal = SIG_FORMAT_CHANGE;
                sigdata->gtx = gtx;
                
                sigparam = g_new0(GtkXineSignalParam, 1);
                sigparam->type = GTK_XINE_SIGPARAM_INT;
                sigparam->data = GINT_TO_POINTER(fdata->width);
                sigdata->params = g_list_append(sigdata->params, sigparam);
                
                sigparam = g_new0(GtkXineSignalParam, 1);
                sigparam->type = GTK_XINE_SIGPARAM_INT;
                sigparam->data = GINT_TO_POINTER(fdata->height);
                sigdata->params = g_list_append(sigdata->params, sigparam);
                
                sigparam = g_new0(GtkXineSignalParam, 1);
                sigparam->type = GTK_XINE_SIGPARAM_INT;
                sigparam->data = GINT_TO_POINTER(fdata->aspect);
                sigdata->params = g_list_append(sigdata->params, sigparam);
                
                sigparam = g_new0(GtkXineSignalParam, 1);
                sigparam->type = GTK_XINE_SIGPARAM_BOOL;
                sigparam->data = GINT_TO_POINTER(fdata->pan_scan);
                sigdata->params = g_list_append(sigdata->params, sigparam);
            }
            break;
        
        case XINE_EVENT_PROGRESS:
            {
                xine_progress_data_t* pdata = (xine_progress_data_t *)event->data;
                
                sigdata = g_new0(GtkXineSignalData, 1);
                sigdata->signal = SIG_PROGRESS_MSG;
                sigdata->gtx = gtx;
                
                sigparam = g_new0(GtkXineSignalParam, 1);
                sigparam->type = GTK_XINE_SIGPARAM_STRING;
                sigparam->data = g_strdup(pdata->description);
                sigdata->params = g_list_append(sigdata->params, sigparam);
                
                sigparam = g_new0(GtkXineSignalParam, 1);
                sigparam->type = GTK_XINE_SIGPARAM_INT;
                sigparam->data = GINT_TO_POINTER(pdata->percent);
                sigdata->params = g_list_append(sigdata->params, sigparam);
            }
            break;
        
        case XINE_EVENT_MRL_REFERENCE:
            {
                xine_mrl_reference_data_t *mdata = (xine_mrl_reference_data_t *)event->data;
                
                sigdata = g_new0(GtkXineSignalData, 1);
                sigdata->signal = SIG_MRL_REFERENCE;
                sigdata->gtx = gtx;
                
                sigparam = g_new0(GtkXineSignalParam, 1);
                sigparam->type = GTK_XINE_SIGPARAM_STRING;
                sigparam->data = g_strdup((gchar *)mdata->mrl);
                sigdata->params = g_list_append(sigdata->params, sigparam);
                
                sigparam = g_new0(GtkXineSignalParam, 1);
                sigparam->type = GTK_XINE_SIGPARAM_INT;
                sigparam->data = GINT_TO_POINTER(mdata->alternative);
                sigdata->params = g_list_append(sigdata->params, sigparam);
            }
            break;
        
        default:
            /*g_message("GtkXine: Unhandled async event of type %d\n", event->type);*/
            return;
    }
    
    gtk_xine_not_so_idle_add(xine_event_emit_idled, sigdata);
}

static void
populate_fullscreen_geom(GtkXine *gtx)
{
    GdkScreen *gscreen;
    gint monitor;
    
    gscreen = gtk_widget_get_screen(GTK_WIDGET(gtx));
    gtx->priv->screen = gdk_screen_get_number(gscreen);
    if(gdk_screen_get_n_monitors(gscreen) == 1) {
        gtx->priv->fullscreen_geom.x = 0;
        gtx->priv->fullscreen_geom.y = 0;
        gtx->priv->fullscreen_geom.width = gdk_screen_get_width(gscreen);
        gtx->priv->fullscreen_geom.height = gdk_screen_get_height(gscreen);
    } else {
        GtkWidget *toplevel = gtk_widget_get_toplevel(GTK_WIDGET(gtx));
        gtx->priv->screen = gdk_screen_get_number(gscreen);
        monitor = gdk_screen_get_monitor_at_window(gscreen, toplevel->window);
        gdk_screen_get_monitor_geometry(gscreen, monitor,
                &gtx->priv->fullscreen_geom);
    }
}

static void
dest_size_cb(gpointer gtx_gen, gint video_width, gint video_height,
        gdouble video_pixel_aspect, gint *dest_width, gint *dest_height,
        gdouble *dest_pixel_aspect)
{

    GtkXine *gtx = (GtkXine *)gtx_gen;

    /* correct size with video_pixel_aspect */
    if(video_pixel_aspect >= gtx->priv->display_ratio)
        video_width = video_width * video_pixel_aspect / gtx->priv->display_ratio + .5;
    else
        video_height = video_height * gtx->priv->display_ratio / video_pixel_aspect + .5;

    if(gtx->priv->fullscreen_mode) {
        *dest_width = gtx->priv->fullscreen_geom.width;
        *dest_height = gtx->priv->fullscreen_geom.height;
    } else {
        if(gtx->priv->resize_factor == 0.0) {
            *dest_width = GTK_WIDGET(gtx)->allocation.width;
            *dest_height = GTK_WIDGET(gtx)->allocation.height;
        } else {
            *dest_width = video_width * gtx->priv->resize_factor;
            *dest_height = video_height * gtx->priv->resize_factor;
        }
    }

    *dest_pixel_aspect = gtx->priv->display_ratio;
}

static gboolean
gtk_xine_idle_resize(gpointer user_data)
{
    GtkXine *gtx = user_data;
    GtkWindow *toplevel;
    gint video_width = gtx->priv->oldwidth;
    gint video_height = gtx->priv->oldheight;

#ifdef LOG
    printf("gtkxine: idle resize to %d x %d\n", video_width, video_height);
#endif

    gdk_threads_enter();

    toplevel = GTK_WINDOW(gtk_widget_get_toplevel(GTK_WIDGET(gtx)));

    gtk_window_set_resizable(toplevel, FALSE);
    GTK_WIDGET(gtx)->allocation.width = video_width;
    GTK_WIDGET(gtx)->allocation.height = video_height;
    gtk_widget_set_size_request(GTK_WIDGET(gtx), video_width, video_height);
    gtk_widget_queue_resize(gtk_widget_get_parent(GTK_WIDGET(gtx)));
    while(gtk_events_pending())
        gtk_main_iteration();

    gtk_window_set_resizable(toplevel, TRUE);

    gdk_threads_leave();

#ifdef LOG
    printf("gtkxine: idle signal done\n");
#endif

    return FALSE;
}

static void
frame_output_cb(gpointer gtx_gen, gint video_width, gint video_height,
        gdouble video_pixel_aspect, gint *dest_x, gint *dest_y,
        gint *dest_width, gint *dest_height, gdouble *dest_pixel_aspect,
        gint *win_x, gint *win_y)
{

    GtkXine *gtx = GTK_XINE(gtx_gen);

    g_return_if_fail(GTK_IS_XINE(gtx));

    /* correct size with video_pixel_aspect */
    if(video_pixel_aspect >= gtx->priv->display_ratio)
        video_width = video_width * video_pixel_aspect / gtx->priv->display_ratio + .5;
    else
        video_height = video_height * gtx->priv->display_ratio / video_pixel_aspect + .5;

    *dest_x = 0;
    *dest_y = 0;

    if(gtx->priv->fullscreen_mode) {
        *win_x = 0;
        *win_y = 0;

        *dest_width = gtx->priv->fullscreen_geom.width;
        *dest_height = gtx->priv->fullscreen_geom.height;
    } else {
        *win_x = gtx->priv->xpos;
        *win_y = gtx->priv->ypos;

        if(gtx->priv->resize_factor != 0.0) {    /* => auto-resize */
            video_width *= gtx->priv->resize_factor;
            video_height *= gtx->priv->resize_factor;

            /* size changed? */

            if((video_width != gtx->priv->oldwidth)
                    || (video_height != gtx->priv->oldheight))
            {
                gtx->priv->oldwidth = video_width;
                gtx->priv->oldheight = video_height;

                /* why can't we do this right here? */
                gtk_xine_not_so_idle_add(gtk_xine_idle_resize, gtx);
            }

            gtx->priv->resize_factor = 0.0;
        }

        *dest_width = GTK_WIDGET(gtx)->allocation.width;
        *dest_height = GTK_WIDGET(gtx)->allocation.height;
    }

    *dest_pixel_aspect = gtx->priv->display_ratio;
}

static xine_video_port_t *
load_video_out_driver(GtkXine *gtx)
{
    gdouble res_h, res_v;
    x11_visual_t vis;
    const gchar *video_driver_id;
    xine_video_port_t *video_port;
    GdkScreen *gscreen = gtk_widget_get_screen(GTK_WIDGET(gtx));
    
    vis.display = gtx->priv->display;
    vis.screen = gtx->priv->screen;
    vis.d = gtx->priv->video_window;
    res_h = gdk_screen_get_width(gscreen) * 1000 / gdk_screen_get_width_mm(gscreen);
    res_v = gdk_screen_get_height(gscreen) * 1000 / gdk_screen_get_height_mm(gscreen);
    /*
    res_h = (DisplayWidth(gtx->priv->display, gtx->priv->screen) * 1000
        / DisplayWidthMM(gtx->priv->display, gtx->priv->screen));
    res_v = (DisplayHeight(gtx->priv->display, gtx->priv->screen) * 1000
        / DisplayHeightMM(gtx->priv->display, gtx->priv->screen));
    */
    gtx->priv->display_ratio = res_v / res_h;

    if(fabs(gtx->priv->display_ratio - 1.0) < 0.01)
        gtx->priv->display_ratio = 1.0;

    vis.dest_size_cb = dest_size_cb;
    vis.frame_output_cb = frame_output_cb;
    vis.user_data = gtx;

    if(gtx->priv->video_driver_id)
        video_driver_id = gtx->priv->video_driver_id;
    else {
        const gchar *const *driver_ids = xine_list_video_output_plugins(gtx->priv->xine);
        const gchar **choices;
        gint i, n = 0;
        
        while(driver_ids[n])
            n++;
        n += 2;
        
        choices = g_malloc0(sizeof(gchar *) * n);
        choices[0] = "auto";
        for(i = 0; driver_ids[i]; i++)
            choices[i+1] = driver_ids[i];

        /* try to init video with stored information */
        i = xine_config_register_enum(gtx->priv->xine, "video.driver", 0,
                (gchar **)choices, "video driver to use", NULL, 10, NULL, NULL);
        video_driver_id = choices[i];
        g_free(choices);
    }
    
    if(strcmp(video_driver_id, "auto")) {
        video_port = xine_open_video_driver(gtx->priv->xine,
                video_driver_id, XINE_VISUAL_TYPE_X11, (void *)&vis);
        if(video_port) {
            if(!gtx->priv->video_driver_id)
                gtx->priv->video_driver_id = g_strdup(video_driver_id);
            return video_port;
        } else
            g_warning("GtkXine: Video driver %s failed.  Attempting autodetection.\n", video_driver_id);
    }

    return xine_open_video_driver(gtx->priv->xine, NULL,
            XINE_VISUAL_TYPE_X11, (void *)&vis);
}

static xine_audio_port_t *
load_audio_out_driver(GtkXine *gtx)
{
    xine_audio_port_t *audio_port;
    const gchar *audio_driver_id;

    if(gtx->priv->audio_driver_id)
        audio_driver_id = gtx->priv->audio_driver_id;
    else {
        gchar **driver_ids = (gchar **)xine_list_audio_output_plugins(gtx->priv->xine);
        gchar **choices;
        gint i;

        choices = malloc(sizeof(gchar *) * 100);
        choices[0] = "auto";
        i = 0;
        while(driver_ids[i]) {
            DBG("%s", driver_ids[i]);
            choices[i + 1] = strdup(driver_ids[i]);
            i++;
        }
        choices[i + 1] = 0;

        /* try to init audio with stored information */
        i = xine_config_register_enum(gtx->priv->xine,
            "audio.driver", 0, choices, "audio driver to use", NULL, 10, NULL, NULL);
        audio_driver_id = choices[i];
    }

    if(!strcmp(audio_driver_id, "null"))
        return NULL;

    DBG("trying audio driver %s", audio_driver_id);
    if(strcmp(audio_driver_id, "auto")) {
        audio_port = xine_open_audio_driver(gtx->priv->xine, audio_driver_id, NULL);
        if(audio_port)
            return audio_port;
        else
            printf("audio driver %s failed\n", audio_driver_id);
    }

    /* autoprobe */
    return xine_open_audio_driver(gtx->priv->xine, NULL, NULL);
}

typedef struct
{
    GObject *object;
    gchar *signal_name;
    GdkEvent gdk_event;
} GtkXineEventFwdData;

static gboolean
gtk_xine_send_window_event_idled(gpointer user_data)
{
    GtkXineEventFwdData *fwddata = user_data;
    gboolean ret;
    
    gdk_threads_enter();
    g_signal_emit_by_name(fwddata->object, fwddata->signal_name,
            &fwddata->gdk_event, &ret);
    gdk_threads_leave();
    
    if(fwddata->gdk_event.type == GDK_KEY_PRESS)
        g_free(((GdkEventKey *)&fwddata->gdk_event)->string);
    
    g_free(fwddata);
    
    return FALSE;
}

void
gtk_xine_port_send_gui_data(GtkXine *gtx, gint type, gpointer data)
{
    Display *dpy;
    xine_video_port_t *vo_port;
    
    g_return_if_fail(GTK_IS_XINE(gtx));
    
    dpy = gtx->priv->display;
    
    XLockDisplay(dpy);
    
    if(gtx->priv->video_port)
        vo_port = gtx->priv->video_port;
    else
        vo_port = gtx->priv->none_video_port;
    
    xine_port_send_gui_data(vo_port, type, data);
    
    XUnlockDisplay(dpy);
}

static gpointer
xine_thread(gpointer gtx_gen)
{
    GtkXine *gtx = GTK_XINE(gtx_gen);
    Display *dpy = gtx->priv->display;
    GValue gval;
    guint dbl_click_time, dbl_click_dist, last_button_press[3] = { 0, 0, 0 };
    guint last_x_coord[3] = { 0, 0, 0 }, last_y_coord[3] = { 0, 0, 0 };
    GAsyncQueue *aqueue = gtx->priv->thread_queue;
    fd_set rfd;
    struct timeval tv;
    GtkXineEventFwdData *fwddata;
    
    g_async_queue_ref(aqueue);
    
    if(gdk_setting_get("Net/DoubleClickTime", &gval)) {
        dbl_click_time = g_value_get_int(&gval);
        g_value_unset(&gval);
    } else
        dbl_click_time = 250;
    
    if(gdk_setting_get("Net/DoubleClickDistance", &gval)) {
        dbl_click_dist = g_value_get_int(&gval);
        g_value_unset(&gval);
    } else
        dbl_click_dist = 5;

    while(1) {
        XEvent event;
        gint message, nevents, i;
        
        message = GPOINTER_TO_INT(g_async_queue_try_pop(aqueue));
        if(message)
            break;
        
        FD_ZERO(&rfd);
        FD_SET(ConnectionNumber(dpy), &rfd);
        tv.tv_sec = 1;
        tv.tv_usec = 0;
        if(select(FD_SETSIZE, &rfd, NULL, NULL, &tv) <= 0
                || !FD_ISSET(ConnectionNumber(dpy), &rfd))
        {
            continue;
        }
        
        nevents = XPending(dpy);
        
        for(i = 0; i < nevents; i++) {
            XNextEvent(dpy, &event);
    
            /* printf ("gtkxine: got an event (%d)\n", event.type);  */\
    
            switch(event.type) {
                case Expose:
                    if(event.xexpose.count != 0)
                        break;
                    
                    gtk_xine_port_send_gui_data(gtx,
                            XINE_GUI_SEND_EXPOSE_EVENT, &event);
                    
                    break;
        
        
                case FocusOut:            /* when in fullscreen mode, keep the focus */
                    if(gtx->priv->fullscreen_mode) {
                        XLockDisplay(gtx->priv->display);
                        XSetInputFocus(gtx->priv->display,
                            gtx->priv->fullscreen_window, RevertToNone, CurrentTime);
                        XUnlockDisplay(gtx->priv->display);
                    }
                    break;
        
                case MotionNotify:
                    {
                        XMotionEvent *mevent = (XMotionEvent *)&event;
                        x11_rectangle_t rect;
                        xine_event_t event;
                        xine_input_data_t input;
        
#ifdef LOG
                        printf("gtkxine: mouse event: mx=%d my=%d\n", mevent->x, mevent->y);
#endif
        
                        rect.x = mevent->x;
                        rect.y = mevent->y;
                        rect.w = 0;
                        rect.h = 0;
                        
                        gtk_xine_port_send_gui_data(gtx,
                                XINE_GUI_SEND_TRANSLATE_GUI_TO_VIDEO, &rect);
                        
                        event.type = XINE_EVENT_INPUT_MOUSE_MOVE;
                        event.data = &input;
                        event.data_length = sizeof(input);
                        input.button = 0;    /* no buttons, just motion */
                        input.x = rect.x;
                        input.y = rect.y;
                        xine_event_send(gtx->priv->stream, &event);
        
                        if(gtx->priv->fullscreen_mode && !gtx->priv->cursor_visible) {
                            /* make mouse pointer visible */
                            XLockDisplay(gtx->priv->display);
                            XDefineCursor(gtx->priv->display, gtx->priv->fullscreen_window,
                                gtx->priv->on_cursor);
                            XFlush(gtx->priv->display);
                            XUnlockDisplay(gtx->priv->display);
        
                            gtx->priv->cursor_visible = TRUE;
                        }
                    }
                    {
                        GdkEventMotion *gdk_event;
                        XMotionEvent *mevent = (XMotionEvent *)&event;
                        GtkWidget *toplevel = gtk_widget_get_toplevel(GTK_WIDGET(gtx));
                        
                        fwddata = g_new0(GtkXineEventFwdData, 1);
                        
                        fwddata->object = G_OBJECT(toplevel);
                        fwddata->signal_name = "motion-notify-event";
                        gdk_event = (GdkEventMotion *)&fwddata->gdk_event;
                        
                        gdk_event->type = GDK_MOTION_NOTIFY;
                        gdk_event->window = toplevel->window;
                        gdk_event->send_event = mevent->send_event;
                        gdk_event->time = mevent->time;
                        gdk_event->x = mevent->x;
                        gdk_event->y = mevent->y;
                        gdk_event->state = mevent->state;
                        gdk_event->is_hint = mevent->is_hint;
                        gdk_event->x_root = mevent->x_root;
                        gdk_event->y_root = mevent->y_root;
                        
                        gtk_xine_not_so_idle_add(gtk_xine_send_window_event_idled, fwddata);
                    }
                    break;
        
                case ButtonPress:
                    {
                        XButtonEvent *bevent = (XButtonEvent *)&event;
#ifdef LOG
                        printf("gtkxine: mouse button event: mx=%d my=%d\n",
                            bevent->x, bevent->y);
#endif
        
                        if(bevent->button == Button1) {
                            x11_rectangle_t rect;
                            xine_event_t event;
                            xine_input_data_t input;
                            
                            rect.x = bevent->x;
                            rect.y = bevent->y;
                            rect.w = 0;
                            rect.h = 0;
                            
                            gtk_xine_port_send_gui_data(gtx,
                                    XINE_GUI_SEND_TRANSLATE_GUI_TO_VIDEO,
                                    &rect);
                            
                            event.type = XINE_EVENT_INPUT_MOUSE_BUTTON;
                            event.data = &input;
                            event.data_length = sizeof(input);
                            input.button = 1;
                            input.x = rect.x;
                            input.y = rect.y;
                            xine_event_send(gtx->priv->stream, &event);
                        }
                    }
                    {
                        GdkEventButton *gdk_event;
                        XButtonEvent *bevent = (XButtonEvent *)&event;
                        GtkWidget *toplevel = gtk_widget_get_toplevel(GTK_WIDGET(gtx));
                        
                        fwddata = g_new0(GtkXineEventFwdData, 1);
                        
                        fwddata->object = G_OBJECT(toplevel);
                        fwddata->signal_name = "button-press-event";
                        gdk_event = (GdkEventButton *)&fwddata->gdk_event;
                        
                        gdk_event->type = GDK_BUTTON_PRESS;
                        if(bevent->button < 4) {
                            guint dx = abs(bevent->x - last_x_coord[bevent->button-1]);
                            guint dy = abs(bevent->y - last_y_coord[bevent->button-1]);
                            dx *= dx; dy *= dy;
                            if(bevent->time - last_button_press[bevent->button-1] <= dbl_click_time
                                    && sqrt(dx + dy) < dbl_click_dist)
                            {
                                gdk_event->type = GDK_2BUTTON_PRESS;
                            }
                            last_button_press[bevent->button-1] = bevent->time;
                            last_x_coord[bevent->button-1] = bevent->x;
                            last_y_coord[bevent->button-1] = bevent->y;
                        }
                        gdk_event->window = toplevel->window;
                        gdk_event->send_event = bevent->send_event;
                        gdk_event->time = bevent->time;
                        gdk_event->x = bevent->x;
                        gdk_event->y = bevent->y;
                        gdk_event->state = bevent->state;
                        gdk_event->button = bevent->button;
                        gdk_event->x_root = bevent->x_root;
                        gdk_event->y_root = bevent->y_root;
                        
                        gtk_xine_not_so_idle_add(gtk_xine_send_window_event_idled, fwddata);
                    }
                    break;
                
                case ButtonRelease:
                    {
                        GdkEventButton *gdk_event;
                        XButtonEvent *bevent = (XButtonEvent *)&event;
                        GtkWidget *toplevel = gtk_widget_get_toplevel(GTK_WIDGET(gtx));
                        
                        fwddata = g_new0(GtkXineEventFwdData, 1);
                        
                        fwddata->object = G_OBJECT(toplevel);
                        fwddata->signal_name = "button-release-event";
                        gdk_event = (GdkEventButton *)&fwddata->gdk_event;
                        
                        gdk_event->type = GDK_BUTTON_RELEASE;
                        gdk_event->window = toplevel->window;
                        gdk_event->send_event = bevent->send_event;
                        gdk_event->time = bevent->time;
                        gdk_event->x = bevent->x;
                        gdk_event->y = bevent->y;
                        gdk_event->state = bevent->state;
                        gdk_event->button = bevent->button;
                        gdk_event->x_root = bevent->x_root;
                        gdk_event->y_root = bevent->y_root;
                        
                        gtk_xine_not_so_idle_add(gtk_xine_send_window_event_idled, fwddata);
                    }
                    break;
        
                case KeyPress:
                    {
                        GdkEventKey *gdk_event;
                        XKeyEvent *kevent = (XKeyEvent *)&event;
                        gchar buffer[20];
                        gint bufsize = 20;
                        KeySym keysym;
                        XComposeStatus compose;
                        GtkWidget *toplevel = gtk_widget_get_toplevel(GTK_WIDGET(gtx));
                        
                        memset(buffer, 0, sizeof(buffer));
                        XLookupString(kevent, buffer, bufsize, &keysym, &compose);
                        
                        fwddata = g_new0(GtkXineEventFwdData, 1);
                        
                        fwddata->object = G_OBJECT(toplevel);
                        fwddata->signal_name = "key-press-event";
                        gdk_event = (GdkEventKey *)&fwddata->gdk_event;
        
                        gdk_event->type = GDK_KEY_PRESS;
                        gdk_event->window = toplevel->window;
                        gdk_event->time = kevent->time;
                        gdk_event->keyval = keysym;
                        gdk_event->state = (GdkModifierType)kevent->state;
                        gdk_event->string = g_strdup(buffer);
                        gdk_event->length = strlen(buffer);
                        
                        gtk_xine_not_so_idle_add(gtk_xine_send_window_event_idled, fwddata);
                    }
        
                    break;
                
                case MapNotify:
                    {
                        int visible = 1;
                        DBG("videowin map notify");
                        gtk_xine_port_send_gui_data(gtx,
                                XINE_GUI_SEND_VIDEOWIN_VISIBLE, &visible);
                    }
                    
                    break;
                
                case UnmapNotify:
                    {
                        int visible = 0;
                        DBG("videowin unmap notify");
                        gtk_xine_port_send_gui_data(gtx,
                                XINE_GUI_SEND_VIDEOWIN_VISIBLE, &visible);
                    }
            }
#if 0  /* deprecated? */
            if(event.type == gtx->priv->completion_event) {
                DBG("completion event");
                gtk_xine_port_send_gui_data(gtx,
                        XINE_GUI_SEND_COMPLETION_EVENT, &event);
            }
#endif
        }
    }
    
    g_async_queue_unref(aqueue);
    
    g_thread_exit(NULL);
    return NULL;
}

static gboolean
configure_cb(GtkWidget *widget, GdkEventConfigure *event, gpointer user_data)
{
    GtkXine *gtx;

    gtx = GTK_XINE(user_data);

    gtx->priv->xpos = event->x + GTK_WIDGET(gtx)->allocation.x;
    gtx->priv->ypos = event->y + GTK_WIDGET(gtx)->allocation.y;

    return FALSE;
}

static void
reset_screen_saver(GtkXine *gtx)
{
    /* printf ("gtkxine: disabling screen saver\n"); */

#ifdef HAVE_XTESTEXTENSION
    if(gtx->priv->have_xtest == True) {
        XTestFakeKeyEvent(gdk_display, gtx->priv->xtest_keycode, True, CurrentTime);
        XTestFakeKeyEvent(gdk_display, gtx->priv->xtest_keycode, False, CurrentTime);
        XSync(gdk_display, False);
    } else
#endif
    {
        XResetScreenSaver(gdk_display);
    }
}

static gint
gtk_timeout_cb(gpointer data)
{
    GtkXine *gtx = GTK_XINE(data);

    if(gtx->priv->fullscreen_mode) {
        XRaiseWindow(gtx->priv->display, gtx->priv->fullscreen_window);
        if( gtx->priv->cursor_visible) {
            XLockDisplay(gtx->priv->display);
            XDefineCursor(gtx->priv->display, gtx->priv->fullscreen_window,
                    gtx->priv->no_cursor);
            XFlush(gtx->priv->display);
            gtx->priv->cursor_visible = FALSE;
            XUnlockDisplay(gtx->priv->display);
        }
    }

    if(xine_get_status(gtx->priv->stream) == XINE_STATUS_PLAY
            && gtx->priv->fullscreen_mode)
    {
        reset_screen_saver(gtx);
    }

    return TRUE;
}

static unsigned char bm_no_data[] = { 0, 0, 0, 0, 0, 0, 0, 0 };

static void
gtk_xine_realize(GtkWidget *widget)
{
    /* GdkWindowAttr attributes; */
    /* gint          attributes_mask; */
    GtkXine *gtx;
    XGCValues values;
    Pixmap bm_no;
    gint screen;
    gulong black_pixel;
    GdkScreen *gscreen;
    xine_cfg_entry_t cfgentry;
    gint real_bufsize = -1;

    g_return_if_fail(GTK_IS_XINE(widget));

    gtx = GTK_XINE(widget);

    /* set realized flag */
    GTK_WIDGET_SET_FLAGS(widget, GTK_REALIZED);
    
    /* load audio driver */
    DBG("creating audio driver");
    gtx->priv->audio_port = load_audio_out_driver(gtx);
    if(!gtx->priv->audio_port)
        g_critical("GtkXine: Unable to load audio output driver.");

    /* create our own video window */
    gscreen = gtk_widget_get_screen(gtk_widget_get_toplevel(widget));
    screen = gdk_screen_get_number(gscreen);
    black_pixel = BlackPixel(gdk_display, screen);
    
    DBG("creating video XWindow");
    gtx->priv->video_window = XCreateSimpleWindow(GDK_DISPLAY(),
            GDK_WINDOW_XWINDOW(gtk_widget_get_parent_window(widget)), 0, 0,
            widget->allocation.width, widget->allocation.height, 1,
            black_pixel, black_pixel);

    widget->window = gdk_window_foreign_new(gtx->priv->video_window);
    
    /* init fullscreen geometry to something sane, at least */
    populate_fullscreen_geom(gtx);
    gtx->priv->fullscreen_mode = FALSE;

    gtx->priv->toplevel =
        GDK_WINDOW_XWINDOW(gdk_window_get_toplevel(gtk_widget_get_parent_window(widget)));

    /* track configure events of toplevel window */
    g_signal_connect(gtk_widget_get_toplevel(widget),
            "configure-event", G_CALLBACK(configure_cb), gtx);
    
    DBG("calling XInitThreads()");
    if(!XInitThreads()) {
        g_critical("GtkXine: XInitThreads() failed; thread-safe X11 libs unavailable!\n");
        gtx->priv->video_port = xine_open_video_driver(gtx->priv->xine,
                "none", XINE_VISUAL_TYPE_NONE, NULL);
        goto create_stream;
    }
    
    DBG("opening a new XDisplay");
    gtx->priv->display = XOpenDisplay(gdk_display_get_name(gdk_screen_get_display(gtk_widget_get_screen(widget))));

    if(!gtx->priv->display) {
        printf("GtkXine: XOpenDisplay failed!\n");
        gtx->priv->video_port = xine_open_video_driver(gtx->priv->xine,
                "none", XINE_VISUAL_TYPE_NONE, NULL);
        goto create_stream;
    }
    
    DBG("locking display");
    XLockDisplay(gtx->priv->display);

    gtx->priv->screen = screen;

    if(XShmQueryExtension(gtx->priv->display) == True) {
        gtx->priv->completion_event = XShmGetEventBase(gtx->priv->display) + ShmCompletion;
    } else {
        gtx->priv->completion_event = -1;
    }

    XSelectInput(gtx->priv->display, gtx->priv->video_window,
            StructureNotifyMask | ExposureMask | ButtonPressMask
            | ButtonReleaseMask |PointerMotionMask | SubstructureNotifyMask
            | VisibilityChangeMask);
    
    /* load "none" driver.  it doesn't need a large buffer (saves RAM). */
    if(xine_config_lookup_entry(gtx->priv->xine, "video.num_buffers", &cfgentry)
            && cfgentry.type == XINE_CONFIG_TYPE_NUM)
    {
        DBG("got video buffer size (%d)", cfgentry.num_value);
        real_bufsize = cfgentry.num_value;
        cfgentry.num_value = 5;
        xine_config_update_entry(gtx->priv->xine, &cfgentry);
    }
    gtx->priv->none_video_port = xine_open_video_driver(gtx->priv->xine,
            "none", XINE_VISUAL_TYPE_NONE, NULL);
    if(real_bufsize != -1) {
        cfgentry.num_value = real_bufsize;
        xine_config_update_entry(gtx->priv->xine, &cfgentry);
    }
    
    values.foreground = BlackPixel(gtx->priv->display, gtx->priv->screen);
    values.background = WhitePixel(gtx->priv->display, gtx->priv->screen);

    gtx->priv->gc = XCreateGC(gtx->priv->display, gtx->priv->video_window,
        (GCForeground | GCBackground), &values);
    
    DBG("unlocking display");
    XUnlockDisplay(gtx->priv->display);

    /* create mouse cursors */
    bm_no = XCreateBitmapFromData(gtx->priv->display,
            gtx->priv->video_window, bm_no_data, 8, 8);
    gtx->priv->no_cursor = XCreatePixmapCursor(gtx->priv->display, bm_no,
            bm_no, (XColor *)&black_pixel, (XColor *)&black_pixel, 0, 0);
    gtx->priv->on_cursor = XCreateFontCursor(gtx->priv->display, XC_left_ptr);
    gtx->priv->cursor_visible = TRUE;
    
    create_stream:
    
    /* create a stream object */
    DBG("creating stream");
    gtx->priv->stream = xine_stream_new(gtx->priv->xine,
            gtx->priv->audio_port, gtx->priv->none_video_port);
    
    /* create xine event handler thread */
    DBG("creating xine event queue");
    gtx->priv->ev_queue = xine_event_new_queue(gtx->priv->stream);
    xine_event_create_listener_thread(gtx->priv->ev_queue,
            event_listener_cb, gtx);
    
    /* now, create a xine thread to handle X events */
    gtx->priv->thread_queue = g_async_queue_new();
    gtx->priv->thread = g_thread_create(xine_thread, gtx, TRUE, NULL);
    if(!gtx->priv->thread)
        g_critical("Unable to spawn xine thread!\n");
    
    /* add timeout to switch off mouse cursor, disable screen saver */
    g_timeout_add(4000, gtk_timeout_cb, gtx);
}


static void
gtk_xine_unrealize(GtkWidget *widget)
{
    GtkXine *gtx;

    g_return_if_fail(widget != NULL);
    g_return_if_fail(GTK_IS_XINE(widget));

    gtx = GTK_XINE(widget);
    
    xine_stop(gtx->priv->stream);

    /* stop X event thread */
    g_async_queue_push(gtx->priv->thread_queue, GINT_TO_POINTER(1));
    g_thread_join(gtx->priv->thread);
    gtx->priv->thread = NULL;
    
    /* ditch xine event thread */
    xine_event_dispose_queue(gtx->priv->ev_queue);
    /* ditch stream */
    xine_close(gtx->priv->stream);
    xine_dispose(gtx->priv->stream);
    /* and a/v drivers */
    if(gtx->priv->video_port)
        xine_close_video_driver(gtx->priv->xine, gtx->priv->video_port);
    if(gtx->priv->audio_port)
        xine_close_audio_driver(gtx->priv->xine, gtx->priv->audio_port);
    if(gtx->priv->none_video_port)
        xine_close_video_driver(gtx->priv->xine, gtx->priv->none_video_port);
    
    if(gtx->priv->video_driver_id)
        g_free(gtx->priv->video_driver_id);
    
    /* Hide all windows */
    if(GTK_WIDGET_MAPPED(widget))
        gtk_widget_unmap(widget);

    GTK_WIDGET_UNSET_FLAGS(widget, GTK_MAPPED);

    /* This destroys widget->window and unsets the realized flag */
    if(GTK_WIDGET_CLASS(parent_class)->unrealize)
        (*GTK_WIDGET_CLASS(parent_class)->unrealize)(widget);
}


GtkWidget *
gtk_xine_new(const gchar *video_driver_id, const gchar *audio_driver_id)
{
    GtkXine *gtx = g_object_new(gtk_xine_get_type(), NULL);

    if(video_driver_id && *video_driver_id)
        gtx->priv->video_driver_id = strdup(video_driver_id);
    else
        gtx->priv->video_driver_id = NULL;

    if(audio_driver_id && audio_driver_id)
        gtx->priv->audio_driver_id = strdup(audio_driver_id);
    else
        gtx->priv->audio_driver_id = NULL;
    
    return GTK_WIDGET(gtx);
}


static gboolean
gtk_xine_expose(GtkWidget *widget, GdkEventExpose *event)
{
    /* eat expose events, as xine_thread() will handle them and pass them
     * to xine itself */
    return TRUE;
}

static void
gtk_xine_size_allocate(GtkWidget *widget, GtkAllocation *allocation)
{
    g_return_if_fail(widget != NULL);
    g_return_if_fail(GTK_IS_XINE(widget));

    widget->allocation = *allocation;

    if(GTK_WIDGET_REALIZED(widget)) {
        gdk_window_move_resize(widget->window,
            allocation->x, allocation->y, allocation->width, allocation->height);
    }
}

gboolean
gtk_xine_open(GtkXine *gtx, const gchar *mrl)
{
    gint ret, has_video;
    
    g_return_val_if_fail(gtx != NULL, FALSE);
    g_return_val_if_fail(GTK_IS_XINE(gtx), FALSE);
    g_return_val_if_fail(gtx->priv->xine != NULL, FALSE);

    if(verbosity)
        printf("gtkxine: calling xine_open, mrl = '%s'\n", mrl);

    if(gtx->priv->mrl) {
        xine_stop(gtx->priv->stream);
        xine_close(gtx->priv->stream);
        g_free(gtx->priv->mrl);
        gtx->priv->mrl = NULL;
    }
    
    DBG("about to call xine_open()");
    ret = xine_open(gtx->priv->stream, mrl);
    DBG("xine_open() returned %d", ret);
    if(ret)
        gtx->priv->mrl = g_strdup(mrl);
    else
        return FALSE;
    
    has_video = xine_get_stream_info(gtx->priv->stream,
            XINE_STREAM_INFO_HAS_VIDEO);
    if((has_video || gtx->priv->vis_plugin_id) && !gtx->priv->video_port) {
        DBG("creating real vid out driver");
        XLockDisplay(gtx->priv->display);
        gtx->priv->video_port = load_video_out_driver(gtx);
        XUnlockDisplay(gtx->priv->display);
        if(gtx->priv->video_port) {
            xine_post_out_t *vid_src = xine_get_video_source(gtx->priv->stream);
            xine_post_wire_video_port(vid_src, gtx->priv->video_port);
            DBG("rewired stream to real vid out driver");
        }
    } else if(!has_video && gtx->priv->video_port && !gtx->priv->vis_plugin_id) {
        xine_post_out_t *vid_src = xine_get_video_source(gtx->priv->stream);
        DBG("wiring stream to none vid out driver and destroying real driver");
        xine_post_wire_video_port(vid_src, gtx->priv->none_video_port);
        xine_close_video_driver(gtx->priv->xine, gtx->priv->video_port);
        gtx->priv->video_port = NULL;
    }
    
    return TRUE;
}

gboolean
gtk_xine_play(GtkXine *gtx, gint pos, gint start_time)
{
    gint res;

    g_return_val_if_fail(gtx != NULL, FALSE);
    g_return_val_if_fail(GTK_IS_XINE(gtx), FALSE);
    g_return_val_if_fail(gtx->priv->xine != NULL, FALSE);

    if(verbosity)
        g_message("gtkxine: calling xine_play start_pos = %d, start_time = %d\n",
                pos, start_time);

    /* visualization */
    if(!xine_get_stream_info(gtx->priv->stream, XINE_STREAM_INFO_HAS_VIDEO)) {
        if(!gtx->priv->vis_plugin && gtx->priv->vis_plugin_id) {
            xine_post_out_t *audio_source;
            xine_post_in_t *input;
            xine_cfg_entry_t cfg;

            gtx->priv->vis_plugin = xine_post_init(gtx->priv->xine, gtx->priv->vis_plugin_id, 0,
                &gtx->priv->audio_port, &gtx->priv->video_port);
            
            xine_config_lookup_entry(gtx->priv->xine, "post.goom_width", &cfg);
            cfg.num_value = DEFAULT_VIS_WIDTH;
            xine_config_update_entry(gtx->priv->xine, &cfg);
            
            xine_config_lookup_entry(gtx->priv->xine, "post.goom_height", &cfg);
            cfg.num_value = DEFAULT_VIS_HEIGHT;
            xine_config_update_entry(gtx->priv->xine, &cfg);
            
            audio_source = xine_get_audio_source(gtx->priv->stream);
            input = xine_post_input(gtx->priv->vis_plugin, "audio in");

            xine_post_wire(audio_source, input);
        }
    } else {
        if(gtx->priv->vis_plugin) {
            xine_post_out_t *pp;

            pp = xine_get_audio_source(gtx->priv->stream);
            xine_post_wire_audio_port(pp, gtx->priv->audio_port);
            xine_post_dispose(gtx->priv->xine, gtx->priv->vis_plugin);
            gtx->priv->vis_plugin = NULL;
        }
    }
    
    res = xine_play(gtx->priv->stream, pos, start_time);

    return res;
}

gboolean
gtk_xine_trick_mode(GtkXine * gtx, gint mode, gint value)
{

    g_return_val_if_fail(gtx != NULL, 0);
    g_return_val_if_fail(GTK_IS_XINE(gtx), 0);
    g_return_val_if_fail(gtx->priv->stream != NULL, 0);

    return xine_trick_mode(gtx->priv->stream, mode, value);
}

gboolean
gtk_xine_get_pos_length(GtkXine *gtx, gint *pos_stream,
    gint *pos_time, gint *length_time)
{
    g_return_val_if_fail(gtx != NULL, 0);
    g_return_val_if_fail(GTK_IS_XINE(gtx), 0);
    g_return_val_if_fail(gtx->priv->stream != NULL, 0);

    return xine_get_pos_length(gtx->priv->stream, pos_stream, pos_time, length_time);
}

void
gtk_xine_stop(GtkXine *gtx)
{
    g_return_if_fail(gtx != NULL);
    g_return_if_fail(GTK_IS_XINE(gtx));
    g_return_if_fail(gtx->priv->stream != NULL);

    xine_stop(gtx->priv->stream);
}

gint
gtk_xine_get_error(GtkXine *gtx)
{
    g_return_val_if_fail(gtx != NULL, 0);
    g_return_val_if_fail(GTK_IS_XINE(gtx), 0);
    g_return_val_if_fail(gtx->priv->stream != NULL, 0);

    return xine_get_error(gtx->priv->stream);
}

gint
gtk_xine_get_status(GtkXine *gtx)
{
    g_return_val_if_fail(gtx != NULL, 0);
    g_return_val_if_fail(GTK_IS_XINE(gtx), 0);
    g_return_val_if_fail(gtx->priv->stream != NULL, 0);

    return xine_get_status(gtx->priv->stream);
}

void
gtk_xine_set_param(GtkXine *gtx, gint param, gint value)
{
    g_return_if_fail(gtx != NULL);
    g_return_if_fail(GTK_IS_XINE(gtx));
    g_return_if_fail(gtx->priv->stream != NULL);

    xine_set_param(gtx->priv->stream, param, value);
}

gint
gtk_xine_get_param(GtkXine *gtx, gint param)
{
    g_return_val_if_fail(gtx != NULL, 0);
    g_return_val_if_fail(GTK_IS_XINE(gtx), 0);
    g_return_val_if_fail(gtx->priv->stream != NULL, 0);

    return xine_get_param(gtx->priv->stream, param);
}

void
gtk_xine_engine_set_param(GtkXine *gtx, gint param, gint value)
{
    g_return_if_fail(gtx != NULL);
    g_return_if_fail(GTK_IS_XINE(gtx));
    g_return_if_fail(gtx->priv->stream != NULL);

    xine_engine_set_param(gtx->priv->xine, param, value);
}

gint
gtk_xine_engine_get_param(GtkXine *gtx, gint param)
{
    g_return_val_if_fail(gtx != NULL, 0);
    g_return_val_if_fail(GTK_IS_XINE(gtx), 0);
    g_return_val_if_fail(gtx->priv->stream != NULL, 0);

    return xine_engine_get_param(gtx->priv->xine, param);
}

gint
gtk_xine_get_audio_lang(GtkXine *gtx, gint channel, gchar * lang)
{
    g_return_val_if_fail(gtx != NULL, 0);
    g_return_val_if_fail(GTK_IS_XINE(gtx), 0);
    g_return_val_if_fail(gtx->priv->stream != NULL, 0);

    return xine_get_audio_lang(gtx->priv->stream, channel, lang);
}

gint
gtk_xine_get_spu_lang(GtkXine *gtx, gint channel, gchar * lang)
{
    g_return_val_if_fail(gtx != NULL, 0);
    g_return_val_if_fail(GTK_IS_XINE(gtx), 0);
    g_return_val_if_fail(gtx->priv->stream != NULL, 0);

    return xine_get_spu_lang(gtx->priv->stream, channel, lang);
}

gint
gtk_xine_get_stream_info(GtkXine *gtx, gint info)
{
    g_return_val_if_fail(gtx != NULL, 0);
    g_return_val_if_fail(GTK_IS_XINE(gtx), 0);
    g_return_val_if_fail(gtx->priv->stream != NULL, 0);

    return xine_get_stream_info(gtx->priv->stream, info);
}

G_CONST_RETURN gchar *
gtk_xine_get_meta_info(GtkXine *gtx, gint info)
{
    g_return_val_if_fail(gtx != NULL, 0);
    g_return_val_if_fail(GTK_IS_XINE(gtx), 0);
    g_return_val_if_fail(gtx->priv->stream != NULL, 0);

    return xine_get_meta_info(gtx->priv->stream, info);
}

G_CONST_RETURN gchar *
gtk_xine_get_current_mrl(GtkXine *gtx)
{
    g_return_val_if_fail(gtx, NULL);
    
    return gtx->priv->mrl;
}

#define MWM_HINTS_DECORATIONS   (1L << 1)
#define PROP_MWM_HINTS_ELEMENTS 5
typedef struct
{
    uint32_t flags;
    uint32_t functions;
    uint32_t decorations;
    int32_t input_mode;
    uint32_t status;
} MWMHints;

void
gtk_xine_set_fullscreen(GtkXine *gtx, gboolean fullscreen)
{
    static char *window_title = "gtx-fullscreen-window";
    XSizeHints hint;
    Atom prop;
    MWMHints mwmhints;
    XEvent xev;

    g_return_if_fail(gtx != NULL);
    g_return_if_fail(GTK_IS_XINE(gtx));
    g_return_if_fail(gtx->priv->stream != NULL);

    if(fullscreen == gtx->priv->fullscreen_mode)
        return;

    XLockDisplay(gtx->priv->display);

    if(fullscreen) {
        gulong black_pixel = BlackPixel(gtx->priv->display, gtx->priv->screen);
        
        populate_fullscreen_geom(gtx);
        
        /*int screen = DefaultScreen(gtx->priv->display);*/

        gtx->priv->fullscreen_window = XCreateSimpleWindow(gtx->priv->display,
            RootWindow(gtx->priv->display, gtx->priv->screen),
            gtx->priv->fullscreen_geom.x, gtx->priv->fullscreen_geom.y,
            gtx->priv->fullscreen_geom.width,
            gtx->priv->fullscreen_geom.height, 1,
            black_pixel, black_pixel);

        hint.flags = PPosition | PSize | PMinSize | PWinGravity;
        hint.x = gtx->priv->fullscreen_geom.x;
        hint.y = gtx->priv->fullscreen_geom.y;
        hint.width = gtx->priv->fullscreen_geom.width;
        hint.height = gtx->priv->fullscreen_geom.height;
        hint.min_width = gtx->priv->fullscreen_geom.width;
        hint.min_height = gtx->priv->fullscreen_geom.height;
        hint.win_gravity = StaticGravity;

        /* set Properties for window manager (always before mapping) */
        XSetStandardProperties(gtx->priv->display, gtx->priv->fullscreen_window,
            window_title, window_title, None, NULL, 0, &hint);

        XSetWMNormalHints(gtx->priv->display, gtx->priv->fullscreen_window, &hint);

        /* XSetWMHints (gtx->priv->display, gtx->priv->fullscreen_window, &hint); */

        /*
         * layer above most other things, like gnome panel
         * WIN_LAYER_ABOVE_DOCK  = 10
         *
         */
        {
            static Atom XA_WIN_LAYER = None;
            long propvalue[1];

            if(XA_WIN_LAYER == None)
                XA_WIN_LAYER = XInternAtom(gtx->priv->display, "_WIN_LAYER", False);

            propvalue[0] = 10;
            XChangeProperty(gtx->priv->display, gtx->priv->fullscreen_window,
                    XA_WIN_LAYER, XA_CARDINAL, 32, PropModeReplace,
                    (unsigned char *)propvalue, 1);
        }

        /*
         * fullscreen the modern (e.g. metacity) way
         */

        {
            static Atom XA_WIN_STATE = None;
            long propvalue[2];

            if(XA_WIN_STATE == None)
                XA_WIN_STATE = XInternAtom(gtx->priv->display, "_NET_WM_STATE", False);

            propvalue[0] = XInternAtom(gtx->priv->display, "_NET_WM_STATE_FULLSCREEN", False);
            propvalue[1] = 0;

            XChangeProperty(gtx->priv->display, gtx->priv->fullscreen_window,
                XA_WIN_STATE, XA_ATOM,
                32, PropModeReplace, (unsigned char *)propvalue, 1);
            XFlush(gtx->priv->display);

        }

        /*
         * wm, no borders please
         */

        prop = XInternAtom(gtx->priv->display, "_MOTIF_WM_HINTS", False);
        mwmhints.flags = MWM_HINTS_DECORATIONS;
        mwmhints.decorations = 0;
        XChangeProperty(gtx->priv->display, gtx->priv->fullscreen_window,
                prop, prop, 32, PropModeReplace,
                (unsigned char *)&mwmhints, PROP_MWM_HINTS_ELEMENTS);

        XSetTransientForHint(gtx->priv->display, gtx->priv->fullscreen_window, None);
        XRaiseWindow(gtx->priv->display, gtx->priv->fullscreen_window);

        XSelectInput(gtx->priv->display, gtx->priv->fullscreen_window,
                ExposureMask | KeyPressMask | ButtonPressMask
                | StructureNotifyMask | FocusChangeMask | PointerMotionMask);

        XMapWindow(gtx->priv->display, gtx->priv->fullscreen_window);

        XFlush(gtx->priv->display);

        /* Wait for map. */
        do {
            XMaskEvent(gtx->priv->display, StructureNotifyMask, &xev);
        } while(xev.type != MapNotify || xev.xmap.event != gtx->priv->fullscreen_window);

        XSetInputFocus(gtx->priv->display, gtx->priv->fullscreen_window, RevertToNone, CurrentTime);

        XMoveWindow(gtx->priv->display, gtx->priv->fullscreen_window,
                gtx->priv->fullscreen_geom.x, gtx->priv->fullscreen_geom.y);

        gtk_xine_port_send_gui_data(gtx, XINE_GUI_SEND_DRAWABLE_CHANGED,
                GUINT_TO_POINTER(gtx->priv->fullscreen_window));
        
        /* switch off mouse cursor */
        XDefineCursor(gtx->priv->display, gtx->priv->fullscreen_window,
                gtx->priv->no_cursor);
        XFlush(gtx->priv->display);

        gtx->priv->cursor_visible = FALSE;
        gtx->priv->fullscreen_gwindow = gdk_window_foreign_new(gtx->priv->fullscreen_window);

    } else {
        gtk_xine_port_send_gui_data(gtx, XINE_GUI_SEND_DRAWABLE_CHANGED,
                GUINT_TO_POINTER(gtx->priv->video_window));

        XDestroyWindow(gtx->priv->display, gtx->priv->fullscreen_window);

        XFlush(gtx->priv->display);
    }

    gtx->priv->fullscreen_mode = fullscreen;

    XUnlockDisplay(gtx->priv->display);
}

gint
gtk_xine_is_fullscreen(GtkXine *gtx)
{
    g_return_val_if_fail(gtx != NULL, 0);
    g_return_val_if_fail(GTK_IS_XINE(gtx), 0);
    g_return_val_if_fail(gtx->priv->stream != NULL, 0);

    return gtx->priv->fullscreen_mode;
}

gint
gtk_xine_get_current_frame(GtkXine *gtx, gint *width, gint *height,
        gint *ratio_code, gint *format, uint8_t *img)
{
    g_return_val_if_fail(gtx != NULL, 0);
    g_return_val_if_fail(GTK_IS_XINE(gtx), 0);
    g_return_val_if_fail(gtx->priv->stream != NULL, 0);

    return xine_get_current_frame(gtx->priv->stream, width, height,
            ratio_code, format, img);
}

gint
gtk_xine_get_log_section_count(GtkXine *gtx)
{
    g_return_val_if_fail(gtx != NULL, 0);
    g_return_val_if_fail(GTK_IS_XINE(gtx), 0);
    g_return_val_if_fail(gtx->priv->xine != NULL, 0);

    return xine_get_log_section_count(gtx->priv->xine);
}

gchar **
gtk_xine_get_log_names(GtkXine *gtx)
{
    g_return_val_if_fail(gtx != NULL, NULL);
    g_return_val_if_fail(GTK_IS_XINE(gtx), NULL);
    g_return_val_if_fail(gtx->priv->xine != NULL, NULL);

    return (gchar **) xine_get_log_names(gtx->priv->xine);
}

gchar **
gtk_xine_get_log(GtkXine *gtx, gint buf)
{

    g_return_val_if_fail(gtx != NULL, NULL);
    g_return_val_if_fail(GTK_IS_XINE(gtx), NULL);
    g_return_val_if_fail(gtx->priv->xine != NULL, NULL);

    return (gchar **)xine_get_log(gtx->priv->xine, buf);
}

void
gtk_xine_register_log_cb(GtkXine *gtx, xine_log_cb_t cb, void *user_data)
{
    g_return_if_fail(gtx != NULL);
    g_return_if_fail(GTK_IS_XINE(gtx));
    g_return_if_fail(gtx->priv->xine != NULL);

    return xine_register_log_cb(gtx->priv->xine, cb, user_data);
}

gchar **
gtk_xine_get_browsable_input_plugin_ids(GtkXine *gtx)
{

    g_return_val_if_fail(gtx != NULL, NULL);
    g_return_val_if_fail(GTK_IS_XINE(gtx), NULL);
    g_return_val_if_fail(gtx->priv->xine != NULL, NULL);

    return (gchar **)xine_get_browsable_input_plugin_ids(gtx->priv->xine);
}

xine_mrl_t **
gtk_xine_get_browse_mrls(GtkXine *gtx, const gchar *plugin_id,
        const gchar *start_mrl, gint *num_mrls)
{
    g_return_val_if_fail(gtx != NULL, NULL);
    g_return_val_if_fail(GTK_IS_XINE(gtx), NULL);
    g_return_val_if_fail(gtx->priv->xine != NULL, NULL);

    return (xine_mrl_t **)xine_get_browse_mrls(gtx->priv->xine, plugin_id,
            start_mrl, num_mrls);
}


GList *
gtk_xine_get_autoplay_input_plugin_ids(GtkXine *gtx)
{
    GList *list = NULL;
    const gchar * const *plugins;
    gint i;
    
    g_return_val_if_fail(gtx != NULL, NULL);
    g_return_val_if_fail(GTK_IS_XINE(gtx), NULL);
    g_return_val_if_fail(gtx->priv->xine != NULL, NULL);
    
    plugins = xine_get_autoplay_input_plugin_ids(gtx->priv->xine);
    if(!plugins)
        return NULL;
    
    for(i = 0; plugins[i]; i++)
        list = g_list_insert_sorted(list, (gpointer)plugins[i],
                (GCompareFunc)g_ascii_strcasecmp);

    return list;
}

GList *
gtk_xine_get_autoplay_mrls(GtkXine *gtx, const gchar *plugin_id)
{
    gchar **mrls;
    GList *list = NULL;
    gint i, num_mrls;
    
    g_return_val_if_fail(gtx != NULL, NULL);
    g_return_val_if_fail(GTK_IS_XINE(gtx), NULL);
    g_return_val_if_fail(gtx->priv->xine != NULL, NULL);
    
    mrls = xine_get_autoplay_mrls(gtx->priv->xine, plugin_id, &num_mrls);
    if(!mrls || num_mrls == 0)
        return NULL;
    
    for(i = 0; i < num_mrls; i++)
        list = g_list_append(list, mrls[i]);

    return list;
}

gchar *
gtk_xine_get_file_extensions(GtkXine *gtx)
{
    g_return_val_if_fail(gtx != NULL, NULL);
    g_return_val_if_fail(GTK_IS_XINE(gtx), NULL);
    g_return_val_if_fail(gtx->priv->xine != NULL, NULL);

    return (gchar *)xine_get_file_extensions(gtx->priv->xine);
}

gchar *
gtk_xine_get_mime_types(GtkXine *gtx)
{
    g_return_val_if_fail(gtx != NULL, NULL);
    g_return_val_if_fail(GTK_IS_XINE(gtx), NULL);
    g_return_val_if_fail(gtx->priv->xine != NULL, NULL);

    return (gchar *)xine_get_mime_types(gtx->priv->xine);
}

const gchar *
gtk_xine_config_register_string(GtkXine *gtx, const gchar *key,
        const gchar *def_value, const gchar *description,
        const gchar *help, gint exp_level, xine_config_cb_t changed_cb,
        void *cb_data)
{

    g_return_val_if_fail(gtx != NULL, NULL);
    g_return_val_if_fail(GTK_IS_XINE(gtx), NULL);
    g_return_val_if_fail(gtx->priv->xine != NULL, NULL);

    return (gchar *)xine_config_register_string(gtx->priv->xine, key,
            def_value, description, help, exp_level, changed_cb, cb_data);
}

gint
gtk_xine_config_register_range(GtkXine *gtx, const gchar *key, gint def_value,
        gint min, gint max, const gchar *description, const gchar *help,
        gint exp_level, xine_config_cb_t changed_cb, void *cb_data)
{
    g_return_val_if_fail(gtx != NULL, 0);
    g_return_val_if_fail(GTK_IS_XINE(gtx), 0);
    g_return_val_if_fail(gtx->priv->xine != NULL, 0);

    return xine_config_register_range(gtx->priv->xine, key, def_value,
            min, max, description, help, exp_level, changed_cb, cb_data);
}

gint
gtk_xine_config_register_enum(GtkXine *gtx, const gchar *key, gint def_value,
        gchar **values, const gchar *description, const gchar *help,
        gint exp_level, xine_config_cb_t changed_cb, void *cb_data)
{
    g_return_val_if_fail(gtx != NULL, 0);
    g_return_val_if_fail(GTK_IS_XINE(gtx), 0);
    g_return_val_if_fail(gtx->priv->xine != NULL, 0);

    return xine_config_register_enum(gtx->priv->xine, key, def_value, values,
            description, help, exp_level, changed_cb, cb_data);
}

gint
gtk_xine_config_register_num(GtkXine *gtx, const gchar *key, gint def_value,
        const gchar *description, const gchar *help, gint exp_level,
        xine_config_cb_t changed_cb, void *cb_data)
{
    g_return_val_if_fail(gtx != NULL, 0);
    g_return_val_if_fail(GTK_IS_XINE(gtx), 0);
    g_return_val_if_fail(gtx->priv->xine != NULL, 0);

    return xine_config_register_num(gtx->priv->xine, key, def_value,
            description, help, exp_level, changed_cb, cb_data);
}

gint
gtk_xine_config_register_bool(GtkXine *gtx, const gchar *key, gint def_value,
        const gchar *description, const gchar *help, gint exp_level,
        xine_config_cb_t changed_cb, void *cb_data)
{

    g_return_val_if_fail(gtx != NULL, 0);
    g_return_val_if_fail(GTK_IS_XINE(gtx), 0);
    g_return_val_if_fail(gtx->priv->xine != NULL, 0);

    return xine_config_register_bool(gtx->priv->xine, key, def_value,
            description, help, exp_level, changed_cb, cb_data);
}

int
gtk_xine_config_get_first_entry(GtkXine *gtx, xine_cfg_entry_t *entry)
{
    g_return_val_if_fail(gtx != NULL, 0);
    g_return_val_if_fail(GTK_IS_XINE(gtx), 0);
    g_return_val_if_fail(gtx->priv->xine != NULL, 0);

    return xine_config_get_first_entry(gtx->priv->xine, entry);
}

int
gtk_xine_config_get_next_entry(GtkXine *gtx, xine_cfg_entry_t *entry)
{
    g_return_val_if_fail(gtx != NULL, 0);
    g_return_val_if_fail(GTK_IS_XINE(gtx), 0);
    g_return_val_if_fail(gtx->priv->xine != NULL, 0);

    return xine_config_get_next_entry(gtx->priv->xine, entry);
}

int
gtk_xine_config_lookup_entry(GtkXine *gtx, const gchar *key,
        xine_cfg_entry_t *entry)
{
    g_return_val_if_fail(gtx != NULL, 0);
    g_return_val_if_fail(GTK_IS_XINE(gtx), 0);
    g_return_val_if_fail(gtx->priv->xine != NULL, 0);

    return xine_config_lookup_entry(gtx->priv->xine, key, entry);
}

void
gtk_xine_config_update_entry(GtkXine *gtx, xine_cfg_entry_t *entry)
{
    g_return_if_fail(gtx != NULL);
    g_return_if_fail(GTK_IS_XINE(gtx));
    g_return_if_fail(gtx->priv->xine != NULL);

    xine_config_update_entry(gtx->priv->xine, entry);
}

void
gtk_xine_config_load(GtkXine *gtx, const gchar *cfg_filename)
{
    g_return_if_fail(gtx != NULL);
    g_return_if_fail(GTK_IS_XINE(gtx));
    g_return_if_fail(gtx->priv->xine != NULL);

    xine_config_load(gtx->priv->xine, cfg_filename);
}

void
gtk_xine_config_save(GtkXine *gtx, const gchar *cfg_filename)
{
    g_return_if_fail(gtx != NULL);
    g_return_if_fail(GTK_IS_XINE(gtx));
    g_return_if_fail(gtx->priv->xine != NULL);

    xine_config_save(gtx->priv->xine, cfg_filename);
}

void
gtk_xine_config_reset(GtkXine *gtx)
{
    g_return_if_fail(gtx != NULL);
    g_return_if_fail(GTK_IS_XINE(gtx));
    g_return_if_fail(gtx->priv->xine != NULL);

    xine_config_reset(gtx->priv->xine);
}

void
gtk_xine_event_send(GtkXine *gtx, const xine_event_t *event)
{
    g_return_if_fail(gtx != NULL);
    g_return_if_fail(GTK_IS_XINE(gtx));
    g_return_if_fail(gtx->priv->stream != NULL);

    xine_event_send(gtx->priv->stream, event);
}


void
gtk_xine_set_resize_factor(GtkXine *gtx, gdouble factor)
{                                        /* 0.0 => don't resize */
    g_return_if_fail(gtx != NULL);
    g_return_if_fail(GTK_IS_XINE(gtx));
    g_return_if_fail(gtx->priv->xine != NULL);

    gtx->priv->resize_factor = factor;
}

void
gtk_xine_set_vis(GtkXine *gtx, const gchar *id)
{                              /* NULL to disable */
    g_return_if_fail(gtx != NULL);
    g_return_if_fail(GTK_IS_XINE(gtx));

    if(gtx->priv->vis_plugin_id)
        g_free(gtx->priv->vis_plugin_id);
    if(id)
        gtx->priv->vis_plugin_id = g_strdup(id);
    else
        gtx->priv->vis_plugin_id = NULL;
    
    if(gtx->priv->stream) {
        if(gtx->priv->vis_plugin) {
            xine_post_out_t *pp;

            pp = xine_get_audio_source(gtx->priv->stream);
            xine_post_wire_audio_port(pp, gtx->priv->audio_port);
            xine_post_dispose(gtx->priv->xine, gtx->priv->vis_plugin);
            gtx->priv->vis_plugin = NULL;
        }

        if(gtx->priv->vis_plugin_id) {
            xine_post_out_t *audio_source;
            xine_post_in_t *input;
            xine_cfg_entry_t cfg;
            
            if(!gtx->priv->video_port) {
                gboolean restart_after = (xine_get_status(gtx->priv->stream) == XINE_STATUS_PLAY);
                gint ps, pt, lt;
                
                if(restart_after) {
                    xine_get_pos_length(gtx->priv->stream, &ps, &pt, &lt);
                    xine_stop(gtx->priv->stream);
                }
                
                XLockDisplay(gtx->priv->display);
                gtx->priv->video_port = load_video_out_driver(gtx);
                XUnlockDisplay(gtx->priv->display);
                if(gtx->priv->video_port) {
                    xine_post_out_t *vid_src = xine_get_video_source(gtx->priv->stream);
                    xine_post_wire_video_port(vid_src, gtx->priv->video_port);
                    DBG("rewired stream to real vid out driver");
                }
                
                if(restart_after)
                    xine_play(gtx->priv->stream, ps, 0);
            }

            gtx->priv->vis_plugin = xine_post_init(gtx->priv->xine, 
                    gtx->priv->vis_plugin_id, 0,
                    &gtx->priv->audio_port, &gtx->priv->video_port);
            
            xine_config_lookup_entry(gtx->priv->xine, "post.goom_width", &cfg);
            cfg.num_value = DEFAULT_VIS_WIDTH;
            xine_config_update_entry(gtx->priv->xine, &cfg);
            
            xine_config_lookup_entry(gtx->priv->xine, "post.goom_height", &cfg);
            cfg.num_value = DEFAULT_VIS_HEIGHT;
            xine_config_update_entry(gtx->priv->xine, &cfg);
            
            audio_source = xine_get_audio_source(gtx->priv->stream);
            input = xine_post_input(gtx->priv->vis_plugin, "audio in");
            
            xine_post_wire(audio_source, input);
        }
    }
}

GList *
gtk_xine_list_post_plugins_typed(GtkXine *gtx, gint type)
{
    const gchar *const *plugins;
    GList *list = NULL;
    gint i;
    
    g_return_val_if_fail(gtx != NULL, 0);
    g_return_val_if_fail(GTK_IS_XINE(gtx), 0);
    g_return_val_if_fail(gtx->priv->xine != NULL, 0);
    
    plugins = xine_list_post_plugins_typed(gtx->priv->xine, type);
    if(!plugins)
        return NULL;
    
    for(i = 0; plugins[i]; i++)
        list = g_list_append(list, (gpointer)plugins[i]);

    return list;
}

xine_t *
gtk_xine_get_raw_xine_engine(GtkXine *gtx)
{
    g_return_val_if_fail(GTK_IS_XINE(gtx), NULL);
    return gtx->priv->xine;
}

xine_stream_t *
gtk_xine_get_raw_stream(GtkXine *gtx)
{
    g_return_val_if_fail(GTK_IS_XINE(gtx), NULL);
    return gtx->priv->stream;
}

xine_audio_port_t *
gtk_xine_get_raw_audio_port(GtkXine *gtx)
{
    g_return_val_if_fail(GTK_IS_XINE(gtx), NULL);
    return gtx->priv->audio_port;
}

xine_video_port_t *
gtk_xine_get_raw_video_port(GtkXine *gtx)
{
    g_return_val_if_fail(GTK_IS_XINE(gtx), NULL);
    return gtx->priv->video_port;
}

xine_post_t *
gtk_xine_get_raw_post_plugin(GtkXine *gtx)
{
    g_return_val_if_fail(GTK_IS_XINE(gtx), NULL);
    return gtx->priv->vis_plugin;
}

GdkWindow *
gtk_xine_get_fullscreen_window(GtkXine *gtx)
{
    g_return_val_if_fail(GTK_IS_XINE(gtx), NULL);
    return gtx->priv->fullscreen_gwindow;
}
