
/*   tubo.c (.992) */

/*  A program independent forking object module for gtk based programs.
 *  
 *  Copyright 2000-2002(C)  Edscott Wilson Garcia under GNU GPL
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

/*#define HYPER_DEBUG*/

#include <sys/time.h>


#ifndef HAVE_SYS_TYPES_H 
#error "This program requieres  the <sys/types.h> header file."
#endif
#ifndef HAVE_SYS_WAIT_H
#error "This program requieres  the <sys/wait.h> header file."
#endif
#ifndef HAVE_SIGNAL_H
#error "This program requieres  the <signal.h> header file."
#endif

/* signal handling */
#ifndef HAVE_SIGNAL
#ifndef HAVE_SIGACTION
#error "This program needs signals to work!"
#endif
#endif


#include <sys/types.h>
#include <sys/wait.h>

#include <unistd.h>
#include <stdarg.h>
#include <signal.h>
#include <errno.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <sys/select.h>

/* GTK */
#include <glib.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <gmodule.h>


/* public stuff */
#include "tubo.h"
/*#define HYPER_DEBUG
#define DEBUG*/

/* private stuff */

/* 
 * Double fork option (DOUBLE_FORK) would be needed if the function provided by
 * the user will fork (once or more). This is the case in xffm, when
 * it is important to hang on to all output produced.
 * */


static pid_t grandchildPID;

/* memcpy is necesary  */
#ifdef __GNUC__
/* memcpy is a GNU extension.*/
#define MEMCPY memcpy
#else
static void *MEMCPY(void *dest, const void *src, size_t n)
{
    char *destC, *srcC;
    size_t i;

    destC = (char *)dest;
    srcC = (char *)src;
    for(i = 0; i < n; i++)
	destC[i] = srcC[i];
    return dest;
}
#endif

static GList *valid_fork_objects=NULL;


typedef struct fork_structure
{
    pid_t PID;
    int tubo[3][3];
    /* user fork function: */
    void (*fork_function) (void *);
    void (*fork_finished_function) (pid_t);
    /* user parse functions: */
    int (*operate_stdout) (int, void *);
    int (*operate_stderr) (int, void *);
    int read_delay;
    int input_redlight;
    gboolean double_fork;
}
fork_struct;

static gint watch_input(gpointer data);


static fork_struct *TuboChupoFaros(fork_struct * forkO)
{
    int i;
#ifdef DEBUG
    printf("TuboChupoFaros()\n");
#endif
    if(!forkO)
    {
	return NULL;
    }
    for(i = 0; i < 3; i++)
    {
	if(forkO->tubo[i][0] > 0)
	{
	    close(forkO->tubo[i][0]);
	}
	if(forkO->tubo[i][1] > 0)
	{
	    close(forkO->tubo[i][1]);
	}
	if(forkO->tubo[i][2] > 0)
	{
#ifdef HYPER_DEBUG
	    printf("removing input signal %d\n", i);
#endif
	    g_source_remove(forkO->tubo[i][2]);
	    forkO->tubo[i][2] = 0;
	}
    }
    valid_fork_objects=g_list_remove(valid_fork_objects,(gpointer)forkO);
    free(forkO);
    return NULL;
}

#define TUBO_BLK_SIZE 256	/* 256 is the size of my cpu internal cache */


static char line[TUBO_BLK_SIZE];



static int get_line(gint src)
{
    int i;
    memset(line, 0, TUBO_BLK_SIZE);

    for(i = 0; i < TUBO_BLK_SIZE - 1; i++)
    {
	if(!read(src, line + i, 1))
	    break;
#ifdef HYPER_DEBUG
	printf("%c",line[i]);fflush(NULL); 
#endif
	if(*(line + i) == '\n')
	{
	    *(line + i + 1) = (char)0;
	    return 0;
	}
    }
    if(i) return i;		/* something has been read */
    return -1;
}


static gboolean TuboInput(gint src, int (*user_parse_function) (int, void *))
{
    int i;

  
    i = get_line(src);
#ifdef HYPER_DEBUG 
    if (i >=0) {
	    printf("TUBO i=%d get_line(%d)=%s", i,src,line);
	    fflush(NULL);
    }
#endif 
    if (i < 0) return FALSE;
    if (i==0 && *((char *)line)==0) return FALSE;
    (*user_parse_function) (i, (void *)line);
    return TRUE;

}

static gint check_read(gpointer data){
    fork_struct *newfork = (fork_struct *) data;
    gboolean retval1=FALSE,retval2=FALSE;
    fd_set rfds;
    struct timeval tv;
    int nfds = ((newfork->tubo[1][0]>newfork->tubo[2][0]) ? newfork->tubo[1][0] : newfork->tubo[2][0]);
    FD_ZERO(&rfds);
	
    if (newfork->tubo[1][0] >= 0) FD_SET(newfork->tubo[1][0], &rfds); /* stdout */
    if (newfork->tubo[2][0] >= 0) FD_SET(newfork->tubo[2][0], &rfds); /* stderr */

    /* Wait up to five microseconds. */
    tv.tv_sec = 0;
    tv.tv_usec = 5;
    
    
    
#ifdef HYPER_DEBUG
    printf("TUBO watch input...\n"); 
#endif
    if (select(nfds+1, &rfds, NULL, NULL, &tv) <= 0) {
#ifdef HYPER_DEBUG
printf("TUBO  select(nfds+1, &rfds, NULL, NULL, &tv) <= 0 : return TRUE\n");
#endif
	return TRUE;
    }
    if (!newfork) {
#ifdef HYPER_DEBUG
printf("TUBO  !newfork: return FALSE\n");
#endif
	return FALSE;
    }
    
    retval1=0;
#if 0
    /* this works nice, almost...
     * the only problem is that it might block at TuboInput.
     * It is blocking on smblist after the password is requested
     * to list the contents of a smb share with lots of files (//sagitario/dxf)
     * ... */
    do {
    	if (FD_ISSET(newfork->tubo[1][0], &rfds)){
	    /*printf("stdout ready for reading...(%d)\n",newfork->tubo[1][0]);*/
	    retval1=TuboInput(newfork->tubo[1][0], newfork->operate_stdout);
	}
    } while (retval1);
    
    do {
    	if (FD_ISSET(newfork->tubo[2][0], &rfds)){
	    /*printf("stderr ready for reading...(%d)\n",newfork->tubo[2][0]);*/
	    retval2=TuboInput(newfork->tubo[2][0], newfork->operate_stderr);
	}
    } while (retval2);

#else
    if (FD_ISSET(newfork->tubo[1][0], &rfds)){
#ifdef HYPER_DEBUG
	    printf("TUBO  stdout ready for reading...(%d)\n",newfork->tubo[1][0]);
#endif
	    retval1=TuboInput(newfork->tubo[1][0], newfork->operate_stdout);
    }
    if (FD_ISSET(newfork->tubo[2][0], &rfds)){
#ifdef HYPER_DEBUG
	    printf("TUBO  stderr ready for reading...(%d)\n",newfork->tubo[2][0]);
#endif
	    retval2=TuboInput(newfork->tubo[2][0], newfork->operate_stderr);
    }
#endif
    return (retval1 || retval2);
	
}

static gint TuboWait(gpointer fork_object)
{
    pid_t PID;
    fork_struct *forkO;
    int status;
    void (*user_end_function) (pid_t);
    forkO = (fork_struct *) ((long)fork_object);
    user_end_function = forkO->fork_finished_function;
    PID = forkO->PID;
    

    if (forkO->input_redlight) {
	return TRUE;
    }
#ifdef HYPER_DEBUG
    printf("tubowait...\n");
#endif
#if 0
    /* this bug workaround for freeBSD <=5.2.1 is no longer necessary
     * in >= 5.3 and only seems to screw things up. We shall no
     * longer support <= 5.3 */
#ifdef	__FreeBSD__
    /* This apparently does the bug workaround for wait failure
     * on FreeBSD 5.1: */
    if (kill(PID,SIGCONT) == 0) return TRUE;
#endif
#endif

#ifdef HYPER_DEBUG
    printf("tubowait...dead...\n");
#endif

    waitpid(forkO->PID, &status, WNOHANG);
    
    if(WIFEXITED(status) || WIFSIGNALED(status))
    {
	    int i;

#ifdef HYPER_DEBUG
    printf("TUBO tubowait...sink...\n");
    printf("TUBO WIFEXITED(status)=%d WIFSIGNALED(status)=%d pid was %d\n",
	    WIFEXITED(status),WIFSIGNALED(status),forkO->PID);
    if (WIFSIGNALED(status)){
	printf("TUBO signal was %d\n",WTERMSIG(status));
    }
#endif
    /* there is a bug in g_stuff with linux 2.4.x:
     * an invalid signal number (quite random) is received 
     * by the process everytime this function is called.
     * So the smart thing to do is to check for the only signal
     * that we need to look at, SIGKILL, which is used
     * by TuboCancel to abort the process... */
    if (WIFSIGNALED(status) && WTERMSIG(status) != SIGKILL){
	return TRUE;
    }
	    
	    for (i=0;i<3;i++) if(forkO->tubo[i][2] > 0){
	      g_source_remove(forkO->tubo[i][2]);
    	      forkO->tubo[i][2]=0;
	    }
	
	watch_input((gpointer) forkO);
    
	forkO->PID = 0;
#ifdef HYPER_DEBUG
    	printf("Parent is terminating tubo\n"); 
#endif
    	TuboChupoFaros(forkO);
    	if(user_end_function) (*user_end_function) (PID);
	if (forkO->double_fork){
	  gchar *pid_file=g_strdup_printf("%s%stubopid.%d",g_get_tmp_dir(),G_DIR_SEPARATOR_S,PID);
	  unlink(pid_file);	
	  g_free(pid_file);

	}
	return FALSE;
    }
    return TRUE;
}

static gint watch_input(gpointer data)
{
    fork_struct *newfork = (fork_struct *) data;
    gboolean retval;
 	
    if (!data) return FALSE;
    if (newfork->tubo[1][0] < 0 && newfork->tubo[2][0] < 0) {
#ifdef HYPER_DEBUG
	printf("TUBO newfork->tubo[1][0] < 0 && newfork->tubo[2][0] < 0\n");
#endif
	return FALSE;
    }

    if (newfork->input_redlight) return TRUE;
    
#if 0
    if (newfork->tubo[2][2] > 0){
#ifdef HYPER_DEBUG
	printf("TUBO g_source_remove(newfork->tubo[2][2]);\n");
#endif
	g_source_remove(newfork->tubo[2][2]);
    }
    newfork->tubo[2][2]=0;
#endif
    
    newfork->input_redlight=TRUE;
#ifdef HYPER_DEBUG
printf("TUBO  \n");
#endif
 
    /* check read will return TRUE if (pipe not ready) or (pipe ready and data read)
     * and FALSE when pipe ready and error reading pipe (child done) */
    retval = check_read(data);
   
    if (!retval && newfork->tubo[0][2]>0) {
	    /* on FALSE, change failure trap to .1 secs. */
	    g_source_remove(newfork->tubo[0][2]);
	    newfork->tubo[0][2] = 0;
	    newfork->tubo[1][2]=g_timeout_add(100, (GtkFunction) TuboWait, (gpointer) (newfork));
#ifdef HYPER_DEBUG
printf("TUBO  g_source_remove(newfork->tubo[0][2]);\n");
printf("TUBO  newfork->tubo[1][2]=g_timeout_add(100\n");
#endif
    }
    
#if 0
    newfork->tubo[2][2] = 
	g_timeout_add(newfork->read_delay,(GtkFunction) watch_input, (gpointer) newfork);
#ifdef HYPER_DEBUG
printf("TUBO  g_timeout_add(newfork->read_delay\n");
#endif
#endif

    newfork->input_redlight=FALSE;
    return TRUE;
}

static void finishit(int sig)
{
    if (grandchildPID){
	kill(grandchildPID,SIGTERM);
	kill(grandchildPID,SIGKILL);
    }
}




G_MODULE_EXPORT
void *Tubo (void (*fork_function) (void *), 
	   void *fork_function_data, 
	   void (*fork_finished_function) (pid_t), 
	   int *stdinFD, 
	   int (*operate_stdout) (int, void *), 
	   int (*operate_stderr) (int, void *),
	   int read_delay,
	   gboolean double_fork)
{
    int i;
    fork_struct tmpfork, *newfork = NULL;
    
    gchar red_light[64];
    {
      gchar *rname;
      rname = g_build_filename(g_get_tmp_dir(),"tubo.XXXXXX",NULL);
      close(mkstemp(rname));
      if (strlen(rname)+1 > 64) {
	g_warning("(strlen(rname)+1 > 64) not met");
	return NULL;
      }
      strcpy(red_light,rname);
      g_free(rname);
    }

#if 0
#ifndef HAVE_SIGNAL
    struct sigaction act;
    sigemptyset(&act.sa_mask);
#ifdef SA_RESTART
    act.sa_flags = SA_RESTART;
#else
    act.sa_flags = 0;
#endif
    act.sa_handler = TuboSemaforo;
#endif
#endif
    
    if((!operate_stdout) && (!operate_stderr))
    {
	printf("TRACE: Using Tubo with NULL functions for stdout and stderr is quite useless except for debugging purposes!\n");
#ifdef NO_MAMES 
	TuboChupoFaros(newfork);	/* no mames condition */
	return NULL;
#endif
    }


    
    for(i = 0; i < 3; i++)
    {
	tmpfork.tubo[i][0] = tmpfork.tubo[i][1] = -1;
	tmpfork.tubo[i][2] = 0;
    
    
	if(pipe(tmpfork.tubo[i]) == -1)
	{
	    TuboChupoFaros(NULL);
	    return NULL;
	}
    }

    /*tmpfork.operate_stdin = stdinFD;*/
     
    tmpfork.operate_stdout = operate_stdout;
    tmpfork.operate_stderr = operate_stderr;
    tmpfork.fork_function = fork_function;
    tmpfork.fork_finished_function = fork_finished_function;
    tmpfork.double_fork = double_fork;

    
    tmpfork.PID = fork();
    if(tmpfork.PID)
    {				/* the parent */
	/* INPUT PIPES *************** */
#ifdef HYPER_DEBUG
 printf("The parent process has forked\n"); 
#endif
	usleep(50);		/* first give the child a time slot */
	newfork = (fork_struct *) malloc(sizeof(fork_struct));
	valid_fork_objects=g_list_append(valid_fork_objects,(gpointer)newfork);
	if(!newfork)
	{
	    /* red light to child */
	    perror("malloc");
#ifdef HYPER_DEBUG
 printf("The parent process is sending a red light\n"); 
#endif
	    kill(tmpfork.PID, SIGTERM);
	    TuboChupoFaros(NULL);
	    return NULL;
	}
	MEMCPY((void *)newfork, (void *)(&tmpfork), sizeof(fork_struct));

	close(newfork->tubo[0][0]);	/* not used */
	newfork->tubo[0][0] = -1;
	

	/* OUTPUT PIPES ************** */
	if(stdinFD) {
		*stdinFD = newfork->tubo[0][1];	
	}
	else {
		close(newfork->tubo[0][1]);
		newfork->tubo[0][1] = -1;
	}
	close(newfork->tubo[1][1]);	/* not used */
	close(newfork->tubo[2][1]);	/* not used */
	newfork->tubo[1][1] = -1;
	newfork->tubo[2][1] = -1;
	
	/* the timeout for failure trap: 1 secs. (seems too short)
	 * now set at 5 seconds... */
	newfork->tubo[0][2]=g_timeout_add(5000, (GtkFunction) TuboWait, (gpointer) (newfork));
	newfork->read_delay=read_delay;
	/*This makes it really slow, to force appearance of race condition problems:
	 *
	 * newfork->read_delay=1000;*/
    	
	if (newfork->read_delay < 15) newfork->read_delay = 15;
	if (operate_stdout || operate_stderr){
		newfork->tubo[2][2] = 
			g_timeout_add(newfork->read_delay,(GtkFunction) watch_input, (gpointer) newfork);
	}
	newfork->input_redlight=FALSE;
	unlink(red_light);
#ifdef DEBUG
	printf("PARENT: removed %s\n",red_light);
#endif
	usleep(50);		/* give the child a time slot again*/
#ifdef HYPER_DEBUG
 printf("The parent process is returning\n"); 
#endif
 
#ifdef HYPER_DEBUG
	/*printf ("TRACE: parent %d<->%d   %d<->%d   %d<->%d\n",
		newfork->tubo[0][0],
		newfork->tubo[0][1],
		newfork->tubo[1][0],
		newfork->tubo[1][1],
		newfork->tubo[2][0],
		newfork->tubo[2][1]);*/
#endif
	return newfork;		/* back to user's program flow */
    }
    else
    {				/* the child */
	
#ifdef HAVE_SIGACTION
	struct sigaction act;
	act.sa_handler = finishit;
	sigemptyset(&act.sa_mask);
#ifdef SA_RESTART
	act.sa_flags = SA_RESTART;
#else
	act.sa_flags = 0;
#endif
	sigaction(SIGHUP,&act,NULL);
#else
	signal(SIGHUP, finishit);
#endif
	newfork = (fork_struct *) malloc(sizeof(fork_struct));
	if(!newfork)
	{

	    _exit(1);
	}
	MEMCPY((void *)newfork, (void *)(&tmpfork), sizeof(fork_struct));

	/* INPUT PIPES */
	if(stdinFD) dup2(newfork->tubo[0][0], 0);	/* stdin */	    
	else {
	    close(newfork->tubo[0][0]);		/* stdin */
	    newfork->tubo[0][0] = -1;
	}
	
	close(newfork->tubo[1][0]);	/* not used */
	close(newfork->tubo[2][0]);	/* not used */
	/* OUTPUT PIPES */
	close(newfork->tubo[0][1]);	/* not used */
	newfork->tubo[1][0] = newfork->tubo[2][0] = newfork->tubo[0][1] = -1;
	
	if(operate_stdout)
	    dup2(newfork->tubo[1][1], 1);	/* stdout */
	else {
	    close(newfork->tubo[1][1]);	/* not used */
	    newfork->tubo[1][1] = -1;
	}
	if(operate_stdout)
	    dup2(newfork->tubo[2][1], 2);	/* stderr */
	else {
	    close(newfork->tubo[2][1]);	/* not used */
	    newfork->tubo[2][1] = -1;
	}
#ifdef HYPER_DEBUG
	/*printf ("TRACE: child %d<->%d   %d<->%d   %d<->%d\n",
		newfork->tubo[0][0],
		newfork->tubo[0][1],
		newfork->tubo[1][0],
		newfork->tubo[1][1],
		newfork->tubo[2][0],
		newfork->tubo[2][1]);*/
#endif
	
     if (double_fork)
     {
	FILE *f;
	int status;
	gchar *pid_file=g_strdup_printf("%s%stubopid.%d",g_get_tmp_dir(),G_DIR_SEPARATOR_S,getpid());
	grandchildPID = fork();
	if (grandchildPID==0) {
	    do {
		usleep(500);
	    } while (access(red_light,F_OK)==0);
	    unlink(red_light);
	    if(newfork->fork_function) (*(newfork->fork_function)) (fork_function_data);
	    /*fprintf(stderr,"TRACE: could not execute forkfunction!\n");*/
	    _exit(1);
	} 
#ifdef HYPER_DEBUG
	fprintf(stderr,"grandCHILD pid is %d\n",grandchildPID);
	
#endif

	if (grandchildPID<0) {
	    printf("TRACE: could not fork!\n");
	    _exit(1);
	} 

	f=fopen(pid_file,"w");
	if (f) {
	   fprintf(f,"%d\n",grandchildPID);
	   fclose(f);
	}
      

	while (wait(&status) > 0);
#ifdef HYPER_DEBUG
	fprintf(stderr,"CHILD done wait for grandchild\n");
	
#endif
	g_free(pid_file);
     }
     else /* ! double_fork */
     {
	do {
#ifdef HYPER_DEBUG
	    printf("CHILD sleep\n");
#endif
	    usleep(500);
	} while (!g_file_test(red_light,G_FILE_TEST_EXISTS)==0);
#ifdef HYPER_DEBUG
	    printf("CHILD OK\n");
#endif
	
	if(newfork->fork_function) (*(newfork->fork_function)) (fork_function_data);
#ifdef HYPER_DEBUG
	printf("CHILD could not execute forkfunction!\n");
#endif
   
     }
	/*TuboChupoFaros(newfork);*/
	fflush(NULL);
	sleep(1);
	_exit(1);
    } /* end child */

}

G_MODULE_EXPORT
int TuboWrite (void *forkObject, void *data, int n)
{
    /* if n!=0 --> binary data */
    int size;
    fork_struct *forkO;
    forkO = (fork_struct *) forkObject;
    if(!forkO)
	return 0;
    if(!data)
	return 0;
    if(!n)
	size = n;
    else
	size = strlen((char *)data);
    write(forkO->tubo[0][1], data, size);	/* watchout for closed pipes */
    return 1;
}


G_MODULE_EXPORT
void *TuboCancel (void *forkObject, void (*cleanup) (void))
{
    fork_struct *forkO;
    forkO = (fork_struct *) forkObject;
    if(!forkO || g_list_find(valid_fork_objects,(gconstpointer)forkO)==NULL)
    {
#ifdef DEBUG
    printf("ignoring TuboCancel() call on invalid fork object.\n");
#endif
	return NULL;
    }
#ifdef HYPER_DEBUG
    printf("cancelling fork object\n");
#endif
    

    if(forkO->PID) {
	
	  /* these aren't killing in linux 2.4:
	  kill(forkO->PID, SIGTERM);*/	
	  kill(forkO->PID, SIGHUP);
	  sleep(1);
	  kill(forkO->PID, SIGKILL);
	  usleep(250);
    }
    if(cleanup)	(*cleanup) ();

    /*note: fork object freed by TuboWaitDone() function */
    return NULL;
}

G_MODULE_EXPORT
pid_t TuboPID (gpointer fork_object)
{
    fork_struct *forkO;
    if (!fork_object || g_list_find(valid_fork_objects,(gconstpointer)fork_object)==NULL)
    {
#ifdef HYPER_DEBUG
    printf("ignoring TuboPID() call on invalid fork object.\n");
#endif
	return 0;
    }
    forkO = (fork_struct *) ((long)fork_object);
    return (forkO->PID);
}
