/* basic_row_c */
/*
 * Copyright (C) 2002-4 Edscott Wilson Garcia
 * EMail: edscott@xfce.org
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#include <sys/stat.h>
#include <stdlib.h>
#include <string.h>
#include <grp.h>
#include <pwd.h>
#include <errno.h>
#include <gtk/gtk.h>


#include "constants.h"
#include "types.h"

#include "primary.h"
#include "treeview.h"


/* static functions: */
#include "treeview_row.i"

/************************************************************************/
/*****************    dummy routines: insert, erase, reset    ***********/
/************************************************************************/
/** 
 * insert_dummy_row:
 *
 * inserts a dummy row so that the parent row changes from leaf to node.
 * Takes either parent's GtkTreeIter or GtkTreeRowReference as a parameter
 * (the other can be NULL). If parent's record_entry_t is known, then it 
 * can be sent as a parameter to avoid looking for it. 
 *
 * icon_id: identifier for icon from mime theme.
 * text: label to print, can be NULL, in which case "" will be printed. 
 *
 **/
G_MODULE_EXPORT
void 
insert_dummy_row (	GtkTreeModel *treemodel, 
			GtkTreeIter *parent_iter,
			GtkTreeRowReference *parent_reference,
			record_entry_t *parent_en,
			gchar *icon_id, 
			gchar *text)
{
    GdkPixbuf *dummy_icon=NULL;
    record_entry_t *dummy_en;
    GtkTreeIter dummy_iter;
    
    if (!parent_iter) {
	if ((parent_iter=get_iter_from_reference(treemodel,parent_reference))==NULL)
	    return;
    }

/* only insert dummies in leaf nodes... */
    /*if(gtk_tree_model_iter_children(treemodel, &dummy_iter, parent_iter)) {
	return;
    }*/

    if (!parent_en) {
	gtk_tree_model_get(treemodel, parent_iter, ENTRY_COLUMN, &parent_en, -1);   
	if (!parent_en) return;
    }

    dummy_en = mk_entry(parent_en->type);
    SET_DUMMY_TYPE(dummy_en->type);
    
    gtk_tree_store_prepend((GtkTreeStore *) treemodel, &dummy_iter, parent_iter);
    if (!text) text=""; 
    if (strcmp(text,"..")==0) dummy_en->path=g_strdup("..");
    if (icon_id) dummy_icon = icon_tell(&(xffm_details->arbol->widgets),0, icon_id);
    blank_column_values(treemodel, &dummy_iter);
    gtk_tree_store_set((GtkTreeStore *) treemodel, &dummy_iter, 
	    NAME_COLUMN, g_strdup(my_utf_string(text)), 
	    STYLE_COLUMN, PANGO_STYLE_ITALIC, 
	    ENTRY_COLUMN, dummy_en, 
	    PIXBUF_COLUMN, dummy_icon,
	    -1);
    if (dummy_icon) g_object_unref (G_OBJECT (dummy_icon));
}

/** 
 * erase_dummy_row:
 *
 * erases any dummy row found in node. Send either parent's GtkTreeIter
 * or parent's GtkTreeRowReference. The other should be NULL.
 *
 **/
G_MODULE_EXPORT
void 
erase_dummy_row (	GtkTreeModel *treemodel, 
			GtkTreeIter * parent_iter,
			GtkTreeRowReference *parent_reference)
{
    GtkTreeIter dummy_iter;
    record_entry_t *en;
    if (!parent_iter) {
	if ((parent_iter=get_iter_from_reference(treemodel,parent_reference))==NULL)
	    return;
    }
    if(gtk_tree_model_iter_children(treemodel, &dummy_iter, parent_iter))
    {
      do {	
        gtk_tree_model_get(treemodel, &dummy_iter, ENTRY_COLUMN, &en, -1);
	if(IS_DUMMY_TYPE(en->type))
	{
	    remove_row(treemodel, &dummy_iter,NULL,en);
	     break; /* To erase the first dummy row only */
	}
      } while(gtk_tree_model_iter_next(treemodel, &dummy_iter));
    }
    return;
}


/** 
 * reset_dummy_row:
 *
 * resets dummy icon and text. Takes either the parent's GtkTreeIter
 * or GtkTreeRowReference. If parent's record_entry_t is known, then it 
 * can be sent as a parameter to avoid looking for it.
 *
 **/

G_MODULE_EXPORT
void 
reset_dummy_row (	GtkTreeModel *treemodel, 
			GtkTreeIter *parent_iter,
			GtkTreeRowReference *parent_reference,
			record_entry_t *parent_en,
			gchar *icon_id, 
			gchar *text)
{
    GdkPixbuf *Icon=NULL;
    GtkTreeIter dummy_iter;
    record_entry_t *dummy_en;
    if (!parent_iter) {
	if ((parent_iter=get_iter_from_reference(treemodel,parent_reference))==NULL)
	    return;
    }
    if(!gtk_tree_model_iter_children(treemodel, &dummy_iter, parent_iter)){
	return;
    }

    if (!parent_en) {
	gtk_tree_model_get(treemodel, parent_iter, ENTRY_COLUMN, &parent_en, -1);
	if (!parent_en) return;
    }
    if (icon_id) Icon = icon_tell(&(xffm_details->arbol->widgets),0, icon_id);
    if (!text) text="";
    do {
	gtk_tree_model_get(treemodel, &dummy_iter, ENTRY_COLUMN, &dummy_en, -1);
	if (!dummy_en) continue;
	if(IS_DUMMY_TYPE(dummy_en->type)){
	    gchar *old_text;
	    gtk_tree_model_get (treemodel, &dummy_iter, NAME_COLUMN, &old_text, -1);
	    g_free(old_text);
	    gtk_tree_store_set((GtkTreeStore *) treemodel, &dummy_iter, 
			    PIXBUF_COLUMN, Icon, 
			    NAME_COLUMN, g_strdup(my_utf_string(text)), -1);
	    if (Icon) g_object_unref (G_OBJECT (Icon));
	}
    } while(gtk_tree_model_iter_next(treemodel, &dummy_iter));
}

/***************************************************************************/
/******************     Removing rows: remove, prune, clear  ***************/
/***************************************************************************/
/**
 * prune_row:
 *
 * Removes all children of the row and inserts a "dummy" child.
 * if the row has no children, then function does nothing.
 *
 * iter or reference can be NULL, but not both
 * en can be NULL, but if you know the value then
 * it is best to send it to the function to avoid 
 * a duplicate lookup.
 *
 **/
G_MODULE_EXPORT
gboolean 
prune_row (	GtkTreeModel *treemodel,
		GtkTreeIter *iter,
		GtkTreeRowReference *reference,
		record_entry_t *en)
{

    if (!iter) {
	if ((iter=get_iter_from_reference(treemodel,reference))==NULL)
	    return FALSE;
    }
    if (!en) {
	gtk_tree_model_get(treemodel, iter, ENTRY_COLUMN, &en, -1);   
	if (!en) return FALSE;
    }

    UNSET_LOADED(en->type);
    UNSET_INCOMPLETE(en->type);

    erase_children(treemodel, iter);

    return TRUE;
}

/**
 * remove_row:
 * 
 * Removes a row and frees memory for the treemodel.
 * removes either leaf or node
 * iter or reference can be NULL, but not both
 * en can be NULL, but if you know the value then
 * it is best to send it to the function to avoid 
 * a duplicate lookup.
 *
 **/

G_MODULE_EXPORT
gboolean 
remove_row (	GtkTreeModel *treemodel,
		GtkTreeIter *iter,
		GtkTreeRowReference *reference,
		record_entry_t *en)
{
    if (!iter) {
	if ((iter=get_iter_from_reference(treemodel,reference))==NULL)
	    return FALSE;
    }

    free_column_values(treemodel,iter);
    free_entry(treemodel,iter,en);
    erase_children(treemodel, iter);
    gtk_tree_store_remove((GtkTreeStore *) treemodel, iter);
    return TRUE;
}

/**
 * clear_row:
 * 
 * Clear the text entries for each column of a row, and
 * wipes out the stat, path, filter, and tag field of the
 * associated entry for that row.
 * 
 * iter or reference can be NULL, but not both
 * en can be NULL, but if you know the value then
 * it is best to send it to the function to avoid 
 * a duplicate lookup.
 *
 **/

G_MODULE_EXPORT
gboolean
clear_row (	GtkTreeModel *treemodel,
		GtkTreeIter *iter,
		GtkTreeRowReference *reference,
		record_entry_t *en)
{
    if (!iter) {
	if ((iter=get_iter_from_reference(treemodel,reference))==NULL)
	    return FALSE;
    }

    free_column_values(treemodel,iter);
    blank_column_values(treemodel,iter);
    if (!en) {
	gtk_tree_model_get(treemodel, iter, ENTRY_COLUMN, &en, -1);   
	if (!en) return FALSE;
    }
    
    if (en->st) {
        g_free(en->st);
        en->st=NULL;
    }
    if (en->tag) {
        g_free(en->tag);
        en->tag=NULL;
    }
    if (en->path) {
        g_free(en->path);
        en->path=NULL;
    }
    if (en->filter) {
        g_free(en->filter);
        en->filter=g_strdup("*");
    }
    return TRUE;
}

/***************************************************************************/
/******************     adding rows: add, reset  ***************/
/***************************************************************************/
/**
 * add_row:
 * 
 * Adds a row to the treemodel.
 * 
 * iter or reference can be NULL. If both are NULL, then
 * the row is added as a top level row.
 *
 * Child record_entry_t cannot be NULL since it will be associated
 * to the row.
 *
 * Name cannot be NULL either: to create a nameless row, use dummy functions.
 *
 * If you do not care what child iter is created, set child_iter to NULL.
 * 
 **/


G_MODULE_EXPORT
gboolean 
add_row (	GtkTreeModel *treemodel, 
		GtkTreeIter *parent_iter, 
		GtkTreeRowReference *parent_reference,
		GtkTreeIter *child_iter, 
		record_entry_t * child_en, 
		gchar *child_name)
{
    GtkTreeIter iter;
    GtkTreeIter *new_iter;
    if (!child_name){
	g_warning("child_name != NULL");
	return FALSE;
    }
    if (!child_en){
	g_warning("child_en != NULL");
	return FALSE;
    }
    if (parent_iter == NULL && parent_reference != NULL) {
	if ((parent_iter=get_iter_from_reference(treemodel,parent_reference))==NULL)
	    return FALSE;
    }
    if (child_iter) new_iter=child_iter;  else new_iter=&iter;
    gtk_tree_store_append((GtkTreeStore *) treemodel, new_iter, parent_iter);
    add_row_info(treemodel, new_iter, child_en, child_name);
    return TRUE;
}

/**
 * update_row:
 * 
 * Updates  MODE_COLUMN,
	    DATE_COLUMN,
	    GROUP_COLUMN,
	    OWNER_COLUMN,
	    SIZE_COLUMN
	    for the row, based on en->st (struct stat) information.
 * 
 * iter or reference can be NULL, but not both.
 *
 * record_entry_t can be NULL, but if it is known beforehand,
 * best to send as a parameter. 
 *
 **/


G_MODULE_EXPORT
gboolean 
update_row (		GtkTreeModel * treemodel, 
		        GtkTreeIter * iter, 
			GtkTreeRowReference *reference,
			record_entry_t * en)
{
    if (!iter) {
	if ((iter=get_iter_from_reference(treemodel,reference))==NULL)
	    return FALSE;
    }

    if (!en) {
	gtk_tree_model_get(treemodel, iter, ENTRY_COLUMN, &en, -1);   
	if (!en) return FALSE;
    }
    return update_row_full(treemodel, iter, en,FALSE);
}

G_MODULE_EXPORT
void 
add_contents_row (	GtkTreeModel *treemodel, 
			GtkTreeIter * iter, 
			xfdir_t * xfdir)
{

    record_entry_t *en;
    unsigned int smallcount = 0;
    long long unsigned tama;
    gint j = 0;

    if(!xfdir) {
	g_warning("xfdir != NULL");
	return;
    }
    if(!iter) {
	g_warning("iter != NULL");
	return;
    }
    gtk_tree_model_get(treemodel, iter, ENTRY_COLUMN, &en, -1);
    if(!en) {
	g_warning("en != NULL");
	return;
    }

    set_relative_tree_id_from_model(treemodel);
    tama = 0;
    for (j=0;j<TREECOUNT;j++) {
	gtk_widget_freeze_child_notify((GtkWidget *) xffm_details->arbol->treestuff[j].treeview);
    }
    /* do a quicksort on reload, before inserting into treeview
     * for the "unsorted" view (by name and subtype, or else the
     * selected sort method.*/
	
      TRACE("Sorting...");
      set_ascending(xffm_details->arbol->treestuff[xffm_details->arbol->relative_one].ascending);
      set_sort_column(xffm_details->arbol->treestuff[xffm_details->arbol->relative_one].sort_column);
      qsort((void *)xfdir->gl, 
	      xfdir->pathc, 
	      sizeof(dir_t), 
#ifdef __COMPAR_FN_T
		(__compar_fn_t)
#endif
	      xfdir_compare);
      TRACE("Sort done ...");

    show_stop(&(xffm_details->arbol->widgets));
    /* load to treemodel */
    TRACE("Loading...");
    for(j = 0; j < xfdir->pathc; j++)
    {
	if (!xfdir->gl[j].pathv || !xfdir->gl[j].en){
	    g_warning("!xfdir->gl[j].pathv || !xfdir->gl[j].en");
	    continue;
	}
	TRACE(" adding %s",xfdir->gl[j].pathv);	
        add_row(treemodel, iter, NULL, NULL, xfdir->gl[j].en, xfdir->gl[j].pathv);
	
	if(smallcount++ & (1 << 7))
	{
	    set_progress_generic(&(xffm_details->arbol->widgets),j, xfdir->pathc,1);
	    smallcount = 1;
	    process_pending_gtk();
	}
	if(xffm_details->arbol->widgets.stop) goto aborted;
	if(!xffm_details->arbol->widgets.window) return;
    }

    set_progress_generic(&(xffm_details->arbol->widgets),j, xfdir->pathc,1);

    SET_LOADED(en->type);
  aborted:
    if (IS_INCOMPLETE(en->type)){
	 print_status(&(xffm_details->arbol->widgets),"xfce/warning",_("Load is incomplete"),NULL);
    }
    for (j=0;j<TREECOUNT;j++) {
	gtk_widget_thaw_child_notify((GtkWidget *) xffm_details->arbol->treestuff[j].treeview);
    }
    set_icon(treemodel, iter);
    return;
}


G_MODULE_EXPORT
void
update_text_cell_for_row (	int cell_id,
				GtkTreeModel *treemodel, 
				GtkTreeIter *iter, 
				const gchar *tag)
{
      gchar *old_text;
      gtk_tree_model_get (treemodel, iter, cell_id, &old_text, -1);
      g_free(old_text);
      gtk_tree_store_set((GtkTreeStore *) treemodel, iter, cell_id, g_strdup(my_utf_string(tag)),-1);
    process_pending_gtk();

}

G_MODULE_EXPORT
void set_icon (GtkTreeModel *treemodel, GtkTreeIter * iterator)
{
    GdkPixbuf *Icon;
    record_entry_t *en;
    record_entry_t *p_en = NULL;
    GtkTreeIter parent;
    /*TRACE("one, iterator=0x%x, treemodel=0x%x",
	    (unsigned)iterator,(unsigned)treemodel);*/
    gtk_tree_model_get(treemodel, iterator, ENTRY_COLUMN, &en, -1);
    if (!en) return;   
    if(gtk_tree_model_iter_parent(treemodel, &parent, iterator))
	gtk_tree_model_get(treemodel, &parent, ENTRY_COLUMN, &p_en, -1);

    if(!en) return;
    if(IS_DUMMY_TYPE(en->type))
	return;
    /*if (en->path) printf("DBG: icon setting for %s\n",en->path); */
    set_font(treemodel,iterator); 
    Icon = resolve_icon(&(xffm_details->arbol->widgets),en, p_en,0);
    update_iconcolumn_for_row(treemodel, iterator,Icon);	
    if (Icon) g_object_unref (G_OBJECT (Icon));    
}

/* XXX probably best as inline macro here... */
G_MODULE_EXPORT
void 
update_iconcolumn_for_row (	GtkTreeModel *treemodel, 
				GtkTreeIter *iter, 
				GdkPixbuf *Icon)
{
      gtk_tree_store_set((GtkTreeStore *) treemodel, iter, PIXBUF_COLUMN, Icon,-1);
}

   
/*void */
G_MODULE_EXPORT
gboolean
increase_size (GtkTreeModel * treemodel, GtkTreePath * path, GtkTreeIter * iter, gpointer data)
{
    record_entry_t *en;
    gtk_tree_model_get(treemodel, iter, ENTRY_COLUMN, &en, -1);
    if(!en)
	return(FALSE);
    set_icon(treemodel,  iter);
    return FALSE;
}

/*void */
G_MODULE_EXPORT
gboolean
decrease_size (GtkTreeModel * treemodel, GtkTreePath * path, GtkTreeIter * iter, gpointer data)
{
    record_entry_t *en;
    gtk_tree_model_get(treemodel, iter, ENTRY_COLUMN, &en, -1);
    if (!en) return(FALSE);   
    set_icon(treemodel, iter);
    return FALSE;
}




G_MODULE_EXPORT
void update_icon (GtkTreeModel *treemodel, GtkTreeIter * iterator)
{
    GtkTreeIter parent;
    GdkPixbuf *Icon;
    record_entry_t *en;
    record_entry_t *p_en = NULL;
    gtk_tree_model_get(treemodel, iterator, ENTRY_COLUMN, &en, -1);
    if (!en) return;   
    if(gtk_tree_model_iter_parent(treemodel, &parent, iterator))
	gtk_tree_model_get(treemodel, &parent, ENTRY_COLUMN, &p_en, -1);    

    if(!xffm_details || !xffm_details->arbol->widgets.window)
	return;
    if(!en)
	return;

    if(IS_DUMMY_TYPE(en->type) || IS_XF_BLK(en->type) || IS_XF_CHR(en->type))
	return;
    if(IS_DIR(en->type))
    {
	struct stat st;
	stat(en->path, &st);
	if(st.st_mtime != en->st->st_mtime || st.st_ctime != en->st->st_ctime)
	{
	    memcpy(en->st, &st, sizeof(struct stat));
	    en->count = count_files(en->path);
	    Icon = resolve_icon(&(xffm_details->arbol->widgets),en, p_en,0);
	    /*D(printf(".");fflush(NULL);)*/
	    update_iconcolumn_for_row(treemodel, iterator,Icon);	
	    update_text_cell_for_row(SIZE_COLUMN,treemodel, iterator,sizetag((off_t)-1, en->count)); 

    	    if (Icon) g_object_unref (G_OBJECT (Icon));    
	}

    }
    /*in_pasteboard(en)==2 implies cut, ==1 implies copy */
    if(IS_CUT(en->type) || in_pasteboard(en)==2)
    {
	/*printf("dbg update icon %s\n",en->path); */
	Icon = resolve_icon(&(xffm_details->arbol->widgets),en, p_en,0);
        update_iconcolumn_for_row(treemodel, iterator,Icon);	
	if (Icon) g_object_unref (G_OBJECT (Icon));    
	if(in_pasteboard(en)==2){
	    SET_CUT(en->type);
	} else {
	    UNSET_CUT(en->type);
	}

    }
    return;
}

G_MODULE_EXPORT
void recreate_icons (GtkTreeView *treeview)
{
    GtkTreeModel *treemodel = gtk_tree_view_get_model(treeview);
    /* create new icons */
	    /* reset icons */
	    gtk_widget_freeze_child_notify((GtkWidget *) (treeview));
	    gtk_tree_model_foreach(treemodel, set_size_icons, (gpointer) (xffm_details));
	    gtk_widget_thaw_child_notify((GtkWidget *) (treeview));
}

