#include "config.h"
#include "i18n.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <fox-1.4/fx.h>
#include <fox-1.4/fxkeys.h>
#include <fox-1.4/FXPNGIcon.h>

#include "xfedefs.h"
#include "icons.h"
#include "DirListBox.h"
#include "FileList.h"
#include "Properties.h"
#include "FilePanel.h"
#include "XFileExplorer.h"
#include "InputDialog.h"
#include "HistInputDialog.h"
#include "File.h"
#include "MessageBox.h"
#include "OverwriteBox.h"


// Global Variables
extern FXMainWindow *mainWindow;
extern FXString editor;
extern FXString viewer;
extern FXString clipboard;
extern FXbool use_trashcan;
extern FXbool confirm_del;
extern char OpenHistory[OPEN_WITH_HIST_SIZE][100];
extern int OpenNum;
#if defined(linux)
extern FXStringDict* fsdevices;
extern FXStringDict* mtdevices;
#endif

FXuint clptype=0;



// Map
FXDEFMAP(FilePanel) FilePanelMap[]={
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_DIRECTORY_UP,FilePanel::onCmdDirectoryUp),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_FILTER,FilePanel::onCmdItemFilter),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_GO_HOME,FilePanel::onCmdGoHome),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_VIEW,FilePanel::onCmdEdit),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_EDIT,FilePanel::onCmdEdit),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_PROPERTIES,FilePanel::onCmdProperties),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_XTERM,FilePanel::onCmdXTerm),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_NEW_DIR,FilePanel::onCmdNewDir),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_NEW_FILE,FilePanel::onCmdNewFile),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_FILE_COPY,FilePanel::onCmdFileMan),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_COPY_CLP,FilePanel::onCmdClp),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_FILE_COPYTO,FilePanel::onCmdFileMan),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_FILE_MOVE,FilePanel::onCmdFileMan),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_FILE_RENAME,FilePanel::onCmdFileMan),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_CUT_CLP,FilePanel::onCmdClp),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_PASTE_CLP,FilePanel::onCmdPaste),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_FILE_DEL,FilePanel::onCmdFileDel),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_FILE_SYMLINK,FilePanel::onCmdFileMan),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_OPEN_WITH,FilePanel::onCmdOpenWith),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_OPEN,FilePanel::onCmdOpen),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_REFRESH,FilePanel::onRefresh),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_SHOW_BIG_ICONS,FilePanel::onCmdShow),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_SHOW_MINI_ICONS,FilePanel::onCmdShow),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_SHOW_DETAILS,FilePanel::onCmdShow),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_TOGGLE_HIDDEN,FilePanel::onCmdToggleHidden),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_TOGGLE_THUMBNAILS,FilePanel::onCmdToggleThumbnails),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_SELECT_ALL,FilePanel::onCmdSelect),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_DESELECT_ALL,FilePanel::onCmdSelect),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_SELECT_INVERSE,FilePanel::onCmdSelect),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_ADD_TO_ARCH,FilePanel::onCmdAddToArch),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_EXTRACT,FilePanel::onCmdExtract),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_EXTRACT_HERE,FilePanel::onCmdExtractHere),
                               FXMAPFUNC(SEL_RIGHTBUTTONRELEASE,FilePanel::ID_FILE_PANEL,FilePanel::onCmdPopupMenu),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_MENU,FilePanel::onCmdPopupMenu),
                               FXMAPFUNC(SEL_DOUBLECLICKED,FilePanel::ID_FILE_PANEL,FilePanel::onCmdItemDblClcked),
                               FXMAPFUNC(SEL_FOCUSIN,FilePanel::ID_FILE_PANEL,FilePanel::onCmdItemClcked),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_STATUS,FilePanel::onUpdStatus),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_LABEL,FilePanel::onUpdLabel),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_DIRECTORY_UP,FilePanel::onUpdUp),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_COPY_CLP,FilePanel::onUpdMenu),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_CUT_CLP,FilePanel::onUpdMenu),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_PASTE_CLP,FilePanel::onUpdPaste),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_PROPERTIES,FilePanel::onUpdMenu),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_FILE_DEL,FilePanel::onUpdMenu),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_FILE_MOVE,FilePanel::onUpdMenu),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_FILE_COPYTO,FilePanel::onUpdMenu),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_FILE_RENAME,FilePanel::onUpdSelMult),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_FILE_SYMLINK,FilePanel::onUpdSelMult),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_EDIT,FilePanel::onUpdSelMult),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_VIEW,FilePanel::onUpdSelMult),
							   FXMAPFUNC(SEL_UPDATE,FilePanel::ID_OPEN,FilePanel::onUpdOpen),
							   FXMAPFUNC(SEL_UPDATE,FilePanel::ID_ADD_TO_ARCH,FilePanel::onUpdAddToArch),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_SHOW_BIG_ICONS,FilePanel::onUpdShow),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_SHOW_MINI_ICONS,FilePanel::onUpdShow),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_SHOW_DETAILS,FilePanel::onUpdShow),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_TOGGLE_HIDDEN,FilePanel::onUpdToggleHidden),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_TOGGLE_THUMBNAILS,FilePanel::onUpdToggleThumbnails),
#if defined(linux)
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_MOUNT,FilePanel::onCmdMount),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_UMOUNT,FilePanel::onCmdMount),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_RPM_QUERY,FilePanel::onCmdRpmQuery),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_RPM_INSTALL,FilePanel::onCmdRpmInstall),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_RPM_UNINSTALL,FilePanel::onCmdRpmUninstall),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_MOUNT,FilePanel::onUpdMount),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_UMOUNT,FilePanel::onUpdUnmount),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_RPM_QUERY,FilePanel::onUpdSelMult),
#endif
                           };

// Object implementation 
FXIMPLEMENT(FilePanel,FXVerticalFrame,FilePanelMap,ARRAYNUMBER(FilePanelMap))

// Construct File Panel
FilePanel::FilePanel(FXchar* nm,FXComposite *p, FXuint name_size, FXuint size_size, FXuint type_size, 
             FXuint modd_size, FXuint user_size, FXuint grou_size, FXuint attr_size,
			 FXuint opts,FXint x,FXint y,FXint w,FXint h):
             FXVerticalFrame(p,opts,x,y,w,h,0,0,0,0)
{
	// Create some icons
	FXPNGIcon *showhiddenicon=new FXPNGIcon(getApp(),showhidden);
	FXPNGIcon *hidehiddenicon=new FXPNGIcon(getApp(),hidehidden);
	FXPNGIcon *showthumbicon=new FXPNGIcon(getApp(),showthumb);
	FXPNGIcon *hidethumbicon=new FXPNGIcon(getApp(),hidethumb);

	// These icons are defined in the class header
    archicon=new FXPNGIcon(getApp(),arch);
    attricon=new FXPNGIcon(getApp(),attrib);
    newfoldericon=new FXPNGIcon(getApp(),newfolder);
    minidocicon=new FXPNGIcon(getApp(),minidoc);
	delicon=new FXPNGIcon(getApp(),filedelete);	
	copyicon=new FXPNGIcon(getApp(),copy_clp);	
	cuticon=new FXPNGIcon(getApp(),cut_clp);	
	pasteicon=new FXPNGIcon(getApp(),paste_clp);	
	renameicon=new FXPNGIcon(getApp(),renameit);
	moveicon=new FXPNGIcon(getApp(),moveit);
	symlinkicon=new FXPNGIcon(getApp(),minilink);
	mapicon=new FXPNGIcon(getApp(),maphost);	
	unmapicon=new FXPNGIcon(getApp(),unmaphost);	
	minixtermicon=new FXPNGIcon(getApp(),minishell);
	minirpmicon=new FXPNGIcon(getApp(),minirpm);
  	locationicon=new FXPNGIcon(getApp(),location);
  	entericon=new FXPNGIcon(getApp(),enter);
	filtericon=new FXPNGIcon(getApp(),filter);
	viewicon=new FXPNGIcon(getApp(),fileview);
	editicon=new FXPNGIcon(getApp(),filedit);
	openicon=new FXPNGIcon(getApp(),fileopen);

    FXVerticalFrame* cont=new FXVerticalFrame(this,LAYOUT_FILL_Y|LAYOUT_FILL_X|FRAME_SUNKEN,0,0,0,0, 0,0,0,0, 1,1);

    name=nm;
    label=new FXLabel(cont," ",NULL,JUSTIFY_LEFT|LAYOUT_LEFT|LAYOUT_FILL_X|FRAME_RAISED);
    label->setTarget(this);
    label->setSelector(FXSEL(0,FilePanel::ID_LABEL));

    list=new FileList(cont,this,ID_FILE_PANEL,LAYOUT_FILL_X|LAYOUT_FILL_Y|ICONLIST_BIG_ICONS);
    list->setHeaderSize(0,name_size);
    list->setHeaderSize(1,size_size);
    list->setHeaderSize(2,type_size);
    list->setHeaderSize(3,modd_size);
    list->setHeaderSize(4,user_size);
    list->setHeaderSize(5,grou_size);
    list->setHeaderSize(6,attr_size);  

	statusbar=new FXHorizontalFrame(cont,JUSTIFY_LEFT|LAYOUT_FILL_X,0,0,0,0, 3,3,3,3);
    new FXToggleButton(statusbar,_("\tShow hidden files (Ctrl-F6)"),_("\tHide hidden files (Ctrl-F6)"),showhiddenicon,hidehiddenicon,this->list,
	                   FileList::ID_TOGGLE_HIDDEN,BUTTON_TOOLBAR|LAYOUT_LEFT|ICON_BEFORE_TEXT|FRAME_RAISED);

    new FXToggleButton(statusbar,_("\tShow thumbnails (Ctrl-F7)"),_("\tHide thumbnails (Ctrl-F7)"),showthumbicon,hidethumbicon,this->list,
	                   FileList::ID_TOGGLE_THUMBNAILS,BUTTON_TOOLBAR|LAYOUT_LEFT|ICON_BEFORE_TEXT|FRAME_RAISED);


    statusbar->setTarget(this);
    statusbar->setSelector(FXSEL(SEL_UPDATE,FilePanel::ID_STATUS));
    status=new FXLabel(statusbar,_("Status"),NULL,JUSTIFY_LEFT|LAYOUT_LEFT|LAYOUT_FILL_X|FRAME_SUNKEN);

	selmult=FALSE;
	current=NULL;
    
	// Home and trahscan locations
	homelocation=getenv("HOME");
    if(homelocation=="")
        homelocation=ROOTDIR;
	trashlocation=homelocation+PATHSEPSTRING+TRASHNAME;
}


// Create X window
void FilePanel::create()
{
	FXVerticalFrame::create();
}

// Destructor
FilePanel::~FilePanel()
{
    delete archicon;
    delete attricon;
    delete newfoldericon;
    delete minidocicon;
	delete delicon;	
	delete copyicon;	
	delete cuticon;	
	delete pasteicon;	
	delete renameicon;
	delete moveicon;
	delete symlinkicon;
	delete mapicon;	
	delete unmapicon;	
	delete minixtermicon;
	delete minirpmicon;
  	delete locationicon;
  	delete entericon;
	delete filtericon;
	delete viewicon;
	delete editicon;
	delete openicon;
    delete list;
    delete label;
    delete current;
    delete next;
    delete status;
	delete statusbar;
}


// FilePanel Gets Focus
void FilePanel::focus()
{
    if(this == current)
        return;
    label->setBackColor(getApp()->getBaseColor());
    label->setTextColor(FXRGB(0,0,0));
    current=this;
    next->unfocus();
    list->setFocus();
}


// FilePanel Loses Focus
void FilePanel::unfocus()
{
    label->setBackColor(FXRGB(128,128,128));
    label->setTextColor(FXRGB(255,255,255));
    current=next;
    list->handle(this,FXSEL(SEL_COMMAND,FileList::ID_DESELECT_ALL),NULL);
}


// Set Pointer to Another FilePanel
void FilePanel::Next(FilePanel* nxt)
{
    next=nxt;
}


// Update location history when changing directory (home, up or double click)
void FilePanel::updateLocation()
{
    FXString item;
	FXint i=0;
	FXComboBox* address=((XFileExplorer*) mainWindow)->address;
  	address->setNumVisible(5);
	FXint count=address->getNumItems();
    FXString p=list->getDirectory();

	// Remember latest directory in the location address
	if(!count)
	{
		count++;
		address->insertItem(0,address->getText());
	}
    while(i < count)
	{
    	item=address->getItem(i++);
    	if(::streq((const char*)&p[0],(const char*)&item[0]))
		{
			i--;
			break;
		}
	}
    if(i==count)
		address->insertItem(0,list->getDirectory());
}


// Double Click on File Item
long FilePanel::onCmdItemDblClcked(FXObject* sender,FXSelector sel, void* ptr)
{
    FXString buf, filename, filepath;

    int item= (int) ptr;
    if(item > -1)
    {
		// File name and path
		filename=list->getItemFilename(item);
		filepath=list->getItemPathname(item);

        // If directory, open the directory
        if(list->isItemDirectory(item))
        {
            // Does not have access
			if(!::isReadExecutable(filepath))
			{
            	MessageBox::error(this,MBOX_OK,_("Error"),_(" Permission to: %s denied."), filepath.text());
                return 0;
			}
            if(filename=="..")
                list->handle(this,FXSEL(SEL_COMMAND,FileList::ID_DIRECTORY_UP),NULL);
            else
                list->setDirectory(filepath);
            
			// Change directory in tree list  
			((XFileExplorer*) mainWindow)->dirpanel->list->setDirectory(filepath,TRUE);
			
			// Update location history			
			updateLocation();
        }
        else if(list->isItemFile(item))
        {
            FileAssoc *association=list->getItemAssoc(item);
            if(association)
                if(association->command.section(',',0).text() != NULL)
                {
                    buf=association->command.section(',',0)+" \""+filepath+"\" &";
                    system(buf.text());
                }
                else if(list->isItemExecutable(item)) //executable
                {
                    buf="'"+filepath+"' &";
                    system(buf.text());
                }    // If regular file return as the selected file
                else
                    current->handle(this,FXSEL(SEL_COMMAND,ID_OPEN_WITH),NULL);
            else if(list->isItemExecutable(item))//executable
            {
                buf="'"+filepath+"' &";
                system(buf.text());
            }    // If regular file return as the selected file
            else
                current->handle(this,FXSEL(SEL_COMMAND,ID_OPEN_WITH),NULL);
        }
    }
    return 1;
}


// Focus when clicked
long FilePanel::onCmdItemClcked(FXObject* sender,FXSelector sel, void* ptr)
{
    focus();
    return 1;
}


// Go to parent directory
long FilePanel::onCmdDirectoryUp(FXObject* sender,FXSelector sel, void* ptr)
{
    current->list->handle(this,FXSEL(SEL_COMMAND,FileList::ID_DIRECTORY_UP),NULL);
    current->list->setFocus();
    ((XFileExplorer*) mainWindow)->dirpanel->list->setDirectory(current->list->getDirectory(),TRUE);
	updateLocation();
    return 1;
}


// Go to home directory
long FilePanel::onCmdGoHome(FXObject* sender,FXSelector sel, void* ptr)
{
    current->list->setDirectory(homelocation);
    current->list->setFocus();
    ((XFileExplorer*) mainWindow)->dirpanel->list->setDirectory(homelocation,TRUE);
	updateLocation();
    return 1;
}


// Copy/Move/Rename/Symlink file(s)
long FilePanel::onCmdFileMan(FXObject* sender,FXSelector sel,void*)
{
    FXString target,sources,name,curdir,param;
    curdir=current->list->getDirectory();

    FXint num;	
	num=current->list->getNumSelectedItems();

	// If no selected files in the file list, use the selected folder from the tree list (if any)
	if (num==0)
	{
    	DirItem* item=(DirItem*)((XFileExplorer*) mainWindow)->dirpanel->list->getCurrentItem();
		if (!item)
			return 0;
			
		FXString filepath=((XFileExplorer*) mainWindow)->dirpanel->list->getItemPathname((TreeItem*)item);	
    	target=FXFile::directory(filepath);
		param=target+"\n"+FXStringVal(1)+"\n"+filepath;
	}
	
	// If exist selected files, use it
	else if (num>=1)
	{
		// Eventually deselect the '..' directory
		if (current->list->isItemSelected(0))
			current->list->deselectItem(0);

    	for(int u=0;u<current->list->getNumItems (); u++)
        	if (current->list->isItemSelected (u))
        	{
            	name=current->list->getItemText(u).text();
            	name=name.section('\t',0);
            	sources += curdir+PATHSEPSTRING+name+"\n";
        	}
		target=current->next->list->getDirectory();
		if(!current->next->shown() || FXSELID(sel)==ID_FILE_RENAME)
        	target=current->list->getDirectory();
		param=target+"\n"+FXStringVal(num)+"\n"+sources;
	}

    switch(FXSELID(sel))
    {
    case ID_FILE_COPY:
        sel=FXSEL(SEL_COMMAND,XFileExplorer::ID_FILE_COPY);
        break;
    case ID_FILE_COPYTO:
        sel=FXSEL(SEL_COMMAND,XFileExplorer::ID_FILE_COPYTO);
        break;
    case ID_FILE_MOVE:
        sel=FXSEL(SEL_COMMAND,XFileExplorer::ID_FILE_MOVE);
        break;
    case ID_FILE_RENAME:
        sel=FXSEL(SEL_COMMAND,XFileExplorer::ID_FILE_RENAME);
        break;
    case ID_FILE_SYMLINK:
        sel=FXSEL(SEL_COMMAND,XFileExplorer::ID_FILE_SYMLINK);
        break;
    }
    mainWindow->handle(this,sel,strdup(param.text()));
    return 1;
}


// Delete files from the file list or the tree list
long FilePanel::onCmdFileDel(FXObject*,FXSelector,void*)
{
	File *f=NULL;
    current->list->setFocus();
	FXString dir=current->list->getDirectory();
    chdir(dir.text());

    // If we don't have permission to write to the parent directory
    if(!::isWritable(dir))
    {
        MessageBox::error(this,MBOX_OK,_("Error"),_("Can't write to %s: Permission denied"),dir.text());
        return 0;
    }

	// Items number in the file list
    FXint num=current->list->getNumSelectedItems();

    // If no selected files, use the selected folder from the tree list (if any)
	if(num==0)
	{
    	DirItem* item=(DirItem*)((XFileExplorer*) mainWindow)->dirpanel->list->getCurrentItem();
		if (!item)
			return 0;
			
		FXString filepath=((XFileExplorer*) mainWindow)->dirpanel->list->getItemPathname((TreeItem*)item);

    	if(confirm_del)
    	{
        	FXPNGIcon icon(getApp(),delete_big);
			FXString message=_("Delete folder ") + filepath + " ?";
         	MessageBox box(this,_("Confirm Delete"),message,&icon,MBOX_OK_CANCEL|DECOR_TITLE|DECOR_BORDER);
 			if(box.execute(PLACEMENT_CURSOR) != MBOX_CLICKED_OK)
            	return 0;
   		}

		// Wait cursor
		getApp()->beginWaitCursor();

		// File object
   		f=new File(this,_("File Delete"),DELETE);
   		f->create();
		list->setAllowRefresh(FALSE);

		if (use_trashcan)
		{
			// Folder is in the trash can
			if (filepath.find(trashlocation.text())==0)
			{
				f->allowPdialog=FALSE;
				f->hide();
				FXPNGIcon icon(getApp(),delete_big);
				FXString message=_("Definitively delete folder ") + filepath + " ?";
         		MessageBox box(this,_("Delete from trash can"),message,&icon,MBOX_OK_CANCEL|DECOR_TITLE|DECOR_BORDER);
				if(box.execute(PLACEMENT_CURSOR) == MBOX_CLICKED_OK)
					f->remove(filepath);
				f->allowPdialog=TRUE;
			}
			else
			{
				// Create trash can if it doesn't exist
				if (!exists(trashlocation))
					::mkdir(trashlocation.text(),0755);
				// Move folder to trash can
				f->move(filepath,trashlocation+PATHSEPSTRING+stampname(filepath));
			}
		}
		else
			// Definitively remove folder
			f->remove(filepath);

		// If action is cancelled in progress dialog
		if (f->isCancelled)
		{
			f->hide();
			MessageBox::error(this,MBOX_OK,_("Error"),_("Delete file operation cancelled!"));
		}

    	// Return to parent dir in DirList		
		FXString parentdir=FXFile::directory(dir);
		((XFileExplorer*) mainWindow)->dirpanel->list->setDirectory(parentdir,TRUE);
    	chdir(parentdir.text());

		getApp()->endWaitCursor();
  		delete f;
	}
	
	// If exist selected files, use it
	else if (num>=1)
	{
		// Eventually deselect the '..' directory
		if (current->list->isItemSelected(0))
			current->list->deselectItem(0);

		if(confirm_del)
    	{
        	FXPNGIcon icon(getApp(),delete_big);
        	FXString message;
			if (num==1)
				message=_("  Delete selected item ?  ");
			else
				message=_("  Delete all selected items ?  ");
			MessageBox box(this,_("Confirm Delete"),message,&icon,MBOX_OK_CANCEL|DECOR_TITLE|DECOR_BORDER);
        	if(box.execute(PLACEMENT_CURSOR) != MBOX_CLICKED_OK)
            	return 0;
   		}

		// Wait cursor
		getApp()->beginWaitCursor();

		// File object
   		f=new File(this,_("File Delete"),DELETE);
   		f->create();
		list->setAllowRefresh(FALSE);	

		// Overwrite initialisations
		FXbool overwrite=FALSE;
		FXbool overwrite_all=FALSE;
		FXbool skip_all=FALSE;
		
		// Trash initialisation
		FXbool trashdel=FALSE;

		// Delete selected files
		FXString filename, filepath;    			
		for(int u=0;u< current->list->getNumItems (); u++)
    	{
        	if (current->list->isItemSelected (u))
        	{
            	// Get file name and path
				filename=current->list->getItemFilename(u);
				filepath=current->list->getItemPathname(u);

    			// If we don't have permission to write to the file
 				if (!::isLWritable(filepath))
    			{
					// Get process uid and file uid
					FXuint uid=getuid();
					struct stat buf;
					lstatout(filepath.text(),&buf);
					FXuint fileuid=buf.st_uid;
					
					// If it is a file and we are owner of it
					if (isFile(filepath) && fileuid==uid)
					{
						// Dialog to confirm file deletion
						f->allowPdialog=FALSE;
						f->hide();
					
						// Overwrite dialog if necessary
						if (!(overwrite_all | skip_all))
    					{
							FXString msg=_("File ") + filepath + _(" is write-protected, delete it anyway?");
       						ConfirmOverWriteBox* dlg=new ConfirmOverWriteBox(this,_("Confirm Delete"),msg);
       						FXuint answer=dlg->execute(PLACEMENT_SCREEN);
							delete dlg;
       						switch(answer)
       						{
       							case 1:
           							overwrite=TRUE;
           							break;
       							case 2:
           							overwrite_all=TRUE;
           							break;
       							case 3:
									overwrite=FALSE;
           							break;
       							case 4:
           							skip_all=TRUE;
									break;
       						}
						}
						if((overwrite | overwrite_all) & !skip_all)
						{
							// Caution!! Don't delete parent directory!!
							if (filename!="..")
							{
								if (use_trashcan)
								{
									// File or folder is in the trash can
									if (filepath.find(trashlocation.text())==0)
									{
										// Confirm delete messsage
										if (!trashdel)
										{
											f->allowPdialog=FALSE;
											f->hide();
											FXPNGIcon icon(getApp(),delete_big);
											FXString message=_("Definitively delete file ") + filepath + " ?";
         									MessageBox box(this,_("Delete from trash can"),message,&icon,MBOX_OK_CANCEL|DECOR_TITLE|DECOR_BORDER);
											if(box.execute(PLACEMENT_CURSOR) == MBOX_CLICKED_OK)
											{
												trashdel=TRUE;
												f->remove(filepath);
											}
											f->allowPdialog=TRUE;
										}
										else
											f->remove(filepath);
									}
									else
									{
										// Create trash can if it doesn't exist
										if (!exists(trashlocation))
											::mkdir(trashlocation.text(),0755);
										// Move file or folder to trash can
										f->move(filepath,trashlocation+PATHSEPSTRING+stampname(filepath));
									}
								}
								else
									// Definitively remove file or folder
									f->remove(filepath);
							}
						}
						f->allowPdialog=TRUE;
					}
					
					// If it is a directory or we are not owner of the file
					else
					{
						f->allowPdialog=FALSE;
						f->hide();
						FXPNGIcon icon(getApp(),errorIcon);
						FXString str;
						if (::isDirectory(filepath))
							str=_("Can't delete folder ") + filepath + _(" : Permission denied");
						else
							str=_("Can't delete file ") + filepath + _(" : Permission denied");
        				MessageBox box(this,_("Error"),str,&icon,MBOX_OK_CANCEL|DECOR_TITLE|DECOR_BORDER);        	
						FXuint answer=box.execute(PLACEMENT_SCREEN);
						if(answer == MBOX_CLICKED_CANCEL)
						{
							MessageBox::error(this,MBOX_OK,_("Error"),_("Delete file operation cancelled!"));
							break;
						}					
						f->allowPdialog=TRUE;
					}
    			}

				// If we have permission to write
				else
				{
					// Caution!! Don't delete parent directory!!
					if (filename!="..")
					{
						if (use_trashcan)
						{
							// File or folder is in the trashcan
							if (filepath.find(trashlocation.text())==0)
							{
								// Confirm delete message
								if (!trashdel)
								{
									f->allowPdialog=FALSE;
									f->hide();
									FXPNGIcon icon(getApp(),delete_big);
									FXString message=_("Definitively delete file ") + filepath + " ?";
         							MessageBox box(this,_("Delete from trashcan"),message,&icon,MBOX_OK_CANCEL|DECOR_TITLE|DECOR_BORDER);
									if(box.execute(PLACEMENT_CURSOR) == MBOX_CLICKED_OK)
									{
										trashdel=TRUE;
										f->remove(filepath);
									}
									f->allowPdialog=TRUE;
								}
								else
									f->remove(filepath);
							}
							else
							{
								// Create trash can if it doesn't exist
								if (!exists(trashlocation))
									::mkdir(trashlocation.text(),0755);
								// Move file or folder to trash can
								f->move(filepath,trashlocation+PATHSEPSTRING+stampname(filepath));
							}
						}
						else
							// Definitively remove file or folder
							f->remove(filepath);
					}
					// If action is cancelled in progress dialog
					if (f->isCancelled)
					{
						f->hide();
						MessageBox::error(this,MBOX_OK,_("Error"),_("Delete file operation cancelled!"));
						break;
					}
				}
        	}
    	}

		getApp()->endWaitCursor();
		delete f;
	}
	// Force FilePanel and DirPanel refresh
	list->setAllowRefresh(TRUE);	
	onRefresh(0,0,0);
	return 1;
}


// Edit file
long FilePanel::onCmdEdit(FXObject*,FXSelector s,void*)
{
    current->list->setFocus();
    chdir(current->list->getDirectory().text());
	FXString filename, filepath;

    FXint num, item;	
	num=current->list->getNumSelectedItems(&item);

	// Only process the case where one file is selected
	if (num==0 || num>1)
		return 0;

    FXString command;
    if(FXSELID(s)==ID_EDIT)
        command=editor;
    else
        command=viewer;

	FileAssoc *association=current->list->getItemAssoc(item);
	if(association)
	{
		if(FXSELID(s)==ID_EDIT)
		{
			command=association->command.section(',',2);
			if(command.length()==0)
				command=editor;
		}
		else
		{
			command=association->command.section(',',1);
			if(command.length()==0)
				command=viewer;
		}
	}

	// Construct command string
	filename=current->list->getItemFilename(item);
	filepath=current->list->getItemPathname(item);
	command+=" \""+filepath+"\"" +" &";

    // Viewer or editor command
    system(command.text());
    return 1;
}


// File or directory properties 
long FilePanel::onCmdProperties(FXObject* sender,FXSelector,void*)
{
    current->list->setFocus();
    chdir(current->list->getDirectory().text());

    FXint num, itm;	
	num=current->list->getNumSelectedItems(&itm);

	// If no selected files in the file list, use the selected folder from the tree list (if any)
	if (num==0)
	{
    	DirItem* item=(DirItem*)((XFileExplorer*) mainWindow)->dirpanel->list->getCurrentItem();
		if (!item)
			return 0;
			
		FXString filepath=((XFileExplorer*) mainWindow)->dirpanel->list->getItemPathname((TreeItem*)item);
    	FXString path=FXFile::directory(filepath);
    	FXString filename=FXFile::name(filepath);
        PropertiesBox* attrdlg=new PropertiesBox(this,filename,path);
        if(attrdlg->execute(PLACEMENT_SCREEN))
        {
            current->list->setDirectory(ROOTDIR);
            current->list->setDirectory(path);
        }
        delete attrdlg;
        return 1;
	}

	// There is one selected file in the file list
	else if (num==1)
    {
		// Eventually deselect the '..' directory
		if (current->list->isItemSelected(0))
			current->list->deselectItem(0);

    	FXString path=current->list->getDirectory();
        FXString filename=current->list->getItemText(itm);
        filename=filename.section('\t',0);
        PropertiesBox* attrdlg=new PropertiesBox(this,filename,path);
        if(attrdlg->execute(PLACEMENT_SCREEN))
        {
            current->list->setDirectory(ROOTDIR);
            current->list->setDirectory(path);
        }
        delete attrdlg;
        return 1;
    }
    
	// There are multiple selected files in the file list
	else if (num>1)
    {
    	FXString path=current->list->getDirectory();
        FXString *files=new FXString[num];

        int i= 0;
        for (int u= 0; u< current->list->getNumItems (); u++)
            if (current->list->isItemSelected(u))
            {
                files[i]=current->list->getItemText(u).section('\t',0);
                i++;
            }

        PropertiesBox* attrdlg=new PropertiesBox(this,files,num,path);
        if(attrdlg->execute(PLACEMENT_SCREEN))
        {
            current->list->setDirectory(ROOTDIR);
            current->list->setDirectory(path);
        }
        delete attrdlg;
        delete[]files;
    }
    return 0;
}


// Create new directory
long FilePanel::onCmdNewDir(FXObject*,FXSelector,void*)
{
	// Current item
	current->list->setFocus();

    FXString dirname="";	
    FXString dirpath=current->list->getDirectory();
    if(dirpath!=ROOTDIR)
        dirpath+=PATHSEPSTRING;

    InputDialog* dialog=new InputDialog(this,dirname,_("Create new folder..."),_("New Folder"));
    dialog->CursorEnd();
    if(dialog->execute(PLACEMENT_CURSOR))
    {
		dirname=dirpath+dialog->getText();
        if(dirname!=dirpath)
        {
			// Create the new dir
			FXint ret=::mkdir(dirname.text(),0755);
			if (ret==-1)
			{
         		MessageBox::error(this,MBOX_OK,_("Error"),"%s",strerror(errno));
    			return 0;
			}
		}							
    }
    delete dialog;
	
	// Force panel refresh
	onRefresh(0,0,0);
    return 1;
}


// Create new file
long FilePanel::onCmdNewFile(FXObject*,FXSelector,void*)
{
	// Current item
    current->list->setFocus();

    FXString filename="";	
    FXString filepath=current->list->getDirectory();
    if(filepath!=ROOTDIR)
        filepath+=PATHSEPSTRING;

    InputDialog* dialog=new InputDialog(this,filename,_("Create new file..."),_("New File"));
    dialog->CursorEnd();
    if(dialog->execute(PLACEMENT_CURSOR))
    {
		filename=filepath+dialog->getText();
        if(filename!=filepath)
        {
			// Test some error conditions
			if (::exists(filename))
        	{
				MessageBox::error(this,MBOX_OK,_("Error"),_("File or folder %s already exists"), filename.text());
        		return 0;
			}
			// Create the new file
			FILE *file=fopen(filename.text(),"w+");
			if (file==NULL)
			{
         		MessageBox::error(this,MBOX_OK,_("Error"),"%s",strerror(errno));
    			return 0;
			}
			fclose(file);								
		}							
    }
    delete dialog;
	
	// Force panel refresh
	onRefresh(0,0,0);
    return 1;
}

// Paste file(s)
long FilePanel::onCmdPaste(FXObject*,FXSelector sel,void*)
{
    if(clipboard.empty())
        return 1;

    FXString param=current->list->getDirectory()+"\n"+clipboard;
    switch(clptype)
    {
    case COPY_CLP:
        sel=FXSEL(SEL_COMMAND,XFileExplorer::ID_FILE_COPY);
        break;
    case CUT_CLP:
        sel=FXSEL(SEL_COMMAND,XFileExplorer::ID_FILE_CUT);
        break;
    }
    mainWindow->handle(this,sel,strdup(param.text()));
    
	// Force panel refresh
	onRefresh(0,0,0);
    return 1;
}


// Clipboard
long FilePanel::onCmdClp(FXObject*,FXSelector sel,void*)
{
    FXString name,curdir;

	// Empty clipboard
	clipboard="";

    // Clipboard type
	if (FXSELID(sel)==ID_CUT_CLP)
		clptype=CUT_CLP;
	else
		clptype=COPY_CLP;

	// Items number in the file list
    FXint num=current->list->getNumSelectedItems();
			
    // If no selected files, use the selected folder from the tree list (if any)
	if (num==0)
	{
    	DirItem* item=(DirItem*)((XFileExplorer*) mainWindow)->dirpanel->list->getCurrentItem();
		if (!item)
			return 0;
			
		FXString pathname=((XFileExplorer*) mainWindow)->dirpanel->list->getItemPathname((TreeItem*)item);
		clipboard=FXStringVal(1)+"\n"+pathname;
	}

	// If exist selected files, use it
	else if (num>=1)
	{
		// Eventually deselect the '..' directory
		if (current->list->isItemSelected(0))
			current->list->deselectItem(0);

    	curdir=current->list->getDirectory();

    	for (int u= 0; u< current->list->getNumItems (); u++)
        	if (current->list->isItemSelected(u))
        	{
            	name=current->list->getItemText(u).text();
            	name=name.section('\t',0);
            	if (curdir==ROOTDIR)
					clipboard += curdir+name+"\n";
				else
					clipboard += curdir+PATHSEPSTRING+name+"\n";
        	}
    	clipboard=FXStringVal(num)+"\n"+clipboard;
	}
    return 1;
}


// Open
long FilePanel::onCmdOpen(FXObject*,FXSelector,void*)
{
    FXString pathname, command;
	FileAssoc* association;
	   
	current->list->setFocus();
    chdir(current->list->getDirectory().text());

	if (current->list->getNumSelectedItems()==0)
		return 0;
    
	for (int u= 0; u< current->list->getNumItems (); u++)
	{
		if (current->list->isItemSelected(u))
		{
			pathname=current->list->getItemPathname(u);
			association=current->list->getItemAssoc(u);
			
			// If there is an association
			if(association)
			{
                // Use it to open the file
				if(association->command.section(',',0).text() != NULL)
                {
                    command=association->command.section(',',0)+" \""+pathname;
                    command+="\" &";
                    system(command.text());
                }
                
				// or execute it
				else if(current->list->isItemExecutable(u))
                {
                    command="'";
                    command+=pathname;
                    command+="' &";
                    system(command.text());
                }
                
				// or call the "Open with..." dialog
				else
                    current->handle(this,FXSEL(SEL_COMMAND,ID_OPEN_WITH),NULL);
			}
            
			// If no association but executable
			else if(current->list->isItemExecutable(u))
            {
                command="'";
                command+=pathname;
                command+="' &";
                system(command.text());
            }
			
			// Other cases
            else
                current->handle(this,FXSEL(SEL_COMMAND,ID_OPEN_WITH),NULL);
		}
	}
    return 1;
}

// Open with
long FilePanel::onCmdOpenWith(FXObject*,FXSelector,void*)
{
	char **str=NULL;
    current->list->setFocus();
    int i;
    chdir(current->list->getDirectory().text());

	if (current->list->getNumSelectedItems()==0)
		return 0;

    HistInputDialog* dialog;
    FXString command="";
    
	// Dialog with history list and associate checkbox
	dialog=new HistInputDialog(this,command,_("Open selected file(s) with :"),_("Open With"), "", NULL, 1, _("A&ssociate"));
    for(int i=0;i<OpenNum;i++)
        dialog->appendItem(OpenHistory[i]);
    dialog->CursorEnd();
    if(dialog->execute())
    {
        command=dialog->getText();

        for (int u= 0; u< current->list->getNumItems (); u++)
            if (current->list->isItemSelected(u))
            {
                // Handles "associate" checkbox for "open with..." dialog
                if (dialog->getOption())
                {
                    FXString filename=current->list->getItemFilename(u);
					FXString ext=FXFile::extension(filename).text();
                    if (ext == "") { ext=FXFile::name(filename); }

                    FileAssoc* association=current->list->getItemAssoc(u);

                    if (association)
                    {
                        // Update existing association
                        FXString oldfileassoc=getApp()->reg().readStringEntry("FILETYPES", ext.text(),"");
						oldfileassoc.remove(0, oldfileassoc.section(';',0).section(',',0).length());
                        oldfileassoc.prepend(dialog->getText());
                        getApp()->reg().writeStringEntry("FILETYPES", ext.text(), oldfileassoc.text());

						// Handle file association
						str=new char*[2];
						str[0]=new char[strlen(ext.text())+1];
						str[1]=new char[strlen(oldfileassoc.text())+1];
						strlcpy(str[0],ext.text(),ext.length()+1);
						strlcpy(str[1],oldfileassoc.text(),oldfileassoc.length()+1);
						mainWindow->handle(this, FXSEL(SEL_COMMAND,XFileExplorer::ID_FILE_ASSOC), str);
                    }
                    else
                    {
                        // New association
						FXString command=dialog->getText().append(";Document;;;;");
						getApp()->reg().writeStringEntry("FILETYPES", ext.text(), command.text());
                  
						// Handle file association
						str=new char*[2];
						str[0]=new char[strlen(ext.text())+1];
						str[1]=new char[strlen(command.text())+1];
						strlcpy(str[0],ext.text(),ext.length()+1);
						strlcpy(str[1],command.text(),command.length()+1);
						mainWindow->handle(this, FXSEL(SEL_COMMAND,XFileExplorer::ID_FILE_ASSOC),str);
                    }
                }
                // End
				
				FXString filepath=current->list->getItemPathname(u);
				command+=" ";
                command=command+"\""+filepath+"\""+" ";
            }

        command+=" &";
        system(command.text());

        OpenNum=dialog->getHistorySize();
        OpenNum++;
        command=dialog->getText();
        if(OpenNum>OPEN_WITH_HIST_SIZE)
            OpenNum=OPEN_WITH_HIST_SIZE;
        for(i=0;i<OpenNum-1;i++)
            if(::streq(OpenHistory[i],command.text()))
                break;
        if(i==OpenNum-1)
        {
            strlcpy(OpenHistory[0],command.text(),command.length()+1);
            for(i=1;i<OpenNum;i++)
                strlcpy(OpenHistory[i],dialog->getHistoryItem(i-1).text(),dialog->getHistoryItem(i-1).length()+1);
        }
        else
            OpenNum--;
    }
    delete dialog;
    
	// Force panel refresh
	onRefresh(0,0,0);
    return 1;
}


long FilePanel::onCmdItemFilter(FXObject*,FXSelector,void*)
{
    current->list->setFocus();
#define FILTER_HIST_SIZE 15

    static char history[FILTER_HIST_SIZE][100];
    static int no=7;
    if (no==7)
	{
		strlcpy(history[0],"*",sizeof(history[0]));
    	strlcpy(history[1],"*.txt",sizeof(history[1]));
    	strlcpy(history[2],"*.c",sizeof(history[2]));
    	strlcpy(history[3],"*.h",sizeof(history[3]));
    	strlcpy(history[4],"*.html",sizeof(history[4]));
    	strlcpy(history[5],"*.pdf",sizeof(history[5]));
    	strlcpy(history[6],"*.jpg",sizeof(history[6]));
	}
    int i;
    HistInputDialog* dialog;
    FXString pat=list->getPattern();
    dialog=new HistInputDialog(this,pat,_("Show files :"),_("Filter"));
    dialog->CursorEnd();
    dialog->SelectAll();
    for(int i=0;i<no;i++)
        dialog->appendItem(history[i]);

    if(dialog->execute() && (pat=dialog->getText()) != "")
    {
        list->setPattern(pat);
        no=dialog->getHistorySize();
        no++;
        if(no>FILTER_HIST_SIZE)
            no=FILTER_HIST_SIZE;

        for(i=0;i<no-1;i++)
            if(::streq(history[i],pat.text()))
                break;
        if(i==no-1)
        {
            strlcpy(history[0],pat.text(),pat.length()+1);
            for(i=1;i<no;i++)
                strlcpy(history[i],dialog->getHistoryItem(i-1).text(),dialog->getHistoryItem(i-1).length()+1);
        }
        else
            no--;
    }
    delete dialog;

    return 1;
}


// File list context menu
long FilePanel::onCmdPopupMenu(FXObject* o,FXSelector s,void* p)
{
    // FilePanel focus
	focus();

	// Check if control key was pressed
	ctrlkey=FALSE;
	if (p!=NULL)
	{
		FXEvent* event=(FXEvent*)p;
		if (event->state&CONTROLMASK)
			ctrlkey=TRUE;
	}

    if(list->getNumSelectedItems()<=1)
    {
        FXint x,y;
        FXuint state;
        list->getCursorPosition(x,y,state);

        int item=list->getItemAt(x,y);

        if (list->getCurrentItem ()>= 0)
            list->deselectItem(list->getCurrentItem());

        if (item>= 0)
        {
            list->setCurrentItem(item);
            if(!list->selectItem(item));
        }
    }

	if (list->getNumSelectedItems()==1 && list->isItemSelected(0))
		ctrlkey=TRUE;
		



	// If control key was pressed, deselect all items
	if(ctrlkey)
		list->handle(o,FXSEL(SEL_COMMAND,FileList::ID_DESELECT_ALL),p);

    // Popup menu pane
	FXMenuPane menu(this);
    FXint x,y;
    FXuint state;
    getRoot()->getCursorPosition(x,y,state);

    FXint num, itm;	
	num=current->list->getNumSelectedItems(&itm);

    // No selection or control key was pressed
	if(num == 0 || current->ctrlkey)
    {
        if(!clipboard.empty())
        {
            new FXMenuCommand(&menu,_("&Paste"),NULL,current,FilePanel::ID_PASTE_CLP);
            new FXMenuSeparator(&menu);
        }
	    new FXMenuCommand(&menu,_("&Terminal"),minixtermicon,current,FilePanel::ID_XTERM);
    	new FXMenuCommand(&menu,_("New &file..."),minidocicon,current,FilePanel::ID_NEW_FILE);
    	new FXMenuCommand(&menu,_("New f&older..."),newfoldericon,current,FilePanel::ID_NEW_DIR);
        new FXMenuCommand(&menu,_("Fi&lter..."),filtericon,current,FilePanel::ID_FILTER);
        new FXMenuSeparator(&menu);
        new FXMenuCheck(&menu,_("&Hidden files"),current->list,FileList::ID_TOGGLE_HIDDEN);
        new FXMenuCheck(&menu,_("T&humbnails"),current->list,FileList::ID_TOGGLE_THUMBNAILS);
        new FXMenuSeparator(&menu);
        new FXMenuRadio(&menu,_("B&ig icons"),current->list,FXIconList::ID_SHOW_BIG_ICONS);
        new FXMenuRadio(&menu,_("S&mall icons"),current->list,FXIconList::ID_SHOW_MINI_ICONS);
        new FXMenuRadio(&menu,_("F&ull file list"),current->list,FXIconList::ID_SHOW_DETAILS);
        new FXMenuSeparator(&menu);
        new FXMenuRadio(&menu,_("&Rows"),current->list,FXIconList::ID_ARRANGE_BY_ROWS);
        new FXMenuRadio(&menu,_("&Columns"),current->list,FXIconList::ID_ARRANGE_BY_COLUMNS);
        new FXMenuSeparator(&menu);
        new FXMenuRadio(&menu,_("&Name"),current->list,FileList::ID_SORT_BY_NAME);
        new FXMenuRadio(&menu,_("&Type"),current->list,FileList::ID_SORT_BY_TYPE);
        new FXMenuRadio(&menu,_("&Size"),current->list,FileList::ID_SORT_BY_SIZE);
        new FXMenuRadio(&menu,_("Ti&me"),current->list,FileList::ID_SORT_BY_TIME);
  		new FXMenuCheck(&menu,_("I&gnore case"),current->list,FileList::ID_SORT_CASE);
    }
    // Non empty selection
	else
    {
		// Deselect the '..' item
		if (current->list->isItemSelected(0))
			current->list->deselectItem(0);

#if defined(linux)
		FXString name=current->list->getItemPathname(itm);
    	if(num==1 && fsdevices->find(name.text()))
		{
			new FXMenuCommand(&menu,_("&Mount"),mapicon,current,FilePanel::ID_MOUNT);
			new FXMenuCommand(&menu,_("Unmount"),unmapicon,current,FilePanel::ID_UMOUNT);
    		new FXMenuSeparator(&menu);
		}
#endif

        FXbool ar=FALSE;
        if(current->list->getItem (itm) && current->list->isItemFile(itm))
        {
            new FXMenuCommand(&menu,_("Open &with..."),NULL,current,FilePanel::ID_OPEN_WITH);
            new FXMenuCommand(&menu,_("&Open"),openicon,current,FilePanel::ID_OPEN);
			FXString name=current->list->getItemText(itm).section('\t',0);
			
			// Last and before last file extensions
			FXString ext1=name.rafter('.',1);
			FXString ext2=name.rafter('.',2);
			
			// Convert these extensions to lower case
			ext1.lower();			
			ext2.lower();

			// Display the extract and RPM menus according to the archive extensions
			if((num==1) && (ext2=="tar.gz" || ext2=="tar.bz2"))
			{
				ar=TRUE;
				new FXMenuCommand(&menu,_("&Extract here"),archicon,current,FilePanel::ID_EXTRACT_HERE);
				new FXMenuCommand(&menu,_("E&xtract to..."),archicon,current,FilePanel::ID_EXTRACT);
			}
			else if ((num==1) && (ext1=="gz" || ext1=="bz2" || ext1=="z"))
			{
				ar=TRUE;
				new FXMenuCommand(&menu,_("&Extract here"),archicon,current,FilePanel::ID_EXTRACT_HERE);
			}
			else if((num==1) && (ext1=="tar" || ext1=="tgz" || ext1=="tbz2" || ext1=="zip" || ext1=="lzh" || ext1=="rar"))
			{
				ar=TRUE;
				new FXMenuCommand(&menu,_("&Extract here"),archicon,current,FilePanel::ID_EXTRACT_HERE);
				new FXMenuCommand(&menu,_("E&xtract to..."),archicon,current,FilePanel::ID_EXTRACT);
			}
#if defined(linux)
			else if(num==1 && ext1=="rpm")
			{
				ar=TRUE;
				new FXMenuCommand(&menu,_("&View"),minirpmicon,current,FilePanel::ID_VIEW);
				new FXMenuCommand(&menu,_("Install/Up&grade"),minirpmicon,current,ID_RPM_INSTALL);
				new FXMenuCommand(&menu,_("Un&install"),minirpmicon,current,ID_RPM_UNINSTALL);
			}
#endif
			// Not archive nor rpm
			if(!ar)
			{
				new FXMenuCommand(&menu,_("&View"),viewicon,current,FilePanel::ID_VIEW);
				new FXMenuCommand(&menu,_("&Edit"),editicon,current,FilePanel::ID_EDIT);
			}
        }
        if(!ar)
            new FXMenuCommand(&menu,_("&Add to archive..."),archicon,current,FilePanel::ID_ADD_TO_ARCH);
#if defined(linux)
        if (!ar)
			new FXMenuCommand(&menu,_("&Rpm query"),minirpmicon,current,FilePanel::ID_RPM_QUERY);
#endif
        new FXMenuSeparator(&menu);
        new FXMenuCommand(&menu,_("&Copy"),copyicon,current,FilePanel::ID_COPY_CLP);
        new FXMenuCommand(&menu,_("C&ut"),cuticon,current,FilePanel::ID_CUT_CLP);
		new FXMenuCommand(&menu,_("&Paste"),pasteicon,current,FilePanel::ID_PASTE_CLP);
        new FXMenuSeparator(&menu);
        new FXMenuCommand(&menu,_("Rename..."),renameicon,current,FilePanel::ID_FILE_RENAME);
        new FXMenuCommand(&menu,_("&Copy &to..."),copyicon,current,FilePanel::ID_FILE_COPYTO);
        new FXMenuCommand(&menu,_("&Move..."),moveicon,current,FilePanel::ID_FILE_MOVE);
        new FXMenuCommand(&menu,_("&Symlink..."),symlinkicon,current,FilePanel::ID_FILE_SYMLINK);
        new FXMenuCommand(&menu,_("&Delete"),delicon,current,FilePanel::ID_FILE_DEL);
        new FXMenuSeparator(&menu);
        new FXMenuCommand(&menu,_("Proper&ties..."),attricon,current,FilePanel::ID_PROPERTIES);
    }
    menu.create();
    menu.popup(NULL,x,y);
	getApp()->runModalWhileShown(&menu);
 	
    return 1;
}


// Run Terminal in the selected directory
long  FilePanel::onCmdXTerm(FXObject*,FXSelector,void*)
{
    chdir(current->list->getDirectory().text());
    FXString cmd=getApp()->reg().readStringEntry("PROGS","xterm","xterm -sb");
    cmd += " &";
    system(cmd.text());
    current->list->setFocus();
    return 0;
}


// Add files or directory to an archive
long FilePanel::onCmdAddToArch(FXObject* o,FXSelector,void*)
{
    FXString name, ext1, ext2, cmd, archive="";
    static char history[ADD_TO_ARCH_HIST_SIZE][100];
    static int n=0;
    int i;
	File *f;
	
	chdir(current->list->getDirectory().text());

    if(current->list->getNumSelectedItems()==0)
        return 0;

  	// Eventually deselect the '..' directory
	if (current->list->isItemSelected(0))
		current->list->deselectItem(0);

    HistInputDialog* dialog;
    FXString archpath=current->list->getDirectory();

    dialog=new HistInputDialog(this,archive,
	      _("Create new archive: (e.g. archive.tar.gz)\n=>Supported formats : tar, tgz, tbz2, taz, gz, bz2, Z, zip, rar, lzh"),
		  _("Add To Archive"));

    for(int i=0;i<n;i++)
        dialog->appendItem(history[i]);
    dialog->CursorEnd();
    if(dialog->execute())
    {
        // Get string and preserve escape characters
		archive=archpath+PATHSEPSTRING+"\""+dialog->getText()+"\"";

        // Get extensions of the archive name
        ext1=dialog->getText().rafter('.',1);
		ext1.lower();
        ext2=dialog->getText().rafter('.',2);
		ext2.lower();
		
		// Handle different archive formats
		if (ext2=="tar.gz")
            cmd="tar -zcvf "+archive+" ";
		else if (ext2=="tar.bz2")
            cmd="tar -jcvf "+archive+" ";
		else if (ext2=="tar.z")
           	cmd="tar -Zcvf "+archive+" ";
		else if (ext1=="tar")
           	cmd="tar -cvf "+archive+" ";
		else if (ext1=="gz")
			cmd="gzip -v ";				
		else if (ext1=="tgz")
            cmd="tar -zcvf "+archive+" ";
		else if (ext1=="bz2")
			cmd="bzip2 -v ";				
		else if (ext1=="tbz2")
            cmd="tar -jcvf "+archive+" ";
		else if (ext1=="z")
            cmd="compress -v ";
		else if (ext1=="zip")
            cmd="zip -r "+archive+" ";
		else if (ext1=="rar")
            cmd="rar a "+archive+" ";
		else if (ext1=="lzh")
            cmd="lha a "+archive+" ";
		else
			cmd="tar -zcvf "+archive+" ";

        for (int u= 0; u< current->list->getNumItems(); u++)
            if (current->list->isItemSelected(u))
            {
				// Don't include '..' in the list
				name=list->getItemFilename(u);				
				if (name!="..")
				{
                	cmd+=" ";
                	cmd=cmd+"\""+name+"\"";
                	cmd+=" ";
				}
            }
		// Wait cursor
		getApp()->beginWaitCursor();

		// File object
		f=new File(this,_("Create archive"),ARCHIVE);
		f->create();

		// Create archive
		f->archive(archive,cmd);
 
		getApp()->endWaitCursor();
		delete f; 

		// Force panel refresh
    	onRefresh(0,0,0);

        // Manage dialog history size
		n=dialog->getHistorySize();
        n++;
        cmd=dialog->getText();
        if(n>ADD_TO_ARCH_HIST_SIZE)
            n=ADD_TO_ARCH_HIST_SIZE;
        for(i=0;i<n-1;i++)
            if(::streq(history[i],cmd.text()))
                break;
        if(i==n-1)
        {
            strlcpy(history[0],cmd.text(),cmd.length()+1);
            for(i=1;i<n;i++)
                strlcpy(history[i],dialog->getHistoryItem(i-1).text(),dialog->getHistoryItem(i-1).length()+1);
        }
        else
            n--;
    }
    delete dialog;
    return 1;
}


// Extract archive
long FilePanel::onCmdExtract(FXObject*,FXSelector,void*)
{
    FXString name, ext1, ext2, cmd, dir, cdir;
	File *f;

    cdir=current->list->getDirectory();
    dir=getApp()->reg().readStringEntry("HISTORY","last_extract_dir",cdir.text());

	DirListBox *dirlistbox=new DirListBox(this);
    dirlistbox->create();
    dirlistbox->setPath(dir);

    FXint num, item;	
	num=current->list->getNumSelectedItems(&item);
    if (current->list->getItem (item))
	{
        if(dirlistbox->execute())
        {
            dir=dirlistbox->getPath();
            if(isWritable(dir))
            {
                getApp()->reg().writeStringEntry("HISTORY","last_extract_dir",dir.text());
                
				// Archive name and extensions
				name=current->list->getItemText(item).text();
        		ext1=name.section('\t',0).rafter('.',1);
         		ext2=name.section('\t',0).rafter('.',2);
				ext1.lower();
				ext2.lower();
				name="\"" + cdir + PATHSEPSTRING + name.section('\t',0) + "\"";

				// Handle different archive formats
				if (ext2=="tar.gz")
            		cmd="tar -zxvf ";
				else if (ext2=="tar.bz2")
            		cmd="tar -jxvf ";
				else if (ext2=="tar.z")
           			cmd="tar -Zxvf ";
				else if (ext1=="tar")
           			cmd="tar -xvf ";
				else if (ext1=="gz")
					cmd="gunzip -v ";				
				else if (ext1=="tgz")
            		cmd="tar -zxvf ";
				else if (ext1=="bz2")
					cmd="bunzip2 -v ";				
				else if (ext1=="tbz2")
            		cmd="tar -jxvf ";
				else if (ext1=="z")
            		cmd="uncompress -v ";
				else if (ext1=="zip")
            		cmd="unzip -o ";
				else if (ext1=="rar")
            		cmd="rar x -o+ ";
				else if (ext1=="lzh")
            		cmd="lha -xf ";
				else
					cmd="tar -zxvf ";

        		// Final extract command
				cmd+=name+" ";

				// Wait cursor
				getApp()->beginWaitCursor();
					
				// File object
   				f=new File(this,_("Extract archive"),EXTRACT);
   				f->create();
				   
				// Extract archive
				f->extract(name,dir,cmd);
 
				getApp()->endWaitCursor();
				delete f; 
            }
            else
                MessageBox::error(this,MBOX_OK,_("Error"),_("Can't write to %s: Permission denied"),dir.text());
        }
	}
	delete dirlistbox;
	
	// Force panel refresh
	current->onRefresh(0,0,0);
	
    return 1;
}

// Extract archive in the current directory
long FilePanel::onCmdExtractHere(FXObject*,FXSelector,void*)
{
    FXString name, ext1, ext2, cmd, cdir;
	File *f;

    // Current directory
	cdir=current->list->getDirectory();

    FXint num, item;	
	num=current->list->getNumSelectedItems(&item);
    if (current->list->getItem (item))
	{
		if(isWritable(cdir))
		{
			// Archive name and extensions
			name=current->list->getItemText(item).text();
        	ext1=name.section('\t',0).rafter('.',1);
         	ext2=name.section('\t',0).rafter('.',2);
			ext1.lower();
			ext2.lower();
			name="\"" + cdir + PATHSEPSTRING + name.section('\t',0) + "\"";

			// Handle different archive formats
			if (ext2=="tar.gz")
            	cmd="tar -zxvf ";
			else if (ext2=="tar.bz2")
            	cmd="tar -jxvf ";
			else if (ext2=="tar.z")
           		cmd="tar -Zxvf ";
			else if (ext1=="tar")
           		cmd="tar -xvf ";
			else if (ext1=="gz")
				cmd="gunzip -v ";				
			else if (ext1=="tgz")
            	cmd="tar -zxvf ";
			else if (ext1=="bz2")
				cmd="bunzip2 -v ";				
			else if (ext1=="tbz2")
            	cmd="tar -jxvf ";
			else if (ext1=="z")
            	cmd="uncompress -v ";
			else if (ext1=="zip")
            	cmd="unzip -o ";
			else if (ext1=="rar")
            	cmd="rar x -o+ ";
			else if (ext1=="lzh")
            	cmd="lha -xf ";
			else
				cmd="tar -zxvf ";

        	// Final extract command
			cmd+=name+" ";

			// Wait cursor
			getApp()->beginWaitCursor();
					
			// File object
   			f=new File(this,_("Extract archive"),EXTRACT);
   			f->create();
				   
			// Extract archive
			f->extract(name,cdir,cmd);
 
			getApp()->endWaitCursor();
			delete f; 
		}
		else
			MessageBox::error(this,MBOX_OK,_("Error"),_("Can't write to %s: Permission denied"),cdir.text());
	}
	
	// Force panel refresh
	current->onRefresh(0,0,0);
	
    return 1;
}

#if defined(linux)
// Install/Upgrade RPM package
long FilePanel::onCmdRpmInstall(FXObject*,FXSelector,void*)
{
    FXString name, path, cmd, dir, cdir;
	File *f;

    cdir=current->list->getDirectory();

    FXint num, itm;	
	num=current->list->getNumSelectedItems(&itm);
    if (current->list->getItem (itm))
	{
        name=current->list->getItemText(itm).text();
		name=name.section('\t',0);
		path="\"" + cdir + PATHSEPSTRING + name + "\"";

		// Upgrade command
		cmd="rpm -Uvh " + path;
                   
		// Wait cursor
		getApp()->beginWaitCursor();
		
		// File object
		f=new File(this,_("RPM Install/Upgrade"),RPM_INSTALL);
		f->create();
				   
		// Install/Upgrade RPM package
		f->rpmInstall(name,cmd);

		getApp()->endWaitCursor();
		delete f; 
      }
    
	// Force panel refresh
	current->onRefresh(0,0,0);
	
	return 1;
}

// Uninstall RPM package based on its name (package version is ignored)
long FilePanel::onCmdRpmUninstall(FXObject*,FXSelector,void*)
{
    FXString name, cmd, dir, cdir;
	File *f;

    cdir=current->list->getDirectory();

    FXint num, itm;	
	num=current->list->getNumSelectedItems(&itm);
    if (current->list->getItem (itm))
	{
        name=current->list->getItemText(itm).text();
		name=name.section('\t',0);
		name=name.section('-',0);

		// Uninstall command
		cmd="rpm -e " + name;
                   
		// Wait cursor
		getApp()->beginWaitCursor();

		// File object
		f=new File(this,_("RPM Uninstall"),RPM_UNINSTALL);
		f->create();

		// Uninstall RPM package
		f->rpmUninstall(name,cmd);

		getApp()->endWaitCursor();
		delete f; 
      }
    
	// Force panel refresh
	current->onRefresh(0,0,0);
	
	return 1;
}

#endif


// FilePanel and DirPanel refresh
long FilePanel::onRefresh(FXObject*,FXSelector,void*)
{
    current->list->setFocus();
    FXString dir=current->list->getDirectory();
    current->list->setDirectory(ROOTDIR);
    current->list->setDirectory(dir);
    return 1;
}


// Handle item selection
long FilePanel::onCmdSelect(FXObject* sender,FXSelector sel,void* ptr)
{
    switch(FXSELID(sel))
    {
    case ID_SELECT_ALL:
        current->list->handle(sender,FXSEL(SEL_COMMAND,FileList::ID_SELECT_ALL),ptr);
        return 1;
    case ID_DESELECT_ALL:
        current->list->handle(sender,FXSEL(SEL_COMMAND,FileList::ID_DESELECT_ALL),ptr);
        return 1;
    case ID_SELECT_INVERSE:
        current->list->handle(sender,FXSEL(SEL_COMMAND,FileList::ID_SELECT_INVERSE),ptr);
        return 1;
    }
    return 1;
}



// Handle show commands
long FilePanel::onCmdShow(FXObject* sender,FXSelector sel,void* ptr)
{
    switch(FXSELID(sel))
    {
    case ID_SHOW_BIG_ICONS:
        current->list->handle(sender,FXSEL(SEL_COMMAND,FileList::ID_SHOW_BIG_ICONS),ptr);
        break;
    case ID_SHOW_MINI_ICONS:
        current->list->handle(sender,FXSEL(SEL_COMMAND,FileList::ID_SHOW_MINI_ICONS),ptr);
        break;
    case ID_SHOW_DETAILS:
        current->list->handle(sender,FXSEL(SEL_COMMAND,FileList::ID_SHOW_DETAILS),ptr);
        break;
    }
    return 1;
}

// Update show commands
long FilePanel::onUpdShow(FXObject* sender,FXSelector sel,void* ptr)
{	
    FXuint msg=FXWindow::ID_UNCHECK;
    FXuint style=current->list->getListStyle() & ~ICONLIST_COLUMNS;
		
    switch(FXSELID(sel))
    {
    case ID_SHOW_BIG_ICONS:
        if(style == ICONLIST_BIG_ICONS)
            msg=FXWindow::ID_CHECK;
        break;
    case ID_SHOW_MINI_ICONS:
        if(style == ICONLIST_MINI_ICONS)
            msg=FXWindow::ID_CHECK;
        break;
    case ID_SHOW_DETAILS:
        if(style == ICONLIST_DETAILED)
            msg=FXWindow::ID_CHECK;
        break;
    }
    sender->handle(this,FXSEL(SEL_COMMAND,msg),ptr);
    return 1;
}

// Handle toggle hidden command
long FilePanel::onCmdToggleHidden(FXObject* sender,FXSelector sel,void* ptr)
{
	current->list->handle(sender,FXSEL(SEL_COMMAND,FileList::ID_TOGGLE_HIDDEN),ptr);
    return 1;
}

// Update toggle hidden command
long FilePanel::onUpdToggleHidden(FXObject* sender,FXSelector sel,void* ptr)
{	
    FXuint msg=FXWindow::ID_UNCHECK;
	FXbool hidden=current->list->showHiddenFiles();
		
	if(hidden == FALSE)
		msg=FXWindow::ID_CHECK;
    sender->handle(this,FXSEL(SEL_COMMAND,msg),ptr);
    return 1;
}

// Handle toggle thumbnails command
long FilePanel::onCmdToggleThumbnails(FXObject* sender,FXSelector sel,void* ptr)
{
	current->list->handle(sender,FXSEL(SEL_COMMAND,FileList::ID_TOGGLE_THUMBNAILS),ptr);
    return 1;
}

// Update toggle hidden command
long FilePanel::onUpdToggleThumbnails(FXObject* sender,FXSelector sel,void* ptr)
{	
    FXuint msg=FXWindow::ID_UNCHECK;
	FXbool showthumb=current->list->showThumbnails();
		
	if(showthumb == FALSE)
		msg=FXWindow::ID_CHECK;
    sender->handle(this,FXSEL(SEL_COMMAND,msg),ptr);
    return 1;
}


#if defined(linux)
// Mount/Unmount file systems
long FilePanel::onCmdMount(FXObject*,FXSelector sel,void*)
{
    FXString cmd, msg, text;
	unsigned int op;
	File *f;
	FXString dir;
	
	current->list->setFocus();

	// Use the selected directory in FilePanel if any
	// or use the selected directory in DirPanel
	if (current->list->getNumSelectedItems()==0)
		dir=current->list->getDirectory();
	else
	{
		for (int u= 0; u< current->list->getNumItems (); u++)
		{
			if (current->list->isItemSelected(u))
				dir=current->list->getItemPathname(u);
		}
	}		

	// If symbolic link, read the linked directory
	if (::isLink(dir))
		dir=readlink(dir);

    if(FXSELID(sel)==ID_MOUNT)
    {
        op=MOUNT;
		msg=_("Mount");
        cmd="mount ";
    }
    else
    {
        op=UNMOUNT;
		msg=_("Unmount");
        cmd="umount ";
    }
    cmd+=dir;
    cmd+=" 2>&1";
    chdir(ROOTDIR);

	// Wait cursor
	getApp()->beginWaitCursor();

	// File object
	text=msg + _(" file system...");
	f=new File(this,text.text(),op);
	f->create();
				   
	// Mount/unmount file system
	text=msg + _(" the directory :");
	f->mount(dir,text,cmd,op);
 
	// If action is cancelled in progress dialog
	if (f->isCancelled)
	{
		f->hide();
		text=msg + _(" operation cancelled!");
		MessageBox::error(this,MBOX_OK,_("Error"),text.text());
		delete f;
		return 0;
	}
					
	getApp()->endWaitCursor();
	delete f; 

	// Force panel refresh
    onRefresh(0,0,0);

    return 1;
}


// Update the Mount button
long FilePanel::onUpdMount(FXObject* o,FXSelector sel,void*)
{
    FXString dir;

   	FXint num=current->list->getNumSelectedItems();
	
	// Use the selected directory in FilePanel if any
	// or use the selected directory in DirPanel
	if (num==0)
		dir=current->list->getDirectory();
	else
	{
		for (int u= 0; u< current->list->getNumItems (); u++)
		{
			if (current->list->isItemSelected(u))
				dir=current->list->getItemPathname(u);
		}
	}		

	if(fsdevices->find(dir.text()) && !mtdevices->find(dir.text()) && !current->list->isItemSelected(0))
        o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
    else
        o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);

    return 1;
}


// Update the Unmount button
long FilePanel::onUpdUnmount(FXObject* o,FXSelector sel,void*)
{
    FXString dir;
	
   	FXint num=current->list->getNumSelectedItems();
	
	// Use the selected directory in FilePanel if any
	// or use the selected directory in DirPanel
	if (num==0)
		dir=current->list->getDirectory();
	else
	{
		for (int u= 0; u< current->list->getNumItems (); u++)
		{
			if (current->list->isItemSelected (u))
				dir=current->list->getItemPathname(u);
		}
	}		

	if(fsdevices->find(dir.text()) && mtdevices->find(dir.text()) && !current->list->isItemSelected(0))
        o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
    else
        o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);

    return 1;
}


// Query RPM data base
long FilePanel::onCmdRpmQuery(FXObject* o,FXSelector sel,void*)
{
	// Name of the current selected file
	FXString file=current->list->getCurrentFile();

	// Query command
	FXString cmd="rpm -qf " + file;
   	cmd+=" 2>&1";

	// Wait cursor
	getApp()->beginWaitCursor();

	// Perform the command
	FILE *pcmd=popen(cmd.text(),"r");
	if(!pcmd)
	{
		MessageBox::error(this,MBOX_OK,_("Error"),_("Failed command : %s"),cmd.text());
		return 0;
	}

	// Get command output
	char text[10000]={0};
	FXString buf;
	while(fgets(text,sizeof(text),pcmd))
		buf+=text;
	snprintf(text,sizeof(text)-1,"%s",buf.text());

	// Close the stream and display error message if any
	if(pclose(pcmd))
	{
		getApp()->endWaitCursor();
		MessageBox::error(this,MBOX_OK,_("Error"),"%s",text);
		return 0;
	}
	getApp()->endWaitCursor();
	
	// Output message
	FXString message=_("File ") + file + _(" belongs to the RPM package : ") + text;
	MessageBox::information(this,MBOX_OK,_("Information"),"%s",message.text());

   	return 1;
}
#endif


// Update the status bar
long FilePanel::onUpdStatus(FXObject* sender,FXSelector,void*)
{
    int item=-1;

    FXString str,path,name,ext,hsize,perm,usr,grp;
    path=list->getDirectory();
    FXchar size[64], buf[MAXPATHLEN+1],*p,*file;

   	FXint num=list->getNumSelectedItems();

	// To handle the update rename (ugly, I know)
	if (current==this)
	{
		if (num<=1) selmult=FALSE;
		else if (num>1) selmult=TRUE;
	}

    item=list->getCurrentItem();
    if(num>1)
    {
        unsigned long long sz=0;
        hsize=_("0 bytes");

        for (int u= 0; u< list->getNumItems (); u++)
            if (list->isItemSelected (u))
            {
				sz+=list->getItemFileSize(u);
				snprintf(size,sizeof(size)-1,"%llu",sz);
				hsize=::hSize(size);
            }
        status->setText(hsize+_(" in ")+FXStringVal(num)+_(" selected files"));
        return 0;
    }

    if(item<0)
    {
        num=list->getNumItems();
        status->setText(FXStringVal(num)+_(" Items"));
    }
    else
    {
        strlcpy(buf,list->getItemText(item).text(),list->getItemText(item).length()+1);
        file=buf;
        if((p=strchr(file,'\t')))
        {
            *p=0;
            name=file;
            file=p+1;
        }
        if((p=strchr(file,'\t')))
        {
            *p=0;
			snprintf(size,sizeof(size)-1,"%llu",list->getItemFileSize(item));			
            file=p+1;
        }
        if((p=strchr(file,'\t')))
        {
            *p=0;
            ext=file;
            file=p+1;
        }
        if((p=strchr(file,'\t')))
        {
            *p=0;
            file=p+1;
        }
        if((p=strchr(file,'\t')))
        {
            *p=0;
            usr=file;
            file=p+1;
        }
        if((p=strchr(file,'\t')))
        {
            *p=0;
            grp=file;
            perm=p+1;
        }
        if(strstr(ext.text(),_("Link")) || strstr(ext.text(),_("Broken link")))
        {
            FXint nread;

            nread=readlink((path+PATHSEPSTRING+name).text(),buf,sizeof(buf)-1);
            if(nread>0)
            {
                buf[nread]='\0';
                status->setText(name+" -> "+buf);
            }
        }
        else
        {
            hsize=::hSize(size);
            status->setText(usr+","+grp+" "+perm+" ("+hsize+") ");
        }

    }
    return 1;
}


// Update label (title) in file list
long FilePanel::onUpdLabel(FXObject*,FXSelector,void*)
{
    char directory[MAXPATHLEN+1];
    register char *dir=directory;
    strlcpy(dir,list->getDirectory().text(),list->getDirectory().length()+1);
    FXFont *normalFont=getApp()->getNormalFont();
    register int len=strlen(dir);
    register int text_width=normalFont->getTextWidth(dir,len);
    int lbl_width=label->getWidth();

    while((text_width > lbl_width) && len)
    {
        dir++;
        len--;
        dir[0]='.';
        dir[1]='.';
        dir[2]='.';
        text_width=normalFont->getTextWidth(dir,len);
    }
	FXString path=dir;
	if (list->getPattern() != "*")
		path=path+PATHSEPSTRING+list->getPattern();
    label->setText(path);
    return 1;
}


// Update the go to parent directory command
long FilePanel::onUpdUp(FXObject* o,FXSelector,void*)
{
    FXButton *button=(FXButton*)o;
    int style=button->getButtonStyle();
    if(style & BUTTON_TOOLBAR)
    {
        if(current->list->getDirectory()!=ROOTDIR)
            o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
        else
            o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);
    }
    else
        o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
    return 1;
}


// Update the paste button and menus
long FilePanel::onUpdPaste(FXObject* o,FXSelector,void*)
{
	FXint num;
	num=current->list->getNumSelectedItems();

	if (num==0 && clipboard.empty())
		o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);
	else
	{
		if(clipboard.empty() || (num==1 && current->list->isItemSelected(0)))
			o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);
		else
			o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
	}

    return 1;
}


// Update menu items and toolbar buttons that are related to file operations
long FilePanel::onUpdMenu(FXObject* o,FXSelector sel,void*)
{
   	// Menu item is disabled when nothing or only ".." is selected
	FXint num;
	num=current->list->getNumSelectedItems();
	DirPanel* dirpanel=((XFileExplorer*) mainWindow)->dirpanel;
	DirItem* item=(DirItem*)dirpanel->list->getCurrentItem();

	if ((dirpanel->shown() && item))
	{
		if (num==0)
			o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
		else if (num==1 && current->list->isItemSelected(0))
			o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);
		else
			o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
	}
	else
	{
		if (num==0)
			o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);
		else if (num==1 && current->list->isItemSelected(0))
			o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);
		else
			o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
	}

    return 1;
}


// Update file open menu
long FilePanel::onUpdOpen(FXObject* o,FXSelector,void*)
{
   	// Menu item is disabled when nothing or a directory (including "..") is selected
    FXint num, item;	
	num=current->list->getNumSelectedItems(&item);
	if (num==0)
		o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);
	else
	{
		if (current->list->getItem (item) && current->list->isItemFile(item))
			o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
		else
			o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);
	}
    return 1;
}


// Update the status of the menu items that should be disabled when selecting multiple files
long FilePanel::onUpdSelMult(FXObject* o, FXSelector sel, void*)
{
   	// Menu item is disabled when nothing is selected or multiple selection or ".." is only selected
	FXint num;
	num=current->list->getNumSelectedItems();
	DirPanel* dirpanel=((XFileExplorer*) mainWindow)->dirpanel;
	DirItem* item=(DirItem*)dirpanel->list->getCurrentItem();

	if (num==0)
	{
		if (!item || !dirpanel->shown())
			o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);
		else
    		o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
	}	
	else if (current->selmult || num==1 && current->list->isItemSelected(0))
		o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);
	else
    	o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);

	return 1;
}

// Update Add to archive menu
long FilePanel::onUpdAddToArch(FXObject* o,FXSelector,void*)
{
   	// Menu item is disabled when nothing or ".." is selected
    FXint num, item;	
	num=current->list->getNumSelectedItems(&item);
	if (num==0)
		o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);	
	else if (num==1 && current->list->isItemSelected(0))
		o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);
	else
		o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
    return 1;
}
