/* $Id: xdg-menu.c,v 1.13 2004/11/04 16:02:52 bmeurer Exp $ */
/*-
 * Copyright (c) 2004 os-cillation
 * All rights reserved.
 *
 * Written by Benedikt Meurer <bm@os-cillation.de>.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_SYS_PARAM_H
#include <sys/param.h>
#endif

#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif

#include <libxfce4util/libxfce4util.h>

#include <modules/menu/xdg-menu.h>


#define XDG_PARSER_STATE(state)   ((XdgParserState)(state))
#define XDG_PARSER(obj)           ((XdgParser *)(obj))


typedef enum   _XdgParserState XdgParserState;
typedef struct _XdgParser      XdgParser;


static void         xdg_menu_consolidate (XdgMenu *menu);
static void         xdg_menu_compress       (XdgMenu              *menu);
static gchar       *xdg_menu_get_dir_int    (const XdgMenu        *menu,
                                             const gchar          *directory);
static void         start_element_handler   (GMarkupParseContext  *context,
                                             const gchar          *element_name,
                                             const gchar         **attribute_names,
                                             const gchar         **attribute_values,
                                             gpointer              user_data,
                                             GError              **error);
static void         end_element_handler     (GMarkupParseContext  *context,
                                             const gchar          *element_name,
                                             gpointer              user_data,
                                             GError              **error);
static void         text_handler            (GMarkupParseContext  *context,
                                             const gchar          *text,
                                             gsize                 text_len,
                                             gpointer              user_data,
                                             GError             **error);
static void
xdg_menu_include_legacy (XdgMenu      *parent,
                         const gchar  *prefix,
                         const gchar  *path, gboolean toplevel);


enum _XdgParserState
{
  XDG_PARSER_START,
  XDG_PARSER_MENU,
  XDG_PARSER_APPDIR,
  XDG_PARSER_DEFAULTAPPDIRS,
  XDG_PARSER_DIRECTORYDIR,
  XDG_PARSER_DEFAULTDIRECTORYDIRS,
  XDG_PARSER_NAME,
  XDG_PARSER_DIRECTORY,
  XDG_PARSER_ONLYUNALLOCATED,
  XDG_PARSER_NOTONLYUNALLOCATED,
  XDG_PARSER_DELETED,
  XDG_PARSER_NOTDELETED,
  XDG_PARSER_INCLUDE,
  XDG_PARSER_EXCLUDE,
  XDG_PARSER_ALL,
  XDG_PARSER_FILENAME,
  XDG_PARSER_CATEGORY,
  XDG_PARSER_OR,
  XDG_PARSER_AND,
  XDG_PARSER_NOT,
  XDG_PARSER_KDELEGACYDIRS,
  XDG_PARSER_LEGACYDIR,
  XDG_PARSER_MOVE,
  XDG_PARSER_OLD,
  XDG_PARSER_NEW,
};


typedef XFCE_GENERIC_STACK(XdgParserState) StateStack;


struct _XdgParser
{
  StateStack      *stack;
  XdgMenu         *menu;
  
  /* absolute location of the menu file being parsed */
  gchar            basedir[PATH_MAX];

  /* temporary parsing strings */
  XdgPattern      *pattern;
  GString         *temp;
  GString         *move_old;
  GString         *move_new;
};


static GMarkupParser markup_parser =
{
  start_element_handler,
  end_element_handler,
  text_handler,
  NULL, 
  NULL,
};



XdgMenu*
xdg_menu_load (const gchar *filename,
               GError     **error)
{
  GMarkupParseContext *context;
  XdgParser            parser;
  gchar               *content;
  gsize                content_len;
  gchar               *tmp;

  g_return_val_if_fail (filename != NULL, NULL);

  /* initialize the parser */
  parser.menu     = NULL;
  parser.temp     = g_string_new ("");
  parser.move_old = g_string_new ("");
  parser.move_new = g_string_new ("");

  parser.stack = xfce_stack_new (StateStack);
  xfce_stack_push (parser.stack, XDG_PARSER_START);

  /* determine the basedir of the menu file for relative AppDir's, etc. */
  tmp = g_path_get_dirname (filename);
  realpath (tmp, parser.basedir);
  g_free (tmp);

  if (!g_file_get_contents (filename, &content, &content_len, error))
    return NULL;

  context = g_markup_parse_context_new (&markup_parser, 0, &parser, NULL);

  if (!g_markup_parse_context_parse (context, content, content_len, error))
    goto failed;

  if (!g_markup_parse_context_end_parse (context, error))
    goto failed;

  g_markup_parse_context_free (context);
  xfce_stack_free (parser.stack);
  g_string_free (parser.temp, TRUE);
  g_free (content);

  if (parser.menu == NULL)
    {
      g_set_error (error, G_MARKUP_ERROR, G_MARKUP_ERROR_INVALID_CONTENT,
                   "No toplevel <Menu> given");
      return NULL;
    }

  xdg_menu_compress (parser.menu);
  xdg_menu_consolidate (parser.menu);

  return parser.menu;

failed:
  g_markup_parse_context_free (context);
  xfce_stack_free (parser.stack);
  if (parser.menu != NULL)
    xdg_menu_free (parser.menu);
  g_string_free (parser.move_old, TRUE);
  g_string_free (parser.move_new, TRUE);
  g_string_free (parser.temp, TRUE);
  g_free (content);
  return NULL;
}



void
xdg_menu_free (XdgMenu *menu)
{
  XdgMenu *child;
  XdgMenu *next;
  GList   *lp;

  for (child = menu->cfirst; child != NULL; child = next)
    {
      next = child->next;
      xdg_menu_free (child);
      child = next;
    }

  for (lp = menu->default_directory_dirs; lp != NULL; lp = lp->next)
    g_free (lp->data);
  g_list_free (menu->default_directory_dirs);

  if (menu->moves != NULL)
    g_hash_table_destroy (menu->moves);
  if (menu->include != NULL)
    xdg_pattern_free (menu->include);
  if (menu->exclude != NULL)
    xdg_pattern_free (menu->exclude);

  g_free (menu->directory_dir);
  g_free (menu->directory);
  g_free (menu->name);
  g_free (menu);
}



XdgMenu*
xdg_menu_query (XdgMenu *menu,
                const gchar *path)
{
  XdgMenu *child;
  gchar  name[256];
  const gchar *s;

  while (*path == '/')
    ++path;

  if (*path == '\0')
    return menu;

  s = strchr (path, '/');
  if (G_UNLIKELY (s == NULL))
    s = path + strlen (path);

  bzero (name, 256);
  memcpy (name, path, s - path);

  for (child = menu->cfirst; child != NULL; child = child->next)
    if (strcmp (child->name, name) == 0)
      return xdg_menu_query (child, s);

  return NULL;
}



gchar*
xdg_menu_get_directory (const XdgMenu *menu)
{
  if (G_LIKELY (menu->directory != NULL))
    return xdg_menu_get_dir_int (menu, menu->directory);
  else
    return NULL;
}



static gboolean
g_list_equal (GList *a, GList *b)
{
  while (a != NULL && b != NULL)
    {
      if (strcmp ((gchar *) a->data, (gchar *) b->data) != 0)
        return FALSE;

      a = a->next;
      b = b->next;
    }

  return (a != NULL && b != NULL);
}



static GList*
concat_appdirs (GList *first, GList *second)
{
  GList *fp;
  GList *sp;

  for (sp = second; sp != NULL; sp = sp->next)
    {
      for (fp = first; fp != NULL; fp = fp->next)
        if (strcmp (XDG_APP_DIR (fp->data)->path,
                    XDG_APP_DIR (sp->data)->path) == 0)
          break;

      if (fp == NULL)
        {
          first = g_list_append (first, sp->data);
        }
      else
        {
          g_free (sp->data);
        }
    }

  g_list_free (second);

  return first;
}



static void
xdg_menu_consolidate (XdgMenu *menu)
{
  XdgMenu *child;
  XdgMenu *other;
  XdgMenu *prev;
  GList   *items = NULL;
  GList   *lp;
  GList   *tp;
  gchar   *tmp;

  /* execute moves */
  if (G_UNLIKELY (menu->moves != NULL))
    {
      for (child = menu->cfirst; child != NULL; child = child->next)
        {
          tmp = g_hash_table_lookup (menu->moves, child->name);
          if (G_UNLIKELY (tmp != NULL))
            {
              g_free (child->name);
              child->name = g_strdup (tmp);
            }
        }
    }
  
  /* consolidate menus */
  for (child = menu->clast; child != NULL; child = prev)
    {
      prev = child->prev;

      for (lp = items; lp != NULL; lp = lp->next)
        if (strcmp (XDG_MENU (lp->data)->name, child->name) == 0)
          break;

      if (lp == NULL)
        {
          /* submenu not yet present, prepend it */
          items = g_list_prepend (items, child);
          child->next = child->prev = NULL;
        }
      else
        {
          other = XDG_MENU (lp->data);

          /* use child Directory* stuff if we have no valid .directory yet */
          tmp = xdg_menu_get_directory (other);
          if (G_UNLIKELY (tmp == NULL))
            {
              if (other->directory_dir == NULL)
                {
                  other->directory_dir = child->directory_dir;
                  child->directory_dir = NULL;
                }

              if (other->default_directory_dirs == NULL)
                {
                  other->default_directory_dirs = child->default_directory_dirs;
                  child->default_directory_dirs = NULL;
                }

              if (other->directory == NULL)
                {
                  other->directory = child->directory;
                  child->directory = NULL;
                }
            }
          else
            {
              g_free (tmp);
            }

          other->only_unallocated = other->only_unallocated
                                 && child->only_unallocated;
          other->deleted          = other->deleted
                                 || child->deleted;

          other->include = xdg_pattern_merge (other->include, child->include);
          other->exclude = xdg_pattern_merge (other->exclude, child->exclude);

          other->appdirs = concat_appdirs (other->appdirs, child->appdirs);
          child->appdirs = NULL;

          if (child->cfirst != NULL)
            {
              if (other->cfirst == NULL)
                {
                  other->cfirst = child->cfirst;
                  other->clast = child->clast;
                }
              else
                {
                  other->clast->next = child->cfirst;
                  child->cfirst->prev = other->clast;
                  other->clast = child->clast;
                }
            }

          /* cleanup old item */
          if (child->directory_dir != NULL)
            g_free (child->directory_dir);
          for (tp = child->default_directory_dirs; tp != NULL; tp = tp->next)
            g_free (tp->data);
          g_list_free (child->default_directory_dirs);
          g_free (child->name);
          if (child->directory != NULL)
            g_free (child->directory);
        }
    }

  menu->cfirst = menu->clast = NULL;

  for (lp = items; lp != NULL; lp = lp->next)
    {
      child = XDG_MENU (lp->data);

      if (menu->cfirst == NULL)
        {
          menu->cfirst = menu->clast = child;
          child->next = child->prev = NULL;
        }
      else
        {
          child->prev = menu->clast;
          child->next = NULL;
          menu->clast->next = child;
          menu->clast = child;
        }

      xdg_menu_consolidate (child);
    }

  g_list_free (items);
}



static void
xdg_menu_compress (XdgMenu *menu)
{
  XdgMenu *parent;
  XdgMenu *child;
  GList   *appdirs = menu->appdirs;
  GList   *lp;

  if (menu->app_dir != NULL)
    {
      appdirs = g_list_prepend (appdirs, menu->app_dir);
      menu->app_dir = NULL;
    }
  else
    {
      appdirs = g_list_concat (appdirs, menu->default_app_dirs);
      menu->default_app_dirs = NULL;
    }

  if (appdirs != NULL)
    {
      /* find the first parent that has appdirs set */
      for (parent = menu->parent; parent != NULL; parent = parent->parent)
        if (parent->appdirs != NULL)
          break;

      /* ok, we have a parent with appdirs, now check if the appdirs
         match */
      if (parent != NULL)
        {
          if (g_list_equal (appdirs, parent->appdirs))
            {
              for (lp = appdirs; lp != NULL; lp = lp->next)
                g_free (lp->data);
              g_list_free (appdirs);
              appdirs = NULL;
            }
        }
    }

  menu->appdirs = appdirs;

  /* compress child menus (2nd pass) */
  for (child = menu->cfirst; child != NULL; child = child->next)
    xdg_menu_compress (child);
}



static gchar*
xdg_menu_get_dir_int (const XdgMenu *menu,
                      const gchar   *directory)
{
  GList *lp;
  gchar *path;

  if (menu->directory_dir != NULL)
    return g_build_filename (menu->directory_dir, directory, NULL);

  if (menu->default_directory_dirs != NULL)
    {
      for (lp = menu->default_directory_dirs; lp != NULL; lp = lp->next)
        {
          path = g_build_filename (lp->data, directory, NULL);
          if (g_file_test (path, G_FILE_TEST_IS_REGULAR))
            return path;
          g_free (path);
        }
    }

  if (menu->parent != NULL)
    return xdg_menu_get_dir_int (menu->parent, directory);

  return NULL;
}



static void
start_element_handler   (GMarkupParseContext  *context,
                         const gchar          *element_name,
                         const gchar         **attribute_names,
                         const gchar         **attribute_values,
                         gpointer              user_data,
                         GError              **error)
{
  XdgPattern *pattern;
  XdgParser  *parser = XDG_PARSER (user_data);
  XdgMenu    *menu;

  switch (xfce_stack_top (parser->stack))
    {
    case XDG_PARSER_START:
      if (strcmp (element_name, "Menu") == 0)
        {
          menu = g_new0 (XdgMenu, 1);

          parser->menu = menu;

          xfce_stack_push (parser->stack, XDG_PARSER_MENU);
        }
      else
        goto unknown_element;
      break;

    case XDG_PARSER_MENU:
      if (strcmp (element_name, "Menu") == 0)
        {
          menu = g_new0 (XdgMenu, 1);
          menu->parent = parser->menu;

          parser->menu = menu;

          xfce_stack_push (parser->stack, XDG_PARSER_MENU);
        }
      else if (strcmp (element_name, "AppDir") == 0)
        {
          g_string_truncate (parser->temp, 0);

          xfce_stack_push (parser->stack, XDG_PARSER_APPDIR);
        }
      else if (strcmp (element_name, "DefaultAppDirs") == 0)
        {
          xfce_stack_push (parser->stack, XDG_PARSER_DEFAULTAPPDIRS);
        }
      else if (strcmp (element_name, "DirectoryDir") == 0)
        {
          g_string_truncate (parser->temp, 0);
          xfce_stack_push (parser->stack, XDG_PARSER_DIRECTORYDIR);
        }
      else if (strcmp (element_name, "DefaultDirectoryDirs") == 0)
        {
          xfce_stack_push (parser->stack, XDG_PARSER_DEFAULTDIRECTORYDIRS);
        }
      else if (strcmp (element_name, "Name") == 0)
        {
          g_string_truncate (parser->temp, 0);
          xfce_stack_push (parser->stack, XDG_PARSER_NAME);
        }
      else if (strcmp (element_name, "Directory") == 0)
        {
          g_string_truncate (parser->temp, 0);
          xfce_stack_push (parser->stack, XDG_PARSER_DIRECTORY);
        }
      else if (strcmp (element_name, "OnlyUnallocated") == 0)
        {
          xfce_stack_push (parser->stack, XDG_PARSER_ONLYUNALLOCATED);
        }
      else if (strcmp (element_name, "NotOnlyUnallocated") == 0)
        {
          xfce_stack_push (parser->stack, XDG_PARSER_NOTONLYUNALLOCATED);
        }
      else if (strcmp (element_name, "Deleted") == 0)
        {
          xfce_stack_push (parser->stack, XDG_PARSER_DELETED);
        }
      else if (strcmp (element_name, "NotDeleted") == 0)
        {
          xfce_stack_push (parser->stack, XDG_PARSER_NOTDELETED);
        }
      else if (strcmp (element_name, "Include") == 0)
        {
          if (G_UNLIKELY (parser->menu->include != NULL))
            xdg_pattern_free (parser->menu->include);

          parser->pattern = xdg_pattern_new (XDG_PATTERN_OR, NULL);
          parser->menu->include = parser->pattern;

          xfce_stack_push (parser->stack, XDG_PARSER_INCLUDE);
        }
      else if (strcmp (element_name, "Exclude") == 0)
        {
          if (G_UNLIKELY (parser->menu->exclude != NULL))
            xdg_pattern_free (parser->menu->exclude);

          parser->pattern = xdg_pattern_new (XDG_PATTERN_OR, NULL);
          parser->menu->exclude = parser->pattern;

          xfce_stack_push (parser->stack, XDG_PARSER_EXCLUDE);
        }
      else if (strcmp (element_name, "KDELegacyDirs") == 0)
        {
          xfce_stack_push (parser->stack, XDG_PARSER_KDELEGACYDIRS);
        }
      else if (strcmp (element_name, "LegacyDir") == 0)
        {
          g_string_truncate (parser->temp, 0);
          xfce_stack_push (parser->stack, XDG_PARSER_LEGACYDIR);
        }
      else if (strcmp (element_name, "Move") == 0)
        {
          xfce_stack_push (parser->stack, XDG_PARSER_MOVE);
        }
      else
        goto unknown_element;
      break;

    case XDG_PARSER_INCLUDE:
    case XDG_PARSER_EXCLUDE:
    case XDG_PARSER_OR:
    case XDG_PARSER_AND:
    case XDG_PARSER_NOT:
      if (strcmp (element_name, "All") == 0)
        {
          xfce_stack_push (parser->stack, XDG_PARSER_ALL);
        }
      else if (strcmp (element_name, "Filename") == 0)
        {
          g_string_truncate (parser->temp, 0);
          xfce_stack_push (parser->stack, XDG_PARSER_FILENAME);
        }
      else if (strcmp (element_name, "Category") == 0)
        {
          g_string_truncate (parser->temp, 0);
          xfce_stack_push (parser->stack, XDG_PARSER_CATEGORY);
        }
      else if (strcmp (element_name, "Or") == 0)
        {
          pattern = xdg_pattern_new (XDG_PATTERN_OR, NULL);
          xdg_pattern_append (parser->pattern, pattern);
          parser->pattern = pattern;
          xfce_stack_push (parser->stack, XDG_PARSER_OR);
        }
      else if (strcmp (element_name, "And") == 0)
        {
          pattern = xdg_pattern_new (XDG_PATTERN_AND, NULL);
          xdg_pattern_append (parser->pattern, pattern);
          parser->pattern = pattern;
          xfce_stack_push (parser->stack, XDG_PARSER_AND);
        }
      else if (strcmp (element_name, "Not") == 0)
        {
          pattern = xdg_pattern_new (XDG_PATTERN_NOT, NULL);
          xdg_pattern_append (parser->pattern, pattern);
          parser->pattern = pattern;
          xfce_stack_push (parser->stack, XDG_PARSER_NOT);
        }
      else
        goto unknown_element;
      break;

    case XDG_PARSER_MOVE:
      if (strcmp (element_name, "Old") == 0)
        {
          g_string_truncate (parser->move_old, 0);
          xfce_stack_push (parser->stack, XDG_PARSER_OLD);
        }
      else if (strcmp (element_name, "New") == 0)
        {
          g_string_truncate (parser->move_new, 0);
          xfce_stack_push (parser->stack, XDG_PARSER_NEW);
        }
      else
        goto unknown_element;
      break;

    default:
      goto unknown_element;
    }

  return;

unknown_element:
  g_set_error (error, G_MARKUP_ERROR, G_MARKUP_ERROR_UNKNOWN_ELEMENT,
               "Unknown element <%s>", element_name);
  return;
}


static void
end_element_handler (GMarkupParseContext  *context,
                     const gchar          *element_name,
                     gpointer              user_data,
                     GError              **error)
{
  XdgPattern *pattern;
  XdgAppDir  *appdir;
  XdgParser  *parser = XDG_PARSER (user_data);
  XdgMenu    *parent;
  XdgMenu    *menu = parser->menu;
  gchar     **dirs;
  guint       n;

  switch (xfce_stack_top (parser->stack))
    {
    case XDG_PARSER_START:
      g_set_error (error, G_MARKUP_ERROR, G_MARKUP_ERROR_INVALID_CONTENT,
                   "End element handler called while in root context");
      return;

    case XDG_PARSER_MENU:
      if (strcmp (element_name, "Menu") == 0)
        {
          if (G_UNLIKELY (menu->name == NULL))
            {
              g_set_error (error, G_MARKUP_ERROR, G_MARKUP_ERROR_INVALID_CONTENT,
                           "Missing <Name> for <Menu>");
              g_free (menu);
              return;
            }

          parent = menu->parent;
          if (parent != NULL)
            {
              if (parent->clast != NULL)
                {
                  menu->prev = parent->clast;
                  parent->clast->next = menu;
                }
              else
                {
                  parent->cfirst = menu;
                }
              parent->clast = menu;
              parser->menu = parent;
            }
        }
      else
        goto unknown_element;
      break;

    case XDG_PARSER_APPDIR:
      if (strcmp (element_name, "AppDir") == 0)
        {
          if (G_LIKELY (menu->app_dir == NULL))
            menu->app_dir = g_new0 (XdgAppDir, 1);

          if (*parser->temp->str == '/')
            {
              g_strlcpy (menu->app_dir->path, parser->temp->str, PATH_MAX);
            }
          else
            {
              g_strlcpy (menu->app_dir->path, parser->basedir, PATH_MAX);
              g_strlcat (menu->app_dir->path, "/", PATH_MAX);
              g_strlcat (menu->app_dir->path, parser->temp->str, PATH_MAX);
            }
        }
      else
        goto unknown_element;
      break;

    case XDG_PARSER_DEFAULTAPPDIRS:
      if (strcmp (element_name, "DefaultAppDirs") == 0)
        {
          if (G_LIKELY (menu->default_app_dirs == NULL))
            {
              dirs = xfce_resource_lookup_all (XFCE_RESOURCE_DATA,
                                               "applications/");
              for (n = 0; dirs[n] != NULL; ++n)
                {
                  appdir = g_new0 (XdgAppDir, 1);
                  g_strlcpy (appdir->path, dirs[n], PATH_MAX);
                  menu->default_app_dirs = g_list_append (menu->default_app_dirs, appdir);
                }
              g_free (dirs);
            }
        }
      else
        goto unknown_element;
      break;

    case XDG_PARSER_DIRECTORYDIR:
      if (strcmp (element_name, "DirectoryDir") == 0)
        {
          if (G_UNLIKELY (menu->directory_dir != NULL))
            g_free (menu->directory_dir);

          if (*parser->temp->str == '/')
            {
              menu->directory_dir = g_strdup (parser->temp->str);
            }
          else
            {
              menu->directory_dir = g_build_filename (parser->basedir,
                                                      parser->temp->str,
                                                      NULL);
            }
        }
      else
        goto unknown_element;
      break;

    case XDG_PARSER_DEFAULTDIRECTORYDIRS:
      if (strcmp (element_name, "DefaultDirectoryDirs") == 0)
        {
          if (G_LIKELY (menu->default_directory_dirs == NULL))
            {
              dirs = xfce_resource_lookup_all (XFCE_RESOURCE_DATA,
                                               "desktop-directories/");
              for (n = 0; dirs[n] != NULL; ++n)
                {
                  menu->default_directory_dirs = g_list_append (menu->default_directory_dirs,
                                                                dirs[n]);
                }
              g_free (dirs);
            }
        }
      else
        goto unknown_element;
      break;

    case XDG_PARSER_NAME:
      if (strcmp (element_name, "Name") == 0)
        {
          if (G_UNLIKELY (menu->name != NULL))
            g_free (menu->name);
          menu->name = g_strdup (parser->temp->str);
        }
      else
        goto unknown_element;
      break;

    case XDG_PARSER_DIRECTORY:
      if (strcmp (element_name, "Directory") == 0)
        {
          if (G_UNLIKELY (menu->directory != NULL))
            g_free (menu->directory);
          menu->directory = g_strdup (parser->temp->str);
        }
      else
        goto unknown_element;
      break;

    case XDG_PARSER_ONLYUNALLOCATED:
      if (strcmp (element_name, "OnlyUnallocated") == 0)
        menu->only_unallocated = TRUE;
      else
        goto unknown_element;
      break;

    case XDG_PARSER_NOTONLYUNALLOCATED:
      if (strcmp (element_name, "NotOnlyUnallocated") == 0)
        menu->only_unallocated = FALSE;
      else
        goto unknown_element;
      break;

    case XDG_PARSER_DELETED:
      if (strcmp (element_name, "Deleted") == 0)
        menu->deleted = TRUE;
      else
        goto unknown_element;
      break;

    case XDG_PARSER_NOTDELETED:
      if (strcmp (element_name, "NotDeleted") == 0)
        menu->deleted = FALSE;
      else
        goto unknown_element;
      break;

    case XDG_PARSER_INCLUDE:
      if (strcmp (element_name, "Include") != 0)
        goto unknown_element;
      break;

    case XDG_PARSER_EXCLUDE:
      if (strcmp (element_name, "Exclude") != 0)
        goto unknown_element;
      break;

    case XDG_PARSER_ALL:
      if (strcmp (element_name, "All") == 0)
        {
          pattern = xdg_pattern_new (XDG_PATTERN_ALL, NULL);
          xdg_pattern_append (parser->pattern, pattern);
        }
      else
        goto unknown_element;
      break;

    case XDG_PARSER_FILENAME:
      if (strcmp (element_name, "Filename") == 0)
        {
          pattern = xdg_pattern_new (XDG_PATTERN_FILENAME, parser->temp->str);
          xdg_pattern_append (parser->pattern, pattern);
        }
      else
        goto unknown_element;
      break;

    case XDG_PARSER_CATEGORY:
      if (strcmp (element_name, "Category") == 0)
        {
          pattern = xdg_pattern_new (XDG_PATTERN_CATEGORY, parser->temp->str);
          xdg_pattern_append (parser->pattern, pattern);
        }
      else
        goto unknown_element;
      break;

    case XDG_PARSER_OR:
      if (strcmp (element_name, "Or") == 0)
        parser->pattern = parser->pattern->parent;
      else
        goto unknown_element;
      break;

    case XDG_PARSER_AND:
      if (strcmp (element_name, "And") == 0)
        parser->pattern = parser->pattern->parent;
      else
        goto unknown_element;
      break;

    case XDG_PARSER_NOT:
      if (strcmp (element_name, "Not") == 0)
        parser->pattern = parser->pattern->parent;
      else
        goto unknown_element;
      break;

    case XDG_PARSER_KDELEGACYDIRS:
      if (strcmp (element_name, "KDELegacyDirs") == 0)
        {
          dirs = xfce_resource_lookup_all (XFCE_RESOURCE_DATA, "applnk/");
          for (n = 0; dirs[n] != NULL; ++n)
            xdg_menu_include_legacy (menu, "kde-", dirs[n], TRUE);
          g_strfreev (dirs);
        }
      else
        goto unknown_element;
      break;

    case XDG_PARSER_LEGACYDIR:
      if (strcmp (element_name, "LegacyDir") == 0)
        {
          if (parser->temp->str != '\0')
            xdg_menu_include_legacy (menu, "", parser->temp->str, TRUE);
        }
      else
        goto unknown_element;
      break;

    case XDG_PARSER_MOVE:
      if (strcmp (element_name, "Move") == 0)
        {
          if (menu->moves == NULL)
            {
              menu->moves = g_hash_table_new_full (g_str_hash,
                                                   g_str_equal,
                                                   g_free,
                                                   g_free);
            }

          g_hash_table_insert (menu->moves, 
                               g_strdup (parser->move_old->str),
                               g_strdup (parser->move_new->str));
        }
      else
        goto unknown_element;
      break;

    case XDG_PARSER_OLD:
      if (strcmp (element_name, "Old") != 0)
        goto unknown_element;
      break;

    case XDG_PARSER_NEW:
      if (strcmp (element_name, "New") != 0)
        goto unknown_element;
      break;

    default:
      goto unknown_element;
    }

  xfce_stack_pop (parser->stack);
  return;

unknown_element:
  g_set_error (error, G_MARKUP_ERROR, G_MARKUP_ERROR_UNKNOWN_ELEMENT,
               "Unknown closing element <%s>", element_name);
  return;
}


static void
text_handler (GMarkupParseContext  *context,
              const gchar          *text,
              gsize                 text_len,
              gpointer              user_data,
              GError              **error)
{
  XdgParser *parser = XDG_PARSER (user_data);

  switch (xfce_stack_top (parser->stack))
    {
    case XDG_PARSER_APPDIR:
    case XDG_PARSER_DIRECTORYDIR:
    case XDG_PARSER_NAME:
    case XDG_PARSER_DIRECTORY:
    case XDG_PARSER_LEGACYDIR:
    case XDG_PARSER_FILENAME:
    case XDG_PARSER_CATEGORY:
      g_string_append_len (parser->temp, text, text_len);
      break;

    case XDG_PARSER_OLD:
      g_string_append_len (parser->move_old, text, text_len);
      break;

    case XDG_PARSER_NEW:
      g_string_append_len (parser->move_new, text, text_len);
      break;

    default:
      break;
    }
}



static void
xdg_menu_include_legacy (XdgMenu      *parent,
                         const gchar  *prefix,
                         const gchar  *path,
                         gboolean      toplevel)
{
  const gchar *entry;
  XdgPattern  *pattern;
  XdgAppDir   *appdir;
  XdgMenu     *menu;
  gchar        fullpath[PATH_MAX];
  gchar        directory[PATH_MAX];
  gchar       *tmp;
  GDir        *dp;

  dp = g_dir_open (path, 0, NULL);
  if (G_UNLIKELY (dp == NULL))
    return;

  if (toplevel)
    {
      appdir = g_new (XdgAppDir, 1);
      g_strlcpy (appdir->path, path, PATH_MAX);
      g_strlcpy (appdir->prefix, prefix, PATH_MAX);
      parent->appdirs = g_list_append (parent->appdirs, appdir);
    }

  for (;;)
    {
      entry = g_dir_read_name (dp);
      if (entry == NULL)
        break;

      /* ignore dirs such as .hidden, etc. (KDE sucks) */
      if (*entry == '.')
        continue;

      g_snprintf (fullpath, PATH_MAX, "%s/%s", path, entry);

      if (g_file_test (fullpath, G_FILE_TEST_IS_DIR))
        {
          menu = g_new0 (XdgMenu, 1);
          menu->name = g_strdup (entry);

          g_snprintf (directory, PATH_MAX, "%s/.directory", fullpath);
          if (g_file_test (directory, G_FILE_TEST_IS_REGULAR))
            {
              menu->directory = g_strdup (".directory");
              menu->directory_dir = g_strdup (fullpath);
            }

          if (parent->clast != NULL)
            {
              menu->prev = parent->clast;
              parent->clast->next = menu;
            }
          else
            {
              parent->cfirst = menu;
            }
          parent->clast = menu;

          tmp = g_strconcat (prefix, entry, "-", NULL);
          xdg_menu_include_legacy (menu, tmp, fullpath, FALSE);
          g_free (tmp);
        }
      else if (!toplevel
          && g_file_test (fullpath, G_FILE_TEST_IS_REGULAR)
          && g_str_has_suffix (entry, ".desktop"))
        {
          if (parent->include == NULL)
            parent->include = xdg_pattern_new (XDG_PATTERN_OR, NULL);

          tmp = g_strconcat (prefix, entry, NULL);
          pattern = xdg_pattern_new (XDG_PATTERN_FILENAME, tmp);
          xdg_pattern_append (parent->include, pattern);
          g_free (tmp);
        }
    }

  g_dir_close (dp);
}



