#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdarg.h>
#include <string.h>
#include <inttypes.h>

#include "xenner.h"

/* ------------------------------------------------------------------ */

struct colors {
    char *name;     /* theme name */
    char *banner;   /* section banner */
    char *fixme;    /* debug: stuff to be coded */
    char *debug;    /* debug: lower priority stuff */
    char *emu;
    char *reset;    /* reset sequence */
};

static const struct colors themes[] = {
    {
	.name    = "off",
	.banner  = "",
	.fixme   = "",
	.debug   = "",
	.emu     = "",
	.reset   = "",
    },{
	.name    = "dark",
	.banner  = "\x1b[41;30m",  /* black on red   */
	.fixme   = "\x1b[31;40m",  /* red   on black */
	.debug   = "\x1b[36;40m",  /* cyan  on black */
	.emu     = "\x1b[36;40m",  /* cyan  on black */
	.reset   = "\x1b[0m",
    },{
	.name    = "light",
	.banner  = "\x1b[7m",     /* reverse */
	.fixme   = "\x1b[31m",    /* red     */
	.debug   = "\x1b[34m",    /* blue    */
	.emu     = "\x1b[34m",    /* blue    */
	.reset   = "\x1b[0m",
    }
};

static const struct colors *theme = themes;

void loginit(char *name)
{
    char *term = getenv("TERM");
    int i;
    
    if (!isatty(fileno(stderr)))
	name = "off";

    if (term && !name) {
	if (0 == strcmp(term, "linux"))
	    name = "dark";
	if (0 == strcmp(term, "xterm"))
	    name = "light";
	if (!name)
	    name = "off";
    }

    for (i = 0; i < sizeof(themes)/sizeof(themes[0]); i++) {
	if (0 == strcmp(name, themes[i].name))
	    theme = themes+i;
    }
}

void dbgprintf(struct xenvm *xen, int msglevel, const char *fmt, ...)
{
    va_list args_file;
    va_list args_tty;

    if (msglevel > xen->debug)
	return;

    va_start(args_file, fmt);
    va_copy(args_tty, args_file);
    if (xen->logfile) {
	fprintf(xen->logfile, "[xenner,%d] ", msglevel);
	vfprintf(xen->logfile, fmt, args_file);
    }
    if (!xen->nostderr) {
	fprintf(stderr, "%s[xenner,%d] ",
		msglevel ? theme->debug : theme->fixme, msglevel);
	vfprintf(stderr, fmt, args_tty);
	fprintf(stderr, "%s", theme->reset);
    }
    va_end(args_file);
    va_end(args_tty);
}

void logprintf(struct xenvm *xen, const char *fmt, ...)
{
    va_list args_file;
    va_list args_tty;

    va_start(args_file, fmt);
    va_copy(args_tty, args_file);
    if (xen && xen->logfile)
	vfprintf(xen->logfile, fmt, args_file);
    if (!xen || !xen->nostderr)
	vfprintf(stderr, fmt, args_tty);
    va_end(args_file);
    va_end(args_tty);
}

void conemu_print(struct xenvcpu *vcpu, const char *buf)
{
    struct xenvm *xen = vcpu->vm;
    
    if (xen->logfile)
	fprintf(xen->logfile, "[emu/%d] %s", vcpu->id, buf);
    if (!xen->nostderr)
	fprintf(stderr, "%s[emu/%d] %s%s", theme->emu, vcpu->id, buf, theme->reset);
}

void banner_print(struct xenvm *xen, const char *msg)
{
    static const char *line = "====================";

    if (xen->logfile)
	fprintf(xen->logfile, "%s %s %s\n", line, msg, line);
    if (!xen->nostderr)
	fprintf(stderr, "%s%s %s %s%s\n",
		theme->banner, line, msg, line, theme->reset);
}

void section_print(struct xenvm *xen, const char *func, char *msg)
{
    static const char *line = "-----";

    if (xen->logfile)
	fprintf(xen->logfile, "%s: %s %s %s\n", func, line, msg, line);
    if (!xen->nostderr)
	fprintf(stderr, "%s: %s %s %s\n", func, line, msg, line);
}
