typedef int (*xen_hypercall)(struct xenvcpu *vcpu, uint64_t *args);

/* ------------------------------------------------------------------ */
/* hypercalls: dead code -- moved to emu                              */

#if 0
static int xen_version(struct xenvm *xen, uint64_t *args)
{
    xen_feature_info_t *fi;
    uint32_t *ptr32;
    
    switch (args[0]) {
    case XENVER_version:
	return (3 << 16) | 1;

    case XENVER_get_features:
	fi = guest_vaddr_to_ptr(xen, args[1]);
	fi->submap = 0;
	if (0 == fi->submap_idx)
	    fi->submap |= (1 << XENFEAT_pae_pgdir_above_4gb);
 	break;

    case XENVER_platform_parameters:
	switch (xen->mode) {
	case XENMODE_32:
	    ptr32 = guest_vaddr_to_ptr(xen, args[1]);
	    *ptr32 = XEN_M2P_32;
	    break;
	case XENMODE_PAE:
	    ptr32 = guest_vaddr_to_ptr(xen, args[1]);
	    *ptr32 = XEN_M2P_PAE;
	    break;
	default:
	    d0printf("%s: no 64bit %s\n", __FUNCTION__, GETNAME(xenver_names, args[0]));
	    return -ENOSYS;
	}
	break;

    default:
	d0printf("%s: FIXME: unknown %d (%s)\n", __FUNCTION__,
		 (int)args[0], GETNAME(xenver_names, args[0]));
	return -ENOSYS;
    }
    return 0;
}

static int vm_assist(struct xenvm *xen, uint64_t *args)
{
    int type = args[1];
    const char *name = GETNAME(vmasst_type_names, type);

    switch (args[0]) {
    case VMASST_CMD_enable:
	d1printf("%s: enable \"%s\"\n", __FUNCTION__, name);
	break;
    case VMASST_CMD_disable:
	d0printf("%s: disable \"%s\"\n", __FUNCTION__, name);
	break;
    default:
	d0printf("%s: FIXME: unknown %d (%s)\n", __FUNCTION__,
		 (int)args[0], GETNAME(vmasst_cmd_names, args[0]));
	return -ENOSYS;
    }
    return 0;
}

static int nmi_op(struct xenvm *xen, uint64_t *args)
{
    switch (args[0]) {
    default:
	d0printf("%s: FIXME: unknown %d (%s)\n", __FUNCTION__,
		 (int)args[0], GETNAME(xennmi_names, args[0]));
	return -ENOSYS;
    }
    return 0;
}

static int physdev_op(struct xenvm *xen, uint64_t *args)
{
    d1printf("%s: %d (%s) -> no permission\n", __FUNCTION__,
	     (int)args[0], GETNAME(physdevop_names, args[0]));
    return -EPERM; /* we don't do dom0 */
}

static int set_debugreg(struct xenvm *xen, uint64_t *args)
{
    d1printf("%s: reg %d val 0x%" PRIx64 " -> no permission\n", __FUNCTION__,
	     (int)args[0], args[1]);
    return -EPERM; /* not supported (yet?) */
}

static int fpu_taskswitch(struct xenvm *xen, uint64_t *args)
{
    d0printf("%s: FIXME (set %d)\n", __FUNCTION__, (int)args[0]);
    return 0;
}

static int grant_table_op(struct xenvm *xen, uint64_t *args)
{
    int rc = 0;
    
    switch (args[0]) {
    default:
	d0printf("%s: FIXME: unknown %d (%s)\n", __FUNCTION__,
		 (int)args[0], GETNAME(gnttabop_names, args[0]));
	rc = -ENOSYS;
    }
    return rc;
}

static int mmu_update_32(struct xenvm *xen, uint64_t *args)
{
    uint64_t *reqs = guest_vaddr_to_ptr(xen, args[0]);
    uint32_t count = args[1];
//    uint32_t *done = guest_paddr_to_ptr(vm, args[2]);
    uint32_t dom   = args[3];
    uint32_t *ptr;
    int i;

    if (dom != DOMID_SELF) {
	d0printf("%s: foreigndom not supported\n", __FUNCTION__);
	return -ENOSYS;
    }

    for (i = 0; i < count; i++) {
	switch (reqs[0] & 3) {
	case MMU_NORMAL_PT_UPDATE:
	    d2printf("%s: normal pt: maddr %" PRIx64 " val %" PRIx64 "\n",
		     __FUNCTION__, reqs[0], reqs[1]);
	    ptr = xen->memory + reqs[0];
	    *ptr = reqs[1];
	    break;
	case MMU_MACHPHYS_UPDATE:
	    d0printf("%s: FIXME: machphys: %" PRIx64 " %" PRIx64 "\n",
		     __FUNCTION__, reqs[0], reqs[1]);
	    return -ENOSYS;
	    break;
	default:
	    d0printf("%s: %" PRIx64 " %p\n", __FUNCTION__, *reqs, reqs);
	    return -ENOSYS;
	}
	reqs += 2;
    }
    return 0;
}

static int mmu_update_pae(struct xenvm *xen, uint64_t *args)
{
    uint64_t *reqs = guest_vaddr_to_ptr(xen, args[0]);
    uint32_t count = args[1];
//    uint32_t *done = guest_paddr_to_ptr(vm, args[2]);
    uint32_t dom   = args[3];
    uint64_t *ptr;
    int i;

    if (dom != DOMID_SELF) {
	d0printf("%s: foreigndom not supported\n", __FUNCTION__);
	return -ENOSYS;
    }

    for (i = 0; i < count; i++) {
	switch (reqs[0] & 3) {
	case MMU_NORMAL_PT_UPDATE:
	    d2printf("%s: normal pt: maddr %" PRIx64 " val %" PRIx64 "\n",
		     __FUNCTION__, reqs[0], reqs[1]);
	    ptr = xen->memory + reqs[0];
	    *ptr = reqs[1];
	    break;
	case MMU_MACHPHYS_UPDATE:
	    d0printf("%s: FIXME: machphys: %" PRIx64 " %" PRIx64 "\n",
		     __FUNCTION__, reqs[0], reqs[1]);
	    return -ENOSYS;
	    break;
	default:
	    d0printf("%s: %" PRIx64 " %p\n", __FUNCTION__, *reqs, reqs);
	    return -ENOSYS;
	}
	reqs += 2;
    }
    return 0;
}

static int stack_switch_32pae(struct xenvm *xen, uint64_t *args)
{
    uint32_t ss  = args[0];
    uint32_t esp = args[1];

    xen->tss_32->ss1  = fix_sel32(ss);
    xen->tss_32->esp1 = esp;
    return 0;
}

static int update_descriptor_32pae(struct xenvm *xen, uint64_t *args)
{
    uint64_t pa   = args[0] | (uint64_t)args[1] << 32;
    struct descriptor_32 desc = {
	.a = args[2],
	.b = args[3],
    };
    int p, index;

    for (p = 0; p < sizeof(xen->gdt_mfn)/sizeof(xen->gdt_mfn[0]); p++) {
	if (addr_to_frame(pa) == xen->gdt_mfn[p])
	    break;
    }
    if (p == sizeof(xen->gdt_mfn)/sizeof(xen->gdt_mfn[0]))
	return -EINVAL;
    index = addr_offset(pa) / sizeof(struct descriptor_32);
    
    d1printf("%s: %" PRIx64 " (pg %d, idx %d) .a %" PRIx32 " .b %" PRIx32 "\n",
	     __FUNCTION__, pa, p, index, desc.a, desc.b);

    fix_desc32(&desc);
    xen->gdt[p * 512 + index] = desc;
    return 0;
}

static int mmuext_op_32pae(struct xenvm *xen, uint64_t *args)
{
    uint32_t *uops = guest_vaddr_to_ptr(xen, args[0]);
    uint32_t count = args[1];
//    uint32_t *done = guest_paddr_to_ptr(vm, args[2]);
    uint32_t dom   = args[3];
    int i;

    if (dom != DOMID_SELF) {
	d0printf("%s: foreigndom not supported\n", __FUNCTION__);
	return -ENOSYS;
    }

    for (i = 0; i < count; i++, uops += 3) {
	switch (uops[0]) {
	case MMUEXT_PIN_L1_TABLE:
	case MMUEXT_PIN_L2_TABLE:
	    d2printf("%s: ignore page pinning\n", __FUNCTION__);
	    break;
	case MMUEXT_SET_LDT:
	    d0printf("%s: FIXME: ignore SET_LDT\n", __FUNCTION__);
	    break;
	case MMUEXT_INVLPG_LOCAL:
	    /* FIXME: ignore for now, vmexit flushes tlb anyway ... */
	    d2printf("%s: invlpg 0x%" PRIx32 " local\n", __FUNCTION__, uops[1]);
	    break;
	case MMUEXT_TLB_FLUSH_LOCAL:
	    /* FIXME: ignore for now, vmexit flushes tlb anyway ... */
	    d2printf("%s: tlbflush local\n", __FUNCTION__);
	    break;
	case MMUEXT_TLB_FLUSH_ALL:
	    /* FIXME: ignore for now, vmexit flushes tlb anyway ... */
	    d2printf("%s: tlbflush all\n", __FUNCTION__);
	    break;
	case MMUEXT_TLB_FLUSH_MULTI:
	    /* FIXME: ignore for now, vmexit flushes tlb anyway ... */
	    d2printf("%s: tlbflush multi\n", __FUNCTION__);
	    break;
	case MMUEXT_NEW_BASEPTR:
	    d1printf("%s: new baseptr: mfn 0x%" PRIx32 "\n", __FUNCTION__, uops[1]);
	    xen->sregs.cr3 = frame_to_addr(uops[1]);
	    xen_emu_map(xen);
	    flush_sregs(xen);
	    break;
	default:
	    d0printf("%s: FIXME: unknown %d (%s)\n", __FUNCTION__,
		     (int)uops[0], GETNAME(mmuext_names, uops[0]));
	    return -ENOSYS;
	}
    }
    return 0;
}

static int update_va_mapping_32(struct xenvm *xen, uint64_t *args)
{
    uint32_t va    = args[0];
    uint32_t val   = args[1];
    /* args[2] is unused here (pae-only) */
    uint32_t flags = args[3];
    uint32_t *pte;

    pte = find_pte_32(xen, va);

    d2printf("%s: at va %" PRIx32
	     " | mfn/flg %" PRIx32 "/%" PRIx32
	     " -> %" PRIx32 "/%" PRIx32
	     " | flags %" PRIx32 "\n", __FUNCTION__,
	     va, get_pgframe_32(*pte), get_pgflags_32(*pte),
	     get_pgframe_32(val), get_pgflags_32(val), flags);

    if ((*pte & ~_PAGE_RW) == val && !test_pgflag_32(val, _PAGE_USER)) {
	d2printf("%s: ignoring mkreadonly (va 0x%" PRIx32 ")\n",
		 __FUNCTION__, va);
	return 0;
    }

    *pte = val;
    /* FIXME: tlb flush (ignore for now, vmexit does that anyway ...) */
    return 0;
}

static int update_va_mapping_pae(struct xenvm *xen, uint64_t *args)
{
    uint32_t va    = args[0];
    uint64_t val   = args[1] | (args[2] << 32 );
    uint32_t flags = args[3];
    uint64_t *pte;

    pte = find_pte_pae(xen, va);

    d2printf("%s: at va %" PRIx32
	     " | mfn/flg %" PRIx32 "/%" PRIx32
	     " -> %" PRIx32 "/%" PRIx32
	     " | flags %" PRIx32 "\n", __FUNCTION__,
	     va, get_pgframe_pae(*pte), get_pgflags_pae(*pte),
	     get_pgframe_pae(val), get_pgflags_pae(val), flags);

    if ((*pte & ~_PAGE_RW) == val && !test_pgflag_pae(val, _PAGE_USER)) {
	d1printf("%s: ignoring mkreadonly (va 0x%" PRIx32 ")\n",
		 __FUNCTION__, va);
	return 0;
    }

    *pte = val;
    /* FIXME: tlb flush (ignore for now, vmexit does that anyway ...) */
    return 0;
}

static int multicall_32pae(struct xenvm *xen, uint64_t *args)
{
    struct multicall_entry_x86_32 *calls = guest_vaddr_to_ptr(xen, args[0]);
    uint32_t i, count = args[1];
    uint64_t margs[6];
    xen_hypercall hcall;

    need_regs(xen);
    need_sregs(xen);

    for (i = 0; i < count; i++) {
	d3printf("%s: %d: %s\n", __FUNCTION__, i,
		 GETNAME(__hypervisor_names, calls[i].op));
	xen->usr_hcalls[calls[i].op]++;
	hcall = (xen->mode == XENMODE_32) ?
	    hcalls_32[calls[i].op] : hcalls_pae[calls[i].op];
	if (!hcall) {
	    d0printf("unhandled xen hypercall #%d (32/pae, %s)\n",
		     calls[i].op, GETNAME(__hypervisor_names, calls[i].op));
	    calls[i].result = -ENOSYS;
	}
	margs[0] = calls[i].args[0];
	margs[1] = calls[i].args[1];
	margs[2] = calls[i].args[2];
	margs[3] = calls[i].args[3];
	margs[4] = calls[i].args[4];
	margs[5] = calls[i].args[5];
	calls[i].result = hcall(xen, margs);
    }
    flush_regs(xen);
    return 0;
}

static int sched_op(struct xenvm *xen, uint64_t *args)
{
    switch (args[0]) {
    case SCHEDOP_yield:
	d0printf("%s: yield\n", __FUNCTION__);
	sleep(1); /* FIXME */
	xen->update_time = 1;
	break;

    case SCHEDOP_block:
	d2printf("%s: block\n", __FUNCTION__);
	wait_event(xen, 0);
	break;

    case SCHEDOP_shutdown:
	d0printf("%s: shutdown\n", __FUNCTION__);
	xenner_cleanup(xen);
	exit(0);

    default:
	d0printf("%s: FIXME: unknown %d (%s)\n", __FUNCTION__,
		 (int)args[0], GETNAME(schedop_names, args[0]));
	return -ENOSYS;
    }
    return 0;
}

static int memory_op(struct xenvm *xen, uint64_t *args)
{
    int cmd = args[0] & 0x0f /* MEMOP_CMD_MASK */;
    int rc;
    
    switch (cmd) {
    case XENMEM_machphys_mapping:
	switch (xen->mode) {
	case XENMODE_32:
	case XENMODE_PAE:
	{
	    struct xen_machphys_mapping_x86_32 map;
	    map.v_start = (xen->mode == XENMODE_32) ? XEN_M2P_32 : XEN_M2P_PAE;
	    map.v_end   = map.v_start + frame_to_addr(xen->pg_m2p);
	    map.max_mfn = xen->pg_m2p << (PAGE_SHIFT-3);
	    if (0 != (rc = copy_to_guest(xen, args[1], &map, sizeof(map))))
		return rc;
	    break;
	}
	case XENMODE_64:
	{
#ifdef __x86_64__
	    struct xen_machphys_mapping_x86_64 map;
	    map.v_start = XEN_M2P_64;
	    map.v_end   = map.v_start + frame_to_addr(xen->pg_m2p);
	    map.max_mfn = xen->pg_m2p << (PAGE_SHIFT-3);
	    if (0 != (rc = copy_to_guest(xen, args[1], &map, sizeof(map))))
		return rc;
	    break;
#else
	    return -ENOSYS;
#endif
	}
	}
	
    case XENMEM_memory_map:
	/* we have no e820 map */
	return -ENOSYS;

    default:
	d0printf("%s: FIXME: unknown %d (%s)\n", __FUNCTION__,
		 cmd, GETNAME(xenmem_names, cmd));
	return -ENOSYS;
    }
    return 0;
}

static int set_trap_table_32pae(struct xenvm *xen, uint64_t *args)
{
    struct trap_info_x86_32 *traps;
    struct trap_info_x86_32 tmp;
    int i, direct;

    if (0 == args[0]) {
	d0printf("%s: FIXME: clear table\n", __FUNCTION__);
	return -EINVAL;
    }

    d1printf("%s: fill table\n", __FUNCTION__);
    traps = guest_vaddr_to_ptr(xen, args[0]);
    for (i = 0;; i++) {
	if (!traps[i].address)
	    break;
	direct = 0;
	tmp = traps[i];
	tmp.cs = fix_sel32(tmp.cs);
	xen->xentr.tr32[traps[i].vector] = tmp;
	if (traps[i].vector > 0x40)
	    direct = 1;
	d2printf("%s: 0x%02x: code %x:%x flags 0x%x%s\n", __FUNCTION__,
		 traps[i].vector, tmp.cs, tmp.address, tmp.flags,
		 direct ? ", direct" : "");
	if (direct) {
	    uint32_t dpl = tmp.flags & 0x03;
	    xen->idt_32[traps[i].vector] =
		mkgate32(tmp.cs, tmp.address, 0x8f | (dpl << 5));
	}
    }
    return 0;
}

static int set_callbacks_32pae(struct xenvm *xen, uint64_t *args)
{
    xen_callback_x86_32_t tmp;

    d1printf("%s\n", __FUNCTION__);

    tmp.cs  = fix_sel32(args[0]);
    tmp.eip = args[1];
    xen->xencb.cb32[CALLBACKTYPE_event] = tmp;

    tmp.cs  = fix_sel32(args[2]);
    tmp.eip = args[3];
    xen->xencb.cb32[CALLBACKTYPE_failsafe] = tmp;
    return 0;
}

static int callback_op_32pae(struct xenvm *xen, uint64_t *args)
{
    struct callback_register_x86_32 *cb;
    xen_callback_x86_32_t tmp;

    cb = guest_vaddr_to_ptr(xen, args[1]);
    d1printf("%s: %s %s\n", __FUNCTION__,
	     GETNAME(callbackop_names, args[0]),
	     GETNAME(callbacktype_names, cb->type));
    if (cb->type >= 8)
	return -EINVAL;
    
    switch (args[0]) {
    case CALLBACKOP_register:
	tmp = cb->address;
	tmp.cs = fix_sel32(tmp.cs);
	xen->xencb.cb32[cb->type] = tmp;
	if (cb->flags)
	    d0printf("%s: FIXME: ignored flags\n", __FUNCTION__);
	break;
    case CALLBACKOP_unregister:
	xen->xencb.cb32[cb->type].cs  = 0;
	xen->xencb.cb32[cb->type].eip = 0;
	break;
    default:
	d0printf("%s: FIXME: unknown %d (%s)\n", __FUNCTION__,
		 (int)args[0], GETNAME(callbackop_names, args[0]));
	return -ENOSYS;
    }
    return 0;
}

static int set_gdt_32pae(struct xenvm *xen, uint64_t *args)
{
    uint32_t *ptrs = guest_vaddr_to_ptr(xen, args[0]);
    uint32_t entries = args[1];
    uint32_t pages = (entries + 511) / 512;
    struct descriptor_32 *src, *dst, tmp;
    int p,e;

    memset(xen->gdt_mfn, 0, sizeof(xen->gdt_mfn));

    if (entries > (0xe000 >> 3))
	return -EINVAL;
    d1printf("%s: ptrs %p pages %d entries %d\n", __FUNCTION__,
	     (void*)(intptr_t)args[0], pages, entries);
    for (p = 0; p < pages; p++) {
	xen->gdt_mfn[p] = ptrs[p];
	src = mfn_to_ptr(xen, ptrs[p]);
	dst = xen->gdt + p * 512;
	for (e = 0; e < 512; e++) {
	    tmp = src[e];
	    fix_desc32(&tmp);
	    dst[e] = tmp;
	}
    }
    return 0;
}

static int callback_op_64(struct xenvm *xen, uint64_t *args)
{
    struct callback_register_x86_64 *cb;

    cb = guest_vaddr_to_ptr(xen, args[1]);
    d1printf("%s: %s %s\n", __FUNCTION__,
	     GETNAME(callbackop_names, args[0]),
	     GETNAME(callbacktype_names, cb->type));
    if (cb->type >= 8)
	return -EINVAL;
    
    switch (args[0]) {
    case CALLBACKOP_register:
	xen->xencb.cb64[cb->type] = cb->address;
	if (cb->flags)
	    d0printf("%s: FIXME: ignored flags\n", __FUNCTION__);
	break;
    case CALLBACKOP_unregister:
	xen->xencb.cb64[cb->type] = 0;
	break;
    default:
	d0printf("%s: FIXME: unknown %d (%s)\n", __FUNCTION__,
		 (int)args[0], GETNAME(callbackop_names, args[0]));
	return -ENOSYS;
    }
    return 0;
}

static int set_callbacks_64(struct xenvm *xen, uint64_t *args)
{
    d1printf("%s: %" PRIx64 " %" PRIx64 "\n", __FUNCTION__,
	     args[0], args[1]);

    xen->xencb.cb64[CALLBACKTYPE_event]    = args[0];
    xen->xencb.cb64[CALLBACKTYPE_failsafe] = args[1];
    return 0;
}

static int set_trap_table_64(struct xenvm *xen, uint64_t *args)
{
    struct trap_info_x86_64 *traps;
    struct trap_info_x86_64 tmp;
    int i;

    if (0 == args[0]) {
	d0printf("%s: FIXME: clear table\n", __FUNCTION__);
	return -EINVAL;
    }

    d1printf("%s: fill table\n", __FUNCTION__);
    traps = guest_vaddr_to_ptr(xen, args[0]);
    for (i = 0;; i++) {
	if (!traps[i].address)
	    break;
	tmp = traps[i];
	tmp.cs = fix_sel64(tmp.cs);
	xen->xentr.tr64[traps[i].vector] = tmp;
	d2printf("%s: 0x%02x: code 0x%" PRIx64 " flags 0x%x\n", __FUNCTION__,
		 traps[i].vector, traps[i].address, traps[i].flags);
    }
    return 0;
}

static int set_gdt_64(struct xenvm *xen, uint64_t *args)
{
    uint64_t *ptrs = guest_vaddr_to_ptr(xen, args[0]);
    uint32_t entries = args[1];
    uint32_t pages = (entries + 511) / 512;
    struct descriptor_32 *src, *dst, tmp;
    int p,e;

    memset(xen->gdt_mfn, 0, sizeof(xen->gdt_mfn));

    if (entries > (0xe000 >> 3))
	return -EINVAL;
    d1printf("%s: ptrs %p pages %d entries %d\n", __FUNCTION__,
	     (void*)args[0], pages, entries);
    for (p = 0; p < pages; p++) {
	xen->gdt_mfn[p] = ptrs[p];
	src = mfn_to_ptr(xen, ptrs[p]);
	dst = xen->gdt + p * 512;
	for (e = 0; e < 512; e++) {
	    tmp = src[e];
	    fix_desc64(&tmp);
	    dst[e] = tmp;
	}
    }
    return 0;
}

static int set_segment_base_64(struct xenvm *xen, uint64_t *args)
{
    int vcpu = 0;
    
    switch (args[0]) {
    case SEGBASE_FS:
	set_msr(xen, vcpu, MSR_FS_BASE, args[1]);
	break;
    case SEGBASE_GS_USER:
	set_msr(xen, vcpu, MSR_KERNEL_GS_BASE, args[1]);
	break;
    case SEGBASE_GS_KERNEL:
	set_msr(xen, vcpu, MSR_GS_BASE, args[1]);
	break;
    case SEGBASE_GS_USER_SEL:
	if (args[1]) {
	    d0printf("%s: gs_user_sel: %" PRIx64 "\n", __FUNCTION__, args[1]);
	}
	return 0;
    default:
	d0printf("%s: unknown %d\n", __FUNCTION__, (int)args[0]);
	return -ENOSYS;
    }
    return 0;
}

#endif

/* ------------------------------------------------------------------ */
/* hypercalls: common all                                             */

static int vcpu_op(struct xenvcpu *vcpu, uint64_t *args)
{
    struct xenvm *xen = vcpu->vm;
    struct xenvcpu *opcpu;
#if 0
    uint32_t *ptr32;
    uint64_t *ptr64;
#endif

    if (args[1] >= xen->vcpus)
	return -ENOENT;
    opcpu = xen->vcpu + args[1];
    
    switch (args[0]) {
#if 0
    case VCPUOP_register_runstate_memory_area:
	switch (xen->mode) {
	case XENMODE_32:
	case XENMODE_PAE:
	    ptr32 = guest_vaddr_to_ptr(vcpu, args[2]);
	    xen->vcpu[vcpu->id].rs.rs32 = guest_vaddr_to_ptr(vcpu, *ptr32);
	    break;
	case XENMODE_64:
	    ptr64 = guest_vaddr_to_ptr(vcpu, args[2]);
	    xen->vcpu[vcpu->id].rs.rs64 = guest_vaddr_to_ptr(vcpu, *ptr64);
	    break;
	}
	return 0;

    case VCPUOP_is_up:
	return opcpu->online;
#endif

    case VCPUOP_set_periodic_timer:
    case VCPUOP_stop_periodic_timer:
    case VCPUOP_set_singleshot_timer:
    case VCPUOP_stop_singleshot_timer:
    case VCPUOP_initialise:
    case VCPUOP_up:
    case VCPUOP_down:
	/* not reached, handled by emu */
	return 0;

    default:
	d0printf("%s/%d: FIXME: unknown %d (%s)\n", __FUNCTION__, vcpu->id,
		 (int)args[0], vcpuop_name(args[0]));
	return -ENOSYS;
    }
    return 0;
}

static int event_channel_op(struct xenvcpu *vcpu, uint64_t *args)
{
    struct xenvm *xen = vcpu->vm;
    int rc;
    
    switch (args[0]) {
#if 0 /* handled by emu now */
    case EVTCHNOP_alloc_unbound:
    {
	struct evtchn_alloc_unbound a;

	if (0 != (rc = copy_from_guest(vcpu, &a, args[1], sizeof(a))))
	    return rc;
	if (a.dom != DOMID_SELF || a.remote_dom != 0)
	    return -EINVAL;
	a.port = evtchn_port(xen, "unbound");
	d1printf("%s: alloc unbound -> port %d\n", __FUNCTION__, a.port);
	if (0 != (rc = copy_to_guest(vcpu, args[1], &a, sizeof(a))))
	    return rc;
	break;
    }
    case EVTCHNOP_bind_virq:
    {
	struct evtchn_bind_virq bind;
	
	if (0 != (rc = copy_from_guest(vcpu, &bind, args[1], sizeof(bind))))
	    return rc;
	if (bind.virq >= NR_VIRQS)
	    return -EINVAL;
	bind.port = evtchn_port(xen, "virq");
	d1printf("%s: bind virq %d (%s) vcpu %d -> port %d\n", __FUNCTION__,
		 bind.virq, GETNAME(virq_names, bind.virq),
		 bind.vcpu, bind.port);
	xen->virq[bind.virq] = bind.port;
	if (0 != (rc = copy_to_guest(vcpu, args[1], &bind, sizeof(bind))))
	    return rc;
	break;
    }
    case EVTCHNOP_bind_ipi:
    {
	struct evtchn_bind_ipi bind;

	if (0 != (rc = copy_from_guest(vcpu, &bind, args[1], sizeof(bind))))
	    return rc;
	bind.port = evtchn_port(xen, "ipi");
	d1printf("%s: bind ipi vcpu %d -> port %d\n", __FUNCTION__,
		 bind.vcpu, bind.port);
	if (0 != (rc = copy_to_guest(vcpu, args[1], &bind, sizeof(bind))))
	    return rc;
	break;
    }
    case EVTCHNOP_unmask:
    {
	struct evtchn_unmask unmask;

	if (0 != (rc = copy_from_guest(vcpu, &unmask, args[1], sizeof(unmask))))
	    return rc;
	evtchn_unmask(xen, unmask.port);
	break;
    }
#endif
    case EVTCHNOP_send:
    {
	struct evtchn_send send;

	if (0 != (rc = copy_from_guest(vcpu, &send, args[1], sizeof(send))))
	    return rc;
	if (0 == evtchn_notify(xen, send.port)) {
	    d2printf("%s: send %d (external, ok)\n", __FUNCTION__, send.port);
	} else {
	    d1printf("%s: send %d (unhandled)\n", __FUNCTION__, send.port);
	}
	break;
    }
    case EVTCHNOP_bind_pirq:
	return -EPERM;
    default:
	d0printf("%s: FIXME: unknown %d (%s)\n", __FUNCTION__,
		 (int)args[0], evtchnop_name(args[0]));
	return -ENOSYS;
    }
    return 0;
}

static int event_channel_op_compat(struct xenvcpu *vcpu, uint64_t *args)
{
    struct evtchn_op op;
    uint64_t nargs[2];
    int rc;

    if (0 != (rc = copy_from_guest(vcpu, &op, args[0], sizeof(op))))
	return rc;
    nargs[0] = op.cmd;
    nargs[1] = args[0] + offsetof(struct evtchn_op, u);
    return event_channel_op(vcpu, nargs);
}

/* ------------------------------------------------------------------ */
/* hypercall tables                                                   */

static xen_hypercall hcalls_32[XEN_HCALL_MAX] = {
    [ __HYPERVISOR_vcpu_op ]                  = vcpu_op,
    [ __HYPERVISOR_event_channel_op ]         = event_channel_op,
    [ __HYPERVISOR_event_channel_op_compat ]  = event_channel_op_compat,
};

static xen_hypercall hcalls_pae[XEN_HCALL_MAX] = {
    [ __HYPERVISOR_vcpu_op ]                  = vcpu_op,
    [ __HYPERVISOR_event_channel_op ]         = event_channel_op,
    [ __HYPERVISOR_event_channel_op_compat ]  = event_channel_op_compat,
};

#ifdef __x86_64__

static xen_hypercall hcalls_64[XEN_HCALL_MAX] = {
    [ __HYPERVISOR_vcpu_op ]                  = vcpu_op,
    [ __HYPERVISOR_event_channel_op ]         = event_channel_op,
    [ __HYPERVISOR_event_channel_op_compat ]  = event_channel_op_compat,
};

#endif

/* ------------------------------------------------------------------ */

int do_xen_hypercall(struct xenvcpu *vcpu, int nr)
{
    struct xenvm *xen = vcpu->vm;
    uint64_t args[6];
    uint64_t retval = -ENOSYS;
    xen_hypercall hcall;

    need_regs(vcpu);
    need_sregs(vcpu);
    if (nr >= XEN_HCALL_MAX)
	goto done;

    d0printf("hypercall #%d (%s)\n", nr, __hypervisor_name(nr));
    switch (xen->mode) {
    case XENMODE_32:
    case XENMODE_PAE:
	hcall = (xen->mode == XENMODE_32) ? hcalls_32[nr] : hcalls_pae[nr];
	if (!hcall) {
	    d0printf("unhandled xen hypercall #%d (32/pae, %s)\n",
		     nr, __hypervisor_name(nr));
	    goto done;
	}
	args[0] = vcpu->regs.rbx & 0xffffffff;
	args[1] = vcpu->regs.rcx & 0xffffffff;
	args[2] = vcpu->regs.rdx & 0xffffffff;
	args[3] = vcpu->regs.rsi & 0xffffffff;
	args[4] = vcpu->regs.rdi & 0xffffffff;
	args[5] = vcpu->regs.rbp & 0xffffffff;
	retval = hcall(vcpu, args);
	break;

#ifdef __x86_64__
    case XENMODE_64:
	hcall = hcalls_64[nr];
	if (!hcall) {
	    d0printf("unhandled xen hypercall #%d (64, %s)\n",
		     nr, __hypervisor_name(nr));
	    goto done;
	}
	args[0] = vcpu->regs.rdi;
	args[1] = vcpu->regs.rsi;
	args[2] = vcpu->regs.rdx;
	args[3] = vcpu->regs.r10;
	args[4] = vcpu->regs.r8;
	args[5] = vcpu->regs.r9;
	retval = hcall(vcpu, args);
	break;
#endif
	
    default:
	d0printf("%s: unhandled xen mode %d\n", __FUNCTION__, xen->mode);
	break;
    }
    
done:
    vcpu->regs.rax = retval;
    flush_regs(vcpu);

    return 0;
}

