
#define	ENTRY(name) \
	.globl name; \
	.align 16; \
	name:

	.macro PUSH_ERROR
	sub $4, %esp		/* space for error code */
	.endm
	
	.macro PUSH_TRAP_EBP trapno
	sub $4, %esp		/* space for trap number */
	push %ebp
	mov $\trapno, %ebp
	mov %ebp, %ss:4(%esp)	/* save trap number on stack */
	.endm

	.macro PUSH_REGS
	mov  %es,%ebp
	push %ebp
	mov  %ds,%ebp
	push %ebp
	mov $0xe010,%ebp	/* ring0 data flat */
	mov %ebp,%ds
	mov %ebp,%es

	push %edi
	push %esi
	push %edx
	push %ecx
	push %ebx
	push %eax
	push %esp               /* struct regs pointer */
	.endm

	.macro POP_REGS
	pop %eax                /* dummy (struct regs pointer) */
	pop %eax
	pop %ebx
	pop %ecx
	pop %edx
	pop %esi
	pop %edi
	pop %ebp
	mov %ebp,%ds
	pop %ebp
	mov %ebp,%es
	pop %ebp
	.endm

	.macro RETURN
	add $8, %esp		/* remove error code & trap number */
	iret			/* jump back */
	.endm	

	.macro DO_TRAP trapno func
	PUSH_TRAP_EBP \trapno
	PUSH_REGS
	call \func
	POP_REGS
	RETURN
	.endm

/* ------------------------------------------------------------------ */

	.code32
	.text

/* --- boot entry point --- */

ENTRY(boot)
	cli
	lea boot_stack_high, %esp	/* setup stack */
	sub $64, %esp			/* sizeof(struct regs_32) */
	push %esp                       /* struct regs pointer */
	cmp $0, %ebx
	jne secondary
	call do_boot
	POP_REGS
	RETURN

secondary:
	push %ebx
	call do_boot_secondary
	pop %ebx
	POP_REGS
	RETURN

/* --- traps/faults handled by emu --- */

ENTRY(debug_int1)
	PUSH_ERROR
	DO_TRAP 1 do_int1

ENTRY(debug_int3)
	PUSH_ERROR
	DO_TRAP 3 do_int3

ENTRY(illegal_instruction)
	PUSH_ERROR
	DO_TRAP 6 do_illegal_instruction

ENTRY(no_device)
	PUSH_ERROR
	DO_TRAP 7 do_lazy_fpu

ENTRY(double_fault)
	DO_TRAP 8 do_double_fault

ENTRY(general_protection)
	DO_TRAP 13 do_general_protection

ENTRY(page_fault)
	DO_TRAP 14 do_page_fault

/* --- traps/faults forwarded to guest --- */

ENTRY(division_by_zero)
	PUSH_ERROR
	PUSH_TRAP_EBP 0
	jmp guest_forward

ENTRY(nmi)
	PUSH_ERROR
	PUSH_TRAP_EBP 2
	jmp guest_forward

ENTRY(overflow)
	PUSH_ERROR
	PUSH_TRAP_EBP 4
	jmp guest_forward

ENTRY(bound_check)
	PUSH_ERROR
	PUSH_TRAP_EBP 5
	jmp guest_forward

ENTRY(coprocessor)
	PUSH_ERROR
	PUSH_TRAP_EBP 9
	jmp guest_forward

ENTRY(invalid_tss)
	PUSH_TRAP_EBP 10
	jmp guest_forward

ENTRY(segment_not_present)
	PUSH_TRAP_EBP 11
	jmp guest_forward

ENTRY(stack_fault)
	PUSH_TRAP_EBP 12
	jmp guest_forward

ENTRY(floating_point)
	PUSH_ERROR
	PUSH_TRAP_EBP 16
	jmp guest_forward

ENTRY(alignment)
	PUSH_TRAP_EBP 17
	jmp guest_forward

ENTRY(machine_check)
	PUSH_ERROR
	PUSH_TRAP_EBP 18
	jmp guest_forward

ENTRY(simd_floating_point)
	PUSH_ERROR
	PUSH_TRAP_EBP 19
	jmp guest_forward

guest_forward:
	PUSH_REGS
	call do_guest_forward
	POP_REGS
	RETURN

/* --- interrupts 32 ... 255 --- */

ENTRY(smp_flush_tlb)
	PUSH_ERROR
	DO_TRAP -1 do_smp_flush_tlb

ENTRY(xen_hypercall)
	PUSH_ERROR
	PUSH_TRAP_EBP -1
	PUSH_REGS
	call do_hypercall
	POP_REGS
	add $8, %esp		/* remove error code & trap number */
	cmp $-1, -4(%esp)
	jne 1f
	out %al, $0xe0          /* let userspace handle it */
1:
	iret

ENTRY(irq_entries)
vector=0
.rept 256
	.align 16
	PUSH_ERROR
	PUSH_TRAP_EBP vector
	jmp irq_common
vector=vector+1
.endr

ENTRY(irq_common)
	PUSH_REGS
	call do_irq
	POP_REGS
	RETURN

/* --- helpers --- */

ENTRY(broken_memcpy_pf)
	mov 4(%esp),%edi
	mov 8(%esp),%esi
	mov 12(%esp),%ecx
	cld
1:	rep movsb
	xor %eax,%eax
8:	ret

	.section .exfix, "ax"
9:	mov $-1, %eax
	jmp 8b
	.previous

	.section .extab, "a"
	.align 4
	.long 1b,9b
	.previous

ENTRY(broken_copy32_pf)
	mov 4(%esp),%edi
	mov 8(%esp),%esi
1:	mov (%esi),%eax
2:	mov %eax,(%edi)
	xor %eax,%eax
8:	ret

	.section .exfix, "ax"
9:	mov $-1, %eax
	jmp 8b
	.previous

	.section .extab, "a"
	.align 4
	.long 1b,9b
	.long 2b,9b
	.previous

ENTRY(broken_copy64_pf)
	mov 4(%esp),%edi
	mov 8(%esp),%esi
1:	mov (%esi), %ebx
2:	mov 4(%esi), %ecx
3:	mov (%edi), %eax
4:	mov 4(%edi), %edx
5:	cmpxchg8b (%edi)
	xor %eax,%eax
8:	ret

	.section .exfix, "ax"
9:	mov $-1, %eax
	jmp 8b
	.previous

	.section .extab, "a"
	.align 4
	.long 1b,9b
	.long 2b,9b
	.long 3b,9b
	.long 4b,9b
	.previous

ENTRY(instructions)
	.byte  0x0f, 0x11, 0x00, 0x0f,  0x11, 0x48, 0x10, 0x0f
	nop
	nop
	nop
	nop
	nop
	nop
	nop
	nop

/* some 16 bit code for smp boot */

	.code16
	.align 4096
ENTRY(sipi)
	mov $0x00060000, %eax  /* EMUDEV_CMD_INIT_SECONDARY_VCPU */
	outl %eax, $0xe8       /* EMUDEV_REG_COMMAND */
	hlt
	.code32

/* data section */

	.data
	.globl boot_stack_low, boot_stack_high
	.globl cpu_ptr
	.align 4096
boot_stack_low:
cpu_ptr:
	.long 0
	.align 4096
boot_stack_high:

/* boot page tables */

#define pageflags 0x063 /* preset, rw, accessed, dirty */
#define largepage 0x080 /* pse */

	.section .pt.32, "a"
	.globl emu_pgd_32

	.align 4096
emu_pgd_32:
	.fill 1020,4,0
	.long pageflags + largepage
	.fill 3,4,0

	.section .pt.pae, "a"
	.globl emu_pgd_pae,emu_pmd_pae

	.align 4096
emu_pgd_pae:
	.fill 3,8,0
	.long emu_pmd_pae - 0xff000000 + 0x001
	.long 0
	.fill 508,8,0

	.align 4096
emu_pmd_pae:
	.fill 504,8,0
	.quad pageflags + largepage
	.fill 7,8,0
