#include <stdarg.h>
#include <stddef.h>
#include <inttypes.h>
#include <xen/xen.h>
#include <xen/callback.h>
#include <xen/grant_table.h>
#include <xen/version.h>
#include <xen/sched.h>
#include <xen/memory.h>
#include <xen/vcpu.h>
#include <xen/physdev.h>

#include "list.h"
#include "spinlock.h"
#include "atomic.h"

#include "shared.h"
#include "xen-emudev.h"
#include "xen/names.h"

/* attributes */
#define asmlinkage __attribute__((regparm(0)))
#define page_aligned __attribute__((aligned(4096))) __attribute__((__section__ (".pgdata")))

#if 0 /* work-in-progress */

#define KVM_FEATURE_CR3_CACHE            1

#define KVM_MSR_SET_CR3_CACHE   0x87655678
#define KVM_CR3_CACHE_SIZE               4
struct kvm_cr3_cache_entry {
	uint64_t guest_cr3;
	uint64_t host_cr3;
};
struct kvm_cr3_cache {
	struct kvm_cr3_cache_entry entry[KVM_CR3_CACHE_SIZE];
	uint32_t max_idx;
};
 
#endif

/* fwd decl */
struct xen_cpu;

/* arch specific bits */
#ifdef __x86_64__
# define longmode 1
# include "emu64.h"
#else
# define longmode 0
# include "emu32.h"
#endif

/* idt entry points */
extern void division_by_zero(void);
extern void debug_int1(void);
extern void nmi(void);
extern void debug_int3(void);
extern void overflow(void);
extern void bound_check(void);
extern void illegal_instruction(void);
extern void no_device(void);
extern void double_fault(void);
extern void coprocessor(void);
extern void invalid_tss(void);
extern void segment_not_present(void);
extern void stack_fault(void);
extern void general_protection(void);
extern void page_fault(void);
extern void floating_point(void);
extern void alignment(void);
extern void machine_check(void);
extern void simd_floating_point(void);
extern void smp_flush_tlb(void);
extern void int_unknown(void);

#if longmode
/* 64bit only */
extern void int_80(void);
#else
/* 32bit only */
extern void xen_hypercall(void);
#endif

/* functions */
#define EMU_PA(_vaddr)       (((uintptr_t)(_vaddr) - (uintptr_t)_vstart))
#define EMU_MFN(_vaddr)      (((uintptr_t)(_vaddr) - (uintptr_t)_vstart) >> PAGE_SHIFT)
extern uint8_t    _vstart[];
extern uint8_t    _vstop[];
extern uintptr_t  _estart[];
extern uintptr_t  _estop[];
extern uint8_t    trampoline_syscall[];

#define STACK_PTR(_cpu,_sym) ((void*)((_cpu)->stack_low + ((_sym) - boot_stack_low)))
#define STACK_PTR(_cpu,_sym) ((void*)((_cpu)->stack_low + ((_sym) - boot_stack_low)))
#define IRQSTACK_PTR(_cpu,_sym) ((void*)((_cpu)->irqstack_low + ((_sym) - boot_stack_low)))
extern uint8_t    boot_stack_low[];
extern uint8_t    cpu_ptr[];
#if longmode
extern uint8_t    trampoline_start[];
extern uint8_t    trampoline_patch[];
extern uint8_t    trampoline_stop[];
#endif
extern uint8_t    boot_stack_high[];

extern uint8_t    irq_entries[];
extern uint8_t    irq_common[];

extern uint8_t    sipi[];

/* emu-data.c */
extern int grant_frames;
extern struct grant_entry page_aligned grant_table[];
extern struct xenner_info vminfo;

struct vmconfig {
    uint64_t      mfn_emu;
    uint64_t      pg_emu;
    uint64_t      mfn_m2p;
    uint64_t      pg_m2p;
    uint64_t      mfn_guest;
    uint64_t      pg_guest;
    uint64_t      pg_total;
    int           debug_level;
    int           nr_cpus;
};
extern struct vmconfig vmconf;

struct xen_vcpu {
    void                  *vcpu_page;
    struct vcpu_info      *vcpu_info;
    uint64_t              vcpu_info_pa;
};

struct xen_cpu {
    /* used by hardware */
#if longmode
    struct tss_64         tss;
#else
    struct tss_32         tss;
#endif
    void                  *lapic;

    /* used by kvm */
    struct kvm_cr3_cache  *cr3_cache;
    
    /* emu state */
    struct xen_vcpu       v;
    struct descriptor_32  *gdt;
    uint64_t              gdt_mfns[16];
    uint32_t              virq_to_vector[NR_VIRQS];
    uint8_t               *stack_low;
    uint8_t               *stack_high;
#if longmode
    uint8_t               *irqstack_low;
    uint8_t               *irqstack_high;
#endif
    ureg_t                kernel_ss;
    ureg_t                kernel_sp;

    /* timer */
    uint64_t              periodic;
    uint64_t              oneshot;
    int                   timerport;

    /* lazy mmu */
    int                   multicall;

    /* I/O */
    int                   iopl;
    int                   nr_ports;

    /* initial state */
    struct vcpu_guest_context *init_ctxt;

#if longmode
    uint64_t              kernel_cr3_mfn;
    uint64_t              user_cr3_mfn;
    int                   user_mode;
#endif

    int                   online;
    int                   id;
    struct list_head      next;
};
extern struct list_head cpus;
extern ureg_t cpumask_all;

extern struct vcpu_guest_context boot_ctxt;
extern struct shared_info page_aligned shared_info;
extern xen_callback_t   xencb[8];
extern struct trap_info xentr[256];

extern uint64_t emu_hcalls[XEN_HCALL_MAX];
extern uint64_t emu_faults[XEN_FAULT_MAX];

struct trapinfo {
    char *name;
    int  ec;      /* has error code  */
    int  lvl;     /* debug log level */
};
extern const struct trapinfo trapinfo[32];
extern const char *cr0_bits[32];
extern const char *cr4_bits[32];
extern const char *pg_bits[32];
extern const char *rflags_bits[32];

/* emu-main.c */
void gdt_init(struct xen_cpu *cpu);
void gdt_load(struct xen_cpu *cpu);
void tss_init(struct xen_cpu *cpu);
void msrs_init(struct xen_cpu *cpu);
void idt_init(void);
void idt_load(void);
void guest_cpu_init(struct xen_cpu *cpu);
void guest_regs_init(struct xen_cpu *cpu, struct regs *regs);
struct xen_cpu *cpu_find(int id);
void print_registers(int level, struct regs *regs);
void print_stack(int level, ureg_t rsp);
void print_state(struct regs *regs);

int panic(char *message, struct regs *regs);
int bounce_trap(struct xen_cpu *cpu, struct regs *regs, int trapno, int cbno);
void flush_tlb_remote(struct xen_cpu *cpu, ureg_t mask, ureg_t addr);

asmlinkage void do_boot(struct regs *regs);
asmlinkage void do_boot_secondary(ureg_t id, struct regs *regs);
asmlinkage void do_illegal_instruction(struct regs *regs);
asmlinkage void do_general_protection(struct regs *regs);
asmlinkage void do_page_fault(struct regs *regs);
asmlinkage void do_double_fault(struct regs *regs);
asmlinkage void do_event_callback(struct regs *regs);
asmlinkage void do_guest_forward(struct regs *regs);
asmlinkage void do_int1(struct regs *regs);
asmlinkage void do_int3(struct regs *regs);
asmlinkage void do_lazy_fpu(struct regs *regs);
asmlinkage void do_smp_flush_tlb(struct regs *regs);

/* emu-mm.c */
void paging_init(void);
void paging_start(struct xen_cpu *cpu);
void update_emu_mappings(ureg_t cr3_mfn);
void *get_pages(int pages, char *purpose);
void *get_memory(int bytes, char *purpose);

/* emu-hcall.c */
#define HCALL_HANDLED  0
#define HCALL_FORWARD -1
#define HCALL_IRET    -2
void guest_gdt_copy_page(struct descriptor_32 *src, struct descriptor_32 *dst);
int guest_gdt_init(struct xen_cpu *cpu, uint32_t entries, ureg_t *mfns);

sreg_t error_noperm(struct xen_cpu *cpu, ureg_t *args);
sreg_t stack_switch(struct xen_cpu *cpu, ureg_t *args);
sreg_t console_io(struct xen_cpu *cpu, ureg_t *args);
sreg_t update_descriptor(struct xen_cpu *cpu, ureg_t *args);
sreg_t fpu_taskswitch(struct xen_cpu *cpu, ureg_t *args);
sreg_t grant_table_op(struct xen_cpu *cpu, ureg_t *args);
sreg_t xen_version(struct xen_cpu *cpu, ureg_t *args);
sreg_t vm_assist(struct xen_cpu *cpu, ureg_t *args);
sreg_t sched_op(struct xen_cpu *cpu, ureg_t *args);
sreg_t sched_op_compat(struct xen_cpu *cpu, ureg_t *args);
sreg_t memory_op(struct xen_cpu *cpu, ureg_t *args);
sreg_t set_trap_table(struct xen_cpu *cpu, ureg_t *args);
sreg_t set_callbacks(struct xen_cpu *cpu, ureg_t *args);
sreg_t callback_op(struct xen_cpu *cpu, ureg_t *args);
sreg_t set_gdt(struct xen_cpu *cpu, ureg_t *args);
sreg_t vcpu_op(struct xen_cpu *cpu, ureg_t *args);
sreg_t set_timer_op(struct xen_cpu *cpu, ureg_t *args);
sreg_t event_channel_op(struct xen_cpu *cpu, ureg_t *args);
sreg_t event_channel_op_compat(struct xen_cpu *cpu, ureg_t *args);
sreg_t mmuext_op(struct xen_cpu *cpu, ureg_t *args);
sreg_t physdev_op(struct xen_cpu *cpu, ureg_t *args);
sreg_t get_debugreg(struct xen_cpu *cpu, ureg_t *args);
sreg_t set_debugreg(struct xen_cpu *cpu, ureg_t *args);

/* emu-pv.c */
int pv_have_clock;
int pv_have_cr3_cache;

void pv_clock_update(int wakeup);
void pv_clock_sys(struct xen_cpu *cpu);
void pv_write_cr3(struct xen_cpu *cpu, ureg_t cr3);
void pv_init(struct xen_cpu *cpu);

/* emu-instr.c */
void real_cpuid(struct kvm_cpuid_entry *entry);
void print_bits(int level, char *msg, uint32_t old, uint32_t new, const char *names[]);
void print_emu_instr(int level, const char *prefix, uint8_t *instr);
int emulate(struct xen_cpu *cpu, struct regs *regs);

/* emu-lapic.c */
#define VECTOR_FLUSH_TLB    0x20
#define VECTOR_EVTCHN_START 0x21
void lapic_eoi(struct xen_cpu *cpu);
void lapic_timer(struct xen_cpu *cpu);
void lapic_ipi_boot(struct xen_cpu *cpu, struct xen_cpu *ap);
void lapic_ipi_flush_tlb(struct xen_cpu *cpu);
int evtchn_route_interdomain(struct xen_cpu *cpu, int port, char *desc);
int evtchn_route_virq(struct xen_cpu *cpu, int virq, int port, char *desc);
int evtchn_route_ipi(struct xen_cpu *cpu, int port);
int evtchn_send(struct xen_cpu *cpu, int port);
void evtchn_unmask(struct xen_cpu *cpu, int port);
int evtchn_alloc(int vcpu_id);
int evtchn_pending(struct xen_cpu *cpu);
void evtchn_try_forward(struct xen_cpu *cpu, struct regs *regs);
int irq_init(struct xen_cpu *cpu);
asmlinkage void do_irq(struct regs *regs);

/* printk.c */
int vscnprintf(char *buf, size_t size, const char *fmt, va_list args);
int snprintf(char * buf, size_t size, const char *fmt, ...);
int printk(int level, const char *fmt, ...) __attribute__((format(printf, 2, 3)));
void write_string(char *msg);

/* inline asm bits */
static inline ureg_t read_cr0(void)
{
    ureg_t val;
    asm volatile("mov %%cr0,%0"
		 : "=r" (val));
    return val;
}

static inline ureg_t read_cr2(void)
{
    ureg_t val;
    asm volatile("mov %%cr2,%0"
		 : "=r" (val));
    return val;
}

static inline ureg_t read_cr3(void)
{
    ureg_t val;
    asm volatile("mov %%cr3,%0"
		 : "=r" (val));
    return val;
}

static inline ureg_t read_cr4(void)
{
    ureg_t val;
    asm volatile("mov %%cr4,%0"
		 : "=r" (val));
    return val;
}

static inline void write_cr0(ureg_t val)
{
    asm volatile("mov %0, %%cr0"
		 : /* no output */
		 : "r" (val)
		 : "memory" );
}

static inline void write_cr3(ureg_t cr3)
{
    asm volatile("mov %0, %%cr3"
		 : /* no output */
		 : "r" (cr3)
		 : "memory");
}

static inline void write_cr4(ureg_t val)
{
    asm volatile("mov %0, %%cr4"
		 : /* no output */
		 : "r" (val)
		 : "memory");
}

static inline void flush_tlb(void)
{
    ureg_t tmpreg;

    asm volatile("mov %%cr3, %0;              \n"
		 "mov %0, %%cr3;  # flush TLB \n"
		 : "=r" (tmpreg)
		 : /* no input */
		 : "memory");
}

static inline void flush_tlb_addr(ureg_t va)
{
    asm volatile("invlpg (%0)"
		 : /* no output */
		 : "r" (va)
		 : "memory");
}

static inline void outb(uint8_t value, uint16_t port)
{
    asm volatile("outb %[value],%w[port]"
		 : /* no output */
		 : [value] "a" (value), [port] "Nd" (port)
		 : "memory");
}

static inline void rdmsr(uint32_t msr, uint32_t *ax, uint32_t *dx)
{
    asm volatile("rdmsr"
		 : "=a" (*ax), "=d" (*dx)
		 : "c" (msr)
		 : "memory");
}

static inline void wrmsr(uint32_t msr, uint32_t ax, uint32_t dx)
{
    asm volatile("wrmsr"
		 : /* no outputs */
		 : "c" (msr), "a" (ax), "d" (dx)
		 : "memory");
}

static inline void wrmsrl(uint32_t msr, uint64_t val)
{
    uint32_t ax = (uint32_t)val;
    uint32_t dx = (uint32_t)(val >> 32);
    wrmsr(msr, ax, dx);
}

static inline int wrmsrl_safe(uint32_t msr, uint64_t val)
{
    uint32_t ax = (uint32_t)val;
    uint32_t dx = (uint32_t)(val >> 32);
    return wrmsr_safe(msr, ax, dx);
}

static inline void lldt(uint16_t sel)
{
    asm volatile("lldt %0"
		 : /* no outputs */
		 : "a" (sel)
		 : "memory");
}

static inline void ltr(uint16_t sel)
{
    asm volatile("ltr %0"
		 : /* no outputs */
		 : "a" (sel)
		 : "memory");
}

static inline void halt(int cpu_id)
{
    vminfo.vcpus_running &= ~(1<<cpu_id);
    asm volatile("hlt" : : : "memory");
    vminfo.vcpus_running |= (1<<cpu_id);
}

static inline void clts(void)
{
    asm volatile("clts" : : : "memory");
}

static inline void vmexit_panic(void)
{
    asm volatile("outb %%al,$0xeb" : : : "memory");
}

static inline void pause(void)
{
    asm volatile("pause" : : : "memory");
}

static inline void sti(void)
{
    asm volatile("sti" : : : "memory");
}

static inline void cli(void)
{
    asm volatile("cli" : : : "memory");
}

static inline void int3(void)
{
    asm volatile("int3" : : : "memory");
}

static inline void set_eflag(ureg_t flag)
{
    ureg_t reg = 0;

    asm volatile("pushf\n"
		 "pop    %[reg]\n"
		 "or     %[tf], %[reg]\n"
		 "push   %[reg]\n"
	         "popf\n"
	         "nop\n"
		 : [reg] "+r" (reg)
		 : [tf]  "r" (flag)
		 : "memory");
}


/*
 * We have 4k stacks (one page).
 *  - there is a pointer to the per-cpu data at the bottom.
 *  - (64bit also has the sysenter trampoline there).
 */
static inline struct xen_cpu *get_cpu(void)
{
    uintptr_t rsp;

#if longmode
    asm volatile("mov %%rsp, %[rsp]" : [rsp] "=a" (rsp) : /* no input */);
#else
    asm volatile("mov %%esp, %[esp]" : [esp] "=a" (rsp) : /* no input */);
#endif
    rsp &= PAGE_MASK;
    return *((void**)rsp);
}

/* gcc builtins */
void *memset(void *s, int c, size_t n);
void *memcpy(void *dest, const void *src, size_t n);

/* guest virtual irq flag */
#define guest_cli(_cpu)		do {				\
		(_cpu)->v.vcpu_info->evtchn_upcall_mask = 1;	\
	} while (0)
#define guest_sti(_cpu)		do {				\
		(_cpu)->v.vcpu_info->evtchn_upcall_mask = 0;	\
	} while (0)
#define guest_irq_flag(_cpu)	(!((_cpu)->v.vcpu_info->evtchn_upcall_mask))

