#include "emu.h"

/* --------------------------------------------------------------------- */

const char *feature_bits[32] = {
    [ KVM_FEATURE_CLOCKSOURCE  ] = "clocksource",
    [ KVM_FEATURE_NOP_IO_DELAY ] = "nop-iodelay",
    [ KVM_FEATURE_MMU_OP       ] = "mmu-op",
};

int pv_have_clock;
int pv_have_cr3_cache;

/* --------------------------------------------------------------------- */

void pv_clock_update(int wakeup)
{
    static int wakeups;
    static int update;
    
    if (pv_have_clock)
	return;

    if (wakeup) {
	/* after halt() -- update clock unconditionally */
	update = 1;
	wakeups++;
    } else {
	/* timer irq -- update only if needed */
	update = (0 == wakeups);
	wakeups = 0;
    }

    /* vmexit to userspace so xenner has a chance to update systime */
    if (update)
	emudev_cmd(EMUDEV_CMD_NOP, 0);
}

static void pv_clock_wall(void)
{
    uint64_t wall = EMU_PA(&shared_info.wc_version);

    if (!pv_have_clock)
	return;
    printk(1, "%s: register wall clock at 0x%" PRIx64 "\n",
	   __FUNCTION__, wall);
    if (0 != wrmsrl_safe(MSR_KVM_WALL_CLOCK, wall))
	panic("MSR_KVM_WALL_CLOCK wrmsr failed", NULL);
    printk(1, "%s: v%d %d.%09d\n", __FUNCTION__,
	   shared_info.wc_version,
	   shared_info.wc_sec,
	   shared_info.wc_nsec);
}

void pv_clock_sys(struct xen_cpu *cpu)
{
    uint64_t sys = cpu->v.vcpu_info_pa + offsetof(struct vcpu_info, time);

    if (!pv_have_clock)
	return;
    printk(1, "%s: register vcpu %d clock at 0x%" PRIx64 "\n",
	   __FUNCTION__, cpu->id, sys);
    if (0 != wrmsrl_safe(MSR_KVM_SYSTEM_TIME, sys | 1))
	panic("MSR_KVM_SYSTEM_TIME wrmsr failed", NULL);
    printk(1, "%s: v%d sys %" PRIu64 " tsc %" PRIu64 " mul %u shift %d\n",
	   __FUNCTION__,
	   cpu->v.vcpu_info->time.version,
	   cpu->v.vcpu_info->time.system_time,
	   cpu->v.vcpu_info->time.tsc_timestamp,
	   cpu->v.vcpu_info->time.tsc_to_system_mul,
	   cpu->v.vcpu_info->time.tsc_shift);
}

/* --------------------------------------------------------------------- */

void pv_write_cr3(struct xen_cpu *cpu, ureg_t cr3)
{
#if 0
    int idx;

    if (cpu->cr3_cache && cpu->cr3_cache->max_idx) {
	for (idx = 0; idx < cpu->cr3_cache->max_idx; idx++) {
	    if (cpu->cr3_cache->entry[idx].guest_cr3 == cr3) {
		/*
		 * Cache-hit: we load the cached host-CR3 value.
		 * This never causes any VM exit. (if it does then the
		 * hypervisor could do nothing with this instruction
		 * and the guest OS would be aborted)
		 */
		vminfo.faults[XEN_FAULT_OTHER_CR3_CACHE_HIT]++;
		write_cr3(cpu->cr3_cache->entry[idx].host_cr3);
		return;
	    }
	}
    }
#endif

    write_cr3(cr3);
    return;
}

static void pv_init_cr3(struct xen_cpu *cpu)
{
    uint64_t cache;

    cpu->cr3_cache = get_pages(1, "cr3 cache");
//  cache = EMU_PA(cpu->cr3_cache);    /* physical */
    cache = (uintptr_t)cpu->cr3_cache; /* virtual  */
    printk(1, "%s: register cr3 cache at 0x%" PRIx64 " ...\n",
	   __FUNCTION__, cache);
#if 0
    if (0 != wrmsrl_safe(KVM_MSR_SET_CR3_CACHE, cache)) {
	printk(1, "%s: ... FAILED\n", __FUNCTION__);
	return -1;
    } else {
	printk(1, "%s: ... OK, %d entries\n",
	       __FUNCTION__, cpu->cr3_cache->max_idx);
	return 0;
    }
#endif
}

/* --------------------------------------------------------------------- */

void pv_init(struct xen_cpu *cpu)
{
    char buf[128];
    struct kvm_cpuid_entry entry;
    uint32_t sig[3];
    uint32_t features;

    entry.function = KVM_CPUID_SIGNATURE;
    real_cpuid(&entry);
    sig[0] = entry.ebx;
    sig[1] = entry.ecx;
    sig[2] = entry.edx;

    entry.function = KVM_CPUID_FEATURES;
    real_cpuid(&entry);
    features = entry.eax;

    snprintf(buf, sizeof(buf), "%s: cpu %d, signature \"%.12s\", features 0x%08x",
	     __FUNCTION__, cpu->id, (char*)sig, features);
    print_bits(1, buf, features, features, feature_bits);

    /* pv clocksource */
    if (features & (1 << KVM_FEATURE_CLOCKSOURCE)) {
	pv_have_clock = 1;
	pv_clock_sys(cpu);
	if (0 == cpu->id)
	    pv_clock_wall();
    }

    /* cr3 cache -- WIP */
    if (0 /* features & (1 << KVM_FEATURE_CR3_CACHE) */) {
	pv_have_cr3_cache = 1;
	pv_init_cr3(cpu);
    }
}

/* --------------------------------------------------------------------- */

