/*
	Copyright (C) 2004 - 2006 rt2x00 SourceForge Project
	<http://rt2x00.serialmonkey.com>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the
	Free Software Foundation, Inc.,
	59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
	Module: rt73usb
	Abstract: rt73usb device specific routines.
	Supported chipsets: RT2573.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/init.h>
#include <linux/usb.h>
#include <linux/delay.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/wireless.h>
#include <linux/firmware.h>
#include <linux/ethtool.h>

#include <net/iw_handler.h>
#include <net/d80211.h>

/*
 * Set enviroment defines for rt2x00.h
 */
#define DRV_NAME			"rt73usb"

#ifdef CONFIG_RT73USB_DEBUG
#define CONFIG_RT2X00_DEBUG
#endif /* CONFIG_RT73USB_DEBUG */

#include "rt2x00.h"
#include "rt2x00usb.h"
#include "rt2x00crc.h"
#include "rt73usb.h"

/*
 * Register access.
 * All access to the CSR registers will go through the methods
 * rt2x00_register_read and rt2x00_register_write.
 * BBP and RF register require indirect register access,
 * and use the CSR registers BBPCSR and RFCSR to achieve this.
 * These indirect registers work with busy bits,
 * and we will try maximal REGISTER_BUSY_COUNT times to access
 * the register while taking a REGISTER_BUSY_DELAY us delay
 * between each attampt. When the busy bit is still set at that time,
 * the access attempt is considered to have failed,
 * and we will print an error.
 * The caller to these register access functions, should take precautions
 * for the correct byte ordering of the values.
 */
static void rt2x00_vendor_request(
	const struct rt2x00_usb *rt2x00usb,
	const u8 request,
	const u8 type,
	const u16 offset,
	u32 value,
	void *buffer,
	const u16 buffer_length)
{
	int status = rt2x00usb_vendor_request(
		rt2x00usb, request, type, offset,
		value, buffer, buffer_length, REGISTER_TIMEOUT);

	if (status < 0)
		ERROR("vendor request error. Request 0x%02x failed "
			"for offset 0x%04x with error %d.\n",
			request, offset, status);
}

static inline void rt2x00_register_read(
	const struct rt2x00_usb *rt2x00usb, const u16 offset, u32 *value)
{
	rt2x00_vendor_request(
		rt2x00usb, USB_MULTI_READ, USB_VENDOR_REQUEST_IN,
		offset, 0x00, value, 4);
}

static inline void rt2x00_register_multiread(
	const struct rt2x00_usb *rt2x00usb,
	const u16 offset, u32 *value, const u32 length)
{
	rt2x00_vendor_request(
		rt2x00usb, USB_MULTI_READ, USB_VENDOR_REQUEST_IN,
		offset, 0x00, value, length);
}

static inline void rt2x00_register_write(
	const struct rt2x00_usb *rt2x00usb, const u16 offset, u32 value)
{
	rt2x00_vendor_request(
		rt2x00usb, USB_MULTI_WRITE, USB_VENDOR_REQUEST_OUT,
		offset, 0x00, &value, 4);
}

static inline void rt2x00_register_multiwrite(
	const struct rt2x00_usb *rt2x00usb,
	const u16 offset, u32 *value, const u32 length)
{
	rt2x00_vendor_request(
		rt2x00usb, USB_MULTI_WRITE, USB_VENDOR_REQUEST_OUT,
		offset, 0x00, value, length);
}

static void rt2x00_bbp_write(
	const struct rt2x00_usb *rt2x00usb, const u8 reg_id, const u8 value)
{
	u32 reg;
	unsigned int i;

	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		rt2x00_register_read(rt2x00usb, PHY_CSR3, &reg);
		if (!rt2x00_get_field32(reg, PHY_CSR3_BUSY))
			goto bbp_write;
		udelay(REGISTER_BUSY_DELAY);
	}

	ERROR("PHY_CSR3 register busy. Write failed.\n");
	return;

bbp_write:
	reg = 0;
	rt2x00_set_field32(&reg, PHY_CSR3_VALUE, value);
	rt2x00_set_field32(&reg, PHY_CSR3_REGNUM, reg_id);
	rt2x00_set_field32(&reg, PHY_CSR3_BUSY, 1);
	rt2x00_set_field32(&reg, PHY_CSR3_READ_CONTROL, 0);

	rt2x00_register_write(rt2x00usb, PHY_CSR3, reg);
}

static void rt2x00_bbp_read(
	const struct rt2x00_usb *rt2x00usb, const u8 reg_id, u8 *value)
{
	u32 reg;
	unsigned int i;

	/*
	 * First request the register we wish to read from.
	 */
	reg =0;
	rt2x00_set_field32(&reg, PHY_CSR3_REGNUM, reg_id);
	rt2x00_set_field32(&reg, PHY_CSR3_BUSY, 1);
	rt2x00_set_field32(&reg, PHY_CSR3_READ_CONTROL, 1);

	rt2x00_register_write(rt2x00usb, PHY_CSR3, reg);

	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		rt2x00_register_read(rt2x00usb, PHY_CSR3, &reg);
		if (!rt2x00_get_field32(reg, PHY_CSR3_BUSY)) {
			*value = rt2x00_get_field32(reg, PHY_CSR3_VALUE);
			return;
		}
		udelay(REGISTER_BUSY_DELAY);
	}

	ERROR("PHY_CSR3 register busy. Read failed.\n");
	*value = 0xff;
}

static void rt2x00_rf_write(
	const struct rt2x00_usb *rt2x00usb, const u32 value)
{
	u32 reg;
	unsigned int i;

	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		rt2x00_register_read(rt2x00usb, PHY_CSR4, &reg);
		if (!rt2x00_get_field32(reg, PHY_CSR4_BUSY))
			goto rf_write;
		udelay(REGISTER_BUSY_DELAY);
	}

	ERROR("RFCSR register busy. Write failed.\n");
	return;

rf_write:
	reg = value;
	rt2x00_set_field32(&reg, PHY_CSR4_NUMBER_OF_BITS, 20);
	rt2x00_set_field32(&reg, PHY_CSR4_IF_SELECT, 0);
	rt2x00_set_field32(&reg, PHY_CSR4_BUSY, 1);

	rt2x00_register_write(rt2x00usb, PHY_CSR4, reg);
}

static inline void rt2x00_eeprom_read(
	const struct rt2x00_usb *rt2x00usb, const u16 word, u16 *data)
{
	rt2x00_vendor_request(
		rt2x00usb, USB_EEPROM_READ, USB_VENDOR_REQUEST_IN,
		word, 0x00, (u32*)data, 2);
}

static void rt2x00_eeprom_multiread(
	const struct rt2x00_usb *rt2x00usb,
	const u8 word, u16 *data, const u16 length)
{
	rt2x00_vendor_request(
		rt2x00usb, USB_EEPROM_READ, USB_VENDOR_REQUEST_IN,
		word, 0x00, (u32*)data, length);
}

static void rt73usb_get_drvinfo(struct net_device *net_dev,
	struct ethtool_drvinfo *drvinfo)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);

	strcpy(drvinfo->driver, DRV_NAME);
	strcpy(drvinfo->version, DRV_VERSION);
	strcpy(drvinfo->bus_info, rt2x00usb->usb_intf->dev.bus_id);
}

static int rt73usb_get_regs_len(struct net_device *net_dev)
{
	return CSR_REG_SIZE;
}

static void rt73usb_get_regs(struct net_device *net_dev,
	struct ethtool_regs *regs, void *data)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);

	regs->version = rt2x00_get_chip_id(&rt2x00usb->chip);

	regs->len = CSR_REG_SIZE;

	rt2x00_register_multiread(rt2x00usb, CSR_REG_BASE, data, CSR_REG_SIZE);
}

#ifdef CONFIG_RT73USB_DEBUG
static u32 rt73usb_get_msglevel(struct net_device *net_dev)
{
	return rt2x00_debug_level;
}

static void rt73usb_set_msglevel(struct net_device *net_dev, u32 msg)
{
	rt2x00_debug_level = !!msg;
}
#endif /* CONFIG_RT73USB_DEBUG */

static int rt73usb_get_eeprom_len(struct net_device *net_dev)
{
	return EEPROM_SIZE;
}

static int rt73usb_get_eeprom(struct net_device *net_dev,
	struct ethtool_eeprom *eeprom, u8 *data)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);

	eeprom->magic = rt2x00_get_chip_id(&rt2x00usb->chip);

	rt2x00_eeprom_multiread(rt2x00usb,
		eeprom->offset / sizeof(u16), (void*)data, eeprom->len);

	return 0;
}

static struct ethtool_ops rt73usb_ethtool_ops = {
	.get_drvinfo	= rt73usb_get_drvinfo,
	.get_regs_len	= rt73usb_get_regs_len,
	.get_regs	= rt73usb_get_regs,
#ifdef CONFIG_RT73USB_DEBUG
	.get_msglevel	= rt73usb_get_msglevel,
	.set_msglevel	= rt73usb_set_msglevel,
#endif /* CONFIG_RT2500USB_DEBUG */
	.get_link	= ethtool_op_get_link,
	.get_eeprom_len	= rt73usb_get_eeprom_len,
	.get_eeprom	= rt73usb_get_eeprom,
};

/*
 * Configuration handlers.
 */
static void rt73usb_config_bssid(struct rt2x00_usb *rt2x00usb, u8 *bssid)
{
	u32 reg[2] = {0, 0};

	rt2x00_set_field32(&reg[0], MAC_CSR4_BYTE0, *(bssid));
	rt2x00_set_field32(&reg[0], MAC_CSR4_BYTE1, *(bssid + 1));
	rt2x00_set_field32(&reg[0], MAC_CSR4_BYTE2, *(bssid + 2));
	rt2x00_set_field32(&reg[0], MAC_CSR4_BYTE3, *(bssid + 3));
	rt2x00_set_field32(&reg[1], MAC_CSR5_BYTE4, *(bssid + 4));
	rt2x00_set_field32(&reg[1], MAC_CSR5_BYTE5, *(bssid + 5));

	rt2x00_register_multiwrite(rt2x00usb, MAC_CSR4, &reg[0], sizeof(reg));
}

static void rt73usb_config_promisc(struct rt2x00_usb *rt2x00usb, int promisc)
{
	u32 reg;

	rt2x00_register_read(rt2x00usb, TXRX_CSR0, &reg);

	if (promisc) {
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_NOT_TO_ME, 0);
		SET_FLAG(rt2x00usb, INTERFACE_ENABLED_PROMISC);
	} else {
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_NOT_TO_ME, 1);
		CLEAR_FLAG(rt2x00usb, INTERFACE_ENABLED_PROMISC);
	}

	rt2x00_register_write(rt2x00usb, TXRX_CSR0, reg);
}

static void rt73usb_config_type(struct rt2x00_usb *rt2x00usb, int type)
{
	u32 reg;

	/*
	 * Only continue when there is something to be done.
	 */
	if (type == rt2x00usb->interface_type
	|| (rt2x00usb->monitor_count
	^ GET_FLAG(rt2x00usb, INTERFACE_ENABLED_MONITOR)))
		return;

	/*
	 * Apply hardware packet filter.
	 */
	rt2x00_register_read(rt2x00usb, TXRX_CSR0, &reg);

	if (type == IEEE80211_IF_TYPE_IBSS
	|| type == IEEE80211_IF_TYPE_STA)
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_TO_DS, 1);
	else
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_TO_DS, 0);

	rt2x00_set_field32(&reg, TXRX_CSR0_DROP_CRC, 1);
	if (rt2x00usb->monitor_count) {
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_PHYSICAL, 0);
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_CONTROL, 0);
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_VERSION_ERROR, 0);
	} else {
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_PHYSICAL, 1);
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_CONTROL, 1);
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_VERSION_ERROR, 1);
	}

	rt2x00_set_field32(&reg, TXRX_CSR0_DROP_MULTICAST, 0);
	rt2x00_set_field32(&reg, TXRX_CSR0_DROP_BORADCAST, 0);

	rt2x00_register_write(rt2x00usb, TXRX_CSR0, reg);

	/*
	 * Enable promisc mode when in monitor mode.
	 */
	if (rt2x00usb->monitor_count)
		rt73usb_config_promisc(rt2x00usb, 1);

	/*
	 * Enable TSF counter.
	 */
	rt2x00_register_read(rt2x00usb, TXRX_CSR9, &reg);
	rt2x00_set_field32(&reg, TXRX_CSR9_TSF_TICKING, 1);
	if (type == IEEE80211_IF_TYPE_IBSS)
		rt2x00_set_field32(&reg, TXRX_CSR9_TSF_SYNC_MODE, 2);
	else if (type == IEEE80211_IF_TYPE_STA)
		rt2x00_set_field32(&reg, TXRX_CSR9_TSF_SYNC_MODE, 1);
	else
		rt2x00_set_field32(&reg, TXRX_CSR9_TSF_SYNC_MODE, 0);
	rt2x00_register_write(rt2x00usb, TXRX_CSR9, reg);

	/*
	 * Update working mode.
	 */
	if (type != IEEE80211_IF_TYPE_MNTR)
		rt2x00usb->interface_type = type;

	if (rt2x00usb->monitor_count)
		SET_FLAG(rt2x00usb, INTERFACE_ENABLED_MONITOR);
	else
		CLEAR_FLAG(rt2x00usb, INTERFACE_ENABLED_MONITOR);
}

static void rt73usb_config_channel(struct rt2x00_usb *rt2x00usb,
	int rf2, int channel, int freq, int txpower)
{
	u8 reg = 0;
	u32 rf1 = rt2x00usb->rf1;
	u32 rf3 = rt2x00usb->rf3;
	u32 rf4 = 0;

	/*
	 * Only continue when there is something to be done.
	 */
	if (channel == rt2x00usb->rx_params.channel)
		return;

	if (txpower == 0xff)
		txpower = rt2x00usb->tx_power;
	txpower = TXPOWER_TO_DEV(txpower);

	if (rt2x00_rf(&rt2x00usb->chip, RF5225)) {
		if (channel <= 14)
			rf3 = cpu_to_le32(0x00068455);
		else if (channel >= 36 && channel <= 48)
			rf3 = cpu_to_le32(0x0009be55);
		else if (channel >= 52 && channel <= 64)
			rf3 = cpu_to_le32(0x0009ae55);
		else if (channel >= 100 && channel <= 112)
			rf3 = cpu_to_le32(0x000bae55);
		else
			rf3 = cpu_to_le32(0x000bbe55);
	}

	if (channel < 14) {
		if (channel & 0x01)
			rf4 = cpu_to_le32(0x000fea0b);
		else
			rf4 = cpu_to_le32(0x000fea1f);
	} else if (channel == 14) {
		rf4 = cpu_to_le32(0x000fea13);
	} else {
		switch (channel) {
			case 36:
			case 56:
			case 116:
			case 136:
				rf4 = cpu_to_le32(0x000fea23);
				break;
			case 40:
			case 60:
			case 100:
			case 120:
			case 140:
				rf4 = cpu_to_le32(0x000fea03);
				break;
			case 44:
			case 64:
			case 104:
			case 124:
				rf4 = cpu_to_le32(0x000fea0b);
				break;
			case 48:
			case 108:
			case 128:
				rf4 = cpu_to_le32(0x000fea13);
				break;
			case 52:
			case 112:
			case 132:
				rf4 = cpu_to_le32(0x000fea1b);
				break;
			case 149:
				rf4 = cpu_to_le32(0x000fea1f);
				break;
			case 153:
				rf4 = cpu_to_le32(0x000fea27);
				break;
			case 157:
				rf4 = cpu_to_le32(0x000fea07);
				break;
			case 161:
				rf4 = cpu_to_le32(0x000fea0f);
				break;
			case 165:
				rf4 = cpu_to_le32(0x000fea17);
				break;
		}
	}

	INFO("Switching channel. RF1: 0x%08x, RF2: 0x%08x, RF3: 0x%08x, "
		"RF4: 0x%08x.\n", rf1, rf2, rf3, rf4);

	if (rt2x00_rf(&rt2x00usb->chip, RF2527)
	|| rt2x00_rf(&rt2x00usb->chip, RF5225))
		rf4 |= cpu_to_le32(0x00010000);

	/*
	 * Set TXpower.
	 */
	rt2x00_set_field32(&rf3, RF3_TXPOWER, txpower);

	/*
	 * Set Frequency offset.
	 */
	rt2x00_set_field32(&rf4, RF4_FREQ_OFFSET, rt2x00usb->freq_offset);

	rt2x00_bbp_read(rt2x00usb, 3, &reg);
	if (rt2x00_rf(&rt2x00usb->chip, RF5225)
	|| rt2x00_rf(&rt2x00usb->chip, RF2527))
		reg &= ~0x01;
	else
		reg |= 0x01;
	rt2x00_bbp_write(rt2x00usb, 3, reg);

	rt2x00_rf_write(rt2x00usb, rf1);
	rt2x00_rf_write(rt2x00usb, rf2);
	rt2x00_rf_write(rt2x00usb, rf3 & ~cpu_to_le32(0x00000004));
	rt2x00_rf_write(rt2x00usb, rf4);

	rt2x00_rf_write(rt2x00usb, rf1);
	rt2x00_rf_write(rt2x00usb, rf2);
	rt2x00_rf_write(rt2x00usb, rf3 | cpu_to_le32(0x00000004));
	rt2x00_rf_write(rt2x00usb, rf4);

	rt2x00_rf_write(rt2x00usb, rf1);
	rt2x00_rf_write(rt2x00usb, rf2);
	rt2x00_rf_write(rt2x00usb, rf3 & ~cpu_to_le32(0x00000004));
	rt2x00_rf_write(rt2x00usb, rf4);

	msleep(1);

	/*
	 * Update active info for RX.
	 */
	rt2x00usb->rx_params.freq = freq;
	rt2x00usb->rx_params.channel = channel;

	/*
	 * Update rf fields
	 */
	rt2x00usb->rf1 = rf1;
	rt2x00usb->rf2 = rf2;
	rt2x00usb->rf3 = rf3;
	rt2x00usb->rf4 = rf4;

	rt2x00usb->tx_power = txpower;
}

static void rt73usb_config_txpower(struct rt2x00_usb *rt2x00usb, int txpower)
{
	txpower = TXPOWER_TO_DEV(txpower);

	/*
	 * Only continue when there is something to be done.
	 */
	if (txpower == rt2x00usb->tx_power)
		return;

	rt2x00_set_field32(&rt2x00usb->rf3, RF3_TXPOWER, txpower);

	rt2x00_rf_write(rt2x00usb, rt2x00usb->rf1);
	rt2x00_rf_write(rt2x00usb, rt2x00usb->rf2);
	rt2x00_rf_write(rt2x00usb, rt2x00usb->rf3 & ~cpu_to_le32(0x00000004));
	rt2x00_rf_write(rt2x00usb, rt2x00usb->rf4);

	rt2x00_rf_write(rt2x00usb, rt2x00usb->rf1);
	rt2x00_rf_write(rt2x00usb, rt2x00usb->rf2);
	rt2x00_rf_write(rt2x00usb, rt2x00usb->rf3 | cpu_to_le32(0x00000004));
	rt2x00_rf_write(rt2x00usb, rt2x00usb->rf4);

	rt2x00_rf_write(rt2x00usb, rt2x00usb->rf1);
	rt2x00_rf_write(rt2x00usb, rt2x00usb->rf2);
	rt2x00_rf_write(rt2x00usb, rt2x00usb->rf3 & ~cpu_to_le32(0x00000004));
	rt2x00_rf_write(rt2x00usb, rt2x00usb->rf4);

	rt2x00usb->tx_power = txpower;
}

static void rt73usb_config_antenna(
	struct rt2x00_usb *rt2x00usb, int antenna, int phymode)
{
	u32 reg;
	u8 reg_r3;
	u8 reg_r4;
	u8 reg_r77;
	u8 frame_type;

	/*
	 * Only continue when there is something to be done.
	 */
	if (rt2x00usb->rx_params.antenna == antenna)
		return;

	rt2x00_register_read(rt2x00usb, PHY_CSR0, &reg);

	if (phymode == MODE_IEEE80211A) {
		if (GET_FLAG(rt2x00usb, CONFIG_EXTERNAL_LNA)) {
			rt2x00_bbp_write(rt2x00usb, 17, 0x38);
			rt2x00_bbp_write(rt2x00usb, 96, 0x78);
			rt2x00_bbp_write(rt2x00usb, 104, 0x48);
			rt2x00_bbp_write(rt2x00usb, 75, 0x80);
			rt2x00_bbp_write(rt2x00usb, 86, 0x80);
			rt2x00_bbp_write(rt2x00usb, 88, 0x80);
		} else {
			rt2x00_bbp_write(rt2x00usb, 17, 0x28);
			rt2x00_bbp_write(rt2x00usb, 96, 0x58);
			rt2x00_bbp_write(rt2x00usb, 104, 0x38);
			rt2x00_bbp_write(rt2x00usb, 75, 0xfe);
			rt2x00_bbp_write(rt2x00usb, 86, 0xfe);
			rt2x00_bbp_write(rt2x00usb, 88, 0xfe);
		}
		rt2x00_bbp_write(rt2x00usb, 35, 0x60);
		rt2x00_bbp_write(rt2x00usb, 97, 0x58);
		rt2x00_bbp_write(rt2x00usb, 98, 0x58);

		rt2x00_set_field32(&reg, PHY_CSR0_PA_PE_BG, 0);
		rt2x00_set_field32(&reg, PHY_CSR0_PA_PE_A, 1);
	} else {
		if (GET_FLAG(rt2x00usb, CONFIG_EXTERNAL_LNA)) {
			rt2x00_bbp_write(rt2x00usb, 17, 0x30);
			rt2x00_bbp_write(rt2x00usb, 96, 0x68);
			rt2x00_bbp_write(rt2x00usb, 104, 0x3c);
			rt2x00_bbp_write(rt2x00usb, 75, 0x80);
			rt2x00_bbp_write(rt2x00usb, 86, 0x80);
			rt2x00_bbp_write(rt2x00usb, 88, 0x80);
		} else {
			rt2x00_bbp_write(rt2x00usb, 17, 0x20);
			rt2x00_bbp_write(rt2x00usb, 96, 0x48);
			rt2x00_bbp_write(rt2x00usb, 104, 0x2c);
			rt2x00_bbp_write(rt2x00usb, 75, 0xfe);
			rt2x00_bbp_write(rt2x00usb, 86, 0xfe);
			rt2x00_bbp_write(rt2x00usb, 88, 0xfe);
		}
		rt2x00_bbp_write(rt2x00usb, 35, 0x50);
		rt2x00_bbp_write(rt2x00usb, 97, 0x48);
		rt2x00_bbp_write(rt2x00usb, 98, 0x48);

		rt2x00_set_field32(&reg, PHY_CSR0_PA_PE_BG, 1);
		rt2x00_set_field32(&reg, PHY_CSR0_PA_PE_A, 0);
	}

	rt2x00_register_write(rt2x00usb, PHY_CSR0, reg);

	rt2x00_bbp_read(rt2x00usb, 3, &reg_r3);
	rt2x00_bbp_read(rt2x00usb, 4, &reg_r4);
	rt2x00_bbp_read(rt2x00usb, 77, &reg_r77);

	reg_r3 &= ~0x01;
	reg_r4 &= ~0x23;

	frame_type = ~(GET_FLAG(rt2x00usb, CONFIG_FRAME_TYPE) << 5);

	if (rt2x00_rf(&rt2x00usb->chip, RF5226)
	|| rt2x00_rf(&rt2x00usb->chip, RF5225)) {
		if (antenna == 0) {
			/* Diversity. */
			reg_r4 |= 0x02;
			if (phymode != MODE_IEEE80211A)
				reg_r4 |= 0x20;
			reg_r4 &= frame_type;
		} else if (antenna == 1) {
			/* RX: Antenna B */
			/* TX: Antenna A */
			reg_r4 |= 0x01;
			reg_r4 &= frame_type;
			if (phymode == MODE_IEEE80211A)
				reg_r77 |= 0x03;
			else
				reg_r77 &= ~0x03;
			rt2x00_bbp_write(rt2x00usb, 77, reg_r77);
		} else if (antenna == 2) {
			/* RX: Antenna A */
			/* TX: Antenna B */
			reg_r4 |= 0x01;
			reg_r4 &= frame_type;
			if (phymode == MODE_IEEE80211A)
				reg_r77 &= ~0x03;
			else
				reg_r77 |= 0x03;
			rt2x00_bbp_write(rt2x00usb, 77, reg_r77);
		}
	} else if (rt2x00_rf(&rt2x00usb->chip, RF2528)
	|| rt2x00_rf(&rt2x00usb->chip, RF2527)) {
		if (antenna == 0) {
			/* Diversity. */
			reg_r4 |= 0x22;
			reg_r4 &= frame_type;
		} else if (antenna == 1) {
			/* RX: Antenna B */
			/* TX: Antenna A */
			reg_r4 |= 0x21;
			reg_r4 &= frame_type;
			reg_r77 &= ~0x03;
			rt2x00_bbp_write(rt2x00usb, 77, reg_r77);
		} else if (antenna == 2) {
			/* RX: Antenna A */
			/* TX: Antenna B */
			reg_r4 |= 0x21;
			reg_r4 &= frame_type;
			reg_r77 |= 0x03;
			rt2x00_bbp_write(rt2x00usb, 77, reg_r77);
		}
	}

	rt2x00_bbp_write(rt2x00usb, 3, reg_r3);
	rt2x00_bbp_write(rt2x00usb, 4, reg_r4);

	/*
	 * Update active info for RX.
	 */
	rt2x00usb->rx_params.antenna = antenna;
}

static void rt73usb_config_duration(
	struct rt2x00_usb *rt2x00usb, int short_slot_time)
{
	u32 reg;
	u32 value;

	short_slot_time = short_slot_time ? SHORT_SLOT_TIME : SLOT_TIME;

	rt2x00_register_read(rt2x00usb, MAC_CSR9, &reg);
	rt2x00_set_field32(&reg, MAC_CSR9_SLOT_TIME, short_slot_time);
	rt2x00_register_write(rt2x00usb, MAC_CSR9, reg);

	rt2x00_register_read(rt2x00usb, MAC_CSR8, &reg);
	rt2x00_set_field32(&reg, MAC_CSR8_SIFS, SIFS);
	rt2x00_set_field32(&reg, MAC_CSR8_SIFS_AFTER_RX_OFDM, 3);
	value = SIFS + get_duration(IEEE80211_HEADER + ACK_SIZE, 10);
	rt2x00_set_field32(&reg, MAC_CSR8_EIFS, value);
	rt2x00_register_write(rt2x00usb, MAC_CSR8, reg);

	rt2x00_register_read(rt2x00usb, TXRX_CSR0, &reg);
	rt2x00_set_field32(&reg, TXRX_CSR0_TSF_OFFSET, IEEE80211_HEADER);
	rt2x00_register_write(rt2x00usb, TXRX_CSR0, reg);

	rt2x00_register_read(rt2x00usb, TXRX_CSR4, &reg);
	rt2x00_set_field32(&reg, TXRX_CSR4_AUTORESPOND_ENABLE, 1);
	rt2x00_register_write(rt2x00usb, TXRX_CSR4, reg);
}

static void rt73usb_config_rate(struct rt2x00_usb *rt2x00usb, const int rate)
{
	struct ieee80211_conf *conf = ieee80211_get_hw_conf(
		usb_get_intfdata(rt2x00usb->usb_intf));
	u32 reg;
	u32 value;
	u32 preamble;

	preamble = DEVICE_RATE_FIELD(rate, PREAMBLE)
		? SHORT_PREAMBLE : PREAMBLE;

	reg = DEVICE_RATE_FIELD(rate, RATEMASK);

	rt2x00_register_write(rt2x00usb, TXRX_CSR5, cpu_to_le32(reg));

	rt2x00_register_read(rt2x00usb, TXRX_CSR0, &reg);
	value = SIFS + PLCP
		+ (2 * (conf->short_slot_time ? SHORT_SLOT_TIME : SLOT_TIME))
		+ preamble
		+ get_duration(ACK_SIZE, 10);
	rt2x00_set_field32(&reg, TXRX_CSR0_RX_ACK_TIMEOUT, value);
	rt2x00_register_write(rt2x00usb, TXRX_CSR0, reg);

	rt2x00_register_read(rt2x00usb, TXRX_CSR4, &reg);
	if (preamble == SHORT_PREAMBLE)
		rt2x00_set_field32(&reg, TXRX_CSR4_AUTORESPOND_PREAMBLE, 1);
	else
		rt2x00_set_field32(&reg, TXRX_CSR4_AUTORESPOND_PREAMBLE, 0);
	rt2x00_register_write(rt2x00usb, TXRX_CSR4, reg);
}

static void rt73usb_config_phymode(
	struct rt2x00_usb *rt2x00usb, const int phymode)
{
	struct ieee80211_rate *rate;

	/*
	 * Only continue when there is something to be done.
	 */
	if (rt2x00usb->rx_params.phymode == phymode)
		return;

	if (phymode == MODE_IEEE80211A
	&& (rt2x00_rf(&rt2x00usb->chip, RF5225)
	|| rt2x00_rf(&rt2x00usb->chip, RF5226)))
		rate = &rt2x00usb->hw.modes[2].rates[
			rt2x00usb->hw.modes[2].num_rates - 1];
	else if (phymode == MODE_IEEE80211B)
		rate = &rt2x00usb->hw.modes[1].rates[
			rt2x00usb->hw.modes[1].num_rates - 1];
	else
		rate = &rt2x00usb->hw.modes[0].rates[
			rt2x00usb->hw.modes[0].num_rates - 1];

	rt73usb_config_rate(rt2x00usb, rate->val2);

	/*
	 * Update physical mode for rx ring.
	 */
	rt2x00usb->rx_params.phymode = phymode;
}

static void rt73usb_config_mac_address(
	struct rt2x00_usb *rt2x00usb, void *addr)
{
	u32 reg[2] = {0, 0};

	rt2x00_set_field32(&reg[0], MAC_CSR2_BYTE0, ((u8*)addr)[0]);
	rt2x00_set_field32(&reg[0], MAC_CSR2_BYTE1, ((u8*)addr)[1]);
	rt2x00_set_field32(&reg[0], MAC_CSR2_BYTE2, ((u8*)addr)[2]);
	rt2x00_set_field32(&reg[0], MAC_CSR2_BYTE3, ((u8*)addr)[3]);
	rt2x00_set_field32(&reg[1], MAC_CSR3_BYTE4, ((u8*)addr)[4]);
	rt2x00_set_field32(&reg[1], MAC_CSR3_BYTE5, ((u8*)addr)[5]);
	rt2x00_set_field32(&reg[1], MAC_CSR3_UNICAST_TO_ME_MASK, 0xff);

	rt2x00_register_multiwrite(rt2x00usb, MAC_CSR2, &reg[0], sizeof(reg));
}

/*
 * Link tuning
 */
static void rt73usb_link_tuner(struct rt2x00_usb *rt2x00usb, char rssi)
{
	u32 reg;
	u8 reg_r17;
	u8 up_bound;
	u8 low_bound;

	/*
	 * Don't perform any tuning during scan.
	 */
	if (rt2x00usb->scan)
		return;

	if (rt2x00usb->rx_params.phymode == MODE_IEEE80211A) {
		up_bound = 0x48;
		low_bound = 0x28;
	} else {
		if (rssi > 38) {
			up_bound = 0x40;
			low_bound = 0x1c;
		} else if (rssi > 36) {
			up_bound = 0x20;
			low_bound = 0x1c;
		} else {
			up_bound = 0x1c;
			low_bound = 0x1c;
		}
	
		if (GET_FLAG(rt2x00usb, CONFIG_EXTERNAL_LNA)) {
			up_bound += 0x10;
			low_bound += 0x14;
		}
	}

	rt2x00_bbp_read(rt2x00usb, 17, &reg_r17);

	if (rssi >= 85) {
		if (reg_r17 != 0x60)
			rt2x00_bbp_write(rt2x00usb, 17, 0x60);
		return;
	} else if (rssi >= 62) {
		if (reg_r17 != up_bound)
			rt2x00_bbp_write(rt2x00usb, 17, up_bound);
		return;
	} else if (rssi >= 54) {
		low_bound += 0x10;
		if (reg_r17 != low_bound)
			rt2x00_bbp_write(rt2x00usb, 17, low_bound);
		return;
	} else if (rssi >= 46) {
		low_bound += 0x08;
		if (reg_r17 != low_bound)
			rt2x00_bbp_write(rt2x00usb, 17, low_bound);
		return;
	} else {
		up_bound -= 2 * (46 - rssi);
		if (up_bound < low_bound)
			up_bound =  low_bound;

		if (reg_r17 > up_bound) {
			rt2x00_bbp_write(rt2x00usb, 17, up_bound);
			return;
		}
	}

	rt2x00_register_read(rt2x00usb, STA_CSR1, &reg);
	reg = rt2x00_get_field32(reg, STA_CSR1_FALSE_CCA_ERROR);

	if (reg > 512 && reg_r17 < up_bound)
		rt2x00_bbp_write(rt2x00usb, 17, ++reg_r17);
	else if (reg < 100 && reg_r17 > low_bound)
		rt2x00_bbp_write(rt2x00usb, 17, --reg_r17);
}

/*
 * LED functions.
 */
static void rt73usb_enable_led(struct rt2x00_usb *rt2x00usb)
{
	u32 reg;

	rt2x00_register_read(rt2x00usb, MAC_CSR14, &reg);
	rt2x00_set_field32(&reg, MAC_CSR14_ON_PERIOD, 30);
	rt2x00_set_field32(&reg, MAC_CSR14_OFF_PERIOD, 70);
	rt2x00_register_write(rt2x00usb, MAC_CSR14, reg);

	rt2x00_set_field16(&rt2x00usb->led_reg, MCU_LEDCS_RADIO_STATUS, 1);
	if (rt2x00usb->rx_params.phymode == MODE_IEEE80211A)
		rt2x00_set_field16(
			&rt2x00usb->led_reg, MCU_LEDCS_LINK_A_STATUS, 1);
	else
		rt2x00_set_field16(
			&rt2x00usb->led_reg, MCU_LEDCS_LINK_BG_STATUS, 1);

	rt2x00_vendor_request(
		rt2x00usb, USB_LED_CONTROL, USB_VENDOR_REQUEST_OUT,
		0x00, rt2x00usb->led_reg, NULL, 0);
}

static void rt73usb_disable_led(struct rt2x00_usb *rt2x00usb)
{
	rt2x00_set_field16(&rt2x00usb->led_reg, MCU_LEDCS_RADIO_STATUS, 0);
	rt2x00_set_field16(&rt2x00usb->led_reg, MCU_LEDCS_LINK_BG_STATUS, 0);
	rt2x00_set_field16(&rt2x00usb->led_reg, MCU_LEDCS_LINK_A_STATUS, 0);

	rt2x00_vendor_request(
		rt2x00usb, USB_LED_CONTROL, USB_VENDOR_REQUEST_OUT,
		0x00, rt2x00usb->led_reg, NULL, 0);
}

static void rt73usb_activity_led(struct rt2x00_usb *rt2x00usb, char rssi)
{
	u32 led;

	if (rt2x00usb->led_mode != LED_MODE_SIGNAL_STRENGTH)
		return;

	if (rssi <= 30)
		led = 0;
	else if (rssi <= 39)
		led = 1;
	else if (rssi <= 49)
		led = 2;
	else if (rssi <= 53)
		led = 3;
	else if (rssi <= 63)
		led = 4;
	else
		led = 5;

	rt2x00_vendor_request(
		rt2x00usb, USB_LED_CONTROL, USB_VENDOR_REQUEST_OUT,
		led, rt2x00usb->led_reg, NULL, 0);
}

/*
 * Device state switch.
 * This will put the device to sleep, or awake it.
 */
static int rt73usb_set_state(
	struct rt2x00_usb *rt2x00usb, enum dev_state state)
{
	u32 reg;
	unsigned int i;
	char put_to_sleep;
	char current_state;

	put_to_sleep = (state != STATE_AWAKE);

	if (!put_to_sleep)
		rt2x00_vendor_request(rt2x00usb,
			USB_DEVICE_MODE, USB_VENDOR_REQUEST_OUT,
			0x00, USB_MODE_WAKEUP, NULL, 0);

	rt2x00_register_read(rt2x00usb, MAC_CSR12, &reg);
	rt2x00_set_field32(&reg, MAC_CSR12_FORCE_WAKEUP, !put_to_sleep);
	rt2x00_set_field32(&reg, MAC_CSR12_PUT_TO_SLEEP, put_to_sleep);
	rt2x00_register_write(rt2x00usb, MAC_CSR12, reg);

	if (put_to_sleep)
		rt2x00_vendor_request(rt2x00usb,
			USB_DEVICE_MODE, USB_VENDOR_REQUEST_OUT,
			0x00, USB_MODE_SLEEP, NULL, 0);

	/*
	 * Device is not guarenteed to be in the requested state yet.
	 * We must wait untill the register indicates that the
	 * device has entered the correct state.
	 */
	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		rt2x00_register_read(rt2x00usb, MAC_CSR12, &reg);
		current_state = rt2x00_get_field32(reg,
			MAC_CSR12_BBP_CURRENT_STATE);
		if (current_state == !put_to_sleep)
			return 0;
		msleep(10);
	}

	NOTICE("Device failed to enter state %d, "
		"current device state %d.\n", !put_to_sleep, current_state);

	return -EBUSY;
}

/*
 * Initialization functions.
 */
static int rt73usb_init_firmware_wait(struct rt2x00_usb *rt2x00usb)
{
	unsigned int i;

	for (i = 0; i < 150; i++) {
		if (GET_FLAG(rt2x00usb, FIRMWARE_FAILED))
			return -EIO;
		if (GET_FLAG(rt2x00usb, FIRMWARE_LOADED))
			return 0;
		msleep(20);
	}

	ERROR("Firmware loading timed out.\n");
	return -EIO;
}

static void rt73usb_init_firmware_cont(
	const struct firmware *fw, void *context)
{
	struct rt2x00_usb *rt2x00usb = context;
	unsigned int i;
	int status;
	u32 reg;
	u16 crc;

	if (!fw || !fw->size || !fw->data) {
		ERROR("Failed to load Firmware.\n");
		goto exit;
	}

	/*
	 * Wait for stable hardware.
	 */
	for (i = 0; i < 100; i++) {
		rt2x00_register_read(rt2x00usb, MAC_CSR0, &reg);
		if (reg)
			break;
		msleep(1);
	}

	if (!reg) {
		ERROR("Unstable hardware.\n");
		goto exit;
	}

	/*
	 * Validate the firmware using 16 bit CRC.
	 * The last 2 bytes of the firmware are the CRC
	 * so substract those 2 bytes from the CRC checksum,
	 * and set those 2 bytes to 0 when calculating CRC.
	 */
	reg = 0;
	crc = rt2x00crc(0, fw->data, fw->size - 2);
	crc = rt2x00crc(crc, (u8*)&reg, 2);

	if (crc != (fw->data[fw->size - 2] << 8 | fw->data[fw->size - 1])) {
		ERROR("Firmware CRC error.\n");
		goto exit;
	}

	DEBUG("Loading Firmware version: %d.%d.\n",
		fw->data[fw->size - 4], fw->data[fw->size - 3]);

	/*
	 * Write firmware to device.
	 */
	for (i = 0; i < fw->size; i += sizeof(u32))
		rt2x00_register_write(rt2x00usb, FIRMWARE_IMAGE_BASE + i,
			*((u32*)(fw->data + i)));

	/*
	 * Send firmware request to device to load firmware,
	 * use the rt2x00usb_vendor_request() function
	 * since we need to specify a long timeout time.
	 */
	status = rt2x00usb_vendor_request(rt2x00usb, USB_DEVICE_MODE,
		USB_VENDOR_REQUEST_OUT, 0x00, USB_MODE_FIRMWARE,
		NULL, 0, REGISTER_TIMEOUT_FIRMWARE);
	if (status  < 0) {
		ERROR("Failed to load Firmware error %d.\n", status);
		goto exit;
	}

	rt73usb_disable_led(rt2x00usb);

	SET_FLAG(rt2x00usb, FIRMWARE_LOADED);

	return;

exit:
	SET_FLAG(rt2x00usb, FIRMWARE_FAILED);
}

static int rt73usb_init_firmware(struct rt2x00_usb *rt2x00usb)
{
	/*
	 * Read correct firmware from harddisk.
	 */
	return _request_firmware_nowait(THIS_MODULE, 1,
		"rt73.bin", &rt2x00usb->usb_intf->dev, rt2x00usb,
		rt73usb_init_firmware_cont);
}

static int rt73usb_alloc_dma_ring(
	struct rt2x00_usb *rt2x00usb,
	struct data_ring *ring,
	void (*handler)(void *),
	const u16 max_entries,
	const u16 data_size,
	const u16 desc_size)
{
	unsigned int i;
	int status;

	/*
	 * Initialize work structure for deferred work.
	 */
	INIT_WORK(&ring->irq_work, handler, ring);

	ring->stats.limit = max_entries;
	ring->data_size = data_size;
	ring->desc_size = desc_size;

	/*
	 * Allocate all ring entries.
	 */
	ring->entry = kmalloc(ring->stats.limit * sizeof(struct data_entry),
		GFP_KERNEL);
	if (!ring->entry)
		return -ENOMEM;

	/*
	 * Allocate DMA memory for descriptor and buffer.
	 */
	ring->data_addr = usb_buffer_alloc(
		interface_to_usbdev(rt2x00usb->usb_intf),
		rt2x00_get_ring_size(ring), GFP_KERNEL, &ring->data_dma);
	if (!ring->data_addr) {
		kfree(ring->entry);
		return -ENOMEM;
	}

	/*
	 * Initialize all ring entries to contain valid
	 * addresses.
	 */
	status = 0;
	for (i = 0; i < ring->stats.limit; i++) {
		ring->entry[i].ring = ring;
		ring->entry[i].priv =
			(!status) ? usb_alloc_urb(0, GFP_KERNEL) :  NULL;
		if (!ring->entry[i].priv)
			status = -ENOMEM;
		ring->entry[i].skb = NULL;
		ring->entry[i].data_addr = ring->data_addr
			+ (i * ring->desc_size)
			+ (i * ring->data_size);
		ring->entry[i].data_dma = ring->data_dma
			+ (i * ring->desc_size)
			+ (i * ring->data_size);
	}

	return status;
}

static void rt73usb_free_ring(
	struct rt2x00_usb *rt2x00usb, struct data_ring *ring)
{
	unsigned int i;

	if (!ring->entry)
		goto exit;

	for (i = 0; i < ring->stats.limit; i++) {
		usb_kill_urb(rt2x00usb_urb(&ring->entry[i]));
		usb_free_urb(rt2x00usb_urb(&ring->entry[i]));
	}

	kfree(ring->entry);
	ring->entry = NULL;

exit:
	if (ring->data_addr)
		usb_buffer_free(
			interface_to_usbdev(rt2x00usb->usb_intf),
			rt2x00_get_ring_size(ring), ring->data_addr,
			ring->data_dma);
	ring->data_addr = NULL;
}

static int rt73usb_allocate_dma_rings(struct rt2x00_usb *rt2x00usb)
{
	if (rt73usb_alloc_dma_ring(
		rt2x00usb, &rt2x00usb->ring[RING_RX], rt73usb_rxdone,
		RX_ENTRIES, DATA_FRAME_SIZE, sizeof(struct rxd))
	|| rt73usb_alloc_dma_ring(
		rt2x00usb, &rt2x00usb->ring[RING_AC_BK], rt73usb_txdone,
		TX_ENTRIES, DATA_FRAME_SIZE, sizeof(struct txd))
	|| rt73usb_alloc_dma_ring(
		rt2x00usb, &rt2x00usb->ring[RING_AC_BE], rt73usb_txdone,
		TX_ENTRIES, DATA_FRAME_SIZE, sizeof(struct txd))
	|| rt73usb_alloc_dma_ring(
		rt2x00usb, &rt2x00usb->ring[RING_AC_VI], rt73usb_txdone,
		TX_ENTRIES, DATA_FRAME_SIZE, sizeof(struct txd))
	|| rt73usb_alloc_dma_ring(
		rt2x00usb, &rt2x00usb->ring[RING_AC_VO], rt73usb_txdone,
		 TX_ENTRIES, DATA_FRAME_SIZE, sizeof(struct txd))
	|| rt73usb_alloc_dma_ring(
		rt2x00usb, &rt2x00usb->ring[RING_PRIO], rt73usb_txdone,
		TX_ENTRIES, DATA_FRAME_SIZE, sizeof(struct txd))
	|| rt73usb_alloc_dma_ring(
		rt2x00usb, &rt2x00usb->ring[RING_BEACON], rt73usb_beacondone,
		BEACON_ENTRIES, MGMT_FRAME_SIZE, sizeof(struct txd)) ) {
		return -ENOMEM;
	}

	return 0;
}

static void rt73usb_free_rings(struct rt2x00_usb *rt2x00usb)
{
	rt73usb_free_ring(rt2x00usb, &rt2x00usb->ring[RING_RX]);
	rt73usb_free_ring(rt2x00usb, &rt2x00usb->ring[RING_AC_BK]);
	rt73usb_free_ring(rt2x00usb, &rt2x00usb->ring[RING_AC_BE]);
	rt73usb_free_ring(rt2x00usb, &rt2x00usb->ring[RING_AC_VI]);
	rt73usb_free_ring(rt2x00usb, &rt2x00usb->ring[RING_AC_VO]);
	rt73usb_free_ring(rt2x00usb, &rt2x00usb->ring[RING_PRIO]);
	rt73usb_free_ring(rt2x00usb, &rt2x00usb->ring[RING_BEACON]);
}

static void rt73usb_init_rxring(
	struct rt2x00_usb *rt2x00usb, unsigned short type)
{
	struct data_ring *ring = &rt2x00usb->ring[type];
	struct usb_device *usb_dev = interface_to_usbdev(rt2x00usb->usb_intf);
	struct urb *urb;
	unsigned int i;

	ring->type = type;

	for (i = 0; i < ring->stats.limit; i++) {
		urb = rt2x00usb_urb(&ring->entry[i]);

		urb->transfer_flags |= URB_NO_TRANSFER_DMA_MAP;
		urb->transfer_dma = ring->entry[i].data_dma;
		usb_fill_bulk_urb(
			urb,
			usb_dev,
			usb_rcvbulkpipe(usb_dev, 1),
			ring->entry[i].data_addr,
			ring->data_size + ring->desc_size,
			rt73usb_interrupt,
			&ring->entry[i]);
	}

	rt2x00_ring_index_clear(ring);
}

static void rt73usb_init_txring(
	struct rt2x00_usb *rt2x00usb, unsigned short type)
{
	struct data_ring *ring = &rt2x00usb->ring[type];
	struct urb *urb;
	unsigned int i;

	ring->type = type;

	for (i = 0; i < ring->stats.limit; i++) {
		urb = rt2x00usb_urb(&ring->entry[i]);

		urb->transfer_flags |= URB_NO_TRANSFER_DMA_MAP;
		urb->transfer_dma = ring->entry[i].data_dma;
		CLEAR_FLAGS(&ring->entry[i]);
	}

	rt2x00_ring_index_clear(ring);
}

static int rt73usb_init_rings(struct rt2x00_usb *rt2x00usb)
{
	rt73usb_init_rxring(rt2x00usb, RING_RX);
	rt73usb_init_txring(rt2x00usb, RING_AC_BK);
	rt73usb_init_txring(rt2x00usb, RING_AC_BE);
	rt73usb_init_txring(rt2x00usb, RING_AC_VI);
	rt73usb_init_txring(rt2x00usb, RING_AC_VO);
	rt73usb_init_txring(rt2x00usb, RING_PRIO);
	rt73usb_init_txring(rt2x00usb, RING_BEACON);

	return 0;
}

static int rt73usb_init_registers(struct rt2x00_usb *rt2x00usb)
{
	u32 reg;

	if (rt73usb_set_state(rt2x00usb, STATE_AWAKE))
		return -EBUSY;

	rt2x00_register_write(rt2x00usb, MAC_CSR10, cpu_to_le32(0x00000718));

	rt2x00_register_write(rt2x00usb, TXRX_CSR0, cpu_to_le32(0x025eb032));

	rt2x00_register_write(rt2x00usb, TXRX_CSR1, cpu_to_le32(0x9eaa9eaf));
	rt2x00_register_write(rt2x00usb, TXRX_CSR2, cpu_to_le32(0x8a8b8c8d));
	rt2x00_register_write(rt2x00usb, TXRX_CSR3, cpu_to_le32(0x00858687));

	rt2x00_register_write(rt2x00usb, TXRX_CSR7, cpu_to_le32(0x2e31353b));
	rt2x00_register_write(rt2x00usb, TXRX_CSR8, cpu_to_le32(0x2a2a2a2c));

	rt2x00_register_write(rt2x00usb, TXRX_CSR15, cpu_to_le32(0x0000000f));

	rt2x00_register_write(rt2x00usb, MAC_CSR6, cpu_to_le32(0x00000fff));

	rt2x00_register_write(rt2x00usb, MAC_CSR13, cpu_to_le32(0x00007f00));

	rt2x00_register_write(rt2x00usb, SEC_CSR0, cpu_to_le32(0x00000000));
	rt2x00_register_write(rt2x00usb, SEC_CSR1, cpu_to_le32(0x00000000));
	rt2x00_register_write(rt2x00usb, SEC_CSR5, cpu_to_le32(0x00000000));

	rt2x00_register_write(rt2x00usb, PHY_CSR1, cpu_to_le32(0x000023b0));
	rt2x00_register_write(rt2x00usb, PHY_CSR5, cpu_to_le32(0x00040a06));
	rt2x00_register_write(rt2x00usb, PHY_CSR6, cpu_to_le32(0x00080606));
	rt2x00_register_write(rt2x00usb, PHY_CSR7, cpu_to_le32(0x00000408));

	rt2x00_register_read(rt2x00usb, AC_TXOP_CSR0, &reg);
	rt2x00_set_field32(&reg, AC_TXOP_CSR0_AC0_TX_OP, 0);
	rt2x00_set_field32(&reg, AC_TXOP_CSR0_AC1_TX_OP, 0);
	rt2x00_register_write(rt2x00usb, AC_TXOP_CSR0, reg);

	rt2x00_register_read(rt2x00usb, AC_TXOP_CSR1, &reg);
	rt2x00_set_field32(&reg, AC_TXOP_CSR1_AC2_TX_OP, 192);
	rt2x00_set_field32(&reg, AC_TXOP_CSR1_AC3_TX_OP, 48);
	rt2x00_register_write(rt2x00usb, AC_TXOP_CSR1, reg);

	rt2x00_register_read(rt2x00usb, MAC_CSR9, &reg);
	rt2x00_set_field32(&reg, MAC_CSR9_CW_SELECT, 0);
	rt2x00_register_write(rt2x00usb, MAC_CSR9, reg);

	rt2x00_register_read(rt2x00usb, TXRX_CSR0, &reg);
	rt2x00_set_field32(&reg, TXRX_CSR0_AUTO_TX_SEQ, 1);
	rt2x00_register_write(rt2x00usb, TXRX_CSR0, reg);

	/*
	 * We must clear the error counters.
	 * These registers are cleared on read,
	 * so we may pass a useless variable to store the value.
	 */
	rt2x00_register_read(rt2x00usb, STA_CSR0, &reg);
	rt2x00_register_read(rt2x00usb, STA_CSR1, &reg);
	rt2x00_register_read(rt2x00usb, STA_CSR2, &reg);

	/*
	 * Reset MAC and BBP registers.
	 */
	reg = 0;
	rt2x00_set_field32(&reg, MAC_CSR1_SOFT_RESET, 1);
	rt2x00_set_field32(&reg, MAC_CSR1_BBP_RESET, 1);
	rt2x00_register_write(rt2x00usb, MAC_CSR1, reg);

	rt2x00_register_read(rt2x00usb, MAC_CSR1, &reg);
	rt2x00_set_field32(&reg, MAC_CSR1_SOFT_RESET, 0);
	rt2x00_set_field32(&reg, MAC_CSR1_BBP_RESET, 0);
	rt2x00_register_write(rt2x00usb, MAC_CSR1, reg);

	rt2x00_register_read(rt2x00usb, MAC_CSR1, &reg);
	rt2x00_set_field32(&reg, MAC_CSR1_HOST_READY, 1);
	rt2x00_register_write(rt2x00usb, MAC_CSR1, reg);

	return 0;
}

static int rt73usb_init_bbp(struct rt2x00_usb *rt2x00usb)
{
	u8 reg_id;
	u8 value;
	unsigned int i;

	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		rt2x00_bbp_read(rt2x00usb, 0, &value);
		if ((value != 0xff) && (value != 0x00))
			goto continue_csr_init;
		NOTICE("Waiting for BBP register.\n");
	}

	ERROR("BBP register access failed, aborting.\n");
	return -EACCES;

continue_csr_init:
	rt2x00_bbp_write(rt2x00usb, 3, 0x80);
	rt2x00_bbp_write(rt2x00usb, 15, 0x30);
	rt2x00_bbp_write(rt2x00usb, 17, 0x20);
	rt2x00_bbp_write(rt2x00usb, 21, 0xc8);
	rt2x00_bbp_write(rt2x00usb, 22, 0x38);
	rt2x00_bbp_write(rt2x00usb, 23, 0x06);
	rt2x00_bbp_write(rt2x00usb, 24, 0xfe);
	rt2x00_bbp_write(rt2x00usb, 25, 0x0a);
	rt2x00_bbp_write(rt2x00usb, 26, 0x0d);
	rt2x00_bbp_write(rt2x00usb, 32, 0x0b);
	rt2x00_bbp_write(rt2x00usb, 34, 0x12);
	rt2x00_bbp_write(rt2x00usb, 37, 0x07);
	rt2x00_bbp_write(rt2x00usb, 39, 0xf8);
	rt2x00_bbp_write(rt2x00usb, 41, 0x60);
	rt2x00_bbp_write(rt2x00usb, 53, 0x10);
	rt2x00_bbp_write(rt2x00usb, 54, 0x18);
	rt2x00_bbp_write(rt2x00usb, 60, 0x10);
	rt2x00_bbp_write(rt2x00usb, 61, 0x04);
	rt2x00_bbp_write(rt2x00usb, 62, 0x04);
	rt2x00_bbp_write(rt2x00usb, 75, 0xfe);
	rt2x00_bbp_write(rt2x00usb, 86, 0xfe);
	rt2x00_bbp_write(rt2x00usb, 88, 0xfe);
	rt2x00_bbp_write(rt2x00usb, 90, 0x0f);
	rt2x00_bbp_write(rt2x00usb, 99, 0x00);
	rt2x00_bbp_write(rt2x00usb, 102, 0x16);
	rt2x00_bbp_write(rt2x00usb, 107, 0x04);

	DEBUG("Start initialization from EEPROM...\n");
	for (i = 0; i < EEPROM_BBP_SIZE; i++) {
		if (rt2x00usb->eeprom[i] != 0xffff
		&& rt2x00usb->eeprom[i] != 0x0000) {
			reg_id = rt2x00_get_field16(
				rt2x00usb->eeprom[i], EEPROM_BBP_REG_ID);
			value = rt2x00_get_field16(
				rt2x00usb->eeprom[i], EEPROM_BBP_VALUE);
			DEBUG("BBP: 0x%02x, value: 0x%02x.\n", reg_id, value);
			rt2x00_bbp_write(rt2x00usb, reg_id, value);
		}
	}
	DEBUG("...End initialization from EEPROM.\n");

	return 0;
}

static int rt73usb_init_channel_time(struct rt2x00_usb *rt2x00usb)
{
	unsigned long jiffies_start;
	unsigned long jiffies_end;

	/*
	 * Determine channel_change_time
	 * by measuring the time it takes
	 * to switch the channel.
	 */
	jiffies_start = jiffies;
	rt73usb_config_channel(rt2x00usb,
		rt2x00usb->hw.modes[0].channels[0].val,
		rt2x00usb->hw.modes[0].channels[0].chan,
		rt2x00usb->hw.modes[0].channels[0].freq, 
		rt2x00usb->hw.modes[0].channels[0].power_level);
	jiffies_end = jiffies;

	rt2x00usb->hw.channel_change_time =
		jiffies_to_usecs((long)jiffies_end - (long)jiffies_start);

	NOTICE("Channel change time has been set to %d.\n",
		rt2x00usb->hw.channel_change_time);

	return 0;
}

/*
 * Device initialization functions.
 */
static int rt73usb_initialize(struct rt2x00_usb *rt2x00usb)
{
	if (GET_FLAG(rt2x00usb, DEVICE_INITIALIZED))
		return 0;

	/*
	 * We must wait on the firmware before
	 * we can safely continue.
	 */
	if (rt73usb_init_firmware_wait(rt2x00usb))
		return -ENODEV;

	/*
	 * Allocate all data rings.
	 */
	if (rt73usb_allocate_dma_rings(rt2x00usb)) {
		ERROR("DMA allocation failed.\n");
		goto exit_fail;
	}

	/*
	 * Initialize all registers.
	 */
	if (rt73usb_init_rings(rt2x00usb)
	|| rt73usb_init_registers(rt2x00usb)
	|| rt73usb_init_bbp(rt2x00usb)) {
		ERROR("Register initialization failed.\n");
		goto exit_fail;
	}

	/*
	 * Determine channel change time.
	 */
	if (rt73usb_init_channel_time(rt2x00usb))
		goto exit_fail;

	SET_FLAG(rt2x00usb, DEVICE_INITIALIZED);

	return 0;

exit_fail:
	rt73usb_free_rings(rt2x00usb);

	return -EIO;
}

static void rt73usb_uninitialize(struct rt2x00_usb *rt2x00usb)
{
	if (!GET_FLAG(rt2x00usb, DEVICE_INITIALIZED))
		return;

	/*
	 * Cancel scanning.
	 */
	if (rt2x00usb->scan) {
		rt2x00usb->scan->status = SCANNING_CANCELLED;
		complete_all(&rt2x00usb->scan->completion);
	}

	/*
	 * Flush out all pending work.
	 */
	flush_workqueue(rt2x00usb->workqueue);

	/*
	 * Free DMA rings.
	 */
	rt73usb_free_rings(rt2x00usb);

	CLEAR_FLAG(rt2x00usb, DEVICE_INITIALIZED);
}

/*
 * Radio control functions.
 */
static int rt73usb_enable_radio(struct rt2x00_usb *rt2x00usb)
{
	struct data_ring *ring;
	u32 reg;
	unsigned int i;

	/*
	 * Don't enable the radio twice.
	 */
	if (GET_FLAG(rt2x00usb, DEVICE_ENABLED_RADIO))
		return 0;

	/*
	 * Check if the hardware has been initialized,
	 * if not then do it now.
	 */
	if (!GET_FLAG(rt2x00usb, DEVICE_INITIALIZED))
		if (rt73usb_initialize(rt2x00usb))
			return -EIO;

	rt2x00_register_write(rt2x00usb, MAC_CSR10, cpu_to_le32(0x00000718));

	/*
	 * Enable RX.
	 */
	rt2x00_register_read(rt2x00usb, TXRX_CSR0, &reg);
	rt2x00_set_field32(&reg, TXRX_CSR0_DISABLE_RX, 0);
	rt2x00_register_write(rt2x00usb, TXRX_CSR0, reg);

	ring = &rt2x00usb->ring[RING_RX];
	for (i = 0; i < ring->stats.limit; i++) {
		SET_FLAG(&ring->entry[i], ENTRY_OWNER_NIC);
		usb_submit_urb(rt2x00usb_urb(&ring->entry[i]), GFP_ATOMIC);
	}

	/*
	 * Enable LED
	 */
	rt73usb_enable_led(rt2x00usb);

	SET_FLAG(rt2x00usb, DEVICE_ENABLED_RADIO);

	return 0;
}

static void rt73usb_disable_radio(struct rt2x00_usb *rt2x00usb)
{
	struct data_ring *ring;
	u32 reg;
	unsigned int i;

	if (!GET_FLAG(rt2x00usb, DEVICE_ENABLED_RADIO))
		return;

	rt2x00_register_write(rt2x00usb, MAC_CSR10, cpu_to_le32(0x00001818));

	/*
	 * Cancel RX and TX.
	 */
	rt2x00_register_read(rt2x00usb, TXRX_CSR0, &reg);
	rt2x00_set_field32(&reg, TXRX_CSR0_DISABLE_RX, 1);
	rt2x00_register_write(rt2x00usb, TXRX_CSR0, reg);

	rt2x00_vendor_request(rt2x00usb, USB_RX_CONTROL,
		USB_VENDOR_REQUEST_OUT, 0x00, 0x00, NULL, 0);

	ring = &rt2x00usb->ring[RING_RX];
	for (i = 0; i < ring->stats.limit; i++)
		usb_kill_urb(rt2x00usb_urb(&ring->entry[i]));

	ring = &rt2x00usb->ring[RING_AC_BK];
	for (i = 0; i < ring->stats.limit; i++)
		usb_kill_urb(rt2x00usb_urb(&ring->entry[i]));

	ring = &rt2x00usb->ring[RING_AC_BE];
	for (i = 0; i < ring->stats.limit; i++)
		usb_kill_urb(rt2x00usb_urb(&ring->entry[i]));

	ring = &rt2x00usb->ring[RING_AC_VI];
	for (i = 0; i < ring->stats.limit; i++)
		usb_kill_urb(rt2x00usb_urb(&ring->entry[i]));

	ring = &rt2x00usb->ring[RING_AC_VO];
	for (i = 0; i < ring->stats.limit; i++)
		usb_kill_urb(rt2x00usb_urb(&ring->entry[i]));

	ring = &rt2x00usb->ring[RING_PRIO];
	for (i = 0; i < ring->stats.limit; i++)
		usb_kill_urb(rt2x00usb_urb(&ring->entry[i]));

	ring = &rt2x00usb->ring[RING_BEACON];
	for (i = 0; i < ring->stats.limit; i++)
		usb_kill_urb(rt2x00usb_urb(&ring->entry[i]));

	/*
	 * Disable LED
	 */
	rt73usb_disable_led(rt2x00usb);

	CLEAR_FLAG(rt2x00usb, DEVICE_ENABLED_RADIO);
}

/*
 * TX descriptor initialization
 */
static void rt73usb_write_tx_desc(
	struct rt2x00_usb *rt2x00usb, struct txd *txd,
	struct sk_buff *skb, struct ieee80211_tx_control *control)
{
	struct ieee80211_hdr *ieee80211hdr = (struct ieee80211_hdr*)skb->data;
	struct data_ring *ring;
	u32 length;
	u32 residual;
	u16 length_high;
	u16 length_low;
	u16 signal;
	u16 service;
	u8 rate;

	rt2x00_set_field32(&txd->word0, TXD_W0_VALID, 1);
	rt2x00_set_field32(&txd->word0, TXD_W0_ACK, !control->no_ack);

	ring = rt2x00usb_get_ring(rt2x00usb, control->queue);
	if (unlikely(!ring))
		return;

	if (control->queue == IEEE80211_TX_QUEUE_BEACON)
		rt2x00_set_field32(&txd->word0, TXD_W0_TIMESTAMP, 1);
	else
		rt2x00_set_field32(&txd->word0, TXD_W0_TIMESTAMP, 0);

	/*
	 * TODO: IFS can be various values, where can we find
	 * which one we want to use?
	 */
	rt2x00_set_field32(&txd->word0, TXD_W0_IFS, 0);

	/*
	 * TODO: How can we determine if we want long retry or short retry?
	 */
	rt2x00_set_field32(&txd->word0, TXD_W0_RETRY_MODE, 0);

	if (ieee80211_get_morefrag(ieee80211hdr))
		rt2x00_set_field32(&txd->word0, TXD_W0_MORE_FRAG, 1);
	else
		rt2x00_set_field32(&txd->word0, TXD_W0_MORE_FRAG, 0);

	rt2x00_set_field32(&txd->word0, TXD_W0_DATABYTE_COUNT, skb->len);

	if (control->queue == IEEE80211_TX_QUEUE_BEACON)
		rt2x00_set_field32(&txd->word1,
			TXD_W1_HOST_Q_ID, 15);
	else if (control->queue == IEEE80211_TX_QUEUE_SVP)
		rt2x00_set_field32(&txd->word1,
			TXD_W1_HOST_Q_ID, 13);
	else
		rt2x00_set_field32(&txd->word1,
			TXD_W1_HOST_Q_ID, control->queue);

	if (rt2x00_require_sequence(ieee80211hdr))
		rt2x00_set_field32(&txd->word1, TXD_W1_HW_SEQUENCE, 1);
	else
		rt2x00_set_field32(&txd->word1, TXD_W1_HW_SEQUENCE, 0);

	rt2x00_set_field32(&txd->word1, TXD_W1_AIFSN, ring->tx_params.aifs);
	rt2x00_set_field32(&txd->word1, TXD_W1_CWMIN, ring->tx_params.cw_min);
	rt2x00_set_field32(&txd->word1, TXD_W1_CWMAX, ring->tx_params.cw_max);

	rt2x00_set_field32(&txd->word5, TXD_W5_TX_POWER, control->power_level);

	/*
	 * Add 4 bytes for FCS.
	 */
	length = skb->len + FCS_LEN;

	/*
	 * Check if we are working with an OFDM rate,
	 * this can be done by checking if bit 4 or higher
	 * is set in the ratemask.
	 */
	if (DEVICE_RATE_FIELD(control->tx_rate, RATEMASK) & 0x0ff0) {
		rt2x00_set_field32(&txd->word0, TXD_W0_OFDM, 1);

		/*
		 * Convert length to microseconds.
		 */
		residual = 0;
		length_high = (length >> 6) & 0x3f;
		length_low = (length & 0x3f);

	} else {
		rate = DEVICE_RATE_FIELD(control->tx_rate, RATE);

		rt2x00_set_field32(&txd->word0, TXD_W0_OFDM, 0);

		residual = get_duration_res(length, rate);
		length = get_duration(length, rate);

		if (residual != 0)
			length++;

		length_high = length >> 8;
		length_low = length & 0xff;
	}

	signal = 0x8500 | DEVICE_RATE_FIELD(control->tx_rate, PLCP);
	if (DEVICE_RATE_FIELD(control->tx_rate, PREAMBLE))
		signal |= 0x0008;

	service = 0x04;
	if (residual <= (8 % 11))
		service |= 0x80;

	rt2x00_set_field32(&txd->word2, TXD_W2_PLCP_SIGNAL, signal);
	rt2x00_set_field32(&txd->word2, TXD_W2_PLCP_SERVICE, service);
	rt2x00_set_field32(&txd->word2, TXD_W2_PLCP_LENGTH_LOW, length_low);
	rt2x00_set_field32(&txd->word2, TXD_W2_PLCP_LENGTH_HIGH, length_high);

	/*
	 * Set this last, after this the device can start transmitting the packet.
	 */
	rt2x00_set_field32(&txd->word5, TXD_W5_WAITING_DMA_DONE_INT, 1);
}

/*
 * Interrupt functions.
 */
static void rt73usb_beacondone(void *data)
{
	struct data_ring *ring = data;
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(ring->net_dev);
	struct sk_buff *skb;
	struct ieee80211_tx_control beacon;

	memset(&beacon, 0x00, sizeof(beacon));

	skb = ieee80211_beacon_get(ring->net_dev,
		rt2x00usb->interface_id, &beacon);
	if (!skb)
		return;

	rt73usb_beacon_update(ring->net_dev, skb, &beacon);

	dev_kfree_skb_any(skb);
}

static void rt73usb_rxdone(void *data)
{
	struct data_ring *ring = data;
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(ring->net_dev);
	struct data_entry *entry;
	struct sk_buff *skb;
	struct rxd *rxd;
	struct urb *urb;
	u16 size;
	u8 rssi_count;
	char total_rssi;

	/*
	 * Initialize variable for average RSSI calculation.
	 */
	rssi_count = 0;
	total_rssi = 0;

	while (1) {
		entry = rt2x00_get_data_entry(ring);
		rxd = rt2x00usb_rxdesc_addr(entry);
		urb = rt2x00usb_urb(entry);

		if (GET_FLAG(entry, ENTRY_OWNER_NIC))
			break;

		/*
		 * There has been a problem. Ignore packet.
		 */
		if (urb->status) {
			SET_FLAG(entry, ENTRY_OWNER_NIC);
			usb_submit_urb(urb, GFP_ATOMIC);
			rt2x00_ring_index_inc(ring);
			continue;
		}

		/*
		 * Received USB packets have 4 bytes of extra data.
		 */
		size = rt2x00_get_field32(rxd->word0, RXD_W0_DATABYTE_COUNT);

		/*
		 * TODO: Don't we need to keep statistics
		 * updated about events like CRC and physical errors?
		 */
		if (!rt2x00_get_field32(rxd->word0, RXD_W0_CRC)
		&& !rt2x00_get_field32(rxd->word0, RXD_W0_CIPHER_ERROR)) {
			skb = dev_alloc_skb(size + NET_IP_ALIGN);
			if (!skb)
				break;

			skb_reserve(skb, NET_IP_ALIGN);

			memcpy(skb_put(skb, size),
				rt2x00usb_rxdata_addr(entry), size);

			rt2x00usb->rx_params.ssi =
				rt2x00_get_field32(rxd->word1, RXD_W1_RSSI);

			__ieee80211_rx(ring->net_dev,
				skb, &rt2x00usb->rx_params);

			rssi_count++;
			total_rssi += rt2x00usb->rx_params.ssi;
		}

		SET_FLAG(entry, ENTRY_OWNER_NIC);
		usb_submit_urb(urb, GFP_ATOMIC);

		rt2x00_ring_index_inc(ring);
	}

	/*
	 * Tune link for optimal performance.
	 */
	if (total_rssi && rssi_count)
		rt73usb_link_tuner(rt2x00usb, total_rssi / rssi_count);

	/*
	 * Update LED.
	 */
	rt73usb_activity_led(rt2x00usb, total_rssi);
}

static void rt73usb_txdone(void *data)
{
	struct data_ring *ring = data;
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(ring->net_dev);
	struct data_entry *entry;
	struct txd *txd;
	struct urb *urb;
	int ack;

	while (!rt2x00_ring_empty(ring)) {
		entry = rt2x00_get_data_entry_done(ring);
		txd = rt2x00usb_txdesc_addr(entry);
		urb = rt2x00usb_urb(entry);

		if (GET_FLAG(entry, ENTRY_OWNER_NIC))
			break;

		ack = rt2x00_get_field32(txd->word0, TXD_W0_ACK);
	
		/*
		 * TODO: How can te below field be set correctly?
		 */
		entry->tx_status.tx_filtered = 0;
	
		entry->tx_status.queue_length = entry->ring->stats.limit;
	
		/*
		 * Check if we have received an
		 * ACK response when ACK was requested and status
		 * was succesfull.
		 */
		entry->tx_status.ack = 0;
		if (ack && (urb->status == TX_SUCCESS))
			entry->tx_status.ack = 1;
		else {
			rt2x00usb->low_level_stats.dot11ACKFailureCount++;
			entry->tx_status.excessive_retries++;
		}
	
		rt2x00_bbp_read(rt2x00usb, 32,
			(u8*)&entry->tx_status.ack_signal);
	
		ieee80211_tx_status(ring->net_dev,
			entry->skb, &entry->tx_status);
	
		entry->skb = NULL;
	
		rt2x00_ring_index_done_inc(entry->ring);
	}

	/*
	 * Check if we are waiting on an empty queue
	 * to start scanning.
	 */
	if (rt2x00usb->scan
	&& rt2x00_ring_empty(&rt2x00usb->ring[RING_AC_BK])
	&& rt2x00_ring_empty(&rt2x00usb->ring[RING_AC_BE])
	&& rt2x00_ring_empty(&rt2x00usb->ring[RING_AC_VI])
	&& rt2x00_ring_empty(&rt2x00usb->ring[RING_AC_VO])
	&& rt2x00_ring_empty(&rt2x00usb->ring[RING_PRIO])) {
		rt2x00usb->scan->status = SCANNING_READY;
		complete(&rt2x00usb->scan->completion);
	}
}

static void rt73usb_interrupt(struct urb *urb, struct pt_regs *regs)
{
	struct data_entry *entry = (struct data_entry*)urb->context;
	struct rt2x00_usb *rt2x00usb =
		ieee80211_dev_hw_data(entry->ring->net_dev);

	if (!GET_FLAG(rt2x00usb, DEVICE_ENABLED_RADIO))
		return;

	CLEAR_FLAG(entry, ENTRY_OWNER_NIC);

	if (urb->status)
		return;

	queue_work(rt2x00usb->workqueue, &entry->ring->irq_work);
}

/*
 * IEEE80211 stack callback functions.
 */
static int rt73usb_tx(struct net_device *net_dev,
	struct sk_buff *skb, struct ieee80211_tx_control *control)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);
	struct usb_device *usb_dev = interface_to_usbdev(rt2x00usb->usb_intf);
	struct data_ring *ring;
	struct data_entry *entry;
	struct txd *txd;

	/*
	 * Determine which ring to put packet on.
	 */
	ring = rt2x00usb_get_ring(rt2x00usb, control->queue);
	if (unlikely(!ring)) {
		ERROR("Attempt to send packet over invalid queue %d.\n"
			"Please file bug report to %s.\n",
			control->queue, DRV_PROJECT);
		return NET_XMIT_DROP;
	}

	if (rt2x00_ring_full(ring))
		return NET_XMIT_DROP;

	entry = rt2x00_get_data_entry(ring);
	txd = rt2x00usb_txdesc_addr(entry);

	memcpy(rt2x00usb_txdata_addr(entry), skb->data, skb->len);
	rt73usb_write_tx_desc(rt2x00usb, txd, skb, control);
	entry->skb = skb;

	SET_FLAG(entry, ENTRY_OWNER_NIC);
	usb_fill_bulk_urb(
		rt2x00usb_urb(entry),
		usb_dev,
		usb_sndbulkpipe(usb_dev, 1),
		entry->data_addr,
		skb->len + ring->desc_size,
		rt73usb_interrupt,
		entry);
	usb_submit_urb(rt2x00usb_urb(entry), GFP_ATOMIC);

	rt2x00_ring_index_inc(ring);

	return 0;
}

static int rt73usb_reset(struct net_device *net_dev)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);

	rt73usb_disable_radio(rt2x00usb);

	rt73usb_init_rings(rt2x00usb);

	return rt73usb_enable_radio(rt2x00usb);
}

static int rt73usb_add_interface(struct net_device *net_dev,
	struct ieee80211_if_init_conf *conf)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);

	/*
	 * We support muliple monitor mode interfaces.
	 * All we need to do is keep increase the
	 * monitor_counter.
	 */
	if (conf->type == IEEE80211_IF_TYPE_MNTR) {
		rt2x00usb->monitor_count++;
		rt2x00usb->interface_promisc = 1;
		goto exit;
	}

	/*
	 * We only support 1 non-monitor interface.
	 */
	if (GET_FLAG(rt2x00usb, INTERFACE_INITIALIZED))
		return -ENOBUFS;

	rt2x00usb->interface_id = conf->if_id;

	rt73usb_config_type(rt2x00usb, conf->type);
	rt73usb_config_mac_address(rt2x00usb, conf->mac_addr);

	SET_FLAG(rt2x00usb, INTERFACE_INITIALIZED);

exit:
	/*
	 * Enable radio when this is the first
	 * interface that is brought up.
	 */
	if (!GET_FLAG(rt2x00usb, DEVICE_ENABLED_RADIO))
		return rt73usb_enable_radio(rt2x00usb);
	return 0;
}

static void rt73usb_remove_interface(struct net_device *net_dev,
	struct ieee80211_if_init_conf *conf)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);

	/*
	 * We support muliple monitor mode interfaces.
	 * Decrease the counter to remove one monitor
	 * interface.
	 */
	if (conf->type == IEEE80211_IF_TYPE_MNTR) {
		rt2x00usb->monitor_count--;
		goto exit;
	}

	/*
	 * We only support 1 non-monitor interface.
	 */
	if (!GET_FLAG(rt2x00usb, INTERFACE_INITIALIZED))
		return;

	rt2x00usb->interface_id = 0;
	rt2x00usb->interface_type = -EINVAL;

	CLEAR_FLAG(rt2x00usb, INTERFACE_INITIALIZED);

exit:
	/*
	 * Disable radio if this was the last interface
	 * that was working with this device.
	 */
	if (!rt2x00usb->monitor_count)
		rt73usb_disable_radio(rt2x00usb);
}

static void rt73usb_config_update(void *data)
{
	struct rt2x00_usb *rt2x00usb = data;
	struct net_device *net_dev = usb_get_intfdata(rt2x00usb->usb_intf);
	struct ieee80211_conf *conf = ieee80211_get_hw_conf(net_dev);
	u32 reg;

	/*
	 * Check if we need to disable the radio,
	 * if this is not the case, at least the RX must be disabled.
	 */
	if (GET_FLAG(rt2x00usb, DEVICE_ENABLED_RADIO)) {
		if (!conf->radio_enabled)
			rt73usb_disable_radio(rt2x00usb);
		else {
			rt2x00_register_read(rt2x00usb, TXRX_CSR0, &reg);
			rt2x00_set_field32(&reg, TXRX_CSR0_DISABLE_RX, 1);
			rt2x00_register_write(rt2x00usb, TXRX_CSR0, reg);
		}
	}

	rt73usb_config_channel(rt2x00usb,
		conf->channel_val, conf->channel, conf->freq,
		conf->power_level);
	rt73usb_config_txpower(rt2x00usb, conf->power_level);
	rt73usb_config_antenna(rt2x00usb, conf->antenna_sel, conf->phymode);
	rt73usb_config_duration(rt2x00usb, conf->short_slot_time);
	rt73usb_config_phymode(rt2x00usb, conf->phymode);

	/*
	 * Reenable RX only if the radio should is on.
	 */
	if (GET_FLAG(rt2x00usb, DEVICE_ENABLED_RADIO)) {
		rt2x00_register_read(rt2x00usb, TXRX_CSR0, &reg);
		rt2x00_set_field32(&reg, TXRX_CSR0_DISABLE_RX, 0);
		rt2x00_register_write(rt2x00usb, TXRX_CSR0, reg);
	} else if (conf->radio_enabled) {
		if (rt73usb_enable_radio(rt2x00usb))
			return;
	}
}

static int rt73usb_config(
	struct net_device *net_dev, struct ieee80211_conf *conf)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);

	/*
	 * Queue work.
	 */
	return !queue_work(rt2x00usb->workqueue, &rt2x00usb->config_work);
}

static void rt73usb_interface_update(void *data)
{
	struct rt2x00_usb *rt2x00usb = data;

	rt73usb_config_bssid(rt2x00usb, &rt2x00usb->interface_bssid[0]);
	rt73usb_config_promisc(rt2x00usb, rt2x00usb->interface_promisc);
}

static int rt73usb_config_interface(struct net_device *net_dev, int if_id,
	struct ieee80211_if_conf *conf)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);

	/*
	 * When configuring monitor mode, we are done now.
	 * but if we are configuring another mode it must be
	 * equal to the interface that has been added.
	 */
	if (conf->type == IEEE80211_IF_TYPE_MNTR)
		return 0;
	else if (conf->type != rt2x00usb->interface_type)
		return -EINVAL;

	if (conf->bssid) {
		memcpy(&rt2x00usb->interface_bssid, conf->bssid, ETH_ALEN);
		return !queue_work(rt2x00usb->workqueue,
				&rt2x00usb->config_work);
	}

	return 0;
}

static void rt73usb_set_multicast_list(struct net_device *net_dev,
	unsigned short flags, int mc_count)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);

	/*
	 * Monitor mode works with PROMISC mode forced on,
	 * so there is nothing to be done here.
	 */
	if (rt2x00usb->monitor_count)
		return;

	if (GET_FLAG(rt2x00usb, INTERFACE_ENABLED_PROMISC)) {
		if (!(flags & IFF_PROMISC)) {
			rt2x00usb->interface_promisc = 0;
			queue_work(rt2x00usb->workqueue,
				&rt2x00usb->config_work);
		}
	} else {
		if (flags & IFF_PROMISC) {
			rt2x00usb->interface_promisc = 1;
			queue_work(rt2x00usb->workqueue,
				&rt2x00usb->config_work);
		}
	}
}

static void rt73usb_scan(void *data)
{
	struct rt2x00_usb *rt2x00usb = data;

	if (unlikely(!rt2x00usb->scan))
		return;

	/*
	 * Before we can start switch the channel for scanning
	 * we need to wait untill all TX rings are empty to
	 * guarentee that all frames are send on the correct channel.
	 */
	if (rt2x00usb->scan->status != SCANNING_READY)
		wait_for_completion(&rt2x00usb->scan->completion);

	/*
	 * Check if this scan has been cancelled while
	 * work was still scheduled.
	 */
	if (rt2x00usb->scan->status == SCANNING_CANCELLED)
		goto exit;

	/*
	 * Switch channel and update active info for RX.
	 */
	if (rt2x00usb->scan->state == IEEE80211_SCAN_START) {
		rt73usb_config_phymode(rt2x00usb,
			rt2x00usb->scan->conf.scan_phymode);

		rt73usb_config_channel(rt2x00usb,
			rt2x00usb->scan->conf.scan_channel_val,
			rt2x00usb->scan->conf.scan_channel,
			rt2x00usb->scan->conf.scan_freq,
			rt2x00usb->scan->conf.scan_power_level);
	} else {
		rt73usb_config_phymode(rt2x00usb,
			rt2x00usb->scan->conf.running_phymode);

		rt73usb_config_channel(rt2x00usb,
			rt2x00usb->scan->conf.running_channel_val,
			rt2x00usb->scan->conf.running_channel,
			rt2x00usb->scan->conf.running_freq,
			rt2x00usb->scan->conf.scan_power_level);
	}

exit:
	kfree(rt2x00usb->scan);
	rt2x00usb->scan = NULL;
}

static int rt73usb_passive_scan(struct net_device *net_dev,
	int state, struct ieee80211_scan_conf *conf)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);

	/*
	 * Check if we are not busy with the previous
	 * passive scan request.
	 */
	if (rt2x00usb->scan)
		return -EBUSY;

	/*
	 * Check if we need to disable the radio
	 */
	if (!GET_FLAG(rt2x00usb, DEVICE_ENABLED_RADIO))
		return -EIO;

	/*
	 * Allocate scanning structure to store scanning info.
	 */
	rt2x00usb->scan = kmalloc(sizeof(struct scanning), GFP_ATOMIC);
	if (!rt2x00usb->scan)
		return -ENOMEM;

	/*
	 * Check if we have to send a packet before the
	 * channel switch.
	 */
	if (conf->skb) {
		if (rt73usb_tx(net_dev, conf->skb, conf->tx_control))
			goto exit;
	}

	/*
	 * Initialize Scanning structure.
	 */
	init_completion(&rt2x00usb->scan->completion);

	memcpy(&rt2x00usb->scan->conf, conf, sizeof(*conf));

	rt2x00usb->scan->state = state;

	rt2x00usb->scan->status = 0;

	/*
	 * Queue work.
	 */
	if (!queue_work(rt2x00usb->workqueue, &rt2x00usb->scan_work))
		goto exit;

	return 0;

exit:
	kfree(rt2x00usb->scan);
	rt2x00usb->scan = NULL;

	return -EIO;
}

static int rt73usb_get_stats(struct net_device *net_dev,
	struct ieee80211_low_level_stats *stats)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);

	/*
	 * The dot11ACKFailureCount is updated in interrupt time.
	 * TODO: dot11FCSErrorCount, dot11RTSFailureCount and
	 * dot11RTSSuccessCount are never updated,
	 * we need to find a method to see where we can update
	 * those statistics from.
	 */
	memcpy(stats, &rt2x00usb->low_level_stats, sizeof(*stats));

	return 0;
}

static int rt73usb_set_retry_limit(struct net_device *net_dev,
	u32 short_retry, u32 long_retry)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);
	u32 reg;

	rt2x00_register_read(rt2x00usb, TXRX_CSR4, &reg);
	rt2x00_set_field32(&reg, TXRX_CSR4_LONG_RETRY_LIMIT, long_retry);
	rt2x00_set_field32(&reg, TXRX_CSR4_SHORT_RETRY_LIMIT, short_retry);
	rt2x00_register_write(rt2x00usb, TXRX_CSR4, reg);

	return 0;
}

static int rt73usb_conf_tx(struct net_device *net_dev,
	int queue, const struct ieee80211_tx_queue_params *params)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);
	struct data_ring *ring;

	ring = rt2x00usb_get_ring(rt2x00usb, queue);
	if (unlikely(!ring)) {
		NOTICE("Ignoring configuration for queue %d.\n", queue);
		return -EINVAL;
	}

	memcpy(&ring->tx_params, params, sizeof(*params));

	/*
	 * The passed variables are stored as real value ((2^n)-1).
	 * RT73 registers require to know the bit number 'n'.
	 */
	if (params->cw_min)
		ring->tx_params.cw_min = HIGHEST_BIT16(params->cw_min) + 1;
	else
		ring->tx_params.cw_min = 4; /* cw_min: 2^4 = 16. */

	if (params->cw_max)
		ring->tx_params.cw_max = HIGHEST_BIT16(params->cw_max) + 1;
	else
		ring->tx_params.cw_max = 10; /* cw_min: 2^10 = 1024. */

	if (!params->aifs)
		ring->tx_params.aifs = 2;

	return 0;
}

static int rt73usb_get_tx_stats(struct net_device *net_dev,
	struct ieee80211_tx_queue_stats *stats)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);

	memcpy(&stats->data[IEEE80211_TX_QUEUE_DATA0],
		&rt2x00usb->ring[RING_AC_BK].stats,
		sizeof(rt2x00usb->ring[RING_AC_BK].stats));
	memcpy(&stats->data[IEEE80211_TX_QUEUE_DATA1],
		&rt2x00usb->ring[RING_AC_BE].stats,
		sizeof(rt2x00usb->ring[RING_AC_BE].stats));
	memcpy(&stats->data[IEEE80211_TX_QUEUE_DATA2],
		&rt2x00usb->ring[RING_AC_VI].stats,
		sizeof(rt2x00usb->ring[RING_AC_VI].stats));
	memcpy(&stats->data[IEEE80211_TX_QUEUE_DATA3],
		&rt2x00usb->ring[RING_AC_VO].stats,
		sizeof(rt2x00usb->ring[RING_AC_VO].stats));
	memcpy(&stats->data[IEEE80211_TX_QUEUE_DATA4],
		&rt2x00usb->ring[RING_PRIO].stats,
		sizeof(rt2x00usb->ring[RING_PRIO].stats));

	return 0;
}

static u64 rt73usb_get_tsf(struct net_device *net_dev)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);
	u64 tsf;
	u32 reg;

	rt2x00_register_read(rt2x00usb, TXRX_CSR13, &reg);
	tsf = (u64)rt2x00_get_field32(reg, TXRX_CSR13_HIGH_TSFTIMER) << 32;
	rt2x00_register_read(rt2x00usb, TXRX_CSR12, &reg);
	tsf |= rt2x00_get_field32(reg, TXRX_CSR12_LOW_TSFTIMER);

	return tsf;
}

static void rt73usb_reset_tsf(struct net_device *net_dev)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);
	u32 reg = 0;

	rt2x00_register_write(rt2x00usb, TXRX_CSR12, reg);
	rt2x00_register_write(rt2x00usb, TXRX_CSR13, reg);
}

static int rt73usb_beacon_update(struct net_device *net_dev,
	struct sk_buff *skb, struct ieee80211_tx_control *control)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);
	struct usb_device *usb_dev = interface_to_usbdev(rt2x00usb->usb_intf);
	struct data_entry *entry;

	entry = rt2x00_get_data_entry(&rt2x00usb->ring[RING_BEACON]);

	/*
	 * Just in case the ieee80211 doesn't set this,
	 * but we need this queue set for the descriptor
	 * initialization.
	 */
	control->queue = IEEE80211_TX_QUEUE_BEACON;

	memcpy(rt2x00usb_txdata_addr(entry), skb->data, skb->len);
	rt73usb_write_tx_desc(rt2x00usb,
		rt2x00usb_txdesc_addr(entry), skb, control);

	SET_FLAG(entry, ENTRY_OWNER_NIC);
	usb_fill_bulk_urb(
		rt2x00usb_urb(entry),
		usb_dev,
		usb_sndbulkpipe(usb_dev, 1),
		entry->data_addr,
		skb->len + rt2x00usb->ring[RING_BEACON].desc_size,
		rt73usb_interrupt,
		entry);
	usb_submit_urb(rt2x00usb_urb(entry), GFP_ATOMIC);

	return 0;
}

/*
 * Device initialization functions.
 */
static int rt73usb_alloc_rings(struct rt2x00_usb *rt2x00usb)
{
	unsigned int i;

	rt2x00usb->ring = kzalloc(
		sizeof(struct data_ring) * RING_NUM, GFP_KERNEL);
	if (!rt2x00usb->ring) {
		ERROR("Ring allocation failed.\n");
		return -ENOMEM;
	}

	for (i = 0; i < RING_NUM; i++) {
		/*
		 *Set device structure.
		 */
		rt2x00usb->ring[i].net_dev =
			usb_get_intfdata(rt2x00usb->usb_intf);
	
		/*
		 * Initialize ring parameters.
		 * cw_min: 2^5 = 32.
		 * cw_max: 2^10 = 1024.
		 */
		rt2x00usb->ring[i].tx_params.aifs = 2;
		rt2x00usb->ring[i].tx_params.cw_min = 5;
		rt2x00usb->ring[i].tx_params.cw_max = 10;
	}

	return 0;
}

static int rt73usb_init_eeprom(struct rt2x00_usb *rt2x00usb)
{
	struct ieee80211_conf *conf = ieee80211_get_hw_conf(
			usb_get_intfdata(rt2x00usb->usb_intf));
	u16 value;
	u16 eeprom;

	/*
	 * 1 - Read EEPROM word for configuration.
	 */
	rt2x00_eeprom_read(rt2x00usb, EEPROM_ANTENNA, &eeprom);

	/*
	 * 2 - Identify RF chipset.
	 */
	value = rt2x00_get_field16(eeprom, EEPROM_ANTENNA_RF_TYPE);
	rt2x00_set_chip(&rt2x00usb->chip, RT73, value);

	if (!rt2x00_rf(&rt2x00usb->chip, RF5226)
	&& !rt2x00_rf(&rt2x00usb->chip, RF2528)
	&& !rt2x00_rf(&rt2x00usb->chip, RF5225)
	&& !rt2x00_rf(&rt2x00usb->chip, RF2527))
		return -ENODEV;

	/*
	 * 3 - Identify default antenna configuration.
	 * Ralink devices have have antenna options for both TX as RX.
	 * The ieee80211 stack currently only provide the user to set
	 * 1 antenna, by default this is considered to be the TX antenna.
	 */
	conf->antenna_sel = rt2x00_get_field16(eeprom,
				EEPROM_ANTENNA_TX_DEFAULT);

	/*
	 * 4 - Read the Frame type.
	 */
	if (rt2x00_get_field16(eeprom, EEPROM_ANTENNA_FRAME_TYPE))
		SET_FLAG(rt2x00usb, CONFIG_FRAME_TYPE);

	/*
	 * 6 - Read frequency offset.
	 */
	rt2x00_eeprom_read(rt2x00usb, EEPROM_FREQ, &eeprom);
	rt2x00usb->freq_offset = rt2x00_get_field16(eeprom,
					EEPROM_FREQ_OFFSET);
	if (rt2x00usb->freq_offset == 0xff)
		rt2x00usb->freq_offset = 0;

	/*
	 * 7 - Read external LNA informations.
	 */
	rt2x00_eeprom_read(rt2x00usb, EEPROM_NIC, &eeprom);
	if (eeprom == 0xffff)
		eeprom = 0;
	if (rt2x00_get_field16(eeprom, EEPROM_NIC_EXTERNAL_LNA))
		SET_FLAG(rt2x00usb, CONFIG_EXTERNAL_LNA);

	/*
	 * 8 - Store led settings, for correct led behaviour.
	 */
	rt2x00_eeprom_read(rt2x00usb, EEPROM_LED, &eeprom);
	/*
	 * If the eeprom value is invalid,
	 * switch to default led mode, and clear all bits.
	 */
	if (eeprom == 0xffff) {
		rt2x00usb->led_mode = LED_MODE_DEFAULT;
		eeprom = 0x0000;
	} else {
		rt2x00usb->led_mode = rt2x00_get_field16(eeprom,
			EEPROM_LED_LED_MODE);
	}

	rt2x00_set_field16(&rt2x00usb->led_reg, MCU_LEDCS_LED_MODE,
		rt2x00usb->led_mode);
	rt2x00_set_field16(&rt2x00usb->led_reg, MCU_LEDCS_POLARITY_GPIO_0,
		rt2x00_get_field16(eeprom, EEPROM_LED_POLARITY_GPIO_0));
	rt2x00_set_field16(&rt2x00usb->led_reg, MCU_LEDCS_POLARITY_GPIO_1,
		rt2x00_get_field16(eeprom, EEPROM_LED_POLARITY_GPIO_1));
	rt2x00_set_field16(&rt2x00usb->led_reg, MCU_LEDCS_POLARITY_GPIO_2,
		rt2x00_get_field16(eeprom, EEPROM_LED_POLARITY_GPIO_2));
	rt2x00_set_field16(&rt2x00usb->led_reg, MCU_LEDCS_POLARITY_GPIO_3,
		rt2x00_get_field16(eeprom, EEPROM_LED_POLARITY_GPIO_3));
	rt2x00_set_field16(&rt2x00usb->led_reg, MCU_LEDCS_POLARITY_GPIO_4,
		rt2x00_get_field16(eeprom, EEPROM_LED_POLARITY_GPIO_4));
	rt2x00_set_field16(&rt2x00usb->led_reg, MCU_LEDCS_POLARITY_ACT,
		rt2x00_get_field16(eeprom, EEPROM_LED_POLARITY_ACT));
	rt2x00_set_field16(&rt2x00usb->led_reg, MCU_LEDCS_POLARITY_READY_BG,
		rt2x00_get_field16(eeprom, EEPROM_LED_POLARITY_RDY_G));
	rt2x00_set_field16(&rt2x00usb->led_reg, MCU_LEDCS_POLARITY_READY_A,
		rt2x00_get_field16(eeprom, EEPROM_LED_POLARITY_RDY_A));

	/*
	 * 9 - Read BBP data from EEPROM and store in private structure.
	 */
	rt2x00usb->eeprom = kzalloc(EEPROM_BBP_SIZE * sizeof(u16), GFP_KERNEL);
	if (!rt2x00usb->eeprom)
		return -ENOMEM;

	rt2x00_eeprom_multiread(rt2x00usb, EEPROM_BBP_START,
		rt2x00usb->eeprom, EEPROM_BBP_SIZE * sizeof(u16));

	return 0;
}

static int rt73usb_init_mac(struct rt2x00_usb *rt2x00usb)
{
	struct net_device *net_dev = usb_get_intfdata(rt2x00usb->usb_intf);
	u16 eeprom[3] = {0, 0, 0};

	if (GET_FLAG(rt2x00usb, DEVICE_INITIALIZED_MAC))
		return 0;

	/*
	 * Read MAC address from EEPROM.
	 */
	rt2x00_eeprom_multiread(rt2x00usb, EEPROM_MAC_ADDR, &eeprom[0], 6);

	net_dev->dev_addr[0] = rt2x00_get_field16(eeprom[0],
					EEPROM_MAC_ADDR_BYTE0);
	net_dev->dev_addr[1] = rt2x00_get_field16(eeprom[0],
					EEPROM_MAC_ADDR_BYTE1);
	net_dev->dev_addr[2] = rt2x00_get_field16(eeprom[1],
					EEPROM_MAC_ADDR_BYTE2);
	net_dev->dev_addr[3] = rt2x00_get_field16(eeprom[1],
					EEPROM_MAC_ADDR_BYTE3);
	net_dev->dev_addr[4] = rt2x00_get_field16(eeprom[2],
					EEPROM_MAC_ADDR_BYTE4);
	net_dev->dev_addr[5] = rt2x00_get_field16(eeprom[2],
					EEPROM_MAC_ADDR_BYTE5);

	net_dev->addr_len = 6;

	if (is_valid_ether_addr(&net_dev->dev_addr[0])) {
		SET_FLAG(rt2x00usb, DEVICE_INITIALIZED_MAC);
		return 0;
	}

	return -EINVAL;
}

static int rt73usb_init_hw(struct rt2x00_usb *rt2x00usb)
{
	struct net_device *net_dev = usb_get_intfdata(rt2x00usb->usb_intf);
	struct ieee80211_hw *hw = &rt2x00usb->hw;

	if (GET_FLAG(rt2x00usb, DEVICE_INITIALIZED_HW))
		return 0;

	/*
	 * IEEE80211 Function callbacks.
	 */
	hw->tx			= rt73usb_tx;
	hw->reset		= rt73usb_reset;
	hw->add_interface	= rt73usb_add_interface;
	hw->remove_interface	= rt73usb_remove_interface;
	hw->config		= rt73usb_config;
	hw->config_interface	= rt73usb_config_interface;
	hw->set_multicast_list	= rt73usb_set_multicast_list;
	hw->passive_scan	= rt73usb_passive_scan;
	hw->get_stats		= rt73usb_get_stats;
	hw->set_retry_limit	= rt73usb_set_retry_limit;
	hw->conf_tx		= rt73usb_conf_tx;
	hw->get_tx_stats	= rt73usb_get_tx_stats;
	hw->get_tsf		= rt73usb_get_tsf;
	hw->reset_tsf		= rt73usb_reset_tsf;
	hw->beacon_update	= rt73usb_beacon_update;

	/*
	 * IEEE80211 Variables.
	 */
	hw->version = IEEE80211_VERSION;
	hw->name = DRV_NAME;
	hw->host_gen_beacon = 1;
	hw->device_hides_wep = 0;
	hw->rx_includes_fcs = 0;
	hw->host_broadcast_ps_buffering = 1;
	hw->wep_include_iv = 1;
	hw->data_nullfunc_ack = 1;
	hw->no_tkip_wmm_hwaccel = 1;
	hw->extra_hdr_room = 0;
	hw->device_strips_mic = 0;
	hw->monitor_during_oper = 1;
	hw->fraglist = 0;

	/*
	 * We have full WMM capabilities.
	 */
	hw->queues = RING_NUM_TX;

	if (ieee80211_register_hw(net_dev, hw))
		return -EIO;

	SET_FLAG(rt2x00usb, DEVICE_INITIALIZED_HW);

	return 0;
}

static void rt73usb_init_hw_channels(struct rt2x00_usb *rt2x00usb,
	struct ieee80211_channel *channels)
{
	unsigned int i;
	u32 rf2_base;
	u16 eeprom;
	static const struct {
		unsigned int chip;
		u32 val[3];
	} rf[] = {
		{ RF5226,	{ 0x00002c0c, 0x00068255 } },
		{ RF2528,	{ 0x00002c0c, 0x00068255 } },
		{ RF5225,	{ 0x00002ccc, 0x00000000 } },
		{ RF2527,	{ 0x00002ccc, 0x00068455 } },
	};
	static const u32 vals[] = {
		0x00000786, 0x00000786, 0x0000078a, 0x0000078a,
		0x0000078e, 0x0000078e, 0x00000792, 0x00000792,
		0x00000796, 0x00000796, 0x0000079a, 0x0000079a,
		0x0000079e, 0x000007a2
	};

	/*
	 * Channel initialization.
	 * First we set the basic variables.
	 */
	for (i = 0; i < 13; i++) {
		channels[i].chan = i + 1;
		channels[i].freq = 2407 + ((i + 1) * 5);
		channels[i].flag = IEEE80211_CHAN_W_IBSS
			| IEEE80211_CHAN_W_ACTIVE_SCAN | IEEE80211_CHAN_W_SCAN;
		channels[i].antenna_max = 0xff;
	}

	channels[13].chan = 14;
	channels[13].freq = 2484;
	channels[13].flag = IEEE80211_CHAN_W_IBSS
		| IEEE80211_CHAN_W_ACTIVE_SCAN | IEEE80211_CHAN_W_SCAN;
	channels[13].antenna_max = 0xff;

	if (rt2x00_rf(&rt2x00usb->chip, RF5225)
	|| rt2x00_rf(&rt2x00usb->chip, RF5226)) {
		for (i = 14; i < 38; i++) {
			if (i < 22)
				channels[i].chan = 36;
			else if (i < 33)
				channels[i].chan = 100;
			else
				channels[i].chan = 149;
			channels[i].chan += ((i - 14) * 4);
			channels[i].freq = ((i - 13) + 1000) * 5;
			channels[i].flag = IEEE80211_CHAN_W_IBSS
				| IEEE80211_CHAN_W_ACTIVE_SCAN
				| IEEE80211_CHAN_W_SCAN;
			channels[i].antenna_max = 0xff;
		}
	}

	/*
	 * Set device specific value.
	 */
	rf2_base = 0;
	if (rt2x00_rf(&rt2x00usb->chip, RF5225)
	|| rt2x00_rf(&rt2x00usb->chip, RF2527))
		rf2_base = 0x00004000;

	for (i = 0; i < ARRAY_SIZE(vals); i++)
		channels[i].val = cpu_to_le32(vals[i] | rf2_base);

	if (rt2x00_rf(&rt2x00usb->chip, RF5225)
	|| rt2x00_rf(&rt2x00usb->chip, RF5226)) {
		static const u32 vals[] = {
			0x0000099a, 0x000009a2, 0x000009a6, 0x000009aa,
			0x000009ae, 0x000009b2, 0x000009ba, 0x000009be,
			0x00000a2a, 0x00000a2e, 0x00000a32, 0x00000a36,
			0x00000a3a, 0x00000a82, 0x00000a86, 0x00000a8a,
			0x00000a8e, 0x00000a92, 0x00000a9a, 0x00000aa2,
			0x00000aa6, 0x00000aae, 0x00000ab2, 0x00000ab6
		};

		struct ieee80211_channel *chan = channels + 14;

		for (i = 0; i < ARRAY_SIZE(vals); i++)
			(chan++)->val = cpu_to_le32(vals[i]);
	}

	/*
	 * Set TX power, each EEPROM TXpower entry
	 * contains the TXpower value for 2 channels.
	 */
	for (i = 0; i < EEPROM_TXPOWER_G_SIZE; i++) {
		rt2x00_eeprom_read(rt2x00usb,
			EEPROM_TXPOWER_G_START + i, &eeprom);

		channels[(i * 2)].power_level =
			rt2x00_get_field16(eeprom, EEPROM_TXPOWER_G_1);
		channels[(i * 2)].power_level = TXPOWER_FROM_DEV(
			channels[(i * 2)].power_level);

		channels[(i * 2) + 1].power_level =
			rt2x00_get_field16(eeprom, EEPROM_TXPOWER_G_2);
		channels[(i * 2) + 1].power_level = TXPOWER_FROM_DEV(
			channels[(i * 2) + 1].power_level);
	}

	if (rt2x00_rf(&rt2x00usb->chip, RF5225)
	|| rt2x00_rf(&rt2x00usb->chip, RF5226)) {
		for (; i < EEPROM_TXPOWER_A_SIZE; i++) {
			rt2x00_eeprom_read(rt2x00usb,
				EEPROM_TXPOWER_A_START + i, &eeprom);

			channels[(i * 2)].power_level =
				rt2x00_get_field16(eeprom, EEPROM_TXPOWER_A_1);
			channels[(i * 2)].power_level = TXPOWER_FROM_DEV(
				channels[(i * 2)].power_level);

			channels[(i * 2) + 1].power_level =
				rt2x00_get_field16(eeprom, EEPROM_TXPOWER_A_2);
			channels[(i * 2) + 1].power_level =
				TXPOWER_FROM_DEV(
				channels[(i * 2) + 1].power_level);
		}
	}

	/*
	 * Set device specific, but channel independent RF values.
	 */
	for (i = 0; i < ARRAY_SIZE(rf); i++) {
		if (rt2x00_rf(&rt2x00usb->chip, rf[i].chip)) {
			rt2x00usb->rf1 = cpu_to_le32(rf[i].val[0]);
			rt2x00usb->rf3 = cpu_to_le32(rf[i].val[1]);
		}
	}
}

static void rt73usb_init_hw_rates(struct rt2x00_usb *rt2x00usb,
	struct ieee80211_rate *rates)
{
	/*
	 * Rates initialization.
	 */
	rates[0].rate = 10;
	rates[0].val = DEVICE_RATE_1MB;
	rates[0].flags = IEEE80211_RATE_CCK;
	rates[0].val2 = DEVICE_RATE_1MB;
	rates[0].min_rssi_ack = 0;
	rates[0].min_rssi_ack_delta = 0;

	rates[1].rate = 20;
	rates[1].val = DEVICE_RATE_2MB;
	rates[1].flags = IEEE80211_RATE_CCK_2;
	rates[1].val2 = DEVICE_RATE_2MB_PREAMBLE;
	rates[1].min_rssi_ack = 0;
	rates[1].min_rssi_ack_delta = 0;

	rates[2].rate = 55;
	rates[2].val = DEVICE_RATE_55MB;
	rates[2].flags = IEEE80211_RATE_CCK_2;
	rates[2].val2 = DEVICE_RATE_55MB_PREAMBLE;
	rates[2].min_rssi_ack = 0;
	rates[2].min_rssi_ack_delta = 0;

	rates[3].rate = 110;
	rates[3].val = DEVICE_RATE_11MB;
	rates[3].flags = IEEE80211_RATE_CCK_2;
	rates[3].val2 = DEVICE_RATE_11MB_PREAMBLE;
	rates[3].min_rssi_ack = 0;
	rates[3].min_rssi_ack_delta = 0;

	rates[4].rate = 60;
	rates[4].val = DEVICE_RATE_6MB;
	rates[4].flags = IEEE80211_RATE_OFDM;
	rates[4].val2 = DEVICE_RATE_6MB;
	rates[4].min_rssi_ack = 0;
	rates[4].min_rssi_ack_delta = 0;

	rates[5].rate = 90;
	rates[5].val = DEVICE_RATE_9MB;
	rates[5].flags = IEEE80211_RATE_OFDM;
	rates[5].val2 = DEVICE_RATE_9MB;
	rates[5].min_rssi_ack = 0;
	rates[5].min_rssi_ack_delta = 0;

	rates[6].rate = 120;
	rates[6].val = DEVICE_RATE_12MB;
	rates[6].flags = IEEE80211_RATE_OFDM;
	rates[6].val2 = DEVICE_RATE_12MB;
	rates[6].min_rssi_ack = 0;
	rates[6].min_rssi_ack_delta = 0;

	rates[7].rate = 180;
	rates[7].val = DEVICE_RATE_18MB;
	rates[7].flags = IEEE80211_RATE_OFDM;
	rates[7].val2 = DEVICE_RATE_18MB;
	rates[7].min_rssi_ack = 0;
	rates[7].min_rssi_ack_delta = 0;

	rates[8].rate = 240;
	rates[8].val = DEVICE_RATE_24MB;
	rates[8].flags = IEEE80211_RATE_OFDM;
	rates[8].val2 = DEVICE_RATE_24MB;
	rates[8].min_rssi_ack = 0;
	rates[8].min_rssi_ack_delta = 0;

	rates[9].rate = 360;
	rates[9].val = DEVICE_RATE_36MB;
	rates[9].flags = IEEE80211_RATE_OFDM;
	rates[9].val2 = DEVICE_RATE_36MB;
	rates[9].min_rssi_ack = 0;
	rates[9].min_rssi_ack_delta = 0;

	rates[10].rate = 480;
	rates[10].val = DEVICE_RATE_48MB;
	rates[10].flags = IEEE80211_RATE_OFDM;
	rates[10].val2 = DEVICE_RATE_48MB;
	rates[10].min_rssi_ack = 0;
	rates[10].min_rssi_ack_delta = 0;

	rates[11].rate = 540;
	rates[11].val = DEVICE_RATE_54MB;
	rates[11].flags = IEEE80211_RATE_OFDM;
	rates[11].val2 = DEVICE_RATE_54MB;
	rates[11].min_rssi_ack = 0;
	rates[11].min_rssi_ack_delta = 0;
}

static int rt73usb_init_hw_modes(struct rt2x00_usb *rt2x00usb)
{
	struct net_device *net_dev = usb_get_intfdata(rt2x00usb->usb_intf);
	struct ieee80211_hw *hw = &rt2x00usb->hw;
	int num_modes;
	int num_channels;

	/*
	 * RF2527 and RF2528 only supports 802.11b & 802.11g,
	 * so we should allocate 14 OFDM channels, 4 CCK rates
	 * and 8 OFDM rates.
	 * RF5225 and RF5226 also supports 802.11a, so allocate an
	 * additional 23 5.2GHz channels.
	 */
	num_modes = 2;
	num_channels = 14;
	if (rt2x00_rf(&rt2x00usb->chip, RF5225)
	|| rt2x00_rf(&rt2x00usb->chip, RF5226)) {
		num_modes = 3;
		num_channels = 38;
	}

	hw->num_modes = num_modes;
	hw->modes =
		kzalloc((sizeof(struct ieee80211_hw_modes) * num_modes),
		GFP_KERNEL);
	if (!hw->modes)
		goto exit;

	hw->modes->channels =
		kzalloc((sizeof(struct ieee80211_channel) * num_channels),
		GFP_KERNEL);
	if (!hw->modes->channels)
		goto exit_free_modes;

	hw->modes->rates =
		kzalloc((sizeof(struct ieee80211_rate) * 12),
		GFP_KERNEL);
	if (!hw->modes->rates)
		goto exit_free_channels;

	/*
	 * Intitialize 802.11g
	 * Rates: CCK, OFDM.
	 * Channels: OFDM.
	 */
	hw->modes[0].mode = MODE_IEEE80211G;
	hw->modes[0].num_channels = 14;
	hw->modes[0].num_rates = 12;

	/*
	 * Intitialize 802.11b
	 * Rates: CCK.
	 * Channels: OFDM.
	 */
	hw->modes[1].mode = MODE_IEEE80211B;
	hw->modes[1].num_channels = 14;
	hw->modes[1].num_rates = 4;
	hw->modes[1].channels = hw->modes[0].channels;
	hw->modes[1].rates = hw->modes[0].rates;

	/*
	 * Intitialize 802.11a
	 * Rates: OFDM.
	 * Channels: OFDM, UNII, HiperLAN2.
	 */
	if (rt2x00_rf(&rt2x00usb->chip, RF5225)
	|| rt2x00_rf(&rt2x00usb->chip, RF5226)) {
		hw->modes[2].mode = MODE_IEEE80211A;
		hw->modes[2].num_channels = 38;
		hw->modes[2].num_rates = 8;
		hw->modes[2].channels = &hw->modes[0].channels[14];
		hw->modes[2].rates = &hw->modes[0].rates[4];
	}

	rt73usb_init_hw_channels(rt2x00usb, hw->modes[0].channels);
	rt73usb_init_hw_rates(rt2x00usb, hw->modes[0].rates);

	/*
	 * xr_end is only used on Atheros cards.
	 */
	hw->modes->xr_end = 0;

	return ieee80211_update_hw(net_dev, hw);

exit_free_channels:
	kfree(hw->modes->channels);
	hw->modes->channels = NULL;

exit_free_modes:
	kfree(hw->modes);
	hw->modes = NULL;

exit:
	ERROR("Allocation ieee80211 modes failed.\n");
	return -ENOMEM;
}

static void rt73usb_free_dev(struct net_device *net_dev)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);

	/*
	 * Free ring structures.
	 */
	kfree(rt2x00usb->ring);
	rt2x00usb->ring = NULL;

	/*
	 * Free EEPROM memory.
	 */
	kfree(rt2x00usb->eeprom);

	/*
	 * Free workqueue.
	 */
	if (likely(rt2x00usb->workqueue)) {
		destroy_workqueue(rt2x00usb->workqueue);
		rt2x00usb->workqueue = NULL;
	}

	/*
	 * Free ieee80211_hw memory.
	 */
	if (likely(rt2x00usb->hw.modes)) {
		kfree(rt2x00usb->hw.modes->channels);
		kfree(rt2x00usb->hw.modes->rates);
		kfree(rt2x00usb->hw.modes);
		rt2x00usb->hw.modes = NULL;
	}
}

static int rt73usb_alloc_dev(struct usb_interface *usb_intf,
	struct net_device *net_dev)
{
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);

	rt2x00usb->usb_intf = usb_intf;

	rt2x00usb->workqueue = create_singlethread_workqueue(DRV_NAME);
	if (!rt2x00usb->workqueue)
		return -ENODEV;

	/*
	 * Initialize cofniguration work.
	 */
	INIT_WORK(&rt2x00usb->config_work, rt73usb_config_update, rt2x00usb);
	INIT_WORK(&rt2x00usb->interface_work,
		rt73usb_interface_update, rt2x00usb);

	/*
	 * Reset current working type.
	 */
	rt2x00usb->interface_type = -EINVAL;

	/*
	 * Intialize scanning attributes.
	 */
	INIT_WORK(&rt2x00usb->scan_work, rt73usb_scan, rt2x00usb);
	rt2x00usb->scan = NULL;

	/*
	 * Allocate ring array.
	 */
	if (rt73usb_alloc_rings(rt2x00usb))
		goto exit;

	/*
	 * Initialize hardware.
	 */
	if (rt73usb_init_eeprom(rt2x00usb)
	|| rt73usb_init_mac(rt2x00usb)
	|| rt73usb_init_hw(rt2x00usb)
	|| rt73usb_init_hw_modes(rt2x00usb)) {
		ERROR("Failed to initialize device.\n");
		goto exit;
	}

	if (rt73usb_init_firmware(rt2x00usb)) {
		ERROR("Failed to load Firmware.\n");
		goto exit;
	}

	return 0;

exit:
	rt73usb_free_dev(net_dev);

	return -ENODEV;
}

/*
 * USB driver handlers.
 */
static int rt73usb_probe(
	struct usb_interface *usb_intf, const struct usb_device_id *id)
{
	struct usb_device *usb_dev = interface_to_usbdev(usb_intf);
	struct net_device *net_dev;
	int status;

	if (unlikely(id->driver_info != RT73)) {
		ERROR("Detected device not supported.\n");
		return -ENODEV;
	}

	usb_dev = usb_get_dev(usb_dev);

	net_dev = ieee80211_alloc_hw(sizeof(struct rt2x00_usb), NULL);
	if (!net_dev) {
		ERROR("Failed to allocate hardware.\n");
		status = -ENOMEM;
		goto exit_put_device;
	}

	SET_ETHTOOL_OPS(net_dev, &rt73usb_ethtool_ops);

	usb_set_intfdata(usb_intf, net_dev);

	status = rt73usb_alloc_dev(usb_intf, net_dev);
	if (status) {
		ERROR("Failed to allocate device.\n");
		status = -ENOMEM;
		goto exit_free_device;
	}

	return 0;

exit_free_device:
	ieee80211_free_hw(net_dev);

exit_put_device:
	usb_put_dev(usb_dev);

	return status;
}

static void rt73usb_disconnect(struct usb_interface *usb_intf)
{
	struct net_device *net_dev = usb_get_intfdata(usb_intf);
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);

	rt73usb_disable_radio(rt2x00usb);

	rt73usb_uninitialize(rt2x00usb);

	rt73usb_free_dev(net_dev);

	ieee80211_unregister_hw(net_dev);

	ieee80211_free_hw(net_dev);

	usb_set_intfdata(usb_intf, NULL);

	usb_put_dev(interface_to_usbdev(usb_intf));
}

#ifdef CONFIG_PM
static int rt73usb_suspend(struct usb_interface *usb_intf, pm_message_t state)
{
	struct net_device *net_dev = usb_get_intfdata(usb_intf);
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);

	NOTICE("Going to sleep.\n");

	/*
	 * Disable the radio.
	 */
	rt73usb_disable_radio(rt2x00usb);

	/*
	 * Set device mode to sleep for power management.
	 */
	if (rt73usb_set_state(rt2x00usb, STATE_SLEEP))
		return -EBUSY;

	/*
	 * Uninitialize device.
	 */
	rt73usb_uninitialize(rt2x00usb);

	/*
	 * Uninitialize hardware.
	 */
	rt73usb_free_dev(net_dev);

	/*
	 * Decrease usbdev refcount.
	 */
	usb_put_dev(interface_to_usbdev(usb_intf));

	return 0;
}

static int rt73usb_resume(struct usb_interface *usb_intf)
{
	struct net_device *net_dev = usb_get_intfdata(usb_intf);
	struct rt2x00_usb *rt2x00usb = ieee80211_dev_hw_data(net_dev);

	NOTICE("Waking up.\n");

	/*
	 * Increase usbdev refcount.
	 */
	usb_get_dev(interface_to_usbdev(usb_intf));

	/*
	 * Initialize hardware.
	 */
	if (rt73usb_alloc_dev(usb_intf, net_dev)) {
		ERROR("Failed to allocate device.\n");
		return -ENOMEM;
	}

	/*
	 * Set device mode to awake for power management.
	 */
	return rt73usb_set_state(rt2x00usb, STATE_AWAKE);
}
#endif /* CONFIG_PM */

/*
 * rt73usb module information.
 */
static char version[] =
	DRV_NAME " - " DRV_VERSION " (" DRV_RELDATE ") by " DRV_PROJECT;

static struct usb_device_id rt73usb_device_table[] = {
	/* Belkin */
	{ USB_DEVICE(0x050d, 0x7050), .driver_info = RT73},
	{ USB_DEVICE(0x050d, 0x705a), .driver_info = RT73},
	/* Conceptronic */
	{ USB_DEVICE(0x14b2, 0x3c22), .driver_info = RT73},
	/* D-Link */
	{ USB_DEVICE(0x07d1, 0x3c03), .driver_info = RT73},
	{ USB_DEVICE(0x07d1, 0x3c04), .driver_info = RT73},
	/* Gigabyte */
	{ USB_DEVICE(0x1044, 0x8008), .driver_info = RT73},
	/* Linksys */
	{ USB_DEVICE(0x13b1, 0x0020), .driver_info = RT73},
	/* MSI */
	{ USB_DEVICE(0x0db0, 0x6877), .driver_info = RT73},
	{ USB_DEVICE(0x0db0, 0xa874), .driver_info = RT73},
	/* Ralink */
	{ USB_DEVICE(0x148f, 0x2573), .driver_info = RT73},
	{ USB_DEVICE(0x148f, 0x2671), .driver_info = RT73},
	/* Qcom */
	{ USB_DEVICE(0x18e8, 0x6196), .driver_info = RT73},
	/* Sitecom */
	{ USB_DEVICE(0x0df6, 0x9712), .driver_info = RT73},
	{0,}
};

MODULE_AUTHOR(DRV_PROJECT);
MODULE_VERSION(DRV_VERSION);
MODULE_DESCRIPTION("Ralink RT73 USB Wireless LAN driver.");
MODULE_SUPPORTED_DEVICE("Ralink RT2573 USB chipset based cards");
MODULE_DEVICE_TABLE(usb, rt73usb_device_table);
MODULE_LICENSE("GPL");

#ifdef CONFIG_RT73USB_DEBUG
module_param_named(debug, rt2x00_debug_level, bool, S_IWUSR | S_IRUGO);
MODULE_PARM_DESC(debug, "Set this parameter to 1 to enable debug output.");
#endif /* CONFIG_RT73USB_DEBUG */

static struct usb_driver rt73usb_driver = {
#if LINUX_VERSION_CODE <= KERNEL_VERSION(2, 6, 15)
	.owner		= THIS_MODULE,
#endif /* LINUX_VERSION_CODE <= KERNEL_VERSION(2, 6, 15) */
	.name		= DRV_NAME,
	.id_table	= rt73usb_device_table,
	.probe		= rt73usb_probe,
	.disconnect	= rt73usb_disconnect,
#ifdef CONFIG_PM
	.suspend	= rt73usb_suspend,
	.resume		= rt73usb_resume,
#endif /* CONFIG_PM */
};

static int __init rt73usb_init(void)
{
	printk(KERN_INFO "Loading module: %s.\n", version);
	return usb_register(&rt73usb_driver);
}

static void __exit rt73usb_exit(void)
{
	printk(KERN_INFO "Unloading module: %s.\n", version);
	usb_deregister(&rt73usb_driver);
}

module_init(rt73usb_init);
module_exit(rt73usb_exit);
