/*
	Copyright (C) 2006 Ivo van Doorn

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the
	Free Software Foundation, Inc.,
	59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
	Radio hardware button support
	Laptops are quite often equiped with support with a hardware button
	to enabled and disable the radio of the integrated wireless network
	or bluetooth interface.
	Altough some devices will make sure that when pressed the radio
	is disabled in hardware, other device depend on the software
	to enabled or disable the radio accordingly.
	This driver will create an input device and will poll registered
	hardware frequently to listen if the button has been pressed.
	When the device requires the software to disable or enable
	the radio it will do so correctly, but it will also in all cases
	send a signal to the input device to inform any listening daemon
	the state has changed and will allow userspace to handle certain
	tasks as well if required.
 */

#ifndef RADIOBTN_H
#define RADIOBTN_H

#include <linux/input.h>

/**
 * struct radio_button - radio hardware structure.
 * @dev_name: Name of the interface. This will become the name
 * 	of the input device created in /dev/radio/.
 * @data: Private data which will be passed along with the radio handlers.
 * @button_poll(unsigned long data): Handler which will be called
 * 	with the poll_delay interval.
 * @enable_radio(unsigned long data): Optional handler to enable the radio
 * 	once the button has been pressed when the hardware does not do this
 * 	automaticly.
 * @disable_radio(unsigned long data): Optional handler to disable the radio
 * 	once the button has been pressed when the hardware does not do this
 * 	automaticly.
 * @poll_delay: Delay in msecs between each poll.
 * @current_state: Current state of the button.
 * 	(Should not be touched by driver)
 * @input_dev: Pointer to input device for this radio button.
 * 	(Should not be touched by driver)
 * @poll_timer: Timer used to poll for the button status.
 *	(Should not be touched by driver)
 */
struct radio_button {
	const char *dev_name;

	unsigned long data;

	int (*button_poll)(unsigned long data);
	void (*enable_radio)(unsigned long data);
	void (*disable_radio)(unsigned long data);

	unsigned int poll_delay;

	unsigned int current_state;

	struct input_dev *input_dev;

	struct timer_list poll_timer;
};

int radiobtn_register_device(struct radio_button *);
void radiobtn_unregister_device(struct radio_button *);

#endif /* RADIOBTN_H */
