#include <iostream>
#include <qobject.h>
#include <qlist.h>
#include <qstringlist.h>
#include <qmessagebox.h>
#include <qregexp.h>

using std::cout;
using std::endl;

#include "drawable.h"
#include "molecule.h"
#include "bond.h"
#include "arrow.h"
#include "bracket.h"
#include "text.h"
#include "chemdata.h"
#include "defs.h"

void ChemData::Retro(Molecule *m1) {
  int frags = m1->Retro();

  // show or place or explain whether no fragments
  if (frags == 0) {
    QMessageBox::information(r, tr("Retrosynthesis"), tr("No fragments returned."), QMessageBox::Ok);
    return;
  }
  QMessageBox::information(r, tr("Retrosynthesis"), tr("Retrosynthesis"), QMessageBox::Ok);
}

// only retro need this function, so it's not part of a class
// determine whether a given bond mtaches a database entry with R groups
bool RParse(QString rlocal, QString rfromdb) {
  QString rwork = rlocal;  // make local copy
  if (rfromdb.contains("R") == 0) { // exact matches only
    if (rlocal == rfromdb)
      return true;
    else
      return false;
  }
  // find if rlocal matches R-containing wild card pattern from DB (rfromdb)

  // find "depth" of db string
  int nlevels, nleft, nright;
  if (rfromdb.contains("[") > 0) nlevels = 3;
  if (rfromdb.contains("(") > 0) nlevels = 2;
  if (rfromdb.contains("<") > 0) nlevels = 1;
  if (nlevels < 3) { // strip [] from rwork
    do {
      nleft = rwork.find("[");
      nright = rwork.find("]") + 1;
      rwork.remove(nleft, nright - nleft);
    } while ( rwork.contains("[") > 0 );
  }
  if (nlevels < 2) { // strip () from rwork
    do {
      nleft = rwork.find("(");
      nright = rwork.find(")") + 1;
      rwork.remove(nleft, nright - nleft);
    } while ( rwork.contains("(") > 0 );
  }

  std::cout << "part of rwork: " << rwork << std::endl;

  return false;
}

int Molecule::Retro() {
  MakeSSSR();

  // basic idea: iterate thru bonds, create something like a canonical
  // SMILES describing the bond and nearest neighbors, and compare to
  // database.  For certain reactions, do a second iteration over atoms
  // and bonds to rule out side products (e.g. Grignards won't work if
  // more than one ketone or -COOH)
  // DB would contain fields like [SMILES],[name or reaction],[conditions],
  // [stoppers (like (C=O) in Grignard as noted)]
  // canonical strings will likely take the form:
  // <1st(2nd[3rd])(2nd)>0-0<1st><1st(2nd)(2nd[3rd][3rd])>
  // ...where canonicalization will be accomplished by alphabetizing
  // within each level.

  QListIterator<Bond> innerIter1(bonds), innerIter2(bonds), innerIter3(bonds), 
    outerIter(bonds);
  QStringList innerstrlist1, innerstrlist2, innerstrlist3, hitlist;
  Bond *local_tmp_bond, *inner_tmp_bond;
  DPoint *outer1, *outer2, *inner1, *inner2, *inner3;
  QString canonicalBond, leveltmpstr, altCanonicalBond, workBond,
    outer1el, outer2el;
  while ( (local_tmp_bond = outerIter.current()) != 0 ) {
    ++outerIter;
    canonicalBond.setLength(0);

    // in the long run, it might be wise to break out this code for
    // making canonical SMILES-type names into a separate function to
    // facilitate making the database

    outer1 = local_tmp_bond->Start();
    outer2 = local_tmp_bond->End();
    if (QString::compare(outer1->element, outer2->element) > 0) {
      inner1 = outer1; outer1 = outer2; outer2 = inner1;
    }
    outer1el = outer1->element;
    canonicalBond = outer1el.remove("H");
    if (outer1->inring) canonicalBond.append("r");
    if (local_tmp_bond->Order() == 1) {
      canonicalBond.append("-");
    }
    if (local_tmp_bond->Order() == 5) {
      canonicalBond.append("-");
    }
    if (local_tmp_bond->Order() == 7) {
      canonicalBond.append("-");
    }
    if (local_tmp_bond->Order() == 2) {
      canonicalBond.append("=");
    }
    if (local_tmp_bond->Order() == 3) {
      canonicalBond.append("#");
    }
    outer2el = outer2->element;
    canonicalBond.append(outer2el.remove("H"));
    if (outer2->inring) canonicalBond.append("r");

    canonicalBond.prepend(RetroTraverseBonds(outer2, outer1, 
					     local_tmp_bond, 3));
    canonicalBond.append(RetroTraverseBonds(outer1, outer2, 
					    local_tmp_bond, 3));

    std::cout << "canonical bond name:" << canonicalBond << std::endl;
    altCanonicalBond = canonicalBond;
    altCanonicalBond.remove("r");
    std::cout << "alternate bond name:" << altCanonicalBond << std::endl;

    // do DB lookup here
    workBond = canonicalBond;
    QFile f(RingDir + "retro.txt");
    if (!f.open(IO_ReadOnly)) {
      QMessageBox::warning(0, tr("Couldn't open file"), tr("Could not open the retrosynthesis data file.") );
      return false;
    }
    f.close();
  }

  return 0;
}

QString Molecule::RetroBondName(Bond *lbond) {
  MakeSSSR();

  // basic idea: iterate thru bonds, create something like a canonical
  // SMILES describing the bond and nearest neighbors, and compare to
  // database.  For certain reactions, do a second iteration over atoms
  // and bonds to rule out side products (e.g. Grignards won't work if
  // more than one ketone or -COOH)
  // DB would conatin fields like [SMILES],[name or reaction],[conditions],
  // [stoppers (like (C=O) in Grignard as noted)]
  // canonical strings will likely take the form:
  // <1st(2nd[3rd])(2nd)>0-0<1st><1st(2nd)(2nd[3rd][3rd])>
  // ...where canonicalization will be accomplished by alphabetizing
  // within each level.

  QListIterator<Bond> innerIter1(bonds), innerIter2(bonds), innerIter3(bonds), 
    outerIter(bonds);
  QStringList innerstrlist1, innerstrlist2, innerstrlist3;
  Bond *local_tmp_bond, *inner_tmp_bond;
  DPoint *outer1, *outer2, *inner1, *inner2, *inner3;
  QString canonicalBond, leveltmpstr, outer1el, outer2el;

  canonicalBond.setLength(0);

  // in the long run, it might be wise to break out this code for
  // making canonical SMILES-type names into a separate function to
  // facilitate making the database

  local_tmp_bond = lbond;
  outer1 = local_tmp_bond->Start();
  outer2 = local_tmp_bond->End();
  if (QString::compare(outer1->element, outer2->element) > 0) {
    inner1 = outer1; outer1 = outer2; outer2 = inner1;
  }
  outer1el = outer1->element;
  canonicalBond = outer1el.remove("H");
  if (outer1->inring) canonicalBond.append("r");
  if (local_tmp_bond->Order() == 1) {
    canonicalBond.append("-");
  }
  if (local_tmp_bond->Order() == 5) {
    canonicalBond.append("-");
  }
  if (local_tmp_bond->Order() == 7) {
    canonicalBond.append("-");
  }
  if (local_tmp_bond->Order() == 2) {
    canonicalBond.append("=");
  }
  if (local_tmp_bond->Order() == 3) {
    canonicalBond.append("#");
  }
  outer2el = outer2->element;
  canonicalBond.append(outer2el.remove("H"));
  if (outer2->inring) canonicalBond.append("r");

  canonicalBond.prepend(RetroTraverseBonds(outer2, outer1, 
					   local_tmp_bond, 3));
  canonicalBond.append(RetroTraverseBonds(outer1, outer2, 
					  local_tmp_bond, 3));

  std::cout << "canonical bond name:" << canonicalBond << std::endl;

  return canonicalBond;
}

QString Molecule::RetroTraverseBonds(DPoint *parentNode, DPoint *thisNode,
				     Bond *this_bond, int tlevel) {
  QListIterator<Bond> innerIter1(bonds), innerIter2(bonds), innerIter(bonds), 
    outerIter(bonds);
  QStringList innerstrlist1, innerstrlist2, innerstrlist3;
  Bond *local_tmp_bond, *inner_tmp_bond;
  DPoint *outer1, *outer2, *inner1, *inner2, *inner3;
  QString canonicalBond, leveltmpstr, leftMarker, rightMarker, thislevelstr,
    outer1el, outer2el;
  if (tlevel == 3) { leftMarker = ""; rightMarker = ""; }
  if (tlevel == 2) { leftMarker = "<"; rightMarker = ">"; }
  if (tlevel == 1) { leftMarker = "("; rightMarker = ")"; }

  outer1 = thisNode;
  
  canonicalBond.setLength(0);
  if (tlevel == 0) { 
    leftMarker = "["; rightMarker = "]"; 
    if (thisNode->element == "H") {
      return canonicalBond;
    } else {
      if (this_bond->Order() == 1) {
	canonicalBond.append("-");
      }
      if (this_bond->Order() == 5) {
	canonicalBond.append("-");
      }
      if (this_bond->Order() == 7) {
	canonicalBond.append("-");
      }
      if (this_bond->Order() == 2) {
	canonicalBond.append("=");
      }
      if (this_bond->Order() == 3) {
	canonicalBond.append("#");
      }
      outer1el = thisNode->element;
      canonicalBond.append(outer1el.remove("H"));
      if (thisNode->inring) canonicalBond.append("r");
      canonicalBond.prepend(leftMarker);
      canonicalBond.append(rightMarker);
      return canonicalBond;
    }
  }

  if (this_bond->Order() == 1) {
    canonicalBond.append("-");
  }
  if (this_bond->Order() == 5) {
    canonicalBond.append("-");
  }
  if (this_bond->Order() == 7) {
    canonicalBond.append("-");
  }
  if (this_bond->Order() == 2) {
    canonicalBond.append("=");
  }
  if (this_bond->Order() == 3) {
    canonicalBond.append("#");
  }
  outer1el = thisNode->element;
  canonicalBond.append(outer1el.remove("H"));
  if (thisNode->inring) canonicalBond.append("r");
  
  innerIter.toFirst();
  innerstrlist1.clear();

  while ( (inner_tmp_bond = innerIter.current()) != 0 ) {
    ++innerIter;
    if (inner_tmp_bond->Find(outer1) == true) {
      inner1 = inner_tmp_bond->otherPoint(outer1);
      if (inner1 != parentNode) {
	std::cout << inner1->element << std::endl;
	if (inner1->element != "H") {
	  // traverse next level (level 0 handled elsewhere)
	  innerstrlist1.append(RetroTraverseBonds(outer1, inner1, 
						  inner_tmp_bond, tlevel - 1));
	}
      }
    }
  }
  innerstrlist1.sort();

  if (tlevel == 3) canonicalBond.setLength(0);
  for ( QStringList::Iterator it = innerstrlist1.begin(); 
	it != innerstrlist1.end(); ++it ) {
    canonicalBond.append(*it);
  }

  canonicalBond.prepend(leftMarker);
  canonicalBond.append(rightMarker);
  return canonicalBond;
}
