/* -*- C++ -*-
 *
 * Copyright (C) 2000 Frans Kaashoek (kaashoek@mit.edu)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 */

#ifndef _SFSROSD_H_
#define _SFSROSD_H_

#define USE_STL

#include "sfsmisc.h"
#include "nfs3_prot.h"
#include "nfstrans.h"
#include "sfs_prot.h"
#include "arpc.h"
#include "crypt.h"
#include "sfsserv.h"
#include "xdfs_cpp.h"

class XdfsSrv;
class Proxy;

struct XNFS_STAT {
    uint32   mode;
    uint32   uid;
    uint32   gid;
    nfstime3 atime;
    nfstime3 mtime;
    nfstime3 ctime;
};

extern XSTCK __XNFS_STAT_STCK;

extern int    __xdfssrv_argc;
extern char** __xdfssrv_argv;

class XdfsSrv
{
public:
    XdfsSrv ()
	: _dbfs (__xdfssrv_argc, __xdfssrv_argv)
    {
	//warnx << "construct XdfsSrv\n";
    }

    ~XdfsSrv ()
    {
	//warnx << "destruct XdfsSrv\n";
    }

    const char* get_path  (guint64 key);

    sfs_servinfo     _servinfo;
    sfs_hash         _hostid;
    ptr<rabin_priv>  _sk;
    DBFS             _dbfs;
};

class Proxy
    : public sfsserv
{
public:

    Proxy (ref<axprt_crypt> x)
	: sfsserv (x),
	  _rwsrv  (asrv::alloc (x, ex_nfs_program_3, wrap (this, &Proxy::nfs3dispatch)))
    {
	//warnx << "construct Proxy\n";
    }

    ~Proxy ()
    {
	//warnx << "destruct Proxy\n";
    }

    void sfs_getfsinfo (svccb *sbp);

private:

    void nfs3dispatch  (svccb *sbp);

    int nfs3_getattr  (TXN *txn, svccb *sbp);
    int nfs3_access   (TXN *txn, svccb *sbp);
    int nfs3_fsinfo   (TXN *txn, svccb *sbp);
    int nfs3_fsstat   (TXN *txn, svccb *sbp);
    int nfs3_lookup   (TXN *txn, svccb *sbp);
    int nfs3_readdir  (TXN *txn, svccb *sbp);
    int nfs3_read     (TXN *txn, svccb *sbp);
    int nfs3_create   (TXN *txn, svccb *sbp);
    int nfs3_write    (TXN *txn, svccb *sbp);
    int nfs3_commit   (TXN *txn, svccb *sbp);
    int nfs3_remove   (TXN *txn, svccb *sbp); // Also handles nfs3_rmdir
    int nfs3_rename   (TXN *txn, svccb *sbp);
    int nfs3_link     (TXN *txn, svccb *sbp);
    int nfs3_mkdir    (TXN *txn, svccb *sbp);
    //int nfs3_symlink  (TXN *txn, svccb *sbp);
    //int nfs3_readlink (TXN *txn, svccb *sbp);
    int nfs3_setattr  (TXN *txn, svccb *sbp);

    int get_node      (TXN *txn, svccb *sbp, guint64 inum, MAJORC &inop);
    int get_dnode     (TXN *txn, svccb *sbp, guint64 inum, MAJORC &inop);
    int get_snode     (TXN *txn, svccb *sbp, guint64 inum, MAJORC &inop);

    int get_attr      (TXN *txn, svccb *sbp, guint64 inum, ex_fattr3 &attr);
    int get_attr      (svccb *sbp, MAJORC &major, ex_post_op_attr &attr);
    int get_attr      (svccb *sbp, MAJORC &major, ex_fattr3 &attr);

    int unlink        (TXN *txn, svccb *sbp, diropargs3* dirop, MAJORC &del_ino);

    int create_node   (TXN          *txn,
			svccb        *sbp,
			diropargs3   *dirop,
			XTYPE         type,
			const sattr3 &attributes,
			MAJORC       &cinop,
			MAJORC       &dinop);

    void xattr_init    (svccb *sbp, const sattr3 &attributes, XNFS_STAT &xa);
    int create_link   (svccb *sbp, MAJORC &dir_inop, MAJORC &linkto, DKEY &dkey, int flags);
    int update_mtime  (svccb *sbp, MAJORC &inop);
    int dir_empty     (svccb *sbp, MAJORC &inop, bool *empty);

    // set_fh
    void set_fh (nfs_fh3 &fh, guint64 inum)
    {
	fh.data.setsize (sizeof (inum));
	memcpy (fh.data.base (), & inum, sizeof (inum));
    }

    void set_fh (post_op_fh3 &fh, guint64 inum)
    {
	set_fh (*fh.handle, inum);
    	fh.set_present (true);
    }

    void set_fh (nfs_fh3     &fh, MAJORC &ino) { set_fh (fh, ino.number ().key ()); }
    void set_fh (post_op_fh3 &fh, MAJORC &ino) { set_fh (fh, ino.number ().key ()); }

    // set_attr
    void set_attr (ex_post_op_attr &attr, const ex_fattr3 &val)
    {
	attr.set_present (true);
	*attr.attributes = val;
    }

    // get_inum
    guint64 get_inum (const nfs_fh3 *fh)
    {
	return get_inum (* fh);
    }

    guint64 get_inum (const nfs_fh3 &fh)
    {
	guint64 inum;

	assert (fh.data.size () == sizeof (inum));

	memcpy (& inum, fh.data.base (), sizeof (inum));

	return inum;
    }

protected:

    ptr<rabin_priv> doconnect (const sfs_connectarg *, sfs_servinfo *);

private:

    XdfsSrv          *_xsrv;
    ptr<asrv>         _rwsrv;
};

// @@@ You should make a macro for THIS
extern MESSAGE_TYPE XNFS_ERROR_MESSAGE;
#define XNFS_ERROR \
	if (XNFS_ERROR_MESSAGE.active) MESSAGE_CLOSURE_FL (XNFS_ERROR_MESSAGE)

int      add_dots   (MAJORC &inop, MAJORC &par_inop);
XdfsSrv* get_server (void);

inline nfstime3 nfs_now (void)
{
    struct timeval now_tv;
    nfstime3 now;

    gettimeofday (& now_tv, NULL);

    now.seconds  = now_tv.tv_sec;
    now.nseconds = now_tv.tv_usec * 1000;

    return now;
}

void xattr_update_time (const set_time &st, const nfstime3 &now, nfstime3 &val);
void xattr_update      (const sattr3 &attributes, const nfstime3 &now, XNFS_STAT &xa);
int  xattr_set         (MAJORC &ino, const XNFS_STAT &xa);
int  xattr_get         (MAJORC &ino, XNFS_STAT &xa);

#endif _SFSROSD_H_
