/* -*-Mode: C++;-*-
 * $Id: xdfs_list.cc 1.1 Wed, 25 Apr 2001 08:44:53 +0400 jmacd $
 *
 * Copyright (C) 1999, 2000, Joshua P. MacDonald <jmacd@CS.Berkeley.EDU>
 * and The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 *    Neither name of The University of California nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "xdfs_cpp.h"
#include "xdfs_list.h"

//////////////////////////////////////////////////////////////////////
// CIRCULAR_LINK
//////////////////////////////////////////////////////////////////////

CIRCULAR_LINK::CIRCULAR_LINK (CIRCULAR_LINKS &clnks,
			      const MKEY     &mkey)
    : _clnks (clnks),
      _mkey  (mkey)
{
}

void
CIRCULAR_LINK::reset ()
{
    _lnk.reset ();
    _lnknode.reset ();
}

int
CIRCULAR_LINK::get (MAJORC &node)
{
    int ret;

    if (! _lnknode.is_valid ()) {

	if ((ret = init_lnk ()) || (ret = _lnk.read_reflink (_lnknode, 0))) {
	    PROP_ERROR (ret) ("read_reflink");
	    return ret;
	}
    }

    node = _lnknode;

    return 0;
}

int
CIRCULAR_LINK::init_lnk ()
{
    int ret;

    if (! _lnk.is_valid ()) {

	if ((ret = _clnks._node.node_mkey (_mkey, _lnk))) {
	    PROP_ERROR (ret) ("node_mkey");
	    return ret;
	}
    }

    return 0;
}

int
CIRCULAR_LINK::link (const MAJORC &node, int flags)
{
    int ret;

    if ((ret = init_lnk ()) ||
	(ret = _lnk.mk_reflink (_lnknode = node, flags))) {
	if (ret != DBFS_EXISTS) {
	    PROP_ERROR (ret) ("mk_reflink");
	}
	return ret;
    }

    return 0;
}

//////////////////////////////////////////////////////////////////////
// CIRCULAR_LINKS
//////////////////////////////////////////////////////////////////////

CIRCULAR_LINKS::CIRCULAR_LINKS (CIRCULAR_LIST &cll,
				const MAJORC  &node)
    : _cll  (cll),
      _node (node),
      _next (*this, cll._nextmk),
      _prev (*this, cll._prevmk)
{
}

int
CIRCULAR_LINKS::reset (const MAJORC &node)
{
    _node = node;

    _next.reset ();
    _prev.reset ();

    return 0;
}

int
CIRCULAR_LINKS::move_next (MAJORC &nextnode)
{
    int ret;

    if ((ret = get_next (nextnode)) != 0) {
	return ret;
    }

    if (nextnode == _cll._head) {
	return DBFS_NOTFOUND;
    }

    return reset (nextnode);
}

//////////////////////////////////////////////////////////////////////
// CIRCULAR_LIST
//////////////////////////////////////////////////////////////////////

CIRCULAR_LIST::CIRCULAR_LIST ()
    : _nextmk (__CLL_NEXT_STCK),
      _prevmk (__CLL_PREV_STCK)
{
}

bool
CIRCULAR_LIST::next (MAJORC &node)
{
    int ret;

    if (! _iter.is_valid ()) {
	_iter = _head;
    }

    CIRCULAR_LINKS lnks (*this, _iter);

    if ((ret = lnks.move_next (node))) {
	if (ret != DBFS_NOTFOUND) {
	    PROP_ERROR (ret) ("move_next");
	}
	return false;
    }

    _iter = node;
    return true;
}

int CIRCULAR_LIST::create (const MAJORC &head)
{
    int ret;

    _head = head;

    CIRCULAR_LINKS headlnks (*this, _head);

    if ((ret = headlnks.link_next (_head, DBFS_NOOVERWRITE)) ||
	(ret = headlnks.link_prev (_head, DBFS_NOOVERWRITE))) {
	PROP_ERROR (ret) ("cll_linkto");
	return ret;
    }

    return 0;
}

int CIRCULAR_LIST::open (const MAJORC &head)
{
    // No checks performed...
    _head = head;

    return 0;
}

int
CIRCULAR_LIST::find_nth (int           N,
			 MAJORC       &node)
{
    int ret = 0;
    CIRCULAR_LINKS l (*this, _head);

    // Counting from zero
    while (N > 0 && (ret = l.move_next (node)) == 0) {

	N -= 1;
    }

    if (ret == DBFS_NOTFOUND) {
	return DBFS_NOTFOUND;
    }

    if (ret != 0) {
	PROP_ERROR (ret) ("move_next");
    }

    return 0;
}

int
CIRCULAR_LIST::get_pos (CIRCULAR_POS pos, MAJORC &node)
{
    if (pos == POS_HEADNODE) {
	node = _head;
	return 0;
    }

    CIRCULAR_LINKS headlnks (*this, _head);

    if (pos == POS_NEXTNODE) {
	return headlnks.get_next (node);
    } else {
	return headlnks.get_prev (node);
    }
}

int
CIRCULAR_LIST::insert (CIRCULAR_POS  next_pos,
		       const MAJORC &item_node)
{
    int ret;
    MAJORC next_node;
    MAJORC prev_node;

    if ((ret = get_pos (next_pos, next_node))) {
	PROP_ERROR (ret) ("get_next_pos");
	return ret;
    }

    CIRCULAR_LINKS next_links (*this, next_node);

    if ((ret = next_links.get_prev (prev_node))) {
	PROP_ERROR (ret) ("next_get_prev");
	return ret;
    }

    CIRCULAR_LINKS item_links (*this, item_node);
    CIRCULAR_LINKS prev_links (*this, prev_node);

    if ((ret = item_links.link_next (next_node, DBFS_NOOVERWRITE)) ||
	(ret = item_links.link_prev (prev_node, DBFS_NOOVERWRITE)) ||
	(ret = prev_links.link_next (item_node, DBFS_OVERWRITE)) ||
	(ret = next_links.link_prev (item_node, DBFS_OVERWRITE))) {
	if (ret != DBFS_EXISTS) {
	    PROP_ERROR (ret) ("linkto");
	}
	return ret;
    }

    return 0;
}
