/* -*-Mode: C++;-*-
 * $Id: xdfs.cc 1.51 Fri, 29 Jun 2001 16:59:43 +0400 jmacd $
 *
 * Copyright (C) 1999, 2000, 2001, Joshua P. MacDonald <jmacd@CS.Berkeley.EDU>
 * and The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 *    Neither name of The University of California nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "xdfs_cpp.h"
#include "xdfs_comp.h"
#include "edsiostdio.h"
#include <errno.h>

#define CONT_MKEY MKEY(__COMP_CONTROL_STCK)
#define IDX_MKEY  MKEY(__COMP_INDEX_STCK)
#define STATE_MKEY MKEY(__COMP_STATE_STCK)
#define SHAREDLNK_MKEY MKEY(__COMP_SHAREDLNK_STCK)

class XdfsView
    : public VIEWDEF
{
public:

    XdfsView ()
	: VIEWDEF (__COMP_RECONSTRUCT_VIEW) { }

    VIEWIMPL* make ()
    {
	return new XdfsViewImpl;
    }

    const char* name ()
    {
	return "XDFS";
    }
};

static XdfsView xdfs_reconstruct_view;

int   xdfs_shared_index (TXN       &txn,
			 XdfsFlags  flags,
			 NODEC     &idxnode)
{
    int ret;

    g_assert (flags & XF_IdxShared);
    g_assert (flags & XF_Approximate);

    if ((ret = txn.root ().node_mkey (IDX_MKEY, idxnode))) {
	PROP_ERROR (ret) ("node_mkey");
	return ret;
    }

    if ((idxnode.xtype () != XTYPE_DIRBTREE) &&
	(ret = idxnode.mk_directory (XTYPE_DIRBTREE, DBFS_NOOVERWRITE))) {
	PROP_ERROR (ret) ("mk_index");
	return ret;
    }

    return 0;
};

/**********************************************************************/
/*			   Maintenence Code                           */
/**********************************************************************/

static const char*
xdfs_policy_to_string (XdfsPolicy policy)
{
    switch (policy) {
    case XP_Reverse:       return "Reverse";
    case XP_Forward:       return "Forward";
    case XP_ForwardSingle: return "Forward (Single Cluster)";
    case XP_NoCompress:    return "None";
    }

    return "Error";
}

const char*
xdfs_flags_to_string (int flags)
{
    static char buf[128];

    buf[0] = 0;

    if (flags & XF_MD5Equality) {
	strcat (buf, "MD5 ");
    }

    if (flags & XF_SHAEquality) {
	strcat (buf, "SHA ");
    }

    if (flags & XF_IdxReverse) {
	strcat (buf, "IdxReverse ");
    }

    if (! buf[0]) {
	strcat (buf, "(none)");
    }

    return buf;
}

int
xdfs_library_init (void)
{
    // @@ Get rid of Ugly
    if (! edsio_library_init ()) {
	goto noprop;
    }

    if (! edsio_edsio_init ()) {
	goto noprop;
    }

    if (! xdfs_edsio_init ()) {
	goto noprop;
    }

    if (! xd_edsio_init ()) {
	goto noprop;
    }

    return 0;

  noprop:

    PROP_ERROR ("xdfs_library_init");
    return -1;
}

int
xdfs_state_print (FileHandle *handle,
		  XdfsState  *state)
{
    guint32 encoded_size = state->literal_size + state->control_size + state->patch_size;

    if (! handle_printf (handle, "Compression policy:          %s\n",
			 xdfs_policy_to_string ((XdfsPolicy) state->policy)) ||
	! handle_printf (handle, "Flags:                       %s\n", xdfs_flags_to_string (state->flags)) ||
	! handle_printf (handle, "Version count:               %d\n", state->total_version_count) ||
	! handle_printf (handle, "Unique Version count:        %d\n", state->unique_version_count) ||
	! handle_printf (handle, "Average file size:           %.0f\n",
			 (double) state->unencoded_size / (double) state->unique_version_count) ||
	! handle_printf (handle, "Unencoded size:              %d\n", state->unencoded_size) ||
	! handle_printf (handle, "Encoded size:                %d\n", encoded_size) ||
	! handle_printf (handle, "Literal size:                %d\n", state->literal_size) ||
	! handle_printf (handle, "Control size:                %d\n", state->control_size) ||
	! handle_printf (handle, "Patch size:                  %d\n", state->patch_size) ||
	! handle_printf (handle, "Ideal compression:           %0.3f\n",
			 (double) encoded_size / (double) state->unencoded_size) ||
	! handle_printf (handle, "Patch files:                 %d\n", state->patches) ||
	! handle_printf (handle, "Literal files:               %d\n", state->literals) ||
	! handle_printf (handle, "Current cluster versions:    %d\n", state->cluster_versions)) {
	PROP_ERROR ("handle_printf");
	return -1;
    }

    if (state->policy & XP_IsReverse &&
	! handle_printf (handle, "Max cluster versions:        %d\n", state->cluster_max_versions)) {
	PROP_ERROR ("handle_printf");
	return -1;
    }

    return 0;
}

int
XdfsLocation::stat (XdfsState    &state)
{
    state = (* _stateX);

    return 0;
}

int
XdfsLocation::print (FileHandle *out)
{
    return xdfs_state_print (out, _stateX);
}

static XdfsState*
xdfs_state_init (XdfsParams *params)
{
    XdfsState *state = g_new0 (XdfsState, 1);

    /* Default values */
    state->policy               = XP_Forward;
    state->cluster_max_versions = 40;

    /* User inputs */
    if (params) {
	state->flags = params->flags;

	if (params->policy != 0) {
	    state->policy = params->policy;
	}

	if (params->cluster_max_versions != 0) {
	    state->cluster_max_versions = params->cluster_max_versions;
	}
    }

    return state;
}

static int
xdfs_state_idxflags (XdfsState *state, const MessageDigest *&md)
{

    bool md5 = (state->flags & XF_MD5Equality);
    bool sha = (state->flags & XF_SHAEquality);

    if (md5 && sha) {
	int ret = DBFS_INVAL;
	XDFS_ERROR (ret) ("too many message digests");
	return ret;
    }

    if (md5) {
	md = edsio_message_digest_md5 ();
    } else if (sha) {
	md = edsio_message_digest_sha ();
    }

    return 0;
}

static int
xdfs_state_read (NODEC &node, XdfsState *&state, const MessageDigest *&md)
{
    SerialSource *src = NULL;
    FileHandle   *set_in = NULL;
    int           ret;

    g_assert (node.mkey () == STATE_MKEY);

    if ((ret = node.read_segment (& set_in, 0))) {
	if (ret != DBFS_NOTFOUND) {
	    PROP_ERROR (ret) ("read_segment");
	}
	return ret;
    }

    src = handle_source (set_in);

    if (! unserialize_xdfsstate (src, & state)) {
	PROP_ERROR ("unserialize_xdfsstate");
	return -1;
    }

    if (! handle_close (set_in)) {
	PROP_ERROR ("handle_close");
	return -1;
    }

    handle_free (set_in);
    src->source_free (src);

    if ((ret = xdfs_state_idxflags (state, md))) {
	PROP_ERROR (ret) ("xdfs_state_idxflags");
	return ret;
    }

    return 0;
}

static int
xdfs_state_write (NODEC &node, XdfsState *state, int flag, const MessageDigest *&md)
{
    SerialSink *sink;
    FileHandle *set_out;
    int         ret;

    g_assert (node.mkey () == STATE_MKEY);

    if ((ret = node.repl_segment (& set_out, flag))) {
	PROP_ERROR (ret) ("repl_segment");
	return ret;
    }

    sink = handle_sink (set_out, NULL, NULL, NULL, NULL);

    if (! serialize_xdfsstate_obj (sink, state)) {
	PROP_ERROR ("serialize_xdfsstate_obj");
	return -1;
    }

    if (! handle_close (set_out)) {
	PROP_ERROR ("handle_close");
	return -1;
    }

    handle_free (set_out);
    sink->sink_free (sink);

    if ((ret = xdfs_state_idxflags (state, md))) {
	PROP_ERROR (ret) ("xdfs_state_idxflags");
	return ret;
    }

    return 0;
}


static int
xdfs_control_write (NODEC         &node,
		    XdeltaControl *control,
		    int            flag)
{
    FileHandle *ch = NULL;
    int ret;

    g_assert (node.mkey () == CONT_MKEY);

    if ((ret = node.repl_segment (& ch, flag))) {
	PROP_ERROR (ret) ("repl_segment");
	goto exit;
    }

    if (! xdp_control_write (control, ch)) {
	ret = -1;
	PROP_ERROR ("xfds_control_write");
	goto exit;
    }

    if (! handle_close (ch)) {
	ret = -1;
	PROP_ERROR ("handle_close");
	goto exit;
    }

    DEBUG_XDFS ("control_write: ") (node);

  exit:

    if (ch) { handle_free (ch); }

    return ret;
}

static int
xdfs_control_read (NODEC          &node,
		   XdeltaControl **control_out)
{
    FileHandle *ch = NULL;
    int ret;

    g_assert (node.mkey () == CONT_MKEY);

    DEBUG_XDFS ("control_read: ") (node);

    if ((ret = node.read_segment (& ch, DBFS_NOFLAG))) {
	PROP_ERROR (ret) ("control_read_segment");
	goto exit;
    }

    if (! ((* control_out) = xdp_control_read (ch))) {
	ret = -1;
	PROP_ERROR ("xdp_control_read");
	goto exit;
    }

    if (! handle_close (ch)) {
	ret = -1;
	PROP_ERROR ("handle_close");
	goto exit;
    }

  exit:

    if (ch) { handle_free (ch); }

    return ret;
}

// This returns 0 if the new version is inserted or DBFS_EXISTS if
// the version was found.
static int
xdfs_drain_copy (FileHandle *fh_in,
		 const SAREA &area,
		 const NODEC &idx,
		 int         flags,
		 const MessageDigest *md,
		 MAJORC     &copy_node,
		 MAJORC     &xdfsloc_node,
		 MAJORC     &exist_node)
{
    int ret;
    FileHandle *copy = NULL;

    // Create the new file entry by draining IN, meanwhile compute digest.
    // The use of DBFS_ALLOC_ASCEND is important due to the furthest() test
    // (which is somewhat bogus)--it allows the comparison of inum for an
    // age test, used for the DAG traversal algorithm.
    if ((ret = area.allocate (copy_node, DBFS_ALLOC_ASCEND))) {
	PROP_ERROR (ret) ("loc_sarea_allocate");
	goto exit;
    }

    if ((ret = copy_node.repl_segment (& copy, DBFS_NOOVERWRITE))) {
	PROP_ERROR (ret) ("copy_repl_segment");
	goto exit;
    }

    if (md && ! handle_digest_compute (copy, md)) {
	PROP_ERROR ("handle_digest_compute");
	goto exit;
    }

    if (! handle_drain (fh_in, copy)) {
	PROP_ERROR ("handle_drain");
	goto exit;
    }

    if (! handle_close (fh_in)) {
	PROP_ERROR ("handle_close");
	goto exit;
    }

    if (! handle_close (copy)) {
	PROP_ERROR ("handle_close");
	goto exit;
    }

    if (md) {

	const guint8 *ins_digest  = NULL;
	DIRC idx_curs;

	if ((ret = idx.dir_open (idx_curs))) {
	    PROP_ERROR (ret) ("idx_dir_open");
	    goto exit;
	}

	// Look for the digest entry in the index, to see if it already exists.
	if (! (ins_digest = handle_digest (copy, md))) {
	    PROP_ERROR ("handle_digest");
	    goto exit;
	}

	// This sets ret to DBFS_NOTFOUND or 0
	if ((ret = idx_curs.set_node (DKEY (ins_digest, md->cksum_len), exist_node, DBFS_LINK_RMW)) &&
	    (ret != DBFS_NOTFOUND)) {

	    PROP_ERROR (ret) ("idx_curs_set_node");
	    goto exit;
	}

	if (ret == 0) {

	    // The insert version exists.
	    // delete the copy and return: node_out is set
	    if ((ret = copy_node.del_major ())) {
		PROP_ERROR (ret) ("copy_del_major");
		goto exit;
	    }

	    // Need to set xdfsloc_node: if XF_IdxShared is set
	    // then read the special ARCHIVE link, otherwise it
	    // is left unset.
	    if (flags & XF_IdxShared) {
		NODEC arlnk;

		if ((ret = exist_node.node_mkey (SHAREDLNK_MKEY, arlnk))) {
		    PROP_ERROR (ret) ("node_mkey: shared-idx link");
		    return ret;
		}

		if ((ret = arlnk.read_reflink (xdfsloc_node, 0))) {
		    PROP_ERROR (ret) ("read_reflink: shared-idx link");
		    return ret;
		}
	    }

	    ret = DBFS_EXISTS;
	} else {

	    //DEBUG_XDFS ("new unique version, insert digest link: ") (copy_node) (" in index ") (idx);

	    if ((ret = idx_curs.insert_link (DKEY (ins_digest, md->cksum_len),
					     copy_node,
					     DBFS_NOOVERWRITE | ((flags & XF_IdxReverse) ?
								 DBFS_LINK_REVERSIBLE :
								 0)))) {
		PROP_ERROR (ret) ("idx_curs_insert_link");
		goto exit;
	    }
	}
    }

    handle_free (copy);

    return ret;

  exit:
    if (ret == 0) { ret = -1; }
    if (copy) { handle_free (copy); }

    return ret;
}

/**********************************************************************/
/*			    XdfsLocation                              */
/**********************************************************************/

int
XdfsLocation::create_first (SAREA      &sarea,
			    XdfsParams *params,
			    FileHandle *in,
			    MAJORC     &xdfsloc_out,
			    MAJORC     &version_out)
{
    int    ret;
    MAJORC copy_node;
    NODEC  idx_node;
    const MessageDigest *md = NULL;

    XdfsState *state = xdfs_state_init (params);

    if (state->flags & XF_IdxShared) {

	if ((ret = xdfs_state_idxflags (state, md))) {
	    PROP_ERROR (ret) ("xdfs_state_idxflags");
	    return ret;
	}

	if ((ret = xdfs_shared_index (sarea.txn (), (XdfsFlags) state->flags, idx_node))) {
	    PROP_ERROR (ret) ("xdfs_shared_index");
	    return ret;
	}
    }

    if ((ret = xdfs_drain_copy (in, sarea, idx_node, state->flags, md,
				copy_node, xdfsloc_out, version_out))) {
	// If DBFS_EXISTS is returned, then both COPY_NODE and
	// VERSION_OUT have been set in drain_copy
	if (ret != DBFS_EXISTS) {
	    PROP_ERROR (ret) ("xdfs_drain_copy");
	} else {
	    g_assert (xdfsloc_out.is_valid () && version_out.is_valid ());
	}
	return ret;
    }

    if ((ret = create_internal (sarea, state, xdfsloc_out))) {
	PROP_ERROR (ret) ("create_internal");
	return ret;
    }

    if ((ret = insert_new_version (copy_node, version_out))) {

	if (ret == DBFS_EXISTS) {
	    XDFS_ERROR ("Internal error: comp-archive-lnk already exists (create)");
	    return DBFS_LINK_INCONSISTENT;
	}

	PROP_ERROR (ret) ("insert_new_version");
	return ret;
    }

    return 0;
}

int
XdfsLocation::create (SAREA &sarea, XdfsParams *params, MAJORC &xdfsloc_out)
{
    int ret;
    XdfsState *state = xdfs_state_init (params);

    if ((ret = create_internal (sarea, state, xdfsloc_out))) {
	PROP_ERROR (ret) ("create_internal");
	return ret;
    }

    if ((ret = xdfs_state_write (_state_node, state, DBFS_NOOVERWRITE, _md))) {
	PROP_ERROR (ret) ("xdfs_state_write");
	return ret;
    }

    return 0;
}

int
XdfsLocation::create_internal (SAREA &sarea, XdfsState *state, MAJORC &xdfsloc_out)
{
    int ret;

    // Location
    if ((ret = sarea.allocate (_loc_node, DBFS_ALLOC_ASCEND))) {
	PROP_ERROR (ret) ("handle_close");
	return ret;
    }

    // State
    if ((ret = _loc_node.node_mkey (STATE_MKEY, _state_node))) {
	PROP_ERROR (ret) ("loc_state_mkey");
	return ret;
    }

    _stateX = state;

    // Index
    if (has_idx ()) {

	if (_stateX->flags & XF_IdxShared) {

	    if ((ret = xdfs_shared_index (sarea.txn (), (XdfsFlags) _stateX->flags, _idx_node))) {
		PROP_ERROR (ret) ("xdfs_shared_index");
		return ret;
	    }

	} else {

	    if ((ret = _loc_node.node_mkey (IDX_MKEY, _idx_node))) {
		PROP_ERROR (ret) ("loc_idx_mkey");
		return ret;
	    }

	    if ((ret = _idx_node.mk_directory (XTYPE_DIRHASH, DBFS_NOOVERWRITE))) {
		PROP_ERROR (ret) ("idx_mk_directory");
		return ret;
	    }
	}

	g_assert (_idx_node.is_container ());
    }

    xdfsloc_out = _loc_node;

    return 0;
}

int
XdfsLocation::open (const MAJORC &node)
{
    int ret;

    // Location
    _loc_node = node;

    // State
    if ((ret = _loc_node.node_mkey (STATE_MKEY, _state_node))) {
	PROP_ERROR (ret) ("loc_state_mkey");
	return ret;
    }

    // Check Notfound
    if (! _state_node.is_segment ()) {
	return DBFS_NOTFOUND;
    }

    if ((ret = xdfs_state_read (_state_node, _stateX, _md))) {
	PROP_ERROR (ret) ("xdfs_state_read");
	return ret;
    }

    // Index
    if (has_idx ()) {

	if (_stateX->flags & XF_IdxShared) {

	    if ((ret = xdfs_shared_index (node.txn (), (XdfsFlags) _stateX->flags, _idx_node))) {
		PROP_ERROR (ret) ("xdfs_shared_index");
		return ret;
	    }

	} else if ((ret = _loc_node.node_mkey (IDX_MKEY, _idx_node))) {
	    PROP_ERROR (ret) ("loc_idx_mkey");
	    return ret;
	}

	g_assert (_idx_node.is_container ());
    }

    return 0;
}

int
XdfsLocation::insert_version (FileHandle *in, MAJORC &node_out)
{
    MAJORC copy_node;
    MAJORC xdfsloc_out;
    int    ret;

    if ((ret = xdfs_drain_copy (in, _loc_node.sarea (), _idx_node, _stateX->flags, _md,
				copy_node, xdfsloc_out, node_out))) {

	if (ret != DBFS_EXISTS) {
	    PROP_ERROR (ret) ("xdfs_drain_copy");
	} else {

	    g_assert (node_out.is_valid ());

	    if (xdfsloc_out.is_valid () && xdfsloc_out != _loc_node) {
		// @@@ Analyze this.
		XDFS_ERROR (ret) ("WARNING: versions exists in another archive");
		return DBFS_LINK_INCONSISTENT;
	    }
	}
	return ret;
    }

    if ((ret = insert_new_version (copy_node, node_out))) {

	if (ret == DBFS_EXISTS) {
	    XDFS_ERROR ("Internal error: comp-archive-lnk already exists (insert)");
	    return DBFS_LINK_INCONSISTENT;
	}

	PROP_ERROR (ret) ("insert_new_version");
	return ret;
    }

    return 0;
}

#define FIXLEN(x) ((x).key ())

// This is where the compression actually happens
int
XdfsLocation::insert_new_version (const MAJORC &copy_node, MAJORC &node_out)
{
    XdeltaGenerator *gen;
    FileHandle      *delta_to;
    FileHandle      *patch_out;
    XdeltaControl   *control;
    XdfsPolicy       policy = (XdfsPolicy) _stateX->policy;
    bool             reject;
    NODEC            cont_node;
    NODEC            curlnk_node;
    MAJORC           patch_link_node;
    MAJORC           patch_node;
    MAJORC           curval_node;
    MAJORC           to_node;
    MAJORC           from_node;
    int              ret;

    // Update counters.
    _stateX->literals             += 1;
    _stateX->unique_version_count += 1;
    _stateX->total_version_count  += 1; // @@ Note: no difference
    _stateX->unencoded_size       += FIXLEN (copy_node.length ());
    _stateX->literal_size         += FIXLEN (copy_node.length ());

    // If no compression, that's all there was to do.  (There's no current
    // link for XP_NoCompress archives).
    if (policy == XP_NoCompress) {
	return 0;
    }

    // Lookup current version in xdfsdir.
    switch ((ret = _loc_node.node_exists (DBFS_DEF_MKEY, curlnk_node))) {
    case DBFS_NOTFOUND: {
	// If the current version link isn't there, then this is the first version.
	// Create the link and return.

	DEBUG_XDFS ("insert_new_version first: ") (copy_node);

	if ((ret = curlnk_node.mk_reflink (copy_node, DBFS_NOOVERWRITE))) {
	    PROP_ERROR (ret) ("curlnk_mk_reflink");
	    return ret;
	}

	_stateX->cluster_versions          = 1;
	_stateX->cluster_uncompressed_size = FIXLEN (copy_node.length ());
	_stateX->cluster_compressed_size   = FIXLEN (copy_node.length ());

	goto done;
    }

    case 0:
	break;

    default:
	PROP_ERROR (ret) ("loc_curlnk_mkey");
	return ret;
    }

    // Prepare to compute the delta
    gen = xdp_generator_new ();

    if ((ret = curlnk_node.read_reflink (curval_node, DBFS_LINK_RMW))) {
	PROP_ERROR (ret) ("curlnk_read_reflink");
	return ret;
    }

    if (policy & XP_IsReverse) {
	// Current is being replaced by reverse delta.

	from_node = copy_node;
	to_node   = curval_node;
    } else {
	// Current stays the same, compute a forward delta.

	from_node = curval_node;
	to_node   = copy_node;
    }

    if ((ret = source_segment (gen, XS_Primary, from_node))) {
	PROP_ERROR (ret) ("source_segment");
	return ret;
    }

    if ((ret = to_node.read_segment (& delta_to, DBFS_NOFLAG))) {
	PROP_ERROR (ret) ("to_read_segment");
	return ret;
    }

    // Now compute the delta.
    if ((ret = _loc_node.sarea ().allocate (patch_node, DBFS_ALLOC_ASCEND))) {
	PROP_ERROR (ret) ("loc_sarea_allocate");
	return ret;
    }

    if ((ret = patch_node.repl_segment (& patch_out, DBFS_NOOVERWRITE))) {
	PROP_ERROR (ret) ("patch_repl_segment");
	return ret;
    }

    if (! (control = xdp_generate_delta (gen, delta_to, NULL, patch_out))) {
	PROP_ERROR ("xdp_generate_delta");
	return -1;
    }

    if (! handle_close (delta_to)) {
	PROP_ERROR ("handle_close");
	return -1;
    }

    handle_free (delta_to);
    handle_free (patch_out);

    // Write its control.  Writing it before deciding to reject, meaning
    // that if it gets rejected the pseudo-delta control will be
    // re-written.  This sets cont_node.
    if ((ret = to_node.node_mkey (CONT_MKEY, cont_node))) {
	PROP_ERROR (ret) ("to_cont_mkey");
	return ret;
    }

    if ((ret = xdfs_control_write (cont_node, control, DBFS_NOOVERWRITE))) {
	PROP_ERROR (ret) ("xdfs_control_write");
	return ret;
    }

    // Decide whether to reject this delta.
    if ((reject = reject_delta (copy_node, to_node, patch_node, cont_node))) {

	DEBUG_XDFS ("insert_new_version reject: copy ") (copy_node)
	           ("to ") (to_node)
	           ("patch ") (patch_node)
	           ("control ") (cont_node);

	// Must create a pseudo delta here.  This is like a delta
	// without a patch file, and it accompanies a literal file.  It
	// is used for delta extraction only.  If there is a patch file,
	// its copy instructions are replaced with copy instructions for
	// the literal file, which will remain.
	if (xdp_has_data (gen)) {

	    for (uint i = 0; i < control->inst_len; i += 1) {
		XdeltaInstruction *xi = control->inst + i;

		if (xi->index == 0) {
		    xi->offset = xi->output_start;
		}
	    }

	    // The patch link now points back at TO.
	    patch_link_node = to_node;
	}

	// Re-write its control.  Note: Could avoid this by estimating
	// the control length, and it can be avoided in any case for
	// computing Reverse deltas, since reject_delta ignores it.
	if ((ret = xdfs_control_write (cont_node, control, DBFS_OVERWRITE))) {
	    PROP_ERROR (ret) ("xdfs_control_write");
	    return ret;
	}

	// Started a new cluster.
	_stateX->cluster_versions          = 1;
	_stateX->cluster_uncompressed_size = FIXLEN (copy_node.length ());
	_stateX->cluster_compressed_size   = FIXLEN (copy_node.length ());

	// patch_node is unreferenced
	if ((ret = patch_node.del_major ())) {
	    PROP_ERROR (ret) ("patch_del_major");
	    return ret;
	}
    } else {

	DEBUG_XDFS ("insert_new_version delta: copy ") (copy_node)
	           ("to ") (to_node)
	           ("patch ") (patch_node)
	           ("control ") (cont_node);

	// Update counters.
	_stateX->literals                  -= 1;
	_stateX->literal_size              -= FIXLEN (to_node   .length ());
	_stateX->patch_size                += FIXLEN (patch_node.length ());
	_stateX->cluster_versions          += 1;
	_stateX->cluster_uncompressed_size += FIXLEN (copy_node .length ());
	_stateX->cluster_compressed_size   += FIXLEN (copy_node .length ());
	_stateX->cluster_compressed_size   -= FIXLEN (to_node   .length ());
	_stateX->cluster_compressed_size   += FIXLEN (patch_node.length ());
	_stateX->cluster_compressed_size   += FIXLEN (cont_node .length ());

	if (xdp_has_data (gen)) {
	    _stateX->patches += 1;

	    patch_link_node = patch_node;
	}

	// Create view in to_ino, replacing old contents.
	if ((ret = to_node.mk_view (to_node.length (), xdfs_reconstruct_view, DBFS_OVERWRITE))) {
	    PROP_ERROR (ret) ("to_mk_view");
	    return ret;
	}
    }

    // Update counters.
    _stateX->control_size += FIXLEN (cont_node.length ());

    // Create links to its sources (one may be the patch_link)
    for (uint i = 0; i < control->src_count; i += 1) {
	MAJORC *src_nodep = (MAJORC*) xdp_source_data (gen, i);
	NODEC   src_link;

	if (xdp_has_data (gen) && i == 0) {
	    // In this case it is the delta's patch file, or
	    // the TO file if the delta is rejected.
	    g_assert (src_nodep == NULL);

	    src_nodep = & patch_link_node;

	    src_nodep->assert_valid ();

	} else {
	    // If its not the patch, not a srcbuf seg.
	    g_assert (*src_nodep == from_node);
	}

	if ((ret = to_node.node_mkey (MKEY (i, __COMP_SRC_STCK), src_link))) {
	    PROP_ERROR (ret) ("to_link_mkey");
	    return ret;
	}

	if ((ret = src_link.mk_reflink (*src_nodep, DBFS_NOOVERWRITE))) {
	    PROP_ERROR (ret) ("src_mk_reflink");
	    return ret;
	}
    }

    xdp_generator_free (gen);

    xdp_control_free (control);

    if ((policy & XP_IsReverse) || ((policy & XP_IsEvolving) && reject)) {
	// If Reverse is true or Forward needs a new literal version,
	// replace the current version (CUR) with the new file (COPY).

	if ((ret = curlnk_node.mk_reflink (copy_node, DBFS_OVERWRITE))) {
	    PROP_ERROR (ret) ("curlnk_mk_reflink");
	    return ret;
	}
    }

  done:

    // Leave a record of the archive for this version so that the
    // global index can locate the archive of each version in
    // drain_copy.
    if (_stateX->flags & XF_IdxShared) {
	NODEC arlnk;

	if ((ret = copy_node.node_mkey (SHAREDLNK_MKEY, arlnk))) {
	    PROP_ERROR (ret) ("node_mkey shared-idx link");
	    return ret;
	}

	if ((ret = arlnk.mk_reflink (_loc_node, DBFS_NOOVERWRITE))) {
	    PROP_ERROR (ret) ("mk_reflink shared-idx link");
	    return ret;
	}
    }

    node_out = copy_node;

    if ((ret = xdfs_state_write (_state_node, _stateX, DBFS_OVERWRITE, _md))) {
	PROP_ERROR (ret) ("xdfs_state_write");
	return ret;
    }

    return 0;
}

int
XdfsLocation::source_segment (XdeltaGenerator  *gen,
			      XdeltaSourceType  type,
			      const MAJORC     &src)
{
    XdeltaSource *xs;
    FileHandle *fh = NULL;
    int ret;

    if ((ret = src.read_segment (& fh, DBFS_NOFLAG))) {
	PROP_ERROR (ret) ("src_read_segment");
	goto exit;
    }

    if (! (xs = xdp_source_new (fh, type, (void*) & src))) {
	PROP_ERROR ("xdp_source_new");
	goto exit;
    }

    if (! xdp_source_add (gen, xs)) {
	PROP_ERROR ("xdp_source_add");
	goto exit;
    }

    return 0;

  exit:

    if (fh) { handle_close (fh); }
    if (ret == 0) { ret = -1; }

    return ret;
}

bool
XdfsLocation::reject_delta (const MAJORC &copy_node,
			    const MAJORC &to_node,
			    const MAJORC &patch_node,
			    const NODEC  &cont_node)
{
    /* Never reject a delta if it is not evolving.
     */
    if (! (_stateX->policy & XP_IsEvolving)) {
	return false;
    }

    if (_stateX->policy & XP_IsForward) {

	// Reject if the compression ratio will decline (finding a local minima)

	size_t new_uncomp = _stateX->cluster_uncompressed_size + FIXLEN (copy_node.length ());
	size_t new_comp   = _stateX->cluster_compressed_size;
	double current_ratio;
	double new_ratio;

	new_comp += FIXLEN (copy_node.length ());
	new_comp -= FIXLEN (to_node.length ());
	new_comp += FIXLEN (patch_node.length ());
	new_comp += FIXLEN (cont_node.length ());

	current_ratio = (double) _stateX->cluster_compressed_size / (double) _stateX->cluster_uncompressed_size;
	new_ratio     = (double) new_comp / (double) new_uncomp;

	if (new_ratio > current_ratio) {
	    DEBUG_XDFS ("reject delta (minima): versions: %d ratio: %0.3f\n",
			_stateX->cluster_versions, current_ratio);
	    return true;
	}
    } else {
	// Reject if the chain is too long.

	if (_stateX->cluster_versions >= _stateX->cluster_max_versions) {
	    DEBUG_XDFS ("reject delta (chain): versions: %d\n", _stateX->cluster_versions);
	    return true;
	}
    }

    return false;
}

/**********************************************************************/
/*			    Retrieval Code                            */
/**********************************************************************/

XdfsViewImpl::XdfsViewImpl ()
{
}

XdfsViewImpl::~XdfsViewImpl ()
{
}

int
XdfsViewImpl::begin (const NODEC &node)
{
    int ret;

    g_assert (node.mkey () == DBFS_DEF_MKEY);

    if ((ret = _index.source (node.majorc ()))) {
	PROP_ERROR (ret) ("index_source");
	return ret;
    }

    return 0;
}

int
XdfsViewImpl::pgin (guint8     *pgbuf,
		    const XSIZ &offsetx,
		    size_t      len)
{
    int   ret;
    size_t written = 0;
    size_t offset = FIXLEN (offsetx);

    const XdfsInstruction *inst = _index.search (offset);

    while (len > 0) {
	size_t inst_offset;
	size_t this_copy;
	FileHandle *fh;

	inst_offset = offset - inst->output_start;
	this_copy   = MIN (len, inst->length - inst_offset);

	inst_offset += inst->offset;

	if ((ret = inst->node.read_segment (& fh, DBFS_STAYOPEN))) {
	    PROP_ERROR (ret) ("inst_read_segment");
	    return ret;
	}

	if ((size_t) handle_seek (fh, inst_offset, HANDLE_SEEK_SET) != inst_offset) {
	    PROP_ERROR ("handle_seek");
	    return -1;
	}

	if ((size_t) handle_read (fh, pgbuf + written, this_copy) != this_copy) {
	    PROP_ERROR ("handle_read");
	    return -1;
	}

	handle_close (fh);
	handle_free  (fh);

	written += this_copy;
	offset  += this_copy;
	len     -= this_copy;
	inst    += 1;
    }

    return 0;
}

#ifdef DEBUG_XDFS_ASSERTIONS
void
XdfsLocalIndex::assert_length ()
{
    size_t len = 0;

    for (size_t i = 0; i < _inst_len; i += 1) {
	len += _inst[i].length;
    }

    g_assert (len == _seg_len);
}
#endif

const XdfsInstruction*
XdfsLocalIndex::search (size_t offset) const
{
    // This is binary search
    // @@ Do a test of this vs. Slp_search overhead?
    size_t l = 0;
    size_t u = _inst_len - 1;
    size_t i;
    const XdfsInstruction *inst;

    g_assert (offset < _seg_len);

  again:

    g_assert (u >= l);

    i = (u+l)/2;

    inst = & _inst[i];

    if (offset < inst->output_start) {
	u = i - 1;
	goto again;
    } else if (offset >= (inst->output_start + inst->length)) {
	l = i + 1;
	goto again;
    } else {
	return inst;
    }
}

/* This function translates the Xdelta instruction TRANSLATE into an
 * XdfsInstruction coresponding to the current, local, literal file
 * set.  TRANSLATE offsets refer to the TRANSLATE_INDEX.  The output
 * is in APPEND_INST_ARRAY.
 */
void
XdfsLocalIndex::translate_copies (const XdeltaInstruction *translate,
				  GArray                  *append_inst_array)
{
    size_t output_start = translate->output_start;
    size_t to_copy      = translate->length;
    size_t cur_offset   = translate->offset;
    const XdfsInstruction *inst;

    g_assert (to_copy > 0);

    inst = search (cur_offset);

    while (to_copy > 0) {
	XdfsInstruction copy;
	size_t inst_offset;
	size_t this_copy;

	inst_offset = cur_offset - inst->output_start;
	this_copy   = MIN (to_copy, inst->length - inst_offset);

	copy.offset = inst->offset + inst_offset;
	copy.length = this_copy;
	copy.node   = inst->node;
	copy.output_start = output_start;

	g_array_append_val (append_inst_array, copy);

	output_start += this_copy;
	cur_offset   += this_copy;
	to_copy      -= this_copy;
	inst         += 1;
    }
}

void
XdfsLocalIndex::delta (const MAJORC   &node,
		       XdfsLocalIndex *src_lis,
		       XdeltaControl  *control)
{
    GArray *narray = g_array_new (FALSE, FALSE, sizeof (XdfsInstruction));

    for (size_t i = 0; i < control->inst_len; i += 1) {

	const XdeltaInstruction *inst   = & control->inst[i];
	XdfsLocalIndex          &src_li = src_lis[inst->index];

	if (src_li._literal) {
	    XdfsInstruction xi;

	    xi.node         = src_li._node;
	    xi.offset       = inst->offset;
	    xi.length       = inst->length;
	    xi.output_start = inst->output_start;

	    g_array_append_val (narray, xi);
	} else {
	    src_li.translate_copies (inst, narray);
	}
    }

    update (node, narray);
}

void
XdfsLocalIndex::update (const MAJORC &node, GArray *array)
{
    clean ();

    _node       = node;
    _literal    = false;
    _seg_len    = FIXLEN (node.length ());
    _array      = array;
    _inst       = (XdfsInstruction*) array->data;
    _inst_len   = array->len;

    assert_length ();
}

void
XdfsLocalIndex::whole (const MAJORC &node)
{
    clean ();

    _if_lit.node         = node;
    _if_lit.offset       = 0;
    _if_lit.length       = FIXLEN (node.length ());
    _if_lit.output_start = 0;

    _node     = node;
    _literal  = true;
    _seg_len  = FIXLEN (node.length ());
    _inst     = & _if_lit;
    _inst_len = 1;

    assert_length ();
}

void
XdfsLocalIndex::clone (XdfsLocalIndex &clone)
{
    clone.clean ();

    clone._node     = _node;
    clone._literal  = _literal;
    clone._seg_len  = _seg_len;
    clone._inst     = _inst;
    clone._inst_len = _inst_len;
}

int
XdfsLocalIndex::source (const MAJORC &node)
{
    int ret = 0;
    XdfsLocalIndex *src_lis = NULL;
    XdeltaControl  *control = NULL;

    switch (node.type ()) {
    case FT_ShortSeg:
    case FT_LongSeg:

	whole (node);
	break;

    case FT_ViewSeg: {

	NODEC cont_node;

	if ((ret = node.node_mkey (CONT_MKEY, cont_node))) {
	    PROP_ERROR (ret) ("node_cont_mkey");
	    goto exit;
	}

	if ((ret = xdfs_control_read (cont_node, & control))) {
	    PROP_ERROR (ret) ("xdfs_control_read");
	    goto exit;
	}

	src_lis = new XdfsLocalIndex [control->src_count];

	for (uint i = 0; i < control->src_count; i += 1) {

	    MAJORC src_nodei;
	    NODEC  src_linki;

	    if ((ret = node.node_mkey (MKEY (i, __COMP_SRC_STCK), src_linki))) {
		PROP_ERROR (ret) ("src_link_mkey");
		goto exit;
	    }

	    if ((ret = src_linki.read_reflink (src_nodei, 0))) {
		PROP_ERROR (ret) ("src_read_reflink");
		goto exit;
	    }

	    if (src_nodei.type () & FT_Readable) {

		if ((ret = src_lis[i].source (src_nodei))) {
		    PROP_ERROR (ret) ("node_source");
		    goto exit;
		}

	    } else {

		g_assert_not_reached ();
	    }
	}

	delta (node, src_lis, control);
    }
    break;

    default:
	g_assert_not_reached ();
    }

  exit:

    if (control) {
	xdp_control_free (control);
    }

    if (src_lis) {
	delete [] src_lis;
    }

    return ret;
}

/**********************************************************************/
/*			Delta Extraction Code                         */
/**********************************************************************/

int
XdfsExtraction::output_delta (XdfsLocalIndex &to_index,
			      FileHandle     *delta_out)
{
    size_t patch_length = 0;
#ifdef DEBUG_XDFS_ASSERTIONS
    size_t copy_length = 0;
#endif
    XdeltaControl      cont;
    GArray            *out_inst = g_array_new (FALSE, FALSE, sizeof (XdeltaInstruction));
    guint32            types[2];
    int patch_index   = -1;
    int primary_index = -1;
    int ret;

    cont.src_count = 0;
    cont.src_types = types;
    cont.length    = to_index._seg_len;

    for (size_t i = 0; i < to_index._inst_len; i += 1) {
	const XdfsInstruction *xi = to_index._inst + i;
	XdeltaInstruction one;

	if (xi->node == _from.home) {
	    if (primary_index < 0) {
		primary_index = cont.src_count;
		types[cont.src_count ++] = XS_Primary;
	    }

	    one.index = primary_index;
	    one.offset = xi->offset;

#ifdef DEBUG_XDFS_ASSERTIONS
	    copy_length += xi->length;
#endif
	} else {
	    if (patch_index < 0) {
		patch_index = cont.src_count;
		types[cont.src_count ++] = XS_Patch;
	    }

	    one.index = patch_index;
	    one.offset = patch_length;

	    patch_length += xi->length;
	}

	one.length = xi->length;

	g_array_append_val (out_inst, one);
    }

#ifdef DEBUG_XDFS_ASSERTIONS
    DEBUG_XDFS ("copy: %d; patch: %d (%0.2f%%)", copy_length, patch_length, 100.0 * (double) patch_length / ((double) copy_length + patch_length));
#endif

    cont.inst_len = out_inst->len;
    cont.inst = (XdeltaInstruction*) out_inst->data;

    if (! xdp_control_write (& cont, delta_out)) {
	PROP_ERROR ("xdp_control_write");
	return -1;
    }

    for (size_t i = 0; i < to_index._inst_len; i += 1) {
	const XdfsInstruction *xi = to_index._inst + i;

	if (xi->node != _from.home) {
	    FileHandle *fh;

	    if ((ret = xi->node.read_segment (& fh, DBFS_STAYOPEN))) {
		PROP_ERROR (ret) ("src_read_segment");
		return ret;
	    }

	    if (! handle_copy (fh, delta_out, xi->offset, xi->length)) {
		PROP_ERROR ("handle_copy");
		return -1;
	    }

	    handle_close (fh);
	    handle_free  (fh);
	}
    }

    g_array_free (out_inst, TRUE);

    return 0;
}

void
XdfsExtraction::extract_to_chain (XdfsLocalIndex &middle)
{
    GSList *chain = _to.chain;

    for (; chain; chain = chain->next) {
	XdfsExtrElt    *elt     = (XdfsExtrElt*) chain->data;
	XdfsLocalIndex *src_lis = new XdfsLocalIndex [elt->_control->src_count];

	for (uint i = 0; i < elt->_control->src_count; i += 1) {
	    if ((int) i == elt->_primary_index) {
		middle.clone (src_lis[i]);
	    } else {
		src_lis[i].whole (elt->_src_nodes[i]);
	    }
	}

	middle.delta (elt->_node, src_lis, elt->_control);

	delete [] src_lis;
    }
}

void
XdfsExtrHalf::free_elt (gpointer data,
			gpointer user_data)
{
    XdfsExtrElt *elt = (XdfsExtrElt*) data;

    delete elt;
}

void
XdfsExtraction::extract_overlay (XdfsLocalIndex &to_middle, XdfsInvert *overlay)
{
    GArray *inst_array = g_array_new (FALSE, FALSE, sizeof (XdfsInstruction));
    size_t pos        = 0;
    size_t inst_i     = 0;
    size_t seg_len    = to_middle._seg_len;
    size_t over_start;

    XdfsInvert::iterator iter = overlay->begin ();

    iter.set_off (seg_len + 1, over_start);

    while (pos < seg_len) {

	g_assert (pos <= over_start);

	if (pos == over_start) {
	    // Inverted copy
	    XdfsInstruction over;

	    over.length       = iter.length ();
	    over.offset       = iter.start ();
	    over.node         = _from.home;
	    over.output_start = over_start;

	    g_array_append_val (inst_array, over);

	    iter.next ();

	    iter.set_off (seg_len + 1, over_start);

	    pos += over.length;
	} else {
	    // Insert instructions, stopping at over_start.

	    size_t copy_length, copy_start, copy_offset;
	    const XdfsInstruction *orig;
	    XdfsInstruction xi;

	    while (to_middle._inst[inst_i].output_start + to_middle._inst[inst_i].length < pos) {
		inst_i += 1;
	    }

	    orig = to_middle._inst + inst_i;

	    copy_start  = orig->output_start;
	    copy_offset = orig->offset;
	    copy_length = orig->length;

	    if (copy_start < pos) {
		int diff = pos - copy_start;

		copy_start  += diff;
		copy_offset += diff;
		copy_length -= diff;
	    }

	    if (pos + copy_length > over_start) {

		g_assert (over_start > pos);

		copy_length = over_start - pos;

	    } else {
		inst_i += 1;
	    }

	    xi.length = copy_length;
	    xi.node   = orig->node;
	    xi.offset = copy_offset;
	    xi.output_start = copy_start;

	    g_array_append_val (inst_array, xi);

	    pos += xi.length;
	}
    }

    to_middle.update (to_middle._node, inst_array);
}

#ifdef DEBUG_XDFS_ASSERTIONS
void
xdfs_print_over (XdfsInvert *over)
{
    XdfsInvert::iterator iter = over->begin ();

    for (; ! iter.end (); iter.next ()) {
	printf ("[offset %d length %d copy start %d]\n",
		iter.offset (),
		iter.length (),
		iter.start ());
    }
}
#endif

XdfsInvert*
XdfsExtraction::extract_from_chain ()
{
    GSList     *chain, *chain_rev;
    XdfsInvert *over = NULL;

    // The chain's head, where it meets the to chain, is the last elt
    // we wish to traverse.  This computes the ranges in the HEAD version
    // copyable from the FROM vesion.
    chain_rev = g_slist_reverse (_from.chain);

    for (chain = chain_rev; chain; chain = chain->next) {
	XdfsExtrElt                  *elt = (XdfsExtrElt*) chain->data;
	const XdeltaInstruction     *inst = elt->_control->inst;
	size_t                   inst_len = elt->_control->inst_len;
	uint                primary_index = elt->_primary_index;
	int                primary_length = FIXLEN (elt->_primary.length ());
	int                   this_length = FIXLEN (elt->_node.length ());
	XdfsInvert              *new_over = new XdfsInvert; // xdfs_skip_list_new (primary_length);

	// First time through: set base case
	if (over == NULL) {
	    over = new XdfsInvert; // xdfs_skip_list_new (elt->_node.length ());

	    over->insert (0, FIXLEN (elt->_node.length ()), 0);
	}

	for (uint i = 0; i < inst_len; i += 1) {

	    const XdeltaInstruction &xi = inst[i];
 	    int copy_length, copy_offset, copy_start;
	    int from_offset, from_length, from_start, this_copy;

	    if (xi.index != primary_index) {
		continue;
	    }

	    // Copy_start is an index in THIS elt, copy_offset is in
	    // PRIMARY elt (next chain elt)
	    copy_start  = xi.output_start;
	    copy_length = xi.length;
	    copy_offset = xi.offset;

	    g_assert (copy_start  + copy_length <= this_length);
	    g_assert (copy_offset + copy_length <= primary_length);

	    while (copy_length > 0) {
		// This is the translate_copies logic for skip lists,
		// i.e. non-continuous instruction ranges

		XdfsInvert::iterator iter = over->invsearch (copy_start);

		g_assert (! iter.end ());

		from_start  = iter.offset ();
		from_length = iter.length ();
		from_offset = iter.start  ();

		if (from_start > copy_start) {
		    int diff = from_start - copy_start;

		    copy_start  += diff;
		    copy_offset += diff;
		    copy_length -= diff; // Note: needs to be signed
		} else if (from_start < copy_start) {
		    int diff = copy_start - from_start;

		    from_start  += diff;
		    from_offset += diff;
		    from_length -= diff; // Note: needs to be signed
		}

		this_copy = MIN (copy_length, from_length);

		if (this_copy <= 0) {
		    break;
		}

		g_assert (copy_offset + this_copy <= primary_length);

		new_over->insert (copy_offset, this_copy, from_offset);

		copy_offset += this_copy;
		copy_start  += this_copy;
		copy_length -= this_copy;
	    }
	}

	delete over;

	over = new_over;
    }

    return over;
}

int
XdfsExtraction::advance (XdfsExtrHalf &half,
			 bool          is_to_chain)
{
    XdeltaControl *control;
    XdfsExtrElt   *elt;
    NODEC          cont_node;
    int            ret;

    DEBUG_XDFS ("advance_extract head: ") (half.head);

    if ((ret = half.head.node_mkey (CONT_MKEY, cont_node))) {
	PROP_ERROR (ret) ("half_cont_mkey");
	return ret;
    }

    if ((ret = xdfs_control_read (cont_node, & control))) {
	PROP_ERROR (ret) ("xdfs_control_read");
	return ret;
    }

    elt = new XdfsExtrElt (half.head, control);

    for (uint i = 0; i < control->src_count; i += 1) {

	bool is_primary = control->src_types[i] == XS_Primary;

	if (is_to_chain || is_primary) {

	    MAJORC src_nodei;
	    NODEC  src_linki;

	    if ((ret = elt->_node.node_mkey (MKEY (i, __COMP_SRC_STCK), src_linki))) {
		PROP_ERROR (ret) ("src_link_mkey");
		return ret;
	    }

	    if ((ret = src_linki.read_reflink (src_nodei, 0))) {
		PROP_ERROR (ret) ("src_link_read_reflink");
		return ret;
	    }

	    elt->_src_nodes[i] = src_nodei;

	    if (! (src_nodei.type () & FT_Readable)) {
		g_assert_not_reached ();
	    }

	    if (is_primary) {
		elt->_primary_index = i;
		elt->_primary       = src_nodei;
	    }
	}
    }

    if (elt->_primary_index < 0) {
	// If there is no primary, the chain is broken.
	delete elt;
	_broken = true;
	return 0;
    }

    half.chain      = g_slist_prepend (half.chain, elt);
    half.chain_len += 1;
    half.head       = elt->_primary;

    return 0;
}

// A function that returns:
//   < 0 if a is less distant than b
//   = 0 if b is equal to a
//   > 0 if a is more distant than b
// where distance is measured from the root, which is the most current
// version.  It is a partial order based on insertion time (inode
// number).
int
XdfsExtraction::furthest ()
{
    // @@@ This is a hack, right?  Need a better DAG traversal
    // algorithm?: See journal pages 31 and 126 (Apr 2001).
    if (_loc._stateX->policy & XP_IsForward) {
	return _from.compare_to (_to); // _from.number ().key () - _to.number ().key ();
    } else {
	return _to.compare_to (_from); // _to.number ().key () - _from.number ().key ();
    }
}

int
XdfsLocation::extract_delta (const MAJORC &from_node,
			     const MAJORC &to_node,
			     FileHandle   *delta_out)
{
    XdfsExtraction  ex (*this, from_node, to_node);
    XdfsLocalIndex  middle;
    XdfsInvert     *from_copies = NULL;
    int f, ret;

    DEBUG_XDFS ("extract_delta from: ") (from_node) (" to ") (to_node);

    g_assert (_stateX->policy != XP_NoCompress);

    // Perform a DAG traversal here, where the edges are delta copies
    // and this loop advances along the oldest edge until the two points
    // meet.
    while (! ex._broken && (f = ex.furthest ()) != 0) {

	if (f > 0) {
	    if ((ret = ex.advance (ex._from, false))) {
		PROP_ERROR (ret) ("extract_advance_from");
		return ret;
	    }
	} else {
	    if ((ret = ex.advance (ex._to, true))) {
		PROP_ERROR (ret) ("extract_advance_to");
		return ret;
	    }
	}
    }

    // @@ There is a special case like RCS in which a delta is copied
    // directly: goes here.

    // If the chain NOT broken and there is no FROM chain, then
    // there will be a total overlay.  That is: (MIDDLE == FROM)
    bool total_overlay = (! ex._broken && ! ex._from.chain);

    if (! total_overlay) {

	// Construct a local index for the middle of the chain (HEAD
	// of the TO chain, same as head of the FROM chain, since both
	// are reversed).
	if ((ret = middle.source (ex._to.head))) {
	    PROP_ERROR (ret) ("middle_source");
	    return ret;
	}
    }

    // If broken is TRUE, then the FROM and TO chains are not joined.
    // This code degenerates into COPY extraction.  This only happens
    // when there are no primary copies at some point in the chain.
    if (! ex._broken) {

	if (ex._from.chain) {
	    // Compute FROM copies in MIDDLE by inverting the FROM chain.
	    from_copies = ex.extract_from_chain ();

	    // FROM_COPIES now represents an overlay for MIDDLE
	    ex.extract_overlay (middle, from_copies);

	} else {
	    // Total overlay.  FROM == MIDLE
	    g_assert (total_overlay);

	    middle.whole (from_node);
	}
    }

    // Finish index construction using the TO chain beginning with
    // MIDDLE, which is updated along the way.  Following this, its
    // rightful name would be TO.
    ex.extract_to_chain (middle);

    // Output delta
    if ((ret = ex.output_delta (middle, delta_out))) {
	PROP_ERROR (ret) ("output_delta");
	return ret;
    }

    if (from_copies) {
	delete from_copies;
    }

    return 0;
}

int
xdfs_apply_delta (FileHandle *from_in,
		  FileHandle *delta_in,
		  FileHandle *cons_out)
{
    XdeltaControl *control;
    const XdeltaInstruction *inst;
    int primary_index = -1;

    if (! (control = xdp_control_read (delta_in))) {
	PROP_ERROR ("xdp_control_read");
	return -1;
    }

    inst = control->inst;

    for (size_t i = 0; i < control->src_count; i += 1) {
	if (control->src_types[i] == XS_Primary) {
	    primary_index = i;
	    break;
	}
    }

    for (size_t i = 0; i < control->inst_len; i += 1) {
	const XdeltaInstruction *xi = inst + i;

	if ((int) xi->index == primary_index) {
	    if (! handle_copy (from_in, cons_out, xi->offset, xi->length)) {
		PROP_ERROR ("handle_copy");
		return -1;
	    }
	} else {
	    if (! handle_copy_len (delta_in, cons_out, xi->length)) {
		PROP_ERROR ("handle_copy_len");
		return -1;
	    }
	}
    }

    xdp_control_free (control);

    return 0;
}

int
XdfsLocation::insert_delta (const MAJORC &against_node,
			    FileHandle   *delta_in,
			    MAJORC       &node_out)
{
    XdeltaControl  *control;
    MAJORC          patch_node;
    MAJORC          tmp_node;
    NODEC           tmp_cont_node;
    int             patch_index   = -1;
    int             primary_index = -1;
    FileHandle     *rh = NULL;
    int ret;

    if (! (control = xdp_control_read (delta_in))) {
	PROP_ERROR ("xdp_control_read");
	return -1;
    }

    for (size_t i = 0; i < control->src_count; i += 1) {

	if (control->src_types[i] == XS_Patch) {
	    patch_index = i;
	} else if (control->src_types[i] == XS_Primary) {
	    primary_index = i;
	}
    }

    // @@ This could be optimized to avoid creating TMP_NODE--measure
    // in xproxy client.
    if ((ret = _loc_node.sarea ().allocate (tmp_node, DBFS_ALLOC_ASCEND))) {
	PROP_ERROR (ret) ("loc_sarea_allocate");
	return ret;
    }

    DEBUG_XDFS ("insert_delta against ") (against_node) ("tmp ") (tmp_node);

    if (patch_index >= 0) {

	FileHandle *p_out;

	if ((ret = _loc_node.sarea ().allocate (patch_node, DBFS_ALLOC_ASCEND))) {
	    PROP_ERROR (ret) ("loc_sarea_allocate");
	    return ret;
	}

	DEBUG_XDFS ("insert_delta patch ") (patch_node);

	if ((ret = patch_node.repl_segment (& p_out, DBFS_NOOVERWRITE))) {
	    PROP_ERROR (ret) ("patch_repl_segment");
	    return ret;
	}

	if (! handle_drain (delta_in, p_out)) {
	    PROP_ERROR ("handle_drain");
	    return -1;
	}

	if (! handle_close (p_out)) {
	    PROP_ERROR ("handle_close");
	    return -1;
	}

	handle_free (p_out);
    }

    if ((ret = tmp_node.node_mkey (CONT_MKEY, tmp_cont_node))) {
	PROP_ERROR (ret) ("tmp_cont_mkey");
	return ret;
    }

    if ((ret = xdfs_control_write (tmp_cont_node, control, DBFS_NOOVERWRITE))) {
	PROP_ERROR (ret) ("xdfs_control_write");
	return ret;
    }

    if ((ret = tmp_node.mk_view (XSIZ (control->length), xdfs_reconstruct_view, DBFS_NOOVERWRITE))) {
	PROP_ERROR (ret) ("tmp_mk_view");
	return ret;
    }

    for (size_t i = 0; i < control->src_count; i += 1) {
	MAJORC src_node;
	NODEC  src_link;

	if ((int) i == patch_index) {
	    src_node = patch_node;
	} else if ((int) i == primary_index) {
	    src_node = against_node;
	}

	if ((ret = tmp_node.node_mkey (MKEY (i, __COMP_SRC_STCK), src_link))) {
	    PROP_ERROR (ret) ("tmp_link_mkey");
	    return ret;
	}

	if ((ret = src_link.mk_reflink (src_node, DBFS_NOOVERWRITE))) {
	    PROP_ERROR (ret) ("tmp_link_mk_reflink");
	    return ret;
	}
    }

    xdp_control_free (control);

    if ((ret = tmp_node.read_segment (& rh, DBFS_NOFLAG))) {
	PROP_ERROR (ret) ("tmp_read_segment");
	return ret;
    }

    if ((ret = insert_version (rh, node_out))) {
	if (ret != DBFS_EXISTS) {
	    PROP_ERROR (ret) ("insert_delta_version");
	}
	goto done;
    }

    // Delete tmp now
    ret = tmp_node.del_major ();

  done:

    handle_free (rh);

    return ret;
}
