/*
 * file mi_host.c - Menu item for host selection in networked games
 *
 * $Id: mi_host.c,v 1.3 2004/05/14 10:00:35 alfie Exp $
 *
 * Program XBLAST 
 * (C) by Oliver Vogel (e-mail: m.vogel@ndh.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2; or (at your option)
 * any later version
 *
 * This program is distributed in the hope that it will be entertaining,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILTY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 * Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include "mi_host.h"

#include "mi_map.h"

#include "gui.h"

/*
 * local macors
 */
#define FF_HOST_NAME_FOCUS     (FF_Small | FF_Black | FF_Center | FF_Outlined)
#define FF_HOST_NAME_NO_FOCUS  (FF_Small | FF_White | FF_Center)
#define FF_HOST_STATE_FOCUS    (FF_Large | FF_Black | FF_Center | FF_Outlined)
#define FF_HOST_STATE_NO_FOCUS (FF_Large | FF_White | FF_Center)

/*
 * local types
 */
typedef struct {
  XBMenuItem    item;
  const char  **pText;
  const char   *cText;
  XBHostState  *pState;
  XBHostState   cState;
  const int    *pPing;
  int           cPing;
  char          tPing[16];
  Sprite       *tSprite;
  Sprite       *sSprite;
  Sprite       *pSprite;
} XBMenuHostItem;

typedef struct {
  XBMenuItem    item;
  XBTeamState  *pTeam;
  XBTeamState   cTeam;
  Sprite       *sSprite;
} XBMenuTeamItem;


/*
 * local variables
 */
static XBHostState serverState = XBHS_Server;
static const char *stateText[NUM_XBHS] = {
  "?",    /* XBHS_None */
  "...",  /* XBHS_Wait */
  "In",   /* XBHS_In */
  "Out",  /* XBHS_Out */
  "Srv",  /* XBHS_Server */
};

static const char *teamText[NUM_XBTS] = {
  "",    /* XBHS_None */
  "Red", // "Team: Red",  
  "Green", // "Team: Green",
  "Blue", //"Team: Blue", 
};

/*
 * a host item receives the focus
 */
static void
MenuHostFocus (XBMenuItem *ptr, XBBool flag)
{
  XBMenuHostItem *host = (XBMenuHostItem *) ptr;

  assert (NULL != host);
  assert (NULL != host->tSprite);
  assert (NULL != host->sSprite);
  SetSpriteAnime (host->tSprite, flag ? FF_HOST_NAME_FOCUS  : FF_HOST_NAME_NO_FOCUS);
  SetSpriteAnime (host->sSprite, flag ? FF_HOST_STATE_FOCUS : FF_HOST_STATE_NO_FOCUS);
  if (NULL != host->pSprite) {
    SetSpriteAnime (host->pSprite, flag ? FF_HOST_NAME_FOCUS : FF_HOST_NAME_NO_FOCUS);
  }
} /* MenuToggleFocus */

static void
MenuTeamFocus (XBMenuItem *ptr, XBBool flag)
{
  XBMenuTeamItem *team = (XBMenuTeamItem *) ptr;

  assert (NULL != team);
  assert (NULL != team->sSprite);
  SetSpriteAnime (team->sSprite, flag ? FF_HOST_NAME_FOCUS : FF_HOST_NAME_NO_FOCUS);

} /* MenuToggleFocus */

/*
 * a server is selected
 */
static void
MenuServerSelect (XBMenuItem *ptr)
{
} /* MenuToggleSelect */

/*
 * a client is selected
 */
static void
MenuClientSelect (XBMenuItem *ptr)
{
  XBMenuHostItem *host = (XBMenuHostItem *) ptr;

  assert (NULL != host);
  assert (NULL != host->pState);
  switch (*host->pState) {
  case XBHS_In:  *host->pState = XBHS_Out; break;
  case XBHS_Out: *host->pState = XBHS_In;  break;
  default: break;
  }
} /* MenuToggleSelect */


/*
 * a team is selected
 */
static void
MenuTeamSelect (XBMenuItem *ptr)
{
  XBMenuTeamItem *team = (XBMenuTeamItem *) ptr;

  assert (NULL != team);
  assert (NULL != team->pTeam);
  if(*team->pTeam < NUM_XBTS-1) { *team->pTeam=*team->pTeam+1;} else { *team->pTeam=1;}
  
} /* MenuToggleSelect */
static void
MenuTeamSelectPeer (XBMenuItem *ptr)
{

} /* MenuToggleSelect */

/*
 * a peer is selected
 */
static void
MenuPeerSelect (XBMenuItem *ptr)
{
} /* MenuToggleSelect */

/*
 * mouse click
 */ 
static void
MenuServerMouse (XBMenuItem *ptr, XBEventCode code)
{
} /* MenuToggleMouse */

/*
 * mouse click
 */ 
static void
MenuClientMouse (XBMenuItem *ptr, XBEventCode code)
{
  if (code == XBE_MOUSE_1) {
    MenuClientSelect (ptr);
  }
} /* MenuToggleMouse */
/*
 * mouse team click
 */ 
static void
MenuTeamMouse (XBMenuItem *ptr, XBEventCode code)
{
  if (code == XBE_MOUSE_1) {
    MenuTeamSelect (ptr);
  }
} /* MenuToggleMouse */
static void
MenuTeamMousePeer (XBMenuItem *ptr, XBEventCode code)
{

} /* MenuToggleMouse */

/*
 * mouse click
 */ 
static void
MenuPeerMouse (XBMenuItem *ptr, XBEventCode code)
{
} /* MenuToggleMouse */


/*
 * polling a host item
 */
static void
MenuHostPoll (XBMenuItem *ptr)
{
  XBMenuHostItem *host = (XBMenuHostItem *) ptr;

  assert (NULL != host);
  assert (NULL != host->tSprite);
  assert (NULL != host->sSprite);
  /* state */
  if (*host->pState != host->cState) {
    host->cState = *host->pState;
    assert (host->cState < NUM_XBHS);
    SetSpriteText (host->sSprite, stateText[host->cState]);
  }
  /* name */
  if (*host->pText != host->cText) {
    host->cText = *host->pText;
    SetSpriteText (host->tSprite, host->cText);
  }
  /* ping */
  if (NULL != host->pPing &&
      *host->pPing != host->cPing) {
    host->cPing = *host->pPing;
    if (host->cPing < 0) {
      host->tPing[0] = 0;
    } else {
      sprintf (host->tPing, "%u ms", host->cPing);
    }
    SetSpriteText (host->pSprite, host->tPing);
  }
} /* MenuHostPoll */
/*
 * polling a team item
 */
static void
MenuTeamPoll (XBMenuItem *ptr)
{
  XBMenuTeamItem *team = (XBMenuTeamItem *) ptr;

  assert (NULL != team);
  /* state */
  if (*team->pTeam != team->cTeam) {
    team->cTeam = *team->pTeam;
    assert (team->cTeam < NUM_XBTS);
    SetSpriteText (team->sSprite, teamText[team->cTeam]);
  }
} /* MenuTeamPoll */

/*
 * create server teamselection button
 */
XBMenuItem *
MenuCreateTeam (int x, int y, int w, XBTeamState *pTeam, int server)
{
  /* create item */
  XBMenuTeamItem *team = calloc (1, sizeof (*team));
  assert (team != NULL);
  if(server) {
    MenuSetItem (&team->item, MIT_Team, x, y, w, CELL_H, MenuTeamFocus, MenuTeamSelect, MenuTeamMouse, MenuTeamPoll);
  } else {
    MenuSetItem (&team->item, MIT_Team, x, y, w, CELL_H, MenuTeamFocus, MenuTeamSelectPeer, MenuTeamMousePeer, MenuTeamPoll);
  }
  /* set server specific data */
  assert (pTeam  != NULL);
  team->pTeam =  pTeam;
  team->cTeam = *pTeam;
  /* sprite with team state */
  team->sSprite = CreateTextSprite (teamText[*pTeam], (x + 1) * BASE_X, y * BASE_Y, (w - 2) * BASE_X, (CELL_H/2) * BASE_Y,
				    FF_HOST_NAME_FOCUS, SPM_MAPPED);
  /* graphics */
  MenuAddSmallFrame (x / CELL_W, (x + w - 1) / CELL_W, y / CELL_H);
  return &team->item;
} /* MenuCreateTeam */

/*
 * create server button
 */
XBMenuItem *
MenuCreateServer (int x, int y, int w, const char **pText)
{
  /* create item */
  XBMenuHostItem *host = calloc (1, sizeof (*host));
  assert (host != NULL);
  MenuSetItem (&host->item, MIT_Host, x, y, w, CELL_H, MenuHostFocus, MenuServerSelect, MenuServerMouse, MenuHostPoll);
  /* set server specific data */
  assert (pText != NULL);
  host->pText  =  pText;
  host->cText  = *pText;
  host->pState = &serverState;
  host->cState =  XBHS_None;
  /* sprite with host name */
  host->tSprite = CreateTextSprite (*pText, (x + 11) * BASE_X, y * BASE_Y, (w - 12) * BASE_X, CELL_H * BASE_Y,
				    FF_HOST_NAME_NO_FOCUS, SPM_MAPPED);
  /* sprite with host state */
  host->sSprite = CreateTextSprite (stateText[0], (x + 1) * BASE_X, y * BASE_Y, 9 * BASE_X, CELL_H * BASE_Y,
				    FF_HOST_STATE_NO_FOCUS, SPM_MAPPED);
  /* graphics */
  MenuAddLargeFrame (x / CELL_W, (x + w - 1) / CELL_W, y / CELL_H);
  return &host->item;
} /* MenuCreateServer */

/*
 * create server button
 */
XBMenuItem *
MenuCreateClient (int x, int y, int w, const char **pText, XBHostState *pState, const int *pPing)
{
  /* create item */
  XBMenuHostItem *host = calloc (1, sizeof (*host));
  assert (host != NULL);
  MenuSetItem (&host->item, MIT_Host, x, y, w, CELL_H, MenuHostFocus, MenuClientSelect, MenuClientMouse, MenuHostPoll);
  /* set server specific data */
  assert (pText  != NULL);
  assert (pState != NULL);
  host->pText  =  pText;
  host->cText  = *pText;
  host->pState =  pState;
  host->cState = *pState;
  host->pPing  =  pPing;
  host->cPing  = -1;
  /* sprite with host name */
  host->tSprite = CreateTextSprite (*pText, (x + 11) * BASE_X, (y+1) * BASE_Y, (w - 12) * BASE_X, (CELL_H/2) * BASE_Y,
				    FF_HOST_NAME_NO_FOCUS, SPM_MAPPED);
  /* sprite with ping to host */
  host->pSprite = CreateTextSprite (host->tPing, (x + 11) * BASE_X, (y+CELL_H/2) * BASE_Y, (w - 12) * BASE_X, (CELL_H/2-1) * BASE_Y,
				    FF_HOST_NAME_NO_FOCUS, SPM_MAPPED);
  /* sprite with host state */
  host->sSprite = CreateTextSprite (stateText[*pState], (x + 1) * BASE_X, y * BASE_Y, 9 * BASE_X, CELL_H * BASE_Y,
				    FF_HOST_STATE_NO_FOCUS, SPM_MAPPED);
  /* graphics */
  MenuAddLargeFrame (x / CELL_W, (x + w - 1) / CELL_W, y / CELL_H);
  return &host->item;
} /* MenuCreateServer */

/*
 * create server button
 */
XBMenuItem *
MenuCreatePeer (int x, int y, int w, const char **pText, XBHostState *pState, const int *pPing)
{
  /* create item */
  XBMenuHostItem *host = calloc (1, sizeof (*host));
  assert (host != NULL);
  MenuSetItem (&host->item, MIT_Host, x, y, w, CELL_H, MenuHostFocus, MenuPeerSelect, MenuPeerMouse, MenuHostPoll);
  /* set server specific data */
  assert (pText  != NULL);
  assert (pState != NULL);
  host->pText  =  pText;
  host->cText  = *pText;
  host->pState =  pState;
  host->cState = *pState;
  host->pPing  =  pPing;
  host->cPing  = -1;
  /* sprite with host name */
  host->tSprite = CreateTextSprite (*pText, (x + 11) * BASE_X, (y+1) * BASE_Y, (w - 12) * BASE_X, (CELL_H/2) * BASE_Y,
				    FF_HOST_NAME_NO_FOCUS, SPM_MAPPED);
  /* sprite with ping to host */
  host->pSprite = CreateTextSprite (host->tPing, (x + 11) * BASE_X, (y+CELL_H/2) * BASE_Y, (w - 12) * BASE_X, (CELL_H/2-1) * BASE_Y,
				    FF_HOST_NAME_NO_FOCUS, SPM_MAPPED);
  /* sprite with host state */
  host->sSprite = CreateTextSprite (stateText[*pState], (x + 1) * BASE_X, y * BASE_Y, 9 * BASE_X, CELL_H * BASE_Y,
				    FF_HOST_STATE_NO_FOCUS, SPM_MAPPED);
  /* graphics */
  MenuAddLargeFrame (x / CELL_W, (x + w - 1) / CELL_W, y / CELL_H);
  return &host->item;
} /* MenuCreateServer */

/*
 * delete a host 
 */
void
MenuDeleteHost (XBMenuItem *item)
{
  XBMenuHostItem *host = (XBMenuHostItem *) item;

  assert (NULL != host);
  assert (NULL != host->tSprite);
  assert (NULL != host->sSprite);
  DeleteSprite (host->tSprite);
  DeleteSprite (host->sSprite);
  if (NULL != host->pSprite) {
    DeleteSprite (host->pSprite);
  }
} /* DeleteButtonItem */

/*
 * delete a team 
 */
void
MenuDeleteTeam (XBMenuItem *item)
{
  XBMenuTeamItem *team = (XBMenuTeamItem *) item;

  assert (NULL != team);
  assert (NULL != team->sSprite);
  DeleteSprite (team->sSprite);
} /* DeleteTeam */

/*
 * end of file mi_host.c
 */
