/*
 * file com_to_central.c - handle communications with centrals
 *
 * $Id: com_to_central.c,v 1.3 2004/05/14 10:00:33 alfie Exp $
 *
 * Program XBLAST 
 * (C) by Oliver Vogel (e-mail: m.vogel@ndh.net)
 * Added by Koen De Raedt for central support
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2; or (at your option)
 * any later version
 *
 * This program is distributed in the hope that it will be entertaining,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILTY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 * Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include "com_to_central.h"

#include "atom.h"
#include "user.h"
#include "com_stream.h"
#include "net_tele.h"
#include "cfg_level.h"
#include "com_stream.h"
#include "net_socket.h"

/*
 * local types
 */
typedef struct {
  XBCommStream stream;
  unsigned     serial;
} XBCommToCentral;

/*
 * local variables
 */

/*
 *
 */
static XBCommResult 
HandleDataAvailable (XBCommStream *toCentral, const XBTelegram *tele)
{
  const void     *data;
  size_t          len;

  assert (toCentral != NULL);
  /* get telegramm data */
  data = Net_TeleData (tele, &len);
  switch (Net_TeleID (tele)) {
  case XBT_ID_PlayerConfig:
    User_ReceivePlayerConfig (data);
    break;
  case XBT_ID_PID:
    User_ReceivePlayerPID (data);
    break;
  default:
    break;
  }
  return XCR_OK;
} /* HandleDataAvailable */

/*
 *
 */
static XBCommResult 
HandleDataNotAvailable (XBCommStream *toCentral, const XBTelegram *tele)
{
  switch (Net_TeleID (tele)) {
  case XBT_ID_PlayerConfig:
    Dbg_Out("No more players disconnecting\n");
    User_NoMorePlayers();
    User_SendDisconnect();
    break;
  case XBT_ID_PID:
    break;
  default:
    break;
  }
  return XCR_OK;
} /* HandleDataAvailable */

/*
 * handle telegrams from central
 */
static XBCommResult 
HandleTelegram (XBCommStream *stream, const XBTelegram *tele)
{
  assert (stream != NULL);
  switch (Net_TeleCOT (tele)) {
    /* central sends requested data */
  case XBT_COT_DataAvailable:
    return HandleDataAvailable (stream, tele);
    /* central has not requested data */
  case XBT_COT_DataNotAvailable:
    return HandleDataNotAvailable (stream, tele);
  default:
    return XCR_Error; 
  }
} /* HandleTelegram */

/*
 *
 */
static XBCommResult 
DeleteToCentral (XBComm *comm)
{
  /* delete communication */
  Stream_CommFinish ((XBCommStream *) comm);
  /* free memory */
  free (comm);
  /* inform application */
  User_SetDisconnected ();
  return XCR_OK;
} /* DeleteToCentral */

/*
 * 
 */
static void
ErrorToCentral (XBCommStream *comm)
{
  /* inform application */
  User_NotifyError ();
} /* ErrorToCentral */

/*
 * 
 */
XBComm *
X2C_CreateComm (const CFGCentralSetup *cfg)
{
  XBSocket       *pSocket;
  XBCommToCentral *toCentral;

  assert (cfg != NULL);
  /* create connection to server */
  pSocket = Net_ConnectInet (cfg->name, cfg->port);
  if (NULL == pSocket) {
    return NULL;
  }
  /* create communication data structure */
  toCentral = calloc (1, sizeof (*toCentral));
  assert (NULL != toCentral);
  /* set values */
  Stream_CommInit (&toCentral->stream, COMM_ToCentral, pSocket, HandleTelegram, ErrorToCentral, DeleteToCentral);
  /* that'S all */
  return &toCentral->stream.comm;
} /* CommCreateToServer */

/*
 * query player config from central
 */
void
X2C_QueryPlayerConfig (XBComm *comm)
{
  XBCommStream *stream = (XBCommStream *) comm;

  /* sanity check */
  assert (stream != NULL);
  assert (stream->sndQueue != NULL);
  /* send data */
  Socket_RegisterWrite (CommSocket (&stream->comm));
  Net_SendTelegram (stream->sndQueue, 
		    Net_CreateTelegram (XBT_COT_RequestData, XBT_ID_PlayerConfig, 0, NULL, 0) );
} /* X2C_QueryPlayerConfig */

/*
 * send game config to central
 */
void
X2C_SendPlayerConfig (XBComm *comm, XBAtom atom)
{
  XBCommStream *stream = (XBCommStream *) comm;

  /* sanity check */
  assert (stream != NULL);
  assert (stream->sndQueue != NULL);  
  /* send database section */
  Socket_RegisterWrite (CommSocket (&stream->comm));
  SendPlayerConfig (CT_Local, stream->sndQueue, XBT_COT_SendData, 0, atom, XBTrue); // XBCC to central
} /* X2C_SendPlayerConfig */

/*
 * send game config to central
 */
void
X2C_SendGameStat (XBComm *comm, int numPlayers, int *PID, int *Score)
{
  XBCommStream *stream = (XBCommStream *) comm;
  char          tmp[256];
  int           i;

  /* sanity check */
  assert (stream != NULL);
  assert (stream->sndQueue != NULL);  
  /* send shit */
  memcpy(tmp,&numPlayers,4);
  for(i=0;i<numPlayers;i++) {
    memcpy(tmp+4+i*8,PID+i,4);
    memcpy(tmp+8+i*8,Score+i,4);
  }
  Socket_RegisterWrite (CommSocket (&stream->comm));
  Net_SendTelegram (stream->sndQueue, 
		    Net_CreateTelegram (XBT_COT_SendData, XBT_ID_GameStat, 0, tmp, 4+numPlayers*8) );
} /* X2C_SendGameStat */

/*
 * send request for disconnect to central
 */
void
X2C_SendDisconnect (XBComm *comm)
{
  XBCommStream *stream = (XBCommStream *) comm;

  /* inform host about disconnect request */
  Socket_RegisterWrite (CommSocket (&stream->comm));
  Net_SendTelegram (stream->sndQueue, 
		    Net_CreateTelegram (XBT_COT_Spontaneous, XBT_ID_HostDisconnected, 0, NULL, 0) );
  Net_SendTelegram (stream->sndQueue, 
		    Net_CreateTelegram (XBT_COT_Activate, XBT_ID_RequestDisconnect, 0, NULL, 0) );
} /* S2C_Disconnect */


/*
 * return address of server in dot-represantation
 */
const char *
X2C_CentralName (XBComm *comm)
{
  return Net_RemoteName (comm->socket);
} /* C2S_ServerName */

/*
 * return address of client (local host) in dot-represantation
 */
const char *
X2C_LocalName (XBComm *comm)
{
  return Net_LocalName (comm->socket);
} /* C2S_ClientName */

/*
 * end of file com_to_central.c
 */
