/* WordPerfectCollector: Collects sections and runs of text from a 
 * wordperfect file (and styles to go along with them) and writes them 
 * to a Writer target document
 *
 * Copyright (C) 2002-2003 William Lachance (william.lachance@sympatico.ca)
 * Copyright (C) 2003 Net Integration Technologies (http://www.net-itech.com)
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 * For further information visit http://libwpd.sourceforge.net
 *
 */

/* "This product is not manufactured, approved, or supported by 
 * Corel Corporation or Corel Corporation Limited."
 */

#include <libwpd/libwpd.h>
#include <glib.h>
#include <string.h> // for strcmp

#include "WordPerfectCollector.h"
#include "DocumentElement.h"
#include "TextRunStyle.h"
#include "FontStyle.h"
#include "ListStyle.h"
#include "PageSpan.h"
#include "SectionStyle.h"
#include "TableStyle.h"
#include "FilterInternal.h"
#include "WriterProperties.h"
#include "FontMap.h"

const char *headerStr = "<?xml version=\"1.0\" encoding=\"UTF-8\"?><!DOCTYPE office:document-content PUBLIC \"-//OpenOffice.org//DTD OfficeDocument 1.0//EN\" \"office.dtd\">\n";

WordPerfectCollector::WordPerfectCollector()
{
	_resetDocumentState();
}

bool WordPerfectCollector::filter(GsfInput *pInput, GsfOutfile *pOutfile)
{
 	// initialize temporary state
	_resetDocumentState();

	// parse & write
 	if (!_parseSourceDocument(pInput))
		return false;
	if (!_writeTargetDocument(pOutfile))
		return false;

 	// clean up the mess we made
 	WRITER_DEBUG_MSG(("WriterWordPerfect: Cleaning up our mess..\n"));

	WRITER_DEBUG_MSG(("Destroying the body elements\n"));
	for (vector<DocumentElement *>::iterator iter = mBodyElements.begin(); iter != mBodyElements.end(); iter++) {
		delete((*iter));
		(*iter) = NULL;
	}

	WRITER_DEBUG_MSG(("Destroying the styles elements\n"));
	for (vector<DocumentElement *>::iterator iter = mStylesElements.begin(); iter != mStylesElements.end(); iter++) {
 		if (!(*iter))
			WRITER_DEBUG_MSG(("NULL\n"));
		else
			(*iter)->print();
 		delete (*iter);
		(*iter) = NULL; // we may pass over the same element again (in the case of headers/footers spanning multiple pages)
		                // so make sure we don't do a double del
	}
	
	WRITER_DEBUG_MSG(("Destroying the rest of the styles elements\n"));
	for (map<UTF8String, TextRunStyle *>::iterator iter = mParagraphStyleHash.begin(); iter != mParagraphStyleHash.end(); iter++) {
		delete(iter->second);
	}
	for (map<UTF8String, FontStyle *>::iterator iter = mFontHash.begin(); iter != mFontHash.end(); iter++) {
		delete(iter->second);
	}

	for (vector<ListStyle *>::iterator iter = mListStyles.begin(); iter != mListStyles.end(); iter++) {
		delete((*iter));
	}
	for (vector<SectionStyle *>::iterator iter = mSectionStyles.begin(); iter != mSectionStyles.end(); iter++) {
		delete((*iter));
	}
	for (vector<TableStyle *>::iterator iter = mTableStyles.begin(); iter != mTableStyles.end(); iter++) {
		delete((*iter));
	}

	for (vector<PageSpan *>::iterator iter = mPageSpans.begin(); iter != mPageSpans.end(); iter++) {
		delete((*iter));
	}

 	return true;
 }

void WordPerfectCollector::_resetDocumentState()
{
	while (!mWriterDocumentState.mFakeSectionStack.empty())
		mWriterDocumentState.mFakeSectionStack.pop();
    
	mWriterDocumentState.mbListElementOpenedAtCurrentLevel = false;

	mWriterDocumentState.mbTableCellOpened = false;	

	miNumStyles = 1;
	miNumSections = 0;
	miNumTables = 0;
	miNumListStyles = 0;
	miCurrentListLevel = 0;
	miLastListLevel = 0;
	miLastListNumber = 0;
	miCurrentNumColumns = 1;
	mfSectionSpaceAfter = 0.0f;
	mbListContinueNumbering = false;
	mbListElementParagraphOpened = false;
	mbListElementOpened = false;

// 	WRITER_DEBUG_MSG(("WriterWordPerfect:Adding no properties style\n"));
// 	mWriterDocumentState.mbFirstElement = false;
//  	_requestParagraphStyle(WP6_PARAGRAPH_JUSTIFICATION_LEFT, 0, 0.0f, 0.0f, IMP_DEFAULT_FONT_NAME, IMP_DEFAULT_FONT_SIZE, 0.0f,
// 			       false, false); 
 	mWriterDocumentState.mbFirstElement = true;

 	mSectionStyles.clear();

	mStylesElements.clear();
	mBodyElements.clear();
	mpCurrentContentElements = &mBodyElements;

 	mTableStyles.clear();
 	mListStyles.clear();

	miNumPageStyles = 0;

	mpCurrentPageSpan = NULL;

	// fixme: should we clear the paragraph styles class as well? 
	// or maybe this whole function is completely redundant? need
	// to think more about how this class is actually being used

	mpCurrentListStyle = NULL;
}

bool WordPerfectCollector::_parseSourceDocument(GsfInput *pInput)
{
	// create a header for the preamble + add some default settings to it

 	WRITER_DEBUG_MSG(("WriterWordPerfect: Attempting to process state\n"));
	bool bRetVal = true;
	try {
		WPDocument::parse(pInput, static_cast<WPXHLListenerImpl *>(this));
	}
	catch (FileException)
		{
			WRITER_DEBUG_MSG(("Caught a file exception..\n"));
			bRetVal = false;
		}

	return bRetVal;
}

void WordPerfectCollector::_writeDefaultStyles(GsfOutput *pOutput)
{
	TagOpenElement stylesOpenElement("office:styles");
	stylesOpenElement.write(pOutput);

	TagOpenElement defaultParagraphStyleOpenElement("style:default-style");
	defaultParagraphStyleOpenElement.addAttribute("style:family", "paragraph");
	defaultParagraphStyleOpenElement.write(pOutput);

	TagOpenElement defaultParagraphStylePropertiesOpenElement("style:properties");
	defaultParagraphStylePropertiesOpenElement.addAttribute("style:family", "paragraph");
	defaultParagraphStylePropertiesOpenElement.addAttribute("style:tab-stop-distance", "0.5inch");
	defaultParagraphStylePropertiesOpenElement.write(pOutput);
	TagCloseElement defaultParagraphStylePropertiesCloseElement("style:properties");
	defaultParagraphStylePropertiesCloseElement.write(pOutput);

	TagCloseElement defaultParagraphStyleCloseElement("style:default-style");
	defaultParagraphStyleCloseElement.write(pOutput);
	TagCloseElement stylesCloseElement("office:styles");
	stylesCloseElement.write(pOutput);

}

void WordPerfectCollector::_writeContentPreamble(GsfOutput *pOutput)
{
// the old content pre-amble: remove me after a few releases
// const char *headerStr = "<?xml version=\"1.0\" encoding=\"UTF-8\"?><!DOCTYPE office:document-content PUBLIC \"-//OpenOffice.org//DTD OfficeDocument 1.0//EN\" \"office.dtd\"><office:document-content xmlns:office=\"http://openoffice.org/2000/office\" xmlns:style=\"http://openoffice.org/2000/style\" xmlns:text=\"http://openoffice.org/2000/text\" xmlns:number=\"http://openoffice.org/2000/datastyle\" xmlns:table=\"http://openoffice.org/2000/table\" xmlns:fo=\"http://www.w3.org/1999/XSL/Format\" office:class=\"text\"><office:script/>";
	TagOpenElement documentContentOpenElement("office:document-content");
	documentContentOpenElement.addAttribute("xmlns:office", "http://openoffice.org/2000/office");
	documentContentOpenElement.addAttribute("xmlns:style", "http://openoffice.org/2000/style");
	documentContentOpenElement.addAttribute("xmlns:text", "http://openoffice.org/2000/text");
	documentContentOpenElement.addAttribute("xmlns:table", "http://openoffice.org/2000/table");
	documentContentOpenElement.addAttribute("xmlns:draw", "http://openoffice.org/2000/draw");
	documentContentOpenElement.addAttribute("xmlns:fo", "http://www.w3.org/1999/XSL/Format");
	documentContentOpenElement.addAttribute("xmlns:xlink", "http://www.w3.org/1999/xlink");
	documentContentOpenElement.addAttribute("xmlns:number", "http://openoffice.org/2000/datastyle");
	documentContentOpenElement.addAttribute("xmlns:svg", "http://www.w3.org/2000/svg");
	documentContentOpenElement.addAttribute("xmlns:chart", "http://openoffice.org/2000/chart");
	documentContentOpenElement.addAttribute("xmlns:dr3d", "http://openoffice.org/2000/dr3d");
	documentContentOpenElement.addAttribute("xmlns:math", "http://www.w3.org/1998/Math/MathML");
	documentContentOpenElement.addAttribute("xmlns:form", "http://openoffice.org/2000/form");
	documentContentOpenElement.addAttribute("xmlns:script", "http://openoffice.org/2000/script");
	documentContentOpenElement.addAttribute("office:class", "text");
	documentContentOpenElement.addAttribute("office:version", "1.0");
	documentContentOpenElement.write(pOutput);
}

void WordPerfectCollector::_writeMasterPages(GsfOutput *pOutput)
{
	TagOpenElement masterStylesOpen("office:master-styles");
	masterStylesOpen.write(pOutput);
	int pageNumber = 1;
	for (int i=0; i<mPageSpans.size(); i++)
	{
		bool bLastPage;
		(i == (mPageSpans.size() - 1)) ? bLastPage = true : bLastPage = false;
		mPageSpans[i]->writeMasterPages(pageNumber, i, bLastPage, pOutput);
		pageNumber += mPageSpans[i]->getSpan();
	}
	TagCloseElement masterStylesClose("office:master-styles");
	masterStylesClose.write(pOutput);	
}

void WordPerfectCollector::_writePageMasters(GsfOutput *pOutput)
{
	int pageNumber = 1;
	for (int i=0; i<mPageSpans.size(); i++)
	{
		mPageSpans[i]->writePageMaster(i, pOutput);
	}
}

bool WordPerfectCollector::_writeTargetDocument(GsfOutfile *pOutfile)
{
	WRITER_DEBUG_MSG(("WriterWordPerfect: Document Body: Printing out the header stuff..\n"));
	GsfOutput *pContentChild = gsf_outfile_new_child(pOutfile, "content.xml", FALSE);

 	if (!gsf_output_write (pContentChild, strlen (headerStr), (guint8 *)headerStr))
 		return false;
	_writeContentPreamble(pContentChild);

 	WRITER_DEBUG_MSG(("WriterWordPerfect: Document Body: Writing out the styles..\n"));

	// write out the font styles
	gsf_output_printf (pContentChild, "<office:font-decls>\n");
	for (map<UTF8String, FontStyle *>::iterator iter = mFontHash.begin(); iter != mFontHash.end(); iter++) {
		iter->second->write(pContentChild);
	}
	TagOpenElement symbolFontOpen("style:font-decl");
	symbolFontOpen.addAttribute("style:name", "StarSymbol");
	symbolFontOpen.addAttribute("fo:font-family", "StarSymbol");
	symbolFontOpen.addAttribute("style:font-charset", "x-symbol");
	symbolFontOpen.write(pContentChild);
	TagCloseElement symbolFontClose("style:font-decl");
	symbolFontClose.write(pContentChild);
	gsf_output_printf (pContentChild, "</office:font-decls>\n");

	// write default styles
	_writeDefaultStyles(pContentChild);

	// write automatic styles: which encompasses quite a bit
       	gsf_output_printf (pContentChild, "<office:automatic-styles>\n");
	for (map<UTF8String, TextRunStyle *>::iterator iter = mParagraphStyleHash.begin(); iter != mParagraphStyleHash.end(); iter++) {
		// writing out the paragraph styles
		if (strcmp((iter->second)->getName()->str, "Standard") || 
		    !(dynamic_cast<ParagraphStyle *>(iter->second))) { 
			// don't write standard paragraph "no styles" style
			(iter->second)->write(pContentChild);
		}
	}

 	// writing out the sections styles
	for (vector<SectionStyle *>::iterator iter = mSectionStyles.begin(); iter != mSectionStyles.end(); iter++) {
		(*iter)->write(pContentChild);
	}

	// writing out the lists styles
	for (vector<ListStyle *>::iterator iter = mListStyles.begin(); iter != mListStyles.end(); iter++) {
		(*iter)->write(pContentChild);
	}

 	// writing out the table styles
	for (vector<TableStyle *>::iterator iter = mTableStyles.begin(); iter != mTableStyles.end(); iter++) {
		(*iter)->write(pContentChild);
	}

	// writing out the page masters
	_writePageMasters(pContentChild);

	gsf_output_printf (pContentChild, "</office:automatic-styles>\n");

	_writeMasterPages(pContentChild);
		
 	WRITER_DEBUG_MSG(("WriterWordPerfect: Document Body: Writing out the document..\n"));
 	// writing out the document
	gsf_output_printf(pContentChild, "<office:body>\n");
	
	for (vector<DocumentElement *>::iterator iter = mBodyElements.begin(); iter != mBodyElements.end(); iter++) {
		(*iter)->write(pContentChild);
	}
 	WRITER_DEBUG_MSG(("WriterWordPerfect: Document Body: Finished writing all doc els..\n"));

	gsf_output_printf(pContentChild, "</office:body>\n");
	gsf_output_printf(pContentChild, "</office:document-content>\n");
	if (!gsf_output_close (pContentChild))
		return false;
	g_object_unref (pContentChild);
	
	return true;
}

// _requestParagraphRunStyle: returns a paragraph style, if it already exists. creates it, adds it 
// to the list of defined styles, and returns it otherwise.
ParagraphStyle * WordPerfectCollector::_requestParagraphStyle(const guint8 iParagraphJustification, const guint32 iTextAttributeBits,
							      const float fMarginLeft, const float fMarginRight,
							      const gchar *pFontName, const float fFontSize, 
							      const float fLineSpacing,
							      const bool bColumnBreak, const bool bPageBreak,
							      const gchar *pParentName, const gchar *pName)
{
	if (mWriterDocumentState.mbFirstElement && mpCurrentContentElements == &mBodyElements)
	{
		WRITER_DEBUG_MSG(("WriterWordPerfect: If.. (mbFirstElement=%i)", mWriterDocumentState.mbFirstElement));
		ParagraphStyle * pParagraphStyle = new ParagraphStyle(iParagraphJustification, iTextAttributeBits, 
								      fMarginLeft, fMarginRight,
								      pFontName, fFontSize, fLineSpacing, 
								      bColumnBreak, bPageBreak, "FS", pParentName);
		UTF8String sParagraphHashKey("P|FS");
		UTF8String sMasterPageName("Page Style 1");
		pParagraphStyle->setMasterPageName(sMasterPageName);
		mParagraphStyleHash[sParagraphHashKey] = pParagraphStyle;
		mWriterDocumentState.mbFirstElement = false;

		return pParagraphStyle;
 	}

	// else.. do the following:
	WRITER_DEBUG_MSG(("WriterWordPerfect: Else.. (mbFirstElement=%i)", mWriterDocumentState.mbFirstElement));

	UTF8String sParagraphHashKey;
	sParagraphHashKey.sprintf("P|%s|%i|%f|%f|%f|%i|%i", pParentName, iParagraphJustification,
				     fMarginLeft, fMarginRight, fLineSpacing, bColumnBreak, bPageBreak);
	WRITER_DEBUG_MSG(("WriterWordPerfect: P|%s|%i|%f|%f|%f|%i|%i|%s|%s\n", pParentName, iParagraphJustification, fMarginLeft, fMarginRight,  fLineSpacing, bColumnBreak, bPageBreak, pParentName, pName));
	WRITER_DEBUG_MSG(("WriterWordPerfect: Paragraph Hash Key: %s\n", sParagraphHashKey.getUTF8()));

	// Get the style
	ParagraphStyle * pParagraphStyle = NULL;	
	if (mParagraphStyleHash.find(sParagraphHashKey) == mParagraphStyleHash.end()) {
		// allocate a new paragraph style
		const char *pFinalStyleName = NULL;
		UTF8String sName;
		sName.sprintf("S%i", miNumStyles); // have to define here so it doesn't go out of scope
		if (pName != NULL)
			pFinalStyleName = pName;
		else {
			pFinalStyleName = sName.getUTF8();
		}
		WRITER_DEBUG_MSG(("WriterWordPerfect: final paragraph style name: %s\n", pFinalStyleName));

		pParagraphStyle = new ParagraphStyle(iParagraphJustification, iTextAttributeBits, 
						     fMarginLeft, fMarginRight,
						     pFontName, fFontSize, fLineSpacing, 
						     bColumnBreak, bPageBreak, pFinalStyleName, pParentName); 
				
		miNumStyles++;
		mParagraphStyleHash[sParagraphHashKey] = pParagraphStyle;
		WRITER_DEBUG_MSG(("WriterWordPerfect: Successfully added to hash, returning it\n"));
	}
	else 
	{
		pParagraphStyle = static_cast<ParagraphStyle *>(mParagraphStyleHash.find(sParagraphHashKey)->second);
	}
	
	return pParagraphStyle;
}


// _requestParagraphRunStyle: returns a paragraph style, if it already exists. creates it, adds it 
// to the list of defined styles, and returns it otherwise.
ParagraphStyle * WordPerfectCollector::_requestListParagraphStyle(const ListStyle * pListStyle, const guint8 iParagraphJustification, 
								  const guint32 iTextAttributeBits,
								  const float fMarginLeft, const float fMarginRight,
								  const gchar *pFontName, const float fFontSize, 
								  const float fLineSpacing)
{		
	if (mWriterDocumentState.mbFirstElement && mpCurrentContentElements == &mBodyElements)
	{
		WRITER_DEBUG_MSG(("WriterWordPerfect: If.. (mbFirstElement=%i)", mWriterDocumentState.mbFirstElement));
		ParagraphStyle * pListParagraphStyle = new ParagraphStyle(iParagraphJustification, iTextAttributeBits, 
									  fMarginLeft, fMarginRight,
									  pFontName, fFontSize, fLineSpacing,
									  false, false,
									  "FS1", "Standard"); 

		UTF8String sParagraphHashKey("P|ListFS");
		UTF8String sMasterPageName("Page Style 1");
		pListParagraphStyle->setMasterPageName(sMasterPageName);
		mParagraphStyleHash[sParagraphHashKey] = pListParagraphStyle;
		mWriterDocumentState.mbFirstElement = false;

		return pListParagraphStyle;
 	}

	UTF8String sParagraphHashKey;
	sParagraphHashKey.sprintf("P|%s|%i|%i|%f|%f|%s|%f|%f", pListStyle->getName(), 
				     iParagraphJustification, iTextAttributeBits,
				     fMarginLeft, fMarginRight, pFontName, fFontSize, fLineSpacing);
	WRITER_DEBUG_MSG(("WriterWordPerfect: Paragraph Hash Key: %s\n", sParagraphHashKey.getUTF8()));

	// Get the style
	ParagraphStyle * pListParagraphStyle = NULL;
	if (mParagraphStyleHash.find(sParagraphHashKey) == mParagraphStyleHash.end()) {
		// allocate a new paragraph style
		UTF8String sName;
		sName.sprintf("S%i", miNumStyles);
		UTF8String sListStyleName(pListStyle->getName()->str);

		pListParagraphStyle = new ParagraphStyle(iParagraphJustification, iTextAttributeBits, 
							 fMarginLeft, fMarginRight,
							 pFontName, fFontSize, fLineSpacing,
							 false, false,
							 sName.getUTF8(), "Standard"); 
		pListParagraphStyle->setListStyleName(sListStyleName);		
		miNumStyles++;
		mParagraphStyleHash[sParagraphHashKey] = pListParagraphStyle;
	}
	
	WRITER_DEBUG_MSG(("WriterWordPerfect: Successfully added to hash, returning it\n"));
	
	return static_cast<ParagraphStyle *>(mParagraphStyleHash.find(sParagraphHashKey)->second);
}

// _allocateFontName: add a (potentially mapped) font style to the hash if it's not already there, do nothing otherwise
void WordPerfectCollector::_allocateFontName(const UTF8String & sFontName)
{
	if (mFontHash.find(sFontName) == mFontHash.end())
	{
		FontStyle *pFontStyle = new FontStyle(sFontName.getUTF8(), sFontName.getUTF8());
		mFontHash[sFontName] = pFontStyle;
	}
}

void WordPerfectCollector::openPageSpan(const int span, const bool isLastPageSpan,
					const float marginLeft, const float marginRight,
					const float marginTop, const float marginBottom)
{
	PageSpan *pPageSpan = new PageSpan(span, marginLeft, marginRight, marginTop, marginBottom);
	mPageSpans.push_back(pPageSpan);
	mpCurrentPageSpan = pPageSpan;
}

void WordPerfectCollector::openHeaderFooter(const WPXHeaderFooterType headerFooterType, const WPXHeaderFooterOccurence headerFooterOccurence)
{
	vector<DocumentElement *> * pHeaderFooterContentElements = new vector<DocumentElement *>;
	
	if (headerFooterType == HEADER)
	{
		switch (headerFooterOccurence)
		{
		case ALL:
		case ODD:
			WRITER_DEBUG_MSG(("WriterWordPerfect: Opening h_all or h_odd\n"));
			mpCurrentPageSpan->setHeaderContent(pHeaderFooterContentElements);
			break;
		case EVEN:
			WRITER_DEBUG_MSG(("WriterWordPerfect: Opening h_even\n"));
			mpCurrentPageSpan->setHeaderLeftContent(pHeaderFooterContentElements);
			break;
		}
	}
	else
	{
		switch (headerFooterOccurence)
		{
		case ALL:
		case ODD:
			WRITER_DEBUG_MSG(("WriterWordPerfect: Opening f_all or f_odd\n"));
			mpCurrentPageSpan->setFooterContent(pHeaderFooterContentElements);
			break;
		case EVEN:
			WRITER_DEBUG_MSG(("WriterWordPerfect: Opening f_even\n"));
			mpCurrentPageSpan->setFooterLeftContent(pHeaderFooterContentElements);
			break;
		}
	}

	mpCurrentContentElements = pHeaderFooterContentElements;
}

void WordPerfectCollector::closeHeaderFooter(const WPXHeaderFooterType headerFooterType, 
					     const WPXHeaderFooterOccurence headerFooterOccurence)
{
	mpCurrentContentElements = &mBodyElements;
}

void WordPerfectCollector::openSection(const unsigned int numColumns, const float spaceAfter)
{
	// OLD: since a section for OO means a different thing than it does for libwpd, we only
	// start a new section if the number of columns has changed
	// FIXME: do we want to remove some of the logic here? libwpd no longer sends
	// fake sections, but maybe this isn't what we want?
	if (miCurrentNumColumns != numColumns) {
		miCurrentNumColumns = numColumns;

		if (miCurrentNumColumns > 1)
		{
			miNumSections++;
			mfSectionSpaceAfter = spaceAfter;
			UTF8String sSectionName;
			sSectionName.sprintf("Section%i", miNumSections);
			WRITER_DEBUG_MSG(("WriterWordPerfect:  New Section: %s\n", sSectionName.getUTF8()));
			
			SectionStyle *pSectionStyle = new SectionStyle(miCurrentNumColumns, sSectionName.getUTF8());
			mSectionStyles.push_back(pSectionStyle);
			
			TagOpenElement *pSectionOpenElement = new TagOpenElement("text:section");
			pSectionOpenElement->addAttribute("text:style-name", pSectionStyle->getName()->str);
			pSectionOpenElement->addAttribute("text:name", pSectionStyle->getName()->str);
			mpCurrentContentElements->push_back(static_cast<DocumentElement *>(pSectionOpenElement));			
		}
		else
			mWriterDocumentState.mFakeSectionStack.push(1);
	}
	else {
		mWriterDocumentState.mFakeSectionStack.push(1);
	}
	
}

void WordPerfectCollector::closeSection()
{
	if (mWriterDocumentState.mFakeSectionStack.empty())
		mpCurrentContentElements->push_back(static_cast<DocumentElement *>(new TagCloseElement("text:section")));	
	else
		mWriterDocumentState.mFakeSectionStack.pop();

	// open as many paragraphs as needed to simulate section space after
	for (float f=0.0f; f<mfSectionSpaceAfter; f+=1.0f) {
		openParagraph(WPX_PARAGRAPH_JUSTIFICATION_LEFT, 0, 0.0f, 0.0f, IMP_DEFAULT_FONT_NAME, IMP_DEFAULT_FONT_SIZE, 0.0f,
 			       false, false);
		closeParagraph();
	}
	mfSectionSpaceAfter = 0.0f;
}

void WordPerfectCollector::openParagraph(const guint8 paragraphJustification, const guint32 textAttributeBits,
					 const float marginLeftOffset, const float marginRightOffset,
					 const gchar *fontName, const float fontSize, 
					 const float lineSpacing,
					 const bool isColumnBreak, const bool isPageBreak)
{
	TextRunStyle *pTextRunStyle;
	UTF8String sMappedFontName(mapFont(fontName));
	_allocateFontName(sMappedFontName);

	if (mWriterDocumentState.mbTableCellOpened)
		pTextRunStyle = _requestParagraphStyle(paragraphJustification, textAttributeBits, 0.0f, 0.0f, 
						       sMappedFontName.getUTF8(), fontSize, lineSpacing, false, false,
						       "Table Contents");
	else if (miCurrentNumColumns > 1)
		pTextRunStyle = _requestParagraphStyle(paragraphJustification, textAttributeBits, 
						       0.0f, 0.0f, 
						       sMappedFontName.getUTF8(), fontSize, lineSpacing, isColumnBreak, isPageBreak,
						       "Standard");
	else
		pTextRunStyle = _requestParagraphStyle(paragraphJustification, textAttributeBits, 
						       marginLeftOffset, marginRightOffset, 
						       sMappedFontName.getUTF8(), fontSize, lineSpacing, isColumnBreak, isPageBreak,
						       "Standard");
	

	// create a document element corresponding to the paragraph, and append it to our list of document elements	
	TagOpenElement *pParagraphOpenElement = new TagOpenElement("text:p");
	pParagraphOpenElement->addAttribute("text:style-name", pTextRunStyle->getName()->str);	
	mpCurrentContentElements->push_back(static_cast<DocumentElement *>(pParagraphOpenElement));	
}

void WordPerfectCollector::closeParagraph()
{
	mpCurrentContentElements->push_back(static_cast<DocumentElement *>(new TagCloseElement("text:p")));	
}

void WordPerfectCollector::openSpan(const guint32 textAttributeBits, const gchar *fontName, const float fontSize)
{
	UTF8String sMappedFontName(mapFont(fontName));
	_allocateFontName(sMappedFontName);
	UTF8String sSpanHashKey;
	sSpanHashKey.sprintf("S|%i|%s|%f", textAttributeBits, sMappedFontName.getUTF8(), fontSize);
	WRITER_DEBUG_MSG(("WriterWordPerfect: Span Hash Key: %s\n", sSpanHashKey.getUTF8()));

	// Get the style
	TextRunStyle * pTextRunStyle = NULL;
	if (mParagraphStyleHash.find(sSpanHashKey) == mParagraphStyleHash.end()) {
		// allocate a new paragraph style
		UTF8String sName;
		sName.sprintf("S%i", miNumStyles);
		pTextRunStyle = new SpanStyle(textAttributeBits, sMappedFontName.getUTF8(), fontSize, sName.getUTF8()); 
		
		miNumStyles++;
		mParagraphStyleHash[sSpanHashKey] = pTextRunStyle;
	}
	else {
		pTextRunStyle = mParagraphStyleHash.find(sSpanHashKey)->second; // yes, this could be optimized (see dup call above)
	}

	// create a document element corresponding to the paragraph, and append it to our list of document elements
	TagOpenElement *pSpanOpenElement = new TagOpenElement("text:span");
	pSpanOpenElement->addAttribute("text:style-name", pTextRunStyle->getName()->str);
	mpCurrentContentElements->push_back(static_cast<DocumentElement *>(pSpanOpenElement));	
}

void WordPerfectCollector::closeSpan()
{
	mpCurrentContentElements->push_back(static_cast<DocumentElement *>(new TagCloseElement("text:span")));	
}

void WordPerfectCollector::defineOrderedListLevel(const int listID, const guint16 listLevel, const WPXNumberingType listType, 
						  const UCSString &textBeforeNumber, const UCSString &textAfterNumber,
						  int startingNumber)
{
	WRITER_DEBUG_MSG(("Define ordered list level (listid: %i)\n", listID));
	OrderedListStyle *pOrderedListStyle = dynamic_cast<OrderedListStyle *>(mpCurrentListStyle);
	// this rather appalling conditional makes sure we only start a new list (rather than continue an old
	// one) iff: (1) we have no prior list OR (2) the prior list is actually definitively different
	// from the list that is just being defined (listIDs differ) OR (3) we can tell that the user actually
	// is starting a new list at level 1 (and only level 1)
	if (pOrderedListStyle == NULL || pOrderedListStyle->getListID() != listID || 
	    (listLevel==1 && (startingNumber != (miLastListNumber+1))))
	{
		WRITER_DEBUG_MSG(("Attempting to create a new ordered list style (listid: %i)\n"));
		UTF8String sName;
		sName.sprintf("OL%i", miNumListStyles);
		miNumListStyles++;
		pOrderedListStyle = new OrderedListStyle(sName.getUTF8(), listID);
		mListStyles.push_back(static_cast<ListStyle *>(pOrderedListStyle));
		mpCurrentListStyle = static_cast<ListStyle *>(pOrderedListStyle);
		mbListContinueNumbering = false;
		miLastListNumber = 0;
	}
	else
		mbListContinueNumbering = true;
	
	pOrderedListStyle->updateListLevel(miCurrentListLevel, listType, textBeforeNumber, textAfterNumber, startingNumber);
}

void WordPerfectCollector::defineUnorderedListLevel(const int listID, const guint16 listLevel, const UCSString &bullet)
{
	WRITER_DEBUG_MSG(("Define unordered list level (listid: %i)\n", listID));
	UnorderedListStyle *pUnorderedListStyle = dynamic_cast<UnorderedListStyle *>(mpCurrentListStyle);
	if (pUnorderedListStyle == NULL) {
		WRITER_DEBUG_MSG(("Attempting to create a new unordered list style (listid: %i)\n", listID));
		GString *psName = g_string_new(NULL);
		g_string_printf(psName, "UL%i", miNumListStyles);
		miNumListStyles++;
		pUnorderedListStyle = new UnorderedListStyle(psName->str, listID);
		g_string_free(psName, TRUE);
		mListStyles.push_back(static_cast<ListStyle *>(pUnorderedListStyle));
		mpCurrentListStyle = static_cast<ListStyle *>(pUnorderedListStyle);
	}
	pUnorderedListStyle->updateListLevel(miCurrentListLevel, bullet);
}

void WordPerfectCollector::openOrderedListLevel(const int listID)
{
	WRITER_DEBUG_MSG(("Open ordered list level (listid: %i)\n", listID));
	miCurrentListLevel++;
	TagOpenElement *pListLevelOpenElement = new TagOpenElement("text:ordered-list");
	_openListLevel(pListLevelOpenElement);

	if (mbListContinueNumbering) {
		pListLevelOpenElement->addAttribute("text:continue-numbering", "true");
	}

	mpCurrentContentElements->push_back(static_cast<DocumentElement *>(pListLevelOpenElement));
}

void WordPerfectCollector::openUnorderedListLevel(const int listID)
{
	WRITER_DEBUG_MSG(("Open unordered list level (listid: %i)\n", listID));
	miCurrentListLevel++;
	TagOpenElement *pListLevelOpenElement = new TagOpenElement("text:unordered-list");
	_openListLevel(pListLevelOpenElement);

	mpCurrentContentElements->push_back(static_cast<DocumentElement *>(pListLevelOpenElement));
}

void WordPerfectCollector::_openListLevel(TagOpenElement *pListLevelOpenElement)
{	
  	if (!mbListElementOpened && miCurrentListLevel > 1)
  	{
  		mpCurrentContentElements->push_back(static_cast<DocumentElement *>(new TagOpenElement("text:list-item")));	
  	}
	else if (mbListElementParagraphOpened)
	{
		mpCurrentContentElements->push_back(static_cast<DocumentElement *>(new TagCloseElement("text:p")));	
		mbListElementParagraphOpened = false;
	}

	if (miCurrentListLevel==1) {
		pListLevelOpenElement->addAttribute("text:style-name", mpCurrentListStyle->getName()->str);
	}		

	mbListElementOpened = false;
}

void WordPerfectCollector::closeOrderedListLevel()
{
	WRITER_DEBUG_MSG(("Close ordered list level)\n"));
	_closeListLevel("ordered-list");
}

void WordPerfectCollector::closeUnorderedListLevel()
{
	WRITER_DEBUG_MSG(("Close unordered list level\n"));
	_closeListLevel("unordered-list");
}

void WordPerfectCollector::_closeListLevel(const char *szListType)
{
	if (mbListElementOpened)
		mpCurrentContentElements->push_back(static_cast<DocumentElement *>(new TagCloseElement("text:list-item")));	

	miCurrentListLevel--;

	UTF8String sCloseElement;
	sCloseElement.sprintf("text:%s", szListType);
	mpCurrentContentElements->push_back(static_cast<DocumentElement *>(new TagCloseElement(sCloseElement.getUTF8())));	

	if (miCurrentListLevel > 0)
		mpCurrentContentElements->push_back(static_cast<DocumentElement *>(new TagCloseElement("text:list-item")));	
	mbListElementOpened = false;
}

void WordPerfectCollector::openListElement(const guint8 paragraphJustification, const guint32 textAttributeBits,
					   const float marginLeftOffset, const float marginRightOffset,
					   const gchar *fontName, const float fontSize, 
					   const float lineSpacing)
{
	UTF8String sMappedFontName(mapFont(fontName));
	_allocateFontName(sMappedFontName);

	miLastListLevel = miCurrentListLevel;
	if (miCurrentListLevel == 1)
		miLastListNumber++;

	if (mbListElementOpened)
		mpCurrentContentElements->push_back(static_cast<DocumentElement *>(new TagCloseElement("text:list-item")));

 	ParagraphStyle * pListParagraphStyle = _requestListParagraphStyle(mpCurrentListStyle, paragraphJustification, textAttributeBits, 
									  marginLeftOffset, marginRightOffset,
									  sMappedFontName.getUTF8(), fontSize, lineSpacing); 

	if (!pListParagraphStyle) {
		throw ParseException();
	}
	TagOpenElement *pOpenListElement = new TagOpenElement("text:list-item");
	TagOpenElement *pOpenListElementParagraph = new TagOpenElement("text:p");
	pOpenListElementParagraph->addAttribute("text:style-name", pListParagraphStyle->getName()->str);
	mpCurrentContentElements->push_back(static_cast<DocumentElement *>(pOpenListElement));
	mpCurrentContentElements->push_back(static_cast<DocumentElement *>(pOpenListElementParagraph));

	mbListElementOpened = true;
	mbListElementParagraphOpened = true;
	mbListContinueNumbering = false;
}

void WordPerfectCollector::closeListElement()
{
	WRITER_DEBUG_MSG(("close list element\n"));

	// this code is kind of tricky, because we don't actually close the list element (because this list element
	// could contain another list level in OOo's implementation of lists). that is done in the closeListLevel
	// code (or when we open another list element)

	if (mbListElementParagraphOpened)
	{
		mpCurrentContentElements->push_back(static_cast<DocumentElement *>(new TagCloseElement("text:p")));	
		mbListElementParagraphOpened = false;
	}
}

void WordPerfectCollector::openFootnote(int number)
{
	WRITER_DEBUG_MSG(("open footnote\n"));
	mpCurrentContentElements->push_back(static_cast<DocumentElement *>(new TagOpenElement("text:footnote")));

	mpCurrentContentElements->push_back(static_cast<DocumentElement *>(new TagOpenElement("text:footnote-citation")));
	UTF8String sFootnoteNumber;
	sFootnoteNumber.sprintf("%i", number);
	mpCurrentContentElements->push_back(static_cast<DocumentElement *>(new CharDataElement(sFootnoteNumber.getUTF8())));
	mpCurrentContentElements->push_back(static_cast<DocumentElement *>(new TagCloseElement("text:footnote-citation")));

	mpCurrentContentElements->push_back(static_cast<DocumentElement *>(new TagOpenElement("text:footnote-body")));

}

void WordPerfectCollector::closeFootnote()
{
	WRITER_DEBUG_MSG(("close footnote\n"));
	mpCurrentContentElements->push_back(static_cast<DocumentElement *>(new TagCloseElement("text:footnote-body")));
	mpCurrentContentElements->push_back(static_cast<DocumentElement *>(new TagCloseElement("text:footnote")));
}

void WordPerfectCollector::openEndnote(int number)
{
	mpCurrentContentElements->push_back(static_cast<DocumentElement *>(new TagOpenElement("text:endnote")));

	mpCurrentContentElements->push_back(static_cast<DocumentElement *>(new TagOpenElement("text:endnote-citation")));
	UTF8String sEndnoteNumber;
	sEndnoteNumber.sprintf("%i", number);
	mpCurrentContentElements->push_back(static_cast<DocumentElement *>(new CharDataElement(sEndnoteNumber.getUTF8())));
	mpCurrentContentElements->push_back(static_cast<DocumentElement *>(new TagCloseElement("text:endnote-citation")));

	mpCurrentContentElements->push_back(static_cast<DocumentElement *>(new TagOpenElement("text:endnote-body")));

}
void WordPerfectCollector::closeEndnote()
{
	WRITER_DEBUG_MSG(("close endnote\n"));
	mpCurrentContentElements->push_back(static_cast<DocumentElement *>(new TagCloseElement("text:endnote-body")));
	mpCurrentContentElements->push_back(static_cast<DocumentElement *>(new TagCloseElement("text:endnote")));
}

void WordPerfectCollector::openTable(const guint8 tablePositionBits, const float marginLeftOffset, const float marginRightOffset,
				     const float leftOffset, const vector < WPXColumnDefinition > &columns)
{
	mbTableFirstRow = true;
	miNumTables++;
	
	UTF8String sTableName;
	sTableName.sprintf("Table%i", miNumTables);
	WRITER_DEBUG_MSG(("WriterWordPerfect:  New Table: %s\n", sTableName.getUTF8()));
	
	// FIXME: we base the table style off of the page's margin left, ignoring (potential) wordperfect margin
	// state which is transmitted inside the page. could this lead to unacceptable behaviour?
	TableStyle *pTableStyle = new TableStyle(mpCurrentPageSpan->getMarginLeft(), mpCurrentPageSpan->getMarginRight(), marginLeftOffset, marginRightOffset, tablePositionBits, leftOffset, columns, sTableName.getUTF8());

	if (mWriterDocumentState.mbFirstElement && mpCurrentContentElements == &mBodyElements)
	{
		UTF8String sMasterPageName("Page Style 1");
		pTableStyle->setMasterPageName(sMasterPageName);
		mWriterDocumentState.mbFirstElement = false;
	}

	mTableStyles.push_back(pTableStyle);

	mpCurrentTableStyle = pTableStyle;
	
	TagOpenElement *pTableOpenElement = new TagOpenElement("table:table");

	pTableOpenElement->addAttribute("table:name", sTableName.getUTF8());
	pTableOpenElement->addAttribute("table:style-name", sTableName.getUTF8());
	mpCurrentContentElements->push_back(static_cast<DocumentElement *>(pTableOpenElement));	

	for (int i=0; i<pTableStyle->getNumColumns(); i++) {
		TagOpenElement *pTableColumnOpenElement = new TagOpenElement("table:table-column");
		UTF8String sColumnStyleName;
		sColumnStyleName.sprintf("%s.Column%i", sTableName.getUTF8(), (i+1));
		pTableColumnOpenElement->addAttribute("table:style-name", sColumnStyleName.getUTF8());
		mpCurrentContentElements->push_back(pTableColumnOpenElement);

		TagCloseElement *pTableColumnCloseElement = new TagCloseElement("table:table-column");
		mpCurrentContentElements->push_back(pTableColumnCloseElement);
	}
}

void WordPerfectCollector::openTableRow()
{

	mpCurrentContentElements->push_back(static_cast<DocumentElement *>(new TagOpenElement("table:table-row")));			
}

void WordPerfectCollector::closeTableRow()
{
	mpCurrentContentElements->push_back(static_cast<DocumentElement *>(new TagCloseElement("table:table-row")));			
}

void WordPerfectCollector::openTableCell(const guint32 col, const guint32 row, const guint32 colSpan, const guint32 rowSpan, 
					 const guint8 borderBits, const RGBSColor * cellFgColor, const RGBSColor * cellBgColor)
{
	float fLeftBorderThickness = (borderBits & WPX_TABLE_CELL_LEFT_BORDER_OFF) ? 0.0f : 0.0007f;
	float fRightBorderThickness = (borderBits & WPX_TABLE_CELL_RIGHT_BORDER_OFF) ? 0.0f : 0.0007f;
	float fTopBorderThickness = (borderBits & WPX_TABLE_CELL_TOP_BORDER_OFF) ? 0.0f : 0.0007f;
	float fBottomBorderThickness = (borderBits & WPX_TABLE_CELL_BOTTOM_BORDER_OFF) ? 0.0f : 0.0007f;
	WRITER_DEBUG_MSG(("WriterWordPerfect: Borderbits=%d\n", borderBits));

	GString *psTableCellStyleName = g_string_new(NULL);
	g_string_printf(psTableCellStyleName, "%s.Cell%i", mpCurrentTableStyle->getName()->str, mpCurrentTableStyle->getNumTableCellStyles());
	TableCellStyle *pTableCellStyle = new TableCellStyle(fLeftBorderThickness, fRightBorderThickness, 
							     fTopBorderThickness, fBottomBorderThickness,
							     cellFgColor, cellBgColor, 
							     psTableCellStyleName->str);
	mpCurrentTableStyle->addTableCellStyle(pTableCellStyle);

	TagOpenElement *pTableCellOpenElement = new TagOpenElement("table:table-cell");
	gchar *szNumColsSpanned = utf8_itoa(colSpan);
	gchar *szNumRowsSpanned = utf8_itoa(rowSpan);
	pTableCellOpenElement->addAttribute("table:style-name", psTableCellStyleName->str);
	pTableCellOpenElement->addAttribute("table:number-columns-spanned", szNumColsSpanned);
	pTableCellOpenElement->addAttribute("table:number-rows-spanned", szNumRowsSpanned);
	pTableCellOpenElement->addAttribute("table:value-type", "string");
	g_free(szNumColsSpanned); g_free(szNumRowsSpanned);
	mpCurrentContentElements->push_back(static_cast<DocumentElement *>(pTableCellOpenElement));


	mWriterDocumentState.mbTableCellOpened = true;
	g_string_free(psTableCellStyleName, TRUE);
}

void WordPerfectCollector::closeTableCell()
{
	mpCurrentContentElements->push_back(static_cast<DocumentElement *>(new TagCloseElement("table:table-cell")));			
	mWriterDocumentState.mbTableCellOpened = false;
}

void WordPerfectCollector::insertCoveredTableCell(const guint32 col, const guint32 row)
{
	mpCurrentContentElements->push_back(static_cast<DocumentElement *>(new TagOpenElement("table:covered-table-cell")));			
	mpCurrentContentElements->push_back(static_cast<DocumentElement *>(new TagCloseElement("table:covered-table-cell")));			
}

void WordPerfectCollector::closeTable()
{
	WRITER_DEBUG_MSG(("WriterWordPerfect: Closing Table\n"));
	mpCurrentContentElements->push_back(static_cast<DocumentElement *>(new TagCloseElement("table:table")));			
}

void WordPerfectCollector::insertLineBreak()
{
	WRITER_DEBUG_MSG(("WriterWordPerfect: Insert Line Break\n"));
	mpCurrentContentElements->push_back(static_cast<DocumentElement *>(new TagOpenElement("text:line-break")));			
	mpCurrentContentElements->push_back(static_cast<DocumentElement *>(new TagCloseElement("text:line-break")));			
}

void WordPerfectCollector::insertTab()
{
	WRITER_DEBUG_MSG(("WriterWordPerfect: Insert Tab\n"));
	mpCurrentContentElements->push_back(static_cast<DocumentElement *>(new TagOpenElement("text:tab-stop")));			
	mpCurrentContentElements->push_back(static_cast<DocumentElement *>(new TagCloseElement("text:tab-stop")));			
}

void WordPerfectCollector::insertText(const UCSString &text)
{
	WRITER_DEBUG_MSG(("WriterWordPerfect: Insert Text\n"));

	DocumentElement *pText = new TextElement(text);
	mpCurrentContentElements->push_back(pText);
}
