/******************************************************************************
 * program:     rasimg library 0.18                                           *
 * function:    Miscellaneous utilities for manipulation with images.         *
 * modul:       rasterut.cc                                                   *
 * licency:     GPL or LGPL                                                   *
 * Copyright: (c) 1998-2019 Jaroslav Fojtik                                   *
 ******************************************************************************/
#include<stdio.h>
#include<string.h>
#include<stdlib.h>

#include "typedfs.h"
#include "common.h"

#include "raster.h"
#include "vecimage.h"
#include "ras_prot.h"

#ifdef _MSC_VER  
  #pragma warning(disable: 4244)
  #pragma warning(disable: 4250)	// Inheriting via dominance, standard C++ feature.
#endif


/* ---------- Global class Image --------- */
void Image::Create(unsigned SizeX, unsigned SizeY, int Planes)
{
  if(Raster)
    {
    if(Raster->GetSize1D()==SizeX && Raster->Size2D==SizeY && 
       Raster->GetPlanes()==(Planes & 0xFF) && 
       Raster->UsageCount==1)		// only this object owns raster
		return;

    Erase();
    }

if(Planes & ImageTrueColor)
  {
  if( (Raster=CreateRaster2DRGB(SizeX, SizeY, Planes&0xFF)) != NULL)
	Raster->UsageCount++;
  }
else
  {
  if( (Raster=CreateRaster2D(SizeX,SizeY,Planes&0xFF))!=NULL)
	Raster->UsageCount++;
  }
}


Image::Image(const Image & I, bool AllFrames)
{
 if((Raster=I.Raster)!=NULL) Raster->UsageCount++;
 if((Palette=I.Palette)!=NULL) Palette->UsageCount++;
 if((VecImage=I.VecImage)!=NULL) VecImage->UsageCount++;
 x=I.x; y=I.y; dx=I.dx; dy=I.dy; RotAngle=I.RotAngle;
 Next = NULL;  

 if(AllFrames)
 {
   Image *PImg, *CurrImg;
   PImg = this;
   CurrImg = I.Next;
   while(CurrImg!=NULL)	//Multiple images
	{
	PImg->Next = new Image(*CurrImg,false);
	if(PImg->Next==NULL) break;	/*Allocation error*/
	PImg = PImg->Next;
	CurrImg = CurrImg->Next;
	}
 }
}

Raster2DAbstract *Image::operator=(Raster2DAbstract *NewRaster)
{
 x=y=dx=dy=0; 
 if(Raster==NewRaster) return(Raster);		//Identical assignment
 if(Raster!=NULL) Erase();
 if( (Raster=NewRaster)!=NULL ) Raster->UsageCount++;
 return(Raster);
}

Image &Image::operator=(const Image &I)
{
Image *PImg;
const Image *CurrImg;
 Erase();

 PImg = this;
 CurrImg = &I;
 goto CopyImg;

 while(CurrImg!=NULL)	//Multiple images
	{
	PImg->Next = new Image;
	if(PImg->Next==NULL) return(*this);
	PImg = PImg->Next;

CopyImg:if((PImg->Raster=CurrImg->Raster)!=NULL)
		CurrImg->Raster->UsageCount++;
	if((PImg->Palette=CurrImg->Palette)!=NULL)
		CurrImg->Palette->UsageCount++;
	PImg->x=CurrImg->x;PImg->y=CurrImg->y;
	PImg->dx=CurrImg->dx;PImg->dy=CurrImg->dy;
	PImg->RotAngle=CurrImg->RotAngle;
        if((PImg->VecImage=CurrImg->VecImage)!=NULL)
                CurrImg->VecImage->UsageCount++;
	CurrImg = CurrImg->Next;
	}
 return(*this);
}


void Image::Erase(void)
{
Image *I,*OldI;

 I = this;
 while(I!=NULL)
   {
   if(I->Raster!=NULL)
      {
      if(I->Raster->UsageCount--<=1) delete I->Raster;
      I->Raster = NULL;
      }
   if(I->Palette!=NULL)
      {
      if(I->Palette->UsageCount--<=1) delete I->Palette;
      I->Palette = NULL;
      }
   if(I->VecImage!=NULL)
      {
      if(I->VecImage->UsageCount--<=1) delete I->VecImage;      
      I->VecImage = NULL;
      }
   OldI = I;
   I = I->Next;
   OldI->Next = NULL;
   if(OldI!=this) delete(OldI);
   }
}


/* 0-none, 1-gray, 2-palette, 3-true color */
IMAGE_TYPE Image::ImageType(void) const
{
  if(Raster==NULL) return ImageNone;
  if(Raster->Channels()>=3) return ImageTrueColor;
  if(Palette!=NULL)
    {
    if(GrayPalette(Palette,Raster->GetPlanes())) return ImageGray;
    if(Raster->GetPlanes()==24) return ImageTrueColor;
    return ImagePalette;
    }
  if(Raster->GetPlanes()==24) return ImageTrueColor;
  return ImageGray;
}


DWORD Image::GetPixelRGB(unsigned Offset1D, unsigned Offset2D) const
{
  if(Raster==NULL) return 0;
  switch(ImageType() & 0xFF00)
    {
    case ImageTrueColor:
	     return Raster->GetValue2D(Offset1D,Offset2D);
    case ImagePalette:
	     {
	     DWORD b = Raster->GetValue2D(Offset1D,Offset2D);
	     return Palette->GetValue1D(b);
	     }
    default: {
	     DWORD b=Raster->GetValue2D(Offset1D,Offset2D);
	     switch(Raster->GetPlanes())
	     {
	       case  1: b *= 0xFF; break;
	       case  2: b *= 85; break;		// 255/3 = 85  -> 85*3=0xFF
	       case  4: b *= 17; break;		// 255/15=17   -> 17*15=0xFF
	       case 16: b >>= 8; break;
	       case 24: b >>= 16; break;
	       case 32: b >>= 24; break;
	     }
	     return b | 0x100*b | 0x10000*b;
	     }
    }
  return 0;
}


void Image::AttachRaster(Raster2DAbstract *NewRaster)
{
  if(Raster==NewRaster) return;		// Raster is already attached
  if(Raster!=NULL)			// Detach previously attached raster
   {
   if(Raster->UsageCount--<=1) delete Raster;
   Raster=NULL;
   }
  if(NewRaster!=NULL)			// Attach raster now
   {
   Raster=NewRaster; Raster->UsageCount++;
   }
}


void Image::AttachPalette(APalette *NewPalette)
{
  if(Palette==NewPalette) return;		// Raster is already attached
  if(Palette!=NULL)			// Detach previously attached raster
   {
   if(Palette->UsageCount--<=1) delete Palette;
   Palette = NULL;
   }
  if(NewPalette!=NULL)			// Attach raster now
   {
   Palette=NewPalette; Palette->UsageCount++;
   }
}


void Image::AttachVecImg(VectorImage *NewVecImg)
{
  if(VecImage==NewVecImg) return;		// Raster is already attached
  if(VecImage!=NULL)			// Detach previously attached raster
   {
   if(VecImage->UsageCount--<=1) delete VecImage;
   VecImage = NULL;
   }
  if(NewVecImg!=NULL)			// Attach raster now
   {
   VecImage=NewVecImg; VecImage->UsageCount++;
   }
}


/* ---------------------------- */


void Flip1D(Raster1DAbstract *r1D)
{
  if(r1D==NULL) return;
  if(r1D->Data1D==NULL) return;
  const int p = labs(r1D->GetPlanes());

  switch(p)
  {
    case 4:  Flip4((BYTE*)r1D->GetRow(), r1D->GetSize1D());
	     return;
    case 8:  Flip8((BYTE*)r1D->GetRow(), r1D->GetSize1D());
	     return;
    case 16: Flip16((WORD*)r1D->GetRow(), r1D->GetSize1D());
	     return;
    case 24: Flip24((BYTE*)r1D->GetRow(), r1D->GetSize1D());
	     return;
    case 32: Flip32((DWORD*)r1D->GetRow(), r1D->GetSize1D());
	     return;
#ifdef QWORD
    case 64: Flip64((QWORD*)r1D->GetRow(), r1D->GetSize1D());
	     return;
#endif
  }

  unsigned x1 = 0;
  unsigned x2 = r1D->Size1D - 1;
  if(p>32)
  {    
    while(x1<x2)
    {
      double tmp = r1D->GetValue1Dd(x1);
      r1D->SetValue1Dd(x1, r1D->GetValue1Dd(x2));
      r1D->SetValue1Dd(x2, tmp);
      x1++;
      x2--;
    }
  }
  else
  {
    while(x1<x2)
    {
      DWORD tmp = r1D->GetValue1D(x1);
      r1D->SetValue1D(x1, r1D->GetValue1D(x2));
      r1D->SetValue1D(x2, tmp);
      x1++;
      x2--;
    }
  }
}


/** This procedure provides vertical flipping of the raster. */
void Flip2D(Raster2DAbstract *r2D)
{
void *ptr, **P1, **P2;

 if(r2D==NULL) return;
 if(r2D->Data2D==NULL) return;
 P1 = r2D->Data2D;
 P2 = P1 + (r2D->Size2D-1); 
 while(P2>P1)
   {
   ptr = *P1;
   *P1 = *P2;
   *P2 = ptr;
   P1++; P2--;
   }
}


/** This procedure provides flipping along 1st dimension (usually horizontal)
 * of the raster. */
void Flip1D(Raster2DAbstract *r2D)
{
unsigned y;
int p;

  if(r2D==NULL) return;
  if(r2D->Data2D==NULL) return;

  p = labs(r2D->GetPlanes());

  if(p<8 && p!=4)	// Speedup for <4 bits
    {
    Raster1D_8Bit Buff;
    Buff.Allocate1D(r2D->GetSize1D());
    if(Buff.Data1D==NULL) return;	// Report error here.

    for(y=0; y<r2D->Size2D; y++)
      {
      r2D->GetRowRaster(y)->Get(Buff);
      Flip8((BYTE*)Buff.Data1D, r2D->GetSize1D());
      r2D->GetRowRaster(y)->Set(Buff);
      }
    Buff.Erase();
    return;
    }

  for(y=0; y<r2D->Size2D; y++)
    {
    switch(p)
      {
      case 4:  Flip4((BYTE*)r2D->GetRow(y), r2D->GetSize1D());
	       break;
      case 8:  Flip8((BYTE*)r2D->GetRow(y), r2D->GetSize1D());
	       break;
      case 16: Flip16((WORD*)r2D->GetRow(y), r2D->GetSize1D());
	       break;
      case 24: Flip24((BYTE*)r2D->GetRow(y), r2D->GetSize1D());
	       break;
      case 32: Flip32((DWORD*)r2D->GetRow(y), r2D->GetSize1D());
	       break;
#ifdef QWORD
      case 64: Flip64((QWORD*)r2D->GetRow(y), r2D->GetSize1D());
	       break;
#endif
      default: {
                 unsigned x1 = 0;
	         unsigned x2 = r2D->Size1D - 1;
	         while(x1<x2)
		 {
		   double tmp = r2D->GetValue2Dd(x1,y);
		   r2D->SetValue2Dd(x1,y, r2D->GetValue2Dd(x2,y));
		   r2D->SetValue2Dd(x2,y, tmp);
		   x1++;
		   x2--;
		 }
               }
      }
   }
}


#if defined(RASTER_3D)

/** This procedure provides flipping along 1st dimension of the raster. */
void Flip1D(Raster3DAbstract *r3D)
{
unsigned y, z;

  if(r3D==NULL) return;
  if(r3D->Size1D<=1 || r3D->Data3D==NULL) return;

  const int p = labs(r3D->GetPlanes());
  if(p<8 && p!=4)	// Speedup for less than 8 bits
    {
    Raster1D_8Bit Buff;
    Buff.Allocate1D(r3D->GetSize1D());
    if(Buff.Data1D==NULL) return;

    for(z=0; z<r3D->Size3D; z++)
      for(y=0; y<r3D->Size2D; y++)
	{
	r3D->GetRowRaster(y,z)->Get(Buff);
	Flip8((BYTE*)Buff.Data1D, r3D->GetSize1D());
	r3D->GetRowRaster(y,z)->Set(Buff);
	}
    Buff.Erase();
    return;
    }

  for(z=0; z<r3D->Size3D; z++)
  {
    void **Row2D = r3D->GetRow(z);
    for(y=0; y<r3D->Size2D; y++)
    {
    switch(p)
      {
      case 4:  Flip4((BYTE*)Row2D[y], r3D->GetSize1D());
	       break;
      case 8:  Flip8((BYTE*)Row2D[y], r3D->GetSize1D());
	       break;
      case 16: Flip16((WORD*)Row2D[y], r3D->GetSize1D());
	       break;
      case 24: Flip24((BYTE*)Row2D[y], r3D->GetSize1D());
	       break;
      case 32: Flip32((DWORD*)Row2D[y], r3D->GetSize1D());
	       break;
#ifdef QWORD
      case 64: Flip64((QWORD*)Row2D[y], r3D->GetSize1D());
	       break;
#endif
      default: unsigned x1 = 0;
	       unsigned x2 = r3D->Size1D - 1;
               
	       while(x1<x2)
		 {
		 double tmp = r3D->GetValue3Dd(x1,y,z);
		 r3D->SetValue3Dd(x1,y,z, r3D->GetValue3Dd(x2,y,z));
		 r3D->SetValue3Dd(x2,y,z, tmp);
		 x1++;
		 x2--;
		 }                
       }
    }
 }
}


/** This procedure provides flipping along 2nd dimension of the raster. */
void Flip2D(Raster3DAbstract *r3D)
{
void *ptr, **P1, **P2;

 if(r3D==NULL) return;
 if(r3D->Size2D==0 || r3D->Data3D==NULL) return;

 for(unsigned z=0; z<r3D->Size3D; z++)
 {
   P1 = r3D->Data3D[z];
   P2 = r3D->Data3D[z] + (r3D->Size2D-1);
   while(P2>P1)
     {
     ptr = *P1;
     *P1 = *P2;
     *P2 = ptr;
     P1++; P2--;
     }
 }
}


/** This procedure provides flipping along 3rd dimension of the raster. */
void Flip3D(Raster3DAbstract *r3D)
{
void **ptr, ***P1, ***P2;

 if(r3D==NULL) return;
 if(r3D->Size3D==0 || r3D->Data3D==NULL) return;

 P1 = r3D->Data3D;
 P2 = r3D->Data3D + (r3D->Size3D-1);
 while(P2>P1)
   {
   ptr = *P1;
   *P1 = *P2;
   *P2 = ptr;
   P1++; P2--;
   }
}

#endif

//////////////////////////////////////////////////////////



class Palette_8bit: virtual public APalette, virtual public Raster1D_8BitRGB
	{
public:	Palette_8bit(int Indices): Raster1D_8BitRGB(Indices) {};
	Palette_8bit(void): Raster1D_8BitRGB() {};

	virtual void Get(unsigned index, RGBQuad *RGB);
	};

class Palette_16bit: virtual public APalette, virtual public Raster1D_16BitRGB
	{
public:	Palette_16bit(int Indices): Raster1D_16BitRGB(Indices) {};
	Palette_16bit(void) {};

	virtual void Get(unsigned index, RGBQuad *RGB);
	};

void Palette_8bit::Get(unsigned index, RGBQuad *RGB)
{
BYTE *ptrb;

 if(RGB==NULL || Data1D==NULL || Size1D<index) return;
 index *= 3; 
 ptrb = ((BYTE *)Data1D)+index;
 RGB->R = *ptrb++;
 RGB->G = *ptrb++;
 RGB->B = *ptrb;
}


void Palette_16bit::Get(unsigned index, RGBQuad *RGB)
{
BYTE *ptrb;

 if(RGB==NULL || Data1D==NULL || Size1D<index) return;
 index *= 6; 
 ptrb = ((BYTE *)Data1D)+index;
 RGB->R = *ptrb;	ptrb += 2;
 RGB->G = *ptrb;	ptrb += 2;
 RGB->B = *ptrb;
}


void APalette::Get(unsigned index, RGBQuad *RGB)
{
 if(RGB==NULL) return;
 RGB->R=R(index);
 RGB->G=G(index);
 RGB->B=B(index);
}


APalette *BuildPalette(unsigned Indices, char type)
{
APalette *pal=NULL;

 switch(type)
	{
	case 0:
	case 8: pal = new Palette_8bit(Indices);  break;
	case 16:pal = new Palette_16bit(Indices); break;
	}
 if(pal)
   if(pal->GetSize1D()<Indices)
	{
	delete pal;
	return(NULL);
	}

return(pal);
}


/** Is this Gray scale palette? */
int GrayPalette(Raster1DAbstractRGB *Palette, int Planes)
{
unsigned i,j;
unsigned PalItems;

 if(Palette==NULL) return(1);
 if(Palette->Data1D==NULL ||
    Palette->Size1D==0) return(1);	//consider invalid palette to be gray

 PalItems = (1<<Planes);
 if(Planes!=0)
   if(PalItems > Palette->GetSize1D())
	{
	//PalItems=Palette->Size1D/3;
	return(0);	//insufficient palette size, ?? non gray ??
	}

 for(i=0;i<PalItems;i++)
	{
	j=i*255 / (PalItems-1);
	if( (Palette->R(i)!=j) ||
	    (Palette->G(i)!=j) ||
	    (Palette->B(i)!=j) ) return(0);
	}
 return(1);
}


/** Make this palette gray. */
int FillGray(Raster1DAbstractRGB *Palette, int Planes)
{
unsigned i,j;

 if(Palette==NULL) return(1);
 if(Planes!=0)
   if(3*(1<<Planes) != Palette->Size1D) return(0);

 if(Palette->Data1D==NULL) return(0);
 for(i=0;i<Palette->Size1D/3-1;i++)
   {
   j=i*255 / (Palette->Size1D/3-1);
   Palette->R(i,j);
   Palette->G(i,j);
   Palette->B(i,j);
   }
 return(1);
}


/** This fuction is intended to schrink image colors.
 * It can work in two modes: 
 *       1, 256 colors to 16 colors.
 *       2, True color to 256 colors.
 * @param[in,out]	Img	Image to be processed.
 * @param[in]		maxColors  Maximal color table size for true color images.
 * @return	0 - image was not changed; ColorCount - image has been reordered. */
int ReducePalette(Image *Img, unsigned maxColors)
{
unsigned x,y;
Raster1DAbstract *RowRas,*RowRas2;
APalette *paletettebuff;
unsigned CurrentColors = 0;
DWORD RGBval, RGBval2;
unsigned i;
Raster2DAbstract *Raster2;

  if(Img==NULL || maxColors<=0) return 0;
  if(Img->Raster==NULL) return 0;
  if(Img->ImageType()!=ImageTrueColor)
    {
    BYTE *Reindex;

    if(Img->Palette==NULL || 
       Img->Raster->GetPlanes()>8 || Img->Raster->GetPlanes()<=4) return 0;

    Reindex = (BYTE *)calloc(256,sizeof(BYTE));
    if(Reindex==NULL) return 0;

	/* Calculate a histogram */
    for(y=Img->Raster->Size2D; y>0; y--)
    {
      RowRas = Img->Raster->GetRowRaster(y-1);
      if(RowRas==NULL) continue;

      for(x=0; x<RowRas->GetSize1D(); x++)
        {
        Reindex[RowRas->GetValue1D(x)] = 1;
        }
    }

	/* Create remapping matrix. */
    y = 0;
    for(x=0; x<((unsigned)1<<labs(Img->Raster->GetPlanes())); x++)
      {
      if(Reindex[x])
        {
        Reindex[x] = y++;
        }
      }
    
    if(y<=16)		// We can schring 256 to 16 levels
      {
	/* Remap a raster. */
      Raster2 = CreateRaster2D(Img->Raster->Size1D,Img->Raster->Size2D,4);

      for(y=Img->Raster->Size2D;y>0;y--)   //fix for 32 bit planes
        {
        RowRas = Img->Raster->GetRowRaster(y-1);
        RowRas2 = Raster2->GetRowRaster(y-1);
        if(RowRas==NULL || RowRas2==NULL) continue;

        for(x=0;x<RowRas->GetSize1D();x++)
	  {
          RowRas2->SetValue1D(x, Reindex[RowRas->GetValue1D(x)]);
	  }
        }

      if(--Img->Raster->UsageCount <= 0)
	delete Img->Raster;
      Img->Raster = Raster2;
      Img->Raster->UsageCount++;
      

	/* Remap a palette. */
      if((paletettebuff=BuildPalette(maxColors,8))!=NULL)
        {
        for(x=0;x<Img->Palette->GetSize1D();x++)
          {
          if(x==0 || Reindex[x])
            {
            paletettebuff->SetValue1D(Reindex[x], Img->Palette->GetValue1D(x));
            }
          }
        
          Img->AttachPalette(paletettebuff);
          paletettebuff = NULL;
        }
      }  

    free(Reindex);
    return 0;
    }

	/* True color reduction process. */
  //printf("\n---- Image check starts %d ------", maxColors);

  if((paletettebuff=BuildPalette(maxColors,8))==NULL) return 0;

  for(y=Img->Raster->Size2D;y>0;y--)   //fix for 32 bit planes
    {
    RowRas = Img->Raster->GetRowRaster(y-1);
    if(RowRas==NULL) continue;
    for(x=0;x<RowRas->GetSize1D();x++)
      {
      RGBval = RowRas->GetValue1D(x);

      i=0;
      while(i<CurrentColors)
	{
	RGBval2 = paletettebuff->GetValue1D(i);
	if(RGBval2 == RGBval) goto Found;

	if(RGBval2>RGBval)		// keep palette sorted
	  {
	  while(i<CurrentColors)
	    {
	    paletettebuff->SetValue1D(i,RGBval);
	    RGBval = RGBval2;
	    i++;
	    RGBval2 = paletettebuff->GetValue1D(i);
	    }
	  break;
	  }

	i++;
	}
	// Insert last value and extend palette
      paletettebuff->SetValue1D(CurrentColors++,RGBval);
      if(CurrentColors>=maxColors) goto Finish;
Found: {}
      }
    }

  if(CurrentColors<=256)
    {
    if(CurrentColors<=16)
      Raster2 = CreateRaster2D(Img->Raster->Size1D,Img->Raster->Size2D,4);
    else
      Raster2 = CreateRaster2D(Img->Raster->Size1D,Img->Raster->Size2D,8);
    if(Raster2->Data1D==NULL) goto Finish;

      {
      APalette *paletettebuff2 = BuildPalette(1<<Raster2->GetPlanes());
      if(paletettebuff2==NULL) goto Finish;
      Img->AttachPalette(paletettebuff2);
      }

    for(i=0; i<Img->Palette->GetSize1D(); i++)
      {
	Img->Palette->SetValue1D(i, paletettebuff->GetValue1D(i));
      }

    for(y=Img->Raster->Size2D;y>0;y--)   //fix for 32 bit planes
      {
      RowRas = Img->Raster->GetRowRaster(y-1);
      RowRas2 = Raster2->GetRowRaster(y-1);
      if(RowRas==NULL || RowRas2==NULL) continue;

      for(x=0;x<RowRas->GetSize1D();x++)
	{
	RGBval = RowRas->GetValue1D(x);

	for(i=0; i<CurrentColors; i++)
	  {
	  if(paletettebuff->GetValue1D(i) == RGBval)
	    {
	    RowRas2->SetValue1D(x,i);	// index has been found
	    break;
	    }
	  }
	}
      }

    Img->AttachRaster(Raster2);
    //printf("---- Image reduced %d ------", CurrentColors);
    }

Finish:
  if(paletettebuff) {delete paletettebuff;paletettebuff=NULL;}
  //printf("---- Image check ended<<<");

return CurrentColors;
}
