/*
 * (C)opyright MMIV-MMV Anselm R. Garbe <garbeam at gmail dot com>
 * See LICENSE file for license details.
 */

#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <time.h>
#include <X11/Xatom.h>
#include <X11/cursorfont.h>
#include <X11/Xproto.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>

#include "wmii.h"

/* array indexes for file pointers */
typedef enum { M_CTL,
    M_SIZE,
    M_PRE_COMMAND,
    M_COMMAND,
    M_HISTORY,
    M_LOOKUP,
    M_TEXT_FONT,
    M_TEXT_SIZE,
    M_SELECTED_BG_COLOR,
    M_SELECTED_TEXT_COLOR,
    M_SELECTED_BORDER_COLOR,
    M_NORMAL_BG_COLOR,
    M_NORMAL_TEXT_COLOR,
    M_NORMAL_BORDER_COLOR,
    M_LAST
} InputIndexes;

IXPServer *ixps = 0;
Display *dpy;
GC gc;
Window win;
XRectangle rect;
XRectangle mrect;
int screen_num;
int lid = 0;
char sockfile[256];
File *files[M_LAST];
File **items = 0;
size_t item_size = 0;
int **offsets = 0;
int sel_item = 0;
int sel_offset = 0;
File **history = 0;
int sel_history = 0;
Pixmap pmap;

static void check_event(Connection * c);
static void draw_menu();
static void handle_kpress(XKeyEvent * e);
static void set_text(char *text);
static void quit(void *obj, char *arg);
static void display(void *obj, char *arg);
static int update_items(char *prefix);

Action acttbl[2] = {
    {"quit", quit},
    {"display", display}
};

char *version[] = {
    "wmimenu - window manager improved menu - " VERSION " (" DRAW ")\n"
        " (C)opyright MMIV-MMV Anselm R. Garbe\n", 0
};

static void
usage()
{
    fprintf(stderr, "%s",
            "usage: wmimenu [-s <socket file>] [-r] [-v] [<x>,<y>,<width>,<height>]\n"
            "      -s      socket file (default: /tmp/.ixp-$USER/wmimenu-%s-%s)\n"
            "      -v      version info\n");
    exit(1);
}

static void
add_history(char *cmd)
{
    char buf[MAX_BUF];
    snprintf(buf, MAX_BUF, "/history/%ld", (long) time(0));
    history = (File **) attach_item_begin((void **) history,
                                          wmii_create_ixpfile(ixps, buf,
                                                              cmd),
                                          sizeof(File *));
}

/* menu mode items ------------------------------------------------ */

static void
_exec(char *cmd)
{
    char *rc = cmd;

    if(!cmd || cmd[0] == '\0')
        return;

    add_history(cmd);
    if(items && (sel_item >= 0) && items[sel_item]
       && items[sel_item]->size)
        rc = cmd = items[sel_item]->content;
    if(files[M_PRE_COMMAND]->content) {
        size_t len = strlen(cmd) + files[M_PRE_COMMAND]->size + 2;
        rc = malloc(len);
        snprintf(rc, len, "%s %s", (char *) files[M_PRE_COMMAND]->content,
                 cmd);
    }
    /* fallback */
    spawn(dpy, rc);
    /* cleanup */
    if(files[M_PRE_COMMAND]->content)
        free(rc);
}

static void
quit(void *obj, char *arg)
{
    ixps->runlevel = SHUTDOWN;
}

static void
show()
{
    XMapRaised(dpy, win);
    XSync(dpy, False);
    update_items(files[M_COMMAND]->content);
    draw_menu();
    while(XGrabKeyboard
          (dpy, RootWindow(dpy, screen_num), True, GrabModeAsync,
           GrabModeAsync, CurrentTime) != GrabSuccess)
        usleep(1000);
}

static void
hide()
{
    XUngrabKeyboard(dpy, CurrentTime);
    set_text(0);
    XUnmapWindow(dpy, win);
    XSync(dpy, False);
}

static void
display(void *obj, char *arg)
{
    if(!arg)
        return;
    if(int_val(arg))
        show();
    else
        hide();
    check_event(0);
}

/* }}} */

/* {{{ buffer functions */

void
set_text(char *text)
{
    if(files[M_COMMAND]->content) {
        free(files[M_COMMAND]->content);
        files[M_COMMAND]->content = 0;
        files[M_COMMAND]->size = 0;
    }
    if(text && strlen(text)) {
        files[M_COMMAND]->content = strdup(text);
        files[M_COMMAND]->size = strlen(text);
    }
}

static int
update_items(char *pattern)
{
    size_t plen = pattern ? strlen(pattern) : 0;
    int matched = pattern ? plen == 0 : 1;
    File *f, *p;
    int x = 0, w, rest, *new, dist, avg_char_len;
    static const char *perf_hack_str =
        "52ns/_ sjsjs6TZHSAN&?={{]]((}}-!.,iPO";
    size_t size = 0;

    sel_item = -1;

    if(offsets)
        free(offsets);
    offsets = 0;
    sel_offset = 0;

    if(!files[M_LOOKUP]->content)
        return 0;
    f = ixp_walk(ixps, files[M_LOOKUP]->content);
    if(!f || !is_directory(f))
        return 0;

    /* build new items */
    for(p = f->content; p; p = p->next)
        size++;
    if(size > item_size) {
        /* stores always the biggest amount of items in memory */
        if(items)
            free((File **) items);
        items = 0;
        item_size = size;
        if(item_size)
            items = (File **) malloc((item_size + 1) * sizeof(File *));
    }
    size = 0;

    /* needed by offset calculation */
    dist = mrect.height / 4;
    /* performance hack */
    avg_char_len = text_width(dpy, files[M_TEXT_FONT]->content,
                              int_val(files[M_TEXT_SIZE]->content),
                              (unsigned char *) perf_hack_str)
        / strlen(perf_hack_str);

    x = rest = 2 * avg_char_len + 2 * dist;

    new = malloc(sizeof(int));
    *new = 0;
    offsets =
        (int **) attach_item_end((void **) offsets, new, sizeof(int *));

    /* build new items */
    for(p = f->content; p; p = p->next) {
        if(!p->content)
            continue;           /* ignore bogus files */
        if(matched || !strncmp(pattern, p->name, plen)) {
            items[size] = p;
            p->parent = 0;      /* HACK to prevent doubled items */

            w = strlen(p->name) * avg_char_len + dist;
            if(x + w >= mrect.width) {
                new = malloc(sizeof(int));
                *new = size;
                offsets =
                    (int **) attach_item_end((void **) offsets, new,
                                             sizeof(int *));
                x = rest;
            }
            x += w;
            size++;
        }
    }

    for(p = f->content; p; p = p->next) {
        if(!p->content)
            continue;           /* ignore bogus files */
        if(p->parent && strstr(p->name, pattern)) {
            items[size] = p;
            w = strlen(p->name) * avg_char_len + dist;
            if(x + w >= mrect.width) {
                new = malloc(sizeof(int));
                *new = size;
                offsets =
                    (int **) attach_item_end((void **) offsets, new,
                                             sizeof(int *));
                x = rest;
            }
            x += w;
            size++;
        } else
            p->parent = f;      /* restore HACK */
    }
    items[size] = 0;

    new = malloc(sizeof(int));
    *new = size;
    offsets =
        (int **) attach_item_end((void **) offsets, new, sizeof(int *));
    return size;
}

/* creates draw structs for menu mode drawing */
static void
draw_menu()
{
    Draw d = { 0 };
    int i, xoff = 0, w, dist;
    int prev_w, next_w;

    d.screen_num = screen_num;
    d.gc = gc;
    d.drawable = pmap;
    d.rect = mrect;
    d.rect.x = 0;
    d.rect.y = 0;
    str_to_color(&d.bg, files[M_NORMAL_BG_COLOR]->content);
    str_to_color4(d.border, files[M_NORMAL_BORDER_COLOR]->content);
    draw_label_noborder(dpy, &d);

    dist = mrect.height / 4;
    /* print command */
    d.rect.height /= 2;
    d.rect.y = d.rect.height;
    d.fnt.font = files[M_TEXT_FONT]->content;
    d.fnt.align = WEST;
    d.fnt.scale = (double) int_val(files[M_TEXT_SIZE]->content);
    str_to_color(&d.fg, files[M_SELECTED_TEXT_COLOR]->content);
    str_to_color(&d.bg, files[M_SELECTED_BG_COLOR]->content);
    d.text = files[M_COMMAND]->content;
    draw_label_noborder(dpy, &d);

    prev_w = text_width(dpy, files[M_TEXT_FONT]->content,
                        int_val(files[M_TEXT_SIZE]->content),
                        (unsigned char *) ">") + dist;
    next_w = text_width(dpy, files[M_TEXT_FONT]->content,
                        int_val(files[M_TEXT_SIZE]->content),
                        (unsigned char *) "<") + dist;
    d.rect.y = 0;
    d.fnt.align = CENTER;
    if(items && items[0] && offsets) {

        if(*offsets[sel_offset] != 0) {
            str_to_color(&d.bg, files[M_NORMAL_BG_COLOR]->content);
            str_to_color(&d.fg, files[M_NORMAL_TEXT_COLOR]->content);
            d.text = "<";
            d.rect.x = xoff;
            d.rect.width = prev_w;
            draw_label_noborder(dpy, &d);
            xoff += d.rect.width;
        }

        /* determine maximum items */
        for(i = *offsets[sel_offset]; i < *offsets[sel_offset + 1]; i++) {
            w = text_width(dpy, files[M_TEXT_FONT]->content,
                           int_val(files[M_TEXT_SIZE]->content),
                           (unsigned char *) items[i]->name) + dist;
            d.text = items[i]->name;
            d.rect.x = xoff;
            d.rect.width = w;
            if(i == sel_item) {
                str_to_color(&d.bg, files[M_SELECTED_BG_COLOR]->content);
                str_to_color(&d.fg, files[M_SELECTED_TEXT_COLOR]->content);
                str_to_color4(d.border,
                              files[M_SELECTED_BORDER_COLOR]->content);
                draw_label(dpy, &d);
            } else if(i == 0 && sel_item == -1) {
                str_to_color(&d.bg, files[M_SELECTED_TEXT_COLOR]->content);
                str_to_color(&d.fg, files[M_SELECTED_BG_COLOR]->content);
                str_to_color4(d.border,
                              files[M_SELECTED_BORDER_COLOR]->content);
                draw_label(dpy, &d);
            } else {
                str_to_color(&d.bg, files[M_NORMAL_BG_COLOR]->content);
                str_to_color(&d.fg, files[M_NORMAL_TEXT_COLOR]->content);
                str_to_color4(d.border,
                              files[M_NORMAL_BORDER_COLOR]->content);
                draw_label_noborder(dpy, &d);
            }
            xoff += w;
        }

        if(items[i] && items[i]->name) {
            str_to_color(&d.bg, files[M_NORMAL_BG_COLOR]->content);
            str_to_color(&d.fg, files[M_NORMAL_TEXT_COLOR]->content);
            d.text = ">";
            d.rect.x = xoff;
            d.rect.width = next_w;
            draw_label_noborder(dpy, &d);
        }
    }
    XCopyArea(dpy, pmap, win, gc, 0, 0, mrect.width, mrect.height, 0, 0);
    XSync(dpy, False);
}

static void
handle_kpress(XKeyEvent * e)
{
    KeySym ksym;
    char buf[32];
    int idx, num;
    static char text[4096];
    size_t len = 0;

    text[0] = '\0';
    if(files[M_COMMAND]->content) {
        STRLCPY(text, files[M_COMMAND]->content, sizeof(text));
        len = strlen(text);
    }
    buf[0] = '\0';
    num = XLookupString(e, buf, sizeof(buf), &ksym, 0);

    if(IsFunctionKey(ksym) || IsKeypadKey(ksym)
       || IsMiscFunctionKey(ksym) || IsPFKey(ksym)
       || IsPrivateKeypadKey(ksym))
        return;

    /* first check if a control mask is omitted */
    if(e->state & ShiftMask) {
        if(ksym == XK_ISO_Left_Tab)
            ksym = XK_Left;
    } else if(e->state & ControlMask) {
        switch (ksym) {
        case XK_E:
        case XK_e:
            ksym = XK_End;
            break;
        case XK_H:
        case XK_h:
            ksym = XK_BackSpace;
            break;
        case XK_J:
        case XK_j:
            ksym = XK_Return;
            break;
        case XK_U:
        case XK_u:
            set_text(0);
            update_items(0);
            draw_menu();
            return;
            break;
        default:               /* ignore other control sequences */
            return;
            break;
        }
    }

    switch (ksym) {
    case XK_Left:
        if(!items || !items[0])
            return;
        if(sel_item > 0) {
            sel_item--;
            set_text(items[sel_item]->name);
        } else
            return;
        break;
    case XK_Right:
    case XK_Tab:
        if(!items || !items[0])
            return;
        if(items[sel_item + 1]) {
            sel_item++;
            set_text(items[sel_item]->name);
        } else
            return;
        break;
    case XK_Down:
        if(history) {
            set_text(history[sel_history]->content);
            idx = index_prev_item((void **) history, history[sel_history]);
            if(idx >= 0)
                sel_history = idx;
        }
        update_items(files[M_COMMAND]->content);
        break;
    case XK_Up:
        if(history) {
            set_text(history[sel_history]->content);
            idx = index_next_item((void **) history, history[sel_history]);
            if(idx >= 0)
                sel_history = idx;
        }
        update_items(files[M_COMMAND]->content);
        break;
    case XK_Return:
        if(items && items[0]) {
            if(sel_item >= 0)
                _exec(items[sel_item]->name);
            else
                _exec(items[0]->name);
        } else if(text)
            _exec(text);
    case XK_Escape:
        hide();
        break;
    case XK_BackSpace:
        if(len) {
            int size = 0;
            size_t i = len;
            for(size = 0; items && items[size]; size++);
            if(i) {
                do
                    text[--i] = '\0';
                while(size && i && size == update_items(text));
            }
            set_text(text);
            update_items(files[M_COMMAND]->content);
        }
        break;
    default:
        if((num == 1) && !iscntrl((int) buf[0])) {
            buf[num] = '\0';
            if(len > 0)
                STRLCAT(text, buf, sizeof(text));
            else
                STRLCPY(text, buf, sizeof(text));
            set_text(text);
            update_items(files[M_COMMAND]->content);
        }
    }
    if(offsets) {
        if(sel_item == *offsets[sel_offset + 1])
            sel_offset++;
        else if((sel_offset > 1) && (sel_item + 1) == *offsets[sel_offset])
            sel_offset--;
    }
    draw_menu();
}

static void
check_event(Connection * c)
{
    XEvent ev;

    while(XPending(dpy)) {
        XNextEvent(dpy, &ev);
        switch (ev.type) {
        case KeyPress:
            handle_kpress(&ev.xkey);
            break;
        case Expose:
            if(ev.xexpose.count == 0) {
                draw_menu();
            }
            break;
        default:
            break;
        }
    }
}

static void
handle_after_write(IXPServer * s, File * f)
{
    int i;
    size_t len;

    if(files[M_CTL] == f) {
        for(i = 0; i < 2; i++) {
            len = strlen(acttbl[i].name);
            if(!strncmp(acttbl[i].name, (char *) f->content, len)) {
                if(strlen(f->content) > len) {
                    acttbl[i].func(0, &((char *) f->content)[len + 1]);
                } else {
                    acttbl[i].func(0, 0);
                }
                break;
            }
        }
    } else if(files[M_SIZE] == f) {
        char *size = files[M_SIZE]->content;
        if(size && strrchr(size, ',')) {
            str_to_rect(dpy, &rect, &mrect, size);
            XFreePixmap(dpy, pmap);
            XMoveResizeWindow(dpy, win, mrect.x, mrect.y,
                              mrect.width, mrect.height);
            XSync(dpy, False);
            pmap = XCreatePixmap(dpy, win, mrect.width, mrect.height,
                                 DefaultDepth(dpy, screen_num));
            XSync(dpy, False);
            draw_menu();
        }
    } else if(files[M_COMMAND] == f) {
        update_items(files[M_COMMAND]->content);
        draw_menu();
    }
    check_event(0);
}

static void
handle_before_read(IXPServer * s, File * f)
{
    char buf[64];
    if(f != files[M_SIZE])
        return;
    snprintf(buf, sizeof(buf), "%d,%d,%d,%d", mrect.x, mrect.y,
             mrect.width, mrect.height);
    if(f->content)
        free(f->content);
    f->content = strdup(buf);
    f->size = strlen(buf);
}

static void
run(char *size)
{
    XSetWindowAttributes wa;
    XGCValues gcv;

    /* init */
    if(!(files[M_CTL] = ixp_create(ixps, "/ctl"))) {
        perror("wmimenu: cannot connect IXP server");
        exit(1);
    }
    files[M_CTL]->after_write = handle_after_write;
    files[M_SIZE] = ixp_create(ixps, "/size");
    files[M_SIZE]->before_read = handle_before_read;
    files[M_SIZE]->after_write = handle_after_write;
    files[M_PRE_COMMAND] = ixp_create(ixps, "/precmd");
    files[M_COMMAND] = ixp_create(ixps, "/cmd");
    files[M_COMMAND]->after_write = handle_after_write;
    files[M_HISTORY] = ixp_create(ixps, "/history");
    add_history("");
    files[M_LOOKUP] = ixp_create(ixps, "/lookup");
    files[M_SELECTED_BG_COLOR] = wmii_create_ixpfile(ixps,
                                                     "/sel-style/bg-color",
                                                     DEFAULT_F_BG);
    files[M_TEXT_FONT] =
        wmii_create_ixpfile(ixps, "/style/text-font", DEFAULT_FT_FAM);
    files[M_TEXT_SIZE] =
        wmii_create_ixpfile(ixps, "/style/text-size", DEFAULT_FT_SIZE);
    files[M_SELECTED_TEXT_COLOR] =
        wmii_create_ixpfile(ixps, "/sel-style/text-color", DEFAULT_F_TEXT);
    files[M_SELECTED_BORDER_COLOR] =
        wmii_create_ixpfile(ixps, "/sel-style/border-color",
                            DEFAULT_F_BORDER);
    files[M_NORMAL_BG_COLOR] =
        wmii_create_ixpfile(ixps, "/norm-style/bg-color", DEFAULT_N_BG);
    files[M_NORMAL_TEXT_COLOR] =
        wmii_create_ixpfile(ixps, "/norm-style/text-color",
                            DEFAULT_N_TEXT);
    files[M_NORMAL_BORDER_COLOR] =
        wmii_create_ixpfile(ixps, "/norm-style/border-color",
                            DEFAULT_N_BORDER);

    wa.override_redirect = 1;
    wa.background_pixmap = ParentRelative;
    wa.event_mask = ExposureMask | ButtonPressMask | KeyPressMask
        | SubstructureRedirectMask | SubstructureNotifyMask;

    rect.x = rect.y = 0;
    rect.width = DisplayWidth(dpy, screen_num);
    rect.height = DisplayHeight(dpy, screen_num);
    str_to_rect(dpy, &rect, &mrect, size);
    if(!mrect.width) {
        mrect.width = DisplayWidth(dpy, screen_num);
    }
    if(!mrect.height) {
        mrect.height = 20;
    }

    win = XCreateWindow(dpy, RootWindow(dpy, screen_num), mrect.x, mrect.y,
                        mrect.width, mrect.height, 0, DefaultDepth(dpy,
                                                                   screen_num),
                        CopyFromParent, DefaultVisual(dpy, screen_num),
                        CWOverrideRedirect | CWBackPixmap | CWEventMask,
                        &wa);
    XDefineCursor(dpy, win, XCreateFontCursor(dpy, XC_xterm));
    XSync(dpy, False);

    /* window pixmap */
    gcv.function = GXcopy;
    gcv.graphics_exposures = False;

    gc = XCreateGC(dpy, win, 0, 0);
    pmap =
        XCreatePixmap(dpy, win, mrect.width, mrect.height,
                      DefaultDepth(dpy, screen_num));

    /* main event loop */
    run_server_with_fd_support(ixps, ConnectionNumber(dpy),
                               check_event, 0);
    deinit_server(ixps);
    XFreePixmap(dpy, pmap);
    XFreeGC(dpy, gc);
    XCloseDisplay(dpy);
}

int
main(int argc, char *argv[])
{
    char size[64];
    int i;

    sockfile[0] = '\0';
    /* command line args */
    for(i = 1; (i < argc) && (argv[i][0] == '-'); i++) {
        switch (argv[i][1]) {
        case 'v':
            fprintf(stdout, "%s", version[0]);
            exit(0);
            break;
        case 's':
            if(i + 1 < argc) {
                STRLCPY(sockfile, argv[++i], sizeof(sockfile));
            } else {
                usage();
            }
            break;
        default:
            usage();
            break;
        }
    }

    dpy = XOpenDisplay(0);
    if(!dpy) {
        fprintf(stderr, "%s", "wmimenu: cannot open display\n");
        exit(1);
    }

    screen_num = DefaultScreen(dpy);

    size[0] = '\0';
    if(argc > i)
        STRLCPY(size, argv[i], sizeof(size));

    ixps = wmii_setup_server(sockfile, sizeof(sockfile), "wmimenu");
    items = 0;

    run(size);

    return 0;
}
