'''This class governs all visible, animateable game objects from ground tiles to the player.
The functionality is extended with the DynamicObject class.'''
import pygame
import os
import random

from math import *

from pygame.locals import *

from locals import *

import data

from animation import Animation

class VisibleObject:

  def __init__(self, screen, x = None, y = None):
    self.screen = screen
    self.animations = {}
    self.animations["default"] = Animation("object", "idle")
    self.current_animation = "default"
    self.image = self.animations[self.current_animation].update_and_get_image()
    self.rect = self.image.get_rect()
    self.x = x
    self.y = y
    if (self.x == None):
      self.x = SCREEN_WIDTH / 2
    if (self.y == None):
      self.y = SCREEN_HEIGHT / 2

    self.flipping = False
    self.flipcounter = 0
    self.flip_init_angle = 0
    self.flip_finished = False
    self.flip_direction = CLOCKWISE #The object will move clockwise

    self.orientation = RIGHT
    self.itemclass = "not_item"

    self.pickable = False

    self.dead = False

    return

  def update(self, level = None):
    self.flip_finished = False

    if self.animations[self.current_animation].finished and self.current_animation == "dying":
      self.dead = True

    if self.flipping:

      if self.flipcounter == 0:
        rela_x = self.x - PLAY_AREA_CENTER_X
        rela_y = self.y - PLAY_AREA_CENTER_Y
        self.rad = sqrt(rela_x**2 + rela_y**2)
        self.flip_init_angle = atan2(rela_y, rela_x)

      self.flipcounter += 1
      self.flip_angle = self.flipcounter * (pi * 0.5 / (FLIP_FRAMES + 1)) * self.flip_direction
      self.angle = self.flip_angle + self.flip_init_angle
      self.x = PLAY_AREA_CENTER_X + cos(self.angle) * self.rad
      self.y = PLAY_AREA_CENTER_Y + sin(self.angle) * self.rad

      if self.flipcounter > FLIP_FRAMES:
        self.flipcounter = 0
        self.flipping = False
        self.dx = 0
        self.dy = 0
        self.flip_finished = True
    return

  #Renders the object - also flips or rotates it visually according to the orientation.
  def render(self, surface = None, topleft = None, static_render = False):
    if (not static_render) or (self.image == None):
      self.image = self.animations[self.current_animation].update_and_get_image()
    if topleft == None:
      self.rect.centerx = int(self.x)
      self.rect.centery = int(self.y)
    else:
      self.rect.left = topleft[0]
      self.rect.top = topleft[1]

    self.orientation = self.get_orientation()

    drawsurface = self.screen
    if surface != None:
      drawsurface = surface
    if self.orientation == RIGHT:
      drawsurface.blit(self.image, self.rect)
    if self.orientation == LEFT:
      drawsurface.blit(pygame.transform.flip(self.image, True, False), self.rect)
    if self.orientation == UP:
      drawsurface.blit(pygame.transform.rotate(self.image, 90), self.rect)
    if self.orientation == DOWN:
      drawsurface.blit(pygame.transform.rotate(self.image, -90), self.rect)
    return

  def get_orientation(self):
    return RIGHT

  def flip(self, flip_direction = CLOCKWISE):
    if self.flipping:
      return
    else:
      self.flipping = True
      self.flip_direction = flip_direction
      return

  #Makes the object die - if the object has a death animation, it will be played first.
  def die(self):
    if self.animations.has_key("dying"):
      self.current_animation = "dying"
    else:
      self.dead = True
    return