/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "bookmarksettings.h"

#include <QSettings>

#include "data/bookmark.h"
#include "configdata.h"

BookmarkSettings::BookmarkSettings()
{
    openBookmarks();
}

BookmarkSettings::~BookmarkSettings()
{
}

Bookmark BookmarkSettings::bookmark( const QString& uuid ) const
{
    int index = findBookmark( uuid );
    if ( index >= 0 )
        return m_bookmarks.at( index );
    return Bookmark();
}

void BookmarkSettings::replaceBookmark( const QString& uuid, const Bookmark& bookmark )
{
    int index = findBookmark( uuid );
    if ( index >= 0 )
        m_bookmarks.removeAt( index );

    addBookmark( bookmark );
}

void BookmarkSettings::addBookmark( const Bookmark& bookmark )
{
    int index = findBookmark( bookmark.serverUuid() );
    if ( index >= 0 )
        m_bookmarks.removeAt( index );

    m_bookmarks.prepend( bookmark );

    while ( m_bookmarks.count() > 10 )
        m_bookmarks.removeLast();

    saveBookmarks();
}

void BookmarkSettings::deleteBookmark( const QString& uuid )
{
    int index = findBookmark( uuid );
    if ( index >= 0 ) {
        m_bookmarks.removeAt( index );

        saveBookmarks();
    }
}

int BookmarkSettings::findBookmark( const QString& uuid ) const
{
    for ( int i = 0; i < m_bookmarks.count(); i++ ) {
        if ( m_bookmarks.at( i ).serverUuid() == uuid )
            return i;
    }
    return -1;
}

void BookmarkSettings::openBookmarks()
{
    m_bookmarks.clear();

    QSettings* settings = configData->settings();

    settings->beginGroup( "Bookmarks" );

    int count = settings->value( "Count", 0 ).toInt();

    for ( int i = 0; i < count; i++ ) {
        QStringList list = settings->value( QString( "Item%1" ).arg( i + 1 ) ).toStringList();
        Bookmark bookmark = deserializeBookmark( list );
        m_bookmarks.append( bookmark );
    }

    settings->endGroup();
}

void BookmarkSettings::saveBookmarks()
{
    QSettings* settings = configData->settings();

    settings->remove( "Bookmarks" );

    settings->beginGroup( "Bookmarks" );

    settings->setValue( "Count", m_bookmarks.count() );

    for ( int i = 0; i < m_bookmarks.count(); i++ ) {
        QStringList list = serializeBookmark( m_bookmarks.at( i ) );
        settings->setValue( QString( "Item%1" ).arg( i + 1 ), list );
    }

    settings->endGroup();
}

QStringList BookmarkSettings::serializeBookmark( const Bookmark& bookmark )
{
    QStringList list;

    list.append( bookmark.serverName() );
    list.append( bookmark.serverUuid() );
    list.append( bookmark.url() );
    list.append( bookmark.login() );
    if ( !bookmark.password().isEmpty() )
        list.append( obscure( bookmark.password() ) );

    return list;
}

Bookmark BookmarkSettings::deserializeBookmark( const QStringList& list )
{
    QString name = list.at( 0 );
    QString uuid = list.at( 1 );
    QString url = list.at( 2 );
    QString login = list.at( 3 );
    QString password;
    if ( list.count() == 5 )
        password = obscure( list.at( 4 ) );

    return Bookmark( name, uuid, url, login, password );
}

QString BookmarkSettings::obscure( const QString& text )
{
    QString result;

    const QChar* unicode = text.unicode();

    for ( int i = 0; i < text.length(); i++ )
        result += ( unicode[ i ].unicode() < 0x20 ) ? unicode[ i ] : QChar( 0x1001F - unicode[ i ].unicode() );

    return result;
}
