/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "managefiltersdialog.h"

#include <QLayout>
#include <QLabel>
#include <QHeaderView>
#include <QPushButton>
#include <QListWidget>
#include <QDialogButtonBox>
#include <QMessageBox>
#include <QMenu>
#include <QAction>

#include "data/datamanager.h"
#include "models/columnconditionsettings.h"
#include "models/treeviewsettings.h"
#include "models/tablemodelshelper.h"
#include "dialogs/filterdialog.h"
#include "dialogs/savefilterdialog.h"
#include "widgets/gradientwidget.h"
#include "viewmanager.h"
#include "iconloader.h"

ManageFiltersDialog::ManageFiltersDialog( int folderId, QWidget* parent ) : QDialog( parent ),
    m_folderId( folderId )
{
    setWindowTitle( tr( "Manage Filters" ) );

    QVBoxLayout* topLayout = new QVBoxLayout( this );
    topLayout->setMargin( 0 );
    topLayout->setSpacing( 0 );

    QWidget* promptWidget = new GradientWidget( this );
    topLayout->addWidget( promptWidget );

    QHBoxLayout* promptLayout = new QHBoxLayout( promptWidget );
    promptLayout->setSpacing( 10 );

    QLabel* promptPixmap = new QLabel( promptWidget );
    promptPixmap->setPixmap( IconLoader::pixmap( "folder-filter", 22 ) );
    promptLayout->addWidget( promptPixmap, 0 );

    QLabel* promptLabel = new QLabel( promptWidget );
    promptLabel->setWordWrap( true );
    promptLabel->setMinimumWidth( 250 );
    promptLayout->addWidget( promptLabel, 1 );

    QFrame* separator = new QFrame( this );
    separator->setFrameStyle( QFrame::HLine | QFrame::Sunken );
    topLayout->addWidget( separator );

    QVBoxLayout* mainLayout = new QVBoxLayout();
    mainLayout->setMargin( 9 );
    mainLayout->setSpacing( 6 );
    topLayout->addLayout( mainLayout );

    m_list = new QListWidget( this );
    m_list->setContextMenuPolicy( Qt::CustomContextMenu ); 
    mainLayout->addWidget( m_list, 1 );

    connect( m_list, SIGNAL( itemActivated( QListWidgetItem* ) ), this, SLOT( itemActivated( QListWidgetItem* ) ) );
    connect( m_list, SIGNAL( customContextMenuRequested( const QPoint& ) ), this, SLOT( listContextMenu( const QPoint& ) ) );

    mainLayout->addSpacing( 10 );

    QDialogButtonBox* buttonBox = new QDialogButtonBox( QDialogButtonBox::Close, Qt::Horizontal, this );
    mainLayout->addWidget( buttonBox );

    buttonBox->button( QDialogButtonBox::Close )->setText( tr( "&Close" ) );

    connect( buttonBox, SIGNAL( rejected() ), this, SLOT( reject() ) );

    QPushButton* newButton = buttonBox->addButton( tr( "&New Filter..." ), QDialogButtonBox::ResetRole );
    connect( newButton, SIGNAL( clicked() ), this, SLOT( newFilter() ) );

    QAction* actionModify = new QAction( IconLoader::icon( "edit-modify" ), tr( "&Modify..." ), this );
    connect( actionModify, SIGNAL( triggered() ), this, SLOT( modify() ) ); 

    QAction* actionRename = new QAction( IconLoader::icon( "edit-rename" ), tr( "Re&name..." ), this );
    connect( actionRename, SIGNAL( triggered() ), this, SLOT( rename() ) ); 

    QAction* actionRemove = new QAction( IconLoader::icon( "edit-delete" ), tr( "&Remove" ), this );
    connect( actionRemove, SIGNAL( triggered() ), this, SLOT( remove() ) );

    m_menu = new QMenu( this );
    m_menu->addAction( actionModify );
    m_menu->addSeparator();
    m_menu->addAction( actionRename );
    m_menu->addAction( actionRemove );

    const FolderRow* folder = dataManager->folders()->find( folderId );
    int typeId = folder ? folder->typeId() : 0;

    const TypeRow* type = dataManager->types()->find( typeId );
    QString name = type ? type->name() : QString();

    promptLabel->setText( tr( "Manage filters for folders of type <b>%1</b>:" ).arg( name ) );
    promptLabel->setFixedHeight( promptLabel->heightForWidth( promptLabel->minimumWidth() ) );

    m_settings = new ColumnConditionSettings();
    m_settings->openIssueFilters( typeId );

    populateList();

    resize( 100, 300 );
}

ManageFiltersDialog::~ManageFiltersDialog()
{
    delete m_settings;
}

void ManageFiltersDialog::populateList()
{
    m_list->clear();

    QStringList filters = m_settings->filterNames();

    filters.sort();

    for ( int i = 0; i < filters.count(); i++ ) {
        QListWidgetItem* item = new QListWidgetItem( m_list );
        item->setText( filters.at( i ) );
        item->setIcon( IconLoader::icon( "filter" ) );
    }
}

void ManageFiltersDialog::itemActivated( QListWidgetItem* item )
{
    modifyFilter( item->text() );
}

void ManageFiltersDialog::listContextMenu( const QPoint& pos )
{
    QModelIndex index = m_list->indexAt( pos );

    if ( index.isValid() ) {
        m_list->selectionModel()->setCurrentIndex( index, QItemSelectionModel::Current );
        m_list->selectionModel()->select( index, QItemSelectionModel::ClearAndSelect );

        m_menu->exec( m_list->viewport()->mapToGlobal( pos ) );
    }
}

void ManageFiltersDialog::newFilter()
{
    const FolderRow* folder = dataManager->folders()->find( m_folderId );
    int typeId = folder ? folder->typeId() : 0;
    int projectId = folder ? folder->projectId() : 0;

    const TypeRow* type = dataManager->types()->find( typeId );
    QString name = type ? type->name() : QString();

    FilterDialog dialog( projectId, this );

    dialog.setPrompt( tr( "Create a new filter for folders of type <b>%1</b>:" ).arg( name ) );

    TreeViewSettings viewSettings;
    viewSettings.openIssuesList( typeId );

    dialog.setAvailableColumns( viewSettings.availableColumns() );
    dialog.setConditions( QList<ColumnCondition>() );

    dialog.setFilterMode( true );
    dialog.setExistingFilters( m_settings->filterNames() );

    if ( dialog.exec() == QDialog::Accepted ) {
        m_settings->saveFilter( dialog.filterName(), dialog.conditions() );

        populateList();

        viewManager->postViewEvent( NULL, ViewEvent::UpdateFilters, typeId ); 
    }
}

void ManageFiltersDialog::modify()
{
    QListWidgetItem* item = m_list->currentItem();
    if ( item )
        modifyFilter( item->text() );
}

void ManageFiltersDialog::modifyFilter( const QString& name )
{
    const FolderRow* folder = dataManager->folders()->find( m_folderId );
    int typeId = folder ? folder->typeId() : 0;
    int projectId = folder ? folder->projectId() : 0;

    FilterDialog dialog( projectId, this );

    dialog.setPrompt( tr( "Modify filter <b>%1</b>:" ).arg( name ) );

    TreeViewSettings viewSettings;
    viewSettings.openIssuesList( typeId );

    dialog.setAvailableColumns( viewSettings.availableColumns() );
    dialog.setConditions( m_settings->loadFilter( name ) );

    dialog.setFilterMode( true );
    dialog.setFilterName( name );
    dialog.setExistingFilters( m_settings->filterNames() );

    if ( dialog.exec() == QDialog::Accepted ) {
        m_settings->saveFilter( dialog.filterName(), dialog.conditions() );

        viewManager->postViewEvent( NULL, ViewEvent::UpdateFilters, typeId ); 
    }
}

void ManageFiltersDialog::rename()
{
    QListWidgetItem* item = m_list->currentItem();
    if ( item ) {
        QString name = item->text();

        const FolderRow* folder = dataManager->folders()->find( m_folderId );
        int typeId = folder ? folder->typeId() : 0;

        SaveFilterDialog dialog( this );
        dialog.setWindowTitle( tr( "Rename Filter" ) );
        dialog.setFilterName( name );
        dialog.setExistingFilters( m_settings->filterNames() );

        if ( dialog.exec() == QDialog::Accepted && dialog.filterName() != name ) {
            m_settings->renameFilter( name, dialog.filterName() );

            populateList();

            viewManager->postViewEvent( NULL, ViewEvent::UpdateFilters, typeId );
        }
    }
}

void ManageFiltersDialog::remove()
{
    QListWidgetItem* item = m_list->currentItem();
    if ( item ) {
        QString name = item->text();

        const FolderRow* folder = dataManager->folders()->find( m_folderId );
        int typeId = folder ? folder->typeId() : 0;

        if ( QMessageBox::warning( this, tr( "Delete Filter" ),
            tr( "<qt>Are you sure you want to delete filter <b>%1</b>?</qt>" ).arg( name ),
            QMessageBox::Yes | QMessageBox::No ) == QMessageBox::No )
            return;

        m_settings->deleteFilter( name );

        populateList();

        viewManager->postViewEvent( NULL, ViewEvent::UpdateFilters, typeId );
    }
}
