/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef ISSUEDIALOGS_H
#define ISSUEDIALOGS_H

#include "commanddialog.h"

class QButtonGroup;
class QLineEdit;

class AbstractValueEditor;

/**
* Base class for dialogs for executing commands related to issues.
*/
class IssueDialog : public CommandDialog
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param parent The parent widget.
    */
    IssueDialog( QWidget* parent );

    /**
    * Destructor.
    */
    ~IssueDialog();

protected:
    /**
    * Return @c true if automatic notifications should be enabled for given issue.
    */
    bool issueAutoNotify( int issueId );

    /**
    * Return @c true if automatic notifications are enabled for given folder.
    */
    bool folderAutoNotify( int folderId );
};

/**
* Dialog for executing the <tt>ADD ISSUE</tt> command.
*/
class AddIssueDialog : public IssueDialog
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param folderId The identifier of the folder containing the issue.
    *
    * @param parent The parent widget.
    */
    AddIssueDialog( int folderId, QWidget* parent );

    /**
    * Destructor.
    */
    ~AddIssueDialog();

public:
    /**
    * Return the identifier of the created issue.
    */
    int issueId() const { return m_issueId; }

public: // overrides
    void accept();

protected: // overrides
    bool batchSuccessful( AbstractBatch* batch );

private:
    int m_folderId;
    int m_issueId;

    QLineEdit* m_nameEdit;
};

/**
* Dialog for executing the <tt>RENAME ISSUE</tt> command.
*/
class RenameIssueDialog : public IssueDialog
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param issueId Identifier of the issue to rename.
    *
    * @param parent The parent widget.
    */
    RenameIssueDialog( int issueId, QWidget* parent );

    /**
    * Destructor.
    */
    ~RenameIssueDialog();

public: // overrides
    void accept();

private:
    int m_issueId;
    QString m_oldName;

    QLineEdit* m_nameEdit;
};

/**
* Dialog for executing the <tt>SET VALUE</tt> command.
*/
class SetValueDialog : public IssueDialog
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param issueId Identifier of the issue to modify.
    *
    * @param attributeId Identifier of the attribute to modify.
    *
    * @param parent The parent widget.
    */
    SetValueDialog( int issueId, int attributeId, QWidget* parent );

    /**
    * Destructor.
    */
    ~SetValueDialog();

public: // overrides
    void accept();

private:
    int m_issueId;
    int m_attributeId;
    QString m_oldValue;

    QButtonGroup* m_group;

    AbstractValueEditor* m_editor;
};

/**
* Dialog for executing the <tt>ADD ATTACHMENT</tt> command.
*/
class AddAttachmentDialog : public IssueDialog
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param issueId Identifier of the issue containing the attachment.
    *
    * @param path Physical path of the file to attach.
    *
    * @param url User visible URL of the file to attach.
    *
    * @param parent The parent widget.
    */
    AddAttachmentDialog( int issueId, const QString& path, const QString& url, QWidget* parent );

    /**
    * Destructor.
    */
    ~AddAttachmentDialog();

public: // overrides
    void accept();

protected: // overrides
    bool batchSuccessful( AbstractBatch* batch );

private slots:
    void uploadProgress( int done );

private:
    int m_issueId;

    QString m_path;
    QString m_name;
    int m_size;

    QLineEdit* m_descriptionEdit;
};

/**
* Dialog for executing the <tt>GET ATTACHMENT</tt> command.
*/
class GetAttachmentDialog : public IssueDialog
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param attachmentId Identifier of the attachment to download.
    *
    * @param path Physical path of the file to attach.
    *
    * @param url User visible URL of the file to attach.
    *
    * @param parent The parent widget.
    */
    GetAttachmentDialog( int attachmentId, const QString& path, const QString& url, QWidget* parent );

    /**
    * Destructor.
    */
    ~GetAttachmentDialog();

public:
    /**
    * Start the download automatically.
    */
    void download();

public: // overrides
    void accept();

protected: // overrides
    bool batchSuccessful( AbstractBatch* batch );

private slots:
    void downloadProgress( int done );

private:
    int m_attachmentId;

    QString m_path;
    int m_size;
};

#endif
