/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "valueeditors.h"

// fix for GCC 4.3-snapshot
#include <climits>

#include <QVariant>
#include <QLineEdit>
#include <QComboBox>
#include <QSpinBox>
#include <QDoubleSpinBox>
#include <QDateEdit>
#include <QDateTimeEdit>

#if ( QT_VERSION >= 0x040400 )
#include <QCalendarWidget>
#endif

#if defined( Q_WS_WIN )
#include <qt_windows.h>
#else
#include <QLocale>
#endif

#include "data/datamanager.h"
#include "rdb/utilities.h"
#include "definitioninfo.h"
#include "datetimehelper.h"

TextValueEditor::TextValueEditor( const DefinitionInfo& info, QObject* parent, QWidget* parentWidget ) :
    AbstractValueEditor( parent )
{
    QLineEdit* edit = new QLineEdit( parentWidget );

    QVariant maxLength = info.metadata( "max-length" );
    if ( maxLength.isValid() )
        edit->setMaxLength( maxLength.toInt() );
    else
        edit->setMaxLength( 80 );

    setWidget( edit );
}

TextValueEditor::~TextValueEditor()
{
}

void TextValueEditor::setValue( const QString& value )
{
    QLineEdit* edit = (QLineEdit*)widget();

    edit->setText( value );
}

QString TextValueEditor::value()
{
    QLineEdit* edit = (QLineEdit*)widget();

    QString text = edit->text().trimmed();
    edit->setText( text );

    return text;
}

EnumValueEditor::EnumValueEditor( const DefinitionInfo& info, QObject* parent, QWidget* parentWidget ) :
    AbstractValueEditor( parent )
{
    QVariant editable = info.metadata( "editable" );
    bool isEditable = editable.isValid() ? editable.toBool() : false;

    QComboBox* combo = new QComboBox( parentWidget );
    combo->setEditable( isEditable );

    QVariant items = info.metadata( "items" );
    combo->addItems( items.toStringList() );

    if ( isEditable )
        combo->lineEdit()->setMaxLength( 80 );

    setWidget( combo );
}

EnumValueEditor::~EnumValueEditor()
{
}

void EnumValueEditor::setValue( const QString& value )
{
    QComboBox* combo = (QComboBox*)widget();

    int index = combo->findText( value );

    if ( index >= 0 )
        combo->setCurrentIndex( index );

    if ( combo->isEditable() )
        combo->setEditText( value );
}

QString EnumValueEditor::value()
{
    QComboBox* combo = (QComboBox*)widget();

    QString text = combo->currentText().trimmed();

    if ( combo->isEditable() )
        combo->setEditText( text );

    return text;
}

NumericValueEditor::NumericValueEditor( const DefinitionInfo& info, QObject* parent, QWidget* parentWidget ) :
    AbstractValueEditor( parent )
{
    QVariant decimal = info.metadata( "decimal" );
    m_decimal = decimal.isValid() ? decimal.toInt() : 0;

    if ( m_decimal > 0 ) {
        double step = 1;
        for ( int i = 0; i < m_decimal; i++ )
            step *= 0.1;

        double range = 1e12 * step;

        double min;
        QVariant minValue = info.metadata( "min-value" );
        if ( minValue.isValid() ) {
            min = minValue.toDouble();
            if ( min < -range )
                min = -range;
        } else {
            min = -range;
        }

        double max;
        QVariant maxValue = info.metadata( "max-value" );
        if ( maxValue.isValid() ) {
            max = maxValue.toDouble();
            if ( max > range )
                max = range;
        } else {
            max = range;
        }

        QDoubleSpinBox* spin = new QDoubleSpinBox( parentWidget );
        spin->setDecimals( m_decimal );
        spin->setRange( min, max );
        spin->setSingleStep( step );

        setWidget( spin );
    } else {
        QVariant minValue = info.metadata( "min-value" );
        int min = minValue.isValid() ? minValue.toInt() : INT_MIN;

        QVariant maxValue = info.metadata( "max-value" );
        int max = maxValue.isValid() ? maxValue.toInt() : INT_MAX;

        QSpinBox* spin = new QSpinBox( parentWidget );
        spin->setRange( min, max );

        setWidget( spin );
    }
}

NumericValueEditor::~NumericValueEditor()
{
}

void NumericValueEditor::setValue( const QString& value )
{
    if ( m_decimal > 0 ) {
        QDoubleSpinBox* spin = (QDoubleSpinBox*)widget();
        spin->setValue( value.toDouble() );
    } else {
        QSpinBox* spin = (QSpinBox*)widget();
        spin->setValue( value.toInt() );
    }
}

QString NumericValueEditor::value()
{
    if ( m_decimal > 0 ) {
        QDoubleSpinBox* spin = (QDoubleSpinBox*)widget();
        return QString::number( spin->value(), 'f', m_decimal );
    } else {
        QSpinBox* spin = (QSpinBox*)widget();
        return QString::number( spin->value() );
    }
}

#if ( QT_VERSION >= 0x040400 )

#if !defined( Q_WS_WIN )

// mapping QLocale::Country to Qt::DayOfWeek based on information from http://cldr.unicode.org/
static const char first_weekday[] = {
1, 6, 1, 6, 7, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 7, 1, 6, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 7, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 7, 1, 1, 1, 1, 1, 7, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 6, 1, 1, 1, 1, 
6, 1, 1, 6, 1, 6, 1, 7, 1, 1, 1, 1, 1, 1, 1, 1, 1, 7, 1, 1, 1, 1, 7, 1, 1, 7, 1, 1, 1, 1, 1, 1, 
1, 7, 1, 7, 7, 1, 6, 6, 7, 7, 1, 7, 7, 6, 1, 6, 1, 1, 7, 6, 7, 7, 1, 1, 1, 1, 6, 1, 1, 1, 7, 1, 
1, 1, 1, 5, 1, 7, 7, 1, 1, 1, 1, 1, 1, 1, 1, 7, 1, 6, 1, 1, 1, 1, 1, 1, 1, 1, 7, 1, 1, 1, 1, 1, 
7, 1, 6, 7, 1, 1, 1, 1, 1, 1, 7, 1, 1, 1, 1, 6, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 6, 1, 1, 1, 7, 1, 
1, 1, 6, 1, 1, 1, 1, 1, 1, 6, 1, 1, 1, 1, 1, 7, 7, 1, 1, 7, 1, 1, 1, 7, 6, 1, 1, 1, 1, 1, 1, 1, 
1, 7, 7, 1, 7, 1, 1, 1, 1, 1, 7, 1, 1, 6, 1, 1, 7, 1 
};

#endif

static Qt::DayOfWeek firstDayOfWeek()
{
#if defined( Q_WS_WIN )
    DWORD day = 0;
    GetLocaleInfoA( GetUserDefaultLCID(), LOCALE_IFIRSTDAYOFWEEK | LOCALE_RETURN_NUMBER, (char*)&day, sizeof( DWORD ) );
    return (Qt::DayOfWeek)( day + 1 );
#else
    QLocale::Country country = QLocale::system().country();
    if ( country > 0 && country < (int)sizeof( first_weekday ) )
        return (Qt::DayOfWeek)first_weekday[ (int)country ];
    return Qt::Monday;
#endif
}

#endif // ( QT_VERSION >= 0x040400 )

DateTimeValueEditor::DateTimeValueEditor( const DefinitionInfo& info, QObject* parent, QWidget* parentWidget ) :
    AbstractValueEditor( parent )
{
    QVariant time = info.metadata( "time" );
    m_time = time.isValid() ? time.toBool() : false;

    if ( m_time ) {
        QDateTimeEdit* dateTime = new QDateTimeEdit( parentWidget );
        dateTime->setDisplayFormat( "yyyy-MM-dd hh:mm" );
        dateTime->setCalendarPopup( true );
#if ( QT_VERSION >= 0x040400 )
        dateTime->calendarWidget()->setFirstDayOfWeek( firstDayOfWeek() );
#endif
        setWidget( dateTime );
    } else {
        QDateEdit* date = new QDateEdit( parentWidget );
        date->setDisplayFormat( "yyyy-MM-dd" );
        date->setCalendarPopup( true );
#if ( QT_VERSION >= 0x040400 )
        date->calendarWidget()->setFirstDayOfWeek( firstDayOfWeek() );
#endif
        setWidget( date );
    }

    setValue( QString() );
}

DateTimeValueEditor::~DateTimeValueEditor()
{
}

void DateTimeValueEditor::setValue( const QString& value )
{
    if ( m_time ) {
        QDateTime dateTime = DateTimeHelper::parseDateTime( value );
        if ( dateTime.isNull() )
            dateTime = QDateTime::currentDateTime();

        QDateTimeEdit* dtWidget = (QDateTimeEdit*)widget();
        dtWidget->setDateTime( dateTime );
    } else {
        QDate date = DateTimeHelper::parseDate( value );
        if ( date.isNull() )
            date = QDate::currentDate();

        QDateEdit* dtWidget = (QDateEdit*)widget();
        dtWidget->setDate( date );
    }
}

QString DateTimeValueEditor::value()
{
    if ( m_time ) {
        QDateTimeEdit* dtWidget = (QDateTimeEdit*)widget();
        return DateTimeHelper::formatDateTime( dtWidget->dateTime() );
    } else {
        QDateEdit* dtWidget = (QDateEdit*)widget();
        return DateTimeHelper::formatDate( dtWidget->date() );
    }
}

UserValueEditor::UserValueEditor( const DefinitionInfo& info, int projectId, QObject* parent, QWidget* parentWidget ) :
    AbstractValueEditor( parent )
{
    QComboBox* combo = new QComboBox( parentWidget );

    QVariant members = info.metadata( "members" );
    bool onlyMembers = ( members.isValid() ? members.toBool() : false ) && projectId != 0;

    RDB::IndexConstIterator<UserRow> it( dataManager->users()->index() );
    QList<const UserRow*> users = localeAwareSortRows( it, &UserRow::name );

    for ( int i = 0; i < users.count(); i++ ) {
        const UserRow* user = users.at( i );
        if ( user->access() == NoAccess )
            continue;
        if ( onlyMembers && !dataManager->members()->find( user->userId(), projectId ) )
            continue;
        combo->addItem( user->name() );
    }

    setWidget( combo );
}

UserValueEditor::~UserValueEditor()
{
}

void UserValueEditor::setValue( const QString& value )
{
    QComboBox* combo = (QComboBox*)widget();

    int index = combo->findText( value );

    if ( index >= 0 )
        combo->setCurrentIndex( index );
}

QString UserValueEditor::value()
{
    QComboBox* combo = (QComboBox*)widget();

    return combo->currentText();
}
