/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "issuehistoryprovider.h"

#include <QtAlgorithms>

#include "attributes/datetimehelper.h"
#include "data/datamanager.h"
#include "data/datarows.h"
#include "tablemodelshelper.h"
#include "linklocator.h"
#include "issuehistorywriter.h"

IssueHistoryProvider::IssueHistoryProvider() :
    m_issueId( 0 ),
    m_writer( NULL )
{
}

IssueHistoryProvider::~IssueHistoryProvider()
{
}

void IssueHistoryProvider::extractIssueHistory( int issueId )
{
    m_issueId = issueId;
    m_items.clear();

    RDB::ForeignConstIterator<CommentRow> itc( dataManager->comments()->parentIndex(), issueId );
    while ( itc.next() )
        m_items.append( itc.get()->commentId() );

    RDB::ForeignConstIterator<AttachmentRow> ita( dataManager->attachments()->parentIndex(), issueId );
    while ( ita.next() )
        m_items.append( ita.get()->attachmentId() );

    RDB::ForeignConstIterator<ChangeRow> itch( dataManager->changes()->parentIndex(), issueId );
    while ( itch.next() )
        m_items.append( itch.get()->changeId() );

    qSort( m_items );

    const IssueRow* issue = dataManager->issues()->find( issueId );
    m_initialName = issue ? issue->name() : QString();

    for ( int i = 0; i < m_items.count(); i++ ) {
        const ChangeRow* change = dataManager->changes()->find( m_items[ i ] );
        if ( change != NULL && change->attributeId() == 0 ) {
            m_initialName = change->oldValue();
            break;
        }
    }
}

void IssueHistoryProvider::write( IssueHistoryWriter* writer )
{
    m_writer = writer;

    writer->begin();

    const IssueRow* issue = dataManager->issues()->find( m_issueId );
    if ( !issue ) {
        writer->end();
        return;
    }

    int changeUser = issue->createdUser();
    QDateTime changeDate = issue->createdDate();

    QString changeHeader = headerCreated( changeUser, changeDate );

    QStringList changes;
    changes.append( itemInitialName( m_initialName ) );

    for ( int i = 0; i < m_items.count(); i++ ) {
        int id = m_items[ i ];

        const ChangeRow* change = dataManager->changes()->find( id );
        if ( change != NULL && changes.count() > 0 ) {
            if ( change->modifiedUser() == changeUser && changeDate.secsTo( change->modifiedDate() ) < 180 ) {
                changes.append( itemChange( change->attributeId(), change->oldValue(), change->newValue() ) );
                continue;
            }
        }

        if ( changes.count() > 0 ) {
            writer->writeChanges( changeHeader, changes );
            changes.clear();
        }

        if ( change != NULL ) {
            changeUser = change->modifiedUser();
            changeDate = change->modifiedDate();
            changeHeader = headerModified( changeUser, changeDate );
            changes.append( itemChange( change->attributeId(), change->oldValue(), change->newValue() ) );
            continue;
        }

        const CommentRow* comment = dataManager->comments()->find( id );
        if ( comment != NULL ) {
            QString header = headerComment( id, comment->createdUser(), comment->createdDate() );
            writer->writeComment( header, writer->comment( comment->text() ), id );
            continue;
        }

        const AttachmentRow* attachment = dataManager->attachments()->find( id );
        if ( attachment != NULL ) {
            QString header = headerAttachment( id, attachment->createdUser(), attachment->createdDate() );
            QString item = itemAttachment( id, attachment->name(), attachment->size(), attachment->description() );
            writer->writeAttachment( header, item, id );
            continue;
        }
    }

    if ( changes.count() > 0 )
        writer->writeChanges( changeHeader, changes );

    writer->end();
}

QString IssueHistoryProvider::headerCreated( int userId, const QDateTime& date )
{
    QString userArg = m_writer->name( TableModelsHelper::userName( userId ) );
    QString dateArg = m_writer->date( DateTimeHelper::formatDateTime( date ) );

    return tr( "Created by %1 on %2" ).arg( userArg, dateArg );
}

QString IssueHistoryProvider::headerModified( int userId, const QDateTime& date )
{
    QString userArg = m_writer->name( TableModelsHelper::userName( userId ) );
    QString dateArg = m_writer->date( DateTimeHelper::formatDateTime( date ) );

    return tr( "Modified by %1 on %2" ).arg( userArg, dateArg );
}

QString IssueHistoryProvider::headerComment( int commentId, int userId, const QDateTime& date )
{
    QString idArg = TableModelsHelper::formatId( commentId );
    QString userArg = m_writer->name( TableModelsHelper::userName( userId ) );
    QString dateArg = m_writer->date( DateTimeHelper::formatDateTime( date ) );

    return tr( "Comment %1 added by %2 on %3" ).arg( idArg, userArg, dateArg );
}

QString IssueHistoryProvider::headerAttachment( int attachmentId, int userId, const QDateTime& date )
{
    QString idArg = TableModelsHelper::formatId( attachmentId );
    QString userArg = m_writer->name( TableModelsHelper::userName( userId ) );
    QString dateArg = m_writer->date( DateTimeHelper::formatDateTime( date ) );

    return tr( "Attachment %1 added by %2 on %3" ).arg( idArg, userArg, dateArg );
}

QString IssueHistoryProvider::itemInitialName( const QString& name )
{
    QString nameArg = m_writer->value( name );

    return tr( "Item named \"%1\"" ).arg( nameArg );
}

QString IssueHistoryProvider::itemChange( int attributeId, const QString& oldValue, const QString& newValue )
{
    QString oldValueArg = m_writer->value( oldValue );
    QString newValueArg = m_writer->value( newValue );

    if ( attributeId == 0 )
        return tr( "Issue renamed from \"%1\" to \"%2\"" ).arg( oldValueArg, newValueArg );

    QString attributeArg = m_writer->name( TableModelsHelper::attributeName( attributeId ) );

    if ( oldValue.isEmpty() )
        return tr( "Attribute %1 changed from empty to \"%2\"" ).arg( attributeArg, newValueArg );

    if ( newValue.isEmpty() )
        return tr( "Attribute %1 changed from \"%2\" to empty" ).arg( attributeArg, oldValueArg );

    return tr( "Attribute %1 changed from \"%2\" to \"%3\"" ).arg( attributeArg, oldValueArg, newValueArg );
}

QString IssueHistoryProvider::itemAttachment( int attachmentId, const QString& name, int size, const QString& description )
{
    QString result = m_writer->attachment( attachmentId, name );

    result += " (" + TableModelsHelper::formatSize( size ) + ')';

    if ( !description.isEmpty() )
        result += " - " + m_writer->description( description );

    return result;
}
