/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "columnconditionsettings.h"

#include <QSettings>

#include "connectionmanager.h"
#include "columncondition.h"
#include "tablemodels.h"

ColumnConditionSettings::ColumnConditionSettings()
{
}

ColumnConditionSettings::~ColumnConditionSettings()
{
}

void ColumnConditionSettings::openIssueFilters( int typeId )
{
    m_group = QString( "IssueFilters/Type%1" ).arg( typeId );
    m_watchesGroup = QString( "FolderWatches/Type%1" ).arg( typeId );

    m_names.clear();

    QSettings* settings = connectionManager->serverSettings();

    if ( !settings->value( QString( "%1/Count" ).arg( m_group ) ).isValid() ) {
        createDefaultIssueFilters();
        return;
    }

    settings->beginGroup( m_group );

    int count = settings->value( "Count", 0 ).toInt();

    for ( int i = 0; i < count; i++ ) {
        QString name = settings->value( QString( "Name%1" ).arg( i + 1 ) ).toString();
        m_names.append( name );
    }

    settings->endGroup();
}

void ColumnConditionSettings::createDefaultIssueFilters()
{
    QList<ColumnCondition> myConditions;
    ColumnCondition myCondition;
    myCondition.setColumn( Column_CreatedBy );
    myCondition.setCondition( ColumnCondition::Check_IsMe );
    myConditions.append( myCondition );

    saveFilter( tr( "Created by Me" ), myConditions );

    QList<ColumnCondition> lastConditions;
    ColumnCondition lastCondition;
    lastCondition.setColumn( Column_ModifiedDate );
    lastCondition.setCondition( ColumnCondition::Age_NewerThan );
    lastCondition.setOperand( "86400" );
    lastConditions.append( lastCondition );

    saveFilter( tr( "Modified in Last 24 Hours" ), lastConditions );
}

bool ColumnConditionSettings::filterExists( const QString& name ) const
{
    return m_names.indexOf( name ) >= 0;
}

QList<ColumnCondition> ColumnConditionSettings::loadFilter( const QString& name )
{
    int index = m_names.indexOf( name );
    if ( index < 0 )
        return QList<ColumnCondition>();

    QSettings* settings = connectionManager->serverSettings();
    settings->beginGroup( m_group );

    QStringList serialized = settings->value( QString( "Conditions%1" ).arg( index + 1 ) ).toStringList();

    settings->endGroup();

    return deserializeConditions( serialized );
}

void ColumnConditionSettings::saveFilter( const QString& name, const QList<ColumnCondition>& conditions )
{
    QSettings* settings = connectionManager->serverSettings();
    settings->beginGroup( m_group );

    QStringList serialized = serializeConditions( conditions );

    int index = m_names.indexOf( name );

    if ( index >= 0 ) {
        settings->setValue( QString( "Conditions%1" ).arg( index + 1 ), serialized );
    } else {
        m_names.append( name );

        int count = m_names.count();
        settings->setValue( "Count", count );

        settings->setValue( QString( "Name%1" ).arg( count ), name );
        settings->setValue( QString( "Conditions%1" ).arg( count ), serialized );
    }

    settings->endGroup();
}

void ColumnConditionSettings::renameFilter( const QString& oldName, const QString& newName )
{
    int oldIndex = m_names.indexOf( oldName );
    if ( oldIndex < 0 )
        return;

    int newIndex = m_names.indexOf( newName );
    if ( newIndex == oldIndex )
        return;

    QSettings* settings = connectionManager->serverSettings();
    settings->beginGroup( m_group );

    if ( newIndex < 0 ) {
        m_names[ oldIndex ] = newName;
        settings->setValue( QString( "Name%1" ).arg( oldIndex + 1 ), newName );
    } else {
        copyConditions( settings, oldIndex, newIndex );
        deleteAt( settings, oldIndex );
    }

    settings->endGroup();
}

void ColumnConditionSettings::deleteFilter( const QString& name )
{
    int index = m_names.indexOf( name );
    if ( index < 0 )
        return ;

    QSettings* settings = connectionManager->serverSettings();
    settings->beginGroup( m_group );

    deleteAt( settings, index );

    settings->endGroup();
}

void ColumnConditionSettings::deleteAt( QSettings* settings, int index )
{
    m_names.removeAt( index );

    int count = m_names.count();
    settings->setValue( "Count", count );

    for ( int i = index; i < count; i++ ) {
        settings->setValue( QString( "Name%1" ).arg( i + 1 ), m_names[ i ] );
        copyConditions( settings, i + 1, i );
    }

    settings->remove( QString( "Name%1" ).arg( count + 1 ) );
    settings->remove( QString( "Conditions%1" ).arg( count + 1 ) );

    deleteWatches( settings, index );
}

void ColumnConditionSettings::copyConditions( QSettings* settings, int oldIndex, int newIndex )
{
    QVariant conditions = settings->value( QString( "Conditions%1" ).arg( oldIndex + 1 ) );
    settings->setValue( QString( "Conditions%1" ).arg( newIndex + 1 ), conditions );
}

QStringList ColumnConditionSettings::serializeConditions( const QList<ColumnCondition>& conditions )
{
    QStringList list;

    for ( int i = 0; i < conditions.count(); i++ ) {
        list.append( QString::number( conditions[ i ].column() ) );
        list.append( QString::number( conditions[ i ].condition() ) );
        list.append( conditions[ i ].operand() );
    }

    return list;
}

const QList<ColumnCondition> ColumnConditionSettings::deserializeConditions( const QStringList& list )
{
    QList<ColumnCondition> conditions;

    QStringList::const_iterator it = list.begin();
    while ( it != list.end() ) {
        ColumnCondition condition;
        condition.setColumn( ( *it++ ).toInt() );
        condition.setCondition( (ColumnCondition::Condition)( *it++ ).toInt() );
        if ( it != list.end() )
            condition.setOperand( *it++ );
        conditions.append( condition );
    }

    return conditions;
}

QStringList ColumnConditionSettings::loadWatches( int folderId )
{
    QSettings* settings = connectionManager->serverSettings();
    settings->beginGroup( m_watchesGroup );

    QStringList list = settings->value( QString( "Folder%1" ).arg( folderId ) ).toStringList();

    QStringList watches;

    for ( int i = 0; i < list.count(); i++ ) {
        int index = list[ i ].toInt();
        if ( index == 0 )
            watches.append( QString() );
        else
            watches.append( m_names[ index - 1 ] );
    }

    settings->endGroup();

    return watches;
}

void ColumnConditionSettings::saveWatches( int folderId, const QStringList& watches )
{
    QSettings* settings = connectionManager->serverSettings();
    settings->beginGroup( m_watchesGroup );

    QStringList list;

    for ( int i = 0; i < watches.count(); i++ ) {
        QString name = watches[ i ];
        if ( name.isEmpty() ) {
            list.append( "0" );
        } else {
            int index = m_names.indexOf( name );
            if ( index >= 0 )
                list.append( QString::number( index + 1 ) );
        }
    }

    if ( list.isEmpty() )
        settings->remove( QString( "Folder%1" ).arg( folderId ) );
    else
        settings->setValue( QString( "Folder%1" ).arg( folderId ), list );

    settings->endGroup();
}

void ColumnConditionSettings::deleteWatches( QSettings* settings, int filterIndex )
{
    QString oldGroup = settings->group();
    settings->endGroup();

    settings->beginGroup( m_watchesGroup );

    QStringList keys = settings->childKeys();

    for ( int i = 0; i < keys.count(); i++ ) {
        QStringList oldList = settings->value( keys[ i ] ).toStringList();
        QStringList newList;
        for ( int j = 0; j < oldList.count(); j++ ) {
            int index = oldList[ j ].toInt();
            if ( index < filterIndex + 1 )
                newList.append( oldList[ j ] );
            else if ( index > filterIndex + 1 )
                newList.append( QString::number( index - 1 ) );
        }
        if ( newList.isEmpty() )
            settings->remove( keys[ i ] );
        else
            settings->setValue( keys[ i ], newList );
    }

    settings->endGroup();
    settings->beginGroup( oldGroup );
}
