/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "columnconditionhelper.h"

#include "attributes/definitioninfo.h"
#include "attributes/definitionparser.h"
#include "attributes/valueeditorfactory.h"
#include "data/datamanager.h"
#include "data/datarows.h"
#include "filtervalueeditors.h"
#include "tablemodels.h"

DefinitionInfo ColumnConditionHelper::columnDefinition( int column )
{
    if ( column > Column_UserDefined ) {
        int attributeId = column - Column_UserDefined;

        const AttributeRow* attribute = dataManager->attributes()->find( attributeId );
        if ( !attribute )
            return DefinitionInfo();

        DefinitionParser parser;
        return parser.parse( attribute->definition() );
    }

    DefinitionInfo info;

    switch ( column ) {
        case Column_Name:
            info.setType( DefinitionInfo::Text );
            break;

        case Column_ID:
            info.setType( DefinitionInfo::Numeric );
            info.setMetadata( "min-value", 1 );
            break;

        case Column_CreatedDate:
        case Column_ModifiedDate:
            info.setType( DefinitionInfo::DateTime );
            info.setMetadata( "time", 1 );
            break;

        case Column_CreatedBy:
        case Column_ModifiedBy:
            info.setType( DefinitionInfo::User );
            info.setMetadata( "members", 1 );
            break;
    }

    info.setRequired( true );

    return info;
}

QList<ColumnCondition::Condition> ColumnConditionHelper::availableConditions( const DefinitionInfo& info )
{
    QList<ColumnCondition::Condition> conditions;

    switch ( info.type() ) {
        case DefinitionInfo::Text:
        case DefinitionInfo::Enum:
            conditions.append( ColumnCondition::String_Equals );
            conditions.append( ColumnCondition::String_NotEquals );
            if ( info.type() == DefinitionInfo::Text || info.metadata( "editable" ).toBool() ) {
                conditions.append( ColumnCondition::String_Contains );
                conditions.append( ColumnCondition::String_NotContains );
                conditions.append( ColumnCondition::String_StartsWith );
                conditions.append( ColumnCondition::String_NotStartsWith );
                conditions.append( ColumnCondition::String_EndsWith );
                conditions.append( ColumnCondition::String_NotEndsWith );
            }
            break;

        case DefinitionInfo::Numeric:
            conditions.append( ColumnCondition::Numeric_Equals );
            conditions.append( ColumnCondition::Numeric_NotEquals );
            conditions.append( ColumnCondition::Numeric_GreaterThan );
            conditions.append( ColumnCondition::Numeric_GreaterOrEqual );
            conditions.append( ColumnCondition::Numeric_LessThan );
            conditions.append( ColumnCondition::Numeric_LessOrEqual );
            break;

        case DefinitionInfo::DateTime:
            conditions.append( ColumnCondition::DateTime_Equals );
            conditions.append( ColumnCondition::DateTime_NotEquals );
            conditions.append( ColumnCondition::DateTime_Before );
            conditions.append( ColumnCondition::DateTime_After );
            conditions.append( ColumnCondition::Age_OlderThan );
            conditions.append( ColumnCondition::Age_NewerThan );
            break;

        case DefinitionInfo::User:
            conditions.append( ColumnCondition::String_Equals );
            conditions.append( ColumnCondition::String_NotEquals );
            conditions.append( ColumnCondition::Check_IsMe );
            conditions.append( ColumnCondition::Check_IsNotMe );
            break;

        default:
            break;
    }

    if ( !info.required() ) {
        conditions.append( ColumnCondition::Check_IsEmpty );
        conditions.append( ColumnCondition::Check_IsNotEmpty );
    }

    return conditions;
}

AbstractValueEditor* ColumnConditionHelper::createValueEditor( const DefinitionInfo& info, ColumnCondition::Condition condition,
                                                               int projectId, QObject* parent, QWidget* parentWidget )
{
    switch ( condition ) {
        case ColumnCondition::String_Contains:
        case ColumnCondition::String_NotContains:
        case ColumnCondition::String_StartsWith:
        case ColumnCondition::String_NotStartsWith:
        case ColumnCondition::String_EndsWith:
        case ColumnCondition::String_NotEndsWith:
            if ( info.type() != DefinitionInfo::Text ) {
                DefinitionInfo textInfo;
                textInfo.setType( DefinitionInfo::Text );
                return ValueEditorFactory::createValueEditor( textInfo, projectId, parent, parentWidget );
            }
            break;

        case ColumnCondition::DateTime_Equals:
        case ColumnCondition::DateTime_NotEquals:
            if ( info.metadata( "time" ).toBool() ) {
                DefinitionInfo dateInfo;
                dateInfo.setType( DefinitionInfo::DateTime );
                return ValueEditorFactory::createValueEditor( dateInfo, projectId, parent, parentWidget );
            }
            break;

        case ColumnCondition::Age_OlderThan:
        case ColumnCondition::Age_NewerThan:
            return new AgeValueEditor( parent, parentWidget );

        case ColumnCondition::Check_IsEmpty:
        case ColumnCondition::Check_IsNotEmpty:
        case ColumnCondition::Check_IsMe:
        case ColumnCondition::Check_IsNotMe:
            return NULL;

        default:
            break;
    }

    return ValueEditorFactory::createValueEditor( info, projectId, parent, parentWidget );
}
