/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "startpage.h"

#include <QLayout>
#include <QLabel>
#include <QListWidget>
#include <QPushButton>
#include <QAction>
#include <QMenu>
#include <QMessageBox>
#include <QUrl>

#include "data/bookmark.h"
#include "application.h"
#include "iconloader.h"

#if defined( Q_WS_WIN )
#include "windowsmodernstyle.h"
#endif

StartPage::StartPage( QWidget* parent ) : QWidget( parent )
{
    QVBoxLayout* topLayout = new QVBoxLayout( this );
    topLayout->setMargin( 0 );
    topLayout->setSpacing( 0 );

    QWidget* headerWidget = new QWidget( this );
#if defined( Q_WS_WIN ) && !defined( NO_STYLE_WINDOWSMODERN )
    if ( !qobject_cast<WindowsModernStyle*>( style() ) ) {
#else
    if ( true ) {
#endif
        headerWidget->setPalette( QPalette( QColor( 255, 255, 255 ) ) );
        headerWidget->setAutoFillBackground( true );
    }
    topLayout->addWidget( headerWidget );

    QHBoxLayout* headerLayout = new QHBoxLayout( headerWidget );
    headerLayout->setMargin( 5 );
    headerLayout->setSpacing( 6 );

    QLabel* pixmapLabel = new QLabel( headerWidget );
    pixmapLabel->setPixmap( IconLoader::pixmap( "webissues", 48 ) );
    headerLayout->addWidget( pixmapLabel );

    QVBoxLayout* captionsLayout = new QVBoxLayout();
    captionsLayout->setMargin( 0 );
    captionsLayout->setSpacing( 0 );
    headerLayout->addLayout( captionsLayout, 1 );

    QString version = ( (Application*)qApp )->version();
    QString title = tr( "WebIssues %1" ).arg( version );

    QLabel* titleLabel = new QLabel( title, headerWidget );
    titleLabel->setSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred );
    captionsLayout->addWidget( titleLabel );

    QFont titleFont;
    titleFont.setPointSize( 10 );
    titleFont.setBold( true );
    titleLabel->setFont( titleFont );

    QLabel* descriptionLabel = new QLabel( tr( "Issue tracking and team collaboration system" ), headerWidget );
    descriptionLabel->setSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred );
    captionsLayout->addWidget( descriptionLabel );

    QString manualUrl = ( (Application*)qApp )->manualIndex().toString();

    QString links;
    links += QLatin1String( "<a style=\"text-decoration: none\" href=\"http://webissues.mimec.org/\">webissues.mimec.org</a>" );
    links += QLatin1String( " &middot; " );
    links += QString( "<a style=\"text-decoration: none\" href=\"%1\">%2</a>" ).arg( manualUrl, tr( "Manual" ) );

    QLabel* linksLabel = new QLabel( links, headerWidget );
    linksLabel->setSizePolicy( QSizePolicy::Ignored, QSizePolicy::Preferred );
    linksLabel->setAlignment( Qt::AlignRight | Qt::AlignVCenter );
    linksLabel->setOpenExternalLinks( true );
    captionsLayout->addWidget( linksLabel );

    QFrame* separator = new QFrame( this );
    separator->setFrameStyle( QFrame::HLine | QFrame::Sunken );
    separator->setFixedHeight( 2 );
    topLayout->addWidget( separator );

    QWidget* mainWidget = new QWidget( this );
    mainWidget->setAutoFillBackground( true );
    topLayout->addWidget( mainWidget );

    QVBoxLayout* mainLayout = new QVBoxLayout( mainWidget );
    mainLayout->setMargin( 9 );
    mainLayout->setSpacing( 6 );

    QLabel* label = new QLabel( tr( "Recent connections:" ), mainWidget );
    mainLayout->addWidget( label );

    m_list = new QListWidget( mainWidget );
    m_list->setContextMenuPolicy( Qt::CustomContextMenu ); 
    mainLayout->addWidget( m_list );

    connect( m_list, SIGNAL( itemActivated( QListWidgetItem* ) ), this, SLOT( itemActivated( QListWidgetItem* ) ) );
    connect( m_list, SIGNAL( customContextMenuRequested( const QPoint& ) ), this, SLOT( listContextMenu( const QPoint& ) ) );

    QHBoxLayout* buttonsLayout = new QHBoxLayout();

    m_newButton = new QPushButton( tr( "&New Connection..." ), mainWidget );
    m_newButton->setIcon( IconLoader::icon( "connection-open" ) );
    buttonsLayout->addWidget( m_newButton );

    m_abortButton = new QPushButton( tr( "&Abort" ), mainWidget );
    m_abortButton->setVisible( false );
    buttonsLayout->addWidget( m_abortButton );

    buttonsLayout->addStretch( 1 );
    mainLayout->addLayout( buttonsLayout );

    QFrame* separator2 = new QFrame( this );
    separator2->setFrameStyle( QFrame::HLine | QFrame::Sunken );
    separator2->setFixedHeight( 2 );
    topLayout->addWidget( separator2 );

    connect( m_newButton, SIGNAL( clicked() ), this, SIGNAL( newConnection() ) );
    connect( m_abortButton, SIGNAL( clicked() ), this, SIGNAL( abort() ) );

    QAction* actionOpen = new QAction( IconLoader::icon( "connection-open" ), tr( "&Open Connection" ), this );
    connect( actionOpen, SIGNAL( triggered() ), this, SLOT( open() ) ); 

    QAction* actionEdit = new QAction( IconLoader::icon( "edit-modify" ), tr( "&Modify..." ), this );
    connect( actionEdit, SIGNAL( triggered() ), this, SLOT( edit() ) ); 

    QAction* actionRemove = new QAction( IconLoader::icon( "edit-delete" ), tr( "&Remove" ), this );
    connect( actionRemove, SIGNAL( triggered() ), this, SLOT( remove() ) );

    m_menu = new QMenu( this );
    m_menu->addAction( actionOpen );
    m_menu->addSeparator();
    m_menu->addAction( actionEdit );
    m_menu->addAction( actionRemove );

    populateList();

    setFocusProxy( m_list );
}

StartPage::~StartPage()
{
}

void StartPage::populateList()
{
    m_list->clear();

    for ( int i = 0; i < m_settings.bookmarks().count(); i++ ) {
        QListWidgetItem* item = new QListWidgetItem( m_list );
        const Bookmark& bookmark = m_settings.bookmarks().at( i );
        item->setText( QString( "%1 (%2)" ).arg( bookmark.serverName(), bookmark.url() ) );
        item->setIcon( IconLoader::icon( "connection" ) );
        item->setData( Qt::UserRole, QVariant( m_settings.bookmarks().at( i ).serverUuid() ) );
     }
}

void StartPage::itemActivated( QListWidgetItem* item )
{
    QString uuid = item->data( Qt::UserRole ).toString();
    emit openConnection( m_settings.bookmark( uuid ) );
}

void StartPage::listContextMenu( const QPoint& pos )
{
    QModelIndex index = m_list->indexAt( pos );
    if ( index.isValid() )
        m_menu->exec( m_list->viewport()->mapToGlobal( pos ) ); 
}

void StartPage::open()
{
    QListWidgetItem* item = m_list->currentItem();
    if ( item ) {
        QString uuid = item->data( Qt::UserRole ).toString();
        emit openConnection( m_settings.bookmark( uuid ) );
    }
}

void StartPage::edit()
{
    QListWidgetItem* item = m_list->currentItem();
    if ( item ) {
        QString uuid = item->data( Qt::UserRole ).toString();
        emit editConnection( m_settings.bookmark( uuid ) );
    }
}

void StartPage::remove()
{
    QListWidgetItem* item = m_list->currentItem();
    if ( item ) {
        QString uuid = item->data( Qt::UserRole ).toString();
        QString name = m_settings.bookmark( uuid ).serverName();

        if ( QMessageBox::warning( this, tr( "Delete Connection" ),
            tr( "<qt>Are you sure you want to delete connection <b>%1</b>?</qt>" ).arg( name ),
            QMessageBox::Yes | QMessageBox::No ) == QMessageBox::No )
            return;

        m_settings.deleteBookmark( uuid );

        populateList();
    }
}

void StartPage::setConnecting( bool connecting )
{
    m_list->setEnabled( !connecting );
    m_newButton->setEnabled( !connecting );
    m_abortButton->setVisible( connecting );
}
