/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "metadatadialogs.h"

#include <QLayout>
#include <QRadioButton>
#include <QButtonGroup>
#include <QLabel>
#include <QRegExp>
#include <QSpinBox>
#include <QLineEdit>
#include <QGroupBox>

#include "widgets/editlistbox.h"
#include "definitioninfo.h"

TextMetadataDialog::TextMetadataDialog( QWidget* parent ) : MetadataDialogBase( parent )
{
    setWindowTitle( tr( "Text Attribute" ) );

    QVBoxLayout* layout = new QVBoxLayout();

    QRadioButton* button1 = new QRadioButton( tr( "&No limit" ), this );
    layout->addWidget( button1 );

    QHBoxLayout* layout2 = new QHBoxLayout();
    layout->addLayout( layout2 );

    QRadioButton* button2 = new QRadioButton( tr( "&Maximum length:" ), this );
    layout2->addWidget( button2, 0 );

    m_lengthGroup = new QButtonGroup( this );
    m_lengthGroup->addButton( button1, 0 );
    m_lengthGroup->addButton( button2, 1 );

    m_lengthSpin = new QSpinBox( this );
    m_lengthSpin->setRange( 1, 80 );
    layout2->addWidget( m_lengthSpin, 1 );

    connect( button2, SIGNAL( toggled( bool ) ), m_lengthSpin, SLOT( setEnabled( bool ) ) );
    connect( button2, SIGNAL( clicked() ), m_lengthSpin, SLOT( setFocus() ) );

    setContentLayout( layout, true );

    button1->setFocus();
}

TextMetadataDialog::~TextMetadataDialog()
{
}

void TextMetadataDialog::updateWidgets( const DefinitionInfo& info )
{
    QVariant maxLength = info.metadata( "max-length" );
    if ( maxLength.isValid() ) {
        m_lengthGroup->button( 1 )->setChecked( true );
        m_lengthSpin->setValue( maxLength.toInt() );
    } else {
        m_lengthGroup->button( 0 )->setChecked( true );
        m_lengthSpin->setEnabled( false );
    }
}

bool TextMetadataDialog::updateMetadata( DefinitionInfo& info )
{
    QVariant maxLength;
    if ( m_lengthGroup->checkedId() == 1 )
        maxLength = m_lengthSpin->value();
    info.setMetadata( "max-length", maxLength );

    return true;
}

EnumMetadataDialog::EnumMetadataDialog( QWidget* parent ) : MetadataDialogBase( parent )
{
    setWindowTitle( tr( "Dropdown List Attribute" ) );

    QVBoxLayout* layout = new QVBoxLayout();

    QRadioButton* button1 = new QRadioButton( tr( "Allow o&nly values from the list" ), this );
    layout->addWidget( button1 );

    QRadioButton* button2 = new QRadioButton( tr( "Allow &entering custom values" ), this );
    layout->addWidget( button2 );

    m_editableGroup = new QButtonGroup( this );
    m_editableGroup->addButton( button1, 0 );
    m_editableGroup->addButton( button2, 1 );

    layout->addSpacing( 5 );

    QLabel* itemsLabel = new QLabel( tr( "Dropdown list items:" ), this );
    layout->addWidget( itemsLabel );

    m_itemsListBox = new EditListBox( this );
    m_itemsListBox->setMaxLength( 80 );
    layout->addWidget( m_itemsListBox );

    setContentLayout( layout, false );

    button1->setFocus();
}

EnumMetadataDialog::~EnumMetadataDialog()
{
}

void EnumMetadataDialog::updateWidgets( const DefinitionInfo& info )
{
    QVariant editable = info.metadata( "editable" );
    if ( editable.isValid() && editable.toBool() )
        m_editableGroup->button( 1 )->setChecked( true );
    else
        m_editableGroup->button( 0 )->setChecked( true );

    QVariant items = info.metadata( "items" );
    m_itemsListBox->setStringList( items.toStringList() );
}

bool EnumMetadataDialog::updateMetadata( DefinitionInfo& info )
{
    QStringList items = m_itemsListBox->stringList();

    if ( items.count() == 0 ) {
        showWarning( tr( "The dropdown list cannot be empty" ) );
        return false;
    }

    if ( m_editableGroup->checkedId() == 1 )
        info.setMetadata( "editable", QVariant( 1 ) );

    info.setMetadata( "items", QVariant( items ) );

    return true;
}

NumericMetadataDialog::NumericMetadataDialog( QWidget* parent ) : MetadataDialogBase( parent )
{
    setWindowTitle( tr( "Numeric Attribute" ) );

    QGridLayout* layout = new QGridLayout();

    QLabel* labelDecimal = new QLabel( tr( "&Decimal places:" ), this );
    layout->addWidget( labelDecimal, 0, 0 );

    m_decimalSpin = new QSpinBox( this );
    m_decimalSpin->setRange( 0, 6 );
    layout->addWidget( m_decimalSpin, 0, 1 );

    labelDecimal->setBuddy( m_decimalSpin );

    layout->setRowMinimumHeight( 1, 5 );

    QRadioButton* minButton1 = new QRadioButton( tr( "&No minimum value" ), this );
    layout->addWidget( minButton1, 2, 0, 1, 2 );

    QRadioButton* minButton2 = new QRadioButton( tr( "&Minimum value:" ), this );
    layout->addWidget( minButton2, 3, 0 );

    m_minGroup = new QButtonGroup( this );
    m_minGroup->addButton( minButton1, 0 );
    m_minGroup->addButton( minButton2, 1 );

    m_minEdit = new QLineEdit( this );
    layout->addWidget( m_minEdit, 3, 1 );

    connect( minButton2, SIGNAL( toggled( bool ) ), m_minEdit, SLOT( setEnabled( bool ) ) );
    connect( minButton2, SIGNAL( clicked() ), m_minEdit, SLOT( setFocus() ) );

    layout->setRowMinimumHeight( 4, 5 );

    QRadioButton* maxButton1 = new QRadioButton( tr( "No m&aximum value" ), this );
    layout->addWidget( maxButton1, 5, 0, 1, 2 );

    QRadioButton* maxButton2 = new QRadioButton( tr( "Ma&ximum value:" ), this );
    layout->addWidget( maxButton2, 6, 0 );

    m_maxGroup = new QButtonGroup( this );
    m_maxGroup->addButton( maxButton1, 0 );
    m_maxGroup->addButton( maxButton2, 1 );

    m_maxEdit = new QLineEdit( this );
    layout->addWidget( m_maxEdit, 6, 1 );

    connect( maxButton2, SIGNAL( toggled( bool ) ), m_maxEdit, SLOT( setEnabled( bool ) ) );
    connect( maxButton2, SIGNAL( clicked() ), m_maxEdit, SLOT( setFocus() ) );

    setContentLayout( layout, true );

    m_decimalSpin->setFocus();
}

NumericMetadataDialog::~NumericMetadataDialog()
{
}

void NumericMetadataDialog::updateWidgets( const DefinitionInfo& info )
{
    QVariant decimal = info.metadata( "decimal" );
    if ( decimal.isValid() )
        m_decimalSpin->setValue( decimal.toInt() );

    QVariant minValue = info.metadata( "min-value" );
    if ( minValue.isValid() ) {
        m_minGroup->button( 1 )->setChecked( true );
        m_minEdit->setText( minValue.toString() );
    } else {
        m_minGroup->button( 0 )->setChecked( true );
        m_minEdit->setEnabled( false );
    }

    QVariant maxValue = info.metadata( "max-value" );
    if ( maxValue.isValid() ) {
        m_maxGroup->button( 1 )->setChecked( true );
        m_maxEdit->setText( maxValue.toString() );
    } else {
        m_maxGroup->button( 0 )->setChecked( true );
        m_maxEdit->setEnabled( false );
    }
}

bool NumericMetadataDialog::updateMetadata( DefinitionInfo& info )
{
    int decimal = m_decimalSpin->value();
    info.setMetadata( "decimal", QVariant( decimal ) );

    QRegExp numberRegExp;
    if ( decimal == 0 )
        numberRegExp.setPattern( "-?\\d+" );
    else
        numberRegExp.setPattern( QString( "-?\\d+(\\.\\d{1,%1})?" ).arg( decimal ) );

    QString minValue;
    if ( m_minGroup->checkedId() == 1 ) {
        minValue = m_minEdit->text();
        if ( !numberRegExp.exactMatch( minValue ) ) {
            showWarning( tr( "Invalid minimum value" ) );
            return false;
        }
        info.setMetadata( "min-value", QVariant( minValue ) );
    }

    QString maxValue;
    if ( m_maxGroup->checkedId() == 1 ) {
        maxValue = m_maxEdit->text();
        if ( !numberRegExp.exactMatch( maxValue ) ) {
            showWarning( tr( "Invalid maximum value" ) );
            return false;
        }
        info.setMetadata( "max-value", QVariant( maxValue ) );
    }

    if ( !minValue.isEmpty() && !maxValue.isEmpty() && minValue.toDouble() > maxValue.toDouble() ) {
        showWarning( tr( "Invalid range" ) );
        return false;
    }

    return true;
}

DateTimeMetadataDialog::DateTimeMetadataDialog( QWidget* parent ) : MetadataDialogBase( parent )
{
    setWindowTitle( tr( "Date & Time Attribute" ) );

    QVBoxLayout* layout = new QVBoxLayout();

    QRadioButton* button1 = new QRadioButton( tr( "&Date only" ), this );
    layout->addWidget( button1 );

    QRadioButton* button2 = new QRadioButton( tr( "Date and &time" ), this );
    layout->addWidget( button2 );

    m_timeGroup = new QButtonGroup( this );
    m_timeGroup->addButton( button1, 0 );
    m_timeGroup->addButton( button2, 1 );

    setContentLayout( layout, true );

    button1->setFocus();
}

DateTimeMetadataDialog::~DateTimeMetadataDialog()
{
}

void DateTimeMetadataDialog::updateWidgets( const DefinitionInfo& info )
{
    QVariant time = info.metadata( "time" );
    if ( time.isValid() && time.toBool() == true )
        m_timeGroup->button( 1 )->setChecked( true );
    else
        m_timeGroup->button( 0 )->setChecked( true );
}

bool DateTimeMetadataDialog::updateMetadata( DefinitionInfo& info )
{
    if ( m_timeGroup->checkedId() == 1 )
        info.setMetadata( "time", QVariant( 1 ) );

    return true;
}

UserMetadataDialog::UserMetadataDialog( QWidget* parent ) : MetadataDialogBase( parent )
{
    setWindowTitle( tr( "User Attribute" ) );

    QVBoxLayout* layout = new QVBoxLayout();

    QRadioButton* button1 = new QRadioButton( tr( "&All users" ), this );
    layout->addWidget( button1 );

    QRadioButton* button2 = new QRadioButton( tr( "Only project &members" ), this );
    layout->addWidget( button2 );

    m_membersGroup = new QButtonGroup( this );
    m_membersGroup->addButton( button1, 0 );
    m_membersGroup->addButton( button2, 1 );

    setContentLayout( layout, true );

    button1->setFocus();
}

UserMetadataDialog::~UserMetadataDialog()
{
}

void UserMetadataDialog::updateWidgets( const DefinitionInfo& info )
{
    QVariant members = info.metadata( "members" );
    if ( members.isValid() && members.toBool() == true )
        m_membersGroup->button( 1 )->setChecked( true );
    else
        m_membersGroup->button( 0 )->setChecked( true );
}

bool UserMetadataDialog::updateMetadata( DefinitionInfo& info )
{
    if ( m_membersGroup->checkedId() == 1 )
        info.setMetadata( "members", QVariant( 1 ) );

    return true;
}
