#include <math.h>

#include <qregexp.h>
#include <qpoint.h>

#include "misc.h"

QString qpointToString (const QPoint &p)
{
	return "(" + QString("%1").arg(p.x()) +","+ QString ("%1").arg (p.y()) +")";
}

ostream &operator<< (ostream &stream, QPoint const &p)
{ 
    return (stream << qpointToString(p) );
}

float getAngle(const QPoint &p)
{	
	// Calculate angle of vector to y-axis
	if (p.y()==0)
	{
		if (p.x()>=0)
			return M_PI_2;
		else
			return 3* M_PI_2;
	} else
	{
		if (p.y()>0) 
			return (float)(M_PI  - atan ( (double)(p.x()) / (double)(p.y()) ) );
		else	
			if (p.x()<0)
				return (float)( 2*M_PI - atan ( (double)(p.x()) / (double)(p.y()) ) );
			else	
				return (float)( - atan ( (double)(p.x()) / (double)(p.y()) ) );
	}	
}

QPoint normalise(const QPoint &p)
{	
	// Calculate normalised position (fixed length) 

	double px=p.x();
	double py=p.y();
	double x;
	double y;
	double r=150;

	if (px==0)
	{
		x=0;
		if (py>=0)
			y=r;
		else
			y=-r;
	} else
	{
		double sign;
		double a;
		if (px>0) 
			sign=1; 
		else 
			sign=-1;
		
		a=atan (py / px);
		x=cos (a) * r *sign;
		y=sin (a) * r *sign;
	}	
	return QPoint ((int) (x),(int) (y));
}

QString maskPath(QString p)
{
	// Change " " to "\ " to enable blanks in filenames
	p=p.replace(QChar('&'),"\\&");
	return p.replace(QChar(' '),"\\ ");
}

QString convertToRel (const QString &src, const QString &dst)
{
	QString s=src;
	QString d=dst;
	int i;

	if (s==d) 
	{
		// Special case, we just need the name of the file,
		// not the complete path
		i=d.findRev ("/");
		d=d.right (d.length()-i-1);
	} else
	{
		// Find relative path from src to dst

		// Remove the first "/"
		if (s.section ("/",0,0).isEmpty()) 
		{
			s=s.right (s.length()-1);
			d=d.right (d.length()-1);
		}
		
		// remove identical left parts
		while (s.section("/",0,0) == d.section("/",0,0) ) 
		{
			i=s.find ("/");
			s=s.right (s.length()-i-1);
			d=d.right (d.length()-i-1);
		}

		int srcsep=s.contains("/");
		int dstsep=d.contains("/");
		if (srcsep >=  dstsep )
		{
			// find path to go up first and then back to dst
			i=1;
			while (i<=srcsep) 
			{
				d="../"+d;
				i++;
			}	
		}
	}	
	return d;
}

// returns masked "<" ">" "&"
QString quotemeta(const QString &s)
{
	QString r=s;
    QRegExp  rx("&(?!amp;)");
    r.replace ( rx,"&amp;");
    rx.setPattern( ">");
    r.replace ( rx,"&gt;");
    rx.setPattern( "<");
    r.replace ( rx,"&lt;");
    rx.setPattern( "\"");
    r.replace ( rx,"&quot;");
    return r;
}

int max(int a, int b)
{
	if (a>b)
		return a;
	return b;
}

int xmlObj::actindent=0;		// make instance of actindent

xmlObj::xmlObj()
{
    indentwidth=4;
}


// returns <s at />
QString xmlObj::singleElement(QString s, QString at)
{
    return indent() + "<" + s +" " + at +" " + "/>\n";
}

// returns <s>
QString xmlObj::beginElement(QString s)
{
    return indent() + "<" + s + ">\n";
}

// returns <s at>
QString xmlObj::beginElement(QString s, QString at)
{
    return indent() + "<" + s + " " + at + ">\n";
}

// returns </s>
QString xmlObj::endElement(QString s)
{
    return indent() + "</" + s + ">\n";
}

// returns  at="val"
QString xmlObj::attribut(QString at, QString val)
{
    return " " + at + "=\"" + quotemeta (val) + "\""; 
}

// returns <s> val </s>
QString xmlObj::valueElement(QString el, QString val)
{
    return indent() + "<" + el + ">" + quotemeta(val) + "</" +el + ">\n";
}

// returns <s at> val </s>
QString xmlObj::valueElement(QString el, QString val, QString at)
{
    return indent() + "<" + el + " " + at + ">" + quotemeta(val) + "</" +el + ">\n";
}

void xmlObj::incIndent()
{
    actindent++;
}	

void xmlObj::decIndent()
{
    actindent--;
    if (actindent<0) actindent=0;
}	

QString xmlObj::indent()
{
    QString s;
    int i;
    for (i=0; i<actindent*indentwidth; i++)
    {
		s= s + " ";
    }
    return s;
}	



ImagePreview::ImagePreview (QWidget *parent=0): QLabel (parent)
{
}

void ImagePreview::previewUrl( const QUrl &u )
{
    QString path = u.path();
    QPixmap pix( path );
    if ( pix.isNull() )
        setText( QObject::tr("This is not an image.") );
    else
	{
		float max_w=300;
		float max_h=300;
		float r;
		if (pix.width()>max_w)
		{
			r=max_w / pix.width();
			pix.resize(qRound(pix.width()*r), qRound(pix.height()*r));
			// TODO not a resize, but a shrink/enlarge is needed here...
		}
		if (pix.height()>max_h)
		{
			r=max_h / pix.height();
			pix.resize(qRound(pix.width()*r), qRound(pix.height()*r));
			// TODO not a resize, but a shrink/enlarge is needed here...
		}
        setPixmap( pix );
	}	
}

