/*---------------------------------------------------------------------------*\

    This program may be used to test caller id transmission.


         Voicetronix Voice Processing Board (VPB) Software

         Copyright (C) 2007-2008, Ron Lee, Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License version 2.1 as published by the Free Software Foundation.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,
         MA  02110-1301  USA

\*---------------------------------------------------------------------------*/

#include "vpbapi.h"

#include <cstdio>
#include <cstdlib>
#include <cstring>

using std::string;


static int arg_exists(int argc, char *argv[], const char *arg)
{ //{{{
	for(int i = 0; i < argc; ++i)
		if(strcmp(argv[i],arg) == 0) return i;

	return 0;
} //}}}

static void usage(const char *argv0)
{ //{{{
	printf("usage: %s [options]\n\n", argv0);
	printf("  --card num         Play tone to card num.  Default is 0\n");
	printf("  --port num         Play tone to port num.  Default is 0\n");
	printf("  --number num       Use num as the caller number\n");
	printf("  --name name        Use name as the caller name\n");
	printf("  --date MMDDHHMM    Use an explicitly specified date string\n");
	printf("  --type2            Send type2 (call waiting) CID\n");
	printf("  --recgain db       Set the record gain to db.\n");
	printf("  --rec outfile      Record to outfile while playing.\n");
	printf("  --help, -?         Show this message.\n\n");
} //}}}

int main(int argc, char *argv[])
{ //{{{
	VPB_CID cid;
	string  recfile;
	int     cardnum  = 0;
	int     portnum  = 0;
	bool    type2    = false;
	int     arg;

	if(argc < 2) {
		usage(argv[0]);
		exit(EXIT_FAILURE);
	}
	if( arg_exists(argc,argv,"--help") || arg_exists(argc, argv,"-?") ) {
		usage(argv[0]);
		exit(EXIT_SUCCESS);
	}
	if((arg = arg_exists(argc,argv,"--card")) != 0) cardnum = atoi(argv[arg+1]);
	if((arg = arg_exists(argc,argv,"--port")) != 0) portnum = atoi(argv[arg+1]);

	if((arg = arg_exists(argc,argv,"--number")) != 0) {
		if( vpb_cid_set(&cid, VPB_CID_CALLING_LINE_DN, argv[arg+1]) ) {
			fprintf(stderr, "Invalid cid number '%s'\n", argv[arg+1]);
			exit(EXIT_FAILURE);
		}
	}

	if((arg = arg_exists(argc,argv,"--name")) != 0) {
		if( vpb_cid_set(&cid, VPB_CID_CALLER_NAME, argv[arg+1]) ) {
			fprintf(stderr, "Invalid cid name '%s'\n", argv[arg+1]);
			exit(EXIT_FAILURE);
		}
	}

	if((arg = arg_exists(argc,argv,"--date")) != 0) {
		if( vpb_cid_set(&cid, VPB_CID_DATE_TIME, argv[arg+1]) ) {
			fprintf(stderr, "Invalid cid date '%s'\n", argv[arg+1]);
			exit(EXIT_FAILURE);
		}
	}

	if((arg = arg_exists(argc,argv,"--type2")) != 0) type2   = true;
	if((arg = arg_exists(argc,argv,"--rec")) != 0)   recfile = argv[arg+1];


	int h = vpb_open(cardnum, portnum);

	if( vpb_get_port_type(h) != VPB_FXS ) {
		fprintf(stderr, "Card %d, port %d is not an FXS port\n",
			cardnum, portnum);
		vpb_close(h);
		exit(EXIT_FAILURE);
	}

	// Wait a moment for the hardware init events to complete.
	sleep(1);

	if((arg = arg_exists(argc, argv, "--recgain")) != 0)
		vpb_record_set_gain(h, atof(argv[arg+1]));

	if( ! recfile.empty() ) {
		vpb_record_file_async(h, recfile, VPB_LINEAR);
		fprintf(stdout, "recording to %s\n",  recfile.c_str());
	}

	if( type2 ) {
		if( vpb_get_hookstate(h) != VPB_OFFHOOK ) {
			fprintf(stderr, "Station must be offhook for type 2 CID\n");
			vpb_close(h);
			exit(EXIT_FAILURE);
		}
		vpb_send_cid_t2_sync(h, cid);
	}
	else {
		if( vpb_get_hookstate(h) != VPB_ONHOOK ) {
			fprintf(stderr, "Station must be onhook for type 1 CID\n");
			vpb_close(h);
			exit(EXIT_FAILURE);
		}
		vpb_ring_with_cid(h, cid);

		// let it ring a couple of seconds
		vpb_sleep(2000);
		vpb_ring_station_async(h, VPB_RING_STATION_OFF);

		// give it a chance to actually stop before we close down
		vpb_sleep(100);
	}

	//fprintf(stdout, "stopping recording...\n");
	//if( ! recfile.empty() )
	//	vpb_record_terminate_sync(h);

	fprintf(stdout, "done.\n\n");
	vpb_close(h);

	return EXIT_SUCCESS;
} //}}}

