/*---------------------------------------------------------------------------*\

    FILE....: TV4LOG.CPP
    TYPE....: C++ Console program
    AUTHOR..: Ian Roberts
    DATE....: 15/04/02

    This program tests the V4LOG, recording all 4 channels to disk files
      using VOX.

    Compile:gcc tv4log.cpp -o tv4log ../linux/libvpb.a -pthread -lm 
                         -Wall -g     
    Run: ./tv4log  
    
    Switches:
	-no	// No file overwrite

	-linev  // Use line voltage detection insted of VOX
		// Note: This only works with Line Voltage Detection module.
		

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

         Voicetronix Voice Processing Board (VPB) Software

         Copyright (C) 1999-2001 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	 USA

\*---------------------------------------------------------------------------*/

#define NUM_BOARDS 1 
#define NUM_CH 4

#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#ifndef WIN32
#include <pthread.h>
#endif
#ifdef WIN32
#include "../threads.h"
#endif
#include "vpbapi.h"
#include "verbose.h"
#include "mess.h"

int arg_exists(int argc, char *argv[], char *arg);

int linev;
int overwrite;
int serial[NUM_BOARDS*NUM_CH];

void WINAPI callback(VPB_EVENT *e, void *context)
{
	char  s[VPB_MAX_STR];
	char  filename[VPB_MAX_STR];
	int h;

	// detach from parent
#ifndef WIN32
        pthread_detach(pthread_self());
#endif
	h= *((int *)context);

	vpb_translate_event(e, s);
	mprintf("%s",s);

	switch(e->type) {
        case VPB_VOXON :
		if(!linev){
			mprintf("VOX:");
			if (overwrite){
				sprintf(filename,"file%d.wav",h);
			} else {
				sprintf(filename,"file%d_%d.wav", h, serial[h]++);
			}
			vpb_record_file_async(h, filename, VPB_LINEAR);
		}
		break;

        case VPB_VOXOFF :
		if(!linev){
			vpb_record_terminate(h);
		}
                break;

        case VPB_LOOP_OFFHOOK :
		if(linev){
			mprintf("LOOP:");
			if (overwrite){
				sprintf(filename,"file%d.wav",h);
			} else {
				sprintf(filename,"file%d_%d.wav", h, serial[h]++);
			}
			vpb_record_file_async(h, filename, VPB_LINEAR);
		}
		break;

        case VPB_LOOP_ONHOOK :
		if(linev) {
			vpb_record_terminate(h);
		}
                break;

        case VPB_RECORDEND :
		break;

	default :
		break;

	}
	#ifdef WIN32
        pthread_detach();
	#endif
}

int main(int argc, char *argv[]) {
	int         h[NUM_BOARDS*NUM_CH];
	VPB_VOX	    v;
	int         i,j;
	char	    *model;	
	int         openpci=0;
	overwrite = 1;
	linev = 0;

	printf("Test V4LOG card / driver \n");
	printf("Switches: -no       No file overwrite\n");
	printf("          -linev    Use Line Voltage detection for record start/stop,\n");
	printf("                    only for cards with Line Voltage Detection module.\n");
	printf("          -opci     Using an OpenPCI card for logging.\n");
	
	if (arg_exists(argc, argv, "-opci") != 0) {
		printf("Using openpci card, with line voltage events!\n");
		//linev = 1;
		openpci=1;
	}
	else {
		model= getenv("VPB_MODEL");
		if((model == NULL) || ( 0 != strcmp(model, "V4LOG") )) {
			printf("%s requires the environment variable VPB_MODEL be set to V4LOG\n", argv[0] );
			printf("This application is compatable with V4LOG & V4PCI cards.\n");	
			return -1;
		}
	}
	
	if (arg_exists(argc, argv, "-no") != 0) {
		overwrite = 0;
	}

	if (arg_exists(argc, argv, "-linev") != 0) {
		linev = 1;
	}

	verbose(1);

	v.runon = 4000;
	if(openpci){
		v.onlevel = -34;
		v.offlevel = -36;
	} else {
		v.onlevel = -40;
		v.offlevel = -46;
	}
    
	for(j=0; j<NUM_BOARDS; j++) {
		for(i=0; i<NUM_CH; i++) {
			int ch = (NUM_CH*j)+i;
			h[ch] = vpb_open(j,i);
			vpb_setvox(h[ch], &v); 
			vpb_set_event_callback(h[ch], callback, (void*)&h[ch]);
		}
	}

	if(!openpci){
		vpb_getvox(h[0], &v); 
		printf("on level = %f dB  off level = %f dB  run on = %d ms\n", v.onlevel,v.offlevel,v.runon);
	}

	printf("Press return to exit\n");
	getchar();

	for(i=0; i<(NUM_CH*NUM_BOARDS); i++) {
		vpb_close(h[i]);
	}

	return 0;
}

int arg_exists(int argc, char *argv[], char *arg) {
  int i;

  for(i=0; i<argc; i++)
    if (strcmp(argv[i],arg) == 0)
      return i;

  return 0;
}
