/* render an image in the background as a set of tiles
 *
 * don't read from mask after closing out
 *
 * JC, 30 sep 03 
 *
 * 22/10/03 JC
 *	- now uses threadgroup kill system, avoiding race condition
 * 2/2/04 JC
 *	- cache failed for large images
 * 8/4/04
 *	- touch reused tiles so they don't get reused again too soon ... helps
 *	  stop thrashing when we've many pending paints and lots of threads
 * 15/4/04
 *	- added im_cache() convenience function
 */

/*

    This file is part of VIPS.
    
    VIPS is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/* Turn on debugging output.
#define DEBUG_MAKE
#define DEBUG
#define DEBUG_PAINT
#define DEBUG_REUSE
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /*HAVE_CONFIG_H*/
#include <vips/intl.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include <vips/vips.h>

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif /*WITH_DMALLOC*/

#ifdef HAVE_THREADS
static const int have_threads = 1;
#else /*!HAVE_THREADS*/
static const int have_threads = 0;
#endif /*HAVE_THREADS*/

/* Notify caller through this.
 */
typedef void (*notify_fn)( IMAGE *, Rect *, void * );

/* A tile in our cache. 
 */
typedef struct {
	struct _Render *render;

	Rect area;		/* Place here (unclipped) */
	REGION *region;		/* REGION with the pixels */
	int time;		/* Time of last use for LRU flush */
	int dirty;		/* TRUE for on the dirty list */
	int painted;		/* TRUE for full of valid pixels */
} Tile;

/*

	FIXME ... should have an LRU queue rather than this thing with times

	FIXME ... could also hash from tile xy to tile pointer

 */

/* Per-call state.
 */
typedef struct _Render {
	/* Parameters.
	 */
	IMAGE *in;		/* Image we render */
	IMAGE *out;		/* Write tiles here on demand */
	IMAGE *mask;		/* Set valid pixels here */
	int width, height;	/* Tile size */
	int max;		/* Maximum number of tiles */
	notify_fn notify;	/* Tell caller about paints here */
	void *client;

	/* Make readers single thread with this. No point allowing
	 * multi-thread read.
	 */
	GMutex *read_lock;	

	/* Tile cache.
	 */
	GSList *cache;		/* List of tiles */
	int ntiles;		/* Number of cache tiles */
	int time;		/* Access ticks */

	/* List of tiles which are to be painted.
	 */
	GMutex *dirty_lock;	/* Lock before we read/write the dirty list */
	GSList *dirty;		/* Tiles which need painting */

	/* Render thread stuff.
	 */
	GThread *thread;	/* tid for background thread */
	im_semaphore_t sem;	/* Start/stop render thread with this */
	im_threadgroup_t *tg;	/* Render with this threadgroup */
} Render;

#ifdef DEBUG
static void *
tile_print( Tile *tile )
{
	printf( "tile: %dx%d, time = %d, dirty = %d, painted = %d\n",
		tile->area.left, tile->area.top,
		tile->time, tile->dirty, tile->painted );

	return( NULL );
}
#endif /*DEBUG*/

static void *
tile_destroy( Tile *tile )
{
#ifdef DEBUG_MAKE
	printf( "tile_destroy\n" );
#endif /*DEBUG_MAKE*/

	if( tile->region ) {
		(void) im_region_free( tile->region );
		tile->region = NULL;
	}

	im_free( tile );

	return( NULL );
}

static int
render_destroy( Render *render )
{
#ifdef DEBUG_MAKE
	printf( "render_destroy: 0x%x\n", (unsigned int) render );
#endif /*DEBUG_MAKE*/

	/* Kill background thread.
	 */
	if( render->thread ) {
		render->tg->kill = 1;
		im_semaphore_up( &render->sem );
		(void) g_thread_join( render->thread );
		render->thread = NULL;
		render->tg->kill = 0;
	}
	im_semaphore_destroy( &render->sem );
	IM_FREEF( im_threadgroup_free, render->tg );

	/* Free cache.
	 */
	im_slist_map2( render->cache,
		(VSListMap2Fn) tile_destroy, NULL, NULL );
	IM_FREEF( g_slist_free, render->cache );
	render->ntiles = 0;
	IM_FREEF( g_slist_free, render->dirty );
	g_mutex_free( render->dirty_lock );
	g_mutex_free( render->read_lock );

	return( 0 );
}

/* Pick a tile off the dirty list.
 */
static Tile *
render_pick_tile( Render *render )
{
	Tile *tile;

	g_mutex_lock( render->dirty_lock );
	if( render->dirty ) {
		tile = (Tile *) render->dirty->data;
		assert( tile->dirty );
		assert( !tile->painted );
		render->dirty = g_slist_remove( render->dirty, tile );
		tile->dirty = 0;
	}
	else
		tile = NULL;
	g_mutex_unlock( render->dirty_lock );

	return( tile );
}

/* Paint a tile and notify.
 */
static void
render_paint_tile( Tile *tile )
{
	Render *render = tile->render;
	int result;

#ifdef DEBUG_PAINT
	printf( "render_fill_tile: paint of tile %dx%d\n",
		tile->area.left, tile->area.top );
#endif /*DEBUG_PAINT*/

	/*

		FIXME ... nice if we did something with the error return

	 */
	result = im_prepare_thread( render->tg, tile->region, &tile->area );
#ifdef DEBUG_PAINT
	if( result )
		printf( "render_fill_tile: im_prepare_thread() failed!\n"
			"\t%s\n", im_errorstring() );
#endif /*DEBUG_PAINT*/
	tile->painted = 1;
	if( render->notify )
		render->notify( render->out, &tile->area, render->client );
}

/* Come here for the background work thread.
 */
static void *
render_background( void *client )
{
	Render *render = (Render *) client;

	for(;;) {
		Tile *tile;

		im_semaphore_down( &render->sem );

		if( render->tg->kill )
			break;

		if( (tile = render_pick_tile( render )) )
			render_paint_tile( tile );
	}

	return( NULL );
}

static Render *
render_new( IMAGE *in, IMAGE *out, IMAGE *mask, 
	int width, int height, int max, notify_fn notify, void *client )
{
	Render *render;

	if( !(render = IM_NEW( out, Render )) )
		return( NULL );
	render->in = in;
	render->out = out;
	render->mask = mask;
	render->width = width;
	render->height = height;
	render->max = max;
	render->notify = notify;
	render->client = client;

	render->read_lock = g_mutex_new();
	render->time = 0;
	render->cache = NULL;
	render->ntiles = 0;
	render->dirty_lock = g_mutex_new();
	render->dirty = NULL;

	render->thread = NULL;
	im_semaphore_init( &render->sem, 0, "render" );
	render->tg = NULL;

	if( im_add_close_callback( out, 
                (im_callback_fn) render_destroy, render, NULL ) ) {
                (void) render_destroy( render );
                return( NULL );
        }

	if( !(render->tg = im_threadgroup_create( in )) )
		return( NULL );

	/* Only need the bg thread for asynch render. 
	 */
	if( have_threads && notify && 
		!(render->thread = g_thread_create_full( 
			render_background, render, 
			IM__DEFAULT_STACK_SIZE, TRUE, FALSE, 
			G_THREAD_PRIORITY_NORMAL, NULL )) ) {
		im_error( "render_new", _( "unable to create thread" ) );
		return( NULL );
	}

	return( render );
}

/* Make a Tile.
 */
static Tile *
tile_new( Render *render )
{
	Tile *tile;

#ifdef DEBUG_MAKE
	printf( "tile_new\n" );
#endif /*DEBUG_MAKE*/

	/* Don't use auto-free: we need to make sure we free the tile after
	 * Render.
	 */
	if( !(tile = IM_NEW( NULL, Tile )) )
		return( NULL );
	tile->render = render;
	tile->region = NULL;
	tile->area.left = 0;
	tile->area.top = 0;
	tile->area.width = 0;
	tile->area.height = 0;
	tile->time = render->time;
	tile->dirty = 0;
	tile->painted = 0;

	if( !(tile->region = im_region_create( render->in )) ) {
		(void) tile_destroy( tile );
		return( NULL );
	}

	render->cache = g_slist_prepend( render->cache, tile );
	render->ntiles += 1;

	return( tile );
}

static void *
tile_test_area( Tile *tile, Rect *area )
{
	if( im_rect_equalsrect( &tile->area, area ) )
		return( tile );

	return( NULL );
}

/* Search the cache for a tile, NULL if not there. Could be *much* faster. If
 * we find a tile, bump to front.
 */
static Tile *
render_tile_lookup( Render *render, Rect *area )
{
	Tile *tile;

	tile = (Tile *) im_slist_map2( render->cache,
		(VSListMap2Fn) tile_test_area, area, NULL );

	/* We've looked at a tile ... bump to end of LRU and front of dirty.
	 */
	if( tile ) {
		tile->time = render->time;
		render->time += 1;

		g_mutex_lock( render->dirty_lock );
		if( tile->dirty ) {
#ifdef DEBUG
			printf( "tile_bump_dirty: bumping tile %dx%d\n",
				tile->area.left, tile->area.top );
#endif /*DEBUG*/

			render->dirty = g_slist_remove( render->dirty, tile );
			render->dirty = g_slist_prepend( render->dirty, tile );
		}
		g_mutex_unlock( render->dirty_lock );
	}

	return( tile );
}

/* Add a tile to the dirty list.
 */
static void
tile_set_dirty( Tile *tile, Rect *area )
{
	Render *render = tile->render;

#ifdef DEBUG_PAINT
	printf( "tile_set_dirty: adding tile %dx%d to dirty\n",
		area->left, area->top );
#endif /*DEBUG_PAINT*/

	assert( !tile->dirty );

	/* Touch the time ... we want to make sure this tile will not be
	 * reused too soon, so it gets a chance to get painted.
	 */
	tile->time = render->time;
	render->time += 1;

	g_mutex_lock( render->dirty_lock );
	tile->dirty = 1;
	tile->area = *area;
	tile->painted = 0;
	render->dirty = g_slist_prepend( render->dirty, tile );
	g_mutex_unlock( render->dirty_lock );

	if( render->notify && have_threads ) 
		/* Tell the bg thread there's a tile to be done.
		 */
		im_semaphore_up( &render->sem );
	else {
		/* No threads, or no notify ... paint the tile ourselves, 
		 * sychronously. No need to notify the client, since they'll 
		 * never see black tiles.
		 */
#ifdef DEBUG_PAINT
		printf( "tile_set_dirty: painting tile %dx%d synchronously\n",
			area->left, area->top );
#endif /*DEBUG_PAINT*/

		(void) im_prepare_thread( render->tg, 
			tile->region, &tile->area );
		tile->painted = 1;
		tile->dirty = 0;
		render->dirty = g_slist_remove( render->dirty, tile );
	}
}

static void *
tile_test_clean_time( Tile *this, Tile **best )
{
	if( this->painted && !this->dirty )
		if( !*best || this->time < (*best)->time )
			*best = this;

	return( NULL );
}

/* Pick a painted tile to reuse. Search for LRU (slow!).
 */
static Tile *
render_tile_get_painted( Render *render )
{
	Tile *tile;

	tile = NULL;
	im_slist_map2( render->cache,
		(VSListMap2Fn) tile_test_clean_time, &tile, NULL );

	if( tile ) {
#ifdef DEBUG_REUSE
		printf( "render_tile_get_painted: reusing painted 0x%x\n",
			(unsigned int) tile );
		assert( !tile->dirty );
		assert( !g_slist_find( render->dirty, tile ) );
#endif /*DEBUG_REUSE*/

		tile->painted = 0;
	}

	return( tile );
}

/* Take a tile off the end of the dirty list.
 */
static Tile *
render_tile_get_dirty( Render *render )
{
	Tile *tile;

	g_mutex_lock( render->dirty_lock );
	if( !render->dirty )
		tile = NULL;
	else {
		tile = (Tile *) g_slist_last( render->dirty )->data;
		render->dirty = g_slist_remove( render->dirty, tile );
		tile->dirty = 0;
	}
	g_mutex_unlock( render->dirty_lock );

#ifdef DEBUG_REUSE
	if( tile )
		printf( "render_tile_get_dirty: reusing dirty 0x%x\n",
			(unsigned int) tile );
#endif /*DEBUG_REUSE*/

	return( tile );
}

static Tile *
render_tile_get( Render *render, Rect *area )
{
	Tile *tile;

	/* Got this tile already?
	 */
	if( (tile = render_tile_lookup( render, area )) ) 
		return( tile );

	/* Have we fewer tiles than teh max? Can just make a new tile.
	 */
	if( render->ntiles < render->max && (tile = tile_new( render )) ) {
		tile_set_dirty( tile, area );
		return( tile );
	}

	/* Need to reuse a tile. Try for an old painted tile first, then if
	 * that fails, reuse a dirty tile.
	 */
	if( (tile = render_tile_get_painted( render )) ||
		(tile = render_tile_get_dirty( render )) ) {
#ifdef DEBUG_REUSE
		printf( "(render_tile_get: was at %dx%d, moving to %dx%d)\n",
			tile->area.left, tile->area.top,
			area->left, area->top );
#endif /*DEBUG_REUSE*/
		tile_set_dirty( tile, area );
		return( tile );
	}

	return( NULL );
}

/* Copy what we can from the tile into the region.
 */
static void
tile_copy( Tile *tile, REGION *to )
{
	Rect ovlap;
	int y;
	int len;

	/* Find common pixels.
	 */
	im_rect_intersectrect( &tile->area, &to->valid, &ovlap );
	assert( !im_rect_isempty( &ovlap ) );
	len = IM_IMAGE_SIZEOF_PEL( to->im ) * ovlap.width;

	/* If the tile is painted, copy over the pixels. Otherwise, fill with
	 * black.
	 */
	if( tile->painted ) 
		for( y = ovlap.top; y < IM_RECT_BOTTOM( &ovlap ); y++ ) {
			PEL *p = (PEL *) IM_REGION_ADDR( tile->region, 
				ovlap.left, y );
			PEL *q = (PEL *) IM_REGION_ADDR( to, ovlap.left, y );

			memcpy( q, p, len );
		}
	else 
		for( y = ovlap.top; y < IM_RECT_BOTTOM( &ovlap ); y++ ) {
			PEL *q = (PEL *) IM_REGION_ADDR( to, ovlap.left, y );

			memset( q, 0, len );
		}
}

/* Loop over the output region, filling with data from cache.
 */
static int
region_fill( REGION *out, void *seq, Render *render )
{
	Rect *r = &out->valid;
	int x, y;

	/* Find top left of tiles we need.
	 */
	int xs = (r->left / render->width) * render->width;
	int ys = (r->top / render->height) * render->height;

	/* Only allow one reader. No point threading this, calculation is
	 * decoupled anyway.
	 */
	g_mutex_lock( render->read_lock );

	/* 

		FIXME ... if r fits inside a single tile, could skip the copy.

	 */

	for( y = ys; y < IM_RECT_BOTTOM( r ); y += render->height )
		for( x = xs; x < IM_RECT_RIGHT( r ); x += render->width ) {
			Rect area;
			Tile *tile;

			area.left = x;
			area.top = y;
			area.width = render->width;
			area.height = render->height;

			if( (tile = render_tile_get( render, &area )) )
				tile_copy( tile, out );
		}

	g_mutex_unlock( render->read_lock );

	return( 0 );
}

/* Paint the state of the 'painted' flag for a tile.
 */
static void
tile_paint_mask( Tile *tile, REGION *to )
{
	Rect ovlap;
	int y;
	int len;

	/* Find common pixels.
	 */
	im_rect_intersectrect( &tile->area, &to->valid, &ovlap );
	assert( !im_rect_isempty( &ovlap ) );
	len = IM_IMAGE_SIZEOF_PEL( to->im ) * ovlap.width;

	for( y = ovlap.top; y < IM_RECT_BOTTOM( &ovlap ); y++ ) {
		PEL *q = (PEL *) IM_REGION_ADDR( to, ovlap.left, y );

		memset( q, tile->painted ? 255 : 0, len );
	}
}

/* The mask image is just 255/0 for the state of painted for each tile.
 */
static int
mask_fill( REGION *out, void *seq, Render *render )
{
	Rect *r = &out->valid;
	int x, y;

	/* Find top left of tiles we need.
	 */
	int xs = (r->left / render->width) * render->width;
	int ys = (r->top / render->height) * render->height;

	/* This will (probably) be NULL if we're called after out has been
	 * closed.
	 */
	assert( render->tg );

	g_mutex_lock( render->read_lock );

	for( y = ys; y < IM_RECT_BOTTOM( r ); y += render->height )
		for( x = xs; x < IM_RECT_RIGHT( r ); x += render->width ) {
			Rect area;
			Tile *tile;

			area.left = x;
			area.top = y;
			area.width = render->width;
			area.height = render->height;

			if( (tile = render_tile_get( render, &area )) )
				tile_paint_mask( tile, out );
		}

	g_mutex_unlock( render->read_lock );

	return( 0 );
}

int
im_render( IMAGE *in, IMAGE *out, IMAGE *mask, 
	int width, int height, int max, notify_fn notify, void *client )
{
	Render *render;

	if( width <= 0 || height <= 0 || max <= 0 ) {
		im_errormsg( "im_render: bad parameters" );
		return( -1 );
	}
	if( im_pincheck( in ) || im_poutcheck( out ) || im_poutcheck( mask ) )
		return( -1 );
	if( !(render = render_new( in, out, mask, 
		width, height, max, notify, client )) )
		return( -1 );
	if( im_cp_desc( out, in ) || im_cp_desc( mask, in ) )
		return( -1 );
	mask->Bands = 1;
	mask->BandFmt = IM_BANDFMT_UCHAR;
	mask->Bbits = IM_BBITS_BYTE;
	mask->Type = IM_TYPE_B_W;
	mask->Coding = IM_CODING_NONE;

#ifdef DEBUG_MAKE
	printf( "im_render: max = %d, 0x%x\n", max, (unsigned int) render );
#endif /*DEBUG_MAKE*/

	if( im_generate( out, NULL, region_fill, NULL, render, NULL ) )
		return( -1 );
	if( im_generate( mask, NULL, mask_fill, NULL, render, NULL ) )
		return( -1 );

	return( 0 );
}

int 
im_cache( IMAGE *in, IMAGE *out, int width, int height, int max )
{
	IMAGE *t;

	if( !(t = im_open_local( out, "im_cache", "p" )) ||
		im_render( in, out, t, width, height, max, NULL, NULL ) )
		return( -1 );

	return( 0 );
}
