/* Convert 1 to 4-band 8 or 16-bit VIPS images to/from PNG.
 *
 * 28/11/03 JC
 *	- better no-overshoot on tile loop
 */

/*

    This file is part of VIPS.
    
    VIPS is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/*
#define DEBUG
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /*HAVE_CONFIG_H*/
#include <vips/intl.h>

#ifndef HAVE_PNG

#include <vips/vips.h>

int
im_vips2png( IMAGE *in, const char *filename )
{
	im_errormsg( "im_vips2png: PNG support disabled" );
	return( -1 );
}

int
im_png2vips( const char *name, IMAGE *out )
{
	im_errormsg( "im_png2vips: PNG support disabled" );
	return( -1 );
}

int
im_png2vips_header( const char *name, IMAGE *out )
{
	im_errormsg( "im_png2vips_header: PNG support disabled" );
	return( -1 );
}

#else /*HAVE_PNG*/

#include <stdio.h>
#include <assert.h>
#include <stdlib.h>

#include <vips/vips.h>

#include <png.h>

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif /*WITH_DMALLOC*/

#if PNG_LIBPNG_VER < 10003
#error "PNG library too old."
#endif

static void
user_error_function( png_structp png_ptr, png_const_charp error_msg )
{
	im_errormsg( "PNG error: \"%s\"", error_msg );
}

static void
user_warning_function( png_structp png_ptr, png_const_charp warning_msg )
{
	im_errormsg( "PNG warning: \"%s\"", warning_msg );
}

/* What we track during a PNG write.
 */
typedef struct {
	IMAGE *in;
	REGION *reg;
	im_threadgroup_t *tg;

	int tile_width, tile_height;
	FILE *fp;
	png_structp pPng;
	png_infop pInfo;
	png_bytep *row_pointer;
} Write;

static void
write_destroy( Write *write )
{
	IM_FREEF( im_region_free, write->reg );
	IM_FREEF( im_threadgroup_free, write->tg );
	IM_FREEF( fclose, write->fp );
	if( write->pPng )
		png_destroy_write_struct( &write->pPng, &write->pInfo );
	IM_FREE( write->row_pointer );

	im_free( write );
}

static Write *
write_new( IMAGE *in )
{
	Write *write;

	if( !(write = IM_NEW( NULL, Write )) )
		return( NULL );

	im__find_demand_size( in, &write->tile_width, &write->tile_height );

	write->in = in;
	write->reg = im_region_create( write->in );
	write->tg = im_threadgroup_create( write->in );
	write->row_pointer = IM_ARRAY( NULL, write->tile_height, png_bytep );
	write->fp = NULL;
	write->pPng = NULL;
	write->pInfo = NULL;

	if( !write->reg || !write->tg || !write->row_pointer ) {
		write_destroy( write );
		return( NULL );
	}

	if( !(write->pPng = png_create_write_struct( 
		PNG_LIBPNG_VER_STRING, NULL,
		user_error_function, user_warning_function )) ) {
		write_destroy( write );
		return( NULL );
	}

	/* Catch PNG errors from png_create_info_struct().
	 */
	if( setjmp( write->pPng->jmpbuf ) ) {
		write_destroy( write );
		return( NULL );
	}

	if( !(write->pInfo = png_create_info_struct( write->pPng )) ) {
		write_destroy( write );
		return( NULL );
	}

	return( write );
}

/* Write a VIPS image to PNG.
 */
static int
write_vips( Write *write, int compress, int interlace )
{
	IMAGE *in = write->in;

	Rect area;
	int j, y, i, nb_passes;

	/* Catch PNG errors.
	 */
	if( setjmp( write->pPng->jmpbuf ) ) 
		return( -1 );

	/* Check input image.
	 */
	if( im_pincheck( in ) )
		return( -1 );
	if( in->Coding != IM_CODING_NONE ) {
		im_errormsg( "im_vips2png: uncoded images only" );
		return( -1 );
	}
	if( in->BandFmt != IM_BANDFMT_UCHAR && 
		in->BandFmt != IM_BANDFMT_USHORT ) {
		im_errormsg( "im_vips2png: uchar or ushort only" );
		return( -1 );
	}
	if( in->Bands < 1 || in->Bands > 4 ) {
		im_errormsg( "im_vips2png: 1 to 4-band only" );
		return( -1 );
	}
	if( compress < 0 || compress > 9 ) {
		im_errormsg( "im_vips2png: compress should be in 0-9 range" );
		return( -1 );
	}

	/* Set compression parameters.
	 */
	png_set_compression_level( write->pPng, compress );

	write->pInfo->width = in->Xsize;
	write->pInfo->height = in->Ysize;
	write->pInfo->bit_depth = (in->BandFmt == IM_BANDFMT_UCHAR ? 8 : 16);
	write->pInfo->gamma = (float) 1.0;

	switch( in->Bands ) {
	case 1: write->pInfo->color_type = PNG_COLOR_TYPE_GRAY; break;
	case 2: write->pInfo->color_type = PNG_COLOR_TYPE_GRAY_ALPHA; break;
	case 3: write->pInfo->color_type = PNG_COLOR_TYPE_RGB; break;
	case 4: write->pInfo->color_type = PNG_COLOR_TYPE_RGB_ALPHA; break;

	default:
		assert( 0 );
	}

	png_write_info( write->pPng, write->pInfo ); 

	/* If we're an intel byte order CPU and this is a 16bit image, we need
	 * to swap bytes.
	 */
	if( write->pInfo->bit_depth > 8 && !im_amiMSBfirst() ) 
		png_set_swap( write->pPng ); 

	if( interlace )	
		nb_passes = png_set_interlace_handling( write->pPng );
	else
		nb_passes = 1;

	/* Write data.
	 */
	for( i = 0; i < nb_passes; i++ ) 
		for( y = 0; y < in->Ysize; y += write->tile_height ) {
			area.left = 0;
			area.top = y;
			area.width = in->Xsize;
			area.height = IM_MIN( write->tile_height, 
				in->Ysize - y );

			if( im_prepare_thread( write->tg, write->reg, &area ) )
				return( -1 );

			for( j = 0; j < area.height; j++ ) 
				write->row_pointer[j] =
					IM_REGION_ADDR( write->reg, 0, y + j );
			png_write_rows( write->pPng, 
				write->row_pointer, area.height );
		}

	png_write_end( write->pPng, write->pInfo );

	return( 0 );
}

/* Write a VIPS image to a file as PNG.
 */
int
im_vips2png( IMAGE *in, const char *filename )
{
	Write *write;
	int compress; 
	int interlace; 

	char *p, *q;

	char name[FILENAME_MAX];
	char mode[FILENAME_MAX];
	char buf[FILENAME_MAX];

	if( !(write = write_new( in )) )
		return( -1 );

	/* Extract write mode from filename and parse.
	 */
	im_filename_split( filename, name, mode );
	strcpy( buf, mode ); 
	p = &buf[0];
	compress = 6;
	interlace = 0;
	if( (q = im_getnextoption( &p )) ) 
		compress = atoi( q );
	if( (q = im_getnextoption( &p )) ) 
		interlace = atoi( q );

	/* Make output.
	 */
#ifdef BINARY_OPEN
        if( !(write->fp = fopen( name, "wb" )) ) {
#else /*BINARY_OPEN*/
        if( !(write->fp = fopen( name, "w" )) ) {
#endif /*BINARY_OPEN*/
		write_destroy( write );
		im_errormsg( "im_vips2png: unable to open \"%s\"", name );

		return( -1 );
	}
	png_init_io( write->pPng, write->fp );

	/* Convert it!
	 */
	if( write_vips( write, compress, interlace ) ) {
		write_destroy( write );
		im_errormsg( "im_vips2png: unable to write \"%s\"", name );

		return( -1 );
	}
	write_destroy( write );

	return( 0 );
}

/* What we track during a PNG read.
 */
typedef struct {
	char *name;
	IMAGE *out;

	FILE *fp;
	png_structp pPng;
	png_infop pInfo;
	png_bytep *row_pointer;
	png_bytep data;
} Read;

static void
read_destroy( Read *read )
{
	if( read->name ) {
		im_free( read->name );
		read->name = NULL;
	}
	if( read->fp ) {
		fclose( read->fp );
		read->fp = NULL;
	}
	if( read->pPng )
		png_destroy_read_struct( &read->pPng, &read->pInfo, NULL );
	if( read->row_pointer ) {
		im_free( read->row_pointer );
		read->row_pointer = NULL;
	}
	if( read->data ) {
		im_free( read->data );
		read->data = NULL;
	}

	im_free( read );
}

static Read *
read_new( const char *name, IMAGE *out )
{
	Read *read;

	if( !(read = IM_NEW( NULL, Read )) )
		return( NULL );

	read->name = im_strdup( NULL, name );
	read->out = out;
	read->fp = NULL;
	read->pPng = NULL;
	read->pInfo = NULL;
	read->row_pointer = NULL;
	read->data = NULL;

#ifdef BINARY_OPEN
        if( !(read->fp = fopen( name, "rb" )) ) {
#else /*BINARY_OPEN*/
        if( !(read->fp = fopen( name, "r" )) ) {
#endif /*BINARY_OPEN*/
		read_destroy( read );
		im_errormsg( "im_png2vips: unable to open \"%s\"", name );
		return( NULL );
	}

	if( !(read->pPng = png_create_read_struct( 
		PNG_LIBPNG_VER_STRING, NULL,
		user_error_function, user_warning_function )) ) {
		read_destroy( read );
		return( NULL );
	}

	/* Catch PNG errors from png_create_info_struct().
	 */
	if( setjmp( read->pPng->jmpbuf ) ) {
		read_destroy( read );
		return( NULL );
	}

	if( !(read->pInfo = png_create_info_struct( read->pPng )) ) {
		read_destroy( read );
		return( NULL );
	}

	return( read );
}

/* Read a PNG file (header) into a VIPS (header).
 */
static int
png2vips( Read *read, int header_only )
{
	int bands, bpp;

	if( setjmp( read->pPng->jmpbuf ) ) 
		return( -1 );

	png_init_io( read->pPng, read->fp );
	png_read_info( read->pPng, read->pInfo );

	switch( read->pInfo->color_type ) {
	case PNG_COLOR_TYPE_PALETTE: bands = 3; break;
	case PNG_COLOR_TYPE_GRAY: bands = 1; break;
	case PNG_COLOR_TYPE_GRAY_ALPHA: bands = 2; break;
	case PNG_COLOR_TYPE_RGB: bands = 3; break;
	case PNG_COLOR_TYPE_RGB_ALPHA: bands = 4; break;

	default:
		im_errormsg( "im_png2vips: unsupported colour type" );
		return( -1 );
	}

	/* 8 or 16 bit.
	 */
	bpp = read->pInfo->bit_depth > 8 ? 2 : 1;

	/* Expand palette images to 8-bit RGB.
	 */
	if( read->pInfo->color_type == PNG_COLOR_TYPE_PALETTE )
	        png_set_expand( read->pPng );

	/* Expand <8 bit images to full bytes.
	 */
	if( read->pInfo->bit_depth < 8 )
		png_set_packing( read->pPng );

	/* If we're an INTEL byte order machine and this is 16bits, we need
	 * to swap bytes.
	 */
	if( read->pInfo->bit_depth > 8 && !im_amiMSBfirst() )
		png_set_swap( read->pPng );

	/* Set VIPS header.
	 */
	im_initdesc( read->out,
		 read->pInfo->width, read->pInfo->height, bands,
		 bpp == 1 ? IM_BBITS_BYTE : IM_BBITS_SHORT, 
		 bpp == 1 ? IM_BANDFMT_UCHAR : IM_BANDFMT_USHORT,
		 IM_CODING_NONE, IM_TYPE_sRGB, 1.0, 1.0, 0, 0 );

	if( !header_only ) {
		int pitch = read->pInfo->width * bands * bpp;
		int y;

		/* Yuk! Have to malloc enough space for the whole image. PNG
		 * is not really designed for large objects ...
		 */
		if( !(read->row_pointer = IM_ARRAY( NULL, 
			read->pInfo->height, png_bytep )) )
			return( -1 );
		if( !(read->data = (png_bytep) im_malloc( NULL,
			read->pInfo->height * pitch ))  )
			return( -1 );

		for( y = 0; y < (int) read->pInfo->height; y++ )
			read->row_pointer[y] = read->data + y * pitch;
		if( im_outcheck( read->out ) || 
			im_setupout( read->out ) || 
			setjmp( read->pPng->jmpbuf ) ) 
			return( -1 );

		png_read_image( read->pPng, read->row_pointer );

		for( y = 0; y < (int) read->pInfo->height; y++ )
			if( im_writeline( y, 
				read->out, read->row_pointer[y] ) )
				return( -1 );
	}

	return( 0 );
}

/* Read a PNG file header into a VIPS header.
 */
int
im_png2vips_header( const char *name, IMAGE *out )
{
	Read *read;

	if( !(read = read_new( name, out )) )
		return( -1 );

	if( png2vips( read, 1 ) ) {
		read_destroy( read );
		return( -1 );
	}

	read_destroy( read );

	return( 0 );
}

/* Read a PNG file into a VIPS image.
 */
int
im_png2vips( const char *name, IMAGE *out )
{
	Read *read;

#ifdef DEBUG
	printf( "im_png2vips: reading \"%s\"\n", name );
#endif /*DEBUG*/

	if( !(read = read_new( name, out )) )
		return( -1 );

	if( png2vips( read, 0 ) ) {
		read_destroy( read );
		return( -1 );
	}

	read_destroy( read );

	return( 0 );
}

#endif /*HAVE_PNG*/
