
// created 06.2003 by Stefan Kleine Stegemann
// 
// licensed under GPL


#include "DocumentWindowController.h"

#include "AppController.h"
#include "DocumentWindow.h"
#include "DocumentPosition.h"
#include "Match.h"
#include "ExtendedImageView.h"

#include <Foundation/NSBundle.h>
#include <Foundation/NSDictionary.h>
#include <Foundation/NSException.h>
#include <Foundation/NSString.h>
#include <Foundation/NSThread.h>
#include <Foundation/NSAutoreleasePool.h>
#include <Foundation/NSDictionary.h>
#include <Foundation/NSValue.h>
#include <Foundation/NSNotification.h>
#include <AppKit/NSClipView.h>
#include <AppKit/NSNibLoading.h>
#include <AppKit/NSTextField.h>
#include <AppKit/NSTableColumn.h>


static NSString* FindThreadParamText      = @"Text";
static NSString* FindThreadParamStartPage = @"StartPage";
static NSString* FindThreadParamStartRect = @"StartRect";

static NSString* N_NewSearchMatchArrived = @"N_NewSearchMatchArrived";
static NSString* UserInfoKeyMatch        = @"UserInfoKeyMatch";
static NSString* UserInfoKeyMatchIndex   = @"UserInfoKeyMatchIndex";


/*
 * A data source for a table that displayes all
 * matches for a search.
 */
@interface MatchesTableDataSource : NSObject
{
   DocumentWindowController* controller;
}

- (id) init;
- (void) dealloc;

- (void) setController: (DocumentWindowController*)_controller;

@end


/*
 * Non-Public methods of DocumentWindowController.
 */
@interface DocumentWindowController(FindPrivate)
- (void) _findAsync: (id)params;
- (void) _findAllOccurrencesOf: (NSString*)text
                startingAtPage: (int)startPage
                    atPosition: (NSRect)startRect;
- (void) _showMatchAt: (int)index autoscroll: (BOOL)scroll;
- (unsigned) _countMatches;
- (Match*) _matchAt: (int)index;
- (NSTableView*) _matchesTable;
- (void) _newMatchArrived: (id)notification;
@end



/*
 * The part of DocumentWindowController that supports
 * text-searching in documents.
 */
@implementation DocumentWindowController(Find)

/*
 * Display a Panel to search for text in searchable Documents.
 */
- (void) showFindPanel: (id)sender
{
   if (!findPanel)
   {
      [NSBundle loadNibNamed: @"find.gorm" owner: self];

      [[NSNotificationCenter defaultCenter] addObserver: self
                                            selector: @selector(_newMatchArrived:)
                                            name: N_NewSearchMatchArrived
                                            object: self];
      // Observer is removed in DocumentWindowController's dealloc
      // method

      [pageCurrentlySearched setStringValue: @""];
      [totalPagesToSearch setStringValue: @""];
      [searchPageLabel setStringValue: @""];
      [ofLabel setStringValue: @""];
   }

   NSAssert(findPanel, @"could not load find panel");

   NSAssert(matchesTable, @"table with matches not found");
   [[matchesTable dataSource] setController: self];
   [matchesTable setTarget: self];
   [matchesTable setDoubleAction: @selector(gotoSelectedMatch:)];

   [findPanel setTitle: [NSString stringWithFormat: @"Find in %@",
                                  (NSWindow*)[[self window] title]]];

   [findPanel orderFrontRegardless];
   [findPanel makeKeyWindow];
   [findPanel display];
}


/*
 * Find the previous occurence of the last searched
 * text (from the current document position). This
 * is only possible if the user has entered a text
 * to search for in the find panel.
 */
- (void) findPrevious: (id)sender
{
   [matchesLock lock];

   if (currentMatch != -1)
   {
      currentMatch--;
      if (currentMatch < 0)
      {
         currentMatch = [matches count] - 1;
      }
      [self _showMatchAt: currentMatch autoscroll: YES];
   }

   [matchesLock unlock];
}


/*
 * Find the next occurence of the last searched text.
 */
- (void) findNext: (id)sender
{
   [matchesLock lock];

   if (currentMatch != -1)
   {
      currentMatch++;
      if (currentMatch >= [matches count])
      {
         currentMatch = 0;
      }
      [self _showMatchAt: currentMatch autoscroll: YES];
   }

   [matchesLock unlock];
}


/*
 * Go to the match that is currently selected in
 * the matches table.
 */
- (void) gotoSelectedMatch: (id)sender
{
   if ([[self _matchesTable] selectedRow] == -1)
   {
      return;
   }

   [matchesLock lock];
   currentMatch = [[self _matchesTable] selectedRow];
   [self _showMatchAt: currentMatch autoscroll: YES];
   [matchesLock unlock];
}


/*
 * If no search is in progress, a serach will be initiated
 * by delegating this method to findAll. Otherwise, the
 * current serach is cancelled.
 */
- (void) findAllOrCancel: (id)sender
{
   if (findInProgress)
   {
      [self cancelSearch: sender];
   }
   else
   {
      [self findAll: sender];
   }
}


/*
 * Initiate a search for a text that has been entered
 * by the user.
 */
- (void) findAll: (id)sender
{
   int                   startPage;
   NSRect                startRect;
   DocumentPosition*     newSelection;
   NSMutableDictionary*  findThreadParams;

   NSAssert(searchText, @"searchText not connected");

   if ([[searchText stringValue] length] == 0)
   {
      NSRunAlertPanel(@"Search for what?",
                      @"Please specify the text you are looking for.",
                      @"Continue",
                      nil,
                      nil);
      return;
   }

   if ((![self contentSelection]) || 
       ([self currentPage] != [[self contentSelection] page]))
      // (![[searchText stringValue] isEqualToString: lastSearchText]))
   {
      startPage = [self currentPage];
      startRect = NSMakeRect(0, 0, 0, -1);
      // -1 is max height
   }
   else
   {
      startPage = [[self contentSelection] page];

      startRect = [[self contentSelection] boundingRect];
      startRect.origin.x = startRect.origin.x + startRect.size.width;
      startRect.origin.y = 
         startRect.origin.y + (startRect.size.height / 2);
      startRect.size.height = 0;
   }

   findThreadParams = [NSMutableDictionary dictionary];

   [findThreadParams setObject: [searchText stringValue]
                     forKey: FindThreadParamText];

   [findThreadParams setObject: [NSNumber numberWithInt: startPage]
                     forKey: FindThreadParamStartPage];

   [findThreadParams setObject: [NSValue valueWithRect: startRect]
                     forKey: FindThreadParamStartRect];

   [NSThread detachNewThreadSelector: @selector(_findAsync:)
             toTarget: self
             withObject: findThreadParams];
}


/*
 * Aborts the current search.
 */
- (void) cancelSearch: (id)sender
{
   if (!findInProgress)
   {
      return;
   }

   findInProgress = NO;
   
   // this 'barrier' is used to synchronize on the
   // find thread
   [findInProgressLock lock];
   [findInProgressLock unlock];
}

@end



@implementation DocumentWindowController(FindPrivate)

/*
 * Searchs for all occurences of a given text
 * in the document and stores the finding places
 * in the matches array.
 */
- (void) _findAllOccurrencesOf: (NSString*)text
                startingAtPage: (int)startPage
                    atPosition: (NSRect)startRect
{
   NSRect                rect       = startRect;
   int                   searchPage = startPage;
   NSString*             context;
   NSMutableDictionary*  userInfo;
   DocumentPosition*     pos;
   Match*                match;
   int                   pageCount;

   [findInProgressLock lock];
   findInProgress = YES;

   [matchesLock lock];
   currentMatch = -1;
   [matches release];
   matches = [[NSMutableArray alloc] initWithCapacity: 0];
   [matchesLock unlock];
   [[self _matchesTable] reloadData];

   pageCount = 0;
   [searchPageLabel setStringValue: @"Searching Page"];
   [totalPagesToSearch setIntValue: [[self pdfDocument] countPages]];
   [ofLabel setStringValue: @"of"];
   [findButton setTitle: @"Abort"];

   do
   {
      pageCount++;
      [pageCurrentlySearched setIntValue: pageCount];

      while (findInProgress &&
             [[self pdfDocument] findText: text
                                 page: &searchPage
                                 toPage: searchPage
                                 position: &rect
                                 context: &context])
      {
         // save match in matches and post a notification
         // about the new match. Go to the matches' position
         // if this is the first match.
         [matchesLock lock];

         pos = [DocumentPosition positionAtPage: searchPage boundingRect: rect];
         match = [Match matchAt: pos context: context searchText: text];

         [matches addObject: match];
         
         userInfo = [NSMutableDictionary dictionary];
         [userInfo setObject: match  forKey: UserInfoKeyMatch];
         [userInfo setObject: [NSNumber numberWithInt: [matches count] - 1]
                   forKey: UserInfoKeyMatchIndex];

         [matchesLock unlock];

         [[NSNotificationCenter defaultCenter]
            postNotificationName: N_NewSearchMatchArrived
                          object: self
                        userInfo: userInfo];

         // move rectangle towards this match
         rect.origin.x = rect.origin.x + rect.size.width;
         rect.origin.y = rect.origin.y + (rect.size.height / 2);
         rect.size.height = 0;
      }

      searchPage++;
      rect = NSMakeRect(0, 0, 0, -1);
      if (searchPage > [[self pdfDocument] countPages])
      {
         searchPage = 1; 
      }
   } while ((searchPage != startPage) && findInProgress);


   if (findInProgress)
   {
      [searchPageLabel setStringValue: @"Completed"];
   }
   else
   {
      [searchPageLabel setStringValue: @"Aborted"];
   }

   [ofLabel setStringValue: @""];
   [totalPagesToSearch setStringValue: @""];
   [pageCurrentlySearched setStringValue: @""];
   [findButton setTitle: @"Find"];

   findInProgress = NO;
   [findInProgressLock unlock];
}


/*
 * To be called from a separate thread. Initiates
 * a _findAllOccurrencesOf.
 */
- (void) _findAsync: (id)params
{
   NSAutoreleasePool* threadPool;
   NSString*          text;
   int                startPage;
   NSRect             startRect;
   
   text      = [params objectForKey: FindThreadParamText];
   startPage = [[params objectForKey: FindThreadParamStartPage] intValue];
   startRect = [[params objectForKey: FindThreadParamStartRect] rectValue];

   threadPool = [[NSAutoreleasePool alloc] init];

   [self _findAllOccurrencesOf: text
         startingAtPage: startPage
         atPosition: startRect];

   [threadPool release];
}


/*
 * Display a match from a search. The position for
 * the match is obtained from the matches array. The
 * method does not protect access to the matches array.
 * If autoscrolling is enabled, this method will ensure
 * that the match is visible.
 */
- (void) _showMatchAt: (int)index autoscroll: (BOOL)scroll
{
   NSPoint  p;
   Match*   match;

   match = [matches objectAtIndex: index];

   [self setContentSelection: [match position]];
   [self setCurrentPage: [[match position] page]];

   // ensure that the selection is visible
   p = NSMakePoint([[match position] boundingRect].origin.x * [self scaleFactor],
                   [[match position] boundingRect].origin.y * [self scaleFactor]);

   if (scroll)
   {
      [[[[(DocumentWindow*)[self window] documentView] scrollView] documentView]
         scrollPoint: p];
   }

   [[self _matchesTable] selectRow: index byExtendingSelection: NO];
   [[self _matchesTable] scrollRowToVisible: index];
}

/*
 * Counts the number of matches that are available
 * for the current search.
 */
- (unsigned) _countMatches
{
   unsigned count;

   [matchesLock lock];
   count = [matches count];
   [matchesLock unlock];

   return count;
}


/*
 * Returns a match at a particular position
 * in the matches list. Returns nil if the
 * given index is invalid.
 */
- (Match*) _matchAt: (int)index
{
   Match* match = nil;

   [matchesLock lock];
   if (index < [matches count])
   {
      match = [matches objectAtIndex: index];
   }
   [matchesLock unlock];

   return match;
}


/*
 * Returns the table view that holds the matches.
 */
- (NSTableView*) _matchesTable
{
   return matchesTable;
}


/*
 * To be invoked by the NSNotificationCenter when
 * a new match was found in the document that is
 * assiociated with this controller.
 */
- (void) _newMatchArrived: (id)notification
{
   Match*      match;
   unsigned    index;
   NSString*   context;

   match   = [[notification userInfo] objectForKey: UserInfoKeyMatch];
   index   = [[[notification userInfo] objectForKey: UserInfoKeyMatchIndex] intValue];

   //NSLog(@"found match at %d (%@)", [[match position] page], [match context]);
   [[self _matchesTable] reloadData];

   [matchesLock lock];
   if (currentMatch == -1)
   {
      currentMatch = 0;
      [self _showMatchAt: currentMatch autoscroll: NO];
   }
   [matchesLock unlock];
}

@end



@implementation MatchesTableDataSource

- (id) init
{
   if ((self = [super init]))
   {
      controller = nil;
   }
   return self;
}


- (void) dealloc
{
   [super dealloc];
}


- (void) setController: (DocumentWindowController*)_controller
{
   controller = _controller;
}


- (int) numberOfRowsInTableView: (NSTableView*)aTableView
{
   return [controller _countMatches];
}

- (id) tableView: (NSTableView*)aTableView
objectValueForTableColumn: (NSTableColumn*)aTableColumn
             row: (int)rowIndex
{
   id      result;
   Match*  match;

   match = [controller _matchAt: rowIndex];
   if (!match)
   {
      return @"??";
   }

   if ([[aTableColumn identifier] isEqualToString: @"page"])
   {
      result = [NSNumber numberWithInt: [[match position] page]];
   }
   else if ([[aTableColumn identifier] isEqualToString: @"context"])
   {
      result = [NSString stringWithFormat: @"... %@ ...", [match context]];
   }
   else
   {
      result = @"invalid_column";
   }

   return result;
}

@end

